/* 
 * Copyright 2015-2024 (c) CoralBlocks LLC - http://www.coralblocks.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package com.coralblocks.coralproto;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.junit.Assert;
import org.junit.Test;

import com.coralblocks.coralproto.field.BooleanField;
import com.coralblocks.coralproto.field.ByteField;
import com.coralblocks.coralproto.field.CharField;
import com.coralblocks.coralproto.field.IntField;
import com.coralblocks.coralproto.field.LongField;
import com.coralblocks.coralproto.field.ShortField;
import com.coralblocks.coralproto.field.SubtypeField;
import com.coralblocks.coralproto.field.TypeField;
import com.coralblocks.coralproto.util.ByteBufferUtils;


public class FieldsTest {
	
	public static class AllFieldsProtoMessage extends AbstractProto {
		
		private static final String IDL_CODE = """
				
				TYPE = A
				SUBTYPE = F
				
				myBoolean: boolean 
				myByte: byte
				myChar: char
				myInt: int
				myLong: long
				myShort: short
				
		""";
		
		public static void main(String[] args) throws IOException {
			IDL idl = new IDL(IDL_CODE, IDL.INDENT + IDL.INDENT);
			String filePath = "src/test/java/com/coralblocks/coralproto/" + FieldsTest.class.getSimpleName() + ".java";
			String sourceCode = idl.getCode();
			IDL.replaceAutoGeneratedCode(filePath, sourceCode);
		}
		
		// Auto-generated code. Do not edit or change anything below here
		
		// BEGIN_AUTO_GENERATED_CODE

        public final static char TYPE = 'A';
        public final static char SUBTYPE = 'F';

        public final TypeField typeField = new TypeField(this, TYPE);
        public final SubtypeField subtypeField = new SubtypeField(this, SUBTYPE);

        public final BooleanField myBoolean = new BooleanField(this);

        public final ByteField myByte = new ByteField(this);

        public final CharField myChar = new CharField(this);

        public final IntField myInt = new IntField(this);

        public final LongField myLong = new LongField(this);

        public final ShortField myShort = new ShortField(this);


		// END_AUTO_GENERATED_CODE
	}
	
	@Test
	public void testAllFields() {
		
		AllFieldsProtoMessage proto = new AllFieldsProtoMessage();
		
		Assert.assertEquals(AllFieldsProtoMessage.TYPE, proto.getType());
		Assert.assertEquals(AllFieldsProtoMessage.SUBTYPE, proto.getSubtype());

		Assert.assertEquals(false, proto.myBoolean.isOptional());
		Assert.assertEquals(true, proto.myBoolean.isPresent());
		Assert.assertEquals(false, proto.myBoolean.get());
		proto.myBoolean.set(true);
		Assert.assertEquals(true, proto.myBoolean.get());
		
		Assert.assertEquals(false, proto.myByte.isOptional());
		Assert.assertEquals(true, proto.myByte.isPresent());
		Assert.assertEquals(0, proto.myByte.get());
		proto.myByte.set(3);
		Assert.assertEquals(3, proto.myByte.get());
		
		Assert.assertEquals(false, proto.myChar.isOptional());
		Assert.assertEquals(true, proto.myChar.isPresent());
		Assert.assertEquals(' ', proto.myChar.get());
		proto.myChar.set('S');
		Assert.assertEquals('S', proto.myChar.get());
		
		Assert.assertEquals(false, proto.myInt.isOptional());
		Assert.assertEquals(true, proto.myInt.isPresent());
		Assert.assertEquals(0, proto.myInt.get());
		proto.myInt.set(4535);
		Assert.assertEquals(4535, proto.myInt.get());
		
		Assert.assertEquals(false, proto.myLong.isOptional());
		Assert.assertEquals(true, proto.myLong.isPresent());
		Assert.assertEquals(0, proto.myLong.get());
		proto.myLong.set(3234234344434L);
		Assert.assertEquals(3234234344434L, proto.myLong.get());
		
		Assert.assertEquals(false, proto.myShort.isOptional());
		Assert.assertEquals(true, proto.myShort.isPresent());
		Assert.assertEquals(0, proto.myShort.get());
		proto.myShort.set(233);
		Assert.assertEquals(233, proto.myShort.get());
		
	}
	
	@Test
	public void testSendReceive() {
		
		ByteBuffer bb = ByteBuffer.allocate(1024);
		
		AllFieldsProtoMessage proto = new AllFieldsProtoMessage();
		
		proto.myBoolean.set(true);
		proto.myByte.set(33);
		proto.myChar.set('S');
		proto.myInt.set(1111);
		proto.myLong.set(222222L);
		proto.myShort.set(3300);
		
		proto.write(bb);
		
		bb.flip();
		
		Assert.assertEquals(AllFieldsProtoMessage.TYPE, bb.get());
		Assert.assertEquals(AllFieldsProtoMessage.SUBTYPE, bb.get());
		
		AllFieldsProtoMessage received = new AllFieldsProtoMessage();
		
		received.read(bb);
		
		Assert.assertEquals(proto.myBoolean.get(), received.myBoolean.get());
		Assert.assertEquals(proto.myByte.get(), received.myByte.get());
		Assert.assertEquals(proto.myChar.get(), received.myChar.get());
		Assert.assertEquals(proto.myInt.get(), received.myInt.get());
		Assert.assertEquals(proto.myLong.get(), received.myLong.get());
		Assert.assertEquals(proto.myShort.get(), received.myShort.get());
		
		bb.clear();
		
		received.writeAscii(true, bb);
		
		bb.flip();
		
		Assert.assertEquals("AF|Y|33|S|1111|222222|3300", ByteBufferUtils.parseString(bb));
		
		bb.clear();
		
		received.writeAscii(false, bb);
		
		bb.flip();
		
		Assert.assertEquals("AF (AllFieldsProtoMessage)|Y|33|S|1111|222222|3300", ByteBufferUtils.parseString(bb));
		
	}
}
