/* 
 * Copyright 2015-2024 (c) CoralBlocks LLC - http://www.coralblocks.com
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package com.coralblocks.coralproto;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.junit.Assert;
import org.junit.Test;

import com.coralblocks.coralproto.field.BytesField;
import com.coralblocks.coralproto.field.CharsField;
import com.coralblocks.coralproto.field.SubtypeField;
import com.coralblocks.coralproto.field.TypeField;
import com.coralblocks.coralproto.field.VarBytesField;
import com.coralblocks.coralproto.field.VarCharsField;
import com.coralblocks.coralproto.util.ByteBufferUtils;


public class CharsAndBytesTest {
	
	public static class CharsAndBytesProtoMessage extends AbstractProto {
		
		private static final String IDL_CODE = """
				
				TYPE = C
				SUBTYPE = B
				
				myChars: chars(8) 
				myVarChars: varchars(1024)
				myBytes: bytes(8) 
				myVarBytes: varbytes(1024)				
				
		""";
		
		public static void main(String[] args) throws IOException {
			IDL idl = new IDL(IDL_CODE, IDL.INDENT + IDL.INDENT);
			String filePath = "src/test/java/com/coralblocks/coralproto/" + CharsAndBytesTest.class.getSimpleName() + ".java";
			String sourceCode = idl.getCode();
			IDL.replaceAutoGeneratedCode(filePath, sourceCode);
		}
		
		// Auto-generated code. Do not edit or change anything below here
		
		// BEGIN_AUTO_GENERATED_CODE

        public final static char TYPE = 'C';
        public final static char SUBTYPE = 'B';

        public final TypeField typeField = new TypeField(this, TYPE);
        public final SubtypeField subtypeField = new SubtypeField(this, SUBTYPE);

        public final CharsField myChars = new CharsField(this, 8);

        public final VarCharsField myVarChars = new VarCharsField(this, 1024);

        public final BytesField myBytes = new BytesField(this, 8);

        public final VarBytesField myVarBytes = new VarBytesField(this, 1024);


		// END_AUTO_GENERATED_CODE
	}
	
	@Test
	public void testCharsField() {
		
		CharsAndBytesProtoMessage proto = new CharsAndBytesProtoMessage();
		
		Assert.assertEquals(CharsAndBytesProtoMessage.TYPE, proto.getType());
		Assert.assertEquals(CharsAndBytesProtoMessage.SUBTYPE, proto.getSubtype());
		
		Assert.assertEquals(false, proto.myChars.isOptional());
		Assert.assertEquals(true, proto.myChars.isPresent());
		
		StringBuilder sb = new StringBuilder(proto.myChars.size());
		for(int i = 0; i < proto.myChars.size(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), proto.myChars.get().toString());
		
		String s = "BLAH";

		proto.myChars.set(s);
		sb.setLength(0);
		sb.append(s);
		for(int i = 0; i < proto.myChars.size() - s.length(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), proto.myChars.get().toString());
		
		proto.myChars.clear();
		sb.setLength(0);
		for(int i = 0; i < proto.myChars.size(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), proto.myChars.get().toString());
		
		s = "FOO1";
		
		proto.myChars.set(s.getBytes());
		sb.setLength(0);
		sb.append(s);
		for(int i = 0; i < proto.myChars.size() - s.length(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), proto.myChars.get().toString());
		
		s = "FOO2";
		
		proto.myChars.set(s.toCharArray());
		sb.setLength(0);
		sb.append(s);
		for(int i = 0; i < proto.myChars.size() - s.length(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), proto.myChars.get().toString());
		
	}
	
	@Test
	public void testVarCharsField() {
		
		CharsAndBytesProtoMessage proto = new CharsAndBytesProtoMessage();
		
		Assert.assertEquals(CharsAndBytesProtoMessage.TYPE, proto.getType());
		Assert.assertEquals(CharsAndBytesProtoMessage.SUBTYPE, proto.getSubtype());
		
		Assert.assertEquals(false, proto.myVarChars.isOptional());
		Assert.assertEquals(true, proto.myVarChars.isPresent());

		StringBuilder sb = new StringBuilder();
		
		String s = "BLAH";
		
		proto.myVarChars.set(s);
		sb.setLength(0);
		sb.append(s);
		Assert.assertEquals(sb.toString(), proto.myVarChars.get().toString());
		
		proto.myVarChars.clear();
		sb.setLength(0);
		Assert.assertEquals(sb.toString(), proto.myVarChars.get().toString());
		
		s = "FOO1";
		
		proto.myVarChars.set(s.getBytes());
		sb.setLength(0);
		sb.append(s);
		Assert.assertEquals(sb.toString(), proto.myVarChars.get().toString());
		
		s = "FOO2";
		
		proto.myVarChars.set(s.toCharArray());
		sb.setLength(0);
		sb.append(s);
		Assert.assertEquals(sb.toString(), proto.myVarChars.get().toString());
		
	}
	
	@Test
	public void testBytesField() {
		
		CharsAndBytesProtoMessage proto = new CharsAndBytesProtoMessage();
		
		Assert.assertEquals(CharsAndBytesProtoMessage.TYPE, proto.getType());
		Assert.assertEquals(CharsAndBytesProtoMessage.SUBTYPE, proto.getSubtype());
		
		Assert.assertEquals(false, proto.myBytes.isOptional());
		Assert.assertEquals(true, proto.myBytes.isPresent());
		
		StringBuilder sb = new StringBuilder(proto.myBytes.size());
		
		String s = "BLAHBLAH";

		proto.myBytes.set(s.getBytes());
		sb.setLength(0);
		sb.append(s);
		for(int i = 0; i < proto.myBytes.size() - s.length(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), ByteBufferUtils.parseString(proto.myBytes.get()));
		
		s = "FOO1FOO1";
		
		proto.myBytes.set(ByteBuffer.wrap(s.getBytes()));
		sb.setLength(0);
		sb.append(s);
		for(int i = 0; i < proto.myBytes.size() - s.length(); i++) sb.append(" ");
		Assert.assertEquals(sb.toString(), ByteBufferUtils.parseString(proto.myBytes.get()));
		
	}
	
	@Test
	public void testVarBytesField() {
		
		CharsAndBytesProtoMessage proto = new CharsAndBytesProtoMessage();
		
		Assert.assertEquals(CharsAndBytesProtoMessage.TYPE, proto.getType());
		Assert.assertEquals(CharsAndBytesProtoMessage.SUBTYPE, proto.getSubtype());
		
		Assert.assertEquals(false, proto.myVarBytes.isOptional());
		Assert.assertEquals(true, proto.myVarBytes.isPresent());
		
		StringBuilder sb = new StringBuilder(proto.myVarBytes.size());
		
		String s = "BLAH";

		proto.myVarBytes.set(s.getBytes());
		sb.setLength(0);
		sb.append(s);
		Assert.assertEquals(sb.toString(), ByteBufferUtils.parseString(proto.myVarBytes.get()));
		
		s = "BLAHFOOO";

		proto.myVarBytes.set(s.getBytes(), 4, 4);
		sb.setLength(0);
		sb.append(s, 4, 8);
		Assert.assertEquals(sb.toString(), ByteBufferUtils.parseString(proto.myVarBytes.get()));
		
		s = "FOO1";
		
		proto.myVarBytes.set(ByteBuffer.wrap(s.getBytes()));
		sb.setLength(0);
		sb.append(s);
		Assert.assertEquals(sb.toString(), ByteBufferUtils.parseString(proto.myVarBytes.get()));
		
	}
	
	@Test
	public void testSendAndReceive() {
		
		CharsAndBytesProtoMessage proto = new CharsAndBytesProtoMessage();
		
		proto.myChars.set("ABCD1234");
		proto.myVarChars.set("0123456789ABC");
		proto.myBytes.set("ZZZZXXXX".getBytes());
		proto.myVarBytes.set("ABCDEFGHIJKLMNOPQRSTUVXZ".getBytes());
		
		ByteBuffer bb = ByteBuffer.allocate(1024);
		
		proto.write(bb);
		
		bb.flip();
		
		Assert.assertEquals(CharsAndBytesProtoMessage.TYPE, bb.get());
		Assert.assertEquals(CharsAndBytesProtoMessage.SUBTYPE, bb.get());
		
		CharsAndBytesProtoMessage received = new CharsAndBytesProtoMessage();
		
		received.read(bb);
		
		Assert.assertEquals("ABCD1234", proto.myChars.get().toString());
		Assert.assertEquals("0123456789ABC", proto.myVarChars.get().toString());
		Assert.assertEquals("ZZZZXXXX", ByteBufferUtils.parseString(proto.myBytes.get()));
		Assert.assertEquals("ABCDEFGHIJKLMNOPQRSTUVXZ", ByteBufferUtils.parseString(proto.myVarBytes.get()));
		
		bb.clear();
		
		received.writeAscii(true, bb);
		
		bb.flip();
		
		Assert.assertEquals("CB|ABCD1234|0123456789ABC|ZZZZXXXX|ABCDEFGHIJKLMNOPQRSTUVXZ", ByteBufferUtils.parseString(bb));
		
		bb.clear();
		
		received.writeAscii(false, bb);
		
		bb.flip();
		
		Assert.assertEquals("CB (CharsAndBytesProtoMessage)|ABCD1234|0123456789ABC|ZZZZXXXX|ABCDEFGHIJKLMNOPQRSTUVXZ", ByteBufferUtils.parseString(bb));
	}
}
