/**
######################################################
# Idea by: Viktor Müller                             #
# Model development: Balázs Könnyű and Viktor Müller #
######################################################

#   Copyright (C) 2012 by Balázs Könnyű and Viktor Müller
#   Contact: konnyu@caesar.elte.hu, mueller.viktor@gmail.com
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   A copy of the GNU General Public License is available via WWW at
#   http://www.gnu.org/licenses/gpl.html
#   You can also obtain it by writing to the Free Software Foundation, Inc.,
#   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, U.S.A.

**/


/*****************************************************************
Cleavage reactions of GAG and GAG-POL polyproteins of HIV-1 by differential equations
1, Kinetics parameters estimated from experimental studies,
2, Enzyme classes: a, PRdPR; b, PR(_)dPR_ and c, _PR_d_PR_. Enzyme classes differ in their dissociation rates. Class c is able to cleave itself but it cannot cleave other sites; the other two classes cannot cleave themselves but can cleave other sites,
3, Etot is the sum of all Class a and Class b enzyme forms, which have the same enzyme activity. Etot catalyzes all heteromolecular cleavage.
4, Enzyme dimerization and cleavage reactions are implemented by differential equations, 
5, RT is produced by equation 79.
6, PR monomers can also be cleaved,
7, Other non-canonocal cleavage sites (e.g. within CA) are not taken into account. 
8, Two types of inhibitors, able to bind reversibly to enzyme classes a and b, respectively.
******************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#define MAXSTP 1e+10
#define TINY 1.0e-30
#define SAFETY 0.9
#define PGROW -0.2
#define PSHRNK -0.25
#define ERRCON 1.89e-4
#define NR_END 1
#define FREE_ARG char*

FILE *output, *input;
time_t start,finish;

void derivs(double,double*,double*,double*,double*);/*equations*/
void rkck(double*,double*,int,double,double,double*,double*,double*,double*); /*RK4-5-6*/
void rkqs(double*,double*,int,double*,double,double,double*,double*,double*,double*,double*); /*stepper*/
void odeint(double*,int,double,double,double,double,double,double*,double*,double**,double,int*,double*);/*driver*/
double FMAX(double,double);
double FMIN(double,double);
double SIGN(double,double);
double *vector(long, long);
void free_vector(double *,long,long);
double **matrix(long,long,long,long);
void free_matrix(double **,long,long,long,long);
void nrerror(char *);

/** names of variables **/
enum sub{nnn,MA,CA,p2,NC,p1,p6gag,TPF,p6pol,PR,p51,p15,INT,p2_NC_TPF_p6pol_PR_p51_p15_INT,CA_p2_NC,PR_p51_p15,PR_p51_p15_INT,NC_TPF_p6pol_PR_p51_p15_INT,p6pol_PR,NC_TPF,p2_NC_p1,CA_p2_NC_p1,p51_p15,MA_CA_p2_NC_TPF_p6pol,p1_p6gag,TPF_p6pol_PR_p51,NC_TPF_p6pol,NC_TPF_p6pol_PR_p51_p15,CA_p2_NC_TPF_p6pol_PR,NC_TPF_p6pol_PR,p2_NC_TPF_p6pol_PR_p51,PR_p51,p2_NC_TPF_p6pol_PR,CA_p2_NC_TPF_p6pol_PR_p51_p15,NC_p1,CA_p2,MA_CA_p2_NC,CA_p2_NC_TPF,p2_NC_TPF_p6pol_PR_p51_p15,TPF_p6pol,MA_CA_p2_NC_TPF,p6pol_PR_p51_p15_INT,p51_p15_INT,TPF_p6pol_PR,MA_CA_p2_NC_p1_p6gag,CA_p2_NC_TPF_p6pol,CA_p2_NC_TPF_p6pol_PR_p51,p6pol_PR_p51_p15,TPF_p6pol_PR_p51_p15_INT,MA_CA_p2_NC_TPF_p6pol_PR,p15_INT,CA_p2_NC_p1_p6gag,p2_NC_TPF_p6pol,NC_TPF_p6pol_PR_p51,MA_CA_p2,TPF_p6pol_PR_p51_p15,NC_p1_p6gag,MA_CA_p2_NC_TPF_p6pol_PR_p51_p15,CA_p2_NC_TPF_p6pol_PR_p51_p15_INT,MA_CA,p2_NC_p1_p6gag,MA_CA_p2_NC_p1,p6pol_PR_p51,p2_NC_TPF,MA_CA_p2_NC_TPF_p6pol_PR_p51,MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT,p2_NC,PRdPR,PRdPR_p51,PRdPR_p51_p15,PRdPR_p51_p15_INT,PR_p51dPR_p51,PR_p51dPR_p51_p15,PR_p51dPR_p51_p15_INT,PR_p51_p15dPR_p51_p15,PR_p51_p15dPR_p51_p15_INT,PR_p51_p15_INTdPR_p51_p15_INT,MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT,MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT,RT,I,dimI,onI,ondimI,dummy,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_3,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11,tPRd_MA_CA_p2_NC_p1_p6gag_3,tPRd_MA_CA_p2_NC_p1_p6gag_4,tPRd_MA_CA_p2_NC_p1_p6gag_5,tPRd_MA_CA_p2_NC_p1_p6gag_1,tPRd_MA_CA_p2_NC_p1_p6gag_2,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11,tPRd_MA_CA_3,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_5,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_6,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_7,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_8,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_9,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_10,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_11,tPRd_MA_CA_p2_3,tPRd_MA_CA_p2_4,tPRd_NC_TPF_p6pol_PR_p51_p15_INT_6,tPRd_NC_TPF_p6pol_PR_p51_p15_INT_7,tPRd_NC_TPF_p6pol_PR_p51_p15_INT_8,tPRd_NC_TPF_p6pol_PR_p51_p15_INT_9,tPRd_NC_TPF_p6pol_PR_p51_p15_INT_10,tPRd_NC_TPF_p6pol_PR_p51_p15_INT_11,tPRd_MA_CA_p2_NC_3,tPRd_MA_CA_p2_NC_4,tPRd_MA_CA_p2_NC_5,tPRd_TPF_p6pol_PR_p51_p15_INT_7,tPRd_TPF_p6pol_PR_p51_p15_INT_8,tPRd_TPF_p6pol_PR_p51_p15_INT_9,tPRd_TPF_p6pol_PR_p51_p15_INT_10,tPRd_TPF_p6pol_PR_p51_p15_INT_11,tPRd_MA_CA_p2_NC_TPF_3,tPRd_MA_CA_p2_NC_TPF_4,tPRd_MA_CA_p2_NC_TPF_5,tPRd_MA_CA_p2_NC_TPF_6,tPRd_p6pol_PR_p51_p15_INT_8,tPRd_p6pol_PR_p51_p15_INT_9,tPRd_p6pol_PR_p51_p15_INT_10,tPRd_p6pol_PR_p51_p15_INT_11,tPRd_MA_CA_p2_NC_TPF_p6pol_3,tPRd_MA_CA_p2_NC_TPF_p6pol_4,tPRd_MA_CA_p2_NC_TPF_p6pol_5,tPRd_MA_CA_p2_NC_TPF_p6pol_6,tPRd_MA_CA_p2_NC_TPF_p6pol_7,tPRd_PR_p51_p15_INT_9,tPRd_PR_p51_p15_INT_10,tPRd_PR_p51_p15_INT_11,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_3,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_4,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_5,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_6,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_7,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_8,tPRd_p51_p15_INT_10,tPRd_p51_p15_INT_11,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_3,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_4,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_5,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_6,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_7,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_8,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_9,tPRd_p15_INT_11,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_3,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_4,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_5,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_6,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_7,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_8,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_9,tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_10,tPRd_CA_p2_NC_p1_p6gag_4,tPRd_CA_p2_NC_p1_p6gag_5,tPRd_CA_p2_NC_p1_p6gag_1,tPRd_CA_p2_NC_p1_p6gag_2,tPRd_p2_NC_p1_p6gag_5,tPRd_p2_NC_p1_p6gag_1,tPRd_p2_NC_p1_p6gag_2,tPRd_NC_p1_p6gag_1,tPRd_NC_p1_p6gag_2,tPRd_p1_p6gag_2,tPRd_MA_CA_p2_NC_p1_3,tPRd_MA_CA_p2_NC_p1_4,tPRd_MA_CA_p2_NC_p1_5,tPRd_MA_CA_p2_NC_p1_1,tPRd_CA_p2_4,tPRd_CA_p2_NC_4,tPRd_CA_p2_NC_5,tPRd_CA_p2_NC_TPF_4,tPRd_CA_p2_NC_TPF_5,tPRd_CA_p2_NC_TPF_6,tPRd_CA_p2_NC_TPF_p6pol_4,tPRd_CA_p2_NC_TPF_p6pol_5,tPRd_CA_p2_NC_TPF_p6pol_6,tPRd_CA_p2_NC_TPF_p6pol_7,tPRd_CA_p2_NC_TPF_p6pol_PR_4,tPRd_CA_p2_NC_TPF_p6pol_PR_5,tPRd_CA_p2_NC_TPF_p6pol_PR_6,tPRd_CA_p2_NC_TPF_p6pol_PR_7,tPRd_CA_p2_NC_TPF_p6pol_PR_8,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_4,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_5,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_6,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_7,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_8,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_9,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_4,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_5,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_6,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_7,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_8,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_9,tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_10,tPRd_p2_NC_5,tPRd_p2_NC_TPF_5,tPRd_p2_NC_TPF_6,tPRd_p2_NC_TPF_p6pol_5,tPRd_p2_NC_TPF_p6pol_6,tPRd_p2_NC_TPF_p6pol_7,tPRd_p2_NC_TPF_p6pol_PR_5,tPRd_p2_NC_TPF_p6pol_PR_6,tPRd_p2_NC_TPF_p6pol_PR_7,tPRd_p2_NC_TPF_p6pol_PR_8,tPRd_p2_NC_TPF_p6pol_PR_p51_5,tPRd_p2_NC_TPF_p6pol_PR_p51_6,tPRd_p2_NC_TPF_p6pol_PR_p51_7,tPRd_p2_NC_TPF_p6pol_PR_p51_8,tPRd_p2_NC_TPF_p6pol_PR_p51_9,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_5,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_6,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_7,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_8,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_9,tPRd_p2_NC_TPF_p6pol_PR_p51_p15_10,tPRd_NC_TPF_6,tPRd_NC_TPF_p6pol_6,tPRd_NC_TPF_p6pol_7,tPRd_NC_TPF_p6pol_PR_6,tPRd_NC_TPF_p6pol_PR_7,tPRd_NC_TPF_p6pol_PR_8,tPRd_NC_TPF_p6pol_PR_p51_6,tPRd_NC_TPF_p6pol_PR_p51_7,tPRd_NC_TPF_p6pol_PR_p51_8,tPRd_NC_TPF_p6pol_PR_p51_9,tPRd_NC_TPF_p6pol_PR_p51_p15_6,tPRd_NC_TPF_p6pol_PR_p51_p15_7,tPRd_NC_TPF_p6pol_PR_p51_p15_8,tPRd_NC_TPF_p6pol_PR_p51_p15_9,tPRd_NC_TPF_p6pol_PR_p51_p15_10,tPRd_TPF_p6pol_7,tPRd_TPF_p6pol_PR_7,tPRd_TPF_p6pol_PR_8,tPRd_TPF_p6pol_PR_p51_7,tPRd_TPF_p6pol_PR_p51_8,tPRd_TPF_p6pol_PR_p51_9,tPRd_TPF_p6pol_PR_p51_p15_7,tPRd_TPF_p6pol_PR_p51_p15_8,tPRd_TPF_p6pol_PR_p51_p15_9,tPRd_TPF_p6pol_PR_p51_p15_10,tPRd_p6pol_PR_8,tPRd_p6pol_PR_p51_8,tPRd_p6pol_PR_p51_9,tPRd_p6pol_PR_p51_p15_8,tPRd_p6pol_PR_p51_p15_9,tPRd_p6pol_PR_p51_p15_10,tPRd_PR_p51_9,tPRd_PR_p51_p15_9,tPRd_PR_p51_p15_10,tPRd_p51_p15_10,tPRd_CA_p2_NC_p1_4,tPRd_CA_p2_NC_p1_5,tPRd_CA_p2_NC_p1_1,tPRd_p2_NC_p1_5,tPRd_p2_NC_p1_1,tPRd_NC_p1_1,tPRd_PR_12,tPRd_CA_13,IPRdPR,IPRdPR_p51,IPRdPR_p51_p15,IPRdPR_p51_p15_INT,IPR_p51dPR_p51,IPR_p51dPR_p51_p15,IPR_p51dPR_p51_p15_INT,IPR_p51_p15dPR_p51_p15,IPR_p51_p15dPR_p51_p15_INT,IPR_p51_p15_INTdPR_p51_p15_INT};
enum rate{nnn2,k1,k2,k3,k4,k5,k6,k7,k8,k9,k10,k11,k12,k13,ko,ka1,kd1,ka2,kd2,ka3,kd3,RTka,RTkd,Ika,Ikd,dimIka,onIka,ondimIka};
enum MM{nnn3,KM1,KM2,KM3,KM4,KM5,KM6,KM7,KM8,KM9,KM10,KM11,KM12,KM13};

int kmax=100000;
int main(int argc, char *argv[])
{
/** PARAMETERS **/	
 double max_t=atof(argv[1]); /*generation time (x2)*/
 double desired_accurancy=/*1.00e-5; */atof(argv[2]); /*desired accuracy (eps): 1e-10*/
	double dxsav=/*1.00e+0;*/ atof(argv[3]);/*result is stored by this interval if kmax>0, 1e-5*/
	int type=atoi(argv[4]); /*Types of runs: 0: only Gag-PR; 1: only GagPol-PR; 2: Gag-GagPol (20:1=Gag:GagPol); 3:Gag-GagPol (20:1=Gag:GagPol)+PR.INT; 4: Gag-GagPol (20:1=Gag:GagPol) + inhibitor */
 double initPR_INT,initPR; /*initial PR.INT concentration (percentage to the GagPol)*//*IF initPR_INT>0 then k0=0.00!!!!!*/

 int kount=0;
 double *xp=NULL,**yp=NULL;
 int NumOfV=286+1/*nvar*/,NumOfP=27+1, NumOfKM=13+1;
	double *yinit=NULL/*ystart*/,t_init=0.00/*x1*/,*par=NULL,*KM=NULL;
	double h_init=1.00e-5/*h1*/,h_min=0.00e-50/*hmin*/;
	register int i,j,jj;

	(void) time(&start);
	printf("start= %f, finish= %f, time of running: %f\n",(double)start,(double)finish,(double)(finish-start));

	/*************************************************************
	max_t:						 end of integration
	desired_accurancy: desired accuracy during integration
	dxsav:             result is stored by this interval if kmax>0
	type:              type of initial state
	initPR_INT:        initial PR.INT concentration (percentage of GagPol)
	kount:             number of integration step
	*xp:               vector of time
	**yp:              result of integration
	NumOfV:				     number of variables
	NumOfP:				     number of parameters
	numOfKM:           number of Michaelis-Menten constants
	yinit:						 initial value of variables
	t_init:					   beginning of integration
	*par:              vector of parameters
	*KM:               vector of Michaelis-Menten constants
	h_init:						 initial step size
	h_min:						 minimal step size
	*************************************************************/

 /*printf("main 1\n");*/

 output=fopen("res.dat", "w");/* external filename -- file of results */
	fclose(output);

 yinit=vector(1,NumOfV);
	par=vector(1,NumOfP);
	KM=vector(1,NumOfKM);
	xp=vector(1,kmax);
	yp=matrix(1,NumOfV,1,kmax);

	for(i=1;i<NumOfV;i++)
  yinit[i]=0.00;
 for(i=1;i<NumOfP;i++)
  par[i]=0.00;
 for(i=1;i<NumOfKM;i++)
  KM[i]=0.00;
 for(i=1;i<kmax;i++)
  xp[i]=0.00;
 for(i=1;i<NumOfV;i++)
 {
  for(j=1;j<kmax;j++)
   yp[i][j]=0.00;
 }

	/*for(i=1;i<=NumOfV;i++)
   printf("yinit[%d]= %e\n",i,yinit[i]);
	printf("\n\n");*/

 /************************************************************************************
																	READING OF PARAMETERS
*************************************************************************************/
	input=fopen ("parameters.dat", "r");/* parameter file */
	switch(type)
	{
		case 0:
						fscanf(input, "%le %le %le %le %le %le\n", &yinit[MA_CA_p2_NC_p1_p6gag],&yinit[PR],&yinit[I],&yinit[dimI],&yinit[onI],&yinit[ondimI]); /*initial cc. of Gag and PR and inhibitors*/
						printf("[Gag]= %f, [PR]= %f,[I]= %f,[dimI]= %f,[onI]= %f, [ondimI]= %f\n\n",yinit[MA_CA_p2_NC_p1_p6gag],yinit[PR],yinit[I],yinit[dimI],yinit[onI],yinit[ondimI]);
						break;
		case 1:
						fscanf(input, "%le %le %le %le %le %le\n", &yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT],&yinit[PR],&yinit[I],&yinit[dimI],&yinit[onI],&yinit[ondimI]); /*initial cc. of  Gag-Pol and PR and inhibitors*/
						printf("[GagPol]= %f, [PR]= %f, [I]= %f,[dimI]= %f,[onI]= %f, [ondimI]= %f\n\n",yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT],yinit[PR],yinit[I],yinit[dimI],yinit[onI],yinit[ondimI]);
						break;
		case 2:
						fscanf(input, "%le %le %le %le %le %le\n", &yinit[MA_CA_p2_NC_p1_p6gag], &yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT], &yinit[I],&yinit[dimI],&yinit[onI],&yinit[ondimI]); /*initial cc. of  Gag and Gag-Pol (20:1) and inhibitors*/
						printf("[Gag]= %f, [GagPol]= %f, [I]= %f,[dimI]= %f,[onI]= %f, [ondimI]= %f\n\n",yinit[MA_CA_p2_NC_p1_p6gag], yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT],yinit[I],yinit[dimI],yinit[onI],yinit[ondimI]);
						break;
		case 3:
						fscanf(input, "%le %le %le %le %le %le %le\n", &yinit[MA_CA_p2_NC_p1_p6gag], &yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT], &yinit[I],&yinit[dimI],&yinit[onI],&yinit[ondimI],&initPR_INT); /*initial cc. of  Gag, Gag-Pol (20:1), inhibitors and PR.INT*/
						printf("[Gag]= %f, [GagPol]= %f, [I]= %f,[dimI]= %f,[onI]= %f, [ondimI]= %f, PR.INT= %f\n\n",yinit[MA_CA_p2_NC_p1_p6gag], yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT],yinit[I],yinit[dimI],yinit[onI],yinit[ondimI],initPR_INT);
						yinit[PR_p51_p15_INT]=yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*initPR_INT;
						yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*=(1.00-initPR_INT);
						/*IF initPR_INT>0 then k0=0.00!!!!!*/
						break;
		case 4:
						fscanf(input, "%le %le %le %le %le %le %le\n", &yinit[MA_CA_p2_NC_p1_p6gag], &yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT], &yinit[I],&yinit[dimI],&yinit[onI],&yinit[ondimI],&yinit[PR]); /*initial cc. of  Gag, Gag-Pol (20:1), inhibitors and PR monomer*/
						printf("[Gag]= %f, [GagPol]= %f, [I]= %f,[dimI]= %f,[onI]= %f, [ondimI]= %f, PR.INT= %f\n\n",yinit[MA_CA_p2_NC_p1_p6gag], yinit[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT],yinit[I],yinit[dimI],yinit[onI],yinit[ondimI],yinit[PR]);
						break;
		default:
						printf("\n\nYou do not give initial concentrations!!!\n\n");
						break;
	}
	/*printf("fscanf, %d\n",scf);*/

	for(i=1;i<=NumOfV;i++)
   printf("yinit[%d]= %e\n",i,yinit[i]);
	printf("\n\n");

	for(jj=1;jj<15;jj++)
  	fscanf(input, "%le ", &par[jj]);
	fscanf(input,"\n");
	for(jj=15;jj<NumOfP;jj++)
  	fscanf(input, "%le ", &par[jj]);
	fscanf(input,"\n");
	for(jj=1;jj<NumOfKM;jj++)
  	fscanf(input, "%le ", &KM[jj]);
	fscanf(input,"\n");
	fclose(input);

	if(initPR_INT>0.00)
		par[ko]=0.00;
	else;

/* check of parameters */
	printf("%s\n", argv[0]);
	printf("max_t= %e, desired_accurancy= %e, dxsav= %e, type= %d\n",max_t,desired_accurancy,dxsav,type);
	printf("k1= %e,k2= %e, k3= %e,k4= %e,k5= %e,k6= %e,k7= %e, k8= %e,k9= %e,k10= %e, k11= %e,k12= %e, k13= %e,ko= %e\n",par[k1],par[k2],par[k3],par[k4],par[k5],par[k6],par[k7],par[k8],par[k9],par[k10],par[k11],par[k12],par[k13],par[ko]);
	printf("ka1= %e,kd1= %e, ka2= %e,kd2= %e,ka3= %e,kd3= %e,RTka= %e, kRTka= %e,Ika= %e,Ikd= %e, dimIka= %e, onIka= %e,ondimIka= %e\n",par[ka1],par[kd1],par[ka2],par[kd2],par[ka3],par[kd3],par[RTka],par[RTkd],par[Ika],par[Ikd],par[dimIka],par[onIka],par[ondimIka]);
		printf("KM1= %e,KM2= %e, KM3= %e,KM4= %e,KM5= %e,KM6= %e,KM7= %e, KM8= %e,KM9= %e,KM10= %e, KM11= %e,KM12= %e, KM13= %e\n",KM[KM1],KM[KM2],KM[KM3],KM[KM4],KM[KM5],KM[KM6],KM[KM7],KM[KM8],KM[KM9],KM[KM10],KM[KM11],KM[KM12],KM[KM13]);


	/*for(jj=1;jj<15;jj++)
		printf("%e ",par[jj]);
	printf("\n");
	for(jj=15;jj<NumOfP;jj++)
		printf("%e ",par[jj]);
	printf("\n");
	for(jj=1;jj<NumOfKM;jj++)
		printf("%e ",KM[jj]);
	printf("\n\n");*/
	/*exit(1);*/
printf("before odeint\n");
odeint(&yinit[0],NumOfV,t_init,max_t,desired_accurancy,h_init,h_min,&par[0],&xp[0],&yp[0],dxsav,&kount,&KM[0]);/* ODE solver -- driver */
printf("after odeint\n");



 printf("kount = %d \n",kount);
	output=fopen("res.dat", "a");/* external filename -- file of results */
	j=0;
	while(j<kount)
	{
	 j++;
	 printf("%e\n",xp[j]);
		fprintf(output,"%e ",xp[j]);
		for(i=1;i<85;i++)
		{
		 /*printf("%e ",yp[i][j]);*/
			fprintf(output,"%e ",yp[i][j]);
		}
		/*printf("\n");*/
		fprintf(output,"\n");
	}
	fclose(output);

	free_vector(par,1,NumOfP);
	free_vector(yinit,1,NumOfV);
	free_vector(KM,1,NumOfKM);
	free_vector(xp,1,kmax);
	free_matrix(yp,1,NumOfV,1,kmax);

 (void) time(&finish);
 printf("start= %f, finish= %f, time of running: %f\n",(double)start,(double)finish,(double)(finish-start));

return 0;

}

void derivs(double x, double *y, double *dydx, double *p, double *km)
{
/******************************************
 x:     time
 *y:    vector of results
 *dydx: vector of derivatives
 *p:    vector of parameters
 *km:   vector of Michaelis-Menten constants
 ******************************************/

/**********************************************************************
             QUASI STEADY STATE OF ENZYME FORMS
**********************************************************************/

double R,tPRd;

/* R: see the equation*/
R=(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM3])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM4])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM5])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM6])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM7])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM8])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM9])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM10])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM11])+(y[MA_CA_p2_NC_p1_p6gag]/km[KM3])+(y[MA_CA_p2_NC_p1_p6gag]/km[KM4])+(y[MA_CA_p2_NC_p1_p6gag]/km[KM5])+(y[MA_CA_p2_NC_p1_p6gag]/km[KM1])+(y[MA_CA_p2_NC_p1_p6gag]/km[KM2])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM4])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM5])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM6])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM7])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM8])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM9])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM10])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM11])+(y[MA_CA]/km[KM3])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM5])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM6])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM7])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM8])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM9])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM10])+(y[p2_NC_TPF_p6pol_PR_p51_p15_INT]/km[KM11])+(y[MA_CA_p2]/km[KM3])+(y[MA_CA_p2]/km[KM4])+(y[NC_TPF_p6pol_PR_p51_p15_INT]/km[KM6])+(y[NC_TPF_p6pol_PR_p51_p15_INT]/km[KM7])+(y[NC_TPF_p6pol_PR_p51_p15_INT]/km[KM8])+(y[NC_TPF_p6pol_PR_p51_p15_INT]/km[KM9])+(y[NC_TPF_p6pol_PR_p51_p15_INT]/km[KM10])+(y[NC_TPF_p6pol_PR_p51_p15_INT]/km[KM11])+(y[MA_CA_p2_NC]/km[KM3])+(y[MA_CA_p2_NC]/km[KM4])+(y[MA_CA_p2_NC]/km[KM5])+(y[TPF_p6pol_PR_p51_p15_INT]/km[KM7])+(y[TPF_p6pol_PR_p51_p15_INT]/km[KM8])+(y[TPF_p6pol_PR_p51_p15_INT]/km[KM9])+(y[TPF_p6pol_PR_p51_p15_INT]/km[KM10])+(y[TPF_p6pol_PR_p51_p15_INT]/km[KM11])+(y[MA_CA_p2_NC_TPF]/km[KM3])+(y[MA_CA_p2_NC_TPF]/km[KM4])+(y[MA_CA_p2_NC_TPF]/km[KM5])+(y[MA_CA_p2_NC_TPF]/km[KM6])+(y[p6pol_PR_p51_p15_INT]/km[KM8])+(y[p6pol_PR_p51_p15_INT]/km[KM9])+(y[p6pol_PR_p51_p15_INT]/km[KM10])+(y[p6pol_PR_p51_p15_INT]/km[KM11])+(y[MA_CA_p2_NC_TPF_p6pol]/km[KM3])+(y[MA_CA_p2_NC_TPF_p6pol]/km[KM4])+(y[MA_CA_p2_NC_TPF_p6pol]/km[KM5])+(y[MA_CA_p2_NC_TPF_p6pol]/km[KM6])+(y[MA_CA_p2_NC_TPF_p6pol]/km[KM7])+(y[PR_p51_p15_INT]/km[KM9])+(y[PR_p51_p15_INT]/km[KM10])+(y[PR_p51_p15_INT]/km[KM11])+(y[MA_CA_p2_NC_TPF_p6pol_PR]/km[KM3])+(y[MA_CA_p2_NC_TPF_p6pol_PR]/km[KM4])+(y[MA_CA_p2_NC_TPF_p6pol_PR]/km[KM5])+(y[MA_CA_p2_NC_TPF_p6pol_PR]/km[KM6])+(y[MA_CA_p2_NC_TPF_p6pol_PR]/km[KM7])+(y[MA_CA_p2_NC_TPF_p6pol_PR]/km[KM8])+(y[p51_p15_INT]/km[KM10])+(y[p51_p15_INT]/km[KM11])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM3])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM4])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM5])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM6])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM7])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM8])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51]/km[KM9])+(y[p15_INT]/km[KM11])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM3])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM4])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM5])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM6])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM7])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM8])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM9])+(y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM10])+(y[CA_p2_NC_p1_p6gag]/km[KM4])+(y[CA_p2_NC_p1_p6gag]/km[KM5])+(y[CA_p2_NC_p1_p6gag]/km[KM1])+(y[CA_p2_NC_p1_p6gag]/km[KM2])+(y[p2_NC_p1_p6gag]/km[KM5])+(y[p2_NC_p1_p6gag]/km[KM1])+(y[p2_NC_p1_p6gag]/km[KM2])+(y[NC_p1_p6gag]/km[KM1])+(y[NC_p1_p6gag]/km[KM2])+(y[p1_p6gag]/km[KM2])+(y[MA_CA_p2_NC_p1]/km[KM3])+(y[MA_CA_p2_NC_p1]/km[KM4])+(y[MA_CA_p2_NC_p1]/km[KM5])+(y[MA_CA_p2_NC_p1]/km[KM1])+(y[CA_p2]/km[KM4])+(y[CA_p2_NC]/km[KM4])+(y[CA_p2_NC]/km[KM5])+(y[CA_p2_NC_TPF]/km[KM4])+(y[CA_p2_NC_TPF]/km[KM5])+(y[CA_p2_NC_TPF]/km[KM6])+(y[CA_p2_NC_TPF_p6pol]/km[KM4])+(y[CA_p2_NC_TPF_p6pol]/km[KM5])+(y[CA_p2_NC_TPF_p6pol]/km[KM6])+(y[CA_p2_NC_TPF_p6pol]/km[KM7])+(y[CA_p2_NC_TPF_p6pol_PR]/km[KM4])+(y[CA_p2_NC_TPF_p6pol_PR]/km[KM5])+(y[CA_p2_NC_TPF_p6pol_PR]/km[KM6])+(y[CA_p2_NC_TPF_p6pol_PR]/km[KM7])+(y[CA_p2_NC_TPF_p6pol_PR]/km[KM8])+(y[CA_p2_NC_TPF_p6pol_PR_p51]/km[KM4])+(y[CA_p2_NC_TPF_p6pol_PR_p51]/km[KM5])+(y[CA_p2_NC_TPF_p6pol_PR_p51]/km[KM6])+(y[CA_p2_NC_TPF_p6pol_PR_p51]/km[KM7])+(y[CA_p2_NC_TPF_p6pol_PR_p51]/km[KM8])+(y[CA_p2_NC_TPF_p6pol_PR_p51]/km[KM9])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM4])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM5])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM6])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM7])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM8])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM9])+(y[CA_p2_NC_TPF_p6pol_PR_p51_p15]/km[KM10])+(y[p2_NC]/km[KM5])+(y[p2_NC_TPF]/km[KM5])+(y[p2_NC_TPF]/km[KM6])+(y[p2_NC_TPF_p6pol]/km[KM5])+(y[p2_NC_TPF_p6pol]/km[KM6])+(y[p2_NC_TPF_p6pol]/km[KM7])+(y[p2_NC_TPF_p6pol_PR]/km[KM5])+(y[p2_NC_TPF_p6pol_PR]/km[KM6])+(y[p2_NC_TPF_p6pol_PR]/km[KM7])+(y[p2_NC_TPF_p6pol_PR]/km[KM8])+(y[p2_NC_TPF_p6pol_PR_p51]/km[KM5])+(y[p2_NC_TPF_p6pol_PR_p51]/km[KM6])+(y[p2_NC_TPF_p6pol_PR_p51]/km[KM7])+(y[p2_NC_TPF_p6pol_PR_p51]/km[KM8])+(y[p2_NC_TPF_p6pol_PR_p51]/km[KM9])+(y[p2_NC_TPF_p6pol_PR_p51_p15]/km[KM5])+(y[p2_NC_TPF_p6pol_PR_p51_p15]/km[KM6])+(y[p2_NC_TPF_p6pol_PR_p51_p15]/km[KM7])+(y[p2_NC_TPF_p6pol_PR_p51_p15]/km[KM8])+(y[p2_NC_TPF_p6pol_PR_p51_p15]/km[KM9])+(y[p2_NC_TPF_p6pol_PR_p51_p15]/km[KM10])+(y[NC_TPF]/km[KM6])+(y[NC_TPF_p6pol]/km[KM6])+(y[NC_TPF_p6pol]/km[KM7])+(y[NC_TPF_p6pol_PR]/km[KM6])+(y[NC_TPF_p6pol_PR]/km[KM7])+(y[NC_TPF_p6pol_PR]/km[KM8])+(y[NC_TPF_p6pol_PR_p51]/km[KM6])+(y[NC_TPF_p6pol_PR_p51]/km[KM7])+(y[NC_TPF_p6pol_PR_p51]/km[KM8])+(y[NC_TPF_p6pol_PR_p51]/km[KM9])+(y[NC_TPF_p6pol_PR_p51_p15]/km[KM6])+(y[NC_TPF_p6pol_PR_p51_p15]/km[KM7])+(y[NC_TPF_p6pol_PR_p51_p15]/km[KM8])+(y[NC_TPF_p6pol_PR_p51_p15]/km[KM9])+(y[NC_TPF_p6pol_PR_p51_p15]/km[KM10])+(y[TPF_p6pol]/km[KM7])+(y[TPF_p6pol_PR]/km[KM7])+(y[TPF_p6pol_PR]/km[KM8])+(y[TPF_p6pol_PR_p51]/km[KM7])+(y[TPF_p6pol_PR_p51]/km[KM8])+(y[TPF_p6pol_PR_p51]/km[KM9])+(y[TPF_p6pol_PR_p51_p15]/km[KM7])+(y[TPF_p6pol_PR_p51_p15]/km[KM8])+(y[TPF_p6pol_PR_p51_p15]/km[KM9])+(y[TPF_p6pol_PR_p51_p15]/km[KM10])+(y[p6pol_PR]/km[KM8])+(y[p6pol_PR_p51]/km[KM8])+(y[p6pol_PR_p51]/km[KM9])+(y[p6pol_PR_p51_p15]/km[KM8])+(y[p6pol_PR_p51_p15]/km[KM9])+(y[p6pol_PR_p51_p15]/km[KM10])+(y[PR_p51]/km[KM9])+(y[PR_p51_p15]/km[KM9])+(y[PR_p51_p15]/km[KM10])+(y[p51_p15]/km[KM10])+(y[CA_p2_NC_p1]/km[KM4])+(y[CA_p2_NC_p1]/km[KM5])+(y[CA_p2_NC_p1]/km[KM1])+(y[p2_NC_p1]/km[KM5])+(y[p2_NC_p1]/km[KM1])+(y[NC_p1]/km[KM1])+(y[PR]/km[KM12])+(y[CA]/km[KM13]);

/* total concentration of enzyme forms */
tPRd =  y[PRdPR] + y[PRdPR_p51] + y[PRdPR_p51_p15] + y[PRdPR_p51_p15_INT] + y[PR_p51dPR_p51] + y[PR_p51dPR_p51_p15] + y[PR_p51dPR_p51_p15_INT] + y[PR_p51_p15dPR_p51_p15] + y[PR_p51_p15dPR_p51_p15_INT] + y[PR_p51_p15_INTdPR_p51_p15_INT];
/*printf("tPRd: %e\n",tPRd);*/

/*printf("R= %e\n",R);*/

/* enzyme complexes */
/*0*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_3]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM3]*(1.00+R));
/*1*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM4]*(1.00+R));
/*2*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM5]*(1.00+R));
/*3*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM6]*(1.00+R));
/*4*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM7]*(1.00+R));
/*5*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM8]*(1.00+R));
/*6*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*7*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*8*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*9*/dydx[tPRd_MA_CA_p2_NC_p1_p6gag_3]=(tPRd*y[MA_CA_p2_NC_p1_p6gag])/(km[KM3]*(1.00+R));
/*10*/dydx[tPRd_MA_CA_p2_NC_p1_p6gag_4]=(tPRd*y[MA_CA_p2_NC_p1_p6gag])/(km[KM4]*(1.00+R));
/*11*/dydx[tPRd_MA_CA_p2_NC_p1_p6gag_5]=(tPRd*y[MA_CA_p2_NC_p1_p6gag])/(km[KM5]*(1.00+R));
/*12*/dydx[tPRd_MA_CA_p2_NC_p1_p6gag_1]=(tPRd*y[MA_CA_p2_NC_p1_p6gag])/(km[KM1]*(1.00+R));
/*13*/dydx[tPRd_MA_CA_p2_NC_p1_p6gag_2]=(tPRd*y[MA_CA_p2_NC_p1_p6gag])/(km[KM2]*(1.00+R));
/*14*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM4]*(1.00+R));
/*15*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM5]*(1.00+R));
/*16*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM6]*(1.00+R));
/*17*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM7]*(1.00+R));
/*18*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM8]*(1.00+R));
/*19*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*20*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*21*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*22*/dydx[tPRd_MA_CA_3]=(tPRd*y[MA_CA])/(km[KM3]*(1.00+R));
/*23*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_5]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM5]*(1.00+R));
/*24*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_6]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM6]*(1.00+R));
/*25*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_7]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM7]*(1.00+R));
/*26*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_8]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM8]*(1.00+R));
/*27*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_9]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*28*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_10]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*29*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_11]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*30*/dydx[tPRd_MA_CA_p2_3]=(tPRd*y[MA_CA_p2])/(km[KM3]*(1.00+R));
/*31*/dydx[tPRd_MA_CA_p2_4]=(tPRd*y[MA_CA_p2])/(km[KM4]*(1.00+R));
/*32*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_6]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM6]*(1.00+R));
/*33*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_7]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM7]*(1.00+R));
/*34*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_8]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM8]*(1.00+R));
/*35*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_9]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*36*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_10]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*37*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_11]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*38*/dydx[tPRd_MA_CA_p2_NC_3]=(tPRd*y[MA_CA_p2_NC])/(km[KM3]*(1.00+R));
/*39*/dydx[tPRd_MA_CA_p2_NC_4]=(tPRd*y[MA_CA_p2_NC])/(km[KM4]*(1.00+R));
/*40*/dydx[tPRd_MA_CA_p2_NC_5]=(tPRd*y[MA_CA_p2_NC])/(km[KM5]*(1.00+R));
/*41*/dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_7]=(tPRd*y[TPF_p6pol_PR_p51_p15_INT])/(km[KM7]*(1.00+R));
/*42*/dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_8]=(tPRd*y[TPF_p6pol_PR_p51_p15_INT])/(km[KM8]*(1.00+R));
/*43*/dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_9]=(tPRd*y[TPF_p6pol_PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*44*/dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_10]=(tPRd*y[TPF_p6pol_PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*45*/dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_11]=(tPRd*y[TPF_p6pol_PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*46*/dydx[tPRd_MA_CA_p2_NC_TPF_3]=(tPRd*y[MA_CA_p2_NC_TPF])/(km[KM3]*(1.00+R));
/*47*/dydx[tPRd_MA_CA_p2_NC_TPF_4]=(tPRd*y[MA_CA_p2_NC_TPF])/(km[KM4]*(1.00+R));
/*48*/dydx[tPRd_MA_CA_p2_NC_TPF_5]=(tPRd*y[MA_CA_p2_NC_TPF])/(km[KM5]*(1.00+R));
/*49*/dydx[tPRd_MA_CA_p2_NC_TPF_6]=(tPRd*y[MA_CA_p2_NC_TPF])/(km[KM6]*(1.00+R));
/*50*/dydx[tPRd_p6pol_PR_p51_p15_INT_8]=(tPRd*y[p6pol_PR_p51_p15_INT])/(km[KM8]*(1.00+R));
/*51*/dydx[tPRd_p6pol_PR_p51_p15_INT_9]=(tPRd*y[p6pol_PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*52*/dydx[tPRd_p6pol_PR_p51_p15_INT_10]=(tPRd*y[p6pol_PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*53*/dydx[tPRd_p6pol_PR_p51_p15_INT_11]=(tPRd*y[p6pol_PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*54*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_3]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol])/(km[KM3]*(1.00+R));
/*55*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_4]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol])/(km[KM4]*(1.00+R));
/*56*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_5]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol])/(km[KM5]*(1.00+R));
/*57*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_6]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol])/(km[KM6]*(1.00+R));
/*58*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_7]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol])/(km[KM7]*(1.00+R));
/*59*/dydx[tPRd_PR_p51_p15_INT_9]=(tPRd*y[PR_p51_p15_INT])/(km[KM9]*(1.00+R));
/*60*/dydx[tPRd_PR_p51_p15_INT_10]=(tPRd*y[PR_p51_p15_INT])/(km[KM10]*(1.00+R));
/*61*/dydx[tPRd_PR_p51_p15_INT_11]=(tPRd*y[PR_p51_p15_INT])/(km[KM11]*(1.00+R));
/*62*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_3]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR])/(km[KM3]*(1.00+R));
/*63*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_4]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR])/(km[KM4]*(1.00+R));
/*64*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_5]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR])/(km[KM5]*(1.00+R));
/*65*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_6]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR])/(km[KM6]*(1.00+R));
/*66*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_7]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR])/(km[KM7]*(1.00+R));
/*67*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_8]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR])/(km[KM8]*(1.00+R));
/*68*/dydx[tPRd_p51_p15_INT_10]=(tPRd*y[p51_p15_INT])/(km[KM10]*(1.00+R));
/*69*/dydx[tPRd_p51_p15_INT_11]=(tPRd*y[p51_p15_INT])/(km[KM11]*(1.00+R));
/*70*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_3]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM3]*(1.00+R));
/*71*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_4]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM4]*(1.00+R));
/*72*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_5]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM5]*(1.00+R));
/*73*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_6]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM6]*(1.00+R));
/*74*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_7]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM7]*(1.00+R));
/*75*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_8]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM8]*(1.00+R));
/*76*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_9]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM9]*(1.00+R));
/*77*/dydx[tPRd_p15_INT_11]=(tPRd*y[p15_INT])/(km[KM11]*(1.00+R));
/*78*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_3]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM3]*(1.00+R));
/*79*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_4]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM4]*(1.00+R));
/*80*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_5]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM5]*(1.00+R));
/*81*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_6]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM6]*(1.00+R));
/*82*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_7]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM7]*(1.00+R));
/*83*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_8]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM8]*(1.00+R));
/*84*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_9]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM9]*(1.00+R));
/*85*/dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_10]=(tPRd*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM10]*(1.00+R));
/*86*/dydx[tPRd_CA_p2_NC_p1_p6gag_4]=(tPRd*y[CA_p2_NC_p1_p6gag])/(km[KM4]*(1.00+R));
/*87*/dydx[tPRd_CA_p2_NC_p1_p6gag_5]=(tPRd*y[CA_p2_NC_p1_p6gag])/(km[KM5]*(1.00+R));
/*88*/dydx[tPRd_CA_p2_NC_p1_p6gag_1]=(tPRd*y[CA_p2_NC_p1_p6gag])/(km[KM1]*(1.00+R));
/*89*/dydx[tPRd_CA_p2_NC_p1_p6gag_2]=(tPRd*y[CA_p2_NC_p1_p6gag])/(km[KM2]*(1.00+R));
/*90*/dydx[tPRd_p2_NC_p1_p6gag_5]=(tPRd*y[p2_NC_p1_p6gag])/(km[KM5]*(1.00+R));
/*91*/dydx[tPRd_p2_NC_p1_p6gag_1]=(tPRd*y[p2_NC_p1_p6gag])/(km[KM1]*(1.00+R));
/*92*/dydx[tPRd_p2_NC_p1_p6gag_2]=(tPRd*y[p2_NC_p1_p6gag])/(km[KM2]*(1.00+R));
/*93*/dydx[tPRd_NC_p1_p6gag_1]=(tPRd*y[NC_p1_p6gag])/(km[KM1]*(1.00+R));
/*94*/dydx[tPRd_NC_p1_p6gag_2]=(tPRd*y[NC_p1_p6gag])/(km[KM2]*(1.00+R));
/*95*/dydx[tPRd_p1_p6gag_2]=(tPRd*y[p1_p6gag])/(km[KM2]*(1.00+R));
/*96*/dydx[tPRd_MA_CA_p2_NC_p1_3]=(tPRd*y[MA_CA_p2_NC_p1])/(km[KM3]*(1.00+R));
/*97*/dydx[tPRd_MA_CA_p2_NC_p1_4]=(tPRd*y[MA_CA_p2_NC_p1])/(km[KM4]*(1.00+R));
/*98*/dydx[tPRd_MA_CA_p2_NC_p1_5]=(tPRd*y[MA_CA_p2_NC_p1])/(km[KM5]*(1.00+R));
/*99*/dydx[tPRd_MA_CA_p2_NC_p1_1]=(tPRd*y[MA_CA_p2_NC_p1])/(km[KM1]*(1.00+R));
/*100*/dydx[tPRd_CA_p2_4]=(tPRd*y[CA_p2])/(km[KM4]*(1.00+R));
/*101*/dydx[tPRd_CA_p2_NC_4]=(tPRd*y[CA_p2_NC])/(km[KM4]*(1.00+R));
/*102*/dydx[tPRd_CA_p2_NC_5]=(tPRd*y[CA_p2_NC])/(km[KM5]*(1.00+R));
/*103*/dydx[tPRd_CA_p2_NC_TPF_4]=(tPRd*y[CA_p2_NC_TPF])/(km[KM4]*(1.00+R));
/*104*/dydx[tPRd_CA_p2_NC_TPF_5]=(tPRd*y[CA_p2_NC_TPF])/(km[KM5]*(1.00+R));
/*105*/dydx[tPRd_CA_p2_NC_TPF_6]=(tPRd*y[CA_p2_NC_TPF])/(km[KM6]*(1.00+R));
/*106*/dydx[tPRd_CA_p2_NC_TPF_p6pol_4]=(tPRd*y[CA_p2_NC_TPF_p6pol])/(km[KM4]*(1.00+R));
/*107*/dydx[tPRd_CA_p2_NC_TPF_p6pol_5]=(tPRd*y[CA_p2_NC_TPF_p6pol])/(km[KM5]*(1.00+R));
/*108*/dydx[tPRd_CA_p2_NC_TPF_p6pol_6]=(tPRd*y[CA_p2_NC_TPF_p6pol])/(km[KM6]*(1.00+R));
/*109*/dydx[tPRd_CA_p2_NC_TPF_p6pol_7]=(tPRd*y[CA_p2_NC_TPF_p6pol])/(km[KM7]*(1.00+R));
/*110*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_4]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR])/(km[KM4]*(1.00+R));
/*111*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_5]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR])/(km[KM5]*(1.00+R));
/*112*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_6]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR])/(km[KM6]*(1.00+R));
/*113*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_7]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR])/(km[KM7]*(1.00+R));
/*114*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_8]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR])/(km[KM8]*(1.00+R));
/*115*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_4]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM4]*(1.00+R));
/*116*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_5]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM5]*(1.00+R));
/*117*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_6]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM6]*(1.00+R));
/*118*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_7]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM7]*(1.00+R));
/*119*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_8]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM8]*(1.00+R));
/*120*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_9]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51])/(km[KM9]*(1.00+R));
/*121*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_4]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM4]*(1.00+R));
/*122*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_5]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM5]*(1.00+R));
/*123*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_6]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM6]*(1.00+R));
/*124*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_7]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM7]*(1.00+R));
/*125*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_8]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM8]*(1.00+R));
/*126*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_9]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM9]*(1.00+R));
/*127*/dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_10]=(tPRd*y[CA_p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM10]*(1.00+R));
/*128*/dydx[tPRd_p2_NC_5]=(tPRd*y[p2_NC])/(km[KM5]*(1.00+R));
/*129*/dydx[tPRd_p2_NC_TPF_5]=(tPRd*y[p2_NC_TPF])/(km[KM5]*(1.00+R));
/*130*/dydx[tPRd_p2_NC_TPF_6]=(tPRd*y[p2_NC_TPF])/(km[KM6]*(1.00+R));
/*131*/dydx[tPRd_p2_NC_TPF_p6pol_5]=(tPRd*y[p2_NC_TPF_p6pol])/(km[KM5]*(1.00+R));
/*132*/dydx[tPRd_p2_NC_TPF_p6pol_6]=(tPRd*y[p2_NC_TPF_p6pol])/(km[KM6]*(1.00+R));
/*133*/dydx[tPRd_p2_NC_TPF_p6pol_7]=(tPRd*y[p2_NC_TPF_p6pol])/(km[KM7]*(1.00+R));
/*134*/dydx[tPRd_p2_NC_TPF_p6pol_PR_5]=(tPRd*y[p2_NC_TPF_p6pol_PR])/(km[KM5]*(1.00+R));
/*135*/dydx[tPRd_p2_NC_TPF_p6pol_PR_6]=(tPRd*y[p2_NC_TPF_p6pol_PR])/(km[KM6]*(1.00+R));
/*136*/dydx[tPRd_p2_NC_TPF_p6pol_PR_7]=(tPRd*y[p2_NC_TPF_p6pol_PR])/(km[KM7]*(1.00+R));
/*137*/dydx[tPRd_p2_NC_TPF_p6pol_PR_8]=(tPRd*y[p2_NC_TPF_p6pol_PR])/(km[KM8]*(1.00+R));
/*138*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_5]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51])/(km[KM5]*(1.00+R));
/*139*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_6]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51])/(km[KM6]*(1.00+R));
/*140*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_7]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51])/(km[KM7]*(1.00+R));
/*141*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_8]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51])/(km[KM8]*(1.00+R));
/*142*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_9]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51])/(km[KM9]*(1.00+R));
/*143*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_5]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM5]*(1.00+R));
/*144*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_6]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM6]*(1.00+R));
/*145*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_7]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM7]*(1.00+R));
/*146*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_8]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM8]*(1.00+R));
/*147*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_9]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM9]*(1.00+R));
/*148*/dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_10]=(tPRd*y[p2_NC_TPF_p6pol_PR_p51_p15])/(km[KM10]*(1.00+R));
/*149*/dydx[tPRd_NC_TPF_6]=(tPRd*y[NC_TPF])/(km[KM6]*(1.00+R));
/*150*/dydx[tPRd_NC_TPF_p6pol_6]=(tPRd*y[NC_TPF_p6pol])/(km[KM6]*(1.00+R));
/*151*/dydx[tPRd_NC_TPF_p6pol_7]=(tPRd*y[NC_TPF_p6pol])/(km[KM7]*(1.00+R));
/*152*/dydx[tPRd_NC_TPF_p6pol_PR_6]=(tPRd*y[NC_TPF_p6pol_PR])/(km[KM6]*(1.00+R));
/*153*/dydx[tPRd_NC_TPF_p6pol_PR_7]=(tPRd*y[NC_TPF_p6pol_PR])/(km[KM7]*(1.00+R));
/*154*/dydx[tPRd_NC_TPF_p6pol_PR_8]=(tPRd*y[NC_TPF_p6pol_PR])/(km[KM8]*(1.00+R));
/*155*/dydx[tPRd_NC_TPF_p6pol_PR_p51_6]=(tPRd*y[NC_TPF_p6pol_PR_p51])/(km[KM6]*(1.00+R));
/*156*/dydx[tPRd_NC_TPF_p6pol_PR_p51_7]=(tPRd*y[NC_TPF_p6pol_PR_p51])/(km[KM7]*(1.00+R));
/*157*/dydx[tPRd_NC_TPF_p6pol_PR_p51_8]=(tPRd*y[NC_TPF_p6pol_PR_p51])/(km[KM8]*(1.00+R));
/*158*/dydx[tPRd_NC_TPF_p6pol_PR_p51_9]=(tPRd*y[NC_TPF_p6pol_PR_p51])/(km[KM9]*(1.00+R));
/*159*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_6]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15])/(km[KM6]*(1.00+R));
/*160*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_7]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15])/(km[KM7]*(1.00+R));
/*161*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_8]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15])/(km[KM8]*(1.00+R));
/*162*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_9]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15])/(km[KM9]*(1.00+R));
/*163*/dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_10]=(tPRd*y[NC_TPF_p6pol_PR_p51_p15])/(km[KM10]*(1.00+R));
/*164*/dydx[tPRd_TPF_p6pol_7]=(tPRd*y[TPF_p6pol])/(km[KM7]*(1.00+R));
/*165*/dydx[tPRd_TPF_p6pol_PR_7]=(tPRd*y[TPF_p6pol_PR])/(km[KM7]*(1.00+R));
/*166*/dydx[tPRd_TPF_p6pol_PR_8]=(tPRd*y[TPF_p6pol_PR])/(km[KM8]*(1.00+R));
/*167*/dydx[tPRd_TPF_p6pol_PR_p51_7]=(tPRd*y[TPF_p6pol_PR_p51])/(km[KM7]*(1.00+R));
/*168*/dydx[tPRd_TPF_p6pol_PR_p51_8]=(tPRd*y[TPF_p6pol_PR_p51])/(km[KM8]*(1.00+R));
/*169*/dydx[tPRd_TPF_p6pol_PR_p51_9]=(tPRd*y[TPF_p6pol_PR_p51])/(km[KM9]*(1.00+R));
/*170*/dydx[tPRd_TPF_p6pol_PR_p51_p15_7]=(tPRd*y[TPF_p6pol_PR_p51_p15])/(km[KM7]*(1.00+R));
/*171*/dydx[tPRd_TPF_p6pol_PR_p51_p15_8]=(tPRd*y[TPF_p6pol_PR_p51_p15])/(km[KM8]*(1.00+R));
/*172*/dydx[tPRd_TPF_p6pol_PR_p51_p15_9]=(tPRd*y[TPF_p6pol_PR_p51_p15])/(km[KM9]*(1.00+R));
/*173*/dydx[tPRd_TPF_p6pol_PR_p51_p15_10]=(tPRd*y[TPF_p6pol_PR_p51_p15])/(km[KM10]*(1.00+R));
/*174*/dydx[tPRd_p6pol_PR_8]=(tPRd*y[p6pol_PR])/(km[KM8]*(1.00+R));
/*175*/dydx[tPRd_p6pol_PR_p51_8]=(tPRd*y[p6pol_PR_p51])/(km[KM8]*(1.00+R));
/*176*/dydx[tPRd_p6pol_PR_p51_9]=(tPRd*y[p6pol_PR_p51])/(km[KM9]*(1.00+R));
/*177*/dydx[tPRd_p6pol_PR_p51_p15_8]=(tPRd*y[p6pol_PR_p51_p15])/(km[KM8]*(1.00+R));
/*178*/dydx[tPRd_p6pol_PR_p51_p15_9]=(tPRd*y[p6pol_PR_p51_p15])/(km[KM9]*(1.00+R));
/*179*/dydx[tPRd_p6pol_PR_p51_p15_10]=(tPRd*y[p6pol_PR_p51_p15])/(km[KM10]*(1.00+R));
/*180*/dydx[tPRd_PR_p51_9]=(tPRd*y[PR_p51])/(km[KM9]*(1.00+R));
/*181*/dydx[tPRd_PR_p51_p15_9]=(tPRd*y[PR_p51_p15])/(km[KM9]*(1.00+R));
/*182*/dydx[tPRd_PR_p51_p15_10]=(tPRd*y[PR_p51_p15])/(km[KM10]*(1.00+R));
/*183*/dydx[tPRd_p51_p15_10]=(tPRd*y[p51_p15])/(km[KM10]*(1.00+R));
/*184*/dydx[tPRd_CA_p2_NC_p1_4]=(tPRd*y[CA_p2_NC_p1])/(km[KM4]*(1.00+R));
/*185*/dydx[tPRd_CA_p2_NC_p1_5]=(tPRd*y[CA_p2_NC_p1])/(km[KM5]*(1.00+R));
/*186*/dydx[tPRd_CA_p2_NC_p1_1]=(tPRd*y[CA_p2_NC_p1])/(km[KM1]*(1.00+R));
/*187*/dydx[tPRd_p2_NC_p1_5]=(tPRd*y[p2_NC_p1])/(km[KM5]*(1.00+R));
/*188*/dydx[tPRd_p2_NC_p1_1]=(tPRd*y[p2_NC_p1])/(km[KM1]*(1.00+R));
/*189*/dydx[tPRd_NC_p1_1]=(tPRd*y[NC_p1])/(km[KM1]*(1.00+R));
/*190*/dydx[tPRd_PR_12]=(tPRd*y[PR])/(km[KM12]*(1.00+R));
/*191*/dydx[tPRd_CA_13]=(tPRd*y[CA])/(km[KM13]*(1.00+R));

/**********************************************************************
                          DIFFERENTIAL EQUATIONS
***********************************************************************/

/* 1:MA*/
  dydx[MA]=(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_3])+(p[k3]*dydx[tPRd_MA_CA_3])+(p[k3]*dydx[tPRd_MA_CA_p2_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_3])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_p1_3]);

		/* 2:CA*/
  dydx[CA]=(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4])+(p[k3]*dydx[tPRd_MA_CA_3])+(p[k4]*dydx[tPRd_CA_p2_NC_p1_p6gag_4])+(p[k4]*dydx[tPRd_CA_p2_4])+(p[k4]*dydx[tPRd_CA_p2_NC_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_4])+(p[k4]*dydx[tPRd_CA_p2_NC_p1_4])-(p[k13]*dydx[tPRd_CA_13]);

		/* 3:p2*/
 dydx[p2]=(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])+(p[k4]*dydx[tPRd_MA_CA_p2_4])+(p[k5]*dydx[tPRd_p2_NC_p1_p6gag_5])+(p[k4]*dydx[tPRd_CA_p2_4])+(p[k5]*dydx[tPRd_p2_NC_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_5])+(p[k5]*dydx[tPRd_p2_NC_p1_5]);

		/* 4:NC*/
 dydx[NC]=(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_5])+(p[k1]*dydx[tPRd_NC_p1_p6gag_1])+(p[k5]*dydx[tPRd_CA_p2_NC_5])+(p[k5]*dydx[tPRd_p2_NC_5])+(p[k6]*dydx[tPRd_NC_TPF_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_6])+(p[k1]*dydx[tPRd_NC_p1_1]);

		/* 5:p1*/
 dydx[p1]=(p[k2]*dydx[tPRd_p1_p6gag_2])+(p[k1]*dydx[tPRd_MA_CA_p2_NC_p1_1])+(p[k1]*dydx[tPRd_CA_p2_NC_p1_1])+(p[k1]*dydx[tPRd_p2_NC_p1_1])+(p[k1]*dydx[tPRd_NC_p1_1]);

		/* 6:p6gag*/
 dydx[p6gag]=(p[k2]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_2])+(p[k2]*dydx[tPRd_CA_p2_NC_p1_p6gag_2])+(p[k2]*dydx[tPRd_p2_NC_p1_p6gag_2])+(p[k2]*dydx[tPRd_NC_p1_p6gag_2])+(p[k2]*dydx[tPRd_p1_p6gag_2]);

		/* 7:TPF*/
 dydx[TPF]=(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_7])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_6])+(p[k6]*dydx[tPRd_NC_TPF_6])+(p[k7]*dydx[tPRd_TPF_p6pol_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_p15_7]);

		/* 8:p6pol*/
 dydx[p6pol]=(p[k8]*dydx[tPRd_p6pol_PR_p51_p15_INT_8])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_TPF_p6pol_7])+(p[k8]*dydx[tPRd_p6pol_PR_8])+(p[k8]*dydx[tPRd_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_p6pol_PR_p51_p15_8]);

		/* 9:PR*/
 dydx[PR]=(p[k9]*dydx[tPRd_PR_p51_p15_INT_9])+(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_p6pol_PR_8])+(p[k9]*dydx[tPRd_PR_p51_9])+(p[k9]*dydx[tPRd_PR_p51_p15_9])-(p[k12]*dydx[tPRd_PR_12])-2.00*(p[ka1]*y[PR]*y[PR])+2.00*(p[kd1]*y[PRdPR])-(p[ka2]*y[PR]*y[PR_p51])+(p[kd2]*y[PRdPR_p51])-(p[ka2]*y[PR]*y[PR_p51_p15])+(p[kd2]*y[PRdPR_p51_p15])-(p[ka2]*y[PR]*y[PR_p51_p15_INT])+(p[kd2]*y[PRdPR_p51_p15_INT])-(p[dimIka]*y[dimI]*y[PR]);

		/* 10:p51*/
 dydx[p51]=(p[k10]*dydx[tPRd_p51_p15_INT_10])+(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_PR_p51_9])+(p[k10]*dydx[tPRd_p51_p15_10])+(p[RTkd]*y[RT])-(p[RTka]*y[p51_p15]*y[p51]);

		/* 11:p15*/ dydx[p15]=(p[k11]*dydx[tPRd_p15_INT_11])+(p[k10]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_10])+(p[k10]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_10])+(p[k10]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_10])+(p[k10]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_10])+(p[k10]*dydx[tPRd_TPF_p6pol_PR_p51_p15_10])+(p[k10]*dydx[tPRd_p6pol_PR_p51_p15_10])+(p[k10]*dydx[tPRd_PR_p51_p15_10])+(p[k10]*dydx[tPRd_p51_p15_10]);

		/* 12:INT*/ dydx[INT]=(p[k11]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_p6pol_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_PR_p51_p15_INT_11])+(p[k11]*dydx[tPRd_p51_p15_INT_11])+(p[k11]*dydx[tPRd_p15_INT_11]);

		/* 13:p2_NC_TPF_p6pol_PR_p51_p15_INT*/ dydx[p2_NC_TPF_p6pol_PR_p51_p15_INT]=(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4])-(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])-(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])-(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])-(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_11]);

		/* 14:CA_p2_NC*/ dydx[CA_p2_NC]=(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_3])+(p[k1]*dydx[tPRd_CA_p2_NC_p1_p6gag_1])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_6])+(p[k1]*dydx[tPRd_CA_p2_NC_p1_1])-(p[k4]*dydx[tPRd_CA_p2_NC_4])-(p[k5]*dydx[tPRd_CA_p2_NC_5]);

		/* 15:PR_p51_p15*/ dydx[PR_p51_p15]=(p[k11]*dydx[tPRd_PR_p51_p15_INT_11])+(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_8])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_8])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_8])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_8])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_p15_8])+(p[k8]*dydx[tPRd_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_PR_p51_p15_9])-(p[k10]*dydx[tPRd_PR_p51_p15_10])-(p[ka2]*y[PR]*y[PR_p51_p15])+(p[kd2]*y[PRdPR_p51_p15])-(p[ka2]*y[PR_p51]*y[PR_p51_p15])+(p[kd2]*y[PR_p51dPR_p51_p15])-2.00*(p[ka2]*y[PR_p51_p15]*y[PR_p51_p15])+2.00*(p[kd2]*y[PR_p51_p15dPR_p51_p15])-(p[ka2]*y[PR_p51_p15]*y[PR_p51_p15_INT])+(p[kd2]*y[PR_p51_p15dPR_p51_p15_INT])-(p[dimIka]*y[dimI]*y[PR_p51_p15]);

		/* 16:PR_p51_p15_INT*/ dydx[PR_p51_p15_INT]=(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_8])+(p[k8]*dydx[tPRd_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_PR_p51_p15_INT_11])-(p[ka2]*y[PR]*y[PR_p51_p15_INT])+(p[kd2]*y[PRdPR_p51_p15_INT])-(p[ka2]*y[PR_p51]*y[PR_p51_p15_INT])+(p[kd2]*y[PR_p51dPR_p51_p15_INT])-(p[ka2]*y[PR_p51_p15]*y[PR_p51_p15_INT])+(p[kd2]*y[PR_p51_p15dPR_p51_p15_INT])-2.00*(p[ka2]*y[PR_p51_p15_INT]*y[PR_p51_p15_INT])+2.00*(p[kd2]*y[PR_p51_p15_INTdPR_p51_p15_INT])-(p[ka3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*y[PR_p51_p15_INT])+(p[kd3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT])-(p[dimIka]*y[dimI]*y[PR_p51_p15_INT])-(p[ondimIka]*y[ondimI]*y[PR_p51_p15_INT]);

		/* 17:NC_TPF_p6pol_PR_p51_p15_INT*/ dydx[NC_TPF_p6pol_PR_p51_p15_INT]=(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])-(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_6])-(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_7])-(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_11]);

		/* 18:p6pol_PR*/ dydx[p6pol_PR]=(p[k9]*dydx[tPRd_p6pol_PR_p51_p15_INT_9])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_7])+(p[k9]*dydx[tPRd_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_p6pol_PR_p51_p15_9])-(p[k8]*dydx[tPRd_p6pol_PR_8]);

		/* 19:NC_TPF*/ dydx[NC_TPF]=(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_5])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_7])-(p[k6]*dydx[tPRd_NC_TPF_6]);

		/* 20:p2_NC_p1*/ dydx[p2_NC_p1]=(p[k2]*dydx[tPRd_p2_NC_p1_p6gag_2])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_p1_4])+(p[k4]*dydx[tPRd_CA_p2_NC_p1_4])-(p[k5]*dydx[tPRd_p2_NC_p1_5])-(p[k1]*dydx[tPRd_p2_NC_p1_1]);

		/* 21:CA_p2_NC_p1*/ dydx[CA_p2_NC_p1]=(p[k2]*dydx[tPRd_CA_p2_NC_p1_p6gag_2])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_p1_3])-(p[k4]*dydx[tPRd_CA_p2_NC_p1_4])-(p[k5]*dydx[tPRd_CA_p2_NC_p1_5])-(p[k1]*dydx[tPRd_CA_p2_NC_p1_1]);

		/* 22:p51_p15*/ dydx[p51_p15]=(p[k11]*dydx[tPRd_p51_p15_INT_11])+(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_9])+(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_9])+(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_9])+(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_9])+(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_p15_9])+(p[k9]*dydx[tPRd_p6pol_PR_p51_p15_9])+(p[k9]*dydx[tPRd_PR_p51_p15_9])-(p[k10]*dydx[tPRd_p51_p15_10])+(p[RTkd]*y[RT])-(p[RTka]*y[p51_p15]*y[p51]);

		/* 23:MA_CA_p2_NC_TPF_p6pol*/ dydx[MA_CA_p2_NC_TPF_p6pol]=(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_8])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_5])-(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_6])-(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_7])+(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT])+(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]);

		/* 24:p1_p6gag*/ dydx[p1_p6gag]=(p[k1]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_1])+(p[k1]*dydx[tPRd_CA_p2_NC_p1_p6gag_1])+(p[k1]*dydx[tPRd_p2_NC_p1_p6gag_1])+(p[k1]*dydx[tPRd_NC_p1_p6gag_1])-(p[k2]*dydx[tPRd_p1_p6gag_2]);

		/* 25:TPF_p6pol_PR_p51*/ dydx[TPF_p6pol_PR_p51]=(p[k10]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_10])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_6])+(p[k10]*dydx[tPRd_TPF_p6pol_PR_p51_p15_10])-(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_7])-(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_8])-(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_9]);

		/* 26:NC_TPF_p6pol*/ dydx[NC_TPF_p6pol]=(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_5])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_8])-(p[k6]*dydx[tPRd_NC_TPF_p6pol_6])-(p[k7]*dydx[tPRd_NC_TPF_p6pol_7]);

		/* 27:NC_TPF_p6pol_PR_p51_p15*/ dydx[NC_TPF_p6pol_PR_p51_p15]=(p[k11]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_5])-(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_6])-(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_7])-(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_9])-(p[k10]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_10]);

		/* 28:CA_p2_NC_TPF_p6pol_PR*/ dydx[CA_p2_NC_TPF_p6pol_PR]=(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_3])+(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_9])-(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_4])-(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_5])-(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_6])-(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_7])-(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_8]);

		/* 29:NC_TPF_p6pol_PR*/ dydx[NC_TPF_p6pol_PR]=(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_5])+(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_9])-(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_6])-(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_7])-(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_8]);

		/* 30:p2_NC_TPF_p6pol_PR_p51*/ dydx[p2_NC_TPF_p6pol_PR_p51]=(p[k10]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_4])+(p[k10]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_10])-(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_5])-(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_6])-(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_7])-(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_8])-(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_9]);

		/* 31:PR_p51*/ dydx[PR_p51]=(p[k10]*dydx[tPRd_PR_p51_p15_INT_10])+(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_p6pol_PR_p51_8])+(p[k10]*dydx[tPRd_PR_p51_p15_10])-(p[k9]*dydx[tPRd_PR_p51_9])-(p[ka2]*y[PR]*y[PR_p51])+(p[kd2]*y[PRdPR_p51])-2.00*(p[ka2]*y[PR_p51]*y[PR_p51])+2.00*(p[kd2]*y[PR_p51dPR_p51])-(p[ka2]*y[PR_p51]*y[PR_p51_p15])+(p[kd2]*y[PR_p51dPR_p51_p15])-(p[ka2]*y[PR_p51]*y[PR_p51_p15_INT])+(p[kd2]*y[PR_p51dPR_p51_p15_INT])-(p[dimIka]*y[dimI]*y[PR_p51]);

		/* 32:p2_NC_TPF_p6pol_PR*/ dydx[p2_NC_TPF_p6pol_PR]=(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_4])+(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_9])-(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_5])-(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_6])-(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_7])-(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_8]);

		/* 33:CA_p2_NC_TPF_p6pol_PR_p51_p15*/ dydx[CA_p2_NC_TPF_p6pol_PR_p51_p15]=(p[k11]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_3])-(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_4])-(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_5])-(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_6])-(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_7])-(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_9])-(p[k10]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_10]);

		/* 34:NC_p1*/ dydx[NC_p1]=(p[k2]*dydx[tPRd_NC_p1_p6gag_2])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_p1_5])+(p[k5]*dydx[tPRd_CA_p2_NC_p1_5])+(p[k5]*dydx[tPRd_p2_NC_p1_5])-(p[k1]*dydx[tPRd_NC_p1_1]);

		/* 35:CA_p2*/ dydx[CA_p2]=(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])+(p[k3]*dydx[tPRd_MA_CA_p2_3])+(p[k5]*dydx[tPRd_CA_p2_NC_p1_p6gag_5])+(p[k5]*dydx[tPRd_CA_p2_NC_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_5])+(p[k5]*dydx[tPRd_CA_p2_NC_p1_5])-(p[k4]*dydx[tPRd_CA_p2_4]);

		/* 36:MA_CA_p2_NC*/ dydx[MA_CA_p2_NC]=(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k1]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_1])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_6])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_6])+(p[k1]*dydx[tPRd_MA_CA_p2_NC_p1_1])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_5]);

		/* 37:CA_p2_NC_TPF*/ dydx[CA_p2_NC_TPF]=(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_3])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_7])-(p[k4]*dydx[tPRd_CA_p2_NC_TPF_4])-(p[k5]*dydx[tPRd_CA_p2_NC_TPF_5])-(p[k6]*dydx[tPRd_CA_p2_NC_TPF_6]);

		/* 38:p2_NC_TPF_p6pol_PR_p51_p15*/dydx[p2_NC_TPF_p6pol_PR_p51_p15]=(p[k11]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_4])-(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_5])-(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_6])-(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_7])-(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_9])-(p[k10]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_10]);

		/* 39:TPF_p6pol*/ dydx[TPF_p6pol]=(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_8])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_6])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_p15_8])-(p[k7]*dydx[tPRd_TPF_p6pol_7]);

		/* 40:MA_CA_p2_NC_TPF*/ dydx[MA_CA_p2_NC_TPF]=(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_7])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_5])-(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_6]);

		/* 41:p6pol_PR_p51_p15_INT*/ dydx[p6pol_PR_p51_p15_INT]=(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_7])-(p[k8]*dydx[tPRd_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_p6pol_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_p6pol_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_p6pol_PR_p51_p15_INT_11]);

		/* 42:p51_p15_INT*/ dydx[p51_p15_INT]=(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_p51_p15_INT_10])-(p[k11]*dydx[tPRd_p51_p15_INT_11]);

		/* 43:TPF_p6pol_PR*/ dydx[TPF_p6pol_PR]=(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_9])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_6])+(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_p15_9])-(p[k7]*dydx[tPRd_TPF_p6pol_PR_7])-(p[k8]*dydx[tPRd_TPF_p6pol_PR_8]);

		/* 44:MA_CA_p2_NC_p1_p6gag*/ dydx[MA_CA_p2_NC_p1_p6gag]=-(p[k3]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_5])-(p[k1]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_1])-(p[k2]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_2]);

		/* 45:CA_p2_NC_TPF_p6pol*/ dydx[CA_p2_NC_TPF_p6pol]=(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_3])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_8])-(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_4])-(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_5])-(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_6])-(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_7]);

		/* 46:CA_p2_NC_TPF_p6pol_PR_p51*/ dydx[CA_p2_NC_TPF_p6pol_PR_p51]=(p[k10]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_3])+(p[k10]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_10])-(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_4])-(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_5])-(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_6])-(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_7])-(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_8])-(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_9]);

		/* 47:p6pol_PR_p51_p15*/ dydx[p6pol_PR_p51_p15]=(p[k11]*dydx[tPRd_p6pol_PR_p51_p15_INT_11])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_p15_7])-(p[k8]*dydx[tPRd_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_p6pol_PR_p51_p15_9])-(p[k10]*dydx[tPRd_p6pol_PR_p51_p15_10]);

		/* 48:TPF_p6pol_PR_p51_p15_INT*/ dydx[TPF_p6pol_PR_p51_p15_INT]=(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_6])-(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_7])-(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_11]);

		/* 49:MA_CA_p2_NC_TPF_p6pol_PR*/ dydx[MA_CA_p2_NC_TPF_p6pol_PR]=(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])+(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_9])+(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_9])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_5])-(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_6])-(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_7])-(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_8]);

		/* 50:p15_INT*/ dydx[p15_INT]=(p[k10]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_p51_p15_INT_10])-(p[k11]*dydx[tPRd_p15_INT_11]);

		/* 51:CA_p2_NC_p1_p6gag*/ dydx[CA_p2_NC_p1_p6gag]=(p[k3]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_3])-(p[k4]*dydx[tPRd_CA_p2_NC_p1_p6gag_4])-(p[k5]*dydx[tPRd_CA_p2_NC_p1_p6gag_5])-(p[k1]*dydx[tPRd_CA_p2_NC_p1_p6gag_1])-(p[k2]*dydx[tPRd_CA_p2_NC_p1_p6gag_2]);

		/* 52:p2_NC_TPF_p6pol*/ dydx[p2_NC_TPF_p6pol]=(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_4])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_8])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_8])+(p[k8]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_8])-(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_5])-(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_6])-(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_7]);

		/* 53:NC_TPF_p6pol_PR_p51*/ dydx[NC_TPF_p6pol_PR_p51]=(p[k10]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_5])+(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_5])+(p[k5]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_5])+(p[k10]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_10])-(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_6])-(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_7])-(p[k8]*dydx[tPRd_NC_TPF_p6pol_PR_p51_8])-(p[k9]*dydx[tPRd_NC_TPF_p6pol_PR_p51_9]);

		/* 54:MA_CA_p2*/ dydx[MA_CA_p2]=(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_5])+(p[k5]*dydx[tPRd_MA_CA_p2_NC_p1_5])-(p[k3]*dydx[tPRd_MA_CA_p2_3])-(p[k4]*dydx[tPRd_MA_CA_p2_4]);

		/* 55:TPF_p6pol_PR_p51_p15*/ dydx[TPF_p6pol_PR_p51_p15]=(p[k11]*dydx[tPRd_TPF_p6pol_PR_p51_p15_INT_11])+(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_6])+(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_6])+(p[k6]*dydx[tPRd_NC_TPF_p6pol_PR_p51_p15_6])-(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_p15_7])-(p[k8]*dydx[tPRd_TPF_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_TPF_p6pol_PR_p51_p15_9])-(p[k10]*dydx[tPRd_TPF_p6pol_PR_p51_p15_10]);

		/* 56:NC_p1_p6gag*/ dydx[NC_p1_p6gag]=(p[k5]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_5])+(p[k5]*dydx[tPRd_CA_p2_NC_p1_p6gag_5])+(p[k5]*dydx[tPRd_p2_NC_p1_p6gag_5])-(p[k1]*dydx[tPRd_NC_p1_p6gag_1])-(p[k2]*dydx[tPRd_NC_p1_p6gag_2]);

		/* 57:MA_CA_p2_NC_TPF_p6pol_PR_p51_p15*/ dydx[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15]=(p[k11]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_5])-(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_6])-(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_7])-(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_8])-(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_9])-(p[k10]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_10]);

		/* 58:CA_p2_NC_TPF_p6pol_PR_p51_p15_INT*/ dydx[CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]=(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_3])-(p[k4]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4])-(p[k5]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])-(p[k6]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])-(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])-(p[k8]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11]);

		/* 59:MA_CA*/ dydx[MA_CA]=(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_4])+(p[k4]*dydx[tPRd_MA_CA_p2_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_4])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_p1_4])-(p[k3]*dydx[tPRd_MA_CA_3]);

		/* 60:p2_NC_p1_p6gag*/ dydx[p2_NC_p1_p6gag]=(p[k4]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_4])+(p[k4]*dydx[tPRd_CA_p2_NC_p1_p6gag_4])-(p[k5]*dydx[tPRd_p2_NC_p1_p6gag_5])-(p[k1]*dydx[tPRd_p2_NC_p1_p6gag_1])-(p[k2]*dydx[tPRd_p2_NC_p1_p6gag_2]);

		/* 61:MA_CA_p2_NC_p1*/ dydx[MA_CA_p2_NC_p1]=(p[k2]*dydx[tPRd_MA_CA_p2_NC_p1_p6gag_2])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_p1_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_p1_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_p1_5])-(p[k1]*dydx[tPRd_MA_CA_p2_NC_p1_1]);

		/* 62:p6pol_PR_p51*/ dydx[p6pol_PR_p51]=(p[k10]*dydx[tPRd_p6pol_PR_p51_p15_INT_10])+(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_CA_p2_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_TPF_p6pol_PR_p51_7])+(p[k10]*dydx[tPRd_p6pol_PR_p51_p15_10])-(p[k8]*dydx[tPRd_p6pol_PR_p51_8])-(p[k9]*dydx[tPRd_p6pol_PR_p51_9]);

		/* 63:p2_NC_TPF*/ dydx[p2_NC_TPF]=(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_4])+(p[k4]*dydx[tPRd_CA_p2_NC_TPF_4])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_7])+(p[k7]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_7])-(p[k5]*dydx[tPRd_p2_NC_TPF_5])-(p[k6]*dydx[tPRd_p2_NC_TPF_6]);

		/* 64:MA_CA_p2_NC_TPF_p6pol_PR_p51*/ dydx[MA_CA_p2_NC_TPF_p6pol_PR_p51]=(p[k10]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])+(p[k10]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_10])-(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_5])-(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_6])-(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_7])-(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_8])-(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_9]);

		/* 65:MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT*/ dydx[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]=-(p[k3]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_3])-(p[k4]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_4])-(p[k5]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_5])-(p[k6]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])-(p[k7]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_7])-(p[k8]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_8])-(p[k9]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_9])-(p[k10]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_10])-(p[k11]*dydx[tPRd_MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT_11])-2.00*(p[ka3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])+2.00*(p[kd3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[ka3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*y[PR_p51_p15_INT])+(p[kd3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT])-(p[ondimIka]*y[ondimI]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]);

		/* 66:p2_NC*/ dydx[p2_NC]=(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_INT_6])+(p[k4]*dydx[tPRd_MA_CA_p2_NC_4])+(p[k1]*dydx[tPRd_p2_NC_p1_p6gag_1])+(p[k4]*dydx[tPRd_CA_p2_NC_4])+(p[k6]*dydx[tPRd_p2_NC_TPF_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_6])+(p[k6]*dydx[tPRd_p2_NC_TPF_p6pol_PR_p51_p15_6])+(p[k1]*dydx[tPRd_p2_NC_p1_1])-(p[k5]*dydx[tPRd_p2_NC_5]);

		/* 67:PRdPR*/ dydx[PRdPR]=(p[ka1]*y[PR]*y[PR])-(p[kd1]*y[PRdPR])-(p[Ika]*y[I]*y[PRdPR])+(p[Ikd]*y[IPRdPR]);

		/* 68:PRdPR_p51*/ dydx[PRdPR_p51]=(p[ka2]*y[PR]*y[PR_p51])-(p[kd2]*y[PRdPR_p51])-(p[Ika]*y[I]*y[PRdPR_p51])+(p[Ikd]*y[IPRdPR_p51]);

		/* 69:PRdPR_p51_p15*/ dydx[PRdPR_p51_p15]=(p[ka2]*y[PR]*y[PR_p51_p15])-(p[kd2]*y[PRdPR_p51_p15])-(p[Ika]*y[I]*y[PRdPR_p51_p15])+(p[Ikd]*y[IPRdPR_p51_p15]);

		/* 70:PRdPR_p51_p15_INT*/ dydx[PRdPR_p51_p15_INT]=(p[ka2]*y[PR]*y[PR_p51_p15_INT])-(p[kd2]*y[PRdPR_p51_p15_INT])-(p[Ika]*y[I]*y[PRdPR_p51_p15_INT])+(p[Ikd]*y[IPRdPR_p51_p15_INT]);

		/* 71:PR_p51dPR_p51*/ dydx[PR_p51dPR_p51]=(p[ka2]*y[PR_p51]*y[PR_p51])-(p[kd2]*y[PR_p51dPR_p51])-(p[Ika]*y[I]*y[PR_p51dPR_p51])+(p[Ikd]*y[IPR_p51dPR_p51]);

		/* 72:PR_p51dPR_p51_p15*/ dydx[PR_p51dPR_p51_p15]=(p[ka2]*y[PR_p51]*y[PR_p51_p15])-(p[kd2]*y[PR_p51dPR_p51_p15])-(p[Ika]*y[I]*y[PR_p51dPR_p51_p15])+(p[Ikd]*y[IPR_p51dPR_p51_p15]);

		/* 73:PR_p51dPR_p51_p15_INT*/ dydx[PR_p51dPR_p51_p15_INT]=(p[ka2]*y[PR_p51]*y[PR_p51_p15_INT])-(p[kd2]*y[PR_p51dPR_p51_p15_INT])-(p[Ika]*y[I]*y[PR_p51dPR_p51_p15_INT])+(p[Ikd]*y[IPR_p51dPR_p51_p15_INT]);

		/* 74:PR_p51_p15dPR_p51_p15*/ dydx[PR_p51_p15dPR_p51_p15]=(p[ka2]*y[PR_p51_p15]*y[PR_p51_p15])-(p[kd2]*y[PR_p51_p15dPR_p51_p15])-(p[Ika]*y[I]*y[PR_p51_p15dPR_p51_p15])+(p[Ikd]*y[IPR_p51_p15dPR_p51_p15]);

		/* 75:PR_p51_p15dPR_p51_p15_INT*/ dydx[PR_p51_p15dPR_p51_p15_INT]=(p[ka2]*y[PR_p51_p15]*y[PR_p51_p15_INT])-(p[kd2]*y[PR_p51_p15dPR_p51_p15_INT])-(p[Ika]*y[I]*y[PR_p51_p15dPR_p51_p15_INT])+(p[Ikd]*y[IPR_p51_p15dPR_p51_p15_INT]);

		/* 76:PR_p51_p15_INTdPR_p51_p15_INT*/ dydx[PR_p51_p15_INTdPR_p51_p15_INT]=(p[ka2]*y[PR_p51_p15_INT]*y[PR_p51_p15_INT])-(p[kd2]*y[PR_p51_p15_INTdPR_p51_p15_INT])+(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT])-(p[Ika]*y[I]*y[PR_p51_p15_INTdPR_p51_p15_INT])+(p[Ikd]*y[IPR_p51_p15_INTdPR_p51_p15_INT]);

		/* 77:MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT*/ dydx[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]=(p[ka3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[kd3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[onIka]*y[onI]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]);

		/* 78:MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT*/ dydx[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT]=(p[ka3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT]*y[PR_p51_p15_INT])-(p[kd3]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT])-(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT])+(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[onIka]*y[onI]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT]);

		/* 79:RT*/  dydx[RT]=(p[RTka]*y[p51_p15]*y[p51])-(p[RTkd]*y[RT]);

		/* 80: Protease inhibitor type 1 (I), which can bind all enzyme-dimer forms a and b in an irreversible way. Its association rate is Ika */  dydx[I]= -(p[Ika]*y[I]*y[PRdPR])-(p[Ika]*y[I]*y[PRdPR_p51])-(p[Ika]*y[I]*y[PRdPR_p51_p15])-(p[Ika]*y[I]*y[PRdPR_p51_p15_INT])-(p[Ika]*y[I]*y[PR_p51dPR_p51])-(p[Ika]*y[I]*y[PR_p51dPR_p51_p15])-(p[Ika]*y[I]*y[PR_p51dPR_p51_p15_INT])-(p[Ika]*y[I]*y[PR_p51_p15dPR_p51_p15])-(p[Ika]*y[I]*y[PR_p51_p15dPR_p51_p15_INT])-(p[Ika]*y[I]*y[PR_p51_p15_INTdPR_p51_p15_INT])+(p[Ikd]*y[IPRdPR])+(p[Ikd]*y[IPRdPR_p51])+(p[Ikd]*y[IPRdPR_p51_p15])+(p[Ikd]*y[IPRdPR_p51_p15_INT])+(p[Ikd]*y[IPR_p51dPR_p51])+(p[Ikd]*y[IPR_p51dPR_p51_p15])+(p[Ikd]*y[IPR_p51dPR_p51_p15_INT])+(p[Ikd]*y[IPR_p51_p15dPR_p51_p15])+(p[Ikd]*y[IPR_p51_p15dPR_p51_p15_INT])+(p[Ikd]*y[IPR_p51_p15_INTdPR_p51_p15_INT]);

		dydx[IPRdPR]=(p[Ika]*y[I]*y[PRdPR])-(p[Ikd]*y[IPRdPR]);
		dydx[IPRdPR_p51]=(p[Ika]*y[I]*y[PRdPR_p51])-(p[Ikd]*y[IPRdPR_p51]);
		dydx[IPRdPR_p51_p15]=(p[Ika]*y[I]*y[PRdPR_p51_p15])-(p[Ikd]*y[IPRdPR_p51_p15]);
		dydx[IPRdPR_p51_p15_INT]=(p[Ika]*y[I]*y[PRdPR_p51_p15_INT])-(p[Ikd]*y[IPRdPR_p51_p15_INT]);
		dydx[IPR_p51dPR_p51]=(p[Ika]*y[I]*y[PR_p51dPR_p51])-(p[Ikd]*y[IPR_p51dPR_p51]);
		dydx[IPR_p51dPR_p51_p15]=(p[Ika]*y[I]*y[PR_p51dPR_p51_p15])-(p[Ikd]*y[IPR_p51dPR_p51_p15]);
		dydx[IPR_p51dPR_p51_p15_INT]=(p[Ika]*y[I]*y[PR_p51dPR_p51_p15_INT])-(p[Ikd]*y[IPR_p51dPR_p51_p15_INT]);
		dydx[IPR_p51_p15dPR_p51_p15]=(p[Ika]*y[I]*y[PR_p51_p15dPR_p51_p15])-(p[Ikd]*y[IPR_p51_p15dPR_p51_p15]);
		dydx[IPR_p51_p15dPR_p51_p15_INT]=(p[Ika]*y[I]*y[PR_p51_p15dPR_p51_p15_INT])-(p[Ikd]*y[IPR_p51_p15dPR_p51_p15_INT]);
		dydx[IPR_p51_p15_INTdPR_p51_p15_INT]=(p[Ika]*y[I]*y[PR_p51_p15_INTdPR_p51_p15_INT])-(p[Ikd]*y[IPR_p51_p15_INTdPR_p51_p15_INT]);


		/* 81: Protease inhibitor type 2 (dimI), which can bind all enzyme-monomer forms a and b in an irreversible way and inhibits the formation of enzyme dimers. Its association rate is dimIka*/  dydx[dimI]=-(p[dimIka]*y[dimI]*y[PR])-(p[dimIka]*y[dimI]*y[PR_p51])-(p[dimIka]*y[dimI]*y[PR_p51_p15])-(p[dimIka]*y[dimI]*y[PR_p51_p15_INT]);

		/* 82: Protease inhibitor type 3 (onI), which can bind all self-cleavage enzyme-dimers (type c) in an irreversible way. Its association rate is onIka */  dydx[onI]=-(p[onIka]*y[onI]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdMA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[onIka]*y[onI]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT]);

		/* 83: Protease inhibitor type 4 (ondimI), which can bind all self-cleavage enzyme-monomers (type c) in an irreversible way and inhibits the formation of enzyme dimers. Its association rate is ondimIka*/  dydx[ondimI]=-(p[ondimIka]*y[ondimI]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INT])-(p[ondimIka]*y[ondimI]*y[PR_p51_p15_INT]);

		/* 84: dummy variable for MA.INTdPR.INT*/  dydx[dummy]=(p[ko]*y[MA_CA_p2_NC_TPF_p6pol_PR_p51_p15_INTdPR_p51_p15_INT]);

}

void rkck(double *y, double *dydx, int n, double x, double h, double *yout,
	double *yerr, double *p, double *km)
{
 /*************************************************************
	Fifth-order Cash-Krap Runge-Kutta method

	*y:      vector of variables
	*dydx:   derivatives of variables
	n:			 number of variables
	x:       time
	h:			 step size
	*yout:	 solution of fifth-order Cash-Krap Runge-Kutta method
	*yerr:	 local error in yout (variables)
	*p:      vector of parameters
	*km:     vector of Michaelis-Menten constants
	*************************************************************/
	int i;
	static double a2=0.2,a3=0.3,a4=0.6,a5=1.0,a6=0.875,b21=0.2,
		b31=3.0/40.0,b32=9.0/40.0,b41=0.3,b42 = -0.9,b43=1.2,
		b51 = -11.0/54.0, b52=2.5,b53 = -70.0/27.0,b54=35.0/27.0,
		b61=1631.0/55296.0,b62=175.0/512.0,b63=575.0/13824.0,
		b64=44275.0/110592.0,b65=253.0/4096.0,c1=37.0/378.0,
		c3=250.0/621.0,c4=125.0/594.0,c6=512.0/1771.0,
		dc5 = -277.0/14336.0;
	double dc1=c1-2825.0/27648.0,dc3=c3-18575.0/48384.0,
		dc4=c4-13525.0/55296.0,dc6=c6-0.25;
	double *ak2,*ak3,*ak4,*ak5,*ak6,*ytemp;

 ak2=vector(1,n);
 ak3=vector(1,n);
 ak4=vector(1,n);
 ak5=vector(1,n);
 ak6=vector(1,n);
 ytemp=vector(1,n);

	for (i=1;i<=n;i++)/*step 1*/
		ytemp[i]=y[i]+b21*h*dydx[i];
	(*derivs)(x+a2*h,ytemp,ak2,&p[0],&km[0]); /*step 2*/

	for (i=1;i<=n;i++)
		ytemp[i]=y[i]+h*(b31*dydx[i]+b32*ak2[i]);
 (*derivs)(x+a3*h,ytemp,ak3,&p[0],&km[0]); /*step 3*/

	for (i=1;i<=n;i++)
		ytemp[i]=y[i]+h*(b41*dydx[i]+b42*ak2[i]+b43*ak3[i]);
 (*derivs)(x+a4*h,ytemp,ak4,&p[0],&km[0]); /*step 4*/

	for (i=1;i<=n;i++)
		ytemp[i]=y[i]+h*(b51*dydx[i]+b52*ak2[i]+b53*ak3[i]+b54*ak4[i]);
 (*derivs)(x+a5*h,ytemp,ak5,&p[0],&km[0]);/*step 5*/

	for (i=1;i<=n;i++)
		ytemp[i]=y[i]+h*(b61*dydx[i]+b62*ak2[i]+b63*ak3[i]+b64*ak4[i]+b65*ak5[i]);
	(*derivs)(x+a6*h,ytemp,ak6,&p[0],&km[0]); /*step 6*/

	for (i=1;i<=n;i++)
		yout[i]=y[i]+h*(c1*dydx[i]+c3*ak3[i]+c4*ak4[i]+c6*ak6[i]);

	for (i=1;i<=n;i++)
		yerr[i]=h*(dc1*dydx[i]+dc3*ak3[i]+dc4*ak4[i]+dc5*ak5[i]+dc6*ak6[i]);
 free_vector(ytemp,1,n);
	free_vector(ak6,1,n);
	free_vector(ak5,1,n);
	free_vector(ak4,1,n);
	free_vector(ak3,1,n);
	free_vector(ak2,1,n);
}
/* (C) Copr. 1986-92 Numerical Recipes Software *(5$r. */

void rkqs(double *y, double *dydx, int n, double *x, double htry, double eps,
	double *yscal, double *hdid, double *hnext, double *p, double *km)
{
 /*************************************************************
	Step monitoring of fifth-order Cash-Krap Runge-Kutta method

	*y:      vector of variables
	*dydx:   derivatives of variables
	n:			 number of variables
	*x:      time
	htry:		 step size
	eps:     desired accuracy
	*yscal:	 error is scaled in this vector
	*hdid:   previous step size
	*hnext:	 next step size
	*p:      vector of parameters
	*km:     vector of Michaelis-Menten constants
	*************************************************************/

	int i;
	double errmax,h,htemp,xnew,*yerr,*ytemp,htmp;

 yerr=vector(1,n);
 ytemp=vector(1,n);
	h=htry;
	for (;;) {
  rkck(y,dydx,n,*x,h,ytemp,yerr,&p[0],&km[0]);
		errmax=0.0;
		for (i=1;i<=n;i++) errmax=FMAX(errmax,fabs(yerr[i]/yscal[i]));
		errmax /= eps;
		if (errmax <= 1.0) break;
  htemp=SAFETY*h*pow(errmax,PSHRNK);
		htmp=h;
		h=(h >=0.00 ? FMAX(htemp, 0.1*h) : FMIN(htemp,0.1*h));
		xnew=(*x)+h;
		/*printf("x= %e, xnew= %e, h= %e\n",*x,xnew,h);*/
		if (xnew == *x) /*
		{
			h=htmp;
			xnew=(*x)+h;
			printf("underflow, the last stepsize is used\n");
		}*/nrerror("Stepsize underflow in rkqs");
 }
 /*printf("x= %e, h= %e\n",*x,h);*/
 if (errmax > ERRCON) *hnext=SAFETY*h*pow(errmax,PGROW);
 else *hnext=5.0*h;
 *x += (*hdid=h);
 for (i=1;i<=n;i++) y[i]=ytemp[i];
	free_vector(ytemp,1,n);
 free_vector(yerr,1,n);
}

void odeint(double *ystart, int nvar, double x1, double x2, double eps, double h1,
	double hmin, double *p,double *xp, double **yp,double dxsav,int *kount, double *km)
{
 /*************************************************************
	Fifth-order Cash-Krap Runge-Kutta method driver with adaptive step size control

	*ystart: initial value of variables
	nvar:		 number of variables
	x1:      start of integration
	x2:			 end of integration
	eps:     desired accuracy
	h1:      first step size
	hmin:		 minimal step size
	*p:      vector of parameters
	*xp:     time vector
	**yp:    result of integration
	dxsav:   result is stored by this interval if kmax>0
	*kount:  number of integration step
	*km:     vector of Michaelis-Menten constants
	*************************************************************/

	int nstp,i;
	double xsav,x,hnext,hdid,h;
	double *yscal,*y,*dydx;

	yscal=vector(1,nvar);
	y=vector(1,nvar);
	dydx=vector(1,nvar);
	x=x1;
	h=SIGN(h1,x2-x1);

	/* *nok = (*nbad) =*/
	(*kount)=0;
 for (i=1;i<=nvar;i++) y[i]=ystart[i];
	if (kmax > 0) xsav=x-dxsav*2.0;
	/*for (nstp=1;nstp<=MAXSTP;nstp++) {*/
	for(;;){
	 		(*derivs)(x,y,dydx,&p[0],&km[0]);
		for (i=1;i<=nvar;i++)
			yscal[i]=fabs(y[i])+fabs(dydx[i]*h)+TINY;
  if (kmax > 0 && (*kount) < kmax-1 && fabs(x-xsav) > fabs(dxsav)) {
			xp[++(*kount)]=x;
   for (i=1;i<=nvar;i++) yp[i][(*kount)]=y[i];
			xsav=x;
		}

		if ((x+h-x2)*(x+h-x1) > 0.0) h=x2-x;
		(*rkqs)(y,dydx,nvar,&x,h,eps,yscal,&hdid,&hnext,&p[0],&km[0]);
		/*if (hdid == h) ++(*nok); else ++(*nbad);*/
		/*printf("%f %f\n",hdid,h);*/
		if ((x-x2)*(x2-x1) >= 0.0) {
			for (i=1;i<=nvar;i++) ystart[i]=y[i];
			if (kmax) {
				xp[++(*kount)]=x;
				for (i=1;i<=nvar;i++) yp[i][(*kount)]=y[i];
			}
   free_vector(dydx,1,nvar);
			free_vector(y,1,nvar);
			free_vector(yscal,1,nvar);
			return ;
		}
		if (fabs(hnext) <= hmin) nrerror("Step size too small in odeint");
		h=hnext;
	}
	nrerror("Too many steps in routine odeint");
}


double FMAX(double a, double b)
{
	if(a>b)
		return a;
	else return b;
}

double FMIN(double a, double b)
{
	if(a>b)
		return b;
	else return a;
}

double SIGN(double a,double b)
{
	 if(b >= 0.0)
		 return fabs(a);
	 else return -fabs(a);
}

double *vector(long nl, long nh)
/*********************************************************
 allocate a double vector with subscript range v[nl..nh]

 nl: start of vector in the memory
 nh: end of vector in the memory
***********************************************************/
{
	double *v;

	v=(double *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(double)));
	if (!v) printf("allocation failure in vector()\n");
	return v-nl+NR_END;
}

void free_vector(double *v, long nl, long nh)
{
/**********************************************
 free a double vector allocated with vector()

 *v: pointer of vector
 nl: start of vector in the memory
 nh: end of vector in the memory
***********************************************/

	free((FREE_ARG) (v+nl-NR_END));
}

double **matrix(long nrl, long nrh, long ncl, long nch)
{
/*********************************************************************
 allocate a double matrix with subscript range m[nrl..nrh][ncl..nch]

 nrl: start of 1 dimenson of matrix (row) in the memory
 nrh: end of 1 dimension of matrix (row) in the memory
 ncl: start of 2 dimenson of matrix (column) in the memory
 nch: end of 2 dimension of matrix (coumn) in the memory
***********************************************************************/

	long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	double **m;

	/* allocate pointers to rows */
	m=(double **) malloc((size_t)((nrow+NR_END)*sizeof(double*)));
	if (!m) printf("allocation failure 1 in matrix()\n");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(double)));
	if (!m[nrl]) printf("allocation failure 2 in matrix()\n");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}

void free_matrix(double **m, long nrl, long nrh, long ncl, long nch)
{
 /*******************************************************
 free a double matrix allocated by matrix()

 **m: pointer of matrix
 nrl: start of 1 dimenson of matrix (row) in the memory
 nrh: end of 1 dimension of matrix (row) in the memory
 ncl: start of 2 dimenson of matrix (column) in the memory
 nch: end of 2 dimension of matrix (coumn) in the memory
**********************************************************/

	free((FREE_ARG) (m[nrl]+ncl-NR_END));
	free((FREE_ARG) (m+nrl-NR_END));
}

void nrerror(char error_text[])
{
/*********************************************
 Numerical Recipes standard error handler
 *********************************************/

	fprintf(stderr,"Numerical Recipes run-time error...\n");
	fprintf(stderr,"%s\n",error_text);
	fprintf(stderr,"...now exiting to system...\n");
	exit(1);
}

#undef MAXSTP
#undef TINY
#undef SAFETY
#undef PGROW
#undef PSHRNK
#undef ERRCON
#undef NR_END
#undef FREE_ARG
