/******************************************************************************\
 *           ___        __                                                    *
 *          /\_ \    __/\ \                                                   *
 *          \//\ \  /\_\ \ \____    ___   _____   _____      __               *
 *            \ \ \ \/\ \ \ '__`\  /'___\/\ '__`\/\ '__`\  /'__`\             *
 *             \_\ \_\ \ \ \ \L\ \/\ \__/\ \ \L\ \ \ \L\ \/\ \L\.\_           *
 *             /\____\\ \_\ \_,__/\ \____\\ \ ,__/\ \ ,__/\ \__/.\_\          *
 *             \/____/ \/_/\/___/  \/____/ \ \ \/  \ \ \/  \/__/\/_/          *
 *                                          \ \_\   \ \_\                     *
 *                                           \/_/    \/_/                     *
 *                                                                            *
 * Copyright (C) 2011-2013                                                    *
 * Dominik Charousset <dominik.charousset@haw-hamburg.de>                     *
 *                                                                            *
 * This file is part of libcppa.                                              *
 * libcppa is free software: you can redistribute it and/or modify it under   *
 * the terms of the GNU Lesser General Public License as published by the     *
 * Free Software Foundation; either version 2.1 of the License,               *
 * or (at your option) any later version.                                     *
 *                                                                            *
 * libcppa is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                       *
 * See the GNU Lesser General Public License for more details.                *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public License   *
 * along with libcppa. If not, see <http://www.gnu.org/licenses/>.            *
\******************************************************************************/


#ifndef CPPA_TDATA_HPP
#define CPPA_TDATA_HPP

#include <typeinfo>
#include <functional>
#include <type_traits>

#include "cppa/get.hpp"
#include "cppa/unit.hpp"
#include "cppa/optional.hpp"

#include "cppa/util/wrapped.hpp"
#include "cppa/util/type_list.hpp"
#include "cppa/util/arg_match_t.hpp"
#include "cppa/util/type_traits.hpp"
#include "cppa/util/rebindable_reference.hpp"

#include "cppa/detail/boxed.hpp"
#include "cppa/detail/types_array.hpp"
#include "cppa/detail/abstract_tuple.hpp"
#include "cppa/detail/tuple_iterator.hpp"
#include "cppa/detail/implicit_conversions.hpp"

namespace cppa {
class uniform_type_info;
const uniform_type_info* uniform_typeid(const std::type_info&);
} // namespace cppa

namespace cppa { namespace detail {

template<typename T>
inline void* ptr_to(T& what) { return &what; }

template<typename T>
inline const void* ptr_to(const T& what) { return &what; }

template<typename T>
inline void* ptr_to(T* what) { return what; }

template<typename T>
inline const void* ptr_to(const T* what) { return what; }

template<typename T>
inline void* ptr_to(const std::reference_wrapper<T>& what) {
    return &(what.get());
}

template<typename T>
inline const void* ptr_to(const std::reference_wrapper<const T>& what) {
    return &(what.get());
}

template<typename T>
inline const uniform_type_info* utype_of(const T&) {
    return static_types_array<T>::arr[0];
}

template<typename T>
inline const uniform_type_info* utype_of(const std::reference_wrapper<T>&) {
    return static_types_array<typename util::rm_const_and_ref<T>::type>::arr[0];
}

template<typename T>
inline const uniform_type_info* utype_of(const T* ptr) {
    return utype_of(*ptr);
}

template<typename T>
struct boxed_or_void {
    static constexpr bool value = is_boxed<T>::value;
};

template<>
struct boxed_or_void<unit_t> {
    static constexpr bool value = true;
};

template<typename T>
struct unbox_ref {
    typedef T type;
};

template<typename T>
struct unbox_ref<std::reference_wrapper<T> > {
    typedef typename std::remove_const<T>::type type;
};

template<typename T>
struct unbox_ref<util::rebindable_reference<T> > {
    typedef typename std::remove_const<T>::type type;
};

/*
 * "enhanced" std::tuple
 */
template<typename...>
struct tdata;

template<>
struct tdata<> {

    typedef tdata super;

    unit_t head;

    typedef unit_t head_type;
    typedef tdata<> tail_type;
    typedef unit_t back_type;
    typedef util::empty_type_list types;

    static constexpr size_t num_elements = 0;

    constexpr tdata() { }

    // swallow any number of additional boxed or unit_t arguments silently
    template<typename... Ts>
    tdata(Ts&&...) {
        typedef util::type_list<typename util::rm_const_and_ref<Ts>::type...> incoming;
        typedef typename util::tl_filter_not_type<incoming, tdata>::type iargs;
        static_assert(util::tl_forall<iargs, boxed_or_void>::value,
                      "Additional unboxed arguments provided");
    }

    typedef tuple_iterator<tdata> const_iterator;

    inline const_iterator begin() const { return {this}; }
    inline const_iterator cbegin() const { return {this}; }

    inline const_iterator end() const { return {this}; }
    inline const_iterator cend() const { return {this}; }

    inline size_t size() const { return num_elements; }

    tdata<>& tail() { return *this; }

    const tdata<>& tail() const { return *this; }

    const tdata<>& ctail() const { return *this; }

    inline const void* at(size_t) const {
        throw std::out_of_range("tdata<>");
    }

    inline void* mutable_at(size_t) {
        throw std::out_of_range("tdata<>");
    }

    inline const uniform_type_info* type_at(size_t) const {
        throw std::out_of_range("tdata<>");
    }

    inline void set() { }

    inline bool operator==(const tdata&) const { return true; }

    inline bool dynamically_typed() const {
        return false;
    }

};

template<bool IsBoxed, bool IsFunction, typename Head, typename T>
struct td_filter_ {
    static inline const T& _(const T& arg) { return arg; }
    static inline T&& _(T&& arg) { return std::move(arg); }
    static inline T& _(T& arg) { return arg; }
};

template<typename Head, typename T>
struct td_filter_<false, true, Head, T> {
    static inline T* _(T* arg) { return arg; }
};

template<typename Head, typename T>
struct td_filter_<true, false, Head, T> {
    static inline Head _(const T&) { return Head{}; }
};

template<typename Head, typename T>
struct td_filter_<true, true, Head, T> : td_filter_<true, false, Head, T> { };

template<typename Head, typename T>
auto td_filter(T&& arg)
    -> decltype(
        td_filter_<
            is_boxed<typename util::rm_const_and_ref<T>::type>::value,
            std::is_function<typename util::rm_const_and_ref<T>::type>::value,
            Head,
            typename util::rm_const_and_ref<T>::type
        >::_(std::forward<T>(arg))) {
    return  td_filter_<
                is_boxed<typename util::rm_const_and_ref<T>::type>::value,
                std::is_function<typename util::rm_const_and_ref<T>::type>::value,
                Head,
                typename util::rm_const_and_ref<T>::type
            >::_(std::forward<T>(arg));
}

template<typename... X, typename... Y>
void tdata_set(tdata<X...>& rhs, const tdata<Y...>& lhs);

template<typename Head, typename... Tail>
struct tdata<Head, Tail...> : tdata<Tail...> {

    typedef tdata<Tail...> super;

    typedef util::type_list<
                typename unbox_ref<Head>::type,
                typename unbox_ref<Tail>::type...>
            types;

    Head head;

    static constexpr size_t num_elements = (sizeof...(Tail) + 1);

    typedef Head head_type;
    typedef tdata<Tail...> tail_type;

    typedef typename std::conditional<
                (sizeof...(Tail) > 0),
                typename tdata<Tail...>::back_type,
                Head
            >::type
            back_type;

    inline tdata() : super(), head() { }

    //tdata(Head arg) : super(), head(std::move(arg)) { }

    tdata(const Head& arg) : super(), head(arg) { }
    tdata(Head&& arg) : super(), head(std::move(arg)) { }

    template<typename T0, typename T1, typename... Ts>
    tdata(T0&& arg0, T1&& arg1, Ts&&... args)
        : super(std::forward<T1>(arg1), std::forward<Ts>(args)...)
        , head(td_filter<Head>(std::forward<T0>(arg0))) {
    }

    tdata(const tdata&) = default;

    tdata(tdata&& other)
        : super(std::move(other.tail())), head(std::move(other.head)) {
    }

    // allow (partial) initialization from a different tdata

    template<typename... Y>
    tdata(tdata<Y...>& other) : super(other.tail()), head(other.head) {
    }

    template<typename... Y>
    tdata(const tdata<Y...>& other) : super(other.tail()), head(other.head) {
    }

    template<typename... Y>
    tdata(tdata<Y...>&& other)
        : super(std::move(other.tail())), head(std::move(other.head)) {
    }

    template<typename... Y>
    tdata& operator=(const tdata<Y...>& other) {
        tdata_set(*this, other);
        return *this;
    }

    template<typename T, typename... Ts>
    inline void set(T&& arg, Ts&&... args) {
        head = std::forward<T>(arg);
        super::set(std::forward<Ts>(args)...);
    }

    inline size_t size() const { return num_elements; }

    typedef tuple_iterator<tdata> const_iterator;

    inline const_iterator begin() const { return {this}; }
    inline const_iterator cbegin() const { return {this}; }

    inline const_iterator end() const { return {this, size()}; }
    inline const_iterator cend() const { return {this, size()}; }


    // upcast
    inline tdata<Tail...>& tail() { return *this; }

    inline const tdata<Tail...>& tail() const { return *this; }

    inline const tdata<Tail...>& ctail() const { return *this; }

    inline const void* at(size_t p) const {
        CPPA_REQUIRE(p < num_elements);
        switch (p) {
            case  0: return ptr_to(head);
            case  1: return ptr_to(super::head);
            case  2: return ptr_to(super::super::head);
            case  3: return ptr_to(super::super::super::head);
            case  4: return ptr_to(super::super::super::super::head);
            default: return super::super::super::super::super::at(p - 5);
        }
    }

    inline void* mutable_at(size_t p) {
#       ifdef CPPA_DEBUG_MODE
        if (p == 0) {
            if (std::is_same<decltype(ptr_to(head)), const void*>::value) {
                throw std::logic_error{"mutable_at with const head"};
            }
            return const_cast<void*>(ptr_to(head));
        }
        return super::mutable_at(p - 1);
#       else
        return const_cast<void*>(at(p));
#       endif
    }

    inline const uniform_type_info* type_at(size_t p) const {
        CPPA_REQUIRE(p < num_elements);
        switch (p) {
            case  0: return utype_of(head);
            case  1: return utype_of(super::head);
            case  2: return utype_of(super::super::head);
            case  3: return utype_of(super::super::super::head);
            case  4: return utype_of(super::super::super::super::head);
            default: return super::super::super::super::super::type_at(p - 5);
        }
    }

    Head& _back(std::integral_constant<size_t, 0>) {
        return head;
    }

    template<size_t Pos>
    back_type& _back(std::integral_constant<size_t, Pos>) {
        std::integral_constant<size_t, Pos-1> token;
        return super::_back(token);
    }

    back_type& back() {
        std::integral_constant<size_t, sizeof...(Tail)> token;
        return _back(token);
    }

    const Head& _back(std::integral_constant<size_t, 0>) const {
        return head;
    }

    template<size_t Pos>
    const back_type& _back(std::integral_constant<size_t, Pos>) const {
        std::integral_constant<size_t, Pos-1> token;
        return super::_back(token);
    }

    const back_type& back() const {
        std::integral_constant<size_t, sizeof...(Tail)> token;
        return _back(token);
    }
};

template<typename... X>
void tdata_set(tdata<X...>&, const tdata<>&) { }

template<typename Head, typename... X, typename... Y>
void tdata_set(tdata<Head, X...>& lhs, tdata<Head, const Y...>& rhs) {
    lhs.head = rhs.head;
    tdata_set(lhs.tail(), rhs.tail());
}

template<size_t N, typename... Tn>
struct tdata_upcast_helper;

template<size_t N, typename Head, typename... Tail>
struct tdata_upcast_helper<N, Head, Tail...> {
    typedef typename tdata_upcast_helper<N-1, Tail...>::type type;
};

template<typename Head, typename... Tail>
struct tdata_upcast_helper<0, Head, Tail...> {
    typedef tdata<Head, Tail...> type;
};

template<typename T>
struct tdata_from_type_list;

template<typename... Ts>
struct tdata_from_type_list<util::type_list<Ts...>> {
    typedef tdata<Ts...> type;
};

template<typename T, typename U>
inline void rebind_value(T& lhs, U& rhs) {
    lhs = rhs;
}

template<typename T, typename U>
inline void rebind_value(util::rebindable_reference<T>& lhs, U& rhs) {
    lhs.rebind(rhs);
}

template<typename... Ts>
inline void rebind_tdata(tdata<Ts...>&) { }

template<typename... Ts, typename... Vs>
void rebind_tdata(tdata<Ts...>& td, const tdata<>&, const Vs&... args) {
    rebind_tdata(td, args...);
}

template<typename... Ts, typename... Us, typename... Vs>
void rebind_tdata(tdata<Ts...>& td, const tdata<Us...>& arg, const Vs&... args) {
    rebind_value(td.head, arg.head);
    rebind_tdata(td.tail(), arg.tail(), args...);
}

} } // namespace cppa::detail

namespace cppa {

template<size_t N, typename... Ts>
const typename util::type_at<N, Ts...>::type& get(const detail::tdata<Ts...>& tv) {
    static_assert(N < sizeof...(Ts), "N >= tv.size()");
    return static_cast<const typename detail::tdata_upcast_helper<N, Ts...>::type&>(tv).head;
}

template<size_t N, typename... Ts>
typename util::type_at<N, Ts...>::type& get_ref(detail::tdata<Ts...>& tv) {
    static_assert(N < sizeof...(Ts), "N >= tv.size()");
    return static_cast<typename detail::tdata_upcast_helper<N, Ts...>::type&>(tv).head;
}

} // namespace cppa

#endif // CPPA_TDATA_HPP
