// Package glogr implements github.com/thockin/logr.Logger in terms of
// github.com/golang/glog.
package glogr

import (
	"fmt"
	"runtime"

	"github.com/golang/glog"
	"github.com/thockin/logr"
)

// New returns a logr.Logger which is implemented by glog.
func New() (logr.Logger, error) {
	return glogger{
		level:  0,
		prefix: "",
	}, nil
}

type glogger struct {
	level  int
	prefix string
}

func prepend(prefix interface{}, args []interface{}) []interface{} {
	return append([]interface{}{prefix}, args...)
}

// Magic string for intermediate frames that we should ignore.
const autogeneratedFrameName = "<autogenerated>"

// Discover how many frames we need to climb to find the caller. This approach
// was suggested by Ian Lance Taylor of the Go team, so it *should* be safe
// enough (famous last words).
func framesToCaller() int {
	// 1 is the immediate caller.  3 should be too many.
	for i := 1; i < 3; i++ {
		_, file, _, _ := runtime.Caller(i + 1) // +1 for this function's frame
		if file != autogeneratedFrameName {
			return i
		}
	}
	return 1 // something went wrong, this is safe
}

func (l glogger) Info(args ...interface{}) {
	if l.Enabled() {
		glog.InfoDepth(framesToCaller(), prepend(l.prefix, args)...)
	}
}

func (l glogger) Infof(format string, args ...interface{}) {
	if l.Enabled() {
		glog.InfoDepth(framesToCaller(), fmt.Sprintf("%s"+format, prepend(l.prefix, args)...))
	}
}

func (l glogger) Enabled() bool {
	return bool(glog.V(glog.Level(l.level)))
}

func (l glogger) Error(args ...interface{}) {
	glog.ErrorDepth(framesToCaller(), prepend(l.prefix, args)...)
}

func (l glogger) Errorf(format string, args ...interface{}) {
	glog.ErrorDepth(framesToCaller(), fmt.Sprintf("%s"+format, prepend(l.prefix, args)...))
}

func (l glogger) V(level int) logr.InfoLogger {
	return glogger{
		level:  level,
		prefix: l.prefix,
	}
}

func (l glogger) NewWithPrefix(prefix string) logr.Logger {
	return glogger{
		level:  l.level,
		prefix: prefix,
	}
}

var _ logr.Logger = glogger{}
var _ logr.InfoLogger = glogger{}
