package dep

import (
	"testing"
	"testing/fstest"

	"github.com/stretchr/testify/assert"
)

func TestParsePoetryLock(t *testing.T) {
	testFS := fstest.MapFS{
		"pyproject.toml": {Data: []byte(`
[tool.poetry]
name = "test-project"
version = "0.1.0"

[tool.poetry.dependencies]
python = "^3.9"
requests = "^2.28.0"
click = "^8.1.0"

[tool.poetry.group.dev.dependencies]
pytest = "^7.0.0"
black = "^22.0.0"
`)},
		"poetry.lock": {Data: []byte(`
# This file is automatically @generated by Poetry 1.4.2 and should not be changed by hand.

[[package]]
name = "certifi"
version = "2023.7.22"

[[package]]
name = "charset-normalizer"
version = "3.2.0"

[[package]]
name = "click"
version = "8.1.7"

[[package]]
name = "idna"
version = "3.4"

[[package]]
name = "requests"
version = "2.31.0"

[[package]]
name = "urllib3"
version = "2.0.4"

[[package]]
name = "pytest"
version = "7.4.0"

[[package]]
name = "black"
version = "23.7.0"
`)},
	}

	manager := newPythonManager(testFS, ".")
	err := manager.Init()
	assert.NoError(t, err)

	// Test finding direct dependencies
	requestsDeps := manager.Find("requests")
	assert.Len(t, requestsDeps, 1)
	requests := requestsDeps[0]
	assert.Equal(t, "requests", requests.Name)
	assert.Equal(t, "^2.28.0", requests.Constraint)
	assert.Equal(t, "2.31.0", requests.Version)
	assert.True(t, requests.IsDirect)
	assert.False(t, requests.IsDevOnly)
	assert.Equal(t, "poetry", requests.ToolName)

	// Test finding dev dependencies
	pytestDeps := manager.Find("pytest")
	assert.Len(t, pytestDeps, 1)
	pytest := pytestDeps[0]
	assert.Equal(t, "pytest", pytest.Name)
	assert.Equal(t, "^7.0.0", pytest.Constraint)
	assert.Equal(t, "7.4.0", pytest.Version)
	assert.True(t, pytest.IsDirect)
	assert.True(t, pytest.IsDevOnly)
	assert.Equal(t, "poetry", pytest.ToolName)

	// Test finding indirect dependencies (only in lock file)
	certifiDeps := manager.Find("certifi")
	assert.Len(t, certifiDeps, 1)
	certifi := certifiDeps[0]
	assert.Equal(t, "certifi", certifi.Name)
	assert.Equal(t, "", certifi.Constraint)
	assert.Equal(t, "2023.7.22", certifi.Version)
	assert.False(t, certifi.IsDirect)
	assert.False(t, certifi.IsDevOnly)
	assert.Equal(t, "poetry", certifi.ToolName)

	// Test Get method
	click, found := manager.Get("click")
	assert.True(t, found)
	assert.Equal(t, "click", click.Name)
	assert.Equal(t, "^8.1.0", click.Constraint)
	assert.Equal(t, "8.1.7", click.Version)
	assert.True(t, click.IsDirect)
	assert.False(t, click.IsDevOnly)
	assert.Equal(t, "poetry", click.ToolName)
}
