package main

import (
	"fmt"
	"io"
	"os"
	"sort"
	"strings"

	"github.com/google/cel-go/cel"

	"github.com/upsun/whatsun/pkg/eval/celfuncs"
)

func main() {
	if len(os.Args) != 2 {
		fmt.Println("Usage: gen_docs <filename>")
		os.Exit(1)
	}

	env, err := cel.NewEnv(celfuncs.DefaultEnvOptions()...)
	if err != nil {
		fmt.Fprintln(os.Stderr, err)
		os.Exit(1)
	}

	if err := writeFile(os.Args[1], func(f io.Writer) error {
		return GenerateDocs(f, env)
	}); err != nil {
		fmt.Fprintln(os.Stderr, err)
		os.Exit(1)
	}
	fmt.Fprintln(os.Stderr, "Documentation generated and saved to:", os.Args[1])
}

func writeFile(path string, fn func(io.Writer) error) error {
	f, err := os.Create(path)
	if err != nil {
		return err
	}
	defer f.Close()
	return fn(f)
}

// GenerateDocs generates documentation for CEL functions.
func GenerateDocs(w io.Writer, env *cel.Env) error {
	functions := env.Functions()
	sortedNames := make([]string, len(functions))
	i := 0
	for name := range functions {
		sortedNames[i] = name
		i++
	}
	sort.Strings(sortedNames)

	categories := []string{"Custom functions", "Built-in functions", "Operators"}
	perCategory := make(map[string]string, len(categories))

	var docs = &celfuncs.Docs{}
	onlyCustomEnv, err := cel.NewCustomEnv(celfuncs.CustomEnvOptions(docs)...)
	if err != nil {
		return err
	}
	isCustom := func(name string) bool {
		for k := range onlyCustomEnv.Functions() {
			if k == name {
				return true
			}
		}
		return false
	}

	for _, name := range sortedNames {
		f := functions[name]

		isOperator := name[0] == '_' || strings.HasSuffix(name, "_")
		category := "Built-in functions"
		if isCustom(name) {
			category = "Custom functions"
		} else if isOperator {
			category = "Operators"
		}

		b := strings.Builder{}

		if isOperator {
			b.WriteString(fmt.Sprintf("\n### `%s`\n", strings.Trim(strings.ReplaceAll(name, "_", " "), "_ ")))
		} else {
			b.WriteString(fmt.Sprintf("\n### `%s`\n", name))
			if funcDoc, ok := docs.GetFunction(name); ok && funcDoc.Comment != "" {
				b.WriteString(fmt.Sprintf("%s.\n", strings.TrimRight(funcDoc.Comment, ".")))
				if funcDoc.Description != "" {
					b.WriteString(fmt.Sprintf("\n%s\n", funcDoc.Description))
				}
			}
		}

		if len(f.OverloadDecls()) > 0 {
			b.WriteString("\n")
		}

		for _, overload := range f.OverloadDecls() {
			argTypes := overload.ArgTypes()
			var argTypesStr = make([]string, len(argTypes))
			for i, at := range argTypes {
				argTypesStr[i] = at.String()
			}

			if funcDoc, ok := docs.GetFunction(name); ok {
				if len(funcDoc.Args) == len(argTypesStr) {
					for i, arg := range funcDoc.Args {
						argTypesStr[i] = arg.Name + " " + argTypesStr[i]
					}
				}
			}

			switch {
			case overload.IsMemberFunction():
				b.WriteString(fmt.Sprintf(
					"* `<%s>.%s(%s)` -> `%s`\n",
					argTypesStr[0],
					name,
					strings.Join(argTypesStr[1:], ", "),
					overload.ResultType(),
				))
			case isOperator:
				descr := f.Name()
				for _, argTypeStr := range argTypesStr {
					descr = strings.Replace(descr, "_", "` `"+argTypeStr+"` `", 1)
				}
				b.WriteString(fmt.Sprintf("* `%s` -> `%s`\n", strings.Trim(descr, " `"), overload.ResultType()))
			default:
				b.WriteString(fmt.Sprintf("* `%s(%s)` -> `%s`\n", f.Name(), strings.Join(argTypesStr, ", "), overload.ResultType()))
			}

			if funcDoc, ok := docs.GetFunction(name); ok {
				if len(funcDoc.Args) == len(argTypesStr) {
					var hasAnyComment bool
					for _, arg := range funcDoc.Args {
						if arg.Comment != "" {
							hasAnyComment = true
							break
						}
					}
					if hasAnyComment {
						for _, arg := range funcDoc.Args {
							if arg.Comment != "" {
								b.WriteString(fmt.Sprintf("    - `%s`: %s\n", arg.Name, arg.Comment))
							} else {
								b.WriteString(fmt.Sprintf("    - `%s`\n", arg.Name))
							}
						}
					}
				}
			}
		}

		if _, ok := perCategory[category]; !ok {
			perCategory[category] = ""
		}
		perCategory[category] += b.String()
	}

	b := strings.Builder{}
	b.WriteString("# CEL function and operator documentation\n\n")
	b.WriteString("This file is generated by `make gen_doc` (DO NOT EDIT).\n\n")
	for _, category := range categories {
		b.WriteString("\n## " + category + "\n")
		b.WriteString(perCategory[category])
	}

	_, err = fmt.Fprint(w, b.String())
	return err
}
