// +build unit

package util

import (
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"path/filepath"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func Test_getLatestReleaseFromGithubUsingHttpRedirect(t *testing.T) {
	t.Parallel()
	type args struct {
		githubOwner string
		githubRepo  string
	}
	tests := []struct {
		name    string
		args    args
		want    string
		wantErr bool
	}{
		{
			"Happy Path",
			args{
				githubOwner: "org",
				githubRepo:  "repo",
			},
			"v1.2.3",
			false,
		},
		{
			"Redirection from one repo to another (redirects to happy path)",
			args{
				githubOwner: "redirect",
				githubRepo:  "repo",
			},
			"v1.2.3",
			false,
		},
		{
			"Not a redirect",
			args{
				githubOwner: "foo",
				githubRepo:  "bar",
			},
			"",
			true,
		},
		{
			"Redirect but no location header",
			args{
				githubOwner: "no",
				githubRepo:  "location",
			},
			"",
			true,
		},
		{
			"Bad Redirect - redirect to a page of an unknown format",
			args{
				githubOwner: "bad",
				githubRepo:  "location",
			},
			"",
			true,
		},
	}

	server := httptest.NewServer(http.HandlerFunc(func(rw http.ResponseWriter, req *http.Request) {
		// Send response to be tested
		if req.URL.Path == "/org/repo/releases/latest" {
			rw.Header().Add("Location", "https://github.com/org/repo/releases/tag/v1.2.3")
			rw.WriteHeader(302)
		} else if req.URL.Path == "/redirect/repo/releases/latest" {
			// permanent Redirect to the happy path
			rw.Header().Add("Location", "/org/repo/releases/latest")
			rw.WriteHeader(301)
		} else if req.URL.Path == "/no/location/releases/latest" {
			// No location header in the response
			rw.WriteHeader(302)
		} else if req.URL.Path == "/bad/location/releases/latest" {
			rw.Header().Add("Location", "https://foo.bar")
			rw.WriteHeader(302)
		} else {
			_, _ = rw.Write([]byte(`This is not the redirect you are looking for`))
		}
	}))
	// Close the server when test finishes
	defer server.Close()

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := getLatestReleaseFromHostUsingHttpRedirect(server.URL, tt.args.githubOwner, tt.args.githubRepo)
			if (err != nil) != tt.wantErr {
				t.Errorf("getLatestReleaseFromGithubUsingHttpRedirect() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("getLatestReleaseFromGithubUsingHttpRedirect() = %v, want %v", got, tt.want)
			}
		})
	}
}

// TestGetLatestReleaseFromGitHubURL tests the util.GetLatestReleaseFromGitHubURL() function
func TestGetLatestReleaseFromGitHubURL(t *testing.T) {
	gitURL := "https://github.com/jenkins-x/jenkins-x-versions.git"
	version, err := GetLatestReleaseFromGitHubURL(gitURL)
	require.NoError(t, err, "failed to call GetLatestReleaseFromGitHubURL with %s", gitURL)
	t.Logf("found version %s for git URL %s\n", version, gitURL)
	assert.True(t, version != "", "failed to find a value for GetLatestReleaseFromGitHubURL with %s", gitURL)
}

// Test_UnTargzAll_NestedNoDir tests that util.UnTargzAll is able to extract
// tgz archives where all folders have been removed (as generated by helm)
func Test_UnTargzAll_NestedNoDir(t *testing.T) {
	dir, err := ioutil.TempDir("", "dowload_test")
	require.NoError(t, err)
	defer os.RemoveAll(dir)

	UnTargzAll("test_data/nested-no-dir.tgz", dir)

	type file struct {
		path    string
		content string
	}
	expected := []file{
		{"dir1/file2.txt", "file2\n"},
		{"dir1/file3.txt", "file3\n"},
		{"dir2/dir3/file4.txt", "file4\n"},
		{"dir2/dir4/file5.txt", "file5\n"},
		{"file1.txt", "file1\n"},
	}
	found := []file{}

	require.NoError(t, filepath.Walk(dir, func(path string, info os.FileInfo, err error) error {
		if !assert.NoError(t, err) {
			return err
		}
		if info.IsDir() {
			return nil
		}
		name := strings.TrimPrefix(path, dir+"/")
		assert.True(t, info.Mode().IsRegular(), "Non regular file %s", name)
		content, err := ioutil.ReadFile(path)
		if !assert.NoError(t, err, "Error reading file %s", name) {
			return err
		}
		found = append(found, file{name, string(content)})
		return nil
	}))
	require.Equal(t, expected, found, "wrong files extracted")
}
