/**
 * Copyright (c) F5, Inc.
 *
 * This source code is licensed under the Apache License, Version 2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

package checksum

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestChecksum(t *testing.T) {
	tests := []struct {
		input    []byte
		expected string
	}{
		{
			input:    []byte("test1"),
			expected: "\x1bO\x0e\x98Q\x97\x19\x98\xe72\a\x85D\xc9k6\xc3\xd0\x1c\xed\xf7ʣ25\x9do\x1d\x83Vp\x14",
		},
		{
			input:    []byte(""),
			expected: "\xe3\xb0\xc4B\x98\xfc\x1c\x14\x9a\xfb\xf4șo\xb9$'\xaeA\xe4d\x9b\x93L\xa4\x95\x99\x1bxR\xb8U",
		},
	}

	for _, test := range tests {
		result := Checksum(test.input)
		assert.Equal(t, test.expected, string(result))
	}
}

func TestHexChecksum(t *testing.T) {
	tests := []struct {
		input    []byte
		expected string
	}{
		{
			input:    []byte{31, 139, 8, 0, 0, 0, 0, 0, 0, 255, 236, 152, 219, 110, 227, 56, 15, 128, 231, 218, 79, 161, 251, 31, 118, 220, 52, 105, 231, 223, 62, 204, 64, 145, 105, 91, 137, 78, 35, 42, 177, 182, 131, 190, 251, 194, 135, 36, 118, 34, 199, 154, 238, 204, 2, 11, 44, 129, 34, 137, 205, 79, 164, 40, 137, 34, 171, 42, 174, 124, 198, 180, 42, 191, 252, 54, 201, 243, 60, 127, 217, 108, 186, 207, 60, 207, 111, 63, 243, 151, 237, 203, 229, 123, 247, 252, 233, 105, 157, 191, 124, 33, 249, 239, 115, 233, 42, 71, 116, 212, 126, 201, 255, 182, 173, 219, 201, 253, 75, 164, 209, 246, 0, 246, 155, 177, 154, 1, 34, 32, 33, 235, 183, 36, 1, 107, 181, 253, 38, 116, 69, 206, 34, 116, 133, 171, 238, 113, 38, 116, 213, 170, 156, 64, 57, 36, 63, 62, 146, 164, 118, 206, 144, 31, 73, 171, 134, 96, 79, 96, 191, 57, 125, 0, 133, 68, 151, 229, 91, 247, 152, 43, 38, 142, 5, 12, 99, 73, 46, 33, 115, 127, 26, 192, 254, 45, 171, 169, 69, 112, 195, 219, 163, 43, 211, 175, 111, 73, 247, 134, 178, 214, 171, 179, 35, 157, 15, 253, 163, 172, 123, 196, 180, 220, 113, 5, 197, 160, 221, 27, 31, 60, 25, 121, 163, 168, 132, 14, 103, 84, 212, 26, 221, 219, 69, 65, 112, 116, 160, 206, 191, 158, 214, 175, 89, 158, 229, 217, 211, 31, 207, 235, 117, 27, 134, 228, 162, 216, 7, 196, 208, 170, 155, 195, 54, 207, 201, 54, 95, 147, 109, 254, 76, 182, 249, 134, 144, 85, 13, 66, 232, 172, 118, 82, 188, 93, 161, 214, 160, 227, 122, 24, 127, 53, 114, 172, 21, 171, 245, 48, 229, 213, 17, 237, 170, 243, 110, 5, 142, 173, 186, 35, 121, 245, 241, 163, 31, 240, 35, 73, 62, 146, 95, 191, 254, 215, 181, 248, 245, 99, 159, 229, 241, 249, 127, 90, 63, 109, 95, 239, 206, 255, 115, 254, 223, 249, 255, 39, 36, 233, 214, 126, 216, 154, 14, 188, 91, 181, 155, 152, 68, 74, 167, 91, 59, 73, 176, 223, 250, 151, 65, 24, 98, 236, 24, 132, 33, 142, 72, 31, 111, 157, 248, 179, 77, 46, 105, 5, 171, 138, 151, 177, 100, 197, 203, 49, 185, 55, 80, 69, 146, 157, 234, 222, 84, 61, 78, 141, 17, 188, 63, 229, 171, 61, 61, 81, 100, 150, 27, 247, 8, 199, 123, 144, 58, 45, 255, 183, 52, 239, 86, 233, 30, 181, 136, 139, 36, 177, 109, 128, 175, 17, 150, 212, 213, 177, 65, 150, 147, 85, 53, 130, 114, 21, 7, 18, 231, 221, 136, 60, 169, 34, 195, 163, 202, 246, 107, 9, 25, 53, 38, 45, 160, 15, 149, 182, 119, 228, 158, 22, 55, 100, 67, 77, 214, 68, 184, 220, 76, 188, 245, 41, 211, 210, 104, 5, 234, 209, 130, 244, 82, 59, 54, 132, 168, 223, 16, 70, 197, 238, 7, 98, 84, 53, 222, 74, 120, 170, 150, 87, 164, 23, 60, 85, 237, 223, 251, 24, 119, 188, 140, 221, 196, 142, 151, 237, 223, 100, 35, 95, 162, 181, 147, 102, 1, 111, 85, 198, 104, 3, 187, 37, 228, 130, 194, 110, 130, 250, 148, 51, 29, 183, 49, 56, 211, 83, 114, 31, 27, 233, 253, 52, 206, 62, 149, 152, 46, 207, 178, 149, 110, 162, 29, 90, 106, 229, 86, 141, 142, 14, 49, 105, 46, 37, 204, 5, 93, 255, 4, 186, 62, 215, 49, 55, 105, 34, 165, 150, 213, 252, 4, 179, 240, 158, 90, 210, 80, 75, 128, 218, 64, 162, 193, 136, 96, 183, 74, 247, 168, 164, 44, 221, 113, 85, 131, 223, 228, 179, 104, 253, 221, 7, 72, 108, 180, 45, 22, 140, 22, 154, 221, 147, 166, 136, 8, 183, 41, 202, 0, 169, 209, 45, 102, 84, 131, 4, 12, 18, 202, 3, 217, 209, 69, 88, 182, 46, 96, 185, 61, 71, 237, 111, 200, 164, 129, 234, 104, 131, 231, 89, 62, 31, 191, 134, 209, 74, 235, 74, 64, 10, 212, 186, 58, 59, 72, 113, 151, 17, 14, 231, 76, 181, 64, 190, 223, 25, 61, 200, 247, 48, 41, 49, 5, 207, 96, 62, 243, 120, 17, 184, 123, 6, 178, 221, 221, 122, 183, 7, 22, 140, 52, 104, 55, 75, 26, 221, 128, 53, 154, 135, 147, 172, 49, 51, 164, 166, 200, 49, 211, 6, 84, 161, 217, 81, 130, 114, 89, 101, 169, 169, 57, 27, 10, 7, 93, 4, 174, 216, 25, 210, 88, 64, 80, 110, 40, 182, 117, 97, 162, 73, 52, 22, 104, 129, 53, 116, 237, 135, 46, 102, 34, 20, 32, 219, 43, 230, 58, 79, 93, 204, 205, 211, 128, 242, 82, 148, 218, 74, 234, 48, 213, 101, 201, 25, 4, 29, 151, 98, 242, 51, 153, 93, 199, 71, 98, 140, 11, 156, 221, 101, 71, 70, 113, 144, 34, 235, 62, 63, 231, 128, 23, 248, 41, 7, 218, 228, 50, 52, 161, 92, 85, 82, 100, 231, 55, 159, 115, 163, 208, 108, 198, 141, 161, 150, 96, 179, 104, 251, 242, 30, 245, 233, 235, 123, 87, 79, 216, 182, 77, 158, 201, 131, 175, 129, 163, 217, 86, 33, 76, 211, 199, 238, 178, 243, 173, 56, 37, 199, 87, 69, 90, 4, 43, 131, 61, 181, 197, 165, 6, 8, 208, 123, 37, 76, 90, 114, 17, 190, 106, 218, 183, 33, 84, 210, 3, 32, 136, 71, 217, 211, 30, 3, 215, 139, 79, 13, 132, 115, 229, 72, 140, 32, 38, 80, 202, 250, 212, 112, 161, 23, 74, 53, 99, 25, 49, 197, 46, 68, 91, 106, 151, 22, 200, 134, 110, 83, 159, 90, 40, 106, 234, 82, 67, 217, 129, 86, 144, 74, 170, 104, 5, 211, 186, 212, 134, 61, 70, 88, 88, 88, 66, 16, 104, 144, 172, 53, 59, 52, 244, 4, 105, 41, 40, 214, 65, 178, 9, 46, 43, 186, 99, 89, 242, 135, 113, 66, 30, 200, 70, 62, 117, 108, 177, 44, 109, 85, 220, 33, 4, 251, 109, 254, 255, 148, 209, 148, 129, 157, 179, 93, 128, 37, 6, 36, 97, 54, 104, 222, 27, 174, 208, 81, 241, 192, 9, 111, 2, 183, 184, 111, 123, 203, 165, 154, 218, 95, 27, 208, 9, 139, 166, 92, 44, 199, 91, 165, 123, 244, 157, 71, 212, 150, 239, 220, 4, 74, 60, 205, 28, 184, 20, 157, 5, 42, 103, 209, 29, 87, 4, 60, 144, 66, 4, 252, 142, 26, 161, 128, 192, 65, 136, 35, 101, 224, 114, 141, 34, 57, 106, 194, 63, 77, 75, 228, 68, 162, 33, 18, 229, 57, 108, 199, 130, 235, 149, 228, 5, 159, 133, 110, 134, 224, 5, 233, 212, 15, 151, 195, 220, 15, 17, 223, 194, 75, 243, 60, 38, 117, 21, 221, 235, 233, 170, 26, 147, 62, 149, 155, 197, 4, 48, 216, 220, 208, 41, 105, 129, 138, 238, 251, 18, 105, 233, 16, 170, 19, 47, 64, 175, 158, 43, 19, 219, 166, 117, 170, 207, 213, 144, 225, 123, 92, 154, 245, 114, 55, 220, 139, 195, 41, 184, 137, 228, 136, 52, 155, 41, 249, 19, 43, 3, 21, 145, 231, 127, 174, 244, 248, 247, 35, 103, 7, 199, 229, 124, 179, 116, 197, 245, 105, 76, 54, 176, 155, 223, 138, 83, 105, 85, 199, 168, 79, 75, 113, 138, 67, 75, 113, 154, 146, 114, 19, 73, 202, 205, 45, 25, 219, 11, 75, 85, 221, 144, 152, 82, 140, 233, 106, 41, 122, 66, 177, 188, 163, 27, 25, 227, 116, 35, 111, 61, 198, 238, 91, 132, 221, 19, 127, 75, 62, 146, 191, 2, 0, 0, 255, 255, 84, 109, 42, 14, 111, 26, 0, 0},
			expected: "92da41bcbab6b67d792531b948d08bbe4922ee4543ac0296aabf9d11b54cc148",
		},
		{
			input:    []byte{},
			expected: "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855",
		},
	}

	for _, test := range tests {
		result := HexChecksum(test.input)
		assert.Equal(t, test.expected, string(result))
	}
}

func TestHexChunk(t *testing.T) {
	tests := []struct {
		input    []byte
		limit    int
		expected [][]byte
	}{
		{
			input:    []byte{1, 2, 3, 4},
			limit:    1,
			expected: [][]byte{{1}, {2}, {3}, {4}},
		},
		{
			input:    []byte{1, 2, 3, 4},
			limit:    2,
			expected: [][]byte{{1, 2}, {3, 4}},
		},
		{
			input:    []byte{1, 2, 3, 4},
			limit:    3,
			expected: [][]byte{{1, 2, 3}, {4}},
		},
		{
			input:    []byte{1, 2, 3, 4},
			limit:    10,
			expected: [][]byte{{1, 2, 3, 4}},
		},
		{
			input:    []byte{},
			limit:    10,
			expected: [][]byte{},
		},
	}
	for _, test := range tests {
		result := Chunk(test.input, test.limit)
		assert.Equal(t, test.expected, result)
	}
}
