/// Macro to duplicate code a per-storage type basis. The given code is duplicated in new modules
/// named for each storage type. A type alias, `V`, is generated that code can use for the type.
/// `@...` match arms are considered private.
///
/// * `$attr`: Module attributes. Generally used to set documentation comments for storage type
///   modules generated by the macro.
/// * `$T`: Types to generate a module for. Accepts all underlying storage types along with a number
///   of different categories:
///   * `All`: `usize`, `u8`, `u16`, `u32`, `u64`, `isize`, `i8`, `i16`, `i32`, `i64`, `BigInt`,
///     `BigUint`, `Rational`, `Rational32`, `Rational64`, `BigRational`, `f32`, and `f64`.
///   * `PrimInt`: `usize`, `u8`, `u16`, `u32`, `u64`, `isize`, `i8`, `i16`, `i32`, `i64`.
///   * `Ratio`: `Rational`, `Rational32`, `Rational64`, and `BigRational`.
///   * `Float`: `f32` and `f64`.
///   * `Signed`: `isize`, `i8`, `i16`, `i32`, `i64`, `BigInt`, `Rational`, `Rational32`,
///     `Rational64`, `BigRational`, `f32`, and `f64`.
///   * `Unsigned`: `usize`, `u8`, `u16`, `u32`, `u64`, and `BigUint`.
/// * `$tt`: Code to place into each storage type module.
///
#[cfg_attr(all(feature = "f32", feature = "f64"), doc = " ```rust")]
#[cfg_attr(not(all(feature = "f32", feature = "f64")), doc = " ```rust,ignore")]
/// #[macro_use]
/// extern crate uom;
///
/// fn main() {
///     f32::do_work(1.234_f32);
///     f64::do_work(1.234_f64);
/// }
///
/// storage_types! {
///     /// Type modules.
///     pub types: Float;
///
///     pub fn do_work(_v: V) {}
/// }
/// ```
#[macro_export]
macro_rules! storage_types {
    ($(#[$attr:meta])* types: $($T:ident),+; $($tt:tt)*) => {
        storage_types!(@types ($(#[$attr])*) @mod $($T),+; ($($tt)*));
    };
    ($(#[$attr:meta])* pub types: $($T:ident),+; $($tt:tt)*) => {
        storage_types!(@types ($(#[$attr])*) @pub_mod $($T),+; ($($tt)*));
    };
    (@types $attr:tt @$M:ident $($T:ident),+; $tt:tt) => {
        $(storage_types!(@type $attr @$M $T $tt);)+
    };
    (@type ($(#[$attr:meta])*) @$M:ident usize ($($tt:tt)*)) => {
        #[cfg(feature = "usize")]
        storage_types!(@$M $(#[$attr])* usize, usize; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident u8 ($($tt:tt)*)) => {
        #[cfg(feature = "u8")]
        storage_types!(@$M $(#[$attr])* u8, u8; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident u16 ($($tt:tt)*)) => {
        #[cfg(feature = "u16")]
        storage_types!(@$M $(#[$attr])* u16, u16; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident u32 ($($tt:tt)*)) => {
        #[cfg(feature = "u32")]
        storage_types!(@$M $(#[$attr])* u32, u32; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident u64 ($($tt:tt)*)) => {
        #[cfg(feature = "u64")]
        storage_types!(@$M $(#[$attr])* u64, u64; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident isize ($($tt:tt)*)) => {
        #[cfg(feature = "isize")]
        storage_types!(@$M $(#[$attr])* isize, isize; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident i8 ($($tt:tt)*)) => {
        #[cfg(feature = "i8")]
        storage_types!(@$M $(#[$attr])* i8, i8; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident i16 ($($tt:tt)*)) => {
        #[cfg(feature = "i16")]
        storage_types!(@$M $(#[$attr])* i16, i16; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident i32 ($($tt:tt)*)) => {
        #[cfg(feature = "i32")]
        storage_types!(@$M $(#[$attr])* i32, i32; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident i64 ($($tt:tt)*)) => {
        #[cfg(feature = "i64")]
        storage_types!(@$M $(#[$attr])* i64, i64; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident BigInt ($($tt:tt)*)) => {
        #[cfg(feature = "bigint")]
        storage_types!(@$M $(#[$attr])* bigint, $crate::num::BigInt; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident BigUint ($($tt:tt)*)) => {
        #[cfg(feature = "biguint")]
        storage_types!(@$M $(#[$attr])* biguint, $crate::num::BigUint; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident Rational ($($tt:tt)*)) => {
        #[cfg(feature = "rational")]
        storage_types!(@$M $(#[$attr])* rational, $crate::num::Rational; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident Rational32 ($($tt:tt)*)) => {
        #[cfg(feature = "rational32")]
        storage_types!(@$M $(#[$attr])* rational32, $crate::num::rational::Rational32; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident Rational64 ($($tt:tt)*)) => {
        #[cfg(feature = "rational64")]
        storage_types!(@$M $(#[$attr])* rational64, $crate::num::rational::Rational64; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident BigRational ($($tt:tt)*)) => {
        #[cfg(feature = "bigrational")]
        storage_types!(@$M $(#[$attr])* bigrational, $crate::num::BigRational; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident f32 ($($tt:tt)*)) => {
        #[cfg(feature = "f32")]
        storage_types!(@$M $(#[$attr])* f32, f32; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident f64 ($($tt:tt)*)) => {
        #[cfg(feature = "f64")]
        storage_types!(@$M $(#[$attr])* f64, f64; $($tt)*);
    };
    (@type ($(#[$attr:meta])*) @$M:ident All ($($tt:tt)*)) => {
        storage_types!(@type ($(#[$attr])*) @$M usize ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u8 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u16 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M isize ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i8 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i16 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigInt ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigUint ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigRational ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M f32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M f64 ($($tt)*));
    };
    (@type ($(#[$attr:meta])*) @$M:ident PrimInt ($($tt:tt)*)) => {
        storage_types!(@type ($(#[$attr])*) @$M usize ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u8 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u16 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M isize ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i8 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i16 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i64 ($($tt)*));
    };
    (@type ($(#[$attr:meta])*) @$M:ident Ratio ($($tt:tt)*)) => {
        storage_types!(@type ($(#[$attr])*) @$M Rational ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigRational ($($tt)*));
    };
    (@type ($(#[$attr:meta])*) @$M:ident Float ($($tt:tt)*)) => {
        storage_types!(@type ($(#[$attr])*) @$M f32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M f64 ($($tt)*));
    };
    (@type ($(#[$attr:meta])*) @$M:ident Signed ($($tt:tt)*)) => {
        storage_types!(@type ($(#[$attr])*) @$M isize ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i8 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i16 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M i64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigInt ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M Rational64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigRational ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M f32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M f64 ($($tt)*));
    };
    (@type ($(#[$attr:meta])*) @$M:ident Unsigned ($($tt:tt)*)) => {
        storage_types!(@type ($(#[$attr])*) @$M usize ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u8 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u16 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u32 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M u64 ($($tt)*));
        storage_types!(@type ($(#[$attr])*) @$M BigUint ($($tt)*));
    };
    (@mod $(#[$attr:meta])* $M:ident, $V:ty; $($tt:tt)*) => {
        $(#[$attr])*
        mod $M {
            #[allow(dead_code)]
            type V = $V;

            $($tt)*
        }
    };
    (@pub_mod $(#[$attr:meta])* $M:ident, $V:ty; $($tt:tt)*) => {
        $(#[$attr])*
        pub mod $M {
            #[allow(dead_code)]
            type V = $V;

            $($tt)*
        }
    };
    ($($tt:tt)*) => {
        storage_types! {
            types: All;

            $($tt)*
        }
    };
}
