/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License") +  you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.openmeetings.web.admin.ldaps;

import static org.apache.openmeetings.web.app.Application.getBean;

import org.apache.openmeetings.db.dao.server.LdapConfigDao;
import org.apache.openmeetings.db.entity.server.LdapConfig;
import org.apache.openmeetings.web.admin.AdminBaseForm;
import org.apache.openmeetings.web.app.WebSession;
import org.apache.openmeetings.web.util.DateLabel;
import org.apache.wicket.ajax.AjaxRequestTarget;
import org.apache.wicket.ajax.form.AjaxFormValidatingBehavior;
import org.apache.wicket.markup.html.WebMarkupContainer;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.form.CheckBox;
import org.apache.wicket.markup.html.form.Form;
import org.apache.wicket.markup.html.form.RequiredTextField;
import org.apache.wicket.markup.html.form.TextArea;
import org.apache.wicket.markup.html.form.TextField;
import org.apache.wicket.model.CompoundPropertyModel;
import org.apache.wicket.model.Model;
import org.apache.wicket.util.time.Duration;

/**
 * Form components to insert/update/delete {@link LdapConfig}
 *
 * @author swagner
 *
 */
public class LdapForm extends AdminBaseForm<LdapConfig> {
	private static final long serialVersionUID = 1L;
	private final WebMarkupContainer listContainer;

	public LdapForm(String id, WebMarkupContainer listContainer, final LdapConfig ldapConfig) {
		super(id, new CompoundPropertyModel<>(ldapConfig));
		setOutputMarkupId(true);
		this.listContainer = listContainer;

		add(new CheckBox("active"));
		add(new DateLabel("inserted"));
		add(new Label("insertedby.login"));
		add(new DateLabel("updated"));
		add(new Label("updatedby.login"));
		add(new CheckBox("addDomainToUserName"));
		add(new TextField<String>("domain"));
		add(new TextArea<String>("comment"));

		// attach an ajax validation behavior to all form component's keydown
		// event and throttle it down to once per second
		add(new AjaxFormValidatingBehavior("keydown", Duration.ONE_SECOND));
	}

	@Override
	protected void onInitialize() {
		add(new RequiredTextField<String>("name").setLabel(Model.of(getString("165"))));
		add(new RequiredTextField<String>("configFileName").setLabel(Model.of(getString("1115"))));
		super.onInitialize();
	}

	@Override
	protected void onSaveSubmit(AjaxRequestTarget target, Form<?> form) {
		setModelObject(getBean(LdapConfigDao.class).update(getModelObject(), WebSession.getUserId()));
		hideNewRecord();
		target.add(this, listContainer);
		reinitJs(target);
	}

	@Override
	protected void onNewSubmit(AjaxRequestTarget target, Form<?> form) {
		this.setModelObject(new LdapConfig());
		target.add(this);
		reinitJs(target);
	}

	@Override
	protected void onRefreshSubmit(AjaxRequestTarget target, Form<?> form) {
		LdapConfig ldapConfig = this.getModelObject();
		if (ldapConfig.getId() != null) {
			ldapConfig = getBean(LdapConfigDao.class).get(ldapConfig.getId());
		} else {
			ldapConfig = new LdapConfig();
		}
		this.setModelObject(ldapConfig);
		target.add(this);
		reinitJs(target);
	}

	@Override
	protected void onDeleteSubmit(AjaxRequestTarget target, Form<?> form) {
		getBean(LdapConfigDao.class).delete(getModelObject(), WebSession.getUserId());
		this.setModelObject(new LdapConfig());
		target.add(listContainer);
		target.add(this);
		reinitJs(target);
	}
}
