/**
 * GitDocumentDB
 * Copyright (c) Hidekazu Kubota
 *
 * This source code is licensed under the Mozilla Public License Version 2.0
 * found in the LICENSE file in the root directory of this source tree.
 */

import path from 'path';
import nodegit from '@sosuisen/nodegit';
import fs from 'fs-extra';
import rimraf from 'rimraf';
import { Logger, TLogLevelName } from 'tslog';
import {
  CannotCreateDirectoryError,
  CannotOpenRepositoryError,
  DatabaseCloseTimeoutError,
  DatabaseClosingError,
  DatabaseExistsError,
  FileRemoveTimeoutError,
  InvalidWorkingDirectoryPathLengthError,
  RemoteAlreadyRegisteredError,
  RepositoryNotFoundError,
  UndefinedDatabaseNameError,
  WorkingDirectoryExistsError,
} from './error';
import { Collection } from './collection';
import { Validator } from './validator';
import {
  AllDocsOptions,
  AllDocsResult,
  CollectionPath,
  DatabaseCloseOption,
  DatabaseInfo,
  DatabaseInfoSuccess,
  DatabaseOption,
  DeleteOptions,
  JsonDoc,
  JsonDocWithMetadata,
  PutOptions,
  PutResult,
  RemoteOptions,
  RemoveResult,
  Schema,
} from './types';
import { CRUDInterface, IDocumentDB } from './types_gitddb';
import { put_worker, putImpl } from './crud/put';
import { getByRevisionImpl, getImpl } from './crud/get';
import { deleteImpl } from './crud/delete';
import { allDocsImpl } from './crud/allDocs';
import { Sync, syncImpl } from './remote/sync';
import { TaskQueue } from './task_queue';
import { FILE_REMOVE_TIMEOUT } from './const';
import { cloneRepository } from './remote/clone';
import { getDocHistoryImpl } from './crud/history';

const defaultLogLevel = 'info';

export const DATABASE_NAME = 'GitDocumentDB';
export const DATABASE_VERSION = '1.0';
export const GIT_DOCUMENTDB_VERSION = `${DATABASE_NAME}: ${DATABASE_VERSION}`;
export const GIT_DOCUMENTDB_VERSION_FILENAME = '.gitddb/lib_version';

interface RepositoryInitOptions {
  description?: string;
  initialHead?: string;
  flags?: number; // https://libgit2.org/libgit2/#HEAD/type/git_repository_init_flag_t
  mode?: number; // https://libgit2.org/libgit2/#HEAD/type/git_repository_init_mode_t
  originUrl?: string;
  templatePath?: string;
  version?: number;
  workdirPath?: string;
}
/*
 const repositoryInitOptionFlags = {
   GIT_REPOSITORY_INIT_BARE: 1,
   GIT_REPOSITORY_INIT_NO_REINIT: 2,
   GIT_REPOSITORY_INIT_NO_DOTGIT_DIR: 4,
   GIT_REPOSITORY_INIT_MKDIR: 8,
   GIT_REPOSITORY_INIT_MKPATH: 16,
   GIT_REPOSITORY_INIT_EXTERNAL_TEMPLATE: 32,
   GIT_REPOSITORY_INIT_RELATIVE_GITLINK: 64,
 };
 */

const defaultLocalDir = './git-documentdb';

/**
 * Main class of GitDocumentDB
 */
export class GitDocumentDB implements IDocumentDB, CRUDInterface {
  /**
   * File extension of a repository document
   */
  readonly fileExt = '.json';
  /**
   * Author name and email
   */
  readonly gitAuthor = {
    name: 'GitDocumentDB',
    email: 'gitddb@example.com',
  } as const;

  readonly defaultBranch = 'main';

  private _firstCommitMessage = 'first commit';

  private _localDir: string;
  private _dbName: string;

  private _currentRepository: nodegit.Repository | undefined;
  private _workingDirectory: string;

  private _synchronizers: { [url: string]: Sync } = {};

  private _dbInfo: DatabaseInfo = {
    ok: true,
    is_new: false,
    is_clone: false,
    is_created_by_gitddb: true,
    is_valid_version: true,
  };

  private _logLevel: TLogLevelName;

  /**
   * Schema
   */
  schema: Schema;

  /**
   * Task queue
   */
  taskQueue: TaskQueue;

  /**
   * Name validator
   */
  validator: Validator;

  /**
   * DB is going to close
   */
  isClosing = false;

  /**
   * Logger
   */
  private _logger!: Logger; // Use definite assignment assertion

  getLogger (): Logger {
    return this._logger;
  }

  setLogLevel (level: TLogLevelName) {
    this._logger = new Logger({
      name: this._dbName,
      minLevel: level as TLogLevelName,
      displayDateTime: false,
      displayFunctionName: false,
      displayFilePath: 'hidden',
    });
    if (this.taskQueue) this.taskQueue.setLogger(this._logger);
  }

  /**
   * Constructor
   *
   * @remarks
   * - The git working directory will be localDir/dbName.
   *
   * @throws {@link InvalidWorkingDirectoryPathLengthError}
   * @throws {@link UndefinedDatabaseNameError}
   *
   */
  constructor (options: DatabaseOption) {
    if (options.db_name === undefined || options.db_name === '') {
      throw new UndefinedDatabaseNameError();
    }

    this._dbName = options.db_name;
    this._localDir = options.local_dir ?? defaultLocalDir;
    this._logLevel = options.log_level ?? defaultLogLevel;

    this.schema = options.schema ?? {
      json: {
        idOfSubtree: undefined,
        plainTextProperties: undefined,
      },
    };

    // Get full-path
    this._workingDirectory = path.resolve(this._localDir, this._dbName);

    this.validator = new Validator(this._workingDirectory);

    this.validator.validateDbName(this._dbName);
    this.validator.validateLocalDir(this._localDir);

    if (
      this._workingDirectory.length === 0 ||
      Validator.byteLengthOf(this._workingDirectory) > Validator.maxWorkingDirectoryLength()
    ) {
      throw new InvalidWorkingDirectoryPathLengthError(
        this._workingDirectory,
        0,
        Validator.maxWorkingDirectoryLength()
      );
    }
    this.setLogLevel(this._logLevel);
    this.taskQueue = new TaskQueue(this.getLogger());
  }

  /**
   * Create and open a repository
   *
   * @remarks
   *  - If localDir does not exist, create it.
   *
   *  - createDB() also opens the repository. createDB() followed by open() has no effect.
   *
   * @returns Database information
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link DatabaseExistsError}
   * @throws {@link WorkingDirectoryExistsError}
   * @throws {@link CannotCreateDirectoryError}
   * @throws {@link CannotConnectError}
   *
   */
  async createDB (remoteOptions?: RemoteOptions): Promise<DatabaseInfo> {
    if (this.isClosing) {
      throw new DatabaseClosingError();
    }
    if (this.isOpened()) {
      throw new DatabaseExistsError();
    }

    if (fs.existsSync(this._workingDirectory)) {
      throw new WorkingDirectoryExistsError();
    }

    /**
     * Create directory
     */
    await fs.ensureDir(this._workingDirectory).catch((err: Error) => {
      throw new CannotCreateDirectoryError(err.message);
    });

    if (remoteOptions?.remote_url === undefined) {
      this._dbInfo = await this._createRepository();
      return this._dbInfo;
    }

    // Clone repository if remoteURL exists
    this._currentRepository = await cloneRepository(
      this.workingDir(),
      remoteOptions,
      this.getLogger()
    ).catch((err: Error) => {
      throw err;
    });

    if (this._currentRepository === undefined) {
      // Clone failed. Try to create remote repository in sync().
      // Please check is_clone flag if you would like to know whether clone is succeeded or not.
      this._dbInfo = await this._createRepository();
    }
    else {
      // this.logger.warn('Clone succeeded.');
      /**
       * TODO: validate db
       */
      (this._dbInfo as DatabaseInfoSuccess).is_clone = true;
    }

    /**
     * Check and sync repository if exists
     */

    await this._setDbInfo();

    if (remoteOptions?.remote_url !== undefined) {
      if (
        (this._dbInfo as DatabaseInfoSuccess).is_created_by_gitddb &&
        (this._dbInfo as DatabaseInfoSuccess).is_valid_version
      ) {
        // Can synchronize
        /**
         * TODO:
         * Handle combine_db_strategy in sync()
         */
        await this.sync(remoteOptions);
      }
    }

    return this._dbInfo;
  }

  /**
   * Open an existing repository
   *
   * @remarks
   *  - GitDocumentDB can load a git repository that is not created by the git-documentdb module.
   *  However, correct behavior is not guaranteed.
   *
   * @returns Database information
   *
   */
  async open (): Promise<DatabaseInfo> {
    const dbInfoError = (err: Error) => {
      this._dbInfo = {
        ok: false,
        error: err,
      };
      return this._dbInfo;
    };

    if (this.isClosing) {
      return dbInfoError(new DatabaseClosingError());
    }
    if (this.isOpened()) {
      (this._dbInfo as DatabaseInfoSuccess).is_new = false;
      return this._dbInfo;
    }

    /**
     * Reset
     */
    this._synchronizers = {};
    this._dbInfo = {
      ok: true,
      is_new: false,
      is_clone: false,
      is_created_by_gitddb: true,
      is_valid_version: true,
    };
    this.taskQueue.clear();

    /**
     * nodegit.Repository.open() throws an error if the specified repository does not exist.
     * open() also throws an error if the path is invalid or not writable,
     */
    try {
      this._currentRepository = await nodegit.Repository.open(this._workingDirectory);
    } catch (err) {
      const gitDir = this._workingDirectory + '/.git/';
      if (!fs.existsSync(gitDir)) {
        return dbInfoError(new RepositoryNotFoundError(gitDir));
      }
      return dbInfoError(new CannotOpenRepositoryError(err));
    }

    await this._setDbInfo();

    return this._dbInfo;
  }

  private async _createRepository () {
    /**
     * Create a repository followed by first commit
     */
    const options: RepositoryInitOptions = {
      initialHead: this.defaultBranch,
    };
    (this._dbInfo as DatabaseInfoSuccess).is_new = true;
    this._currentRepository = await nodegit.Repository.initExt(
      this._workingDirectory,
      // @ts-ignore
      options
    ).catch(err => {
      return Promise.reject(err);
    });

    // First commit
    await put_worker(
      this,
      GIT_DOCUMENTDB_VERSION_FILENAME,
      '',
      GIT_DOCUMENTDB_VERSION,
      this._firstCommitMessage
    );
    return this._dbInfo;
  }

  private async _setDbInfo () {
    const version = await fs
      .readFile(
        path.resolve(this._workingDirectory, GIT_DOCUMENTDB_VERSION_FILENAME),
        'utf8'
      )
      .catch(() => {
        (this._dbInfo as DatabaseInfoSuccess).is_created_by_gitddb = false;
        (this._dbInfo as DatabaseInfoSuccess).is_valid_version = false;
        return undefined;
      });
    if (version === undefined) return this._dbInfo;

    if (new RegExp('^' + DATABASE_NAME).test(version)) {
      (this._dbInfo as DatabaseInfoSuccess).is_created_by_gitddb = true;
      if (new RegExp('^' + GIT_DOCUMENTDB_VERSION).test(version)) {
        (this._dbInfo as DatabaseInfoSuccess).is_valid_version = true;
      }
      else {
        (this._dbInfo as DatabaseInfoSuccess).is_valid_version = false;
        /**
         * TODO: Need migration
         */
      }
    }
    else {
      (this._dbInfo as DatabaseInfoSuccess).is_created_by_gitddb = false;
      (this._dbInfo as DatabaseInfoSuccess).is_valid_version = false;
    }
  }

  /**
   * Get dbName
   *
   */
  dbName () {
    return this._dbName;
  }

  /**
   * Get a full path of the current Git working directory
   *
   * @returns Full path of the directory (trailing slash is omitted)
   *
   */
  workingDir () {
    return this._workingDirectory;
  }

  /**
   * Get a current repository
   * @remarks Be aware that direct operation of the current repository can corrupt the database.
   *
   */
  repository (): nodegit.Repository | undefined {
    return this._currentRepository;
  }

  /**
   * Get a collection
   *
   * @remarks
   * - Notice that this function does not make a sub-directory under the working directory.
   *
   * @param collectionPath - path from localDir. Sub-directories are also permitted. e.g. 'pages', 'pages/works'.
   *
   */
  collection (collectionPath: CollectionPath) {
    return new Collection(this, collectionPath);
  }

  /**
   * Get collections
   *
   * @param rootPath Get collections directly under the path.
   * @returns Promise<Collection[]>
   * @throws {@link RepositoryNotOpenError}
   */
  async getCollections (rootPath?: string): Promise<Collection[]> {
    return await Collection.getCollections(this, rootPath);
  }

  /**
   * Test if a database is opened
   *
   */
  isOpened () {
    return this._currentRepository !== undefined;
  }

  /**
   * Close a database
   *
   * @remarks
   * - New CRUD operations are not available while closing.
   *
   * - Queued operations are executed before the database is closed.
   *
   * @param options - The options specify how to close database.
   * @throws {@link DatabaseClosingError}
   * @throws {@link DatabaseCloseTimeoutError}
   *
   */
  async close (options?: DatabaseCloseOption): Promise<void> {
    if (this.isClosing) {
      return Promise.reject(new DatabaseClosingError());
    }
    // Stop remote
    Object.values(this._synchronizers).forEach(_sync => _sync.close());

    options ??= { force: undefined, timeout: undefined };
    options.force ??= false;
    options.timeout ??= 10000;

    // Wait taskQueue
    if (this._currentRepository instanceof nodegit.Repository) {
      try {
        this.isClosing = true;
        if (!options.force) {
          const isTimeout = await this.taskQueue.waitCompletion(options.timeout);
          if (isTimeout) {
            return Promise.reject(new DatabaseCloseTimeoutError());
          }
        }
      } finally {
        this.taskQueue.clear();

        /**
         * The types are wrong. Repository does not have free() method.
         * See https://github.com/nodegit/nodegit/issues/1817#issuecomment-776844425
         * https://github.com/nodegit/nodegit/pull/1570
         *
         * Use cleanup() instead.
         * http://carlosmn.github.io/libgit2/#v0.23.0/group/repository/git_repository__cleanup
         */
        // this._currentRepository.free();

        this._currentRepository.cleanup();
        this._currentRepository = undefined;

        this._synchronizers = {};

        this.isClosing = false;
      }
    }
  }

  /**
   * Destroy a database
   *
   * @remarks
   * - {@link GitDocumentDB.close} is called automatically before destroying.
   *
   * - options.force is true if undefined.
   *
   * - The Git repository and the working directory are removed from the filesystem.
   *
   * - local_dir (which is specified in constructor) is not removed.
   *
   * - destroy() can remove a database that has not been created yet if a working directory exists.
   *
   * @param options - The options specify how to close database.
   * @throws {@link DatabaseClosingError}
   * @throws {@link DatabaseCloseTimeoutError}
   * @throws {@link FileRemoveTimeoutError}
   *
   */
  async destroy (options: DatabaseCloseOption = {}): Promise<{ ok: true }> {
    if (this.isClosing) {
      return Promise.reject(new DatabaseClosingError());
    }

    let closeError: Error | undefined;
    if (this._currentRepository !== undefined) {
      // NOTICE: options.force is true by default.
      options.force = options.force ?? true;
      await this.close(options).catch(err => {
        closeError = err;
      });
    }
    // If the path does not exist, remove() silently does nothing.
    // https://github.com/jprichardson/node-fs-extra/blob/master/docs/remove.md
    //      await fs.remove(this._workingDirectory).catch(err => {

    await new Promise<void>((resolve, reject) => {
      // Set timeout because rimraf sometimes does not catch EPERM error.
      setTimeout(() => {
        reject(new FileRemoveTimeoutError());
      }, FILE_REMOVE_TIMEOUT);
      rimraf(this._workingDirectory, error => {
        if (error) {
          reject(error);
        }
        resolve();
      });
    });

    if (closeError instanceof Error) {
      throw closeError;
    }

    return {
      ok: true,
    };
  }

  /**
   * Insert a document if not exists. Otherwise, update it.
   *
   * @remarks
   * - put() does not check a write permission of your file system (unlike open()).
   *
   * - Saved file path is `${workingDir()}/${document._id}.json`. {@link InvalidIdLengthError} will be thrown if the path length exceeds the maximum length of a filepath on the device.
   *
   * - A put operation is not skipped when no change occurred on a specified document.
   *
   * @param jsonDoc - See {@link JsonDoc} for restriction
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link CannotWriteDataError}
   * @throws {@link CannotCreateDirectoryError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   *
   */
  put (jsonDoc: JsonDoc, options?: PutOptions): Promise<PutResult>;
  /**
    * Insert a document if not exists. Otherwise, update it.
    *
    * @remarks
    * - put() does not check a write permission of your file system (unlike open()).
    *
    * - Saved file path is `${workingDir()}/${document._id}.json`. {@link InvalidIdLengthError} will be thrown if the path length exceeds the maximum length of a filepath on the device.
    
    * - A put operation is not skipped when no change occurred on a specified document.
    *
    * @param id - _id property of a document
    * @param document - This is a {@link JsonDoc}, but _id property is ignored.
    *
    * @throws {@link DatabaseClosingError}
    * @throws {@link RepositoryNotOpenError}
    * @throws {@link UndefinedDocumentIdError}
    * @throws {@link InvalidJsonObjectError}
    * @throws {@link CannotWriteDataError}
    * @throws {@link CannotCreateDirectoryError}
    * @throws {@link InvalidIdCharacterError}
    * @throws {@link InvalidIdLengthError}
    * 
       */
  put (
    id: string,
    document: { [key: string]: any },
    options?: PutOptions
  ): Promise<PutResult>;

  put (
    idOrDoc: string | JsonDoc,
    docOrOptions: { [key: string]: any } | PutOptions,
    options?: PutOptions
  ) {
    return putImpl.call(this, idOrDoc, docOrOptions, options);
  }

  /**
   * Insert a document
   *
   * @remarks
   * - Throws SameIdExistsError when a document which has the same id exists. It might be better to use put() instead of insert().
   *
   * - insert() does not check a write permission of your file system (unlike open()).
   *
   * - Saved file path is `${workingDir()}/${document._id}.json`. {@link InvalidIdLengthError} will be thrown if the path length exceeds the maximum length of a filepath on the device.
   *
   * @param jsonDoc - See {@link JsonDoc} for restriction
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link CannotWriteDataError}
   * @throws {@link CannotCreateDirectoryError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link SameIdExistsError}
   *
   */
  insert (jsonDoc: JsonDoc, options?: PutOptions): Promise<PutResult>;
  /**
   * Insert a document
   *
   * @remarks
   * - Throws SameIdExistsError when a document which has the same id exists. It might be better to use put() instead of insert().
   *
   * - insert() does not check a write permission of your file system (unlike open()).
   *
   * - Saved file path is `${workingDir()}/${document._id}.json`. {@link InvalidIdLengthError} will be thrown if the path length exceeds the maximum length of a filepath on the device.
   *
   * @param id - _id property of a document
   * @param document - This is a {@link JsonDoc}, but _id property is ignored.
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link CannotWriteDataError}
   * @throws {@link CannotCreateDirectoryError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link SameIdExistsError}
   *
   */
  insert (
    id: string,
    document: { [key: string]: any },
    options?: PutOptions
  ): Promise<PutResult>;

  insert (
    idOrDoc: string | JsonDoc,
    docOrOptions: { [key: string]: any } | PutOptions,
    options?: PutOptions
  ) {
    if (typeof idOrDoc === 'object') {
      docOrOptions = { ...docOrOptions, insertOrUpdate: 'insert' };
    }

    return putImpl.call(this, idOrDoc, docOrOptions, {
      ...options,
      insertOrUpdate: 'insert',
    });
  }

  /**
   * Update a document
   *
   * @remarks
   * - Throws DocumentNotFoundError if the document does not exist. It might be better to use put() instead of update().
   *
   * - update() does not check a write permission of your file system (unlike open()).
   *
   * - Saved file path is `${workingDir()}/${document._id}.json`. {@link InvalidIdLengthError} will be thrown if the path length exceeds the maximum length of a filepath on the device.
   *
   * - A update operation is not skipped when no change occurred on a specified document.
   *
   * @param jsonDoc - See {@link JsonDoc} for restriction
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link CannotWriteDataError}
   * @throws {@link CannotCreateDirectoryError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link DocumentNotFoundError}
   *
   */
  update (jsonDoc: JsonDoc, options?: PutOptions): Promise<PutResult>;
  /**
   * Update a document
   *
   * @remarks
   * - Throws DocumentNotFoundError if the document does not exist. It might be better to use put() instead of update().
   *
   * - update() does not check a write permission of your file system (unlike open()).
   *
   * - Saved file path is `${workingDir()}/${document._id}.json`. {@link InvalidIdLengthError} will be thrown if the path length exceeds the maximum length of a filepath on the device.
   *
   * - A update operation is not skipped when no change occurred on a specified document.
   *
   * @param id - _id property of a document
   * @param document - This is a {@link JsonDoc}, but _id property is ignored.
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link CannotWriteDataError}
   * @throws {@link CannotInsertedirectoryError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link DocumentNotFoundError}
   *
   */
  update (
    id: string,
    document: { [key: string]: any },
    options?: PutOptions
  ): Promise<PutResult>;

  update (
    idOrDoc: string | JsonDoc,
    docOrOptions: { [key: string]: any } | PutOptions,
    options?: PutOptions
  ) {
    if (typeof idOrDoc === 'object') {
      docOrOptions = { ...docOrOptions, insertOrUpdate: 'update' };
    }

    return putImpl.call(this, idOrDoc, docOrOptions, {
      ...options,
      insertOrUpdate: 'update',
    });
  }

  /**
   * Get a document
   *
   * @param docId - id of a target document
   * @param backNumber - Specify a number to go back to old revision. Default is 0. When backNumber equals 0, a document in the current DB is returned.
   * When backNumber is 0 and a document has been deleted in the current DB, it returns undefined.
   *
   * @returns
   *  - JsonDoc if exists.
   *
   *  - undefined if not exists.
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link CorruptedRepositoryError}
   * @throws {@link InvalidBackNumberError}
   */
  get (docId: string, backNumber?: number): Promise<JsonDoc | undefined> {
    // Do not use 'get = getImpl;' because api-extractor(TsDoc) recognizes this not as a function but a property.
    return getImpl.call(this, docId, { back_number: backNumber, with_metadata: false });
  }

  /**
   * Get a document with metadata
   *
   * @param docId - id of a target document
   * @param backNumber - Specify a number to go back to old revision. Default is 0. When backNumber is 0, a document in the current DB is returned.
   * When backNumber is 0 and a document has been deleted in the current DB, it returns undefined.
   *
   * @returns
   *  - JsonDocWithMetadata if exists.
   *
   *  - undefined if not exists.
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link CorruptedRepositoryError}
   * @throws {@link InvalidBackNumberError}
   */
  getDocWithMetaData (
    docId: string,
    backNumber?: number
  ): Promise<JsonDocWithMetadata | undefined> {
    return (getImpl.call(this, docId, {
      back_number: backNumber,
      with_metadata: true,
    }) as unknown) as Promise<JsonDocWithMetadata>;
  }

  /**
   * Get a specific revision of a document
   *
   * @param - fileSHA SHA-1 hash of Git object (40 characters)
   *
   * @returns
   *  - JsonDoc if exists.
   *
   *  - undefined if not exists.
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedFileSHAError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link CannotGetEntryError}
   */
  getByRevision (fileSHA: string): Promise<JsonDoc | undefined> {
    return getByRevisionImpl.call(this, fileSHA);
  }

  /**
   * Get revision history of a file from new to old
   *
   * @param - docId - id of a target document
   * @returns Array of fileSHA (NOTE: getDocHistory returns empty array if document does not exist in history.)
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidCollectionPathCharacterError}
   * @throws {@link InvalidCollectionPathLengthError}
   * @throws {@link InvalidIdLengthError}
   * @throws {@link CannotGetEntryError}
   */
  getDocHistory (docID: string): Promise<string[]> {
    return getDocHistoryImpl.call(this, docID);
  }

  /**
   * Remove a document
   *
   * @param id - id of a target document
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link DocumentNotFoundError} when the specified document does not exist.
   * @throws {@link CannotDeleteDataError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   *
   */
  delete (id: string, options?: DeleteOptions): Promise<RemoveResult>;
  /**
   * Remove a document
   *
   * @param jsonDoc - Target document
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link UndefinedDocumentIdError}
   * @throws {@link DocumentNotFoundError} when the specified document does not exist.
   * @throws {@link CannotDeleteDataError}
   * @throws {@link InvalidIdCharacterError}
   * @throws {@link InvalidIdLengthError}
   *
   */
  delete (jsonDoc: JsonDoc, options?: DeleteOptions): Promise<RemoveResult>;
  delete (idOrDoc: string | JsonDoc, options?: DeleteOptions): Promise<RemoveResult> {
    return deleteImpl.call(this, idOrDoc, options);
  }

  /**
   * This is an alias of remove()
   */

  remove (id: string, options?: DeleteOptions): Promise<RemoveResult>;
  /**
   * This is an alias of remove()
   */
  remove (jsonDoc: JsonDoc, options?: DeleteOptions): Promise<RemoveResult>;
  remove (idOrDoc: string | JsonDoc, options?: DeleteOptions): Promise<RemoveResult> {
    return deleteImpl.call(this, idOrDoc, options);
  }

  /**
   * Get all the documents
   *
   * @remarks
   *
   * @param options - The options specify how to get documents.
   *
   * @throws {@link DatabaseClosingError}
   * @throws {@link RepositoryNotOpenError}
   * @throws {@link InvalidJsonObjectError}
   * @throws {@link InvalidCollectionPathCharacterError}
   * @throws {@link InvalidCollectionPathLengthError}
   *
   */
  allDocs (options?: AllDocsOptions): Promise<AllDocsResult> {
    // Do not use 'allDocs = allDocsImpl;' because api-extractor(TsDoc) recognizes this not as a function but a property.
    return allDocsImpl.call(this, options);
  }

  /**
   * getRemoteURLs
   *
   */
  getRemoteURLs (): string[] {
    return Object.keys(this._synchronizers);
  }

  /**
   * Get synchronizer
   *
   */
  getSynchronizer (remoteURL: string): Sync {
    return this._synchronizers[remoteURL];
  }

  /**
   * Stop and unregister remote synchronization
   *
   */
  unregisterRemote (remoteURL: string) {
    this._synchronizers[remoteURL].cancel();
    delete this._synchronizers[remoteURL];
  }

  /**
   * Synchronize with a remote repository
   *
   * @throws {@link UndefinedRemoteURLError} (from Sync#constructor())
   * @throws {@link IntervalTooSmallError}  (from Sync#constructor())
   *
   * @throws {@link RemoteRepositoryConnectError} (from Sync#init())
   * @throws {@link PushWorkerError} (from Sync#init())
   * @throws {@link SyncWorkerError} (from Sync#init())
   *
   * @remarks
   * Register and synchronize with a remote repository. Do not register the same remote repository again. Call unregisterRemote() before register it again.
   */
  async sync (remoteURL: string, options?: RemoteOptions): Promise<Sync>;
  /**
   * Synchronize with a remote repository
   *
   * @throws {@link UndefinedRemoteURLError} (from Sync#constructor())
   * @throws {@link IntervalTooSmallError}  (from Sync#constructor())
   *
   * @throws {@link RemoteRepositoryConnectError} (from Sync#init())
   * @throws {@link PushWorkerError} (from Sync#init())
   * @throws {@link SyncWorkerError} (from Sync#init())
   *
   * @remarks
   * Register and synchronize with a remote repository. Do not register the same remote repository again. Call removeRemote() before register it again.
   */
  async sync (options?: RemoteOptions): Promise<Sync>;
  async sync (
    remoteUrlOrOption?: string | RemoteOptions,
    options?: RemoteOptions
  ): Promise<Sync> {
    if (typeof remoteUrlOrOption === 'string') {
      options ??= {};
      options.remote_url = remoteUrlOrOption;
    }
    else {
      options = remoteUrlOrOption;
    }

    if (
      options?.remote_url !== undefined &&
      this._synchronizers[options?.remote_url] !== undefined
    ) {
      throw new RemoteAlreadyRegisteredError(options.remote_url);
    }
    const remote = await syncImpl.call(this, options);
    this._synchronizers[remote.remoteURL()] = remote;
    return remote;
  }
}
