defexception Kernel.CompilationError, message: "compilation failed"

defmodule Kernel.ParallelCompiler do
  alias :orddict, as: OrdDict

  @moduledoc """
  A module responsible for compiling files in parallel.
  """

  defmacrop default_callback, do: quote(do: fn x -> x end)

  @doc """
  Compiles the given files.

  Those files are compiled in parallel and can automatically
  detect dependencies between them. Once a dependency is found,
  the current file stops being compiled until the dependency is
  resolved.

  If there is any error during compilation or if warnings_as_errors
  is set to true and there is a warning, this function will fail
  with an exception.

  A callback that is invoked every time a file is compiled
  with its name can be optionally given as argument.
  """
  def files(files, callback // default_callback) do
    spawn_compilers(files, nil, callback)
  end

  @doc """
  Compiles the given files to the given path.
  Read files/2 for more information.
  """
  def files_to_path(files, path, callback // default_callback) when is_binary(path) do
    spawn_compilers(files, path, callback)
  end

  defp spawn_compilers(files, path, callback) do
    Code.ensure_loaded(Kernel.ErrorHandler)
    :elixir_code_server.cast(:reset_warnings)
    schedulers = max(:erlang.system_info(:schedulers_online), 2)

    result = spawn_compilers(files, path, callback, [], [], schedulers, [])
    case :elixir_code_server.call(:compilation_status) do
      :ok    -> result
      :error -> raise Kernel.CompilationError, [], []
    end
  end

  # We already have 4 currently running, don't spawn new ones
  defp spawn_compilers(files, output, callback, waiting, queued, schedulers, result) when
      length(queued) - length(waiting) >= schedulers do
    wait_for_messages(files, output, callback, waiting, queued, schedulers, result)
  end

  # Spawn a compiler for each file in the list until we reach the limit
  defp spawn_compilers([h|t], output, callback, waiting, queued, schedulers, result) do
    parent = self()

    child  = spawn_link fn ->
      :erlang.put(:elixir_compiler_pid, parent)
      :erlang.put(:elixir_ensure_compiled, true)
      :erlang.process_flag(:error_handler, Kernel.ErrorHandler)

      try do
        if output do
          :elixir_compiler.file_to_path(h, output)
        else
          :elixir_compiler.file(h)
        end
        parent <- { :compiled, self(), h }
      catch
        kind, reason ->
          parent <- { :failure, self(), kind, reason, System.stacktrace }
      end
    end

    spawn_compilers(t, output, callback, waiting, [{child, h}|queued], schedulers, result)
  end

  # No more files, nothing waiting, queue is empty, we are done
  defp spawn_compilers([], _output, _callback, [], [], _schedulers, result), do: result

  # Queued x, waiting for x: POSSIBLE ERROR! Release processes so we get the failures
  defp spawn_compilers([], output, callback, waiting, queued, schedulers, result) when length(waiting) == length(queued) do
    Enum.each queued, fn { child, _ } -> child <- { :release, self() } end
    wait_for_messages([], output, callback, waiting, queued, schedulers, result)
  end

  # No more files, but queue and waiting are not full or do not match
  defp spawn_compilers([], output, callback, waiting, queued, schedulers, result) do
    wait_for_messages([], output, callback, waiting, queued, schedulers, result)
  end

  # Wait for messages from child processes
  defp wait_for_messages(files, output, callback, waiting, queued, schedulers, result) do
    receive do
      { :compiled, child, file } ->
        callback.(file)
        new_queued  = List.keydelete(queued, child, 0)
        # Sometimes we may have spurious entries in the waiting
        # list because someone invoked try/rescue UndefinedFunctionError
        new_waiting = List.keydelete(waiting, child, 0)
        spawn_compilers(files, output, callback, new_waiting, new_queued, schedulers, result)
      { :module_available, _child, module, binary } ->
        new_waiting = release_waiting_processes(module, waiting)
        new_result  = [{module, binary}|result]
        wait_for_messages(files, output, callback, new_waiting, queued, schedulers, new_result)
      { :waiting, child, on } ->
        new_waiting = OrdDict.store(child, on, waiting)
        spawn_compilers(files, output, callback, new_waiting, queued, schedulers, result)
      { :failure, child, kind, reason, stacktrace } ->
        if many_missing?(child, files, waiting, queued) do
          IO.puts "== Compilation failed =="
          IO.puts "Compilation failed on the following files:\n"

          Enum.each Enum.reverse(queued), fn { pid, file } ->
            case List.keyfind(waiting, pid, 0) do
              { _, mod } -> IO.puts "* #{file} is missing module #{inspect mod}"
              _ -> :ok
            end
          end

          IO.puts "\nThe first failure is shown below..."
        end

        {^child, file} = List.keyfind(queued, child, 0)
        IO.puts "== Compilation error on file #{file} =="
        :erlang.raise(kind, reason, stacktrace)
    end
  end

  defp many_missing?(child, files, waiting, queued) do
    waiting_length = length(waiting)

    match?({ ^child, _ }, List.keyfind(waiting, child, 0)) and
      waiting_length > 1 and files == [] and
      waiting_length == length(queued)
  end

  # Release waiting processes that are waiting for the given module
  defp release_waiting_processes(module, waiting) do
    Enum.filter waiting, fn { child, waiting_module } ->
      if waiting_module == module do
        child <- { :release, self() }
        false
      else
        true
      end
    end
  end
end
