# Copyright 2015 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Tests for SavedModel utils."""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from tensorflow.core.framework import types_pb2
from tensorflow.python.framework import dtypes
from tensorflow.python.framework import ops
from tensorflow.python.framework import sparse_tensor
from tensorflow.python.ops import array_ops
from tensorflow.python.platform import test
from tensorflow.python.saved_model import utils


class UtilsTest(test.TestCase):

  def testBuildTensorInfoDense(self):
    x = array_ops.placeholder(dtypes.float32, 1, name="x")
    x_tensor_info = utils.build_tensor_info(x)
    self.assertEqual("x:0", x_tensor_info.name)
    self.assertEqual(types_pb2.DT_FLOAT, x_tensor_info.dtype)
    self.assertEqual(1, len(x_tensor_info.tensor_shape.dim))
    self.assertEqual(1, x_tensor_info.tensor_shape.dim[0].size)

  def testBuildTensorInfoSparse(self):
    x = sparse_tensor.SparseTensor(indices=[[3, 3], [4, 4], [5, 5]],
                                   values=[103.0, 104.0, 105.0],
                                   dense_shape=[42, 69])
    x_tensor_info = utils.build_tensor_info(x)
    self.assertEqual(x.values.name,
                     x_tensor_info.coo_sparse.values_tensor_name)
    self.assertEqual(x.indices.name,
                     x_tensor_info.coo_sparse.indices_tensor_name)
    self.assertEqual(x.dense_shape.name,
                     x_tensor_info.coo_sparse.dense_shape_tensor_name)
    self.assertEqual(types_pb2.DT_FLOAT, x_tensor_info.dtype)
    self.assertEqual(2, len(x_tensor_info.tensor_shape.dim))
    self.assertEqual(42, x_tensor_info.tensor_shape.dim[0].size)
    self.assertEqual(69, x_tensor_info.tensor_shape.dim[1].size)

  def testGetTensorFromInfoDense(self):
    expected = array_ops.placeholder(dtypes.float32, 1, name="x")
    tensor_info = utils.build_tensor_info(expected)
    actual = utils.get_tensor_from_tensor_info(tensor_info)
    self.assertIsInstance(actual, ops.Tensor)
    self.assertEqual(expected.name, actual.name)

  def testGetTensorFromInfoSparse(self):
    expected = array_ops.sparse_placeholder(dtypes.float32, name="x")
    tensor_info = utils.build_tensor_info(expected)
    actual = utils.get_tensor_from_tensor_info(tensor_info)
    self.assertIsInstance(actual, sparse_tensor.SparseTensor)
    self.assertEqual(expected.values.name, actual.values.name)
    self.assertEqual(expected.indices.name, actual.indices.name)
    self.assertEqual(expected.dense_shape.name, actual.dense_shape.name)

  def testGetTensorFromInfoInOtherGraph(self):
    with ops.Graph().as_default() as expected_graph:
      expected = array_ops.placeholder(dtypes.float32, 1, name="right")
      tensor_info = utils.build_tensor_info(expected)
    with ops.Graph().as_default():  # Some other graph.
      array_ops.placeholder(dtypes.float32, 1, name="other")
    actual = utils.get_tensor_from_tensor_info(tensor_info,
                                               graph=expected_graph)
    self.assertIsInstance(actual, ops.Tensor)
    self.assertIs(actual.graph, expected_graph)
    self.assertEqual(expected.name, actual.name)

  def testGetTensorFromInfoInScope(self):
    # Build a TensorInfo with name "bar/x:0".
    with ops.Graph().as_default():
      with ops.name_scope("bar"):
        unscoped = array_ops.placeholder(dtypes.float32, 1, name="x")
        tensor_info = utils.build_tensor_info(unscoped)
        self.assertEqual("bar/x:0", tensor_info.name)
    # Build a graph with node "foo/bar/x:0", akin to importing into scope foo.
    with ops.Graph().as_default():
      with ops.name_scope("foo"):
        with ops.name_scope("bar"):
          expected = array_ops.placeholder(dtypes.float32, 1, name="x")
      self.assertEqual("foo/bar/x:0", expected.name)
      # Test that tensor is found by prepending the import scope.
      actual = utils.get_tensor_from_tensor_info(tensor_info,
                                                 import_scope="foo")
      self.assertEqual(expected.name, actual.name)

  def testGetTensorFromInfoRaisesErrors(self):
    expected = array_ops.placeholder(dtypes.float32, 1, name="x")
    tensor_info = utils.build_tensor_info(expected)
    tensor_info.name = "blah:0"  # Nonexistant name.
    with self.assertRaises(KeyError):
      utils.get_tensor_from_tensor_info(tensor_info)
    tensor_info.ClearField("name")  # Malformed (missing encoding).
    with self.assertRaises(ValueError):
      utils.get_tensor_from_tensor_info(tensor_info)

if __name__ == "__main__":
  test.main()
