/* Copyright 2017 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

#ifndef TENSORFLOW_COMPILER_XLA_SERVICE_GPU_GEMM_THUNK_H_
#define TENSORFLOW_COMPILER_XLA_SERVICE_GPU_GEMM_THUNK_H_

#include "tensorflow/compiler/xla/service/buffer_assignment.h"
#include "tensorflow/compiler/xla/service/gpu/buffer_allocations.h"
#include "tensorflow/compiler/xla/service/gpu/gpu_executable.h"
#include "tensorflow/compiler/xla/service/gpu/thunk.h"
#include "tensorflow/compiler/xla/service/hlo_instruction.h"
#include "tensorflow/compiler/xla/xla_data.pb.h"
#include "tensorflow/core/lib/core/status.h"
#include "tensorflow/core/platform/stream_executor_no_cuda.h"

namespace xla {
namespace gpu {

// This class stores everything that StreamExecutor needs to launch a BLAS gemm.
// It is generated by IrEmitter.
//
// This is thread-compatible.
class GemmThunk : public Thunk {
 public:
  // Constructs a thunk that computes "output = lhs <dot> rhs" using BLAS gemm.
  // transpose_lhs and transpose_rhs indicate whether gemm should transpose the
  // lhs and rhs operand. hlo_instruction is as in Thunk.
  GemmThunk(BufferAllocation::Index lhs_buffer,
            BufferAllocation::Index rhs_buffer,
            BufferAllocation::Index output_buffer, const Shape& lhs_shape,
            const Shape& rhs_shape, const Shape& output_shape,
            bool transpose_lhs, bool transpose_rhs,
            const HloInstruction* hlo_instruction);

  GemmThunk(const GemmThunk&) = delete;
  GemmThunk& operator=(const GemmThunk&) = delete;

  // Does the gemm operation for the thunk on "stream", which must be non-null.
  tensorflow::Status ExecuteOnStream(
      const BufferAllocations& buffer_allocations,
      perftools::gputools::Stream* stream) override;

 private:
  BufferAllocation::Index lhs_buffer_;
  BufferAllocation::Index rhs_buffer_;
  BufferAllocation::Index output_buffer_;

  Shape lhs_shape_;
  Shape rhs_shape_;
  Shape output_shape_;

  bool transpose_lhs_;
  bool transpose_rhs_;
};

}  // namespace gpu
}  // namespace xla

#endif  // TENSORFLOW_COMPILER_XLA_SERVICE_GPU_GEMM_THUNK_H_
