#!/usr/bin/env bash
# Copyright 2015 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

set -e -o errexit

if [ -d "../org_tensorflow" ]; then
  script_path="../org_tensorflow"
else
  # Prefix expected paths with ./ locally and external/reponame/ for remote repos.
  # TODO(kchodorow): remove once runfiles paths are fixed, see
  # https://github.com/bazelbuild/bazel/issues/848.
  script_path=$(dirname $(dirname $(dirname "$0")))
  script_path=${script_path:-.}
fi

EXPECTED_PATHS="$script_path/util/python/python_include"\
" $script_path/util/python/python_lib"\
" $script_path/third_party/py/numpy/numpy_include"

function main {
  argument="$1"
  shift
  case $argument in
    --check)
      check_python
      exit 0
      ;;
    --setup)
      setup_python "$1"
      exit 0
      ;;
  esac
}

function setup_python {
  PYTHON_BIN_PATH="$1";

  if [ -z "$PYTHON_BIN_PATH" ]; then
    echo "PYTHON_BIN_PATH was not provided.  Did you run configure?"
    exit 1
  fi
  if [ ! -x "$PYTHON_BIN_PATH" ]  || [ -d "$PYTHON_BIN_PATH" ]; then
    echo "PYTHON_BIN_PATH is not executable.  Is it the python binary?"
    exit 1
  fi

  local python_major_version=$("${PYTHON_BIN_PATH}" -c 'from __future__ import print_function; import sys; print(sys.version_info[0]);')
  if [ "$python_major_version" == "" ]; then
    echo -e "\n\nERROR: Problem getting python version.  Is $PYTHON_BIN_PATH the correct python binary?"
    exit 1
  fi

  local python_include=$("${PYTHON_BIN_PATH}" -c 'from __future__ import print_function; from distutils import sysconfig; print(sysconfig.get_python_inc());')
  if [ "$python_include" == "" ]; then
    echo -e "\n\nERROR: Problem getting python include path.  Is distutils installed?"
    exit 1
  fi
  local python_lib=$("${PYTHON_BIN_PATH}" -c 'from __future__ import print_function; from distutils import sysconfig; print(sysconfig.get_python_lib());')
  if [ "$python_lib" == "" ]; then
    echo -e "\n\nERROR: Problem getting python lib path.  Is distutils installed?"
    exit 1
  fi
  local numpy_include=$("${PYTHON_BIN_PATH}" -c 'from __future__ import print_function; import numpy; print(numpy.get_include());')
  if [ "$numpy_include" == "" ]; then
    echo -e "\n\nERROR: Problem getting numpy include path.  Is numpy installed?"
    exit 1
  fi

  for x in $EXPECTED_PATHS; do
    if [ -e "$x" ]; then
      rm "$x"
    fi
  done

  ln -sf "${python_include}" util/python/python_include
  ln -sf "${python_lib}" util/python/python_lib
  ln -sf "${numpy_include}" third_party/py/numpy/numpy_include

  # Write tools/bazel.rc
  echo "# Autogenerated by configure: DO NOT EDIT" > tools/bazel.rc
  sed -e "s/\$PYTHON_MAJOR_VERSION/$python_major_version/g" \
      -e "s[\$PYTHON_BINARY[$PYTHON_BIN_PATH[g" \
      tools/bazel.rc.template >> tools/bazel.rc
  # Write tools/python_bin_path.sh
  echo "export PYTHON_BIN_PATH=$PYTHON_BIN_PATH" > tools/python_bin_path.sh
}

function check_python {
  for x in $EXPECTED_PATHS; do
    if [ ! -e "$x" ]; then
      echo -e "\n\nERROR: Cannot find '${x}'.  Did you run configure?\n\n" 1>&2
      exit 1
    fi
    if [ ! -L "${x}" ]; then
      echo -e "\n\nERROR: '${x}' is not a symbolic link.  Internal error.\n\n" 1>&2
      exit 1
    fi
    true_path=$(readlink "${x}")
    if [ ! -d "${true_path}" ]; then
      echo -e "\n\nERROR: '${x}' does not refer to an existing directory: ${true_path}.  Do you need to rerun configure?\n\n" 1>&2
      exit 1
    fi
  done
}

main "$@"
