package scala

import scala.annotation.nowarn
import scala.reflect.io.File
import java.util.Base64
import org.junit.Test

package collection {
  object accessWrappers { val Wrappers = convert.JavaCollectionWrappers }
}

import collection.accessWrappers.Wrappers._

// This test is self-modifying when run as follows:
//
//    junit/testOnly scala.SerializationStabilityTest -- -Doverwrite.source=test/junit/scala/SerializationStabilityTest.scala
//
// Use this to re-establish a baseline for serialization compatibility.

// based on run/t8549.scala partest
@annotation.nowarn("cat=deprecation&origin=scala.collection.mutable.AnyRefMap")
object SerializationStability {

  def parseBase64Binary(s: String): Array[Byte] = Base64.getDecoder.decode(s)

  def printBase64Binary(data: Array[Byte]): String = Base64.getEncoder.encode(data).map(_.toChar).mkString

  val overwrite: Option[File] = sys.props.get("overwrite.source").map(s => new File(new java.io.File(s)))
  private class CheckInitProbe {
    val x = "initialized"
  }
  val isCheckInitEnabled = classOf[CheckInitProbe].getDeclaredFields.exists(_.getName.startsWith("bitmap$init"))

  def serialize(o: AnyRef): String = {
    val bos = new java.io.ByteArrayOutputStream()
    val out = new java.io.ObjectOutputStream(bos)
    out.writeObject(o)
    out.flush()
    printBase64Binary(bos.toByteArray())
  }

  def amend(file: File)(f: String => String): Unit = file.writeAll(f(file.slurp()))
  def quote(s: String) = List("\"", s, "\"").mkString

  def patch(file: File, line: Int, prevResult: String, result: String): Unit = {
    amend(file) {
      content =>
        content.linesIterator.toList.zipWithIndex.map {
          case (content, i) if i == line - 1 =>
            val newContent = content.replace(quote(prevResult), quote(result))
            if (newContent != content)
              println(s"- $content\n+ $newContent\n")
            newContent
          case (content, _) => content
        }.mkString("\n")
    }
  }

  def updateComment(file: File): Unit = {
    val timestamp = {
      import java.text.SimpleDateFormat
      val sdf = new SimpleDateFormat("yyyyMMdd-HH:mm:ss")
      sdf.format(new java.util.Date)
    }
    val newComment = s"  // Generated on $timestamp with Scala ${scala.util.Properties.versionString})"
    amend(file) {
      content =>
        content.linesIterator.toList.map {
          f => f.replaceAll("""^ +// Generated on.*""", newComment)
        }.mkString("\n")
    }
  }

  def deserialize(string: String): AnyRef = {
    val bis = new java.io.ByteArrayInputStream(parseBase64Binary(string))
    val in = new java.io.ObjectInputStream(bis)
    in.readObject()
  }

  def checkRoundTrip[T <: AnyRef](instance: T)(f: T => AnyRef): Unit = {
    val result = serialize(instance)
    val reconstituted = deserialize(result).asInstanceOf[T]
    assert(f(instance) == f(reconstituted), (f(instance), f(reconstituted)))
  }

  def check[T <: AnyRef](stack: Array[StackTraceElement])(instance: => T)(prevResult: String, f: T => AnyRef = (x: T) => x): Unit = {
    val result = serialize(instance)
    overwrite match {
      case Some(ov) =>
        val lineNumberOfLiteralString = stack.apply(2).getLineNumber
        patch(ov, lineNumberOfLiteralString, prevResult, result)
      case None =>
        if (!isCheckInitEnabled) {
          checkRoundTrip(instance)(f)
          assert(f(deserialize(prevResult).asInstanceOf[T]) == f(instance), s"$instance != f(deserialize(prevResult))")
          assert(prevResult == result, s"instance = $instance : ${instance.getClass}\n serialization unstable: ${prevResult}\n   found: ${result}")
        }
    }
  }

  // Generated on 20210115-11:32:52 with Scala version 2.13.5-20210114-144616-3802815)
  def main(args: Array[String]): Unit = {
    overwrite.foreach(updateComment)
    def g = Thread.currentThread.getStackTrace

    check(g)(Some(1))("rO0ABXNyAApzY2FsYS5Tb21lESLyaV6hi3QCAAFMAAV2YWx1ZXQAEkxqYXZhL2xhbmcvT2JqZWN0O3hyAAxzY2FsYS5PcHRpb27+aTf92w5mdAIAAHhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAE=")
    check(g)(None)("rO0ABXNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAC3NjYWxhLk5vbmUkRlAk9lPKlKwCAAB4cgAMc2NhbGEuT3B0aW9u/mk3/dsOZnQCAAB4cA==")

    // TODO what is the status of the following checks?
    //val f0: Function0[Int] = () => 0
    //val f1: Function1[Int, Int] = _ => 0
    //val f2: Function2[Int, Int, Int] = (_, _) => 0
    
    // check(g)(f0)("rO0ABXNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgADTAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgADTAAJaW1wbENsYXNzcQB+AANMAA5pbXBsTWV0aG9kTmFtZXEAfgADTAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgADTAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgADeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAHZyAB1zY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JAAAAAAAAAAAAAAAeHB0ACVzY2FsYS9ydW50aW1lL2phdmE4L0pGdW5jdGlvbjAkbWNJJHNwdAAMYXBwbHkkbWNJJHNwdAADKClJdAAdc2NhbGEvU2VyaWFsaXphdGlvblN0YWJpbGl0eSR0AA8kYW5vbmZ1biRtYWluJDRxAH4AC3EAfgAL")
    // check(g)(f1)("rO0ABXNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgADTAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgADTAAJaW1wbENsYXNzcQB+AANMAA5pbXBsTWV0aG9kTmFtZXEAfgADTAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgADTAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgADeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAHZyAB1zY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JAAAAAAAAAAAAAAAeHB0ACZzY2FsYS9ydW50aW1lL2phdmE4L0pGdW5jdGlvbjEkbWNJSSRzcHQADWFwcGx5JG1jSUkkc3B0AAQoSSlJdAAdc2NhbGEvU2VyaWFsaXphdGlvblN0YWJpbGl0eSR0AA8kYW5vbmZ1biRtYWluJDVxAH4AC3EAfgAL")
    // check(g)(f2)("rO0ABXNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgADTAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgADTAAJaW1wbENsYXNzcQB+AANMAA5pbXBsTWV0aG9kTmFtZXEAfgADTAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgADTAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgADeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAHZyAB1zY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JAAAAAAAAAAAAAAAeHB0ACdzY2FsYS9ydW50aW1lL2phdmE4L0pGdW5jdGlvbjIkbWNJSUkkc3B0AA5hcHBseSRtY0lJSSRzcHQABShJSSlJdAAdc2NhbGEvU2VyaWFsaXphdGlvblN0YWJpbGl0eSR0AA8kYW5vbmZ1biRtYWluJDZxAH4AC3EAfgAL")

    class TypeClass[T](t: T) extends Serializable

    abstract class Typeable[T] extends Serializable {
      val typeClass: TypeClass[T]
    }

    case class Typed[T](t: T) extends Typeable[T] {
      lazy val typeClass = new TypeClass(t)
    }

    check(g)(List(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdCQAAAAAAAAAAwIAAHhwdwT/////c3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3EAfgALAAAAA3NxAH4ABnZyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")
    check(g)(Nil)(           "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdCQAAAAAAAAAAwIAAHhwdwQAAAAAeA==")
    check(g)(List("1", "2", "3"))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdCQAAAAAAAAAAwIAAHhwdwT/////dAABMXQAATJ0AAEzc3EAfgAGdnIAJnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5TZXJpYWxpemVFbmQkAAAAAAAAAAMCAAB4cHg=")

    //check(g)(new TypeClass(1))("rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")
    //check(g)(new TypeClass("1"))( "rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")
    //check(g)(new TypeClass(List(1, 2, 3)))( "rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")
    //check(g)(new TypeClass(Nil))( "rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")
    //check(g)(new TypeClass(f0))( "rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")
    //check(g)(new TypeClass(f1))( "rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")
    //check(g)(new TypeClass(f2))( "rO0ABXNyAChzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxXdOtc6kLOZgCAAB4cA==")

    check(g)(Typed(1))("rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABcA==")
    check(g)(Typed("1"))( "rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHQAATFw")
    check(g)(Typed(List(1, 2, 3)))( "rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdCQAAAAAAAAAAwIAAHhwdwT/////c3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+ABAAAAACc3EAfgAQAAAAA3NxAH4AC3ZyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4cA==")
    check(g)(Typed(Nil))( "rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAgc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdCQAAAAAAAAAAwIAAHhwdwQAAAAAeHA=")
    // check(g)(Typed(f0))( "rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgAITAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgAITAAJaW1wbENsYXNzcQB+AAhMAA5pbXBsTWV0aG9kTmFtZXEAfgAITAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgAITAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgAIeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAHZyAB1zY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JAAAAAAAAAAAAAAAeHB0ACVzY2FsYS9ydW50aW1lL2phdmE4L0pGdW5jdGlvbjAkbWNJJHNwdAAMYXBwbHkkbWNJJHNwdAADKClJdAAdc2NhbGEvU2VyaWFsaXphdGlvblN0YWJpbGl0eSR0AA8kYW5vbmZ1biRtYWluJDRxAH4AEHEAfgAQcA==")
    // check(g)(Typed(f1))( "rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgAITAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgAITAAJaW1wbENsYXNzcQB+AAhMAA5pbXBsTWV0aG9kTmFtZXEAfgAITAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgAITAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgAIeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAHZyAB1zY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JAAAAAAAAAAAAAAAeHB0ACZzY2FsYS9ydW50aW1lL2phdmE4L0pGdW5jdGlvbjEkbWNJSSRzcHQADWFwcGx5JG1jSUkkc3B0AAQoSSlJdAAdc2NhbGEvU2VyaWFsaXphdGlvblN0YWJpbGl0eSR0AA8kYW5vbmZ1biRtYWluJDVxAH4AEHEAfgAQcA==")
    // check(g)(Typed(f2))( "rO0ABXNyACRzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVkJDEGkA3aN1HC9wIAA1oACGJpdG1hcCQwTAABdHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wACXR5cGVDbGFzc3QAKkxzY2FsYS9TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVDbGFzcyQxO3hyACdzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JFR5cGVhYmxlJDFnfHu9Y0dh9wIAAHhwAHNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgAITAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgAITAAJaW1wbENsYXNzcQB+AAhMAA5pbXBsTWV0aG9kTmFtZXEAfgAITAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgAITAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgAIeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAHZyAB1zY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JAAAAAAAAAAAAAAAeHB0ACdzY2FsYS9ydW50aW1lL2phdmE4L0pGdW5jdGlvbjIkbWNJSUkkc3B0AA5hcHBseSRtY0lJSSRzcHQABShJSSlJdAAdc2NhbGEvU2VyaWFsaXphdGlvblN0YWJpbGl0eSR0AA8kYW5vbmZ1biRtYWluJDZxAH4AEHEAfgAQcA==")

    check(g)(Vector(1))(     "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAic2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuVmVjdG9yJAAAAAAAAAADAgAAeHB3BAAAAAFzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXg=")
    check(g)(Vector())(      "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAic2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuVmVjdG9yJAAAAAAAAAADAgAAeHB3BAAAAAB4")

    import collection.{ mutable, immutable }

    class C
    check(g)(reflect.classTag[C])("rO0ABXNyACZzY2FsYS5yZWZsZWN0LkNsYXNzVGFnJEdlbmVyaWNDbGFzc1RhZwAAAAAAAAABAgABTAAMcnVudGltZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACBzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JEMkMQAAAAAAAAAAAAAAeHA=")
    check(g)(reflect.classTag[Int])("rO0ABXNyAClzY2FsYS5yZWZsZWN0Lk1hbmlmZXN0RmFjdG9yeSRJbnRNYW5pZmVzdAAAAAAAAAABAgAAeHIAHHNjYWxhLnJlZmxlY3QuQW55VmFsTWFuaWZlc3QAAAAAAAAAAQIAAUwACHRvU3RyaW5ndAASTGphdmEvbGFuZy9TdHJpbmc7eHB0AANJbnQ=")
    check(g)(reflect.classTag[String])("rO0ABXNyACZzY2FsYS5yZWZsZWN0LkNsYXNzVGFnJEdlbmVyaWNDbGFzc1RhZwAAAAAAAAABAgABTAAMcnVudGltZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyABBqYXZhLmxhbmcuU3RyaW5noPCkOHo7s0ICAAB4cA==")
    check(g)(reflect.classTag[Object])("rO0ABXNyACxzY2FsYS5yZWZsZWN0Lk1hbmlmZXN0RmFjdG9yeSRPYmplY3RNYW5pZmVzdAAAAAAAAAABAgAAeHIALXNjYWxhLnJlZmxlY3QuTWFuaWZlc3RGYWN0b3J5JFBoYW50b21NYW5pZmVzdAAAAAAAAAABAgABTAAIdG9TdHJpbmd0ABJMamF2YS9sYW5nL1N0cmluZzt4cgAvc2NhbGEucmVmbGVjdC5NYW5pZmVzdEZhY3RvcnkkQ2xhc3NUeXBlTWFuaWZlc3QAAAAAAAAAAQIAA0wABnByZWZpeHQADkxzY2FsYS9PcHRpb247TAAMcnVudGltZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMAA10eXBlQXJndW1lbnRzdAAhTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0xpc3Q7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzcQB+AAV4cHZyAAtzY2FsYS5Ob25lJEZQJPZTypSsAgAAeHIADHNjYWxhLk9wdGlvbv5pN/3bDmZ0AgAAeHB2cgAQamF2YS5sYW5nLk9iamVjdAAAAAAAAAAAAAAAeHBzcgAyc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLkRlZmF1bHRTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAwMAAUwAB2ZhY3Rvcnl0ABpMc2NhbGEvY29sbGVjdGlvbi9GYWN0b3J5O3hwc3IAKnNjYWxhLmNvbGxlY3Rpb24uSXRlcmFibGVGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAIkxzY2FsYS9jb2xsZWN0aW9uL0l0ZXJhYmxlRmFjdG9yeTt4cHNxAH4ACHZyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0JAAAAAAAAAADAgAAeHB3BAAAAAB4dAAGT2JqZWN0")

    // TODO Reinstate test now that this suite skips under -Xcheckinit
    // check(g)(Enum)(         "rO0ABXNyAApUZXN0JEVudW0ketCIyQ8C23MCAAJMAAJWMXQAGUxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZTtMAAJWMnQAF0xzY2FsYS9FbnVtZXJhdGlvbiRWYWw7eHIAEXNjYWxhLkVudW1lcmF0aW9udaDN3ZgOWY4CAAhJAAZuZXh0SWRJABtzY2FsYSRFbnVtZXJhdGlvbiQkYm90dG9tSWRJABhzY2FsYSRFbnVtZXJhdGlvbiQkdG9wSWRMABRWYWx1ZU9yZGVyaW5nJG1vZHVsZXQAIkxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZU9yZGVyaW5nJDtMAA9WYWx1ZVNldCRtb2R1bGV0AB1Mc2NhbGEvRW51bWVyYXRpb24kVmFsdWVTZXQkO0wACG5leHROYW1ldAAbTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmF0b3I7TAAXc2NhbGEkRW51bWVyYXRpb24kJG5tYXB0AB5Mc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL01hcDtMABdzY2FsYSRFbnVtZXJhdGlvbiQkdm1hcHEAfgAHeHAAAAArAAAAAAAAACtwcHBzcgAgc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAAAAAAAAAAAQMAAHhwdw0AAALuAAAAAAAAAAQAeHNxAH4ACXcNAAAC7gAAAAEAAAAEAHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAqc3IAFXNjYWxhLkVudW1lcmF0aW9uJFZhbM9pZ6/J/O1PAgACSQAYc2NhbGEkRW51bWVyYXRpb24kVmFsJCRpTAAEbmFtZXQAEkxqYXZhL2xhbmcvU3RyaW5nO3hyABdzY2FsYS5FbnVtZXJhdGlvbiRWYWx1ZWJpfC/tIR1RAgACTAAGJG91dGVydAATTHNjYWxhL0VudW1lcmF0aW9uO0wAHHNjYWxhJEVudW1lcmF0aW9uJCRvdXRlckVudW1xAH4AEnhwcQB+AAhxAH4ACAAAACpweHNyABFUZXN0JEVudW0kJGFub24kMVlIjlmE1sXaAgAAeHEAfgARcQB+AAhxAH4ACHEAfgAT")
    // check(g)(Enum.V1)(      "rO0ABXNyABFUZXN0JEVudW0kJGFub24kMVlIjlmE1sXaAgAAeHIAF3NjYWxhLkVudW1lcmF0aW9uJFZhbHVlYml8L+0hHVECAAJMAAYkb3V0ZXJ0ABNMc2NhbGEvRW51bWVyYXRpb247TAAcc2NhbGEkRW51bWVyYXRpb24kJG91dGVyRW51bXEAfgACeHBzcgAKVGVzdCRFbnVtJHrQiMkPAttzAgACTAACVjF0ABlMc2NhbGEvRW51bWVyYXRpb24kVmFsdWU7TAACVjJ0ABdMc2NhbGEvRW51bWVyYXRpb24kVmFsO3hyABFzY2FsYS5FbnVtZXJhdGlvbnWgzd2YDlmOAgAISQAGbmV4dElkSQAbc2NhbGEkRW51bWVyYXRpb24kJGJvdHRvbUlkSQAYc2NhbGEkRW51bWVyYXRpb24kJHRvcElkTAAUVmFsdWVPcmRlcmluZyRtb2R1bGV0ACJMc2NhbGEvRW51bWVyYXRpb24kVmFsdWVPcmRlcmluZyQ7TAAPVmFsdWVTZXQkbW9kdWxldAAdTHNjYWxhL0VudW1lcmF0aW9uJFZhbHVlU2V0JDtMAAhuZXh0TmFtZXQAG0xzY2FsYS9jb2xsZWN0aW9uL0l0ZXJhdG9yO0wAF3NjYWxhJEVudW1lcmF0aW9uJCRubWFwdAAeTHNjYWxhL2NvbGxlY3Rpb24vbXV0YWJsZS9NYXA7TAAXc2NhbGEkRW51bWVyYXRpb24kJHZtYXBxAH4AC3hwAAAAKwAAAAAAAAArcHBwc3IAIHNjYWxhLmNvbGxlY3Rpb24ubXV0YWJsZS5IYXNoTWFwAAAAAAAAAAEDAAB4cHcNAAAC7gAAAAAAAAAEAHhzcQB+AA13DQAAAu4AAAABAAAABABzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAKnNyABVzY2FsYS5FbnVtZXJhdGlvbiRWYWzPaWevyfztTwIAAkkAGHNjYWxhJEVudW1lcmF0aW9uJFZhbCQkaUwABG5hbWV0ABJMamF2YS9sYW5nL1N0cmluZzt4cQB+AAFxAH4ADHEAfgAMAAAAKnB4cQB+AANxAH4AFXEAfgAM")
    // check(g)(Enum.V2)(      "rO0ABXNyABVzY2FsYS5FbnVtZXJhdGlvbiRWYWzPaWevyfztTwIAAkkAGHNjYWxhJEVudW1lcmF0aW9uJFZhbCQkaUwABG5hbWV0ABJMamF2YS9sYW5nL1N0cmluZzt4cgAXc2NhbGEuRW51bWVyYXRpb24kVmFsdWViaXwv7SEdUQIAAkwABiRvdXRlcnQAE0xzY2FsYS9FbnVtZXJhdGlvbjtMABxzY2FsYSRFbnVtZXJhdGlvbiQkb3V0ZXJFbnVtcQB+AAN4cHNyAApUZXN0JEVudW0ketCIyQ8C23MCAAJMAAJWMXQAGUxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZTtMAAJWMnQAF0xzY2FsYS9FbnVtZXJhdGlvbiRWYWw7eHIAEXNjYWxhLkVudW1lcmF0aW9udaDN3ZgOWY4CAAhJAAZuZXh0SWRJABtzY2FsYSRFbnVtZXJhdGlvbiQkYm90dG9tSWRJABhzY2FsYSRFbnVtZXJhdGlvbiQkdG9wSWRMABRWYWx1ZU9yZGVyaW5nJG1vZHVsZXQAIkxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZU9yZGVyaW5nJDtMAA9WYWx1ZVNldCRtb2R1bGV0AB1Mc2NhbGEvRW51bWVyYXRpb24kVmFsdWVTZXQkO0wACG5leHROYW1ldAAbTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmF0b3I7TAAXc2NhbGEkRW51bWVyYXRpb24kJG5tYXB0AB5Mc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL01hcDtMABdzY2FsYSRFbnVtZXJhdGlvbiQkdm1hcHEAfgAMeHAAAAArAAAAAAAAACtwcHBzcgAgc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAAAAAAAAAAAQMAAHhwdw0AAALuAAAAAAAAAAQAeHNxAH4ADncNAAAC7gAAAAEAAAAEAHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAqcQB+AAR4c3IAEVRlc3QkRW51bSQkYW5vbiQxWUiOWYTWxdoCAAB4cQB+AAJxAH4ADXEAfgANcQB+AARxAH4ADQAAACpw")

    // IndexedSeqLike#Elements
    check(g)(immutable.Range(0, 1, 1).iterator)("rO0ABXNyAChzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5SYW5nZUl0ZXJhdG9yAAAAAAAAAAMCAARaAAhfaGFzTmV4dEkABV9uZXh0SQALbGFzdEVsZW1lbnRJAARzdGVweHABAAAAAAAAAAAAAAAB"
      , _.toList)

    // check(g)(new collection.concurrent.TrieMap[Any, Any]())( "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmNvbmN1cnJlbnQuVHJpZU1hcKckxpgOIYHPAwAETAALZXF1YWxpdHlvYmp0ABJMc2NhbGEvbWF0aC9FcXVpdjtMAApoYXNoaW5nb2JqdAAcTHNjYWxhL3V0aWwvaGFzaGluZy9IYXNoaW5nO0wABHJvb3R0ABJMamF2YS9sYW5nL09iamVjdDtMAAtyb290dXBkYXRlcnQAOUxqYXZhL3V0aWwvY29uY3VycmVudC9hdG9taWMvQXRvbWljUmVmZXJlbmNlRmllbGRVcGRhdGVyO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uY29uY3VycmVudC5UcmllTWFwJE1hbmdsZWRIYXNoaW5nhTBoJQ/mgb0CAAB4cHNyABhzY2FsYS5tYXRoLkVxdWl2JCRhbm9uJDLBbyx4dy/qGwIAAHhwc3IANHNjYWxhLmNvbGxlY3Rpb24uY29uY3VycmVudC5UcmllTWFwU2VyaWFsaXphdGlvbkVuZCSbjdgbbGCt2gIAAHhweA==")
    // not sure why this one needs stable serialization.

    check(g)(collection.accessWrappers.Wrappers)( "rO0ABXNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAMHNjYWxhLmNvbGxlY3Rpb24uY29udmVydC5KYXZhQ29sbGVjdGlvbldyYXBwZXJzJAAAAAAAAAADAgAAeHA=")

    check(g)(new SetWrapper(immutable.Set()))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vU2V0O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAoc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuU2V0JEVtcHR5U2V0JAAAAAAAAAADAgAAeHA=")
    check(g)(new SetWrapper(immutable.Set(1, 2, 3)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vU2V0O3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCRTZXQzAAAAAAAAAAMCAANMAAVlbGVtMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wABWVsZW0ycQB+AARMAAVlbGVtM3EAfgAEeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ABgAAAAJzcQB+AAYAAAAD")
    check(g)(new SetWrapper(mutable.Set()))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vU2V0O3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaFNldCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAAEHcEAAAAAHg=")
    check(g)(new SetWrapper(mutable.Set(1, 2, 3)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vU2V0O3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaFNldCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAACHcEAAAAA3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAIAAAAAnNxAH4ACAAAAAN4")
    check(g)(new MutableSetWrapper(mutable.Set()))("rO0ABXNyAEFzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNdXRhYmxlU2V0V3JhcHBlcgAAAAAAAAADAgABTAAKdW5kZXJseWluZ3QAHkxzY2FsYS9jb2xsZWN0aW9uL211dGFibGUvU2V0O3hyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vU2V0O3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaFNldCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAAEHcEAAAAAHhxAH4ABw==")
    check(g)(new MutableSetWrapper(mutable.Set(1, 2, 3)))("rO0ABXNyAEFzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNdXRhYmxlU2V0V3JhcHBlcgAAAAAAAAADAgABTAAKdW5kZXJseWluZ3QAHkxzY2FsYS9jb2xsZWN0aW9uL211dGFibGUvU2V0O3hyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRTZXRXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vU2V0O3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaFNldCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAACHcEAAAAA3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAKAAAAAnNxAH4ACgAAAAN4cQB+AAc=")
    check(g)(new MapWrapper(immutable.Map()))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAoc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTWFwJEVtcHR5TWFwJAAAAAAAAAADAgAAeHA=")
    check(g)(new MapWrapper(immutable.Map[Int, Int](1 -> 2)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLk1hcCRNYXAxAAAAAAAAAAMCAAJMAARrZXkxdAASTGphdmEvbGFuZy9PYmplY3Q7TAAGdmFsdWUxcQB+AAR4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAGAAAAAg==")
    check(g)(new MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLk1hcCRNYXAyAAAAAAAAAAMCAARMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMiQka2V5MXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAKXNjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXAyJCRrZXkycQB+AARMACtzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMiQkdmFsdWUxcQB+AARMACtzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMiQkdmFsdWUycQB+AAR4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAGAAAAAnEAfgAJc3EAfgAGAAAAAw==")
    check(g)(new MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLk1hcCRNYXAzAAAAAAAAAAMCAAZMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMyQka2V5MXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAKXNjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXAzJCRrZXkycQB+AARMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMyQka2V5M3EAfgAETAArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDMkJHZhbHVlMXEAfgAETAArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDMkJHZhbHVlMnEAfgAETAArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDMkJHZhbHVlM3EAfgAEeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ABgAAAAJzcQB+AAYAAAADcQB+AAlxAH4ACnNxAH4ABgAAAAQ=")
    check(g)(new MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLk1hcCRNYXA0AAAAAAAAAAMCAAhMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQka2V5MXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAKXNjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXA0JCRrZXkycQB+AARMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQka2V5M3EAfgAETAApc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDQkJGtleTRxAH4ABEwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXA0JCR2YWx1ZTFxAH4ABEwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXA0JCR2YWx1ZTJxAH4ABEwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXA0JCR2YWx1ZTNxAH4ABEwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXA0JCR2YWx1ZTRxAH4ABHhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAYAAAACc3EAfgAGAAAAA3NxAH4ABgAAAARxAH4ACXEAfgAKcQB+AAtzcQB+AAYAAAAF")
    check(g)(new MapWrapper(immutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5, 5 -> 6)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyACVzY2FsYS5jb2xsZWN0aW9uLk1hcEZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAdTHNjYWxhL2NvbGxlY3Rpb24vTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAI3NjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkhhc2hNYXAkAAAAAAAAAAMCAAB4cHcEAAAABXNyAAxzY2FsYS5UdXBsZTIB+93NIuc0egIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4AD3hwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAVzcQB+ABEAAAAGc3EAfgAOc3EAfgARAAAAAXNxAH4AEQAAAAJzcQB+AA5xAH4AF3NxAH4AEQAAAANzcQB+AA5xAH4AGXNxAH4AEQAAAARzcQB+AA5xAH4AG3EAfgATeA==")
    check(g)(new MapWrapper(mutable.Map()))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaE1hcCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAAEHcEAAAAAHg=")
    check(g)(new MapWrapper(mutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4)))("rO0ABXNyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaE1hcCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAACHcEAAAAA3NyAAxzY2FsYS5UdXBsZTIB+93NIuc0egIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4ACXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3EAfgAIcQB+AA5zcQB+AAsAAAADc3EAfgAIcQB+ABBzcQB+AAsAAAAEeA==")
    check(g)(new MutableMapWrapper(mutable.Map()))("rO0ABXNyAEFzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNdXRhYmxlTWFwV3JhcHBlcgAAAAAAAAADAgABTAAKdW5kZXJseWluZ3QAHkxzY2FsYS9jb2xsZWN0aW9uL211dGFibGUvTWFwO3hyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaE1hcCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAAEHcEAAAAAHhxAH4ABw==")
    check(g)(new MutableMapWrapper(mutable.Map[Int, Int](1 -> 2, 2 -> 3, 3 -> 4)))("rO0ABXNyAEFzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNdXRhYmxlTWFwV3JhcHBlcgAAAAAAAAADAgABTAAKdW5kZXJseWluZ3QAHkxzY2FsYS9jb2xsZWN0aW9uL211dGFibGUvTWFwO3hyADpzY2FsYS5jb2xsZWN0aW9uLmNvbnZlcnQuSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyAAAAAAAAAAMCAAFMAEZzY2FsYSRjb2xsZWN0aW9uJGNvbnZlcnQkSmF2YUNvbGxlY3Rpb25XcmFwcGVycyRNYXBXcmFwcGVyJCR1bmRlcmx5aW5ndAAWTHNjYWxhL2NvbGxlY3Rpb24vTWFwO3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uZ2VuZXJpYy5EZWZhdWx0U2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAFMAAdmYWN0b3J5dAAaTHNjYWxhL2NvbGxlY3Rpb24vRmFjdG9yeTt4cHNyADdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuSGFzaE1hcCREZXNlcmlhbGl6YXRpb25GYWN0b3J5AAAAAAAAAAMCAAJEAApsb2FkRmFjdG9ySQALdGFibGVMZW5ndGh4cD/oAAAAAAAAAAAACHcEAAAAA3NyAAxzY2FsYS5UdXBsZTIB+93NIuc0egIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4AC3hwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AA0AAAACc3EAfgAKcQB+ABBzcQB+AA0AAAADc3EAfgAKcQB+ABJzcQB+AA0AAAAEeHEAfgAH")

    case class ConstantHash(value: Int) {
      override def hashCode() = 1
    }

    check(g)(immutable.BitSet(1, 2, 3))( "rO0ABXNyADRzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5CaXRTZXQkU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMCAAB4cgAqc2NhbGEuY29sbGVjdGlvbi5CaXRTZXQkU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAMDAAB4cHcMAAAAAQAAAAAAAAAOeA==")
    check(g)(immutable.HashMap())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5IYXNoTWFwJAAAAAAAAAADAgAAeHB3BAAAAAB4")
    check(g)(immutable.HashMap(1 -> 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5IYXNoTWFwJAAAAAAAAAADAgAAeHB3BAAAAAFzcgAMc2NhbGEuVHVwbGUyAfvdzSLnNHoCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAOAAAAAng=")
    check(g)(immutable.HashMap(1 -> 2, 3 -> 4))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5IYXNoTWFwJAAAAAAAAAADAgAAeHB3BAAAAAJzcgAMc2NhbGEuVHVwbGUyAfvdzSLnNHoCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAOAAAAAnNxAH4AC3NxAH4ADgAAAANzcQB+AA4AAAAEeA==")
    check(g)(immutable.HashMap(ConstantHash(1) -> 1, ConstantHash(2) -> 1, ConstantHash(3) -> 1))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5IYXNoTWFwJAAAAAAAAAADAgAAeHB3BAAAAANzcgAMc2NhbGEuVHVwbGUyAfvdzSLnNHoCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyACtzY2FsYS5TZXJpYWxpemF0aW9uU3RhYmlsaXR5JENvbnN0YW50SGFzaCQxxIY3l6PJ0cgCAAFJAAV2YWx1ZXhwAAAAAXNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgALc3EAfgAOAAAAAnEAfgASc3EAfgALc3EAfgAOAAAAA3EAfgASeA==")

    check(g)(immutable.HashSet())(        "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaFNldCQAAAAAAAAAAwIAAHhwdwQAAAAAeA==")
    check(g)(immutable.HashSet(1))(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaFNldCQAAAAAAAAAAwIAAHhwdwQAAAABc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAF4")
    check(g)(immutable.HashSet(1, 2))(    "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaFNldCQAAAAAAAAAAwIAAHhwdwQAAAACc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACeA==")
    check(g)(immutable.HashSet(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaFNldCQAAAAAAAAAAwIAAHhwdwQAAAADc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAsAAAACc3EAfgALAAAAA3g=")
    check(g)(immutable.HashSet(ConstantHash(1), ConstantHash(1), ConstantHash(1)))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaFNldCQAAAAAAAAAAwIAAHhwdwQAAAABc3IAK3NjYWxhLlNlcmlhbGl6YXRpb25TdGFiaWxpdHkkQ29uc3RhbnRIYXNoJDHEhjeXo8nRyAIAAUkABXZhbHVleHAAAAABeA==")

    check(g)(immutable.ListSet())(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdFNldCQAAAAAAAAAAwIAAHhwdwQAAAAAeA==")
    check(g)(immutable.ListSet(1))(      "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuTGlzdFNldCQAAAAAAAAAAwIAAHhwdwT/////c3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAZ2cgAmc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLlNlcmlhbGl6ZUVuZCQAAAAAAAAAAwIAAHhweA==")
    check(g)(immutable.ListMap())(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0TWFwJAAAAAAAAAADAgAAeHB3BAAAAAB4")
    check(g)(immutable.ListMap(1 -> 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5MaXN0TWFwJAAAAAAAAAADAgAAeHB3BP////9zcgAMc2NhbGEuVHVwbGUyAfvdzSLnNHoCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgAOAAAAAnNxAH4ABnZyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")
    check(g)(immutable.Queue())(         "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAhc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuUXVldWUkAAAAAAAAAAMCAAB4cHcE/////3NxAH4ABnZyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")
    check(g)(immutable.Queue(1, 2, 3))(  "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAhc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuUXVldWUkAAAAAAAAAAMCAAB4cHcE/////3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgALAAAAAnNxAH4ACwAAAANzcQB+AAZ2cgAmc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLlNlcmlhbGl6ZUVuZCQAAAAAAAAAAwIAAHhweA==")

    check(g)(immutable.Range(0, 1, 1))( "rO0ABXNyACpzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5SYW5nZSRFeGNsdXNpdmUAAAAAAAAAAwIAAHhyACBzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5SYW5nZQAAAAAAAAADAgAGSQADZW5kWgAHaXNFbXB0eUkALXNjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFJhbmdlJCRsYXN0RWxlbWVudEkAMnNjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJFJhbmdlJCRudW1SYW5nZUVsZW1lbnRzSQAFc3RhcnRJAARzdGVweHAAAAABAAAAAAAAAAABAAAAAAAAAAE=")

    check(g)(immutable.Set())(              "rO0ABXNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAKHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlNldCRFbXB0eVNldCQAAAAAAAAAAwIAAHhw")
    check(g)(immutable.Set(1))(             "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TZXQkU2V0MQAAAAAAAAADAgABTAAFZWxlbTF0ABJMamF2YS9sYW5nL09iamVjdDt4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAB")
    check(g)(immutable.Set(1, 2))(          "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TZXQkU2V0MgAAAAAAAAADAgACTAAFZWxlbTF0ABJMamF2YS9sYW5nL09iamVjdDtMAAVlbGVtMnEAfgABeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4AAwAAAAI=")
    check(g)(immutable.Set(1, 2, 3))(       "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TZXQkU2V0MwAAAAAAAAADAgADTAAFZWxlbTF0ABJMamF2YS9sYW5nL09iamVjdDtMAAVlbGVtMnEAfgABTAAFZWxlbTNxAH4AAXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAMAAAACc3EAfgADAAAAAw==")
    check(g)(immutable.Set(1, 2, 3, 4))(    "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TZXQkU2V0NAAAAAAAAAADAgAETAAFZWxlbTF0ABJMamF2YS9sYW5nL09iamVjdDtMAAVlbGVtMnEAfgABTAAFZWxlbTNxAH4AAUwABWVsZW00cQB+AAF4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgADAAAAAnNxAH4AAwAAAANzcQB+AAMAAAAE")
    check(g)(immutable.Set(1, 2, 3, 4, 5))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuSGFzaFNldCQAAAAAAAAAAwIAAHhwdwQAAAAFc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAVzcQB+AAsAAAABc3EAfgALAAAAAnNxAH4ACwAAAANzcQB+AAsAAAAEeA==")

    check(g)(immutable.Map())(                                       "rO0ABXNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAKHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLk1hcCRFbXB0eU1hcCQAAAAAAAAAAwIAAHhw")
    check(g)(immutable.Map(1 -> 2))(                                 "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5NYXAkTWFwMQAAAAAAAAADAgACTAAEa2V5MXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wABnZhbHVlMXEAfgABeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4AAwAAAAI=")
    check(g)(immutable.Map(1 -> 2, 2 -> 3))(                         "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5NYXAkTWFwMgAAAAAAAAADAgAETAApc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDIkJGtleTF0ABJMamF2YS9sYW5nL09iamVjdDtMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMiQka2V5MnEAfgABTAArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDIkJHZhbHVlMXEAfgABTAArc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDIkJHZhbHVlMnEAfgABeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4AAwAAAAJxAH4ABnNxAH4AAwAAAAM=")
    check(g)(immutable.Map(1 -> 2, 2 -> 3, 3 -> 4))(                 "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5NYXAkTWFwMwAAAAAAAAADAgAGTAApc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDMkJGtleTF0ABJMamF2YS9sYW5nL09iamVjdDtMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwMyQka2V5MnEAfgABTAApc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDMkJGtleTNxAH4AAUwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXAzJCR2YWx1ZTFxAH4AAUwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXAzJCR2YWx1ZTJxAH4AAUwAK3NjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXAzJCR2YWx1ZTNxAH4AAXhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAMAAAACc3EAfgADAAAAA3EAfgAGcQB+AAdzcQB+AAMAAAAE")
    check(g)(immutable.Map(1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5))(         "rO0ABXNyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5NYXAkTWFwNAAAAAAAAAADAgAITAApc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDQkJGtleTF0ABJMamF2YS9sYW5nL09iamVjdDtMAClzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQka2V5MnEAfgABTAApc2NhbGEkY29sbGVjdGlvbiRpbW11dGFibGUkTWFwJE1hcDQkJGtleTNxAH4AAUwAKXNjYWxhJGNvbGxlY3Rpb24kaW1tdXRhYmxlJE1hcCRNYXA0JCRrZXk0cQB+AAFMACtzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQkdmFsdWUxcQB+AAFMACtzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQkdmFsdWUycQB+AAFMACtzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQkdmFsdWUzcQB+AAFMACtzY2FsYSRjb2xsZWN0aW9uJGltbXV0YWJsZSRNYXAkTWFwNCQkdmFsdWU0cQB+AAF4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgADAAAAAnNxAH4AAwAAAANzcQB+AAMAAAAEcQB+AAZxAH4AB3EAfgAIc3EAfgADAAAABQ==")
    check(g)(immutable.Map(1 -> 2, 2 -> 3, 3 -> 4, 4 -> 5, 5 -> 6))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5IYXNoTWFwJAAAAAAAAAADAgAAeHB3BAAAAAVzcgAMc2NhbGEuVHVwbGUyAfvdzSLnNHoCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAx4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAAFc3EAfgAOAAAABnNxAH4AC3NxAH4ADgAAAAFzcQB+AA4AAAACc3EAfgALcQB+ABRzcQB+AA4AAAADc3EAfgALcQB+ABZzcQB+AA4AAAAEc3EAfgALcQB+ABhxAH4AEHg=")

    check(g)(immutable.Stream(1, 2, 3): @nowarn)( "rO0ABXNyACZzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5TdHJlYW0kQ29ucwAAAAAAAAADAgADTAAEaGVhZHQAEkxqYXZhL2xhbmcvT2JqZWN0O0wABXRsR2VudAARTHNjYWxhL0Z1bmN0aW9uMDtMAAV0bFZhbHQAI0xzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9TdHJlYW07eHIAIXNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlN0cmVhbQAAAAAAAAADAgAAeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNyACFqYXZhLmxhbmcuaW52b2tlLlNlcmlhbGl6ZWRMYW1iZGFvYdCULCk2hQIACkkADmltcGxNZXRob2RLaW5kWwAMY2FwdHVyZWRBcmdzdAATW0xqYXZhL2xhbmcvT2JqZWN0O0wADmNhcHR1cmluZ0NsYXNzdAARTGphdmEvbGFuZy9DbGFzcztMABhmdW5jdGlvbmFsSW50ZXJmYWNlQ2xhc3N0ABJMamF2YS9sYW5nL1N0cmluZztMAB1mdW5jdGlvbmFsSW50ZXJmYWNlTWV0aG9kTmFtZXEAfgAMTAAiZnVuY3Rpb25hbEludGVyZmFjZU1ldGhvZFNpZ25hdHVyZXEAfgAMTAAJaW1wbENsYXNzcQB+AAxMAA5pbXBsTWV0aG9kTmFtZXEAfgAMTAATaW1wbE1ldGhvZFNpZ25hdHVyZXEAfgAMTAAWaW5zdGFudGlhdGVkTWV0aG9kVHlwZXEAfgAMeHAAAAAGdXIAE1tMamF2YS5sYW5nLk9iamVjdDuQzlifEHMpbAIAAHhwAAAAAXNyAC5zY2FsYS5jb2xsZWN0aW9uLkFycmF5T3BzJEFycmF5SXRlcmF0b3IkbWNJJHNwAAAAAAAAAAMCAAFbAAl4cyRtY0kkc3B0AAJbSXhyACdzY2FsYS5jb2xsZWN0aW9uLkFycmF5T3BzJEFycmF5SXRlcmF0b3IAAAAAAAAAAwIAA0kAA2xlbkkALHNjYWxhJGNvbGxlY3Rpb24kQXJyYXlPcHMkQXJyYXlJdGVyYXRvciQkcG9zTAACeHNxAH4AAXhwAAAAAwAAAAF1cgACW0lNumAmduqypQIAAHhwAAAAAwAAAAEAAAACAAAAA3EAfgAVdnIAInNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLlN0cmVhbSQAAAAAAAAAAwIAAHhwdAAPc2NhbGEvRnVuY3Rpb24wdAAFYXBwbHl0ABQoKUxqYXZhL2xhbmcvT2JqZWN0O3QAInNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL1N0cmVhbSR0ABckYW5vbmZ1biRmcm9tSXRlcmF0b3IkMXQAQChMc2NhbGEvY29sbGVjdGlvbi9JdGVyYXRvcjspTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL1N0cmVhbTt0ACUoKUxzY2FsYS9jb2xsZWN0aW9uL2ltbXV0YWJsZS9TdHJlYW07cA==")

    check(g)(Ordering.Int)("rO0ABXNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHA=")
    check(g)(immutable.TreeMap[Int, Int]())("rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDJ0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHBzcQB+AAd2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuVHJlZU1hcCQAAAAAAAAAAwIAAHhwdwQAAAAAeA==")
    check(g)(immutable.TreeMap(1 -> 1, 3 -> 6))("rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDJ0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHBzcQB+AAd2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuVHJlZU1hcCQAAAAAAAAAAwIAAHhwdwQAAAACc3IADHNjYWxhLlR1cGxlMgH73c0i5zR6AgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAQeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXEAfgAUc3EAfgAPc3EAfgASAAAAA3NxAH4AEgAAAAZ4")
    check(g)(immutable.TreeMap(1 -> 1, 3 -> 6).range(1, 2))("rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDJ0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHBzcQB+AAd2cgAjc2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuVHJlZU1hcCQAAAAAAAAAAwIAAHhwdwQAAAABc3IADHNjYWxhLlR1cGxlMgH73c0i5zR6AgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAQeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXEAfgAUeA==")
    check(g)(immutable.TreeSet[Int]())("rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkiz96UGruasMCAAB4cHNxAH4AB3ZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5UcmVlU2V0JAAAAAAAAAADAgAAeHB3BP////9zcQB+AAd2cgAmc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLlNlcmlhbGl6ZUVuZCQAAAAAAAAAAwIAAHhweA==")
    check(g)(immutable.TreeSet(1, 3))("rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkiz96UGruasMCAAB4cHNxAH4AB3ZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5UcmVlU2V0JAAAAAAAAAADAgAAeHB3BP////9zcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ADwAAAANzcQB+AAd2cgAmc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLlNlcmlhbGl6ZUVuZCQAAAAAAAAAAwIAAHhweA==")
    check(g)(immutable.TreeSet(1, 3).range(1, 2))("rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkiz96UGruasMCAAB4cHNxAH4AB3ZyACNzY2FsYS5jb2xsZWN0aW9uLmltbXV0YWJsZS5UcmVlU2V0JAAAAAAAAAADAgAAeHB3BP////9zcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4AB3ZyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")

    check(g)(mutable.ArrayBuffer(1, 2, 3))(      "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAlc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkFycmF5QnVmZmVyJAAAAAAAAAADAgAAeHB3BAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ACwAAAAJzcQB+AAsAAAADeA==")
    check(g)(mutable.ArraySeq(1, 2, 3))(         "rO0ABXNyACdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuQXJyYXlTZXEkb2ZJbnQAAAAAAAAAAwIAAVsABWFycmF5dAACW0l4cgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkFycmF5U2VxAAAAAAAAAAMCAAB4cHVyAAJbSU26YCZ26rKlAgAAeHAAAAADAAAAAQAAAAIAAAAD")
    check(g)(mutable.AnyRefMap("a" -> "A"))(       "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyAC1zY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuQW55UmVmTWFwJFRvRmFjdG9yeSQAAAAAAAAAAwIAAHhwdwQAAAABc3IADHNjYWxhLlR1cGxlMgH73c0i5zR6AgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAJeHB0AAFhdAABQXg=")

    check(g)(mutable.HashMap())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgA3c2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAkRGVzZXJpYWxpemF0aW9uRmFjdG9yeQAAAAAAAAADAgACRAAKbG9hZEZhY3RvckkAC3RhYmxlTGVuZ3RoeHA/6AAAAAAAAAAAABB3BAAAAAB4")
    check(g)(mutable.HashMap(1 -> 1))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgA3c2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hNYXAkRGVzZXJpYWxpemF0aW9uRmFjdG9yeQAAAAAAAAADAgACRAAKbG9hZEZhY3RvckkAC3RhYmxlTGVuZ3RoeHA/6AAAAAAAAAAAAAh3BAAAAAFzcgAMc2NhbGEuVHVwbGUyAfvdzSLnNHoCAAJMAAJfMXQAEkxqYXZhL2xhbmcvT2JqZWN0O0wAAl8ycQB+AAZ4cHNyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABcQB+AAp4")
    check(g)(mutable.HashSet(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgA3c2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkhhc2hTZXQkRGVzZXJpYWxpemF0aW9uRmFjdG9yeQAAAAAAAAADAgACRAAKbG9hZEZhY3RvckkAC3RhYmxlTGVuZ3RoeHA/6AAAAAAAAAAAAAh3BAAAAANzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ABQAAAAJzcQB+AAUAAAADeA==")
    check(g)(mutable.TreeMap[Int, Int]())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDJ0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHBzcQB+AAd2cgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlRyZWVNYXAkAAAAAAAAAAMCAAB4cHcEAAAAAHg=")
    check(g)(mutable.TreeMap(1 -> 1, 3 -> 6))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDJ0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHBzcQB+AAd2cgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlRyZWVNYXAkAAAAAAAAAAMCAAB4cHcEAAAAAnNyAAxzY2FsYS5UdXBsZTIB+93NIuc0egIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4AEHhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFxAH4AFHNxAH4AD3NxAH4AEgAAAANzcQB+ABIAAAAGeA==")
    check(g)(mutable.TreeMap(1 -> 1, 3 -> 6).range(1, 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgArc2NhbGEuY29sbGVjdGlvbi5Tb3J0ZWRNYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgACTAALZXZpZGVuY2UkNDJ0ABVMc2NhbGEvbWF0aC9PcmRlcmluZztMAAdmYWN0b3J5dAAjTHNjYWxhL2NvbGxlY3Rpb24vU29ydGVkTWFwRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAGHNjYWxhLm1hdGguT3JkZXJpbmckSW50JIs/elBq7mrDAgAAeHBzcQB+AAd2cgAhc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlRyZWVNYXAkAAAAAAAAAAMCAAB4cHcE/////3NyAAxzY2FsYS5UdXBsZTIB+93NIuc0egIAAkwAAl8xdAASTGphdmEvbGFuZy9PYmplY3Q7TAACXzJxAH4AEHhwc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFxAH4AFHNxAH4AB3ZyACZzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuU2VyaWFsaXplRW5kJAAAAAAAAAADAgAAeHB4")
    check(g)(mutable.TreeSet[Int]())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkiz96UGruasMCAAB4cHNxAH4AB3ZyACFzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuVHJlZVNldCQAAAAAAAAAAwIAAHhwdwQAAAAAeA==")
    check(g)(mutable.TreeSet(1, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkiz96UGruasMCAAB4cHNxAH4AB3ZyACFzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuVHJlZVNldCQAAAAAAAAAAwIAAHhwdwQAAAACc3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AA8AAAADeA==")
    check(g)(mutable.TreeSet(1, 3).range(1, 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAyc2NhbGEuY29sbGVjdGlvbi5FdmlkZW5jZUl0ZXJhYmxlRmFjdG9yeSRUb0ZhY3RvcnkAAAAAAAAAAwIAAkwAC2V2aWRlbmNlJDE1dAASTGphdmEvbGFuZy9PYmplY3Q7TAAHZmFjdG9yeXQAKkxzY2FsYS9jb2xsZWN0aW9uL0V2aWRlbmNlSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAYc2NhbGEubWF0aC5PcmRlcmluZyRJbnQkiz96UGruasMCAAB4cHNxAH4AB3ZyACFzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuVHJlZVNldCQAAAAAAAAAAwIAAHhwdwT/////c3IAEWphdmEubGFuZy5JbnRlZ2VyEuKgpPeBhzgCAAFJAAV2YWx1ZXhyABBqYXZhLmxhbmcuTnVtYmVyhqyVHQuU4IsCAAB4cAAAAAFzcQB+AAd2cgAmc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLlNlcmlhbGl6ZUVuZCQAAAAAAAAAAwIAAHhweA==")
    check(g)(mutable.LinkedHashMap(1 -> 2))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAlc2NhbGEuY29sbGVjdGlvbi5NYXBGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAHUxzY2FsYS9jb2xsZWN0aW9uL01hcEZhY3Rvcnk7eHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyACdzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuTGlua2VkSGFzaE1hcCQAAAAAAAAAAwIAAHhwdwQAAAABc3IADHNjYWxhLlR1cGxlMgH73c0i5zR6AgACTAACXzF0ABJMamF2YS9sYW5nL09iamVjdDtMAAJfMnEAfgAMeHBzcgARamF2YS5sYW5nLkludGVnZXIS4qCk94GHOAIAAUkABXZhbHVleHIAEGphdmEubGFuZy5OdW1iZXKGrJUdC5TgiwIAAHhwAAAAAXNxAH4ADgAAAAJ4")
    check(g)(mutable.LinkedHashSet(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAnc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkxpbmtlZEhhc2hTZXQkAAAAAAAAAAMCAAB4cHcEAAAAA3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgALAAAAAnNxAH4ACwAAAAN4")

    check(g)(mutable.ListBuffer(1, 2, 3))( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAkc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLkxpc3RCdWZmZXIkAAAAAAAAAAMCAAB4cHcEAAAAA3NyABFqYXZhLmxhbmcuSW50ZWdlchLioKT3gYc4AgABSQAFdmFsdWV4cgAQamF2YS5sYW5nLk51bWJlcoaslR0LlOCLAgAAeHAAAAABc3EAfgALAAAAAnNxAH4ACwAAAAN4")
    check(g)(new mutable.StringBuilder(new java.lang.StringBuilder("123")))( "rO0ABXNyACZzY2FsYS5jb2xsZWN0aW9uLm11dGFibGUuU3RyaW5nQnVpbGRlcgAAAAAAAAADAgABTAAKdW5kZXJseWluZ3QAGUxqYXZhL2xhbmcvU3RyaW5nQnVpbGRlcjt4cHNyABdqYXZhLmxhbmcuU3RyaW5nQnVpbGRlcjzV+xRaTGrLAwAAeHB3BAAAAAN1cgACW0OwJmaw4l2ErAIAAHhwAAAAEwAxADIAMwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAeA==")
    check(g)(mutable.UnrolledBuffer[Int]())( "rO0ABXNyADJzY2FsYS5jb2xsZWN0aW9uLmdlbmVyaWMuRGVmYXVsdFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwABTAAHZmFjdG9yeXQAGkxzY2FsYS9jb2xsZWN0aW9uL0ZhY3Rvcnk7eHBzcgAqc2NhbGEuY29sbGVjdGlvbi5JdGVyYWJsZUZhY3RvcnkkVG9GYWN0b3J5AAAAAAAAAAMCAAFMAAdmYWN0b3J5dAAiTHNjYWxhL2NvbGxlY3Rpb24vSXRlcmFibGVGYWN0b3J5O3hwc3IAMnNjYWxhLmNvbGxlY3Rpb24uQ2xhc3NUYWdTZXFGYWN0b3J5JEFueVNlcURlbGVnYXRlAAAAAAAAAAMCAAB4cgA8c2NhbGEuY29sbGVjdGlvbi5DbGFzc1RhZ0l0ZXJhYmxlRmFjdG9yeSRBbnlJdGVyYWJsZURlbGVnYXRlAAAAAAAAAAMCAAFMAAhkZWxlZ2F0ZXQAKkxzY2FsYS9jb2xsZWN0aW9uL0NsYXNzVGFnSXRlcmFibGVGYWN0b3J5O3hwc3IAJnNjYWxhLnJ1bnRpbWUuTW9kdWxlU2VyaWFsaXphdGlvblByb3h5AAAAAAAAAAECAAFMAAttb2R1bGVDbGFzc3QAEUxqYXZhL2xhbmcvQ2xhc3M7eHB2cgAoc2NhbGEuY29sbGVjdGlvbi5tdXRhYmxlLlVucm9sbGVkQnVmZmVyJAAAAAAAAAADAgAAeHB3BAAAAAB4")

    check(g)(AnEnum.X)("rO0ABXNyABVzY2FsYS5FbnVtZXJhdGlvbiRWYWzPaWevyfztTwIAAkkAAWlMAARuYW1ldAASTGphdmEvbGFuZy9TdHJpbmc7eHIAF3NjYWxhLkVudW1lcmF0aW9uJFZhbHVlYml8L+0hHVECAAJMAAYkb3V0ZXJ0ABNMc2NhbGEvRW51bWVyYXRpb247TAAcc2NhbGEkRW51bWVyYXRpb24kJG91dGVyRW51bXEAfgADeHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyAA1zY2FsYS5BbkVudW0kpusjI6FdLmYCAAB4cgARc2NhbGEuRW51bWVyYXRpb251oM3dmA5ZjgIACEkABm5leHRJZEkAG3NjYWxhJEVudW1lcmF0aW9uJCRib3R0b21JZEkAGHNjYWxhJEVudW1lcmF0aW9uJCR0b3BJZEwAFFZhbHVlT3JkZXJpbmckbW9kdWxldAAiTHNjYWxhL0VudW1lcmF0aW9uJFZhbHVlT3JkZXJpbmckO0wAD1ZhbHVlU2V0JG1vZHVsZXQAHUxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZVNldCQ7TAAIbmV4dE5hbWV0ABtMc2NhbGEvY29sbGVjdGlvbi9JdGVyYXRvcjtMAARubWFwdAAeTHNjYWxhL2NvbGxlY3Rpb24vbXV0YWJsZS9NYXA7TAAXc2NhbGEkRW51bWVyYXRpb24kJHZtYXBxAH4ADXhwcQB+AAcAAAAAcA==")
    check(g)(AnEnum.ValueSet(AnEnum.X))("rO0ABXNyABpzY2FsYS5FbnVtZXJhdGlvbiRWYWx1ZVNldGRU86cAWE+SAgACTAAGJG91dGVydAATTHNjYWxhL0VudW1lcmF0aW9uO0wABW5uSWRzdAAjTHNjYWxhL2NvbGxlY3Rpb24vaW1tdXRhYmxlL0JpdFNldDt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIADXNjYWxhLkFuRW51bSSm6yMjoV0uZgIAAHhyABFzY2FsYS5FbnVtZXJhdGlvbnWgzd2YDlmOAgAISQAGbmV4dElkSQAbc2NhbGEkRW51bWVyYXRpb24kJGJvdHRvbUlkSQAYc2NhbGEkRW51bWVyYXRpb24kJHRvcElkTAAUVmFsdWVPcmRlcmluZyRtb2R1bGV0ACJMc2NhbGEvRW51bWVyYXRpb24kVmFsdWVPcmRlcmluZyQ7TAAPVmFsdWVTZXQkbW9kdWxldAAdTHNjYWxhL0VudW1lcmF0aW9uJFZhbHVlU2V0JDtMAAhuZXh0TmFtZXQAG0xzY2FsYS9jb2xsZWN0aW9uL0l0ZXJhdG9yO0wABG5tYXB0AB5Mc2NhbGEvY29sbGVjdGlvbi9tdXRhYmxlL01hcDtMABdzY2FsYSRFbnVtZXJhdGlvbiQkdm1hcHEAfgAMeHBzcgA0c2NhbGEuY29sbGVjdGlvbi5pbW11dGFibGUuQml0U2V0JFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAgAAeHIAKnNjYWxhLmNvbGxlY3Rpb24uQml0U2V0JFNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAADAwAAeHB3DAAAAAEAAAAAAAAAAXg=")
    check(g)(AnEnum.X)("rO0ABXNyABVzY2FsYS5FbnVtZXJhdGlvbiRWYWzPaWevyfztTwIAAkkAAWlMAARuYW1ldAASTGphdmEvbGFuZy9TdHJpbmc7eHIAF3NjYWxhLkVudW1lcmF0aW9uJFZhbHVlYml8L+0hHVECAAJMAAYkb3V0ZXJ0ABNMc2NhbGEvRW51bWVyYXRpb247TAAcc2NhbGEkRW51bWVyYXRpb24kJG91dGVyRW51bXEAfgADeHBzcgAmc2NhbGEucnVudGltZS5Nb2R1bGVTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAQIAAUwAC21vZHVsZUNsYXNzdAARTGphdmEvbGFuZy9DbGFzczt4cHZyAA1zY2FsYS5BbkVudW0kpusjI6FdLmYCAAB4cgARc2NhbGEuRW51bWVyYXRpb251oM3dmA5ZjgIACEkABm5leHRJZEkAG3NjYWxhJEVudW1lcmF0aW9uJCRib3R0b21JZEkAGHNjYWxhJEVudW1lcmF0aW9uJCR0b3BJZEwAFFZhbHVlT3JkZXJpbmckbW9kdWxldAAiTHNjYWxhL0VudW1lcmF0aW9uJFZhbHVlT3JkZXJpbmckO0wAD1ZhbHVlU2V0JG1vZHVsZXQAHUxzY2FsYS9FbnVtZXJhdGlvbiRWYWx1ZVNldCQ7TAAIbmV4dE5hbWV0ABtMc2NhbGEvY29sbGVjdGlvbi9JdGVyYXRvcjtMAARubWFwdAAeTHNjYWxhL2NvbGxlY3Rpb24vbXV0YWJsZS9NYXA7TAAXc2NhbGEkRW51bWVyYXRpb24kJHZtYXBxAH4ADXhwcQB+AAcAAAAAcA==") // scala/bug#11919-ish

    check(g)("...".r)("rO0ABXNyABlzY2FsYS51dGlsLm1hdGNoaW5nLlJlZ2V44u3Vap7wIb8CAAJMAAdwYXR0ZXJudAAZTGphdmEvdXRpbC9yZWdleC9QYXR0ZXJuO0wAJXNjYWxhJHV0aWwkbWF0Y2hpbmckUmVnZXgkJGdyb3VwTmFtZXN0ACBMc2NhbGEvY29sbGVjdGlvbi9pbW11dGFibGUvU2VxO3hwc3IAF2phdmEudXRpbC5yZWdleC5QYXR0ZXJuRmfVa25JAg0CAAJJAAVmbGFnc0wAB3BhdHRlcm50ABJMamF2YS9sYW5nL1N0cmluZzt4cAAAAAB0AAMuLi5zcgAyc2NhbGEuY29sbGVjdGlvbi5nZW5lcmljLkRlZmF1bHRTZXJpYWxpemF0aW9uUHJveHkAAAAAAAAAAwMAAUwAB2ZhY3Rvcnl0ABpMc2NhbGEvY29sbGVjdGlvbi9GYWN0b3J5O3hwc3IAKnNjYWxhLmNvbGxlY3Rpb24uSXRlcmFibGVGYWN0b3J5JFRvRmFjdG9yeQAAAAAAAAADAgABTAAHZmFjdG9yeXQAIkxzY2FsYS9jb2xsZWN0aW9uL0l0ZXJhYmxlRmFjdG9yeTt4cHNyACZzY2FsYS5ydW50aW1lLk1vZHVsZVNlcmlhbGl6YXRpb25Qcm94eQAAAAAAAAABAgABTAALbW9kdWxlQ2xhc3N0ABFMamF2YS9sYW5nL0NsYXNzO3hwdnIAIHNjYWxhLmNvbGxlY3Rpb24uaW1tdXRhYmxlLkxpc3QkAAAAAAAAAAMCAAB4cHcEAAAAAHg=",
      r => (r.toString))
  }
}

object AnEnum extends Enumeration {
  val X, Y = Value
}

class SerializationStabilityTest {
  @Test
  def testAll(): Unit = SerializationStability.main(new Array[String](0))
}
