/*
 * SonarQube
 * Copyright (C) 2009-2020 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
import * as React from 'react';
import BoxedTabs from 'sonar-ui-common/components/controls/BoxedTabs';
import DeferredSpinner from 'sonar-ui-common/components/ui/DeferredSpinner';
import { translate } from 'sonar-ui-common/helpers/l10n';
import { isDiffMetric } from 'sonar-ui-common/helpers/measures';
import { rawSizes } from '../../../app/theme';
import { findMeasure } from '../../../helpers/measures';
import { ApplicationPeriod } from '../../../types/application';
import { BranchLike } from '../../../types/branch-like';
import { ComponentQualifier } from '../../../types/component';
import { MetricKey } from '../../../types/metrics';
import MeasurementLabel from '../components/MeasurementLabel';
import { IssueType, MeasurementType } from '../utils';
import { DrilldownMeasureValue } from './DrilldownMeasureValue';
import { LeakPeriodInfo } from './LeakPeriodInfo';
import MeasuresPanelIssueMeasureRow from './MeasuresPanelIssueMeasureRow';
import MeasuresPanelNoNewCode from './MeasuresPanelNoNewCode';

export interface MeasuresPanelProps {
  appLeak?: ApplicationPeriod;
  branchLike?: BranchLike;
  component: T.Component;
  loading?: boolean;
  measures?: T.MeasureEnhanced[];
  period?: T.Period;
}

export enum MeasuresPanelTabs {
  New,
  Overall
}

export function MeasuresPanel(props: MeasuresPanelProps) {
  const { appLeak, branchLike, component, loading, measures = [], period } = props;

  const hasDiffMeasures = measures.some(m => isDiffMetric(m.metric.key));
  const isApp = component.qualifier === ComponentQualifier.Application;
  const leakPeriod = isApp ? appLeak : period;

  const [tab, selectTab] = React.useState(MeasuresPanelTabs.New);

  const isNewCodeTab = tab === MeasuresPanelTabs.New;

  React.useEffect(() => {
    // Open Overall tab by default if there are no new measures.
    if (loading === false && !hasDiffMeasures && isNewCodeTab) {
      selectTab(MeasuresPanelTabs.Overall);
    }
    // In this case, we explicitly do NOT want to mark tab as a dependency, as
    // it would prevent the user from selecting it, even if it's empty.
    /* eslint-disable-next-line react-hooks/exhaustive-deps */
  }, [loading, hasDiffMeasures]);

  const tabs = [
    {
      key: MeasuresPanelTabs.New,
      label: (
        <div className="text-left overview-measures-tab">
          <span className="text-bold">{translate('overview.new_code')}</span>
          {leakPeriod && <LeakPeriodInfo leakPeriod={leakPeriod} />}
        </div>
      )
    },
    {
      key: MeasuresPanelTabs.Overall,
      label: (
        <div className="text-left overview-measures-tab">
          <span className="text-bold" style={{ position: 'absolute', top: 2 * rawSizes.grid }}>
            {translate('overview.overall_code')}
          </span>
        </div>
      )
    }
  ];

  return (
    <div className="overview-panel" data-test="overview__measures-panel">
      <h2 className="overview-panel-title">{translate('overview.measures')}</h2>

      {loading ? (
        <div className="overview-panel-content overview-panel-big-padded">
          <DeferredSpinner loading={loading} />
        </div>
      ) : (
        <>
          <BoxedTabs onSelect={selectTab} selected={tab} tabs={tabs} />

          <div className="overview-panel-content flex-1 bordered">
            {!hasDiffMeasures && isNewCodeTab ? (
              <MeasuresPanelNoNewCode
                branchLike={branchLike}
                component={component}
                period={period}
              />
            ) : (
              <>
                {[
                  IssueType.Bug,
                  IssueType.Vulnerability,
                  IssueType.SecurityHotspot,
                  IssueType.CodeSmell
                ].map((type: IssueType) => (
                  <MeasuresPanelIssueMeasureRow
                    branchLike={branchLike}
                    component={component}
                    isNewCodeTab={isNewCodeTab}
                    key={type}
                    measures={measures}
                    type={type}
                  />
                ))}

                <div className="display-flex-row overview-measures-row">
                  {(findMeasure(measures, MetricKey.coverage) ||
                    findMeasure(measures, MetricKey.new_coverage)) && (
                    <div
                      className="overview-panel-huge-padded flex-1 bordered-right display-flex-center"
                      data-test="overview__measures-coverage">
                      <MeasurementLabel
                        branchLike={branchLike}
                        centered={isNewCodeTab}
                        component={component}
                        measures={measures}
                        type={MeasurementType.Coverage}
                        useDiffMetric={isNewCodeTab}
                      />

                      {tab === MeasuresPanelTabs.Overall && (
                        <div className="huge-spacer-left">
                          <DrilldownMeasureValue
                            branchLike={branchLike}
                            component={component}
                            measures={measures}
                            metric={MetricKey.tests}
                          />
                        </div>
                      )}
                    </div>
                  )}
                  <div className="overview-panel-huge-padded flex-1 display-flex-center">
                    <MeasurementLabel
                      branchLike={branchLike}
                      centered={isNewCodeTab}
                      component={component}
                      measures={measures}
                      type={MeasurementType.Duplication}
                      useDiffMetric={isNewCodeTab}
                    />

                    {tab === MeasuresPanelTabs.Overall && (
                      <div className="huge-spacer-left">
                        <DrilldownMeasureValue
                          branchLike={branchLike}
                          component={component}
                          measures={measures}
                          metric={MetricKey.duplicated_blocks}
                        />
                      </div>
                    )}
                  </div>
                </div>
              </>
            )}
          </div>
        </>
      )}
    </div>
  );
}

export default React.memo(MeasuresPanel);
