/*
 * SonarQube
 * Copyright (C) 2009-2020 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.db.permission.template;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.sonar.api.utils.System2;
import org.sonar.api.web.UserRole;
import org.sonar.core.util.UuidFactory;
import org.sonar.core.util.UuidFactoryFast;
import org.sonar.db.DbSession;
import org.sonar.db.DbTester;
import org.sonar.db.organization.OrganizationDto;
import org.sonar.db.user.GroupDto;
import org.sonar.db.user.UserDto;

import static java.util.Arrays.asList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.tuple;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.sonar.api.web.UserRole.ADMIN;
import static org.sonar.api.web.UserRole.CODEVIEWER;
import static org.sonar.api.web.UserRole.ISSUE_ADMIN;
import static org.sonar.api.web.UserRole.USER;
import static org.sonar.core.permission.GlobalPermissions.SCAN_EXECUTION;
import static org.sonar.db.permission.template.PermissionTemplateTesting.newPermissionTemplateDto;
import static org.sonar.db.user.GroupTesting.newGroupDto;

public class PermissionTemplateDaoTest {

  private static final Date PAST = new Date(100_000_000_000L);
  private static final Date NOW = new Date(500_000_000_000L);

  @Rule
  public ExpectedException expectedException = ExpectedException.none();
  @Rule
  public DbTester db = DbTester.create();

  private System2 system2 = mock(System2.class);
  private DbSession dbSession = db.getSession();
  private PermissionTemplateDbTester templateDb = db.permissionTemplates();
  private UuidFactory uuidFactory = UuidFactoryFast.getInstance();

  private PermissionTemplateDao underTest = new PermissionTemplateDao(uuidFactory, system2);

  @Before
  public void setUp() {
    when(system2.now()).thenReturn(NOW.getTime());
  }

  @Test
  public void create_permission_template() {
    PermissionTemplateDto permissionTemplate = underTest.insert(db.getSession(), newPermissionTemplateDto()
      .setUuid("ABCD")
      .setName("my template")
      .setDescription("my description")
      .setKeyPattern("myregexp")
      .setOrganizationUuid("org")
      .setCreatedAt(PAST)
      .setUpdatedAt(NOW));
    db.commit();

    assertThat(underTest.selectByUuid(db.getSession(), permissionTemplate.getUuid()))
      .extracting(PermissionTemplateDto::getUuid, PermissionTemplateDto::getName, PermissionTemplateDto::getDescription, PermissionTemplateDto::getKeyPattern,
        PermissionTemplateDto::getOrganizationUuid, PermissionTemplateDto::getCreatedAt, PermissionTemplateDto::getUpdatedAt)
      .containsOnly("ABCD", "my template", "my description", "myregexp", "org", PAST, NOW);
  }

  @Test
  public void select_permission_template_by_uuid() {
    templateDb.insertTemplate(newPermissionTemplateDto()
      .setUuid("ABCD")
      .setName("my template")
      .setDescription("my description")
      .setKeyPattern("myregexp")
      .setOrganizationUuid("org"));

    assertThat(underTest.selectByUuid(db.getSession(), "ABCD"))
      .extracting(PermissionTemplateDto::getUuid, PermissionTemplateDto::getName, PermissionTemplateDto::getDescription, PermissionTemplateDto::getKeyPattern,
        PermissionTemplateDto::getOrganizationUuid)
      .containsOnly("ABCD", "my template", "my description", "myregexp", "org");
  }

  @Test
  public void selectAll_without_name_filtering() {
    templateDb.insertTemplate(newPermissionTemplateDto()
      .setUuid("tpl1")
      .setName("template1")
      .setDescription("description1")
      .setOrganizationUuid("org"));
    templateDb.insertTemplate(newPermissionTemplateDto()
      .setUuid("tpl2")
      .setName("template2")
      .setDescription("description2")
      .setOrganizationUuid("org"));
    templateDb.insertTemplate(newPermissionTemplateDto()
      .setUuid("tpl3")
      .setName("template3")
      .setDescription("description3")
      .setOrganizationUuid("org"));

    assertThat(underTest.selectAll(dbSession, "org", null))
      .extracting(PermissionTemplateDto::getUuid, PermissionTemplateDto::getName, PermissionTemplateDto::getDescription)
      .containsOnly(
        tuple("tpl1", "template1", "description1"),
        tuple("tpl2", "template2", "description2"),
        tuple("tpl3", "template3", "description3"));
    assertThat(underTest.selectAll(dbSession, "missingOrg", null)).isEmpty();
  }

  @Test
  public void selectAll_with_name_filtering() {
    PermissionTemplateDto t1InOrg1 = templateDb.insertTemplate(newPermissionTemplateDto().setName("aBcDeF").setOrganizationUuid("org1"));
    PermissionTemplateDto t2InOrg1 = templateDb.insertTemplate(newPermissionTemplateDto().setName("cdefgh").setOrganizationUuid("org1"));
    PermissionTemplateDto t3InOrg1 = templateDb.insertTemplate(newPermissionTemplateDto().setName("hijkl").setOrganizationUuid("org2"));
    PermissionTemplateDto t4InOrg2 = templateDb.insertTemplate(newPermissionTemplateDto().setName("cdefgh").setOrganizationUuid("org2"));

    assertThat(underTest.selectAll(dbSession, "org1", "def")).extracting(PermissionTemplateDto::getUuid).containsExactly(t1InOrg1.getUuid(), t2InOrg1.getUuid());
    assertThat(underTest.selectAll(dbSession, "org1", "missing")).isEmpty();
  }

  @Test
  public void should_update_permission_template() {
    PermissionTemplateDto permissionTemplateDto = templateDb.insertTemplate(newPermissionTemplateDto()
      .setUuid("ABCD")
      .setName("name")
      .setDescription("description")
      .setKeyPattern("regexp")
      .setOrganizationUuid("org")
      .setCreatedAt(PAST)
      .setUpdatedAt(PAST));

    underTest.update(dbSession, permissionTemplateDto
      .setName("new_name")
      .setDescription("new_description")
      .setKeyPattern("new_regexp")
      .setUpdatedAt(NOW)
      // Invariant fields, should not be updated
      .setUuid("ABCD")
      .setOrganizationUuid("new org")
      .setCreatedAt(NOW));
    db.commit();

    assertThat(underTest.selectByUuid(db.getSession(), "ABCD"))
      .extracting(PermissionTemplateDto::getUuid, PermissionTemplateDto::getName, PermissionTemplateDto::getDescription, PermissionTemplateDto::getKeyPattern,
        PermissionTemplateDto::getOrganizationUuid, PermissionTemplateDto::getCreatedAt, PermissionTemplateDto::getUpdatedAt)
      .containsOnly("ABCD", "new_name", "new_description", "new_regexp", "org", PAST, NOW);
  }

  @Test
  public void delete_permission_template() {
    UserDto user1 = db.users().insertUser();
    UserDto user2 = db.users().insertUser();
    GroupDto group1 = db.users().insertGroup();
    GroupDto group2 = db.users().insertGroup();
    PermissionTemplateDto permissionTemplate1 = templateDb.insertTemplate(db.getDefaultOrganization());
    PermissionTemplateDto permissionTemplate2 = templateDb.insertTemplate(db.getDefaultOrganization());
    templateDb.addUserToTemplate(permissionTemplate1, user1, "user");
    templateDb.addUserToTemplate(permissionTemplate1, user2, "user");
    templateDb.addUserToTemplate(permissionTemplate1, user2, "admin");
    templateDb.addUserToTemplate(permissionTemplate2, user2, "admin");
    templateDb.addGroupToTemplate(permissionTemplate1, group1, "user");
    templateDb.addGroupToTemplate(permissionTemplate1, group2, "user");
    templateDb.addAnyoneToTemplate(permissionTemplate1, "admin");
    templateDb.addAnyoneToTemplate(permissionTemplate2, "admin");
    templateDb.addProjectCreatorToTemplate(permissionTemplate1.getUuid(), "user");
    templateDb.addProjectCreatorToTemplate(permissionTemplate2.getUuid(), "user");

    underTest.deleteByUuid(dbSession, permissionTemplate1.getUuid());
    dbSession.commit();

    assertThat(underTest.selectAll(db.getSession(), db.getDefaultOrganization().getUuid(), null))
      .extracting(PermissionTemplateDto::getUuid)
      .containsOnly(permissionTemplate2.getUuid());
    assertThat(db.getDbClient().permissionTemplateDao().selectUserPermissionsByTemplateId(db.getSession(), permissionTemplate1.getUuid())).isEmpty();
    assertThat(db.getDbClient().permissionTemplateDao().selectUserPermissionsByTemplateId(db.getSession(), permissionTemplate2.getUuid())).hasSize(1);
    assertThat(db.getDbClient().permissionTemplateDao().selectGroupPermissionsByTemplateUuid(db.getSession(), permissionTemplate1.getUuid())).isEmpty();
    assertThat(db.getDbClient().permissionTemplateDao().selectGroupPermissionsByTemplateUuid(db.getSession(), permissionTemplate2.getUuid())).hasSize(1);
    assertThat(db.getDbClient().permissionTemplateCharacteristicDao().selectByTemplateUuids(db.getSession(), asList(permissionTemplate1.getUuid(), permissionTemplate2.getUuid())))
      .extracting(PermissionTemplateCharacteristicDto::getTemplateUuid)
      .containsOnly(permissionTemplate2.getUuid());
  }

  @Test
  public void add_user_permission_to_template() {
    PermissionTemplateDto permissionTemplate = templateDb.insertTemplate(db.getDefaultOrganization());
    UserDto user = db.users().insertUser();

    underTest.insertUserPermission(dbSession, permissionTemplate.getUuid(), user.getUuid(), "user");

    assertThat(db.getDbClient().permissionTemplateDao().selectUserPermissionsByTemplateId(db.getSession(), permissionTemplate.getUuid()))
      .extracting(PermissionTemplateUserDto::getTemplateUuid, PermissionTemplateUserDto::getUserUuid, PermissionTemplateUserDto::getPermission,
        PermissionTemplateUserDto::getCreatedAt,
        PermissionTemplateUserDto::getUpdatedAt)
      .containsOnly(tuple(permissionTemplate.getUuid(), user.getUuid(), "user", NOW, NOW));
  }

  @Test
  public void remove_user_permission_from_template() {
    PermissionTemplateDto permissionTemplate = templateDb.insertTemplate(db.getDefaultOrganization());
    UserDto user1 = db.users().insertUser();
    UserDto user2 = db.users().insertUser();
    templateDb.addUserToTemplate(permissionTemplate, user1, "user");
    templateDb.addUserToTemplate(permissionTemplate, user1, "admin");
    templateDb.addUserToTemplate(permissionTemplate, user2, "user");

    underTest.deleteUserPermission(dbSession, permissionTemplate.getUuid(), user1.getUuid(), "user");

    assertThat(db.getDbClient().permissionTemplateDao().selectUserPermissionsByTemplateId(db.getSession(), permissionTemplate.getUuid()))
      .extracting(PermissionTemplateUserDto::getUserUuid, PermissionTemplateUserDto::getPermission)
      .containsOnly(tuple(user1.getUuid(), "admin"), tuple(user2.getUuid(), "user"));
  }

  @Test
  public void add_group_permission_to_template() {
    PermissionTemplateDto permissionTemplate = templateDb.insertTemplate(db.getDefaultOrganization());
    GroupDto group = db.users().insertGroup();

    underTest.insertGroupPermission(dbSession, permissionTemplate.getUuid(), group.getUuid(), "user");
    dbSession.commit();

    assertThat(db.getDbClient().permissionTemplateDao().selectGroupPermissionsByTemplateUuid(db.getSession(), permissionTemplate.getUuid()))
      .extracting(PermissionTemplateGroupDto::getTemplateUuid, PermissionTemplateGroupDto::getGroupUuid, PermissionTemplateGroupDto::getPermission,
        PermissionTemplateGroupDto::getCreatedAt,
        PermissionTemplateGroupDto::getUpdatedAt)
      .containsOnly(tuple(permissionTemplate.getUuid(), group.getUuid(), "user", NOW, NOW));
  }

  @Test
  public void remove_by_group() {
    PermissionTemplateDto permissionTemplate = templateDb.insertTemplate(db.getDefaultOrganization());
    GroupDto group1 = db.users().insertGroup();
    GroupDto group2 = db.users().insertGroup();
    templateDb.addGroupToTemplate(permissionTemplate, group1, "user");
    templateDb.addGroupToTemplate(permissionTemplate, group1, "admin");
    templateDb.addGroupToTemplate(permissionTemplate, group2, "user");

    underTest.deleteByGroup(db.getSession(), group1.getUuid());
    db.getSession().commit();

    assertThat(db.getDbClient().permissionTemplateDao().selectGroupPermissionsByTemplateUuid(db.getSession(), permissionTemplate.getUuid()))
      .extracting(PermissionTemplateGroupDto::getGroupUuid, PermissionTemplateGroupDto::getPermission)
      .containsOnly(tuple(group2.getUuid(), "user"));
  }

  @Test
  public void add_group_permission_to_anyone() {
    PermissionTemplateDto permissionTemplate = templateDb.insertTemplate(db.getDefaultOrganization());

    underTest.insertGroupPermission(dbSession, permissionTemplate.getUuid(), null, "user");
    dbSession.commit();

    assertThat(db.getDbClient().permissionTemplateDao().selectGroupPermissionsByTemplateUuid(db.getSession(), permissionTemplate.getUuid()))
      .extracting(PermissionTemplateGroupDto::getTemplateUuid, PermissionTemplateGroupDto::getGroupUuid, PermissionTemplateGroupDto::getGroupName,
        PermissionTemplateGroupDto::getPermission)
      .containsOnly(tuple(permissionTemplate.getUuid(), "Anyone", "Anyone", "user"));
  }

  @Test
  public void group_count_by_template_and_permission() {
    PermissionTemplateDto template1 = templateDb.insertTemplate();
    PermissionTemplateDto template2 = templateDb.insertTemplate();
    PermissionTemplateDto template3 = templateDb.insertTemplate();
    PermissionTemplateDto template4 = templateDb.insertTemplate();
    GroupDto group1 = db.users().insertGroup(newGroupDto());
    GroupDto group2 = db.users().insertGroup(newGroupDto());
    GroupDto group3 = db.users().insertGroup(newGroupDto());
    templateDb.addGroupToTemplate(template1.getUuid(), group1.getUuid(), CODEVIEWER);
    templateDb.addGroupToTemplate(template1.getUuid(), group2.getUuid(), CODEVIEWER);
    templateDb.addGroupToTemplate(template1.getUuid(), group3.getUuid(), CODEVIEWER);
    templateDb.addGroupToTemplate(template1.getUuid(), null, CODEVIEWER);
    templateDb.addGroupToTemplate(template1.getUuid(), group1.getUuid(), ADMIN);
    templateDb.addGroupToTemplate(template2.getUuid(), group1.getUuid(), ADMIN);
    templateDb.addGroupToTemplate(template4.getUuid(), group1.getUuid(), ISSUE_ADMIN);

    final List<CountByTemplateAndPermissionDto> result = new ArrayList<>();
    underTest.groupsCountByTemplateUuidAndPermission(dbSession, asList(template1.getUuid(), template2.getUuid(), template3.getUuid()),
      context -> result.add(context.getResultObject()));

    assertThat(result).extracting(CountByTemplateAndPermissionDto::getPermission, CountByTemplateAndPermissionDto::getTemplateUuid, CountByTemplateAndPermissionDto::getCount)
      .containsOnly(tuple(ADMIN, template1.getUuid(), 1), tuple(CODEVIEWER, template1.getUuid(), 4), tuple(ADMIN, template2.getUuid(), 1));
  }

  @Test
  public void user_count_by_template_and_permission() {
    PermissionTemplateDto template1 = templateDb.insertTemplate();
    PermissionTemplateDto template2 = templateDb.insertTemplate();
    PermissionTemplateDto template3 = templateDb.insertTemplate();
    PermissionTemplateDto anotherTemplate = templateDb.insertTemplate();

    UserDto user1 = db.users().insertUser();
    UserDto user2 = db.users().insertUser();
    UserDto user3 = db.users().insertUser();

    templateDb.addUserToTemplate(template1.getUuid(), user1.getUuid(), ADMIN);
    templateDb.addUserToTemplate(template1.getUuid(), user2.getUuid(), ADMIN);
    templateDb.addUserToTemplate(template1.getUuid(), user3.getUuid(), ADMIN);
    templateDb.addUserToTemplate(template1.getUuid(), user1.getUuid(), USER);
    templateDb.addUserToTemplate(template2.getUuid(), user1.getUuid(), USER);
    templateDb.addUserToTemplate(anotherTemplate.getUuid(), user1.getUuid(), ISSUE_ADMIN);

    final List<CountByTemplateAndPermissionDto> result = new ArrayList<>();
    underTest.usersCountByTemplateUuidAndPermission(dbSession, asList(template1.getUuid(), template2.getUuid(), template3.getUuid()),
      context -> result.add(context.getResultObject()));
    assertThat(result)
      .extracting(CountByTemplateAndPermissionDto::getPermission, CountByTemplateAndPermissionDto::getTemplateUuid, CountByTemplateAndPermissionDto::getCount)
      .containsExactlyInAnyOrder(
        tuple(ADMIN, template1.getUuid(), 3),
        tuple(USER, template1.getUuid(), 1),
        tuple(USER, template2.getUuid(), 1));
  }

  @Test
  public void selectPotentialPermissions_with_unknown_template_and_no_user() {
    List<String> result = underTest.selectPotentialPermissionsByUserUuidAndTemplateUuid(dbSession, null, "42");

    assertThat(result).isEmpty();
  }

  @Test
  public void selectPotentialPermissions_with_empty_template_and_new_user() {
    UserDto user = db.users().insertUser();
    PermissionTemplateDto template = templateDb.insertTemplate();

    List<String> result = underTest.selectPotentialPermissionsByUserUuidAndTemplateUuid(dbSession, user.getUuid(), template.getUuid());

    assertThat(result).isEmpty();
  }

  @Test
  public void selectPotentialPermission_with_template_users_groups_and_project_creator() {
    UserDto user = db.users().insertUser();
    GroupDto group = db.users().insertGroup(newGroupDto());
    db.users().insertMember(group, user);
    PermissionTemplateDto template = templateDb.insertTemplate();
    templateDb.addProjectCreatorToTemplate(template.getUuid(), SCAN_EXECUTION);
    templateDb.addProjectCreatorToTemplate(template.getUuid(), UserRole.ADMIN);
    templateDb.addUserToTemplate(template.getUuid(), user.getUuid(), UserRole.USER);
    templateDb.addUserToTemplate(template.getUuid(), user.getUuid(), UserRole.ADMIN);
    templateDb.addGroupToTemplate(template.getUuid(), group.getUuid(), UserRole.CODEVIEWER);
    templateDb.addGroupToTemplate(template.getUuid(), group.getUuid(), UserRole.ADMIN);
    templateDb.addGroupToTemplate(template.getUuid(), null, UserRole.ISSUE_ADMIN);

    List<String> resultWithUser = underTest.selectPotentialPermissionsByUserUuidAndTemplateUuid(dbSession, user.getUuid(), template.getUuid());
    List<String> resultWithoutUser = underTest.selectPotentialPermissionsByUserUuidAndTemplateUuid(dbSession, null, template.getUuid());

    assertThat(resultWithUser).containsOnlyOnce(SCAN_EXECUTION, UserRole.ADMIN, UserRole.USER, UserRole.CODEVIEWER, UserRole.ISSUE_ADMIN);
    // only permission from anyone group
    assertThat(resultWithoutUser).containsOnly(UserRole.ISSUE_ADMIN);
  }

  @Test
  public void selectAllGroupPermissionTemplatesByGroupUuid() {
    PermissionTemplateDto permissionTemplate = templateDb.insertTemplate(db.getDefaultOrganization());
    GroupDto group1 = db.users().insertGroup();
    GroupDto group2 = db.users().insertGroup();
    templateDb.addGroupToTemplate(permissionTemplate, group1, "user");
    templateDb.addGroupToTemplate(permissionTemplate, group1, "admin");
    templateDb.addGroupToTemplate(permissionTemplate, group2, "user");

    assertThat(db.getDbClient().permissionTemplateDao().selectAllGroupPermissionTemplatesByGroupUuid(db.getSession(), group1.getUuid()))
      .extracting(PermissionTemplateGroupDto::getGroupUuid, PermissionTemplateGroupDto::getPermission)
      .containsOnly(tuple(group1.getUuid(), "user"), tuple(group1.getUuid(), "admin"));
  }

  @Test
  public void deleteByOrganization_does_not_fail_on_empty_db() {
    underTest.deleteByOrganization(dbSession, "some uuid");
    dbSession.commit();
  }

  @Test
  public void deleteByOrganization_does_not_fail_when_organization_has_no_template() {
    OrganizationDto organization = db.organizations().insert();

    underTest.deleteByOrganization(dbSession, organization.getUuid());
    dbSession.commit();
  }

  @Test
  public void deleteByOrganization_delete_all_templates_of_organization_and_content_of_child_tables() {
    OrganizationDto organization1 = db.organizations().insert();
    OrganizationDto organization2 = db.organizations().insert();
    OrganizationDto organization3 = db.organizations().insert();

    PermissionTemplateDto[] templates = {
      createTemplate(organization1),
      createTemplate(organization2),
      createTemplate(organization3),
      createTemplate(organization1),
      createTemplate(organization2)
    };

    verifyTemplateUuidsInDb(templates[0].getUuid(), templates[1].getUuid(), templates[2].getUuid(), templates[3].getUuid(), templates[4].getUuid());

    underTest.deleteByOrganization(dbSession, organization2.getUuid());
    dbSession.commit();
    verifyTemplateUuidsInDb(templates[0].getUuid(), templates[2].getUuid(), templates[3].getUuid());

    underTest.deleteByOrganization(dbSession, organization3.getUuid());
    dbSession.commit();
    verifyTemplateUuidsInDb(templates[0].getUuid(), templates[3].getUuid());

    underTest.deleteByOrganization(dbSession, organization1.getUuid());
    dbSession.commit();
    verifyTemplateUuidsInDb();
  }

  @Test
  public void delete_user_permissions_by_organization() {
    OrganizationDto organization = db.organizations().insert();
    OrganizationDto anotherOrganization = db.organizations().insert();
    UserDto user = db.users().insertUser();
    UserDto anotherUser = db.users().insertUser();
    PermissionTemplateDto template = db.permissionTemplates().insertTemplate(organization);
    PermissionTemplateDto anotherTemplate = db.permissionTemplates().insertTemplate(anotherOrganization);
    String permission = "PERMISSION";
    db.permissionTemplates().addUserToTemplate(template.getUuid(), user.getUuid(), permission);
    db.permissionTemplates().addUserToTemplate(template.getUuid(), anotherUser.getUuid(), permission);
    db.permissionTemplates().addUserToTemplate(anotherTemplate.getUuid(), user.getUuid(), permission);

    underTest.deleteUserPermissionsByOrganization(dbSession, organization.getUuid(), user.getUuid());

    assertThat(underTest.selectUserPermissionsByTemplateId(dbSession, template.getUuid())).extracting(PermissionTemplateUserDto::getUserUuid).containsOnly(anotherUser.getUuid());
    assertThat(underTest.selectUserPermissionsByTemplateId(dbSession, anotherTemplate.getUuid())).extracting(PermissionTemplateUserDto::getUserUuid).containsOnly(user.getUuid());
  }

  @Test
  public void delete_user_permissions_by_user_uuid() {
    OrganizationDto organization = db.organizations().insert();
    OrganizationDto anotherOrganization = db.organizations().insert();
    UserDto user = db.users().insertUser();
    UserDto anotherUser = db.users().insertUser();
    PermissionTemplateDto template = db.permissionTemplates().insertTemplate(organization);
    PermissionTemplateDto anotherTemplate = db.permissionTemplates().insertTemplate(anotherOrganization);
    String permission = "PERMISSION";
    db.permissionTemplates().addUserToTemplate(template.getUuid(), user.getUuid(), permission);
    db.permissionTemplates().addUserToTemplate(template.getUuid(), anotherUser.getUuid(), permission);
    db.permissionTemplates().addUserToTemplate(anotherTemplate.getUuid(), user.getUuid(), permission);

    underTest.deleteUserPermissionsByUserUuid(dbSession, user.getUuid());
    db.commit();

    assertThat(db.select("select template_uuid as \"templateUuid\", user_uuid as \"userUuid\", permission_reference as \"permission\" from perm_templates_users"))
      .extracting((row) -> row.get("templateUuid"), (row) -> row.get("userUuid"), (row) -> row.get("permission"))
      .containsOnly(tuple(template.getUuid(), anotherUser.getUuid(), permission));
  }

  private PermissionTemplateDto createTemplate(OrganizationDto organization) {
    UserDto user = db.users().insertUser();
    GroupDto group = db.users().insertGroup();
    db.users().insertMember(group, user);
    PermissionTemplateDto template = templateDb.insertTemplate(organization);
    templateDb.addProjectCreatorToTemplate(template.getUuid(), SCAN_EXECUTION);
    templateDb.addProjectCreatorToTemplate(template.getUuid(), UserRole.ADMIN);
    templateDb.addUserToTemplate(template.getUuid(), user.getUuid(), UserRole.USER);
    templateDb.addUserToTemplate(template.getUuid(), user.getUuid(), UserRole.ADMIN);
    templateDb.addGroupToTemplate(template.getUuid(), group.getUuid(), UserRole.CODEVIEWER);
    templateDb.addGroupToTemplate(template.getUuid(), group.getUuid(), UserRole.ADMIN);
    templateDb.addGroupToTemplate(template.getUuid(), null, UserRole.ISSUE_ADMIN);
    return template;
  }

  private void verifyTemplateUuidsInDb(String... expectedTemplateUuids) {
    assertThat(db.select("select distinct template_uuid as \"templateUuid\" from perm_templates_groups"))
      .extracting((row) -> (String) row.get("templateUuid"))
      .containsOnly(expectedTemplateUuids);
    assertThat(db.select("select distinct template_uuid as \"templateUuid\" from perm_templates_users"))
      .extracting((row) -> (String) row.get("templateUuid"))
      .containsOnly(expectedTemplateUuids);
    assertThat(db.select("select distinct template_uuid as \"templateUuid\" from perm_tpl_characteristics"))
      .extracting((row) -> (String) row.get("templateUuid"))
      .containsOnly(expectedTemplateUuids);
    assertThat(db.select("select distinct uuid as \"templateUuid\" from permission_templates"))
      .extracting((row) -> (String) row.get("templateUuid"))
      .containsOnly(expectedTemplateUuids);
  }

}
