/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.monitor.os;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.Constants;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.io.PathUtils;
import org.elasticsearch.monitor.Probes;
import org.elasticsearch.monitor.os.OsInfo;
import org.elasticsearch.monitor.os.OsStats;

public class OsProbe {
    private static final OperatingSystemMXBean osMxBean = ManagementFactory.getOperatingSystemMXBean();
    private static final Method getFreePhysicalMemorySize = OsProbe.getMethod("getFreePhysicalMemorySize");
    private static final Method getTotalPhysicalMemorySize = OsProbe.getMethod("getTotalPhysicalMemorySize");
    private static final Method getFreeSwapSpaceSize = OsProbe.getMethod("getFreeSwapSpaceSize");
    private static final Method getTotalSwapSpaceSize = OsProbe.getMethod("getTotalSwapSpaceSize");
    private static final Method getSystemLoadAverage = OsProbe.getMethod("getSystemLoadAverage");
    private static final Method getSystemCpuLoad = OsProbe.getMethod("getSystemCpuLoad");
    private static final String CONTROL_GROUPS_HIERARCHY_OVERRIDE = System.getProperty("es.cgroups.hierarchy.override");
    private final Logger logger = LogManager.getLogger(this.getClass());

    public long getFreePhysicalMemorySize() {
        if (getFreePhysicalMemorySize == null) {
            this.logger.warn("getFreePhysicalMemorySize is not available");
            return 0L;
        }
        try {
            long freeMem = (Long)getFreePhysicalMemorySize.invoke((Object)osMxBean, new Object[0]);
            if (freeMem < 0L) {
                this.logger.warn("OS reported a negative free memory value [{}]", (Object)freeMem);
                return 0L;
            }
            return freeMem;
        }
        catch (Exception e) {
            this.logger.warn("exception retrieving free physical memory", (Throwable)e);
            return 0L;
        }
    }

    public long getTotalPhysicalMemorySize() {
        if (getTotalPhysicalMemorySize == null) {
            this.logger.warn("getTotalPhysicalMemorySize is not available");
            return 0L;
        }
        try {
            long totalMem = (Long)getTotalPhysicalMemorySize.invoke((Object)osMxBean, new Object[0]);
            if (totalMem < 0L) {
                this.logger.warn("OS reported a negative total memory value [{}]", (Object)totalMem);
                return 0L;
            }
            return totalMem;
        }
        catch (Exception e) {
            this.logger.warn("exception retrieving total physical memory", (Throwable)e);
            return 0L;
        }
    }

    public long getFreeSwapSpaceSize() {
        if (getFreeSwapSpaceSize == null) {
            return -1L;
        }
        try {
            return (Long)getFreeSwapSpaceSize.invoke((Object)osMxBean, new Object[0]);
        }
        catch (Exception e) {
            return -1L;
        }
    }

    public long getTotalSwapSpaceSize() {
        if (getTotalSwapSpaceSize == null) {
            return -1L;
        }
        try {
            return (Long)getTotalSwapSpaceSize.invoke((Object)osMxBean, new Object[0]);
        }
        catch (Exception e) {
            return -1L;
        }
    }

    final double[] getSystemLoadAverage() {
        if (Constants.WINDOWS) {
            return null;
        }
        if (Constants.LINUX) {
            try {
                String procLoadAvg = this.readProcLoadavg();
                assert (procLoadAvg.matches("(\\d+\\.\\d+\\s+){3}\\d+/\\d+\\s+\\d+"));
                String[] fields = procLoadAvg.split("\\s+");
                return new double[]{Double.parseDouble(fields[0]), Double.parseDouble(fields[1]), Double.parseDouble(fields[2])};
            }
            catch (IOException e) {
                if (this.logger.isDebugEnabled()) {
                    this.logger.debug("error reading /proc/loadavg", (Throwable)e);
                }
                return null;
            }
        }
        assert (Constants.MAC_OS_X);
        if (getSystemLoadAverage == null) {
            return null;
        }
        try {
            double oneMinuteLoadAverage = (Double)getSystemLoadAverage.invoke((Object)osMxBean, new Object[0]);
            return new double[]{oneMinuteLoadAverage >= 0.0 ? oneMinuteLoadAverage : -1.0, -1.0, -1.0};
        }
        catch (IllegalAccessException | InvocationTargetException e) {
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("error reading one minute load average from operating system", (Throwable)e);
            }
            return null;
        }
    }

    @SuppressForbidden(reason="access /proc/loadavg")
    String readProcLoadavg() throws IOException {
        return this.readSingleLine(PathUtils.get("/proc/loadavg", new String[0]));
    }

    public short getSystemCpuPercent() {
        return Probes.getLoadAndScaleToPercent(getSystemCpuLoad, osMxBean);
    }

    private String readSingleLine(Path path) throws IOException {
        List<String> lines = Files.readAllLines(path);
        assert (lines != null && lines.size() == 1);
        return lines.get(0);
    }

    private Map<String, String> getControlGroups() throws IOException {
        List<String> lines = this.readProcSelfCgroup();
        HashMap<String, String> controllerMap = new HashMap<String, String>();
        for (String line : lines) {
            String[] controllers;
            String[] fields = line.split(":");
            assert (fields.length == 3);
            for (String controller : controllers = fields[1].split(",")) {
                String controlGroupPath = CONTROL_GROUPS_HIERARCHY_OVERRIDE != null ? CONTROL_GROUPS_HIERARCHY_OVERRIDE : fields[2];
                String previous = controllerMap.put(controller, controlGroupPath);
                assert (previous == null);
            }
        }
        return controllerMap;
    }

    @SuppressForbidden(reason="access /proc/self/cgroup")
    List<String> readProcSelfCgroup() throws IOException {
        List<String> lines = Files.readAllLines(PathUtils.get("/proc/self/cgroup", new String[0]));
        assert (lines != null && !lines.isEmpty());
        return lines;
    }

    private long getCgroupCpuAcctUsageNanos(String controlGroup) throws IOException {
        return Long.parseLong(this.readSysFsCgroupCpuAcctCpuAcctUsage(controlGroup));
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpuacct")
    String readSysFsCgroupCpuAcctCpuAcctUsage(String controlGroup) throws IOException {
        return this.readSingleLine(PathUtils.get("/sys/fs/cgroup/cpuacct", controlGroup, "cpuacct.usage"));
    }

    private long getCgroupCpuAcctCpuCfsPeriodMicros(String controlGroup) throws IOException {
        return Long.parseLong(this.readSysFsCgroupCpuAcctCpuCfsPeriod(controlGroup));
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu")
    String readSysFsCgroupCpuAcctCpuCfsPeriod(String controlGroup) throws IOException {
        return this.readSingleLine(PathUtils.get("/sys/fs/cgroup/cpu", controlGroup, "cpu.cfs_period_us"));
    }

    private long getCgroupCpuAcctCpuCfsQuotaMicros(String controlGroup) throws IOException {
        return Long.parseLong(this.readSysFsCgroupCpuAcctCpuAcctCfsQuota(controlGroup));
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu")
    String readSysFsCgroupCpuAcctCpuAcctCfsQuota(String controlGroup) throws IOException {
        return this.readSingleLine(PathUtils.get("/sys/fs/cgroup/cpu", controlGroup, "cpu.cfs_quota_us"));
    }

    private OsStats.Cgroup.CpuStat getCgroupCpuAcctCpuStat(String controlGroup) throws IOException {
        List<String> lines = this.readSysFsCgroupCpuAcctCpuStat(controlGroup);
        long numberOfPeriods = -1L;
        long numberOfTimesThrottled = -1L;
        long timeThrottledNanos = -1L;
        for (String line : lines) {
            String[] fields = line.split("\\s+");
            switch (fields[0]) {
                case "nr_periods": {
                    numberOfPeriods = Long.parseLong(fields[1]);
                    break;
                }
                case "nr_throttled": {
                    numberOfTimesThrottled = Long.parseLong(fields[1]);
                    break;
                }
                case "throttled_time": {
                    timeThrottledNanos = Long.parseLong(fields[1]);
                }
            }
        }
        assert (numberOfPeriods != -1L);
        assert (numberOfTimesThrottled != -1L);
        assert (timeThrottledNanos != -1L);
        return new OsStats.Cgroup.CpuStat(numberOfPeriods, numberOfTimesThrottled, timeThrottledNanos);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/cpu")
    List<String> readSysFsCgroupCpuAcctCpuStat(String controlGroup) throws IOException {
        List<String> lines = Files.readAllLines(PathUtils.get("/sys/fs/cgroup/cpu", controlGroup, "cpu.stat"));
        assert (lines != null && lines.size() == 3);
        return lines;
    }

    private String getCgroupMemoryLimitInBytes(String controlGroup) throws IOException {
        return this.readSysFsCgroupMemoryLimitInBytes(controlGroup);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/memory")
    String readSysFsCgroupMemoryLimitInBytes(String controlGroup) throws IOException {
        return this.readSingleLine(PathUtils.get("/sys/fs/cgroup/memory", controlGroup, "memory.limit_in_bytes"));
    }

    private String getCgroupMemoryUsageInBytes(String controlGroup) throws IOException {
        return this.readSysFsCgroupMemoryUsageInBytes(controlGroup);
    }

    @SuppressForbidden(reason="access /sys/fs/cgroup/memory")
    String readSysFsCgroupMemoryUsageInBytes(String controlGroup) throws IOException {
        return this.readSingleLine(PathUtils.get("/sys/fs/cgroup/memory", controlGroup, "memory.usage_in_bytes"));
    }

    @SuppressForbidden(reason="access /proc/self/cgroup, /sys/fs/cgroup/cpu, /sys/fs/cgroup/cpuacct and /sys/fs/cgroup/memory")
    boolean areCgroupStatsAvailable() {
        if (!Files.exists(PathUtils.get("/proc/self/cgroup", new String[0]), new LinkOption[0])) {
            return false;
        }
        if (!Files.exists(PathUtils.get("/sys/fs/cgroup/cpu", new String[0]), new LinkOption[0])) {
            return false;
        }
        if (!Files.exists(PathUtils.get("/sys/fs/cgroup/cpuacct", new String[0]), new LinkOption[0])) {
            return false;
        }
        return Files.exists(PathUtils.get("/sys/fs/cgroup/memory", new String[0]), new LinkOption[0]);
    }

    private OsStats.Cgroup getCgroup() {
        try {
            if (!this.areCgroupStatsAvailable()) {
                return null;
            }
            Map<String, String> controllerMap = this.getControlGroups();
            assert (!controllerMap.isEmpty());
            String cpuAcctControlGroup = controllerMap.get("cpuacct");
            assert (cpuAcctControlGroup != null);
            long cgroupCpuAcctUsageNanos = this.getCgroupCpuAcctUsageNanos(cpuAcctControlGroup);
            String cpuControlGroup = controllerMap.get("cpu");
            assert (cpuControlGroup != null);
            long cgroupCpuAcctCpuCfsPeriodMicros = this.getCgroupCpuAcctCpuCfsPeriodMicros(cpuControlGroup);
            long cgroupCpuAcctCpuCfsQuotaMicros = this.getCgroupCpuAcctCpuCfsQuotaMicros(cpuControlGroup);
            OsStats.Cgroup.CpuStat cpuStat = this.getCgroupCpuAcctCpuStat(cpuControlGroup);
            String memoryControlGroup = controllerMap.get("memory");
            assert (memoryControlGroup != null);
            String cgroupMemoryLimitInBytes = this.getCgroupMemoryLimitInBytes(memoryControlGroup);
            String cgroupMemoryUsageInBytes = this.getCgroupMemoryUsageInBytes(memoryControlGroup);
            return new OsStats.Cgroup(cpuAcctControlGroup, cgroupCpuAcctUsageNanos, cpuControlGroup, cgroupCpuAcctCpuCfsPeriodMicros, cgroupCpuAcctCpuCfsQuotaMicros, cpuStat, memoryControlGroup, cgroupMemoryLimitInBytes, cgroupMemoryUsageInBytes);
        }
        catch (IOException e) {
            this.logger.debug("error reading control group stats", (Throwable)e);
            return null;
        }
    }

    public static OsProbe getInstance() {
        return OsProbeHolder.INSTANCE;
    }

    OsProbe() {
    }

    OsInfo osInfo(long refreshInterval, int allocatedProcessors) throws IOException {
        return new OsInfo(refreshInterval, Runtime.getRuntime().availableProcessors(), allocatedProcessors, Constants.OS_NAME, this.getPrettyName(), Constants.OS_ARCH, Constants.OS_VERSION);
    }

    private String getPrettyName() throws IOException {
        if (Constants.LINUX) {
            Optional maybePrettyNameLine;
            List<String> etcOsReleaseLines = this.readOsRelease();
            List prettyNameLines = etcOsReleaseLines.stream().filter(line -> line.startsWith("PRETTY_NAME")).collect(Collectors.toList());
            assert (prettyNameLines.size() <= 1) : prettyNameLines;
            Optional<Object> optional = maybePrettyNameLine = prettyNameLines.size() == 1 ? Optional.of((String)prettyNameLines.get(0)) : Optional.empty();
            if (maybePrettyNameLine.isPresent()) {
                String trimmedPrettyNameLine = ((String)maybePrettyNameLine.get()).trim();
                Matcher matcher = Pattern.compile("PRETTY_NAME=(\"?|'?)?([^\"']+)\\1").matcher(trimmedPrettyNameLine);
                boolean matches = matcher.matches();
                assert (matches) : trimmedPrettyNameLine;
                assert (matcher.groupCount() == 2) : trimmedPrettyNameLine;
                return matcher.group(2);
            }
            return Constants.OS_NAME;
        }
        return Constants.OS_NAME;
    }

    @SuppressForbidden(reason="access /etc/os-release or /usr/lib/os-release or /etc/system-release")
    List<String> readOsRelease() throws IOException {
        if (Files.exists(PathUtils.get("/etc/os-release", new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get("/etc/os-release", new String[0]));
            assert (lines != null && !lines.isEmpty());
            return lines;
        }
        if (Files.exists(PathUtils.get("/usr/lib/os-release", new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get("/usr/lib/os-release", new String[0]));
            assert (lines != null && !lines.isEmpty());
            return lines;
        }
        if (Files.exists(PathUtils.get("/etc/system-release", new String[0]), new LinkOption[0])) {
            List<String> lines = Files.readAllLines(PathUtils.get("/etc/system-release", new String[0]));
            assert (lines != null && lines.size() == 1);
            return Collections.singletonList("PRETTY_NAME=\"" + lines.get(0) + "\"");
        }
        return Collections.emptyList();
    }

    public OsStats osStats() {
        OsStats.Cpu cpu = new OsStats.Cpu(this.getSystemCpuPercent(), this.getSystemLoadAverage());
        OsStats.Mem mem = new OsStats.Mem(this.getTotalPhysicalMemorySize(), this.getFreePhysicalMemorySize());
        OsStats.Swap swap = new OsStats.Swap(this.getTotalSwapSpaceSize(), this.getFreeSwapSpaceSize());
        OsStats.Cgroup cgroup = Constants.LINUX ? this.getCgroup() : null;
        return new OsStats(System.currentTimeMillis(), cpu, mem, swap, cgroup);
    }

    private static Method getMethod(String methodName) {
        try {
            return Class.forName("com.sun.management.OperatingSystemMXBean").getMethod(methodName, new Class[0]);
        }
        catch (Exception e) {
            return null;
        }
    }

    private static class OsProbeHolder {
        private static final OsProbe INSTANCE = new OsProbe();

        private OsProbeHolder() {
        }
    }
}

