/*
 * SonarQube
 * Copyright (C) 2009-2020 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
import * as React from 'react';
import { Link } from 'react-router';
import Checkbox from 'sonar-ui-common/components/controls/Checkbox';
import Tooltip from 'sonar-ui-common/components/controls/Tooltip';
import QualifierIcon from 'sonar-ui-common/components/icons/QualifierIcon';
import DateTooltipFormatter from 'sonar-ui-common/components/intl/DateTooltipFormatter';
import { Project } from '../../api/components';
import PrivacyBadgeContainer from '../../components/common/PrivacyBadgeContainer';
import { getComponentOverviewUrl } from '../../helpers/urls';
import './ProjectRow.css';
import ProjectRowActions from './ProjectRowActions';

interface Props {
  currentUser: Pick<T.LoggedInUser, 'login'>;
  onProjectCheck: (project: Project, checked: boolean) => void;
  organization: string | undefined;
  project: Project;
  selected: boolean;
}

export default class ProjectRow extends React.PureComponent<Props> {
  handleProjectCheck = (checked: boolean) => {
    this.props.onProjectCheck(this.props.project, checked);
  };

  render() {
    const { organization, project, selected } = this.props;

    return (
      <tr data-project-key={project.key}>
        <td className="thin">
          <Checkbox checked={selected} onCheck={this.handleProjectCheck} />
        </td>

        <td className="nowrap hide-overflow project-row-text-cell">
          <Link
            className="link-with-icon"
            to={getComponentOverviewUrl(project.key, project.qualifier)}>
            <QualifierIcon className="little-spacer-right" qualifier={project.qualifier} />

            <Tooltip overlay={project.name} placement="left">
              <span>{project.name}</span>
            </Tooltip>
          </Link>
        </td>

        <td className="thin nowrap">
          <PrivacyBadgeContainer qualifier={project.qualifier} visibility={project.visibility} />
        </td>

        <td className="nowrap hide-overflow project-row-text-cell">
          <Tooltip overlay={project.key} placement="left">
            <span className="note">{project.key}</span>
          </Tooltip>
        </td>

        <td className="thin nowrap text-right">
          {project.lastAnalysisDate ? (
            <DateTooltipFormatter date={project.lastAnalysisDate} />
          ) : (
            <span className="note">—</span>
          )}
        </td>

        <td className="thin nowrap">
          <ProjectRowActions
            currentUser={this.props.currentUser}
            organization={organization}
            project={project}
          />
        </td>
      </tr>
    );
  }
}
