/*
 * SonarQube
 * Copyright (C) 2009-2020 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.server.platform.db.migration.sql;

import java.util.List;
import org.sonar.db.dialect.Dialect;
import org.sonar.db.dialect.H2;
import org.sonar.db.dialect.MsSql;
import org.sonar.db.dialect.Oracle;
import org.sonar.db.dialect.PostgreSql;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.sonar.server.platform.db.migration.def.Validations.validateTableName;

public class RenameTableBuilder {

  private final Dialect dialect;
  private String name;
  private String newName;
  private boolean autoGeneratedId = true;

  public RenameTableBuilder(Dialect dialect) {
    this.dialect = dialect;
  }

  public RenameTableBuilder setName(String s) {
    this.name = s;
    return this;
  }

  public RenameTableBuilder setNewName(String s) {
    this.newName = s;
    return this;
  }

  /**
   * When a table has no auto generated id, this parameter has to be set to false.
   * On Oracle, it will allow to not try to drop and recreate the trigger.
   * On other databases, this method is useless.
   *
   * Default value is true.
   */
  public RenameTableBuilder setAutoGeneratedId(boolean autoGeneratedId) {
    this.autoGeneratedId = autoGeneratedId;
    return this;
  }

  public List<String> build() {
    validateTableName(name);
    validateTableName(newName);
    checkArgument(!name.equals(newName), "Names must be different");
    return createSqlStatement();
  }

  private List<String> createSqlStatement() {
    switch (dialect.getId()) {
      case H2.ID:
      case PostgreSql.ID:
        return singletonList("ALTER TABLE " + name + " RENAME TO " + newName);
      case MsSql.ID:
        return singletonList("EXEC sp_rename '" + name + "', '" + newName + "'");
      case Oracle.ID:
        String renameSqlCommand = "RENAME " + name + " TO " + newName;
        return autoGeneratedId ? asList(
          "DROP TRIGGER " + name + "_idt", renameSqlCommand,
          "RENAME " + name + "_seq TO " + newName + "_seq",
            CreateTableBuilder.createOracleTriggerForTable(newName))
          : singletonList(renameSqlCommand);
      default:
        throw new IllegalArgumentException("Unsupported dialect id " + dialect.getId());
    }
  }
}
