Abstract base class for probability distributions.

This class, along with `ContinuousDistribution` and `DiscreteDistribution`,
defines the API for probability distributions.

Users will never instantiate a `BaseDistribution`, but will instead
instantiate subclasses of either `ContinuousDistribution` or
`DiscreteDistribution`.

Developers of new distributions should prefer to subclass
`ContinuousDistribution` or `DiscreteDistribution`.

### API

The key methods for probability distributions are defined here. The likelihood
functions (`pdf`, `log_pdf`) and (`pmf`, `log_pmf`) are defined in
`ContinuousDistribution` and `DiscreteDistribution`, respectively.

To keep ops generated by the distribution tied together by name, subclasses
should override `name` and use it to preprend names of ops in other methods
(see `cdf` for an example).

Subclasses that wish to support `cdf` and `log_cdf` can override `log_cdf`
and use the base class's implementation for `cdf`.

### Broadcasting, batching, and shapes

All distributions support batches of independent distributions of that type.
The batch shape is determined by broadcasting together the parameters.

The shape of arguments to `__init__`, `cdf`, `log_cdf`, and the likelihood
functions defined in `ContinuousDistribution` and `DiscreteDistribution`
reflect this broadcasting, as does the return value of `sample`.

`sample_shape = (n,) + batch_shape + event_shape`, where `sample_shape` is the
shape of the `Tensor` returned from `sample`, `n` is the number of samples,
`batch_shape` defines how many independent distributions there are, and
`event_shape` defines the shape of samples from each of those independent
distributions. Samples are independent along the `batch_shape` dimensions,
but not necessarily so along the `event_shape` dimensions (dependending on
the particulars of the underlying distribution).

Using the `Uniform` distribution as an example:

```python
minval = 3.0
maxval = [[4.0, 6.0],
          [10.0, 12.0]]

# Broadcasting:
# This instance represents 4 Uniform distributions. Each has a lower bound at
# 3.0 as the `minval` parameter was broadcasted to match `maxval`'s shape.
u = Uniform(minval, maxval)

# `event_shape` is `TensorShape([])`.
event_shape = u.get_event_shape()
# `event_shape_t` is a `Tensor` which will evaluate to [].
event_shape_t = u.event_shape

# Sampling returns a sample per distribution.  `samples` has shape
# (5, 2, 2), which is (n,) + batch_shape + event_shape, where n=5,
# batch_shape=(2, 2), and event_shape=().
samples = u.sample(5)

# The broadcasting holds across methods. Here we use `cdf` as an example. The
# same holds for `log_cdf` and the likelihood functions.

# `cum_prob` has shape (2, 2) as the `value` argument was broadcasted to the
# shape of the `Uniform` instance.
cum_prob_broadcast = u.cdf(4.0)

# `cum_prob`'s shape is (2, 2), one per distribution. No broadcasting
# occurred.
cum_prob_per_dist = u.cdf([[4.0, 5.0],
                           [6.0, 7.0]])

# INVALID as the `value` argument is not broadcastable to the distribution's
# shape.
cum_prob_invalid = u.cdf([4.0, 5.0, 6.0])
```
- - -

#### `tf.contrib.distributions.BaseDistribution.batch_shape(name='batch_shape')` {#BaseDistribution.batch_shape}

Batch dimensions of this instance as a 1-D int32 `Tensor`.

The product of the dimensions of the `batch_shape` is the number of
independent distributions of this kind the instance represents.

##### Args:


*  <b>`name`</b>: name to give to the op

##### Returns:

  `Tensor` `batch_shape`


- - -

#### `tf.contrib.distributions.BaseDistribution.cdf(value, name='cdf')` {#BaseDistribution.cdf}

Cumulative distribution function.


- - -

#### `tf.contrib.distributions.BaseDistribution.dtype` {#BaseDistribution.dtype}

dtype of samples from this distribution.


- - -

#### `tf.contrib.distributions.BaseDistribution.entropy(name='entropy')` {#BaseDistribution.entropy}

Entropy of the distribution in nats.


- - -

#### `tf.contrib.distributions.BaseDistribution.event_shape(name='event_shape')` {#BaseDistribution.event_shape}

Shape of a sample from a single distribution as a 1-D int32 `Tensor`.

##### Args:


*  <b>`name`</b>: name to give to the op

##### Returns:

  `Tensor` `event_shape`


- - -

#### `tf.contrib.distributions.BaseDistribution.get_batch_shape()` {#BaseDistribution.get_batch_shape}

`TensorShape` available at graph construction time.

Same meaning as `batch_shape`. May be only partially defined.


- - -

#### `tf.contrib.distributions.BaseDistribution.get_event_shape()` {#BaseDistribution.get_event_shape}

`TensorShape` available at graph construction time.

Same meaning as `event_shape`. May be only partially defined.


- - -

#### `tf.contrib.distributions.BaseDistribution.log_cdf(value, name='log_cdf')` {#BaseDistribution.log_cdf}

Log CDF.


- - -

#### `tf.contrib.distributions.BaseDistribution.mean(name='mean')` {#BaseDistribution.mean}

Mean of the distribution.


- - -

#### `tf.contrib.distributions.BaseDistribution.mode(name='mode')` {#BaseDistribution.mode}

Mode of the distribution.


- - -

#### `tf.contrib.distributions.BaseDistribution.name` {#BaseDistribution.name}

Name to prepend to all ops.


- - -

#### `tf.contrib.distributions.BaseDistribution.sample(n, seed=None, name='sample')` {#BaseDistribution.sample}

Generate `n` samples.

##### Args:


*  <b>`n`</b>: scalar. Number of samples to draw from each distribution.
*  <b>`seed`</b>: Python integer seed for RNG
*  <b>`name`</b>: name to give to the op.

##### Returns:


*  <b>`samples`</b>: a `Tensor` of shape `(n,) + self.batch_shape + self.event_shape`
      with values of type `self.dtype`.


- - -

#### `tf.contrib.distributions.BaseDistribution.std(name='std')` {#BaseDistribution.std}

Standard deviation of the distribution.


- - -

#### `tf.contrib.distributions.BaseDistribution.variance(name='variance')` {#BaseDistribution.variance}

Variance of the distribution.


