use crate::config::{Config, CONFIG_KEY};
use crate::{convert, handler, lsp_ext, LspError, UrlExt, Vfs, MAX_FILE_LEN};
use anyhow::{anyhow, bail, Context, Result};
use crossbeam_channel::{Receiver, Sender};
use ide::{Analysis, AnalysisHost, Cancelled, FlakeInfo, VfsPath};
use lsp_server::{ErrorCode, Message, Notification, ReqQueue, Request, RequestId, Response};
use lsp_types::notification::Notification as _;
use lsp_types::{
    notification as notif, request as req, ConfigurationItem, ConfigurationParams, Diagnostic,
    InitializeParams, MessageType, NumberOrString, PublishDiagnosticsParams, ShowMessageParams,
    Url,
};
use nix_interop::{flake_lock, FLAKE_FILE, FLAKE_LOCK_FILE};
use std::backtrace::Backtrace;
use std::cell::Cell;
use std::collections::HashMap;
use std::io::ErrorKind;
use std::panic::UnwindSafe;
use std::path::{Path, PathBuf};
use std::sync::{Arc, Once, RwLock};
use std::{fs, panic, thread};

type ReqHandler = Box<dyn FnOnce(&mut Server, Response) + 'static>;

type Task = Box<dyn FnOnce() -> Event + Send + 'static>;

enum Event {
    Response(Response),
    Diagnostics {
        uri: Url,
        version: u64,
        diagnostics: Vec<Diagnostic>,
    },
    ClientExited,
    LoadFlake(Result<LoadFlakeResult>),
}

enum LoadFlakeResult {
    IsFlake {
        flake_info: FlakeInfo,
        missing_inputs: bool,
    },
    NotFlake,
}

pub struct Server {
    // States.
    /// This contains an internal RWLock and must not lock together with `vfs`.
    host: AnalysisHost,
    vfs: Arc<RwLock<Vfs>>,
    opened_files: HashMap<Url, FileData>,
    config: Arc<Config>,
    is_shutdown: bool,
    /// Monotonic version counter for diagnostics calculation ordering.
    version_counter: u64,

    // Message passing.
    req_queue: ReqQueue<(), ReqHandler>,
    lsp_tx: Sender<Message>,
    task_tx: Sender<Task>,
    event_tx: Sender<Event>,
    event_rx: Receiver<Event>,
}

#[derive(Debug, Default)]
struct FileData {
    diagnostics_version: u64,
    diagnostics: Vec<Diagnostic>,
}

impl Server {
    pub fn new(lsp_tx: Sender<Message>, root_path: PathBuf) -> Self {
        let (task_tx, task_rx) = crossbeam_channel::unbounded();
        let (event_tx, event_rx) = crossbeam_channel::unbounded();
        let worker_cnt = thread::available_parallelism().map_or(1, |n| n.get());
        for _ in 0..worker_cnt {
            let task_rx = task_rx.clone();
            let event_tx = event_tx.clone();
            thread::Builder::new()
                .name("Worker".into())
                .spawn(move || Self::worker(task_rx, event_tx))
                .expect("Failed to spawn worker threads");
        }
        tracing::info!("Started {worker_cnt} workers");

        Self {
            host: AnalysisHost::default(),
            vfs: Arc::new(RwLock::new(Vfs::new())),
            opened_files: HashMap::default(),
            config: Arc::new(Config::new(root_path)),
            is_shutdown: false,
            version_counter: 0,

            req_queue: ReqQueue::default(),
            lsp_tx,
            task_tx,
            event_tx,
            event_rx,
        }
    }

    fn worker(task_rx: Receiver<Task>, event_tx: Sender<Event>) {
        while let Ok(task) = task_rx.recv() {
            if event_tx.send(task()).is_err() {
                break;
            }
        }
    }

    pub fn run(&mut self, lsp_rx: Receiver<Message>, init_params: InitializeParams) -> Result<()> {
        #[cfg(target_os = "linux")]
        if let Some(pid) = init_params.process_id {
            use std::io;
            use std::mem::MaybeUninit;
            use std::os::unix::io::{AsRawFd, FromRawFd, OwnedFd, RawFd};
            use std::ptr::null_mut;

            fn wait_remote_pid(pid: libc::pid_t) -> Result<(), io::Error> {
                let pidfd = unsafe {
                    let ret = libc::syscall(libc::SYS_pidfd_open, pid, 0 as libc::c_int);
                    if ret == -1 {
                        return Err(io::Error::last_os_error());
                    }
                    OwnedFd::from_raw_fd(ret as RawFd)
                };
                unsafe {
                    let mut fdset = MaybeUninit::uninit();
                    libc::FD_ZERO(fdset.as_mut_ptr());
                    libc::FD_SET(pidfd.as_raw_fd(), fdset.as_mut_ptr());
                    let nfds = pidfd.as_raw_fd() + 1;
                    let ret =
                        libc::select(nfds, fdset.as_mut_ptr(), null_mut(), null_mut(), null_mut());
                    if ret == -1 {
                        return Err(io::Error::last_os_error());
                    }
                }
                Ok(())
            }

            let event_tx = self.event_tx.clone();
            thread::spawn(move || {
                match wait_remote_pid(pid as _) {
                    Ok(()) => {}
                    Err(err) if err.raw_os_error() == Some(libc::ESRCH) => {}
                    Err(err) => {
                        tracing::warn!("Failed to monitor parent pid {}: {}", pid, err);
                        return;
                    }
                }
                let _ = event_tx.send(Event::ClientExited);
            });
        }

        // Load configurations before loading flake.
        // The latter depends on `nix.binary`.
        self.load_config(|st| {
            // TODO: Register file watcher for flake.lock.
            st.load_flake();
        });

        loop {
            crossbeam_channel::select! {
                recv(lsp_rx) -> msg => {
                    match msg.context("Channel closed")? {
                        Message::Request(req) => self.dispatch_request(req),
                        Message::Notification(notif) => {
                            if notif.method == notif::Exit::METHOD {
                                return Ok(());
                            }
                            self.dispatch_notification(notif);
                        }
                        Message::Response(resp) => {
                            if let Some(callback) = self.req_queue.outgoing.complete(resp.id.clone()) {
                                callback(self, resp);
                            }
                        }
                    }
                }
                recv(self.event_rx) -> event => {
                    self.dispatch_event(event.context("Worker panicked")?)?;
                }
            }
        }
    }

    fn dispatch_event(&mut self, event: Event) -> Result<()> {
        match event {
            Event::Response(resp) => {
                if let Some(()) = self.req_queue.incoming.complete(resp.id.clone()) {
                    self.lsp_tx.send(resp.into()).unwrap();
                }
            }
            Event::Diagnostics {
                uri,
                version,
                diagnostics,
            } => match self.opened_files.get_mut(&uri) {
                Some(f) if f.diagnostics_version < version => {
                    f.diagnostics_version = version;
                    f.diagnostics = diagnostics.clone();
                    tracing::trace!(
                        "Push {} diagnostics of {uri}, version {version}",
                        diagnostics.len(),
                    );
                    self.send_notification::<notif::PublishDiagnostics>(PublishDiagnosticsParams {
                        uri,
                        diagnostics,
                        version: None,
                    });
                }
                _ => tracing::debug!("Ignore raced diagnostics of {uri}, version {version}"),
            },
            Event::ClientExited => {
                bail!("The process initializing this server is exited. Exit now")
            }
            Event::LoadFlake(ret) => match ret {
                Err(err) => {
                    self.show_message(
                        MessageType::ERROR,
                        format!("Failed to load flake workspace: {err:#}"),
                    );
                }
                Ok(LoadFlakeResult::IsFlake {
                    flake_info,
                    missing_inputs,
                }) => {
                    tracing::info!(
                        "Workspace is a flake (missing_inputs = {missing_inputs}): {flake_info:?}"
                    );
                    if missing_inputs {
                        self.show_message(MessageType::WARNING, "Some flake inputs are not available, please run `nix flake archive` to fetch all inputs");
                    }
                    self.vfs.write().unwrap().set_flake_info(Some(flake_info));
                    self.apply_vfs_change();
                }
                Ok(LoadFlakeResult::NotFlake) => {
                    tracing::info!("Workspace is not a flake");
                    self.vfs.write().unwrap().set_flake_info(None);
                    self.apply_vfs_change();
                }
            },
        }
        Ok(())
    }

    fn dispatch_request(&mut self, req: Request) {
        if self.is_shutdown {
            let resp = Response::new_err(
                req.id,
                ErrorCode::InvalidRequest as i32,
                "Shutdown already requested.".into(),
            );
            self.lsp_tx.send(resp.into()).unwrap();
            return;
        }

        RequestDispatcher(self, Some(req))
            .on_sync_mut::<req::Shutdown>(|st, ()| {
                st.is_shutdown = true;
                Ok(())
            })
            .on::<req::GotoDefinition>(handler::goto_definition)
            .on::<req::References>(handler::references)
            .on::<req::Completion>(handler::completion)
            .on::<req::SelectionRangeRequest>(handler::selection_range)
            .on::<req::PrepareRenameRequest>(handler::prepare_rename)
            .on::<req::Rename>(handler::rename)
            .on::<req::SemanticTokensFullRequest>(handler::semantic_token_full)
            .on::<req::SemanticTokensRangeRequest>(handler::semantic_token_range)
            .on::<req::HoverRequest>(handler::hover)
            .on::<req::DocumentSymbolRequest>(handler::document_symbol)
            .on::<req::Formatting>(handler::formatting)
            .on::<req::DocumentLinkRequest>(handler::document_links)
            .on::<req::CodeActionRequest>(handler::code_action)
            .on::<req::DocumentHighlightRequest>(handler::document_highlight)
            .on::<lsp_ext::ParentModule>(handler::parent_module)
            .finish();
    }

    fn dispatch_notification(&mut self, notif: Notification) {
        NotificationDispatcher(self, Some(notif))
            .on_sync_mut::<notif::Cancel>(|st, params| {
                let id: RequestId = match params.id {
                    NumberOrString::Number(id) => id.into(),
                    NumberOrString::String(id) => id.into(),
                };
                if let Some(resp) = st.req_queue.incoming.cancel(id) {
                    st.lsp_tx.send(resp.into()).unwrap();
                }
            })
            .on_sync_mut::<notif::DidOpenTextDocument>(|st, params| {
                // Ignore the open event for unsupported files, thus all following interactions
                // will error due to unopened files.
                let len = params.text_document.text.len();
                if len > MAX_FILE_LEN {
                    st.show_message(
                        MessageType::WARNING,
                        "Disable LSP functionalities for too large file ({len} > {MAX_FILE_LEN})",
                    );
                    return;
                }
                let uri = &params.text_document.uri;
                st.set_vfs_file_content(uri, params.text_document.text);
                st.opened_files.insert(uri.clone(), FileData::default());
            })
            .on_sync_mut::<notif::DidCloseTextDocument>(|st, params| {
                // N.B. Don't clear text here.
                st.opened_files.remove(&params.text_document.uri);
            })
            .on_sync_mut::<notif::DidChangeTextDocument>(|st, params| {
                let mut vfs = st.vfs.write().unwrap();
                let uri = &params.text_document.uri;
                // Ignore files not maintained in Vfs.
                let Ok(file) = vfs.file_for_uri(uri) else { return };
                for change in params.content_changes {
                    let ret = (|| {
                        let del_range = match change.range {
                            None => None,
                            Some(range) => Some(convert::from_range(&vfs, file, range).ok()?.1),
                        };
                        vfs.change_file_content(file, del_range, &change.text)
                            .ok()?;
                        Some(())
                    })();
                    if ret.is_none() {
                        tracing::error!(
                            "File is out of sync! Failed to apply change for {uri}: {change:?}"
                        );

                        // Clear file states to minimize pollution of the broken state.
                        st.opened_files.remove(uri);
                        // TODO: Remove the file from Vfs.
                    }
                }
                drop(vfs);
                st.apply_vfs_change();
            })
            // As stated in https://github.com/microsoft/language-server-protocol/issues/676,
            // this notification's parameters should be ignored and the actual config queried separately.
            .on_sync_mut::<notif::DidChangeConfiguration>(|st, _params| {
                st.load_config(|_| {});
            })
            // Workaround:
            // > In former implementations clients pushed file events without the server actively asking for it.
            // Ref: https://microsoft.github.io/language-server-protocol/specifications/lsp/3.17/specification/#workspace_didChangeWatchedFiles
            .on_sync_mut::<notif::DidChangeWatchedFiles>(|_st, _params| {})
            .finish();
    }

    /// Enqueue a task to reload the flake.{nix,lock} and the locked inputs.
    fn load_flake(&self) {
        tracing::info!("Loading flake configuration");

        let flake_path = self.config.root_path.join(FLAKE_FILE);
        let lock_path = self.config.root_path.join(FLAKE_LOCK_FILE);
        let nix_bin_path = self.config.nix_binary.clone();

        let vfs = self.vfs.clone();
        let task = move || {
            let flake_vpath = VfsPath::new(&flake_path);
            let flake_src = match fs::read_to_string(&flake_path) {
                Ok(src) => src,
                // Not a flake.
                Err(err) if err.kind() == ErrorKind::NotFound => {
                    return Ok(LoadFlakeResult::NotFlake);
                }
                // Read failure.
                Err(err) => {
                    return Err(anyhow::Error::new(err)
                        .context(format!("Failed to read flake root {flake_path:?}")));
                }
            };

            // Load the flake file in Vfs.
            let flake_file = {
                let mut vfs = vfs.write().unwrap();
                match vfs.file_for_path(&flake_vpath) {
                    // If the file is already opened (transferred from client),
                    // prefer the managed one. It contains more recent unsaved changes.
                    Ok(file) => file,
                    // Otherwise, cache the file content from disk.
                    Err(_) => vfs.set_path_content(flake_vpath, flake_src),
                }
            };

            let lock_src = match fs::read(&lock_path) {
                Ok(lock_src) => lock_src,
                // Flake without inputs.
                Err(err) if err.kind() == ErrorKind::NotFound => {
                    return Ok(LoadFlakeResult::IsFlake {
                        missing_inputs: false,
                        flake_info: FlakeInfo {
                            flake_file,
                            input_store_paths: HashMap::new(),
                        },
                    });
                }
                Err(err) => {
                    return Err(anyhow::Error::new(err)
                        .context(format!("Failed to read flake lock {lock_path:?}")));
                }
            };

            let inputs = flake_lock::resolve_flake_locked_inputs(&nix_bin_path, &lock_src)
                .context("Failed to resolve flake inputs from lock file")?;

            // We only need the map for input -> store path.
            let inputs_cnt = inputs.len();
            let input_store_paths = inputs
                .into_iter()
                .filter(|(_, input)| Path::new(&input.store_path).exists())
                .map(|(key, input)| (key, VfsPath::new(input.store_path)))
                .collect::<HashMap<_, _>>();

            Ok(LoadFlakeResult::IsFlake {
                missing_inputs: input_store_paths.len() != inputs_cnt,
                flake_info: FlakeInfo {
                    flake_file,
                    input_store_paths,
                },
            })
        };
        self.task_tx
            .send(Box::new(move || Event::LoadFlake(task())))
            .unwrap();
    }

    fn send_request<R: req::Request>(
        &mut self,
        params: R::Params,
        callback: impl FnOnce(&mut Self, Result<R::Result>) + 'static,
    ) {
        let callback = |this: &mut Self, resp: Response| {
            let ret = match resp.error {
                None => serde_json::from_value(resp.result.unwrap_or_default()).map_err(Into::into),
                Some(err) => Err(anyhow!(
                    "Request failed with {}: {}, data: {:?}",
                    err.code,
                    err.message,
                    err.data
                )),
            };
            callback(this, ret);
        };
        let req = self
            .req_queue
            .outgoing
            .register(R::METHOD.into(), params, Box::new(callback));
        self.lsp_tx.send(req.into()).unwrap();
    }

    fn send_notification<N: notif::Notification>(&self, params: N::Params) {
        self.lsp_tx
            .send(Notification::new(N::METHOD.into(), params).into())
            .unwrap();
    }

    // Maybe connect all tracing::* to LSP ShowMessage?
    fn show_message(&self, typ: MessageType, message: impl Into<String>) {
        let message = message.into();
        if typ == MessageType::ERROR {
            tracing::error!("{message}");
        }

        self.send_notification::<notif::ShowMessage>(ShowMessageParams { typ, message });
    }

    fn load_config(&mut self, callback: impl FnOnce(&mut Self) + 'static) {
        self.send_request::<req::WorkspaceConfiguration>(
            ConfigurationParams {
                items: vec![ConfigurationItem {
                    scope_uri: None,
                    section: Some(CONFIG_KEY.into()),
                }],
            },
            move |st, resp| {
                match resp {
                    Ok(mut v) => {
                        tracing::debug!("Updating config: {:?}", v);
                        st.update_config(v.pop().unwrap_or_default());
                    }
                    Err(err) => tracing::error!("Failed to update config: {}", err),
                }
                callback(st);
            },
        );
    }

    fn update_config(&mut self, value: serde_json::Value) {
        let mut config = Config::clone(&self.config);
        let (errors, updated_diagnostics) = config.update(value);
        tracing::debug!("Updated config, errors: {errors:?}, config: {config:?}");
        self.config = Arc::new(config);

        if !errors.is_empty() {
            let msg = ["Failed to apply some settings:"]
                .into_iter()
                .chain(errors.iter().flat_map(|s| ["\n- ", s]))
                .collect::<String>();
            self.show_message(MessageType::ERROR, msg);
        }

        // Refresh all diagnostics since the filter may be changed.
        if updated_diagnostics {
            let version = self.next_version();
            for uri in self.opened_files.keys() {
                tracing::trace!("Recalculate diagnostics of {uri}, version {version}");
                self.update_diagnostics(uri.clone(), version);
            }
        }
    }

    fn update_diagnostics(&self, uri: Url, version: u64) {
        let snap = self.snapshot();
        let task = move || {
            // Return empty diagnostics for ignored files.
            let diagnostics = (!snap.config.diagnostics_excluded_files.contains(&uri))
                .then(|| {
                    with_catch_unwind("diagnostics", || handler::diagnostics(snap, &uri))
                        .unwrap_or_else(|err| {
                            tracing::error!("Failed to calculate diagnostics: {err}");
                            Vec::new()
                        })
                })
                .unwrap_or_default();
            Event::Diagnostics {
                uri,
                version,
                diagnostics,
            }
        };
        self.task_tx.send(Box::new(task)).unwrap();
    }

    fn next_version(&mut self) -> u64 {
        self.version_counter += 1;
        self.version_counter
    }

    fn snapshot(&self) -> StateSnapshot {
        StateSnapshot {
            analysis: self.host.snapshot(),
            vfs: Arc::clone(&self.vfs),
            config: Arc::clone(&self.config),
        }
    }

    fn set_vfs_file_content(&mut self, uri: &Url, text: String) {
        let vpath = uri.to_vfs_path();
        self.vfs.write().unwrap().set_path_content(vpath, text);
        self.apply_vfs_change();
    }

    fn apply_vfs_change(&mut self) {
        let changes = self.vfs.write().unwrap().take_change();
        tracing::trace!("Change: {:?}", changes);
        let file_changes = changes.file_changes.clone();

        // N.B. This acquires the internal write lock.
        // Must be called without holding the lock of `vfs`.
        self.host.apply_change(changes);

        let version = self.next_version();
        let vfs = self.vfs.read().unwrap();
        for (file, text) in file_changes {
            let uri = vfs.uri_for_file(file);
            if !self.opened_files.contains_key(&uri) {
                continue;
            }

            // FIXME: Removed or closed files are indistinguishable from empty files.
            if !text.is_empty() {
                self.update_diagnostics(uri, version);
            } else {
                // Clear diagnostics.
                self.event_tx
                    .send(Event::Diagnostics {
                        uri,
                        version,
                        diagnostics: Vec::new(),
                    })
                    .unwrap();
            }
        }
    }
}

#[must_use = "RequestDispatcher::finish not called"]
struct RequestDispatcher<'s>(&'s mut Server, Option<Request>);

impl<'s> RequestDispatcher<'s> {
    fn on_sync_mut<R: req::Request>(
        mut self,
        f: fn(&mut Server, R::Params) -> Result<R::Result>,
    ) -> Self {
        if matches!(&self.1, Some(notif) if notif.method == R::METHOD) {
            let req = self.1.take().unwrap();
            let ret = (|| {
                let params = serde_json::from_value::<R::Params>(req.params)?;
                let v = f(self.0, params)?;
                Ok(serde_json::to_value(v).unwrap())
            })();
            let resp = result_to_response(req.id, ret);
            self.0.lsp_tx.send(resp.into()).unwrap();
        }
        self
    }

    fn on<R>(mut self, f: fn(StateSnapshot, R::Params) -> Result<R::Result>) -> Self
    where
        R: req::Request,
        R::Params: 'static,
        R::Result: 'static,
    {
        if matches!(&self.1, Some(notif) if notif.method == R::METHOD) {
            let req = self.1.take().unwrap();
            let snap = self.0.snapshot();
            self.0.req_queue.incoming.register(req.id.clone(), ());
            let task = move || {
                let ret = with_catch_unwind(R::METHOD, || {
                    let params = serde_json::from_value::<R::Params>(req.params)?;
                    let resp = f(snap, params)?;
                    Ok(serde_json::to_value(resp)?)
                });
                Event::Response(result_to_response(req.id, ret))
            };
            self.0.task_tx.send(Box::new(task)).unwrap();
        }
        self
    }

    fn finish(self) {
        if let Some(req) = self.1 {
            let resp = Response::new_err(req.id, ErrorCode::MethodNotFound as _, String::new());
            self.0.lsp_tx.send(resp.into()).unwrap();
        }
    }
}

#[must_use = "NotificationDispatcher::finish not called"]
struct NotificationDispatcher<'s>(&'s mut Server, Option<Notification>);

impl<'s> NotificationDispatcher<'s> {
    fn on_sync_mut<N: notif::Notification>(mut self, f: fn(&mut Server, N::Params)) -> Self {
        if matches!(&self.1, Some(notif) if notif.method == N::METHOD) {
            match serde_json::from_value::<N::Params>(self.1.take().unwrap().params) {
                Ok(params) => {
                    f(self.0, params);
                }
                Err(err) => {
                    tracing::error!("Failed to parse notification {}: {}", N::METHOD, err);
                }
            }
        }
        self
    }

    fn finish(self) {
        if let Some(notif) = self.1 {
            if !notif.method.starts_with("$/") {
                tracing::error!("Unhandled notification: {:?}", notif);
            }
        }
    }
}

fn with_catch_unwind<T>(ctx: &str, f: impl FnOnce() -> Result<T> + UnwindSafe) -> Result<T> {
    static INSTALL_PANIC_HOOK: Once = Once::new();
    thread_local! {
        static PANIC_LOCATION: Cell<String> = Cell::new(String::new());
    }

    INSTALL_PANIC_HOOK.call_once(|| {
        let old_hook = panic::take_hook();
        panic::set_hook(Box::new(move |info| {
            let loc = info
                .location()
                .map(|loc| loc.to_string())
                .unwrap_or_default();
            let backtrace = Backtrace::force_capture();
            PANIC_LOCATION.with(|inner| {
                inner.set(format!("Location: {loc:#}\nBacktrace: {backtrace:#}"));
            });
            old_hook(info);
        }));
    });

    match panic::catch_unwind(f) {
        Ok(ret) => ret,
        Err(payload) => {
            let reason = payload
                .downcast_ref::<String>()
                .map(|s| &**s)
                .or_else(|| payload.downcast_ref::<&str>().map(|s| &**s))
                .unwrap_or("unknown");
            let mut loc = PANIC_LOCATION.with(|inner| inner.take());
            if loc.is_empty() {
                loc = "Location: unknown".into();
            }
            tracing::error!("Panicked in {ctx}: {reason}\n{loc}");
            bail!("Panicked in {ctx}: {reason}\n{loc}");
        }
    }
}

fn result_to_response(id: RequestId, ret: Result<serde_json::Value>) -> Response {
    let err = match ret {
        Ok(v) => {
            return Response {
                id,
                result: Some(v),
                error: None,
            }
        }
        Err(err) => err,
    };

    if err.is::<Cancelled>() {
        // When client cancelled a request, a response is immediately sent back,
        // and the response will be ignored.
        return Response::new_err(id, ErrorCode::ServerCancelled as i32, "Cancelled".into());
    }
    if let Some(err) = err.downcast_ref::<LspError>() {
        return Response::new_err(id, err.code as i32, err.to_string());
    }
    if let Some(err) = err.downcast_ref::<serde_json::Error>() {
        return Response::new_err(id, ErrorCode::InvalidParams as i32, err.to_string());
    }
    Response::new_err(id, ErrorCode::InternalError as i32, err.to_string())
}

#[derive(Debug)]
pub struct StateSnapshot {
    pub(crate) analysis: Analysis,
    vfs: Arc<RwLock<Vfs>>,
    pub(crate) config: Arc<Config>,
}

impl StateSnapshot {
    pub(crate) fn vfs(&self) -> impl std::ops::Deref<Target = Vfs> + '_ {
        self.vfs.read().unwrap()
    }
}
