/* Copyright 2015 Google Inc. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

/// <reference path="../../typings/tsd.d.ts" />
/// <reference path="../plottable/plottable.d.ts" />

module TF {
  export module Urls {
    export type RunTagUrlFn = (tag: string, run: string) => string;
    export interface Router {
      runs: () => string;
      scalars: RunTagUrlFn;
      histograms: RunTagUrlFn;
      compressedHistograms: RunTagUrlFn;
      images: RunTagUrlFn;
      individualImage: (query: string) => string;
      graph: (run: string) => string;
    };

    export var routes = ["runs", "scalars", "histograms",
                         "compressedHistograms", "images",
                         "individualImage", "graph"];

    export function productionRouter(): Router {
      /* The standard router for communicating with the TensorBoard backend */
      function standardRoute(route: string): ((tag: string, run: string) => string) {
        return function(tag: string, run: string): string {
          return "/data/" + route + "?tag=" + encodeURIComponent(tag)
                                  + "&run=" + encodeURIComponent(run);
        };
      }
      function individualImageUrl(query: string) {
        return "/data/individualImage?" + query;
      }
      function graphUrl(run: string) {
        return "/data/graph?run=" + encodeURIComponent(run);
      }
      return {
        runs: () => "/data/runs",
        individualImage: individualImageUrl,
        graph: graphUrl,
        scalars: standardRoute("scalars"),
        histograms: standardRoute("histograms"),
        compressedHistograms: standardRoute("compressedHistograms"),
        images: standardRoute("images"),
      };
    };

    export function demoRouter(dataDir: string): Router {
      /* Retrieves static .json data generated by demo_from_server.py */
      function demoRoute(route) {
        return function(tag, run) {
          run = run.replace(/[ \)\(]/g, "_");
          tag = tag.replace(/[ \)\(]/g, "_");
          return dataDir + "/" + route + "/" + run + "/" + tag + ".json";
        };
      };
      function individualImageUrl(query) {
        return dataDir + "/individualImage/" + query + ".png";
      };
      function graphUrl(run) {
        run = run.replace(/ /g, "_");
        return dataDir + "/graph/" + run + ".pbtxt";
      };

      return {
        runs: () => dataDir + "/runs.json",
        individualImage: individualImageUrl,
        graph: graphUrl,
        scalars: demoRoute("scalars"),
        histograms: demoRoute("histograms"),
        compressedHistograms: demoRoute("compressedHistograms"),
        images: demoRoute("images"),
      };
    }
  }
}
