# Copyright 2016 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Tests for tf.contrib.tensor_forest.ops.scatter_add_ndim_op."""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import tensorflow as tf

from tensorflow.contrib.tensor_forest.python.ops import training_ops

from tensorflow.python.framework import test_util
from tensorflow.python.platform import googletest


class ScatterAddNdimTest(test_util.TensorFlowTestCase):

  def setUp(self):
    self.ops = training_ops.Load()

  def test1dim(self):
    input_data = tf.Variable([1., 2., 3., 4., 5., 6.,
                              7., 8., 9., 10., 11., 12.])
    indices = [[1], [10]]
    updates = [100., 200.]

    with self.test_session():
      tf.initialize_all_variables().run()
      self.ops.scatter_add_ndim(input_data, indices, updates).run()
      self.assertAllEqual([1., 102., 3., 4., 5., 6.,
                           7., 8., 9., 10., 211., 12.], input_data.eval())

  def test3dim(self):
    input_data = tf.Variable([[[1., 2., 3.], [4., 5., 6.]],
                              [[7., 8., 9.], [10., 11., 12.]]])
    indices = [[0, 0, 1], [1, 1, 2]]
    updates = [100., 200.]

    with self.test_session():
      tf.initialize_all_variables().run()
      self.ops.scatter_add_ndim(input_data, indices, updates).run()
      self.assertAllEqual([[[1., 102., 3.], [4., 5., 6.]],
                           [[7., 8., 9.], [10., 11., 212.]]], input_data.eval())

  def testNoUpdates(self):
    init_val = [[[1., 2., 3.], [4., 5., 6.]], [[7., 8., 9.], [10., 11., 12.]]]
    input_data = tf.Variable(init_val)
    indices = []
    updates = []

    with self.test_session():
      tf.initialize_all_variables().run()
      self.ops.scatter_add_ndim(input_data, indices, updates).run()
      self.assertAllEqual(init_val, input_data.eval())

  def testBadInput(self):
    init_val = [[[1., 2., 3.], [4., 5., 6.]], [[7., 8., 9.], [10., 11., 12.]]]
    input_data = tf.Variable(init_val)
    indices = [[0, 0, 1], [1, 1, 2]]
    updates = [100.]
    with self.test_session():
      tf.initialize_all_variables().run()
      with self.assertRaisesOpError(
          'Number of updates should be same as number of indices.'):
        self.ops.scatter_add_ndim(input_data, indices, updates).run()
        self.assertAllEqual(init_val, input_data.eval())

  def testIncompleteIndices(self):
    input_data = tf.Variable([[[1., 2., 3.], [4., 5., 6.]],
                              [[7., 8., 9.], [10., 11., 12.]]])
    indices = [[0, 0], [1, 1]]
    updates = [[100., 200., 300.], [400., 500., 600.]]

    with self.test_session():
      tf.initialize_all_variables().run()
      self.ops.scatter_add_ndim(input_data, indices, updates).run()
      self.assertAllEqual([[[101., 202., 303.], [4., 5., 6.]],
                           [[7., 8., 9.], [410., 511., 612.]]],
                          input_data.eval())


if __name__ == '__main__':
  googletest.main()
