#include "stdafx.h"
#include "util.h"

static acl::string __keypre("test_key_cluster");

static bool test_del(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;

	key.format("%s_%d", __keypre.c_str(), i);
	int ret = cmd.del(key.c_str(), NULL);
	if (ret < 0)
	{
		printf("del key: %s error\r\n", key.c_str());
		return false;
	}
	else if (i < 10)
		printf("del ok, key: %s\r\n", key.c_str());
	return true;
}

static bool test_expire(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;

	key.format("%s_%d", __keypre.c_str(), i);
	if (cmd.expire(key.c_str(), 100) < 0)
	{
		printf("expire key: %s error\r\n", key.c_str());
		return false;
	}
	else if (i < 10)
		printf("expire ok, key: %s\r\n", key.c_str());
	return true;
}

static bool test_ttl(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;
	int ttl;

	key.format("%s_%d", __keypre.c_str(), i);
	if ((ttl = cmd.ttl(key.c_str())) < 0)
	{
		printf("get ttl key: %s error\r\n", key.c_str());
		return false;
	}
	else if (i < 10)
		printf("ttl ok, key: %s, ttl: %d\r\n", key.c_str(), ttl);
	return true;
}

static bool test_exists(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;

	key.format("%s_%d", __keypre.c_str(), i);
	if (cmd.exists(key.c_str()) == false)
	{
		if (i < 10)
			printf("no exists key: %s\r\n", key.c_str());
	}
	else
	{
		if (i < 10)
			printf("exists key: %s\r\n", key.c_str());
	}
	return true;
}

static bool test_type(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;

	key.format("%s_%d", __keypre.c_str(), i);
	acl::redis_key_t ret = cmd.type(key.c_str());
	if (ret == acl::REDIS_KEY_NONE)
	{
		printf("unknown type key: %s\r\n", key.c_str());
		return false;
	}
	else if (i < 10)
		printf("type ok, key: %s, ret: %d\r\n", key.c_str(), ret);
	return true;
}

static bool test_set(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;
	key.format("%s_%d", __keypre.c_str(), i);

	acl::string value;
	value.format("value_%s", key.c_str());

	bool ret = cmd.set(key.c_str(), value.c_str());
	return ret;
	if (i < 10)
		printf("set key: %s, value: %s %s\r\n", key.c_str(),
			value.c_str(), ret ? "ok" : "error");
	return ret;
}

static bool test_get(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;
	key.format("%s_%d", __keypre.c_str(), i);

	acl::string value;

	bool ret = cmd.get(key.c_str(), value);
	if (i < 10)
		printf("get key: %s, value: %s %s, len: %d\r\n",
			key.c_str(), value.c_str(), ret ? "ok" : "error",
			(int) value.length());
	return ret;
}

static bool test_lrem(acl::redis& cmd, int i)
{
	cmd.clear();

	acl::string key;
	key.format("list_%s_%d", __keypre.c_str(), i);

	int ret = cmd.lrem(key.c_str(), 0, "list_value");
	if (i < 10)
		printf("lrem key: %s, ret: %d\r\n", key.c_str(), ret);
	return i >= 0 ? true : false;
}

static int __threads_exit = 0;

class test_thread : public acl::thread
{
public:
	test_thread(acl::locker& locker, acl::redis_client_cluster& cluster,
		int max_conns, const char* cmd, int n)
	: locker_(locker)
	, cluster_(cluster)
	, max_conns_(max_conns)
	, cmd_(cmd)
	, n_(n)
	{}

	~test_thread() {}

protected:
	virtual void* run()
	{
		bool ret;
		acl::redis cmd;
		cmd.set_cluster(&cluster_, max_conns_);

		for (int i = 0; i < n_; i++)
		{
			if (cmd_ == "set")
				ret = test_set(cmd, i);
			else if (cmd_ == "get")
				ret = test_get(cmd, i);
			else if (cmd_ == "del")
				ret = test_del(cmd, i);
			else if (cmd_ == "expire")
				ret = test_expire(cmd, i);
			else if (cmd_ == "ttl")
				ret = test_ttl(cmd, i);
			else if (cmd_ == "exists")
				ret = test_exists(cmd, i);
			else if (cmd_ == "type")
				ret = test_type(cmd, i);
			else if (cmd_ == "lrem")
				ret = test_lrem(cmd, i);
			else if (cmd_ == "all")
			{
				if (test_set(cmd, i) == false
					|| test_get(cmd, i) == false
					|| test_exists(cmd, i) == false
					|| test_type(cmd, i) == false
					|| test_expire(cmd, i) == false
					|| test_ttl(cmd, i) == false
					|| test_del(cmd, i) == false)
				{
					ret = false;
				}
				else
					ret = true;
			}
			else
			{
				printf("unknown cmd: %s\r\n", cmd_.c_str());
				break;
			}

			if (ret == false)
			{
				printf("cmd: %s error, tid: %lu\r\n",
					cmd_.c_str(), thread_self());
				break;
			}

			if (i > 0 && i % 1000 == 0)
			{
				char tmp[128];
				acl::safe_snprintf(tmp, sizeof(tmp), "%d", i);
				acl::meter_time(__FILE__, __LINE__, tmp);
			}
		}

		locker_.lock();
		__threads_exit++;
		locker_.unlock();

		return NULL;
	}

private:
	acl::locker& locker_;
	acl::redis_client_cluster& cluster_;
	int max_conns_;
	acl::string cmd_;
	int n_;
};

static void usage(const char* procname)
{
	printf("usage: %s -h[help]\r\n"
		"-s redis_addr_list[127.0.0.1:6379]\r\n"
		"-n count[default: 10]\r\n"
		"-C connect_timeout[default: 10]\r\n"
		"-I rw_timeout[default: 10]\r\n"
		"-c max_threads[default: 10]\r\n"
		"-w wait_for_cluster_resume[default: 500 ms]\r\n"
		"-r retry_for_cluster_resnum[default: 10]\r\n"
		"-p [preset all hash-slots of the cluster]\r\n"
		"-a cmd[set|get|expire|ttl|exists|type|del|lrem]\r\n",
		procname);
}

int main(int argc, char* argv[])
{
	int  ch, n = 1, conn_timeout = 10, rw_timeout = 10;
	int  max_threads = 10, nsleep = 500, nretry = 10;
	acl::string addrs("127.0.0.1:6379"), cmd;
	bool preset = false;

	while ((ch = getopt(argc, argv, "hs:n:C:I:c:a:w:r:p")) > 0)
	{
		switch (ch)
		{
		case 'h':
			usage(argv[0]);
			return 0;
		case 's':
			addrs = optarg;
			break;
		case 'n':
			n = atoi(optarg);
			break;
		case 'C':
			conn_timeout = atoi(optarg);
			break;
		case 'I':
			rw_timeout = atoi(optarg);
			break;
		case 'c':
			max_threads = atoi(optarg);
			break;
		case 'a':
			cmd = optarg;
			break;
		case 'w':
			nsleep = atoi(optarg);
			break;
		case 'r':
			nretry = atoi(optarg);
			break;
		case 'p':
			preset = true;
			break;
		default:
			break;
		}
	}

	acl::acl_cpp_init();
	acl::log::stdout_open(true);

	acl::redis_client_cluster cluster(conn_timeout, rw_timeout);

	// ĳӳؽ⣬̽ӽǷָʱ()ֵ
	// Ϊ 0 ʱ򲻼
	cluster.set_retry_inter(1);

	// ضֵض˷ֵ򱨴
	cluster.set_redirect_max(nretry);

	// ض >= 2 ʱÿض˺Ϣʱ()
	cluster.set_redirect_sleep(nsleep);

	cluster.init(NULL, addrs.c_str(), max_threads);

	// ǷҪйϣ۵ĶӦϵǰúãȥʱ̬
	// ϣ۵Ḷ́ӶʱЧ
	if (preset)
	{
		const std::vector<acl::string>& token = addrs.split2(",; \t");
		cluster.set_all_slot(token[0], max_threads);
	}

	struct timeval begin;
	gettimeofday(&begin, NULL);

	acl::locker locker;

	std::vector<test_thread*> threads;
	for (int i = 0; i < max_threads; i++)
	{
		test_thread* thread = new test_thread(locker, cluster,
			max_threads, cmd.c_str(), n);
		threads.push_back(thread);
		thread->set_detachable(true);
		thread->start();
	}

	while (true)
	{
		locker.lock();
		if (__threads_exit == max_threads)
		{
			locker.unlock();
			printf("All threads over now!\r\n");
			break;
		}
		locker.unlock();

		//printf("max_threads: %d, threads_exit: %d\r\n",
		//	max_threads, __threads_exit);
		sleep(1);
	}

	std::vector<test_thread*>::iterator it = threads.begin();
	for (; it != threads.end(); ++it)
	{
		//(*it)->wait();
		delete (*it);
	}

	struct timeval end;
	gettimeofday(&end, NULL);

	long long int total = max_threads * n;
	double inter = util::stamp_sub(&end, &begin);
	printf("total %s: %lld, spent: %0.2f ms, speed: %0.2f\r\n", cmd.c_str(),
		total, inter, (total * 1000) /(inter > 0 ? inter : 1));

#ifdef WIN32
	printf("enter any key to exit\r\n");
	getchar();
#endif
	return 0;
}
