// Copyright (c) 2019 Minoru Osuka
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// 		http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"encoding/json"
	"errors"
	"fmt"
	"os"

	"github.com/golang/protobuf/ptypes/any"
	"github.com/mosuka/blast/index"
	"github.com/mosuka/blast/protobuf"
	pbindex "github.com/mosuka/blast/protobuf/index"
	"github.com/urfave/cli"
)

func execIndex(c *cli.Context) error {
	grpcAddr := c.String("grpc-addr")

	id := c.Args().Get(0)
	if id == "" {
		err := errors.New("key argument must be set")
		return err
	}

	fields := c.Args().Get(1)
	if fields == "" {
		err := errors.New("value argument must be set")
		return err
	}

	// string -> map[string]interface{}
	var fieldsMap map[string]interface{}
	err := json.Unmarshal([]byte(fields), &fieldsMap)
	if err != nil {
		return err
	}

	// map[string]interface{} -> Any
	fieldsAny := &any.Any{}
	err = protobuf.UnmarshalAny(fieldsMap, fieldsAny)
	if err != nil {
		return err
	}

	// create document
	doc := &pbindex.Document{
		Id:     id,
		Fields: fieldsAny,
	}

	client, err := index.NewGRPCClient(grpcAddr)
	if err != nil {
		return err
	}
	defer func() {
		err := client.Close()
		if err != nil {
			fmt.Fprintln(os.Stderr, err)
		}
	}()

	err = client.Index(doc)
	if err != nil {
		return err
	}

	return nil
}
