//  Copyright (c) 2014 Couchbase, Inc.
//  Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file
//  except in compliance with the License. You may obtain a copy of the License at
//    http://www.apache.org/licenses/LICENSE-2.0
//  Unless required by applicable law or agreed to in writing, software distributed under the
//  License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
//  either express or implied. See the License for the specific language governing permissions
//  and limitations under the License.

package bleve

import (
	"github.com/blevesearch/bleve/index"
	"github.com/blevesearch/bleve/search"
)

type queryStringQuery struct {
	Query    string  `json:"query"`
	BoostVal float64 `json:"boost,omitempty"`
}

// NewQueryStringQuery creates a new Query used for
// finding documents that satisfy a query string.  The
// query string is a small query language for humans.
func NewQueryStringQuery(query string) *queryStringQuery {
	return &queryStringQuery{
		Query:    query,
		BoostVal: 1.0,
	}
}

func (q *queryStringQuery) Boost() float64 {
	return q.BoostVal
}

func (q *queryStringQuery) SetBoost(b float64) Query {
	q.BoostVal = b
	return q
}

func (q *queryStringQuery) Searcher(i index.IndexReader, m *IndexMapping, explain bool) (search.Searcher, error) {
	newQuery, err := parseQuerySyntax(q.Query)
	if err != nil {
		return nil, err
	}
	return newQuery.Searcher(i, m, explain)
}

func (q *queryStringQuery) Validate() error {
	newQuery, err := parseQuerySyntax(q.Query)
	if err != nil {
		return err
	}
	return newQuery.Validate()
}

func (q *queryStringQuery) Field() string {
	return ""
}

func (q *queryStringQuery) SetField(f string) Query {
	return q
}
