//
//
// Copyright 2016 gRPC authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//

#include <string.h>

#include <grpc/byte_buffer.h>
#include <grpc/load_reporting.h>
#include <grpc/support/alloc.h>
#include <grpc/support/log.h>
#include <grpc/support/string_util.h>
#include <grpc/support/time.h>

#include "src/core/ext/filters/load_reporting/server_load_reporting_filter.h"
#include "src/core/ext/filters/load_reporting/server_load_reporting_plugin.h"
#include "src/core/lib/channel/channel_args.h"
#include "src/core/lib/gprpp/crash.h"
#include "test/core/end2end/cq_verifier.h"
#include "test/core/end2end/end2end_tests.h"

enum { TIMEOUT = 200000 };

static void* tag(intptr_t t) { return (void*)t; }

typedef struct {
  gpr_mu mu;
  intptr_t channel_id;
  intptr_t call_id;

  char* initial_md_str;
  char* trailing_md_str;
  char* method_name;

  uint64_t incoming_bytes;
  uint64_t outgoing_bytes;

  grpc_status_code call_final_status;

  bool fully_processed;
} load_reporting_data;

static std::unique_ptr<CoreTestFixture> begin_test(
    const CoreTestConfiguration& config, const char* test_name,
    grpc_channel_args* client_args, grpc_channel_args* server_args) {
  gpr_log(GPR_INFO, "Running test: %s/%s", test_name, config.name);

  auto f = config.create_fixture(grpc_core::ChannelArgs::FromC(client_args),
                                 grpc_core::ChannelArgs::FromC(server_args));
  f->InitServer(grpc_core::ChannelArgs::FromC(server_args));
  f->InitClient(grpc_core::ChannelArgs::FromC(client_args));

  return f;
}

static void request_response_with_payload(
    const CoreTestConfiguration& config, CoreTestFixture f,
    const char* method_name, const char* request_msg, const char* response_msg,
    grpc_metadata* initial_lr_metadata, grpc_metadata* trailing_lr_metadata) {
  grpc_slice request_payload_slice = grpc_slice_from_static_string(request_msg);
  grpc_slice response_payload_slice =
      grpc_slice_from_static_string(response_msg);
  grpc_call* c;
  grpc_call* s;
  grpc_byte_buffer* request_payload =
      grpc_raw_byte_buffer_create(&request_payload_slice, 1);
  grpc_byte_buffer* response_payload =
      grpc_raw_byte_buffer_create(&response_payload_slice, 1);
  grpc_core::CqVerifier cqv(f->cq());
  grpc_op ops[6];
  grpc_op* op;
  grpc_metadata_array initial_metadata_recv;
  grpc_metadata_array trailing_metadata_recv;
  grpc_metadata_array request_metadata_recv;
  grpc_byte_buffer* request_payload_recv = nullptr;
  grpc_byte_buffer* response_payload_recv = nullptr;
  grpc_call_details call_details;
  grpc_status_code status;
  grpc_call_error error;
  grpc_slice details;
  int was_cancelled = 2;

  gpr_timespec deadline = grpc_timeout_seconds_to_deadline(5);
  c = grpc_channel_create_call(
      f->client(), nullptr, GRPC_PROPAGATE_DEFAULTS, f->cq(),
      grpc_slice_from_static_string(method_name), nullptr, deadline, nullptr);
  GPR_ASSERT(c);

  grpc_metadata_array_init(&initial_metadata_recv);
  grpc_metadata_array_init(&trailing_metadata_recv);
  grpc_metadata_array_init(&request_metadata_recv);
  grpc_call_details_init(&call_details);

  memset(ops, 0, sizeof(ops));
  op = ops;
  op->op = GRPC_OP_SEND_INITIAL_METADATA;
  GPR_ASSERT(initial_lr_metadata != nullptr);
  op->data.send_initial_metadata.count = 1;
  op->data.send_initial_metadata.metadata = initial_lr_metadata;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_SEND_MESSAGE;
  op->data.send_message.send_message = request_payload;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_SEND_CLOSE_FROM_CLIENT;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_RECV_INITIAL_METADATA;
  op->data.recv_initial_metadata.recv_initial_metadata = &initial_metadata_recv;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_RECV_MESSAGE;
  op->data.recv_message.recv_message = &response_payload_recv;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_RECV_STATUS_ON_CLIENT;
  op->data.recv_status_on_client.trailing_metadata = &trailing_metadata_recv;
  op->data.recv_status_on_client.status = &status;
  op->data.recv_status_on_client.status_details = &details;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  error = grpc_call_start_batch(c, ops, static_cast<size_t>(op - ops),
                                grpc_core::CqVerifier::tag(1), nullptr);
  GPR_ASSERT(GRPC_CALL_OK == error);

  error = grpc_server_request_call(f->server(), &s, &call_details,
                                   &request_metadata_recv, f->cq(), f->cq(),
                                   grpc_core::CqVerifier::tag(101));
  GPR_ASSERT(GRPC_CALL_OK == error);
  cqv.Expect(grpc_core::CqVerifier::tag(101), true);
  cqv.Verify();

  memset(ops, 0, sizeof(ops));
  op = ops;
  op->op = GRPC_OP_SEND_INITIAL_METADATA;
  op->data.send_initial_metadata.count = 0;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_RECV_MESSAGE;
  op->data.recv_message.recv_message = &request_payload_recv;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  error = grpc_call_start_batch(s, ops, static_cast<size_t>(op - ops),
                                grpc_core::CqVerifier::tag(102), nullptr);
  GPR_ASSERT(GRPC_CALL_OK == error);

  cqv.Expect(grpc_core::CqVerifier::tag(102), true);
  cqv.Verify();

  memset(ops, 0, sizeof(ops));
  op = ops;
  op->op = GRPC_OP_RECV_CLOSE_ON_SERVER;
  op->data.recv_close_on_server.cancelled = &was_cancelled;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_SEND_MESSAGE;
  op->data.send_message.send_message = response_payload;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  op->op = GRPC_OP_SEND_STATUS_FROM_SERVER;
  GPR_ASSERT(trailing_lr_metadata != nullptr);
  op->data.send_status_from_server.trailing_metadata_count = 1;
  op->data.send_status_from_server.trailing_metadata = trailing_lr_metadata;
  op->data.send_status_from_server.status = GRPC_STATUS_OK;
  grpc_slice status_details = grpc_slice_from_static_string("xyz");
  op->data.send_status_from_server.status_details = &status_details;
  op->flags = 0;
  op->reserved = nullptr;
  op++;
  error = grpc_call_start_batch(s, ops, static_cast<size_t>(op - ops),
                                grpc_core::CqVerifier::tag(103), nullptr);
  GPR_ASSERT(GRPC_CALL_OK == error);

  cqv.Expect(grpc_core::CqVerifier::tag(103), true);
  cqv.Expect(grpc_core::CqVerifier::tag(1), true);
  cqv.Verify();

  GPR_ASSERT(status == GRPC_STATUS_OK);

  grpc_slice_unref(details);
  grpc_metadata_array_destroy(&initial_metadata_recv);
  grpc_metadata_array_destroy(&trailing_metadata_recv);
  grpc_metadata_array_destroy(&request_metadata_recv);
  grpc_call_details_destroy(&call_details);

  grpc_call_unref(c);
  grpc_call_unref(s);

  grpc_byte_buffer_destroy(request_payload);
  grpc_byte_buffer_destroy(response_payload);
  grpc_byte_buffer_destroy(request_payload_recv);
  grpc_byte_buffer_destroy(response_payload_recv);
}

// override the default for testing purposes
extern void (*g_load_reporting_fn)(
    const grpc_load_reporting_call_data* call_data);

static void test_load_reporting_hook(const CoreTestConfiguration& config) {
  // TODO(dgq): this test is currently a noop until LR is fully defined.
  // Leaving the rest here, as it'll likely be reusable.

  // Introduce load reporting for the server through its arguments
  grpc_arg arg = grpc_load_reporting_enable_arg();
  grpc_channel_args* lr_server_args =
      grpc_channel_args_copy_and_add(nullptr, &arg, 1);

  auto f =
      begin_test(config, "test_load_reporting_hook", nullptr, lr_server_args);

  const char* method_name = "/gRPCFTW";
  const char* request_msg = "the msg from the client";
  const char* response_msg = "... and the response from the server";

  grpc_metadata initial_lr_metadata;
  grpc_metadata trailing_lr_metadata;

  initial_lr_metadata.key = GRPC_MDSTR_LB_TOKEN;
  initial_lr_metadata.value = grpc_slice_from_static_string("client-token");
  memset(&initial_lr_metadata.internal_data, 0,
         sizeof(initial_lr_metadata.internal_data));

  trailing_lr_metadata.key = GRPC_MDSTR_LB_COST_BIN;
  trailing_lr_metadata.value = grpc_slice_from_static_string("server-token");
  memset(&trailing_lr_metadata.internal_data, 0,
         sizeof(trailing_lr_metadata.internal_data));

  request_response_with_payload(config, f, method_name, request_msg,
                                response_msg, &initial_lr_metadata,
                                &trailing_lr_metadata);

  {
    grpc_core::ExecCtx exec_ctx;
    grpc_channel_args_destroy(lr_server_args);
  }
}

void load_reporting_hook(const CoreTestConfiguration& config) {
  test_load_reporting_hook(config);
}

void load_reporting_hook_pre_init(void) {}
