//
//
// Copyright 2023 gRPC authors.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//

#include <chrono>
#include <csignal>
#include <iostream>
#include <memory>
#include <string>
#include <thread>

#include "absl/flags/flag.h"
#include "absl/flags/parse.h"
#include "absl/strings/str_format.h"

#include <grpcpp/ext/gcp_observability.h>
#include <grpcpp/ext/proto_server_reflection_plugin.h>
#include <grpcpp/grpcpp.h>
#include <grpcpp/health_check_service_interface.h>

#ifdef BAZEL_BUILD
#include "examples/protos/helloworld.grpc.pb.h"
#else
#include "helloworld.grpc.pb.h"
#endif

using grpc::Server;
using grpc::ServerBuilder;
using grpc::ServerContext;
using grpc::Status;
using helloworld::Greeter;
using helloworld::HelloReply;
using helloworld::HelloRequest;

ABSL_FLAG(uint16_t, port, 50051, "Server port for the service");

namespace {

volatile std::sig_atomic_t g_shutdown_flag = 0;

void signal_handler(int signal) {
  g_shutdown_flag = 1;
  std::signal(signal, SIG_DFL);
}

// Logic and data behind the server's behavior.W
class GreeterServiceImpl final : public Greeter::Service {
  Status SayHello(ServerContext* context, const HelloRequest* request,
                  HelloReply* reply) override {
    std::string prefix("Hello ");
    reply->set_message(prefix + request->name());
    return Status::OK;
  }
};

void RunServer(uint16_t port) {
  std::string server_address = absl::StrFormat("0.0.0.0:%d", port);
  GreeterServiceImpl service;
  grpc::EnableDefaultHealthCheckService(true);
  grpc::reflection::InitProtoReflectionServerBuilderPlugin();
  ServerBuilder builder;
  // Listen on the given address without any authentication mechanism.
  builder.AddListeningPort(server_address, grpc::InsecureServerCredentials());
  // Register "service" as the instance through which we'll communicate with
  // clients. In this case it corresponds to an *synchronous* service.
  builder.RegisterService(&service);
  // Finally assemble the server.
  std::unique_ptr<Server> server(builder.BuildAndStart());
  std::cout << "Server listening on " << server_address << std::endl;
  // Instead of server->Wait(), we are waiting on a shutdown notification from
  // SIGINT.
  while (!g_shutdown_flag) {
    std::this_thread::sleep_for(std::chrono::milliseconds(100));
  }
  server->Shutdown();
}

}  // namespace

int main(int argc, char** argv) {
  absl::ParseCommandLine(argc, argv);
  // Install a signal handler for an indication to shut down server and flush
  // out observability data;
  std::signal(SIGINT, signal_handler);
  // Turn on GCP Observability for the whole binary. Based on the configuration,
  // this will emit observability data (stats, tracing and logging) to GCP
  // backends. Note that this should be done before any other gRPC operation.
  auto status = grpc::experimental::GcpObservabilityInit();
  if (!status.ok()) {
    std::cerr << "GcpObservabilityInit() failed: " << status.ToString()
              << std::endl;
    return static_cast<int>(status.code());
  }
  std::cout << "Initialized GCP Observability" << std::endl;
  RunServer(absl::GetFlag(FLAGS_port));
  // Flush out any pending Observability data
  std::cout << "Closing GCP Observability" << std::endl;
  grpc::experimental::GcpObservabilityClose();
  std::cout << "Sleeping for 25 seconds to make sure Observability stats and "
               "tracing are flushed.(Another Ctrl+C will immediately exit the "
               "program.)"
            << std::endl;
  // Currently, GcpObservabilityClose() only supports flushing logs. Stats and
  // tracing get automatically flushed at a regular interval, so sleep for an
  // interval to make sure that those are flushed too.
  std::this_thread::sleep_for(std::chrono::seconds(25));
  return 0;
}
