---
id: jsx-in-depth
title: JSX in Depth
permalink: jsx-in-depth.html
prev: displaying-data.html
next: jsx-spread.html
---

[JSX](http://facebook.github.io/jsx/) is a JavaScript syntax extension that looks similar to XML. You can use a simple JSX syntactic transform with React.

## Why JSX?

You don't have to use JSX with React. You can just use plain JS. However, we recommend using JSX because it is a concise and familiar syntax for defining tree structures with attributes.

It's more familiar for casual developers such as designers.

XML has the benefit of balanced open and closing tags. This helps make large trees easier to read than function calls or object literals.

It doesn't alter the semantics of JavaScript.

## HTML Tags vs. React Components

React can either render HTML tags (strings) or React components (classes).

To render a HTML tag, just use lower-case tag names in JSX:

```javascript
var myDivElement = <div className="foo" />;
React.render(myDivElement, document.body);
```

To render a React Component, just a local variable that starts with an upper-case letter:

```javascript
var MyComponent = React.createClass({/*...*/});
var myElement = <MyComponent someProperty={true} />;
React.render(myElement, document.body);
```

React's JSX uses the upper vs. lower case convention to distinguish between local component classes and HTML tags.

> Note:
>
> Since JSX is JavaScript, identifiers such as `class` and `for` are discouraged
> as XML attribute names. Instead, React DOM components expect DOM property
> names like `className` and `htmlFor`, respectively.

## The Transform

React JSX transforms from an XML-like syntax into native JavaScript. XML elements, attributes and children are transformed into arguments to `React.createElement`.

```javascript
var Nav;
// Input (JSX):
var app = <Nav color="blue" />;
// Output (JS):
var app = React.createElement(Nav, {color:"blue"});
```

Notice that in order to use `<Nav />`, the `Nav` variable must be in scope.

JSX also allows specifying children using XML syntax:

```javascript
var Nav, Profile;
// Input (JSX):
var app = <Nav color="blue"><Profile>click</Profile></Nav>;
// Output (JS):
var app = React.createElement(
  Nav,
  {color:"blue"},
  React.createElement(Profile, null, "click")
);
```

Use the [JSX Compiler](/react/jsx-compiler.html) to try out JSX and see how it
desugars into native JavaScript, and the
[HTML to JSX converter](/react/html-jsx.html) to convert your existing HTML to
JSX.

If you want to use JSX, the [Getting Started](/react/docs/getting-started.html) guide shows how to setup compilation.

> Note:
>
> The JSX expression always evaluates to a ReactElement. The actual
> implementation details may vary. An optimized mode could inline the 
> ReactElement as an object literal to bypass the validation code in
> `React.createElement`.

## JavaScript Expressions

### Attribute Expressions

To use a JavaScript expression as an attribute value, wrap the expression in a
pair of curly braces (`{}`) instead of quotes (`""`).

```javascript
// Input (JSX):
var person = <Person name={window.isLoggedIn ? window.name : ''} />;
// Output (JS):
var person = React.createElement(
  Person,
  {name: window.isLoggedIn ? window.name : ''}
);
```

### Child Expressions

Likewise, JavaScript expressions may be used to express children:

```javascript
// Input (JSX):
var content = <Container>{window.isLoggedIn ? <Nav /> : <Login />}</Container>;
// Output (JS):
var content = React.createElement(
  Container,
  null,
  window.isLoggedIn ? React.createElement(Nav) : React.createElement(Login)
);
```

### Comments

It's easy to add comments within your JSX; they're just JS expressions. You just need to be careful to put `{}` around the comments when you are within the children section of a tag.

```javascript
var content = (
  <Nav>
    {/* child comment, put {} around */}
    <Person
      /* multi
         line
         comment */
      name={window.isLoggedIn ? window.name : ''} // end of line comment
    />
  </Nav>
);
```

> NOTE:
> 
> JSX is similar to HTML, but not exactly the same. See [JSX gotchas](/react/docs/jsx-gotchas.html) for some key differences.
