/* Copyright 2017 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

// Native XLA implementations of indexing ops.

#include "tensorflow/compiler/tf2xla/kernels/index_ops.h"

#include "tensorflow/compiler/tf2xla/type_util.h"
#include "tensorflow/compiler/tf2xla/xla_helpers.h"
#include "tensorflow/compiler/tf2xla/xla_op_kernel.h"
#include "tensorflow/compiler/tf2xla/xla_op_registry.h"
#include "tensorflow/compiler/xla/client/lib/arithmetic.h"
#include "tensorflow/compiler/xla/shape_util.h"
#include "tensorflow/core/framework/kernel_def_builder.h"
#include "tensorflow/core/framework/op_kernel.h"
#include "tensorflow/core/framework/register_types.h"
#include "tensorflow/core/framework/tensor.h"
#include "tensorflow/core/framework/tensor_shape.h"
#include "tensorflow/core/kernels/bounds_check.h"

namespace tensorflow {
XlaArgMinMaxOp::XlaArgMinMaxOp(OpKernelConstruction* ctx, bool is_min)
    : XlaOpKernel(ctx), is_min_(is_min) {}

void XlaArgMinMaxOp::Compile(XlaOpKernelContext* ctx) {
  const TensorShape input_shape = ctx->InputShape(0);
  const TensorShape dimension_shape = ctx->InputShape(1);

  OP_REQUIRES(ctx, TensorShapeUtils::IsScalar(dimension_shape),
              errors::InvalidArgument(
                  "dim must be a scalar, but received tensor of shape: ",
                  dimension_shape.DebugString()));

  int64 dim;
  OP_REQUIRES_OK(ctx, ctx->ConstantInputAsIntScalar(1, &dim));

  const int input_dims = input_shape.dims();
  const int axis = dim < 0 ? dim + input_dims : dim;

  OP_REQUIRES(
      ctx, axis >= 0 && axis < input_dims,
      errors::InvalidArgument("Expected dimension in the range [", -input_dims,
                              ", ", input_dims, "), but got ", dim));
  const int64 axis_size = input_shape.dim_size(axis);
  OP_REQUIRES(
      ctx, axis_size > 0,
      errors::InvalidArgument("Reduction axis ", dim, " is empty in shape ",
                              input_shape.DebugString()));

  DataType index_type = output_type(0);

  xla::ComputationBuilder* b = ctx->builder();
  xla::ComputationDataHandle input = ctx->Input(0);

  xla::ComputationDataHandle output;
  if (is_min_) {
    OP_REQUIRES_OK(ctx,
                   XlaHelpers::ArgMin(b, ctx, input, input_shape, input_type(0),
                                      index_type, axis, &output));
  } else {
    OP_REQUIRES_OK(ctx,
                   XlaHelpers::ArgMax(b, ctx, input, input_shape, input_type(0),
                                      index_type, axis, &output));
  }

  ctx->SetOutput(0, output);
}

XlaArgMaxOp::XlaArgMaxOp(OpKernelConstruction* ctx)
    : XlaArgMinMaxOp(ctx, /*is_min=*/false) {}
REGISTER_XLA_OP(Name("ArgMax")
                    .Device(DEVICE_GPU_XLA_JIT)
                    .CompileTimeConstInput("dimension"),
                XlaArgMaxOp);

namespace {

class XlaArgMinOp : public XlaArgMinMaxOp {
 public:
  explicit XlaArgMinOp(OpKernelConstruction* ctx);
};
XlaArgMinOp::XlaArgMinOp(OpKernelConstruction* ctx)
    : XlaArgMinMaxOp(ctx, /*is_min=*/true) {}
REGISTER_XLA_OP(Name("ArgMin").CompileTimeConstInput("dimension"), XlaArgMinOp);

}  // namespace
}  // namespace tensorflow
