﻿using Grand.Business.Common.Extensions;
using Grand.Business.Common.Interfaces.Configuration;
using Grand.Business.Common.Interfaces.Localization;
using Grand.Infrastructure.Plugins;

namespace Widgets.FacebookPixel
{
    public class FacebookPixelPlugin : BasePlugin, IPlugin
    {
        private readonly ISettingService _settingService;
        private readonly ITranslationService _translationService;
        private readonly ILanguageService _languageService;

        public FacebookPixelPlugin(
            ISettingService settingService, 
            ITranslationService translationService, 
            ILanguageService languageService)
        {
            _settingService = settingService;
            _translationService = translationService;
            _languageService = languageService;
        }

        /// <summary>
        /// Gets a configuration page URL
        /// </summary>
        public override string ConfigurationUrl()
        {
            return FacebookPixelDefaults.ConfigurationUrl;
        }

        public override async Task Install()
        {
            var settings = new FacebookPixelSettings
            {
                PixelId = "000000000000000",
                PixelScript = @"<!-- Facebook Pixel Code -->
                                    <script>
                                      !function(f,b,e,v,n,t,s)
                                      {if(f.fbq)return;n=f.fbq=function(){n.callMethod?
                                      n.callMethod.apply(n,arguments):n.queue.push(arguments)};
                                      if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';
                                      n.queue=[];t=b.createElement(e);t.async=!0;
                                      t.src=v;s=b.getElementsByTagName(e)[0];
                                      s.parentNode.insertBefore(t,s)}(window, document,'script',
                                      'https://connect.facebook.net/en_US/fbevents.js');
                                      fbq('init', '{PIXELID}');
                                      fbq('track', 'PageView');
                                    </script>
                                    <noscript>
                                      <img height='1' width='1' style='display:none'
                                           src='https://www.facebook.com/tr?id={PIXELID}&ev=PageView&noscript=1'/>
                                    </noscript>
                                <!-- End Facebook Pixel Code -->",
                AddToCartScript = @"<script> fbq('track', 'AddToCart', {contents: [{ id: '{PRODUCTID}', quantity: {QTY} }], content_name: '{PRODUCTNAME}', content_type: 'product', value: {AMOUNT}, currency: '{CURRENCY}'}); </script>",
                DetailsOrderScript = @"<script> fbq('track', 'Purchase', {currency: '{CURRENCY}', value: {AMOUNT}}); </script>",
                ConsentName = "Facebook pixel",
                ConsentDescription = "Allows us to identifies visitors from Facebook"
            };
            await _settingService.SaveSetting(settings);

            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.FriendlyName", "Facebook Pixel");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelId", "Pixel Id");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelId.Hint", "Enter Facebook Pixel ID.");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelScript", "Facebook Pixel Code");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelScript.Hint", "Paste the facebook pixel code generated by Facebook here. {PIXELID} will be dynamically replaced.");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.AddToCartScript", "Facebook Event - Add to cart");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.AddToCartScript.Hint", "Paste the event add to cart pixel code. {PRODUCTNAME}, {PRODUCTID}, {QTY}, {AMOUNT}, {CURRENCY} will be dynamically replaced.");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.DetailsOrderScript", "Facebook Event - Purchase");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.DetailsOrderScript.Hint", "Paste the event purchase pixel code. {AMOUNT} and {CURRENCY} will be dynamically replaced.");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.AllowToDisableConsentCookie", "Allow disabling consent cookie");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.AllowToDisableConsentCookie.Hint", "Get or set the value to disable consent cookie");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDefaultState", "Consent default state");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDefaultState.Hint", "Get or set the value to consent default state");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentName", "Consent cookie name");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentName.Hint", "Get or set the value to consent cookie name");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDescription", "Consent cookie description");
            await this.AddOrUpdatePluginTranslateResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDescription.Hint", "Get or set the value to consent cookie description");

            await base.Install();
        }

        /// <summary>
        /// Uninstall plugin
        /// </summary>
        public override async Task Uninstall()
        {
            //settings
            await _settingService.DeleteSetting<FacebookPixelSettings>();

            //locales
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelId");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelId.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelScript");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.PixelScript.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.AddToCartScript");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.AddToCartScript.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.DetailsOrderScript");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.DetailsOrderScript.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.AllowToDisableConsentCookie");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.AllowToDisableConsentCookie.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDefaultState");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDefaultState.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentName");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentName.Hint");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDescription");
            await this.DeletePluginTranslationResource(_translationService, _languageService, "Widgets.FacebookPixel.ConsentDescription.Hint");

            await base.Uninstall();
        }


    }
}
