using System.Collections.ObjectModel;

namespace Grand.Business.Core.Interfaces.Common.Directory;

/// <summary>
///     Represents a datetime service
/// </summary>
public interface IDateTimeService
{
    /// <summary>
    ///     Gets the current user time zone
    /// </summary>
    TimeZoneInfo CurrentTimeZone { get; }

    /// <summary>
    ///     Retrieves a System.TimeZoneInfo object from the registry based on its identifier.
    /// </summary>
    /// <param name="id">The time zone identifier, which corresponds to the System.TimeZoneInfo.Id property.</param>
    /// <returns>A System.TimeZoneInfo object whose identifier is the value of the id parameter.</returns>
    TimeZoneInfo FindTimeZoneById(string id);

    /// <summary>
    ///     Returns a sorted collection of all the time zones
    /// </summary>
    /// <returns>A read-only collection of System.TimeZoneInfo objects.</returns>
    ReadOnlyCollection<TimeZoneInfo> GetSystemTimeZones();

    /// <summary>
    ///     Converts the date and time to current user date and time
    /// </summary>
    /// <param name="dt">The date and time (represents local system time or UTC time) to convert.</param>
    /// <returns>A DateTime value that represents time that corresponds to the dateTime parameter in customer time zone.</returns>
    DateTime ConvertToUserTime(DateTime dt);

    /// <summary>
    ///     Converts the date and time to current user date and time
    /// </summary>
    /// <param name="dt">The date and time (represents local system time or UTC time) to convert.</param>
    /// <param name="sourceDateTimeKind">The source datetime kind</param>
    /// <returns>A DateTime value that represents time that corresponds to the dateTime parameter in customer time zone.</returns>
    DateTime ConvertToUserTime(DateTime dt, DateTimeKind sourceDateTimeKind);

    /// <summary>
    ///     Converts the date and time to current user date and time
    /// </summary>
    /// <param name="dt">The date and time to convert.</param>
    /// <param name="sourceTimeZone">The time zone of dateTime.</param>
    /// <param name="destinationTimeZone">The time zone to convert dateTime to.</param>
    /// <returns>A DateTime value that represents time that corresponds to the dateTime parameter in customer time zone.</returns>
    DateTime ConvertToUserTime(DateTime dt, TimeZoneInfo sourceTimeZone, TimeZoneInfo destinationTimeZone);

    /// <summary>
    ///     Converts the date and time to Coordinated Universal Time (UTC)
    /// </summary>
    /// <param name="dt">The date and time (represents local system time or UTC time) to convert.</param>
    /// <returns>
    ///     A DateTime value that represents the Coordinated Universal Time (UTC) that corresponds to the dateTime
    ///     parameter. The DateTime value's Kind property is always set to DateTimeKind.Utc.
    /// </returns>
    DateTime ConvertToUtcTime(DateTime dt);

    /// <summary>
    ///     Converts the date and time to Coordinated Universal Time (UTC)
    /// </summary>
    /// <param name="dt">The date and time (represents local system time or UTC time) to convert.</param>
    /// <param name="sourceDateTimeKind">The source datetime kind</param>
    /// <returns>
    ///     A DateTime value that represents the Coordinated Universal Time (UTC) that corresponds to the dateTime
    ///     parameter. The DateTime value's Kind property is always set to DateTimeKind.Utc.
    /// </returns>
    DateTime ConvertToUtcTime(DateTime dt, DateTimeKind sourceDateTimeKind);

    /// <summary>
    ///     Converts the date and time to Coordinated Universal Time (UTC)
    /// </summary>
    /// <param name="dt">The date and time to convert.</param>
    /// <param name="sourceTimeZone">The time zone of dateTime.</param>
    /// <returns>
    ///     A DateTime value that represents the Coordinated Universal Time (UTC) that corresponds to the dateTime
    ///     parameter. The DateTime value's Kind property is always set to DateTimeKind.Utc.
    /// </returns>
    DateTime ConvertToUtcTime(DateTime dt, TimeZoneInfo sourceTimeZone);
}