# Copyright 2017 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
"""Tests for call_trees module."""

from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import numpy as np

from tensorflow.contrib.autograph.converters import call_trees
from tensorflow.contrib.autograph.converters import converter_test_base
from tensorflow.python.framework import constant_op
from tensorflow.python.framework import dtypes
from tensorflow.python.framework import ops
from tensorflow.python.ops import math_ops
from tensorflow.python.platform import test


class CallTreesTest(converter_test_base.TestCase):

  def test_basic(self):

    def test_fn_1(_):
      raise ValueError('This should not be called in the compiled version.')

    def renamed_test_fn_1(a):
      return a + 1

    def test_fn_2(a):
      return test_fn_1(a) + 1

    node = self.parse_and_analyze(test_fn_2, {'test_fn_1': test_fn_1})
    node = call_trees.transform(node, self.ctx, (), ())

    with self.compiled(node) as result:
      # Only test_fn_2 is transformed, so we'll insert renamed_test_fn_1
      # manually.
      result.renamed_test_fn_1 = renamed_test_fn_1
      self.assertEquals(3, result.test_fn_2(1))

  def test_dynamic_function(self):

    def test_fn_1():
      raise ValueError('This should be masked by the mock.')

    def test_fn_2(f):
      return f() + 3

    node = self.parse_and_analyze(test_fn_2, {})
    node = call_trees.transform(node, self.ctx, (), ())

    with self.compiled(node) as result:
      # 10 = 7 (from the mock) + 3 (from test_fn_2)
      self.assertEquals(10, result.test_fn_2(test_fn_1))

  def test_simple_methods(self):

    class TestClass(object):

      def test_fn_1(self, a):
        return a + 1

      def test_fn_2(self, a):
        return self.test_fn_1(a) + 1

    node = self.parse_and_analyze(
        TestClass.test_fn_2, {'TestClass': TestClass},
        namer=converter_test_base.FakeNoRenameNamer(),
        arg_types={'self': (TestClass.__name__, TestClass)})
    node = call_trees.transform(node, self.ctx, (), ())

    with self.compiled(node) as result:
      tc = TestClass()
      self.assertEquals(3, result.test_fn_2(tc, 1))

  def test_py_func_wrap_no_retval(self):

    def test_fn(a):
      setattr(a, 'foo', 'bar')

    node = self.parse_and_analyze(test_fn, {'setattr': setattr})
    node = call_trees.transform(node, self.ctx, (), ())

    with self.compiled(node) as result:
      with self.test_session() as sess:
        # The function has no return value, so we do some tricks to grab the
        # generated py_func node and ensure its effect only happens at graph
        # execution.

        class Dummy(object):
          pass

        a = Dummy()
        result.test_fn(a)
        self.assertFalse(hasattr(a, 'foo'))
        sess.run(sess.graph.get_operations()[0])
        self.assertEquals('bar', a.foo)

  def test_py_func_wrap_known_function(self):

    def test_fn():
      return np.random.binomial(2, 0.5)

    node = self.parse_and_analyze(test_fn, {'np': np})
    node = call_trees.transform(node, self.ctx, (), ())

    with self.compiled(node, dtypes.int64) as result:
      result.np = np
      with self.test_session() as sess:
        self.assertTrue(isinstance(result.test_fn(), ops.Tensor))
        self.assertIn(sess.run(result.test_fn()), (0, 1, 2))

  def test_uncompiled_modules(self):

    def test_fn(a):
      a = math_ops.multiply(a, constant_op.constant(2))
      a = math_ops.add(a, constant_op.constant(1))
      return a

    node = self.parse_and_analyze(test_fn, {
        'math_ops': math_ops,
        'constant_op': constant_op
    })
    node = call_trees.transform(node, self.ctx,
                                set(((math_ops.__name__,),
                                     (constant_op.__name__,))), ())

    with self.compiled(node) as result:
      result.math_ops = math_ops
      result.constant_op = constant_op
      with self.test_session() as sess:
        # Not renamed, because the converter doesn't rename the definition
        # itself (the caller is responsible for that).
        result_tensor = result.test_fn(constant_op.constant(1))
        self.assertEquals(3, sess.run(result_tensor))


if __name__ == '__main__':
  test.main()
