# Copyright 2015 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================
# pylint: disable=g-bad-import-order,unused-import
"""Tests the graph freezing tool."""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

import tensorflow as tf
import os

from tensorflow.examples.image_retraining import retrain
from tensorflow.python.framework import test_util


class ImageRetrainingTest(test_util.TensorFlowTestCase):

  def dummyImageLists(self):
    return {'label_one': {'dir': 'somedir', 'training': ['image_one.jpg',
                                                         'image_two.jpg'],
                          'testing': ['image_three.jpg', 'image_four.jpg'],
                          'validation': ['image_five.jpg', 'image_six.jpg']},
            'label_two': {'dir': 'otherdir', 'training': ['image_one.jpg',
                                                          'image_two.jpg'],
                          'testing': ['image_three.jpg', 'image_four.jpg'],
                          'validation': ['image_five.jpg', 'image_six.jpg']}}

  def testGetImagePath(self):
    image_lists = self.dummyImageLists()
    self.assertEqual('image_dir/somedir/image_one.jpg', retrain.get_image_path(
        image_lists, 'label_one', 0, 'image_dir', 'training'))
    self.assertEqual('image_dir/otherdir/image_four.jpg',
                     retrain.get_image_path(image_lists, 'label_two', 1,
                                            'image_dir', 'testing'))

  def testGetBottleneckPath(self):
    image_lists = self.dummyImageLists()
    self.assertEqual('bottleneck_dir/somedir/image_five.jpg_imagenet_v3.txt',
                     retrain.get_bottleneck_path(
                         image_lists, 'label_one', 0, 'bottleneck_dir',
                         'validation', 'imagenet_v3'))

  def testShouldDistortImage(self):
    self.assertEqual(False, retrain.should_distort_images(False, 0, 0, 0))
    self.assertEqual(True, retrain.should_distort_images(True, 0, 0, 0))
    self.assertEqual(True, retrain.should_distort_images(False, 10, 0, 0))
    self.assertEqual(True, retrain.should_distort_images(False, 0, 1, 0))
    self.assertEqual(True, retrain.should_distort_images(False, 0, 0, 50))

  def testAddInputDistortions(self):
    with tf.Graph().as_default():
      with tf.Session() as sess:
        retrain.add_input_distortions(True, 10, 10, 10, 299, 299, 3, 128, 128)
        self.assertIsNotNone(sess.graph.get_tensor_by_name('DistortJPGInput:0'))
        self.assertIsNotNone(sess.graph.get_tensor_by_name('DistortResult:0'))

  @tf.test.mock.patch.object(retrain, 'FLAGS', learning_rate=0.01)
  def testAddFinalRetrainOps(self, flags_mock):
    with tf.Graph().as_default():
      with tf.Session() as sess:
        bottleneck = tf.placeholder(tf.float32, [1, 1024], name='bottleneck')
        # Test creating final training op with quantization.
        retrain.add_final_retrain_ops(5, 'final', bottleneck, 1024, False,
                                      False)
        self.assertIsNotNone(sess.graph.get_tensor_by_name('final:0'))

  @tf.test.mock.patch.object(retrain, 'FLAGS', learning_rate=0.01)
  def testAddFinalRetrainOpsQuantized(self, flags_mock):
    # Ensure that the training and eval graph for quantized models are correctly
    # created.
    with tf.Graph().as_default() as g:
      with tf.Session() as sess:
        bottleneck = tf.placeholder(tf.float32, [1, 1024], name='bottleneck')
        # Test creating final training op with quantization, set is_training to
        # true.
        retrain.add_final_retrain_ops(5, 'final', bottleneck, 1024, True, True)
        self.assertIsNotNone(sess.graph.get_tensor_by_name('final:0'))
        found_fake_quant = 0
        for op in g.get_operations():
          if op.type == 'FakeQuantWithMinMaxVars':
            found_fake_quant += 1
            # Ensure that the inputs of each FakeQuant operations has 2 Assign
            # operations in the training graph (Assign[Min,Max]Last,
            # Assign[Min,Max]Ema)
            self.assertEqual(2,
                             len([i for i in op.inputs if 'Assign' in i.name]))
        self.assertEqual(found_fake_quant, 2)
    with tf.Graph().as_default() as g:
      with tf.Session() as sess:
        bottleneck = tf.placeholder(tf.float32, [1, 1024], name='bottleneck')
        # Test creating final training op with quantization, set is_training to
        # false.
        retrain.add_final_retrain_ops(5, 'final', bottleneck, 1024, True, False)
        self.assertIsNotNone(sess.graph.get_tensor_by_name('final:0'))
        found_fake_quant = 0
        for op in g.get_operations():
          if op.type == 'FakeQuantWithMinMaxVars':
            found_fake_quant += 1
            for i in op.inputs:
              # Ensure that no operations are Assign operation since this is the
              # evaluation graph.
              self.assertTrue('Assign' not in i.name)
        self.assertEqual(found_fake_quant, 2)

  def testAddEvaluationStep(self):
    with tf.Graph().as_default():
      final = tf.placeholder(tf.float32, [1], name='final')
      gt = tf.placeholder(tf.int64, [1], name='gt')
      self.assertIsNotNone(retrain.add_evaluation_step(final, gt))

  def testAddJpegDecoding(self):
    with tf.Graph().as_default():
      jpeg_data, mul_image = retrain.add_jpeg_decoding(10, 10, 3, 0, 255)
      self.assertIsNotNone(jpeg_data)
      self.assertIsNotNone(mul_image)

  def testCreateModelInfo(self):
    did_raise_value_error = False
    try:
      retrain.create_model_info('no_such_model_name')
    except ValueError:
      did_raise_value_error = True
    self.assertTrue(did_raise_value_error)
    model_info = retrain.create_model_info('inception_v3')
    self.assertIsNotNone(model_info)
    self.assertEqual(299, model_info['input_width'])

  def testCreateModelInfoQuantized(self):
    # Test for mobilenet_quantized
    model_info = retrain.create_model_info('mobilenet_1.0_224')
    self.assertIsNotNone(model_info)
    self.assertEqual(224, model_info['input_width'])


if __name__ == '__main__':
  tf.test.main()
