#!/usr/bin/env python

import glob
import os
import re
import shutil
import subprocess
import sys
import stat

from lib.config import LIBCHROMIUMCONTENT_COMMIT, BASE_URL, PLATFORM, \
                       get_target_arch, get_chromedriver_version, \
                       get_platform_key
from lib.util import scoped_cwd, rm_rf, get_atom_shell_version, make_zip, \
                     execute, atom_gyp


ATOM_SHELL_VERSION = get_atom_shell_version()

SOURCE_ROOT = os.path.abspath(os.path.dirname(os.path.dirname(__file__)))
DIST_DIR = os.path.join(SOURCE_ROOT, 'dist')
OUT_DIR = os.path.join(SOURCE_ROOT, 'out', 'R')
CHROMIUM_DIR = os.path.join(SOURCE_ROOT, 'vendor', 'brightray', 'vendor',
                            'download', 'libchromiumcontent', 'static_library')

PROJECT_NAME = atom_gyp()['project_name%']
PRODUCT_NAME = atom_gyp()['product_name%']

TARGET_BINARIES = {
  'darwin': [
  ],
  'win32': [
    '{0}.exe'.format(PROJECT_NAME),  # 'electron.exe'
    'content_shell.pak',
    'd3dcompiler_47.dll',
    'icudtl.dat',
    'libEGL.dll',
    'libGLESv2.dll',
    'msvcp120.dll',
    'msvcr120.dll',
    'ffmpeg.dll',
    'node.dll',
    'pdf.dll',
    'content_resources_200_percent.pak',
    'ui_resources_200_percent.pak',
    'xinput1_3.dll',
    'natives_blob.bin',
    'snapshot_blob.bin',
    'vccorlib120.dll',
  ],
  'linux': [
    PROJECT_NAME,  # 'electron'
    'content_shell.pak',
    'icudtl.dat',
    'libffmpeg.so',
    'libnode.so',
    'natives_blob.bin',
    'snapshot_blob.bin',
  ],
}
TARGET_DIRECTORIES = {
  'darwin': [
    '{0}.app'.format(PRODUCT_NAME),
  ],
  'win32': [
    'resources',
    'locales',
  ],
  'linux': [
    'resources',
    'locales',
  ],
}

SYSTEM_LIBRARIES = [
  'libgcrypt.so',
]


def main():
  rm_rf(DIST_DIR)
  os.makedirs(DIST_DIR)

  target_arch = get_target_arch()

  force_build()
  create_symbols()
  copy_binaries()
  copy_chrome_binary('chromedriver')
  copy_chrome_binary('mksnapshot')
  copy_license()

  if PLATFORM == 'linux':
    strip_binaries()
    if target_arch != 'arm':
      copy_system_libraries()

  create_version()
  create_dist_zip()
  create_chrome_binary_zip('chromedriver', get_chromedriver_version())
  create_chrome_binary_zip('mksnapshot', ATOM_SHELL_VERSION)
  create_ffmpeg_zip()
  create_symbols_zip()


def force_build():
  build = os.path.join(SOURCE_ROOT, 'script', 'build.py')
  execute([sys.executable, build, '-c', 'Release'])


def copy_binaries():
  for binary in TARGET_BINARIES[PLATFORM]:
    shutil.copy2(os.path.join(OUT_DIR, binary), DIST_DIR)

  for directory in TARGET_DIRECTORIES[PLATFORM]:
    shutil.copytree(os.path.join(OUT_DIR, directory),
                    os.path.join(DIST_DIR, directory),
                    symlinks=True)


def copy_chrome_binary(binary):
  if PLATFORM == 'win32':
    binary += '.exe'
  src = os.path.join(CHROMIUM_DIR, binary)
  dest = os.path.join(DIST_DIR, binary)

  # Copy file and keep the executable bit.
  shutil.copyfile(src, dest)
  os.chmod(dest, os.stat(dest).st_mode | stat.S_IEXEC)


def copy_license():
  shutil.copy2(os.path.join(CHROMIUM_DIR, '..', 'LICENSES.chromium.html'),
               DIST_DIR)
  shutil.copy2(os.path.join(SOURCE_ROOT, 'LICENSE'), DIST_DIR)


def strip_binaries():
  if get_target_arch() == 'arm':
    strip = 'arm-linux-gnueabihf-strip'
  else:
    strip = 'strip'
  for binary in TARGET_BINARIES[PLATFORM]:
    if binary.endswith('.so') or '.' not in binary:
      execute([strip, os.path.join(DIST_DIR, binary)])


def copy_system_libraries():
  executable_path = os.path.join(OUT_DIR, PROJECT_NAME)  # our/R/electron
  ldd = execute(['ldd', executable_path])
  lib_re = re.compile('\t(.*) => (.+) \(.*\)$')
  for line in ldd.splitlines():
    m = lib_re.match(line)
    if not m:
      continue
    for i, library in enumerate(SYSTEM_LIBRARIES):
      real_library = m.group(1)
      if real_library.startswith(library):
        shutil.copyfile(m.group(2), os.path.join(DIST_DIR, real_library))
        SYSTEM_LIBRARIES[i] = real_library


def create_version():
  version_path = os.path.join(SOURCE_ROOT, 'dist', 'version')
  with open(version_path, 'w') as version_file:
    version_file.write(ATOM_SHELL_VERSION)


def create_symbols():
  destination = os.path.join(DIST_DIR, '{0}.breakpad.syms'.format(PROJECT_NAME))
  dump_symbols = os.path.join(SOURCE_ROOT, 'script', 'dump-symbols.py')
  execute([sys.executable, dump_symbols, destination])

  if PLATFORM == 'darwin':
    dsyms = glob.glob(os.path.join(OUT_DIR, '*.dSYM'))
    for dsym in dsyms:
      shutil.copytree(dsym, os.path.join(DIST_DIR, os.path.basename(dsym)))


def create_dist_zip():
  dist_name = '{0}-{1}-{2}-{3}.zip'.format(PROJECT_NAME, ATOM_SHELL_VERSION,
                                           get_platform_key(),
                                           get_target_arch())
  zip_file = os.path.join(SOURCE_ROOT, 'dist', dist_name)

  with scoped_cwd(DIST_DIR):
    files = TARGET_BINARIES[PLATFORM] +  ['LICENSE', 'LICENSES.chromium.html',
                                          'version']
    if PLATFORM == 'linux':
      files += [lib for lib in SYSTEM_LIBRARIES if os.path.exists(lib)]
    dirs = TARGET_DIRECTORIES[PLATFORM]
    make_zip(zip_file, files, dirs)


def create_chrome_binary_zip(binary, version):
  dist_name = '{0}-{1}-{2}-{3}.zip'.format(binary, version, get_platform_key(),
                                           get_target_arch())
  zip_file = os.path.join(SOURCE_ROOT, 'dist', dist_name)

  with scoped_cwd(DIST_DIR):
    files = ['LICENSE', 'LICENSES.chromium.html']
    if PLATFORM == 'win32':
      files += [binary + '.exe']
    else:
      files += [binary]
    make_zip(zip_file, files, [])


def create_ffmpeg_zip():
  dist_name = 'ffmpeg-{0}-{1}-{2}.zip'.format(
      ATOM_SHELL_VERSION, get_platform_key(), get_target_arch())
  zip_file = os.path.join(SOURCE_ROOT, 'dist', dist_name)

  if PLATFORM == 'darwin':
    ffmpeg_name = 'libffmpeg.dylib'
  elif PLATFORM == 'linux':
    ffmpeg_name = 'libffmpeg.so'
  elif PLATFORM == 'win32':
    ffmpeg_name = 'ffmpeg.dll'

  shutil.copy2(os.path.join(CHROMIUM_DIR, '..', 'ffmpeg', ffmpeg_name),
               DIST_DIR)
  with scoped_cwd(DIST_DIR):
    make_zip(zip_file, [ffmpeg_name, 'LICENSE', 'LICENSES.chromium.html'], [])


def create_symbols_zip():
  dist_name = '{0}-{1}-{2}-{3}-symbols.zip'.format(PROJECT_NAME,
                                                   ATOM_SHELL_VERSION,
                                                   get_platform_key(),
                                                   get_target_arch())
  zip_file = os.path.join(DIST_DIR, dist_name)
  licenses = ['LICENSE', 'LICENSES.chromium.html', 'version']

  with scoped_cwd(DIST_DIR):
    dirs = ['{0}.breakpad.syms'.format(PROJECT_NAME)]
    make_zip(zip_file, licenses, dirs)

  if PLATFORM == 'darwin':
    dsym_name = '{0}-{1}-{2}-{3}-dsym.zip'.format(PROJECT_NAME,
                                                  ATOM_SHELL_VERSION,
                                                  get_platform_key(),
                                                  get_target_arch())
    with scoped_cwd(DIST_DIR):
      dsyms = glob.glob('*.dSYM')
      make_zip(os.path.join(DIST_DIR, dsym_name), licenses, dsyms)


if __name__ == '__main__':
  sys.exit(main())
