package egin

import (
	healthcheck "github.com/RaMin0/gin-health-check"
	"github.com/opentracing/opentracing-go"

	"github.com/gotomicro/ego/core/econf"
	"github.com/gotomicro/ego/core/elog"
	"github.com/gotomicro/ego/core/util/xnet"
)

// Container 容器
type Container struct {
	config *Config
	name   string
	logger *elog.Component
}

// DefaultContainer 默认容器
func DefaultContainer() *Container {
	return &Container{
		config: DefaultConfig(),
		logger: elog.EgoLogger.With(elog.FieldComponent(PackageName)),
	}
}

// Load 加载配置key
func Load(key string) *Container {
	c := DefaultContainer()
	c.logger = c.logger.With(elog.FieldComponentName(key))
	if err := econf.UnmarshalKey(key, &c.config); err != nil {
		c.logger.Panic("parse config error", elog.FieldErr(err), elog.FieldKey(key))
		return c
	}
	var (
		host string
		err  error
	)
	// 获取网卡ip
	if c.config.EnableLocalMainIP {
		host, _, err = xnet.GetLocalMainIP()
		if err != nil {
			host = ""
		}
		c.config.Host = host
	}
	c.name = key
	return c
}

// Build 构建组件
func (c *Container) Build(options ...Option) *Component {
	for _, option := range options {
		option(c)
	}
	server := newComponent(c.name, c.config, c.logger)
	server.Use(healthcheck.Default())
	server.Use(defaultServerInterceptor(c.logger, c.config))

	if c.config.EnableMetricInterceptor {
		server.Use(metricServerInterceptor())
	}

	if c.config.EnableTraceInterceptor && opentracing.IsGlobalTracerRegistered() {
		server.Use(traceServerInterceptor())
	}

	if c.config.EnableSentinel {
		server.Use(sentinelMiddleware(c.config))
	}

	return server
}
