//
// Copyright Aliaksei Levin (levlam@telegram.org), Arseny Smirnov (arseny30@gmail.com) 2014-2020
//
// Distributed under the Boost Software License, Version 1.0. (See accompanying
// file LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//
#include "td/telegram/MessagesManager.h"

#include "td/telegram/AuthManager.h"
#include "td/telegram/ChatId.h"
#include "td/telegram/ConfigShared.h"
#include "td/telegram/ContactsManager.h"
#include "td/telegram/DialogDb.h"
#include "td/telegram/DialogLocation.h"
#include "td/telegram/DraftMessage.h"
#include "td/telegram/DraftMessage.hpp"
#include "td/telegram/FileReferenceManager.h"
#include "td/telegram/files/FileId.hpp"
#include "td/telegram/files/FileLocation.h"
#include "td/telegram/files/FileManager.h"
#include "td/telegram/files/FileType.h"
#include "td/telegram/Global.h"
#include "td/telegram/InlineQueriesManager.h"
#include "td/telegram/InputMessageText.h"
#include "td/telegram/Location.h"
#include "td/telegram/logevent/LogEvent.h"
#include "td/telegram/logevent/LogEventHelper.h"
#include "td/telegram/MessageContent.h"
#include "td/telegram/MessageEntity.h"
#include "td/telegram/MessageEntity.hpp"
#include "td/telegram/MessagesDb.h"
#include "td/telegram/misc.h"
#include "td/telegram/net/DcId.h"
#include "td/telegram/net/NetActor.h"
#include "td/telegram/net/NetQuery.h"
#include "td/telegram/NotificationGroupType.h"
#include "td/telegram/NotificationManager.h"
#include "td/telegram/NotificationSettings.hpp"
#include "td/telegram/NotificationType.h"
#include "td/telegram/Payments.h"
#include "td/telegram/ReplyMarkup.h"
#include "td/telegram/ReplyMarkup.hpp"
#include "td/telegram/SecretChatActor.h"
#include "td/telegram/SecretChatsManager.h"
#include "td/telegram/SequenceDispatcher.h"
#include "td/telegram/Td.h"
#include "td/telegram/TdDb.h"
#include "td/telegram/TopDialogCategory.h"
#include "td/telegram/UpdatesManager.h"
#include "td/telegram/Version.h"
#include "td/telegram/WebPageId.h"

#include "td/actor/PromiseFuture.h"
#include "td/actor/SleepActor.h"

#include "td/db/binlog/BinlogEvent.h"
#include "td/db/binlog/BinlogHelper.h"
#include "td/db/SqliteKeyValue.h"
#include "td/db/SqliteKeyValueAsync.h"

#include "td/utils/format.h"
#include "td/utils/misc.h"
#include "td/utils/Random.h"
#include "td/utils/Slice.h"
#include "td/utils/Time.h"
#include "td/utils/tl_helpers.h"
#include "td/utils/tl_storers.h"
#include "td/utils/utf8.h"

#include <algorithm>
#include <cstring>
#include <iterator>
#include <limits>
#include <set>
#include <tuple>
#include <type_traits>
#include <unordered_map>
#include <unordered_set>
#include <utility>

namespace td {

void dummyUpdate::store(TlStorerToString &s, const char *field_name) const {
  s.store_class_begin(field_name, "dummyUpdate");
  s.store_class_end();
}

class GetOnlinesQuery : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    CHECK(dialog_id.get_type() == DialogType::Channel);
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }

    send_query(
        G()->net_query_creator().create(create_storer(telegram_api::messages_getOnlines(std::move(input_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getOnlines>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    td->messages_manager_->on_update_dialog_online_member_count(dialog_id_, result->onlines_, true);
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetOnlinesQuery");
    td->messages_manager_->on_update_dialog_online_member_count(dialog_id_, 0, true);
  }
};

class GetAllDraftsQuery : public Td::ResultHandler {
 public:
  void send() {
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getAllDrafts())));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getAllDrafts>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetAllDraftsQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));
  }

  void on_error(uint64 id, Status status) override {
    LOG(ERROR) << "Receive error for GetAllDraftsQuery: " << status;
    status.ignore();
  }
};

class GetDialogQuery : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getPeerDialogs(
        td->messages_manager_->get_input_dialog_peers({dialog_id}, AccessRights::Read)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive chat: " << to_string(result);

    td->contacts_manager_->on_get_users(std::move(result->users_), "GetDialogQuery");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetDialogQuery");
    td->messages_manager_->on_get_dialogs(
        FolderId(), std::move(result->dialogs_), -1, std::move(result->messages_),
        PromiseCreator::lambda([td = td, dialog_id = dialog_id_](Result<> result) {
          if (result.is_ok()) {
            td->messages_manager_->on_get_dialog_query_finished(dialog_id, Status::OK());
          } else {
            if (G()->close_flag()) {
              return;
            }
            td->messages_manager_->on_get_dialog_error(dialog_id, result.error(), "OnGetDialogs");
            td->messages_manager_->on_get_dialog_query_finished(dialog_id, result.move_as_error());
          }
        }));
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogQuery");
    td->messages_manager_->on_get_dialog_query_finished(dialog_id_, std::move(status));
  }
};

class GetPinnedDialogsActor : public NetActorOnce {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit GetPinnedDialogsActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  NetQueryRef send(FolderId folder_id, uint64 sequence_id) {
    folder_id_ = folder_id;
    auto query =
        G()->net_query_creator().create(create_storer(telegram_api::messages_getPinnedDialogs(folder_id.get())));
    auto result = query.get_weak();
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_id);
    return result;
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getPinnedDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive pinned chats: " << to_string(result);

    td->contacts_manager_->on_get_users(std::move(result->users_), "GetPinnedDialogsActor");
    td->contacts_manager_->on_get_chats(std::move(result->chats_), "GetPinnedDialogsActor");
    std::reverse(result->dialogs_.begin(), result->dialogs_.end());
    td->messages_manager_->on_get_dialogs(folder_id_, std::move(result->dialogs_), -2, std::move(result->messages_),
                                          std::move(promise_));
  }

  void on_error(uint64 id, Status status) override {
    promise_.set_error(std::move(status));
  }
};

class GetDialogUnreadMarksQuery : public Td::ResultHandler {
 public:
  void send() {
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getDialogUnreadMarks())));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogUnreadMarks>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto results = result_ptr.move_as_ok();
    for (auto &result : results) {
      td->messages_manager_->on_update_dialog_is_marked_as_unread(DialogId(result), true);
    }

    G()->td_db()->get_binlog_pmc()->set("fetched_marks_as_unread", "1");
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for GetDialogUnreadMarksQuery: " << status;
    }
    status.ignore();
  }
};

class GetMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit GetMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<tl_object_ptr<telegram_api::InputMessage>> &&message_ids) {
    send_query(
        G()->net_query_creator().create(create_storer(telegram_api::messages_getMessages(std::move(message_ids)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetMessagesQuery");
    LOG_IF(ERROR, info.is_channel_messages) << "Receive channel messages in GetMessagesQuery";
    td->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, false,
                                           "GetMessagesQuery");

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    promise_.set_error(std::move(status));
  }
};

class GetChannelMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit GetChannelMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, tl_object_ptr<telegram_api::InputChannel> &&input_channel,
            vector<tl_object_ptr<telegram_api::InputMessage>> &&message_ids) {
    channel_id_ = channel_id;
    CHECK(input_channel != nullptr);
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::channels_getMessages(std::move(input_channel), std::move(message_ids)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_getMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetChannelMessagesQuery");
    LOG_IF(ERROR, !info.is_channel_messages) << "Receive ordinary messages in GetChannelMessagesQuery";
    td->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, false,
                                           "GetChannelMessagesQuery");

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    td->contacts_manager_->on_get_channel_error(channel_id_, status, "GetChannelMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetScheduledMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit GetScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, tl_object_ptr<telegram_api::InputPeer> &&input_peer, vector<int32> &&message_ids) {
    dialog_id_ = dialog_id;
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_getScheduledMessages(std::move(input_peer), std::move(message_ids)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetScheduledMessagesQuery");
    LOG_IF(ERROR, info.is_channel_messages != (dialog_id_.get_type() == DialogType::Channel))
        << "Receive wrong messages constructor in GetScheduledMessagesQuery";
    td->messages_manager_->on_get_messages(std::move(info.messages), info.is_channel_messages, true,
                                           "GetScheduledMessagesQuery");

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.message() == "MESSAGE_IDS_EMPTY") {
      promise_.set_value(Unit());
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetScheduledMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogPinnedMessageQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId message_id_;

 public:
  explicit UpdateDialogPinnedMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, bool disable_notification) {
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't update pinned message because have no write access to " << dialog_id;
      return on_error(0, Status::Error(500, "Can't update pinned message"));
    }

    int32 flags = 0;
    if (disable_notification) {
      flags |= telegram_api::messages_updatePinnedMessage::SILENT_MASK;
    }

    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_updatePinnedMessage(
        flags, false /*ignored*/, std::move(input_peer), message_id.get_server_message_id().get()))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_updatePinnedMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for UpdateDialogPinnedMessageQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      td->messages_manager_->on_update_dialog_pinned_message_id(dialog_id_, message_id_);
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UpdateDialogPinnedMessageQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class ExportChannelMessageLinkQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId message_id_;
  bool for_group_ = false;
  bool ignore_result_ = false;

 public:
  explicit ExportChannelMessageLinkQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId message_id, bool for_group, bool ignore_result) {
    channel_id_ = channel_id;
    message_id_ = message_id;
    for_group_ = for_group;
    ignore_result_ = ignore_result;
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);
    send_query(G()->net_query_creator().create(create_storer(telegram_api::channels_exportMessageLink(
        std::move(input_channel), message_id.get_server_message_id().get(), for_group))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_exportMessageLink>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(DEBUG) << "Receive result for ExportChannelMessageLinkQuery: " << to_string(ptr);
    if (!ignore_result_) {
      td->messages_manager_->on_get_public_message_link({DialogId(channel_id_), message_id_}, for_group_,
                                                        std::move(ptr->link_), std::move(ptr->html_));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!ignore_result_) {
      td->contacts_manager_->on_get_channel_error(channel_id_, status, "ExportChannelMessageLinkQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogListActor : public NetActorOnce {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit GetDialogListActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, int32 offset_date, ServerMessageId offset_message_id, DialogId offset_dialog_id,
            int32 limit, uint64 sequence_id) {
    folder_id_ = folder_id;
    auto input_peer = td->messages_manager_->get_input_peer(offset_dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      input_peer = make_tl_object<telegram_api::inputPeerEmpty>();
    }

    int32 flags =
        telegram_api::messages_getDialogs::EXCLUDE_PINNED_MASK | telegram_api::messages_getDialogs::FOLDER_ID_MASK;
    auto query = G()->net_query_creator().create(
        create_storer(telegram_api::messages_getDialogs(flags, false /*ignored*/, folder_id.get(), offset_date,
                                                        offset_message_id.get(), std::move(input_peer), limit, 0)));
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetDialogListActor: " << to_string(ptr);
    switch (ptr->get_id()) {
      case telegram_api::messages_dialogs::ID: {
        auto dialogs = move_tl_object_as<telegram_api::messages_dialogs>(ptr);
        td->contacts_manager_->on_get_users(std::move(dialogs->users_), "GetDialogListActor");
        td->contacts_manager_->on_get_chats(std::move(dialogs->chats_), "GetDialogListActor");
        td->messages_manager_->on_get_dialogs(folder_id_, std::move(dialogs->dialogs_),
                                              narrow_cast<int32>(dialogs->dialogs_.size()),
                                              std::move(dialogs->messages_), std::move(promise_));
        break;
      }
      case telegram_api::messages_dialogsSlice::ID: {
        auto dialogs = move_tl_object_as<telegram_api::messages_dialogsSlice>(ptr);
        td->contacts_manager_->on_get_users(std::move(dialogs->users_), "GetDialogListActor");
        td->contacts_manager_->on_get_chats(std::move(dialogs->chats_), "GetDialogListActor");
        td->messages_manager_->on_get_dialogs(folder_id_, std::move(dialogs->dialogs_), max(dialogs->count_, 0),
                                              std::move(dialogs->messages_), std::move(promise_));
        break;
      }
      case telegram_api::messages_dialogsNotModified::ID:
        LOG(ERROR) << "Receive " << to_string(ptr);
        return on_error(id, Status::Error(500, "Receive wrong server response messages.dialogsNotModified"));
      default:
        UNREACHABLE();
    }
  }

  void on_error(uint64 id, Status status) override {
    promise_.set_error(std::move(status));
  }
};

class SearchPublicDialogsQuery : public Td::ResultHandler {
  string query_;

 public:
  void send(const string &query) {
    query_ = query;
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::contacts_search(query, 3 /* ignored server-side */))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::contacts_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto dialogs = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SearchPublicDialogsQuery: " << to_string(dialogs);
    td->contacts_manager_->on_get_users(std::move(dialogs->users_), "SearchPublicDialogsQuery");
    td->contacts_manager_->on_get_chats(std::move(dialogs->chats_), "SearchPublicDialogsQuery");
    td->messages_manager_->on_get_public_dialogs_search_result(query_, std::move(dialogs->my_results_),
                                                               std::move(dialogs->results_));
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for SearchPublicDialogsQuery: " << status;
    }
    td->messages_manager_->on_failed_public_dialogs_search(query_, std::move(status));
  }
};

class GetCommonDialogsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  UserId user_id_;
  int32 offset_chat_id_ = 0;

 public:
  explicit GetCommonDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(UserId user_id, int32 offset_chat_id, int32 limit) {
    user_id_ = user_id;
    offset_chat_id_ = offset_chat_id;
    LOG(INFO) << "Get common dialogs with " << user_id << " from " << offset_chat_id << " with limit " << limit;

    auto input_user = td->contacts_manager_->get_input_user(user_id);
    CHECK(input_user != nullptr);

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_getCommonChats(std::move(input_user), offset_chat_id, limit))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getCommonChats>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto chats_ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetCommonDialogsQuery: " << to_string(chats_ptr);
    switch (chats_ptr->get_id()) {
      case telegram_api::messages_chats::ID: {
        auto chats = move_tl_object_as<telegram_api::messages_chats>(chats_ptr);
        td->messages_manager_->on_get_common_dialogs(user_id_, offset_chat_id_, std::move(chats->chats_),
                                                     narrow_cast<int32>(chats->chats_.size()));
        break;
      }
      case telegram_api::messages_chatsSlice::ID: {
        auto chats = move_tl_object_as<telegram_api::messages_chatsSlice>(chats_ptr);
        td->messages_manager_->on_get_common_dialogs(user_id_, offset_chat_id_, std::move(chats->chats_),
                                                     chats->count_);
        break;
      }
      default:
        UNREACHABLE();
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    promise_.set_error(std::move(status));
  }
};

class CreateChatQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;

 public:
  explicit CreateChatQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<tl_object_ptr<telegram_api::InputUser>> &&input_users, const string &title, int64 random_id) {
    random_id_ = random_id;
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_createChat(std::move(input_users), title))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_createChat>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for createChat " << to_string(ptr);
    td->messages_manager_->on_create_new_dialog_success(random_id_, std::move(ptr), DialogType::Chat,
                                                        std::move(promise_));
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_create_new_dialog_fail(random_id_, std::move(status), std::move(promise_));
  }
};

class CreateChannelQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;

 public:
  explicit CreateChannelQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(const string &title, bool is_megagroup, const string &about, const DialogLocation &location,
            int64 random_id) {
    int32 flags = 0;
    if (is_megagroup) {
      flags |= telegram_api::channels_createChannel::MEGAGROUP_MASK;
    } else {
      flags |= telegram_api::channels_createChannel::BROADCAST_MASK;
    }
    if (!location.empty()) {
      flags |= telegram_api::channels_createChannel::GEO_POINT_MASK;
    }

    random_id_ = random_id;
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::channels_createChannel(flags, false /*ignored*/, false /*ignored*/, title, about,
                                                           location.get_input_geo_point(), location.get_address()))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_createChannel>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for createChannel " << to_string(ptr);
    td->messages_manager_->on_create_new_dialog_success(random_id_, std::move(ptr), DialogType::Channel,
                                                        std::move(promise_));
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_create_new_dialog_fail(random_id_, std::move(status), std::move(promise_));
  }
};

class EditDialogPhotoQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  FileId file_id_;
  bool was_uploaded_ = false;
  string file_reference_;
  DialogId dialog_id_;

 public:
  explicit EditDialogPhotoQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FileId file_id, tl_object_ptr<telegram_api::InputChatPhoto> &&input_chat_photo) {
    CHECK(input_chat_photo != nullptr);
    file_id_ = file_id;
    was_uploaded_ = FileManager::extract_was_uploaded(input_chat_photo);
    file_reference_ = FileManager::extract_file_reference(input_chat_photo);
    dialog_id_ = dialog_id;

    switch (dialog_id.get_type()) {
      case DialogType::Chat:
        send_query(G()->net_query_creator().create(create_storer(
            telegram_api::messages_editChatPhoto(dialog_id.get_chat_id().get(), std::move(input_chat_photo)))));
        break;
      case DialogType::Channel: {
        auto channel_id = dialog_id.get_channel_id();
        auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
        CHECK(input_channel != nullptr);
        send_query(G()->net_query_creator().create(
            create_storer(telegram_api::channels_editPhoto(std::move(input_channel), std::move(input_chat_photo)))));
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    static_assert(std::is_same<telegram_api::messages_editChatPhoto::ReturnType,
                               telegram_api::channels_editPhoto::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_editChatPhoto>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for editDialogPhoto: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    if (file_id_.is_valid() && was_uploaded_) {
      td->file_manager_->delete_partial_remote_location(file_id_);
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (file_id_.is_valid() && was_uploaded_) {
      td->file_manager_->delete_partial_remote_location(file_id_);
    }
    if (FileReferenceManager::is_file_reference_error(status)) {
      if (file_id_.is_valid() && !was_uploaded_) {
        VLOG(file_references) << "Receive " << status << " for " << file_id_;
        td->file_manager_->delete_file_reference(file_id_, file_reference_);
        td->messages_manager_->upload_dialog_photo(dialog_id_, file_id_, std::move(promise_));
        return;
      } else {
        LOG(ERROR) << "Receive file reference error, but file_id = " << file_id_
                   << ", was_uploaded = " << was_uploaded_;
      }
    }

    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditDialogPhotoQuery");
    }
    td->updates_manager_->get_difference("EditDialogPhotoQuery");
    promise_.set_error(std::move(status));
  }
};

class EditDialogTitleQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditDialogTitleQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &title) {
    dialog_id_ = dialog_id;
    switch (dialog_id.get_type()) {
      case DialogType::Chat:
        send_query(G()->net_query_creator().create(
            create_storer(telegram_api::messages_editChatTitle(dialog_id.get_chat_id().get(), title))));
        break;
      case DialogType::Channel: {
        auto channel_id = dialog_id.get_channel_id();
        auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
        CHECK(input_channel != nullptr);
        send_query(G()->net_query_creator().create(
            create_storer(telegram_api::channels_editTitle(std::move(input_channel), title))));
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    static_assert(std::is_same<telegram_api::messages_editChatTitle::ReturnType,
                               telegram_api::channels_editTitle::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_editChatTitle>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for editDialogTitle " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->updates_manager_->get_difference("EditDialogTitleQuery");

    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditDialogTitleQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class EditDialogDefaultBannedRightsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditDialogDefaultBannedRightsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, RestrictedRights permissions) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_editChatDefaultBannedRights(
        std::move(input_peer), permissions.get_chat_banned_rights()))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_editChatDefaultBannedRights>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for editDialogPermissions " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.message() == "CHAT_NOT_MODIFIED") {
      if (!td->auth_manager_->is_bot()) {
        promise_.set_value(Unit());
        return;
      }
    } else {
      td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditDialogDefaultBannedRightsQuery");
    }
    promise_.set_error(std::move(status));
  }
};

class SaveDraftMessageQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SaveDraftMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const unique_ptr<DraftMessage> &draft_message) {
    LOG(INFO) << "Save draft in " << dialog_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      LOG(INFO) << "Can't update draft message because have no write access to " << dialog_id;
      return on_error(0, Status::Error(500, "Can't save draft message"));
    }

    int32 flags = 0;
    ServerMessageId reply_to_message_id;
    if (draft_message != nullptr) {
      if (draft_message->reply_to_message_id.is_valid() && draft_message->reply_to_message_id.is_server()) {
        reply_to_message_id = draft_message->reply_to_message_id.get_server_message_id();
        flags |= MessagesManager::SEND_MESSAGE_FLAG_IS_REPLY;
      }
      if (draft_message->input_message_text.disable_web_page_preview) {
        flags |= MessagesManager::SEND_MESSAGE_FLAG_DISABLE_WEB_PAGE_PREVIEW;
      }
      if (draft_message->input_message_text.text.entities.size()) {
        flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
      }
    }

    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_saveDraft(
        flags, false /*ignored*/, reply_to_message_id.get(), std::move(input_peer),
        draft_message == nullptr ? "" : draft_message->input_message_text.text.text,
        draft_message == nullptr
            ? vector<tl_object_ptr<telegram_api::MessageEntity>>()
            : get_input_message_entities(td->contacts_manager_.get(), draft_message->input_message_text.text.entities,
                                         "SaveDraftMessageQuery")))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_saveDraft>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      on_error(id, Status::Error(400, "Save draft failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SaveDraftMessageQuery")) {
      LOG(ERROR) << "Receive error for SaveDraftMessageQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ClearAllDraftsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ClearAllDraftsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send() {
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_clearAllDrafts())));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_clearAllDrafts>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.move_as_ok();
    if (!result) {
      LOG(INFO) << "Receive false for clearAllDrafts";
    } else {
      LOG(INFO) << "All draft messages has been cleared";
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for ClearAllDraftsQuery: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogPinQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_pinned_;

 public:
  explicit ToggleDialogPinQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_pinned) {
    dialog_id_ = dialog_id;
    is_pinned_ = is_pinned;

    auto input_peer = td->messages_manager_->get_input_dialog_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(500, "Can't update dialog is_pinned"));
    }

    int32 flags = 0;
    if (is_pinned) {
      flags |= telegram_api::messages_toggleDialogPin::PINNED_MASK;
    }
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_toggleDialogPin(flags, false /*ignored*/, std::move(input_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_toggleDialogPin>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      on_error(id, Status::Error(400, "Toggle dialog pin failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogPinQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogPinQuery: " << status;
    }
    td->messages_manager_->set_dialog_is_pinned(dialog_id_, !is_pinned_);
    promise_.set_error(std::move(status));
  }
};

class ReorderPinnedDialogsQuery : public Td::ResultHandler {
  FolderId folder_id_;
  Promise<Unit> promise_;

 public:
  explicit ReorderPinnedDialogsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, const vector<DialogId> &dialog_ids) {
    folder_id_ = folder_id;
    int32 flags = telegram_api::messages_reorderPinnedDialogs::FORCE_MASK;
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_reorderPinnedDialogs(
        flags, true /*ignored*/, folder_id.get(),
        td->messages_manager_->get_input_dialog_peers(dialog_ids, AccessRights::Read)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_reorderPinnedDialogs>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.move_as_ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Result is false"));
    }
    LOG(INFO) << "Pinned chats reordered";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for ReorderPinnedDialogsQuery: " << status;
    }
    td->messages_manager_->on_update_pinned_dialogs(folder_id_);
    promise_.set_error(std::move(status));
  }
};

class ToggleDialogUnreadMarkQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  bool is_marked_as_unread_;

 public:
  explicit ToggleDialogUnreadMarkQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_marked_as_unread) {
    dialog_id_ = dialog_id;
    is_marked_as_unread_ = is_marked_as_unread;

    auto input_peer = td->messages_manager_->get_input_dialog_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(500, "Can't update dialog is_marked_as_unread"));
    }

    int32 flags = 0;
    if (is_marked_as_unread) {
      flags |= telegram_api::messages_markDialogUnread::UNREAD_MASK;
    }
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_markDialogUnread(flags, false /*ignored*/, std::move(input_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_markDialogUnread>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      on_error(id, Status::Error(400, "Toggle dialog mark failed"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ToggleDialogUnreadMarkQuery")) {
      LOG(ERROR) << "Receive error for ToggleDialogUnreadMarkQuery: " << status;
    }
    td->messages_manager_->on_update_dialog_is_marked_as_unread(dialog_id_, !is_marked_as_unread_);
    promise_.set_error(std::move(status));
  }
};

class GetMessagesViewsQuery : public Td::ResultHandler {
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

 public:
  void send(DialogId dialog_id, vector<MessageId> &&message_ids, bool increment_view_counter) {
    dialog_id_ = dialog_id;
    message_ids_ = std::move(message_ids);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't update message views because doesn't have info about the " << dialog_id;
      return on_error(0, Status::Error(500, "Can't update message views"));
    }

    LOG(INFO) << "View " << message_ids_.size() << " messages in " << dialog_id
              << ", increment = " << increment_view_counter;
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getMessagesViews(
        std::move(input_peer), MessagesManager::get_server_message_ids(message_ids_), increment_view_counter))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getMessagesViews>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    vector<int32> views = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for GetMessagesViewsQuery: " << format::as_array(views);
    if (message_ids_.size() != views.size()) {
      return on_error(id, Status::Error(500, "Wrong number of message views returned"));
    }

    for (size_t i = 0; i < message_ids_.size(); i++) {
      td->messages_manager_->on_update_message_views({dialog_id_, message_ids_[i]}, views[i]);
    }
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetMessagesViewsQuery")) {
      LOG(ERROR) << "Receive error for GetMessagesViewsQuery: " << status;
    }
  }
};

class ReadMessagesContentsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ReadMessagesContentsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<MessageId> &&message_ids) {
    LOG(INFO) << "Receive ReadMessagesContentsQuery for messages " << format::as_array(message_ids);

    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::messages_readMessageContents(MessagesManager::get_server_message_ids(message_ids)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_readMessageContents>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      td->messages_manager_->add_pending_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                affected_messages->pts_count_, false, "read messages content query");
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for read message contents: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadChannelMessagesContentsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit ReadChannelMessagesContentsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, vector<MessageId> &&message_ids) {
    channel_id_ = channel_id;

    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    if (input_channel == nullptr) {
      LOG(ERROR) << "Have no input channel for " << channel_id;
      return on_error(0, Status::Error(500, "Can't read channel message contents"));
    }

    LOG(INFO) << "Receive ReadChannelMessagesContentsQuery for messages " << format::as_array(message_ids) << " in "
              << channel_id;

    send_query(G()->net_query_creator().create(create_storer(telegram_api::channels_readMessageContents(
        std::move(input_channel), MessagesManager::get_server_message_ids(message_ids)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_readMessageContents>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(ERROR, !result) << "Read channel messages contents failed";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "ReadChannelMessagesContentsQuery")) {
      LOG(ERROR) << "Receive error for read messages contents in " << channel_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogMessageByDateQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 date_;
  int64 random_id_;

 public:
  explicit GetDialogMessageByDateQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 date, int64 random_id) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(500, "Have no info about the chat"));
    }

    dialog_id_ = dialog_id;
    date_ = date;
    random_id_ = random_id;

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_getHistory(std::move(input_peer), 0, date, -3, 5, 0, 0, 0))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetDialogMessageByDateQuery");
    td->messages_manager_->on_get_dialog_message_by_date_success(dialog_id_, date_, random_id_,
                                                                 std::move(info.messages));
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogMessageByDateQuery")) {
      LOG(ERROR) << "Receive error for GetDialogMessageByDateQuery in " << dialog_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
    td->messages_manager_->on_get_dialog_message_by_date_fail(random_id_);
  }
};

class GetHistoryQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId from_message_id_;
  int32 offset_;
  int32 limit_;
  bool from_the_end_;

 public:
  explicit GetHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't get chat history in " << dialog_id << " because doesn't have info about the chat";
      return promise_.set_error(Status::Error(500, "Have no info about the chat"));
    }
    CHECK(offset < 0);

    dialog_id_ = dialog_id;
    from_message_id_ = from_message_id;
    offset_ = offset;
    limit_ = limit;
    from_the_end_ = false;
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getHistory(
        std::move(input_peer), from_message_id.get_server_message_id().get(), 0, offset, limit, 0, 0, 0))));
  }

  void send_get_from_the_end(DialogId dialog_id, int32 limit) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't get chat history because doesn't have info about the chat";
      return promise_.set_error(Status::Error(500, "Have no info about the chat"));
    }

    dialog_id_ = dialog_id;
    offset_ = 0;
    limit_ = limit;
    from_the_end_ = true;
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_getHistory(std::move(input_peer), 0, 0, 0, limit, 0, 0, 0))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetHistoryQuery");
    // TODO use info.total_count, info.pts
    td->messages_manager_->on_get_history(dialog_id_, from_message_id_, offset_, limit_, from_the_end_,
                                          std::move(info.messages));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetHistoryQuery")) {
      LOG(ERROR) << "Receive error for getHistoryQuery in " << dialog_id_ << ": " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadHistoryQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReadHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId max_message_id) {
    dialog_id_ = dialog_id;
    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::messages_readHistory(td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read),
                                           max_message_id.get_server_message_id().get()))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_readHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);
    LOG(INFO) << "Receive result for readHistory: " << to_string(affected_messages);

    if (affected_messages->pts_count_ > 0) {
      td->messages_manager_->add_pending_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                affected_messages->pts_count_, false, "read history query");
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReadHistoryQuery")) {
      LOG(ERROR) << "Receive error for readHistory: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class ReadChannelHistoryQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;

 public:
  explicit ReadChannelHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId max_message_id) {
    channel_id_ = channel_id;
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);

    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::channels_readHistory(std::move(input_channel), max_message_id.get_server_message_id().get()))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_readHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "ReadChannelHistoryQuery")) {
      LOG(ERROR) << "Receive error for readChannelHistory: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class SearchMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  string query_;
  UserId sender_user_id_;
  MessageId from_message_id_;
  int32 offset_;
  int32 limit_;
  SearchMessagesFilter filter_;
  int64 random_id_;

 public:
  explicit SearchMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &query, UserId sender_user_id,
            telegram_api::object_ptr<telegram_api::InputUser> &&sender_input_user, MessageId from_message_id,
            int32 offset, int32 limit, SearchMessagesFilter filter, int64 random_id) {
    auto input_peer = dialog_id.is_valid() ? td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read)
                                           : make_tl_object<telegram_api::inputPeerEmpty>();
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't search messages because doesn't have info about the chat";
      return promise_.set_error(Status::Error(500, "Have no info about the chat"));
    }

    dialog_id_ = dialog_id;
    query_ = query;
    sender_user_id_ = sender_user_id;
    from_message_id_ = from_message_id;
    offset_ = offset;
    limit_ = limit;
    filter_ = filter;
    random_id_ = random_id;

    if (filter == SearchMessagesFilter::UnreadMention) {
      send_query(G()->net_query_creator().create(create_storer(
          telegram_api::messages_getUnreadMentions(std::move(input_peer), from_message_id.get_server_message_id().get(),
                                                   offset, limit, std::numeric_limits<int32>::max(), 0))));
    } else {
      int32 flags = 0;
      if (sender_input_user != nullptr) {
        flags |= telegram_api::messages_search::FROM_ID_MASK;
      }

      send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_search(
          flags, std::move(input_peer), query, std::move(sender_input_user),
          MessagesManager::get_input_messages_filter(filter), 0, std::numeric_limits<int32>::max(),
          from_message_id.get_server_message_id().get(), offset, limit, std::numeric_limits<int32>::max(), 0, 0))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    static_assert(std::is_same<telegram_api::messages_getUnreadMentions::ReturnType,
                               telegram_api::messages_search::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_search>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "SearchMessagesQuery");
    td->messages_manager_->on_get_dialog_messages_search_result(dialog_id_, query_, sender_user_id_, from_message_id_,
                                                                offset_, limit_, filter_, random_id_, info.total_count,
                                                                std::move(info.messages));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SearchMessagesQuery");
    td->messages_manager_->on_failed_dialog_messages_search(dialog_id_, random_id_);
    promise_.set_error(std::move(status));
  }
};

class SearchMessagesGlobalQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  string query_;
  int32 offset_date_;
  DialogId offset_dialog_id_;
  MessageId offset_message_id_;
  int32 limit_;
  int64 random_id_;

 public:
  explicit SearchMessagesGlobalQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(FolderId folder_id, bool ignore_folder_id, const string &query, int32 offset_date,
            DialogId offset_dialog_id, MessageId offset_message_id, int32 limit, int64 random_id) {
    query_ = query;
    offset_date_ = offset_date;
    offset_dialog_id_ = offset_dialog_id;
    offset_message_id_ = offset_message_id;
    limit_ = limit;
    random_id_ = random_id;

    auto input_peer = td->messages_manager_->get_input_peer(offset_dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      input_peer = make_tl_object<telegram_api::inputPeerEmpty>();
    }

    int32 flags = 0;
    if (!ignore_folder_id) {
      flags |= telegram_api::messages_searchGlobal::FOLDER_ID_MASK;
    }
    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::messages_searchGlobal(flags, folder_id.get(), query, offset_date_, std::move(input_peer),
                                            offset_message_id.get_server_message_id().get(), limit))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_searchGlobal>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "SearchMessagesGlobalQuery");
    td->messages_manager_->on_get_messages_search_result(query_, offset_date_, offset_dialog_id_, offset_message_id_,
                                                         limit_, random_id_, info.total_count,
                                                         std::move(info.messages));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_failed_messages_search(random_id_);
    promise_.set_error(std::move(status));
  }
};

class GetAllScheduledMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  uint32 generation_;

 public:
  explicit GetAllScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 hash, uint32 generation) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    dialog_id_ = dialog_id;
    generation_ = generation;

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_getScheduledHistory(std::move(input_peer), hash))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getScheduledHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (result_ptr.ok()->get_id() == telegram_api::messages_messagesNotModified::ID) {
      td->messages_manager_->on_get_scheduled_server_messages(dialog_id_, generation_, Auto(), true);
    } else {
      auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetAllScheduledMessagesQuery");
      td->messages_manager_->on_get_scheduled_server_messages(dialog_id_, generation_, std::move(info.messages), false);
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetAllScheduledMessagesQuery");
    promise_.set_error(std::move(status));
  }
};

class GetRecentLocationsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int32 limit_;
  int64 random_id_;

 public:
  explicit GetRecentLocationsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int32 limit, int64 random_id) {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't get recent locations because doesn't have info about the chat";
      return promise_.set_error(Status::Error(500, "Have no info about the chat"));
    }

    dialog_id_ = dialog_id;
    limit_ = limit;
    random_id_ = random_id;

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_getRecentLocations(std::move(input_peer), limit, 0))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getRecentLocations>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto info = td->messages_manager_->on_get_messages(result_ptr.move_as_ok(), "GetRecentLocationsQuery");
    td->messages_manager_->on_get_recent_locations(dialog_id_, limit_, random_id_, info.total_count,
                                                   std::move(info.messages));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetRecentLocationsQuery");
    td->messages_manager_->on_get_recent_locations_failed(random_id_);
    promise_.set_error(std::move(status));
  }
};

class DeleteHistoryQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  MessageId max_message_id_;
  bool remove_from_dialog_list_;
  bool revoke_;

  void send_request() {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(3, "Chat is not accessible"));
    }

    int32 flags = 0;
    if (!remove_from_dialog_list_) {
      flags |= telegram_api::messages_deleteHistory::JUST_CLEAR_MASK;
    }
    if (revoke_) {
      flags |= telegram_api::messages_deleteHistory::REVOKE_MASK;
    }
    LOG(INFO) << "Delete " << dialog_id_ << " history up to " << max_message_id_ << " with flags " << flags;

    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::messages_deleteHistory(flags, false /*ignored*/, false /*ignored*/, std::move(input_peer),
                                             max_message_id_.get_server_message_id().get()))));
  }

 public:
  explicit DeleteHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId max_message_id, bool remove_from_dialog_list, bool revoke) {
    dialog_id_ = dialog_id;
    max_message_id_ = max_message_id;
    remove_from_dialog_list_ = remove_from_dialog_list;
    revoke_ = revoke;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      td->messages_manager_->add_pending_update(make_tl_object<dummyUpdate>(), affected_history->pts_,
                                                affected_history->pts_count_, false, "delete history query");
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "DeleteHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class DeleteChannelHistoryQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  MessageId max_message_id_;
  bool allow_error_;

 public:
  explicit DeleteChannelHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, MessageId max_message_id, bool allow_error) {
    channel_id_ = channel_id;
    max_message_id_ = max_message_id;
    allow_error_ = allow_error;
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);

    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::channels_deleteHistory(std::move(input_channel), max_message_id.get_server_message_id().get()))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_deleteHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    LOG_IF(ERROR, !allow_error_ && !result)
        << "Delete history in " << channel_id_ << " up to " << max_message_id_ << " failed";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "DeleteChannelHistoryQuery")) {
      LOG(ERROR) << "Receive error for deleteChannelHistory: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteUserHistoryQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  UserId user_id_;

  void send_request() {
    auto input_channel = td->contacts_manager_->get_input_channel(channel_id_);
    if (input_channel == nullptr) {
      return promise_.set_error(Status::Error(3, "Chat is not accessible"));
    }
    auto input_user = td->contacts_manager_->get_input_user(user_id_);
    if (input_user == nullptr) {
      return promise_.set_error(Status::Error(3, "Usat is not accessible"));
    }

    LOG(INFO) << "Delete all messages from " << user_id_ << " in " << channel_id_;

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::channels_deleteUserHistory(std::move(input_channel), std::move(input_user)))));
  }

 public:
  explicit DeleteUserHistoryQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, UserId user_id) {
    channel_id_ = channel_id;
    user_id_ = user_id;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_deleteUserHistory>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      td->messages_manager_->add_pending_channel_update(DialogId(channel_id_), make_tl_object<dummyUpdate>(),
                                                        affected_history->pts_, affected_history->pts_count_,
                                                        "delete user history query");
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->contacts_manager_->on_get_channel_error(channel_id_, status, "DeleteUserHistoryQuery");
    promise_.set_error(std::move(status));
  }
};

class ReadAllMentionsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

  void send_request() {
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id_, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_error(Status::Error(3, "Chat is not accessible"));
    }

    LOG(INFO) << "Read all mentions in " << dialog_id_;

    send_query(
        G()->net_query_creator().create(create_storer(telegram_api::messages_readMentions(std::move(input_peer)))));
  }

 public:
  explicit ReadAllMentionsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    send_request();
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_readMentions>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_history = result_ptr.move_as_ok();
    CHECK(affected_history->get_id() == telegram_api::messages_affectedHistory::ID);

    if (affected_history->pts_count_ > 0) {
      if (dialog_id_.get_type() == DialogType::Channel) {
        LOG(ERROR) << "Receive pts_count " << affected_history->pts_count_ << " in result of ReadAllMentionsQuery in "
                   << dialog_id_;
        td->updates_manager_->get_difference("Wrong messages_readMentions result");
      } else {
        td->messages_manager_->add_pending_update(make_tl_object<dummyUpdate>(), affected_history->pts_,
                                                  affected_history->pts_count_, false, "read all mentions query");
      }
    }

    if (affected_history->offset_ > 0) {
      send_request();
      return;
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReadAllMentionsQuery");
    promise_.set_error(std::move(status));
  }
};

class SendSecretMessageActor : public NetActor {
  int64 random_id_;

 public:
  void send(DialogId dialog_id, int64 reply_to_random_id, int32 ttl, const string &text, SecretInputMedia media,
            vector<tl_object_ptr<secret_api::MessageEntity>> &&entities, UserId via_bot_user_id, int64 media_album_id,
            int64 random_id) {
    if (false && !media.empty()) {
      td->messages_manager_->on_send_secret_message_error(random_id, Status::Error(400, "FILE_PART_1_MISSING"), Auto());
      stop();
      return;
    }

    CHECK(dialog_id.get_type() == DialogType::SecretChat);
    random_id_ = random_id;

    int32 flags = 0;
    if (reply_to_random_id != 0) {
      flags |= secret_api::decryptedMessage::REPLY_TO_RANDOM_ID_MASK;
    }
    if (via_bot_user_id.is_valid()) {
      flags |= secret_api::decryptedMessage::VIA_BOT_NAME_MASK;
    }
    if (!media.empty()) {
      flags |= secret_api::decryptedMessage::MEDIA_MASK;
    }
    if (!entities.empty()) {
      flags |= secret_api::decryptedMessage::ENTITIES_MASK;
    }
    if (media_album_id != 0) {
      CHECK(media_album_id < 0);
      flags |= secret_api::decryptedMessage::GROUPED_ID_MASK;
    }

    send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_message, dialog_id.get_secret_chat_id(),
                 make_tl_object<secret_api::decryptedMessage>(
                     flags, random_id, ttl, text, std::move(media.decrypted_media_), std::move(entities),
                     td->contacts_manager_->get_user_username(via_bot_user_id), reply_to_random_id, -media_album_id),
                 std::move(media.input_file_),
                 PromiseCreator::event(self_closure(this, &SendSecretMessageActor::done)));
  }

  void done() {
    stop();
  }
};

class SendMessageActor : public NetActorOnce {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  void send(int32 flags, DialogId dialog_id, MessageId reply_to_message_id, int32 schedule_date,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities, const string &text, int64 random_id,
            NetQueryRef *send_query_ref, uint64 sequence_dispatcher_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }

    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }

    auto query = G()->net_query_creator().create(create_storer(telegram_api::messages_sendMessage(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), text, random_id, std::move(reply_markup),
        std::move(entities), schedule_date)));
    if (G()->shared_config().get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_id](Unit) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          },
          PromiseCreator::Ignore());
    }
    *send_query_ref = query.get_weak();
    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_sendMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for sendMessage for " << random_id_ << ": " << to_string(ptr);

    auto constructor_id = ptr->get_id();
    if (constructor_id != telegram_api::updateShortSentMessage::ID) {
      td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendMessage");
      return td->updates_manager_->on_get_updates(std::move(ptr));
    }
    auto sent_message = move_tl_object_as<telegram_api::updateShortSentMessage>(ptr);
    td->messages_manager_->on_update_sent_text_message(random_id_, std::move(sent_message->media_),
                                                       std::move(sent_message->entities_));

    auto message_id = MessageId(ServerMessageId(sent_message->id_));
    if (dialog_id_.get_type() == DialogType::Channel) {
      td->messages_manager_->add_pending_channel_update(
          dialog_id_, make_tl_object<updateSentMessage>(random_id_, message_id, sent_message->date_),
          sent_message->pts_, sent_message->pts_count_, "send message actor");
      return;
    }

    td->messages_manager_->add_pending_update(
        make_tl_object<updateSentMessage>(random_id_, message_id, sent_message->date_), sent_message->pts_,
        sent_message->pts_count_, false, "send message actor");
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for sendMessage: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendMessageActor");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class StartBotQuery : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  NetQueryRef send(tl_object_ptr<telegram_api::InputUser> bot_input_user, DialogId dialog_id,
                   tl_object_ptr<telegram_api::InputPeer> input_peer, const string &parameter, int64 random_id) {
    CHECK(bot_input_user != nullptr);
    CHECK(input_peer != nullptr);
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto query = G()->net_query_creator().create(create_storer(
        telegram_api::messages_startBot(std::move(bot_input_user), std::move(input_peer), random_id, parameter)));
    if (G()->shared_config().get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_id](Unit) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          },
          PromiseCreator::Ignore());
    }
    auto send_query_ref = query.get_weak();
    send_query(std::move(query));
    return send_query_ref;
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_startBot>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for startBot for " << random_id_ << ": " << to_string(ptr);
    // Result may contain messageActionChatAddUser
    // td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "StartBot");
    td->updates_manager_->on_get_updates(std::move(ptr));
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for startBot: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message should be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "StartBotQuery");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class SendInlineBotResultQuery : public Td::ResultHandler {
  int64 random_id_;
  DialogId dialog_id_;

 public:
  NetQueryRef send(int32 flags, DialogId dialog_id, MessageId reply_to_message_id, int32 schedule_date, int64 random_id,
                   int64 query_id, const string &result_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    auto query = G()->net_query_creator().create(create_storer(telegram_api::messages_sendInlineBotResult(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), random_id, query_id, result_id, schedule_date)));
    auto send_query_ref = query.get_weak();
    send_query(std::move(query));
    return send_query_ref;
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_sendInlineBotResult>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for sendInlineBotResult for " << random_id_ << ": " << to_string(ptr);
    td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendInlineBotResult");
    td->updates_manager_->on_get_updates(std::move(ptr));
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for sendInlineBotResult: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendInlineBotResultQuery");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class SendMultiMediaActor : public NetActorOnce {
  vector<FileId> file_ids_;
  vector<string> file_references_;
  vector<int64> random_ids_;
  DialogId dialog_id_;

 public:
  void send(int32 flags, DialogId dialog_id, MessageId reply_to_message_id, int32 schedule_date,
            vector<FileId> &&file_ids, vector<tl_object_ptr<telegram_api::inputSingleMedia>> &&input_single_media,
            uint64 sequence_dispatcher_id) {
    for (auto &single_media : input_single_media) {
      random_ids_.push_back(single_media->random_id_);
      CHECK(FileManager::extract_was_uploaded(single_media->media_) == false);
      file_references_.push_back(FileManager::extract_file_reference(single_media->media_));
    }
    dialog_id_ = dialog_id;
    file_ids_ = std::move(file_ids);
    CHECK(file_ids_.size() == random_ids_.size());

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }

    auto query = G()->net_query_creator().create(create_storer(telegram_api::messages_sendMultiMedia(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(input_peer),
        reply_to_message_id.get_server_message_id().get(), std::move(input_single_media), schedule_date)));
    // no quick ack, because file reference errors are very likely to happen
    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_sendMultiMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for sendMultiMedia for " << format::as_array(random_ids_) << ": " << to_string(ptr);

    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    auto sent_random_ids_size = sent_random_ids.size();
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        if (random_ids_.size() == 1) {
          is_result_wrong = true;
        }
        td->messages_manager_->on_send_message_fail(random_id, Status::Error(400, "Message was not sent"));
      } else {
        sent_random_ids.erase(it);
      }
    }
    if (!sent_random_ids.empty()) {
      is_result_wrong = true;
    }
    if (!is_result_wrong) {
      auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
      if (sent_random_ids_size != sent_messages.size()) {
        is_result_wrong = true;
      }
      for (auto &sent_message : sent_messages) {
        if (td->messages_manager_->get_message_dialog_id(*sent_message) != dialog_id_) {
          is_result_wrong = true;
        }
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for sendMultiMedia with random_ids " << format::as_array(random_ids_)
                 << " to " << dialog_id_ << ": " << oneline(to_string(ptr));
      td->updates_manager_->schedule_get_difference("Wrong sendMultiMedia result");
    }

    td->updates_manager_->on_get_updates(std::move(ptr));
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for sendMultiMedia: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    if (FileReferenceManager::is_file_reference_error(status)) {
      auto pos = FileReferenceManager::get_file_reference_error_pos(status);
      if (1 <= pos && pos <= file_ids_.size() && file_ids_[pos - 1].is_valid()) {
        VLOG(file_references) << "Receive " << status << " for " << file_ids_[pos - 1];
        td->file_manager_->delete_file_reference(file_ids_[pos - 1], file_references_[pos - 1]);
        td->messages_manager_->on_send_media_group_file_reference_error(dialog_id_, std::move(random_ids_));
        return;
      } else {
        LOG(ERROR) << "Receive file reference error " << status << ", but file_ids = " << file_ids_
                   << ", message_count = " << file_ids_.size();
      }
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendMultiMediaActor");
    for (auto &random_id : random_ids_) {
      td->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
  }
};

class SendMediaActor : public NetActorOnce {
  int64 random_id_ = 0;
  FileId file_id_;
  FileId thumbnail_file_id_;
  DialogId dialog_id_;
  string file_reference_;
  bool was_uploaded_ = false;
  bool was_thumbnail_uploaded_ = false;

 public:
  void send(FileId file_id, FileId thumbnail_file_id, int32 flags, DialogId dialog_id, MessageId reply_to_message_id,
            int32 schedule_date, tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities, const string &text,
            tl_object_ptr<telegram_api::InputMedia> &&input_media, int64 random_id, NetQueryRef *send_query_ref,
            uint64 sequence_dispatcher_id) {
    random_id_ = random_id;
    file_id_ = file_id;
    thumbnail_file_id_ = thumbnail_file_id;
    dialog_id_ = dialog_id;
    file_reference_ = FileManager::extract_file_reference(input_media);
    was_uploaded_ = FileManager::extract_was_uploaded(input_media);
    was_thumbnail_uploaded_ = FileManager::extract_was_thumbnail_uploaded(input_media);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }
    if (!entities.empty()) {
      flags |= telegram_api::messages_sendMedia::ENTITIES_MASK;
    }

    telegram_api::messages_sendMedia request(flags, false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                             std::move(input_peer), reply_to_message_id.get_server_message_id().get(),
                                             std::move(input_media), text, random_id, std::move(reply_markup),
                                             std::move(entities), schedule_date);
    LOG(INFO) << "Send media: " << to_string(request);
    auto query = G()->net_query_creator().create(create_storer(request));
    if (G()->shared_config().get_option_boolean("use_quick_ack") && was_uploaded_) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_id](Unit) {
            send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
          },
          PromiseCreator::Ignore());
    }
    *send_query_ref = query.get_weak();
    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_sendMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (was_thumbnail_uploaded_) {
      CHECK(thumbnail_file_id_.is_valid());
      // always delete partial remote location for the thumbnail, because it can't be reused anyway
      // TODO delete it only in the case it can't be merged with file thumbnail
      td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for sendMedia for " << random_id_ << ": " << to_string(ptr);
    td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(), "SendMedia");
    td->updates_manager_->on_get_updates(std::move(ptr));
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for sendMedia: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    if (was_uploaded_) {
      if (was_thumbnail_uploaded_) {
        CHECK(thumbnail_file_id_.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
      }

      CHECK(file_id_.is_valid());
      if (begins_with(status.message(), "FILE_PART_") && ends_with(status.message(), "_MISSING")) {
        td->messages_manager_->on_send_message_file_part_missing(random_id_,
                                                                 to_integer<int32>(status.message().substr(10)));
        return;
      } else {
        if (status.code() != 429 && status.code() < 500 && !G()->close_flag()) {
          td->file_manager_->delete_partial_remote_location(file_id_);
        }
      }
    } else if (FileReferenceManager::is_file_reference_error(status)) {
      if (file_id_.is_valid() && !was_uploaded_) {
        VLOG(file_references) << "Receive " << status << " for " << file_id_;
        td->file_manager_->delete_file_reference(file_id_, file_reference_);
        td->messages_manager_->on_send_message_file_reference_error(random_id_);
        return;
      } else {
        LOG(ERROR) << "Receive file reference error, but file_id = " << file_id_
                   << ", was_uploaded = " << was_uploaded_;
      }
    }

    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendMediaActor");
    td->messages_manager_->on_send_message_fail(random_id_, std::move(status));
  }
};

class UploadMediaQuery : public Td::ResultHandler {
  DialogId dialog_id_;
  MessageId message_id_;
  FileId file_id_;
  FileId thumbnail_file_id_;
  string file_reference_;
  bool was_uploaded_ = false;
  bool was_thumbnail_uploaded_ = false;

 public:
  void send(DialogId dialog_id, MessageId message_id, FileId file_id, FileId thumbnail_file_id,
            tl_object_ptr<telegram_api::InputMedia> &&input_media) {
    CHECK(input_media != nullptr);
    dialog_id_ = dialog_id;
    message_id_ = message_id;
    file_id_ = file_id;
    thumbnail_file_id_ = thumbnail_file_id;
    file_reference_ = FileManager::extract_file_reference(input_media);
    was_uploaded_ = FileManager::extract_was_uploaded(input_media);
    was_thumbnail_uploaded_ = FileManager::extract_was_thumbnail_uploaded(input_media);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Have no write access to the chat"));
    }

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_uploadMedia(std::move(input_peer), std::move(input_media)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_uploadMedia>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    if (was_thumbnail_uploaded_) {
      CHECK(thumbnail_file_id_.is_valid());
      // always delete partial remote location for the thumbnail, because it can't be reused anyway
      td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for uploadMedia for " << message_id_ << " in " << dialog_id_ << ": " << to_string(ptr);
    td->messages_manager_->on_upload_message_media_success(dialog_id_, message_id_, std::move(ptr));
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for uploadMedia for " << message_id_ << " in " << dialog_id_ << ": " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, message will be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UploadMediaQuery");
    if (was_uploaded_) {
      if (was_thumbnail_uploaded_) {
        CHECK(thumbnail_file_id_.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td->file_manager_->delete_partial_remote_location(thumbnail_file_id_);
      }

      CHECK(file_id_.is_valid());
      if (begins_with(status.message(), "FILE_PART_") && ends_with(status.message(), "_MISSING")) {
        td->messages_manager_->on_upload_message_media_file_part_missing(
            dialog_id_, message_id_, to_integer<int32>(status.message().substr(10)));
        return;
      } else {
        if (status.code() != 429 && status.code() < 500 && !G()->close_flag()) {
          td->file_manager_->delete_partial_remote_location(file_id_);
        }
      }
    } else if (FileReferenceManager::is_file_reference_error(status)) {
      LOG(ERROR) << "Receive file reference error for UploadMediaQuery";
    }
    td->messages_manager_->on_upload_message_media_fail(dialog_id_, message_id_, std::move(status));
  }
};

class SendScheduledMessageActor : public NetActorOnce {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SendScheduledMessageActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, uint64 sequence_dispatcher_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat"));
      stop();
      return;
    }

    LOG(DEBUG) << "Send " << FullMessageId{dialog_id, message_id};

    int32 server_message_id = message_id.get_scheduled_server_message_id().get();
    auto query = G()->net_query_creator().create(
        create_storer(telegram_api::messages_sendScheduledMessages(std::move(input_peer), {server_message_id})));

    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_sendScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendScheduledMessage: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for SendScheduledMessage: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendScheduledMessageActor");
    promise_.set_error(std::move(status));
  }
};

class EditMessageActor : public NetActorOnce {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditMessageActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, DialogId dialog_id, MessageId message_id, const string &text,
            vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities,
            tl_object_ptr<telegram_api::InputMedia> &&input_media,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup, int32 schedule_date,
            uint64 sequence_dispatcher_id) {
    dialog_id_ = dialog_id;

    if (false && input_media != nullptr) {
      on_error(0, Status::Error(400, "FILE_PART_1_MISSING"));
      stop();
      return;
    }

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat"));
      stop();
      return;
    }

    if (reply_markup != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (!text.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_MESSAGE;
    }
    if (input_media != nullptr) {
      flags |= telegram_api::messages_editMessage::MEDIA_MASK;
    }
    if (schedule_date != 0) {
      flags |= telegram_api::messages_editMessage::SCHEDULE_DATE_MASK;
    }
    LOG(DEBUG) << "Edit message with flags " << flags;

    int32 server_message_id = schedule_date != 0 ? message_id.get_scheduled_server_message_id().get()
                                                 : message_id.get_server_message_id().get();
    auto query = G()->net_query_creator().create(create_storer(telegram_api::messages_editMessage(
        flags, false /*ignored*/, std::move(input_peer), server_message_id, text, std::move(input_media),
        std::move(reply_markup), std::move(entities), schedule_date)));

    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_editMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for editMessage: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for editMessage: " << status;
    if (!td->auth_manager_->is_bot() && status.message() == "MESSAGE_NOT_MODIFIED") {
      return promise_.set_value(Unit());
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditMessageActor");
    promise_.set_error(std::move(status));
  }
};

class EditInlineMessageQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit EditInlineMessageQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, tl_object_ptr<telegram_api::inputBotInlineMessageID> input_bot_inline_message_id,
            const string &text, vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities,
            tl_object_ptr<telegram_api::InputMedia> &&input_media,
            tl_object_ptr<telegram_api::ReplyMarkup> &&reply_markup) {
    CHECK(input_bot_inline_message_id != nullptr);

    // file in an inline message can't be uploaded to another datacenter,
    // so only previously uploaded files or URLs can be used in the InputMedia
    CHECK(!FileManager::extract_was_uploaded(input_media));

    if (reply_markup != nullptr) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
    }
    if (!entities.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_ENTITIES;
    }
    if (!text.empty()) {
      flags |= MessagesManager::SEND_MESSAGE_FLAG_HAS_MESSAGE;
    }
    if (input_media != nullptr) {
      flags |= telegram_api::messages_editInlineBotMessage::MEDIA_MASK;
    }
    LOG(DEBUG) << "Edit inline message with flags " << flags;

    auto dc_id = DcId::internal(input_bot_inline_message_id->dc_id_);
    send_query(
        G()->net_query_creator().create(create_storer(telegram_api::messages_editInlineBotMessage(
                                            flags, false /*ignored*/, std::move(input_bot_inline_message_id), text,
                                            std::move(input_media), std::move(reply_markup), std::move(entities))),
                                        dc_id));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_editInlineBotMessage>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    LOG_IF(ERROR, !result_ptr.ok()) << "Receive false in result of editInlineMessage";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for editInlineMessage: " << status;
    promise_.set_error(std::move(status));
  }
};

class SetGameScoreActor : public NetActorOnce {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetGameScoreActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, bool edit_message,
            tl_object_ptr<telegram_api::InputUser> input_user, int32 score, bool force, uint64 sequence_dispatcher_id) {
    int32 flags = 0;
    if (edit_message) {
      flags |= telegram_api::messages_setGameScore::EDIT_MESSAGE_MASK;
    }
    if (force) {
      flags |= telegram_api::messages_setGameScore::FORCE_MASK;
    }

    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Edit);
    if (input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat"));
      stop();
      return;
    }

    CHECK(input_user != nullptr);
    auto query = G()->net_query_creator().create(create_storer(
        telegram_api::messages_setGameScore(flags, false /*ignored*/, false /*ignored*/, std::move(input_peer),
                                            message_id.get_server_message_id().get(), std::move(input_user), score)));

    LOG(INFO) << "Set game score to " << score;

    query->debug("send to MessagesManager::MultiSequenceDispatcher");
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_setGameScore>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for setGameScore: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for setGameScore: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SetGameScoreActor");
    promise_.set_error(std::move(status));
  }
};

class SetInlineGameScoreQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit SetInlineGameScoreQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(tl_object_ptr<telegram_api::inputBotInlineMessageID> input_bot_inline_message_id, bool edit_message,
            tl_object_ptr<telegram_api::InputUser> input_user, int32 score, bool force) {
    CHECK(input_bot_inline_message_id != nullptr);
    CHECK(input_user != nullptr);

    int32 flags = 0;
    if (edit_message) {
      flags |= telegram_api::messages_setInlineGameScore::EDIT_MESSAGE_MASK;
    }
    if (force) {
      flags |= telegram_api::messages_setInlineGameScore::FORCE_MASK;
    }

    LOG(INFO) << "Set inline game score to " << score;
    auto dc_id = DcId::internal(input_bot_inline_message_id->dc_id_);
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_setInlineGameScore(flags, false /*ignored*/, false /*ignored*/,
                                                                std::move(input_bot_inline_message_id),
                                                                std::move(input_user), score)),
        dc_id));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_setInlineGameScore>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    LOG_IF(ERROR, !result_ptr.ok()) << "Receive false in result of setInlineGameScore";

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for setInlineGameScore: " << status;
    promise_.set_error(std::move(status));
  }
};

class GetGameHighScoresQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;
  int64 random_id_;

 public:
  explicit GetGameHighScoresQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, MessageId message_id, tl_object_ptr<telegram_api::InputUser> input_user,
            int64 random_id) {
    dialog_id_ = dialog_id;
    random_id_ = random_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    CHECK(input_user != nullptr);
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getGameHighScores(
        std::move(input_peer), message_id.get_server_message_id().get(), std::move(input_user)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getGameHighScores>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_game_high_scores(random_id_, result_ptr.move_as_ok());
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for getGameHighScores: " << status;
    td->messages_manager_->on_get_game_high_scores(random_id_, nullptr);
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetGameHighScoresQuery");
    promise_.set_error(std::move(status));
  }
};

class GetInlineGameHighScoresQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;

 public:
  explicit GetInlineGameHighScoresQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(tl_object_ptr<telegram_api::inputBotInlineMessageID> input_bot_inline_message_id,
            tl_object_ptr<telegram_api::InputUser> input_user, int64 random_id) {
    CHECK(input_bot_inline_message_id != nullptr);
    CHECK(input_user != nullptr);

    random_id_ = random_id;

    auto dc_id = DcId::internal(input_bot_inline_message_id->dc_id_);
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_getInlineGameHighScores(
                                                   std::move(input_bot_inline_message_id), std::move(input_user))),
                                               dc_id));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getInlineGameHighScores>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_game_high_scores(random_id_, result_ptr.move_as_ok());
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for getInlineGameHighScores: " << status;
    td->messages_manager_->on_get_game_high_scores(random_id_, nullptr);
    promise_.set_error(std::move(status));
  }
};

class ForwardMessagesActor : public NetActorOnce {
  Promise<Unit> promise_;
  vector<int64> random_ids_;
  DialogId to_dialog_id_;

 public:
  explicit ForwardMessagesActor(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(int32 flags, DialogId to_dialog_id, DialogId from_dialog_id, const vector<MessageId> &message_ids,
            vector<int64> &&random_ids, int32 schedule_date, uint64 sequence_dispatcher_id) {
    LOG(INFO) << "Forward " << format::as_array(message_ids) << " from " << from_dialog_id << " to " << to_dialog_id;

    random_ids_ = random_ids;
    to_dialog_id_ = to_dialog_id;

    auto to_input_peer = td->messages_manager_->get_input_peer(to_dialog_id, AccessRights::Write);
    if (to_input_peer == nullptr) {
      on_error(0, Status::Error(400, "Have no write access to the chat"));
      stop();
      return;
    }

    auto from_input_peer = td->messages_manager_->get_input_peer(from_dialog_id, AccessRights::Read);
    if (from_input_peer == nullptr) {
      on_error(0, Status::Error(400, "Can't access the chat to forward messages from"));
      stop();
      return;
    }

    auto query = G()->net_query_creator().create(create_storer(telegram_api::messages_forwardMessages(
        flags, false /*ignored*/, false /*ignored*/, false /*ignored*/, false /*ignored*/, std::move(from_input_peer),
        MessagesManager::get_server_message_ids(message_ids), std::move(random_ids), std::move(to_input_peer),
        schedule_date)));
    if (G()->shared_config().get_option_boolean("use_quick_ack")) {
      query->quick_ack_promise_ = PromiseCreator::lambda(
          [random_ids = random_ids_](Unit) {
            for (auto random_id : random_ids) {
              send_closure(G()->messages_manager(), &MessagesManager::on_send_message_get_quick_ack, random_id);
            }
          },
          PromiseCreator::Ignore());
    }
    send_closure(td->messages_manager_->sequence_dispatcher_, &MultiSequenceDispatcher::send_with_callback,
                 std::move(query), actor_shared(this), sequence_dispatcher_id);
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_forwardMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for forwardMessages for " << format::as_array(random_ids_) << ": " << to_string(ptr);
    auto sent_random_ids = UpdatesManager::get_sent_messages_random_ids(ptr.get());
    bool is_result_wrong = false;
    auto sent_random_ids_size = sent_random_ids.size();
    for (auto &random_id : random_ids_) {
      auto it = sent_random_ids.find(random_id);
      if (it == sent_random_ids.end()) {
        if (random_ids_.size() == 1) {
          is_result_wrong = true;
        }
        td->messages_manager_->on_send_message_fail(random_id, Status::Error(400, "Message was not forwarded"));
      } else {
        sent_random_ids.erase(it);
      }
    }
    if (!sent_random_ids.empty()) {
      is_result_wrong = true;
    }
    if (!is_result_wrong) {
      auto sent_messages = UpdatesManager::get_new_messages(ptr.get());
      if (sent_random_ids_size != sent_messages.size()) {
        is_result_wrong = true;
      }
      for (auto &sent_message : sent_messages) {
        if (td->messages_manager_->get_message_dialog_id(*sent_message) != to_dialog_id_) {
          is_result_wrong = true;
        }
      }
    }
    if (is_result_wrong) {
      LOG(ERROR) << "Receive wrong result for forwarding messages with random_ids " << format::as_array(random_ids_)
                 << " to " << to_dialog_id_ << ": " << oneline(to_string(ptr));
      td->updates_manager_->schedule_get_difference("Wrong forwardMessages result");
    }

    td->updates_manager_->on_get_updates(std::move(ptr));
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for forward messages: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, messages should be re-sent
      return;
    }
    // no on_get_dialog_error call, because two dialogs are involved
    for (auto &random_id : random_ids_) {
      td->messages_manager_->on_send_message_fail(random_id, status.clone());
    }
    promise_.set_error(std::move(status));
  }
};

class SendScreenshotNotificationQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  int64 random_id_;
  DialogId dialog_id_;

 public:
  explicit SendScreenshotNotificationQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, int64 random_id) {
    random_id_ = random_id;
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    auto query = G()->net_query_creator().create(
        create_storer(telegram_api::messages_sendScreenshotNotification(std::move(input_peer), 0, random_id)));
    send_query(std::move(query));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_sendScreenshotNotification>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for SendScreenshotNotificationQuery for " << random_id_ << ": " << to_string(ptr);
    td->messages_manager_->check_send_message_result(random_id_, dialog_id_, ptr.get(),
                                                     "SendScreenshotNotificationQuery");
    td->updates_manager_->on_get_updates(std::move(ptr));
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for SendScreenshotNotificationQuery: " << status;
    if (G()->close_flag() && G()->parameters().use_message_db) {
      // do not send error, messages should be re-sent
      return;
    }
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SendScreenshotNotificationQuery");
    td->messages_manager_->on_send_message_fail(random_id_, status.clone());
    promise_.set_error(std::move(status));
  }
};

class SetTypingQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit SetTypingQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  NetQueryRef send(DialogId dialog_id, tl_object_ptr<telegram_api::SendMessageAction> &&action) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Write);
    CHECK(input_peer != nullptr);

    auto net_query = G()->net_query_creator().create(
        create_storer(telegram_api::messages_setTyping(std::move(input_peer), std::move(action))));
    auto result = net_query.get_weak();
    send_query(std::move(net_query));
    return result;
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_setTyping>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    // ignore result

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.code() == NetQuery::Cancelled) {
      return promise_.set_value(Unit());
    }

    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "SetTypingQuery")) {
      LOG(INFO) << "Receive error for set typing: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  int32 query_count_;

 public:
  explicit DeleteMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(vector<MessageId> &&message_ids, bool revoke) {
    LOG(INFO) << "Send deleteMessagesQuery to delete " << format::as_array(message_ids);
    int32 flags = 0;
    if (revoke) {
      flags |= telegram_api::messages_deleteMessages::REVOKE_MASK;
    }

    query_count_ = 0;
    auto server_message_ids = MessagesManager::get_server_message_ids(message_ids);
    const size_t MAX_SLICE_SIZE = 100;
    for (size_t i = 0; i < server_message_ids.size(); i += MAX_SLICE_SIZE) {
      auto end_i = i + MAX_SLICE_SIZE;
      auto end = end_i < server_message_ids.size() ? server_message_ids.begin() + end_i : server_message_ids.end();
      vector<int32> slice(server_message_ids.begin() + i, end);

      query_count_++;
      send_query(G()->net_query_creator().create(
          create_storer(telegram_api::messages_deleteMessages(flags, false /*ignored*/, std::move(slice)))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_deleteMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      td->messages_manager_->add_pending_update(make_tl_object<dummyUpdate>(), affected_messages->pts_,
                                                affected_messages->pts_count_, false, "delete messages query");
    }
    if (--query_count_ == 0) {
      promise_.set_value(Unit());
    }
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for delete messages: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteChannelMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  int32 query_count_;
  ChannelId channel_id_;

 public:
  explicit DeleteChannelMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, vector<MessageId> &&message_ids) {
    channel_id_ = channel_id;
    LOG(INFO) << "Send deleteChannelMessagesQuery to delete " << format::as_array(message_ids) << " in the "
              << channel_id;

    query_count_ = 0;
    auto server_message_ids = MessagesManager::get_server_message_ids(message_ids);
    const size_t MAX_SLICE_SIZE = 100;
    for (size_t i = 0; i < server_message_ids.size(); i += MAX_SLICE_SIZE) {
      auto end_i = i + MAX_SLICE_SIZE;
      auto end = end_i < server_message_ids.size() ? server_message_ids.begin() + end_i : server_message_ids.end();
      vector<int32> slice(server_message_ids.begin() + i, end);

      query_count_++;
      auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
      CHECK(input_channel != nullptr);
      send_query(G()->net_query_creator().create(
          create_storer(telegram_api::channels_deleteMessages(std::move(input_channel), std::move(slice)))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_deleteMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto affected_messages = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for deleteChannelMessages: " << to_string(affected_messages);
    CHECK(affected_messages->get_id() == telegram_api::messages_affectedMessages::ID);

    if (affected_messages->pts_count_ > 0) {
      td->messages_manager_->add_pending_channel_update(DialogId(channel_id_), make_tl_object<dummyUpdate>(),
                                                        affected_messages->pts_, affected_messages->pts_count_,
                                                        "DeleteChannelMessagesQuery");
    }
    if (--query_count_ == 0) {
      promise_.set_value(Unit());
    }
  }

  void on_error(uint64 id, Status status) override {
    if (!td->contacts_manager_->on_get_channel_error(channel_id_, status, "DeleteChannelMessagesQuery")) {
      LOG(ERROR) << "Receive error for delete channel messages: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class DeleteScheduledMessagesQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit DeleteScheduledMessagesQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, vector<MessageId> &&message_ids) {
    dialog_id_ = dialog_id;
    LOG(INFO) << "Send deleteScheduledMessagesQuery to delete " << format::as_array(message_ids);

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return on_error(0, Status::Error(400, "Can't access the chat"));
    }
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_deleteScheduledMessages(
        std::move(input_peer), MessagesManager::get_scheduled_server_message_ids(message_ids)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_deleteScheduledMessages>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for DeleteScheduledMessagesQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "DeleteScheduledMessagesQuery")) {
      LOG(ERROR) << "Receive error for delete scheduled messages: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetDialogNotifySettingsQuery : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;
    auto input_notify_peer = td->messages_manager_->get_input_notify_peer(dialog_id);
    CHECK(input_notify_peer != nullptr);
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::account_getNotifySettings(std::move(input_notify_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::account_getNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    td->messages_manager_->on_update_dialog_notify_settings(dialog_id_, std::move(ptr), "GetDialogNotifySettingsQuery");
    td->messages_manager_->on_get_dialog_notification_settings_query_finished(dialog_id_, Status::OK());
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetDialogNotifySettingsQuery");
    td->messages_manager_->on_get_dialog_notification_settings_query_finished(dialog_id_, std::move(status));
  }
};

class GetNotifySettingsExceptionsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit GetNotifySettingsExceptionsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(NotificationSettingsScope scope, bool filter_scope, bool compare_sound) {
    int32 flags = 0;
    tl_object_ptr<telegram_api::InputNotifyPeer> input_notify_peer;
    if (filter_scope) {
      flags |= telegram_api::account_getNotifyExceptions::PEER_MASK;
      input_notify_peer = get_input_notify_peer(scope);
    }
    if (compare_sound) {
      flags |= telegram_api::account_getNotifyExceptions::COMPARE_SOUND_MASK;
    }
    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::account_getNotifyExceptions(flags, false /* ignored */, std::move(input_notify_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::account_getNotifyExceptions>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto updates_ptr = result_ptr.move_as_ok();
    auto dialog_ids = UpdatesManager::get_update_notify_settings_dialog_ids(updates_ptr.get());
    vector<tl_object_ptr<telegram_api::User>> users;
    vector<tl_object_ptr<telegram_api::Chat>> chats;
    switch (updates_ptr->get_id()) {
      case telegram_api::updatesCombined::ID: {
        auto updates = static_cast<telegram_api::updatesCombined *>(updates_ptr.get());
        users = std::move(updates->users_);
        chats = std::move(updates->chats_);
        reset_to_empty(updates->users_);
        reset_to_empty(updates->chats_);
        break;
      }
      case telegram_api::updates::ID: {
        auto updates = static_cast<telegram_api::updates *>(updates_ptr.get());
        users = std::move(updates->users_);
        chats = std::move(updates->chats_);
        reset_to_empty(updates->users_);
        reset_to_empty(updates->chats_);
        break;
      }
    }
    td->contacts_manager_->on_get_users(std::move(users), "GetNotifySettingsExceptionsQuery");
    td->contacts_manager_->on_get_chats(std::move(chats), "GetNotifySettingsExceptionsQuery");
    for (auto &dialog_id : dialog_ids) {
      td->messages_manager_->force_create_dialog(dialog_id, "GetNotifySettingsExceptionsQuery");
    }
    td->updates_manager_->on_get_updates(std::move(updates_ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    promise_.set_error(std::move(status));
  }
};

class GetScopeNotifySettingsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  NotificationSettingsScope scope_;

 public:
  explicit GetScopeNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(NotificationSettingsScope scope) {
    scope_ = scope;
    auto input_notify_peer = get_input_notify_peer(scope);
    CHECK(input_notify_peer != nullptr);
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::account_getNotifySettings(std::move(input_notify_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::account_getNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    td->messages_manager_->on_update_scope_notify_settings(scope_, std::move(ptr));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    promise_.set_error(std::move(status));
  }
};

class UpdateDialogNotifySettingsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit UpdateDialogNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const DialogNotificationSettings &new_settings) {
    dialog_id_ = dialog_id;

    auto input_notify_peer = td->messages_manager_->get_input_notify_peer(dialog_id);
    if (input_notify_peer == nullptr) {
      return on_error(0, Status::Error(500, "Can't update chat notification settings"));
    }

    int32 flags = 0;
    if (!new_settings.use_default_mute_until) {
      flags |= telegram_api::inputPeerNotifySettings::MUTE_UNTIL_MASK;
    }
    if (!new_settings.use_default_sound) {
      flags |= telegram_api::inputPeerNotifySettings::SOUND_MASK;
    }
    if (!new_settings.use_default_show_preview) {
      flags |= telegram_api::inputPeerNotifySettings::SHOW_PREVIEWS_MASK;
    }
    if (new_settings.silent_send_message) {
      flags |= telegram_api::inputPeerNotifySettings::SILENT_MASK;
    }
    send_query(G()->net_query_creator().create(create_storer(telegram_api::account_updateNotifySettings(
        std::move(input_notify_peer), make_tl_object<telegram_api::inputPeerNotifySettings>(
                                          flags, new_settings.show_preview, new_settings.silent_send_message,
                                          new_settings.mute_until, new_settings.sound)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::account_updateNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UpdateDialogNotifySettingsQuery")) {
      LOG(INFO) << "Receive error for set chat notification settings: " << status;
    }

    if (!td->auth_manager_->is_bot() && td->messages_manager_->get_input_notify_peer(dialog_id_) != nullptr) {
      // trying to repair notification settings for this dialog
      td->messages_manager_->send_get_dialog_notification_settings_query(dialog_id_, Promise<>());
    }

    promise_.set_error(std::move(status));
  }
};

class UpdateScopeNotifySettingsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  NotificationSettingsScope scope_;

 public:
  explicit UpdateScopeNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(NotificationSettingsScope scope, const ScopeNotificationSettings &new_settings) {
    auto input_notify_peer = get_input_notify_peer(scope);
    CHECK(input_notify_peer != nullptr);
    int32 flags = telegram_api::inputPeerNotifySettings::MUTE_UNTIL_MASK |
                  telegram_api::inputPeerNotifySettings::SOUND_MASK |
                  telegram_api::inputPeerNotifySettings::SHOW_PREVIEWS_MASK;
    send_query(G()->net_query_creator().create(create_storer(telegram_api::account_updateNotifySettings(
        std::move(input_notify_peer),
        make_tl_object<telegram_api::inputPeerNotifySettings>(flags, new_settings.show_preview, false,
                                                              new_settings.mute_until, new_settings.sound)))));
    scope_ = scope;
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::account_updateNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for set notification settings: " << status;

    if (!td->auth_manager_->is_bot()) {
      // trying to repair notification settings for this scope
      td->messages_manager_->send_get_scope_notification_settings_query(scope_, Promise<>());
    }

    promise_.set_error(std::move(status));
  }
};

class ResetNotifySettingsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;

 public:
  explicit ResetNotifySettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send() {
    send_query(G()->net_query_creator().create(create_storer(telegram_api::account_resetNotifySettings())));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::account_resetNotifySettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!G()->close_flag()) {
      LOG(ERROR) << "Receive error for reset notification settings: " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetPeerSettingsQuery : public Td::ResultHandler {
  DialogId dialog_id_;

 public:
  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    send_query(
        G()->net_query_creator().create(create_storer(telegram_api::messages_getPeerSettings(std::move(input_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getPeerSettings>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_peer_settings(dialog_id_, result_ptr.move_as_ok());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for get peer settings: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetPeerSettingsQuery");
  }
};

class UpdatePeerSettingsQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit UpdatePeerSettingsQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, bool is_spam_dialog) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      return promise_.set_value(Unit());
    }

    if (is_spam_dialog) {
      send_query(
          G()->net_query_creator().create(create_storer(telegram_api::messages_reportSpam(std::move(input_peer)))));
    } else {
      send_query(G()->net_query_creator().create(
          create_storer(telegram_api::messages_hidePeerSettingsBar(std::move(input_peer)))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    static_assert(std::is_same<telegram_api::messages_reportSpam::ReturnType,
                               telegram_api::messages_hidePeerSettingsBar::ReturnType>::value,
                  "");
    auto result_ptr = fetch_result<telegram_api::messages_reportSpam>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_peer_settings(
        dialog_id_,
        make_tl_object<telegram_api::peerSettings>(0, false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                                   false /*ignored*/, false /*ignored*/, false /*ignored*/),
        true);

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for update peer settings: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "UpdatePeerSettingsQuery");
    td->messages_manager_->repair_dialog_action_bar(dialog_id_, "UpdatePeerSettingsQuery");
    promise_.set_error(std::move(status));
  }
};

class ReportEncryptedSpamQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReportEncryptedSpamQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_encrypted_chat(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    LOG(INFO) << "Report spam in " << to_string(input_peer);

    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::messages_reportEncryptedSpam(std::move(input_peer)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_reportEncryptedSpam>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_peer_settings(
        dialog_id_,
        make_tl_object<telegram_api::peerSettings>(0, false /*ignored*/, false /*ignored*/, false /*ignored*/,
                                                   false /*ignored*/, false /*ignored*/, false /*ignored*/),
        true);

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for report encrypted spam: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReportEncryptedSpamQuery");
    td->messages_manager_->repair_dialog_action_bar(
        DialogId(td->contacts_manager_->get_secret_chat_user_id(dialog_id_.get_secret_chat_id())),
        "ReportEncryptedSpamQuery");
    promise_.set_error(std::move(status));
  }
};

class ReportPeerQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit ReportPeerQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, tl_object_ptr<telegram_api::ReportReason> &&report_reason,
            const vector<MessageId> &message_ids) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    if (message_ids.empty()) {
      send_query(G()->net_query_creator().create(
          create_storer(telegram_api::account_reportPeer(std::move(input_peer), std::move(report_reason)))));
    } else {
      send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_report(
          std::move(input_peer), MessagesManager::get_server_message_ids(message_ids), std::move(report_reason)))));
    }
  }

  void on_result(uint64 id, BufferSlice packet) override {
    static_assert(
        std::is_same<telegram_api::account_reportPeer::ReturnType, telegram_api::messages_report::ReturnType>::value,
        "");
    auto result_ptr = fetch_result<telegram_api::account_reportPeer>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    bool result = result_ptr.ok();
    if (!result) {
      return on_error(id, Status::Error(400, "Receive false as result"));
    }

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    LOG(INFO) << "Receive error for report peer: " << status;
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "ReportPeerQuery");
    td->messages_manager_->repair_dialog_action_bar(dialog_id_, "ReportPeerQuery");
    promise_.set_error(std::move(status));
  }
};

class EditPeerFoldersQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  DialogId dialog_id_;

 public:
  explicit EditPeerFoldersQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, FolderId folder_id) {
    dialog_id_ = dialog_id;

    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);

    vector<telegram_api::object_ptr<telegram_api::inputFolderPeer>> input_folder_peers;
    input_folder_peers.push_back(
        telegram_api::make_object<telegram_api::inputFolderPeer>(std::move(input_peer), folder_id.get()));
    send_query(G()->net_query_creator().create(
        create_storer(telegram_api::folders_editPeerFolders(std::move(input_folder_peers)))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::folders_editPeerFolders>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(INFO) << "Receive result for EditPeerFoldersQuery: " << to_string(ptr);
    td->updates_manager_->on_get_updates(std::move(ptr));
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "EditPeerFoldersQuery")) {
      LOG(INFO) << "Receive error for EditPeerFoldersQuery: " << status;
    }

    // trying to repair folder ID for this dialog
    td->messages_manager_->get_dialog_info_full(dialog_id_, Auto());

    promise_.set_error(std::move(status));
  }
};

class GetStatsUrlQuery : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::httpUrl>> promise_;
  DialogId dialog_id_;

 public:
  explicit GetStatsUrlQuery(Promise<td_api::object_ptr<td_api::httpUrl>> &&promise) : promise_(std::move(promise)) {
  }

  void send(DialogId dialog_id, const string &parameters, bool is_dark) {
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    int32 flags = 0;
    if (is_dark) {
      flags |= telegram_api::messages_getStatsURL::DARK_MASK;
    }
    send_query(G()->net_query_creator().create(create_storer(
        telegram_api::messages_getStatsURL(flags, false /*ignored*/, std::move(input_peer), parameters))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_getStatsURL>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    promise_.set_value(td_api::make_object<td_api::httpUrl>(result->url_));
  }

  void on_error(uint64 id, Status status) override {
    td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetStatsUrlQuery");
    promise_.set_error(std::move(status));
  }
};

class RequestUrlAuthQuery : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::LoginUrlInfo>> promise_;
  string url_;
  DialogId dialog_id_;

 public:
  explicit RequestUrlAuthQuery(Promise<td_api::object_ptr<td_api::LoginUrlInfo>> &&promise)
      : promise_(std::move(promise)) {
  }

  void send(string url, DialogId dialog_id, MessageId message_id, int32 button_id) {
    url_ = std::move(url);
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_requestUrlAuth(
        std::move(input_peer), message_id.get_server_message_id().get(), button_id))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_requestUrlAuth>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive " << to_string(result);
    switch (result->get_id()) {
      case telegram_api::urlAuthResultRequest::ID: {
        auto request = telegram_api::move_object_as<telegram_api::urlAuthResultRequest>(result);
        UserId bot_user_id = ContactsManager::get_user_id(request->bot_);
        if (!bot_user_id.is_valid()) {
          return on_error(id, Status::Error(500, "Receive invalid bot_user_id"));
        }
        td->contacts_manager_->on_get_user(std::move(request->bot_), "RequestUrlAuthQuery");
        bool request_write_access =
            (request->flags_ & telegram_api::urlAuthResultRequest::REQUEST_WRITE_ACCESS_MASK) != 0;
        promise_.set_value(td_api::make_object<td_api::loginUrlInfoRequestConfirmation>(
            url_, request->domain_, td->contacts_manager_->get_user_id_object(bot_user_id, "RequestUrlAuthQuery"),
            request_write_access));
        break;
      }
      case telegram_api::urlAuthResultAccepted::ID: {
        auto accepted = telegram_api::move_object_as<telegram_api::urlAuthResultAccepted>(result);
        promise_.set_value(td_api::make_object<td_api::loginUrlInfoOpen>(accepted->url_, true));
        break;
      }
      case telegram_api::urlAuthResultDefault::ID:
        promise_.set_value(td_api::make_object<td_api::loginUrlInfoOpen>(url_, false));
        break;
    }
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "RequestUrlAuthQuery")) {
      LOG(INFO) << "RequestUrlAuthQuery returned " << status;
    }
    promise_.set_value(td_api::make_object<td_api::loginUrlInfoOpen>(url_, false));
  }
};

class AcceptUrlAuthQuery : public Td::ResultHandler {
  Promise<td_api::object_ptr<td_api::httpUrl>> promise_;
  string url_;
  DialogId dialog_id_;

 public:
  explicit AcceptUrlAuthQuery(Promise<td_api::object_ptr<td_api::httpUrl>> &&promise) : promise_(std::move(promise)) {
  }

  void send(string url, DialogId dialog_id, MessageId message_id, int32 button_id, bool allow_write_access) {
    url_ = std::move(url);
    dialog_id_ = dialog_id;
    auto input_peer = td->messages_manager_->get_input_peer(dialog_id, AccessRights::Read);
    CHECK(input_peer != nullptr);
    int32 flags = 0;
    if (allow_write_access) {
      flags |= telegram_api::messages_acceptUrlAuth::WRITE_ALLOWED_MASK;
    }
    send_query(G()->net_query_creator().create(create_storer(telegram_api::messages_acceptUrlAuth(
        flags, false /*ignored*/, std::move(input_peer), message_id.get_server_message_id().get(), button_id))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::messages_acceptUrlAuth>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto result = result_ptr.move_as_ok();
    LOG(INFO) << "Receive " << to_string(result);
    switch (result->get_id()) {
      case telegram_api::urlAuthResultRequest::ID:
        LOG(ERROR) << "Receive unexpected " << to_string(result);
        return on_error(id, Status::Error(500, "Receive unexpected urlAuthResultRequest"));
      case telegram_api::urlAuthResultAccepted::ID: {
        auto accepted = telegram_api::move_object_as<telegram_api::urlAuthResultAccepted>(result);
        promise_.set_value(td_api::make_object<td_api::httpUrl>(accepted->url_));
        break;
      }
      case telegram_api::urlAuthResultDefault::ID:
        promise_.set_value(td_api::make_object<td_api::httpUrl>(url_));
        break;
    }
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "AcceptUrlAuthQuery")) {
      LOG(INFO) << "AcceptUrlAuthQuery returned " << status;
    }
    promise_.set_error(std::move(status));
  }
};

class GetChannelDifferenceQuery : public Td::ResultHandler {
  DialogId dialog_id_;
  int32 pts_;
  int32 limit_;

 public:
  void send(DialogId dialog_id, tl_object_ptr<telegram_api::InputChannel> &&input_channel, int32 pts, int32 limit,
            bool force) {
    CHECK(pts >= 0);
    dialog_id_ = dialog_id;
    pts_ = pts;
    limit_ = limit;
    CHECK(input_channel != nullptr);

    int32 flags = 0;
    if (force) {
      flags |= telegram_api::updates_getChannelDifference::FORCE_MASK;
    }
    send_query(G()->net_query_creator().create(create_storer(telegram_api::updates_getChannelDifference(
        flags, false /*ignored*/, std::move(input_channel), make_tl_object<telegram_api::channelMessagesFilterEmpty>(),
        pts, limit))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::updates_getChannelDifference>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_channel_difference(dialog_id_, pts_, limit_, result_ptr.move_as_ok());
  }

  void on_error(uint64 id, Status status) override {
    if (!td->messages_manager_->on_get_dialog_error(dialog_id_, status, "GetChannelDifferenceQuery")) {
      LOG(ERROR) << "Receive updates.getChannelDifference error for " << dialog_id_ << " with pts " << pts_
                 << " and limit " << limit_ << ": " << status;
    }
    td->messages_manager_->on_get_channel_difference(dialog_id_, pts_, limit_, nullptr);
  }
};

class ResolveUsernameQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  string username_;

 public:
  explicit ResolveUsernameQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(const string &username) {
    username_ = username;

    LOG(INFO) << "Send ResolveUsernameQuery with username = " << username;
    send_query(G()->net_query_creator().create(create_storer(telegram_api::contacts_resolveUsername(username))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::contacts_resolveUsername>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    auto ptr = result_ptr.move_as_ok();
    LOG(DEBUG) << "Receive result for resolveUsername " << to_string(ptr);
    td->contacts_manager_->on_get_users(std::move(ptr->users_), "ResolveUsernameQuery");
    td->contacts_manager_->on_get_chats(std::move(ptr->chats_), "ResolveUsernameQuery");

    td->messages_manager_->on_resolved_username(username_, DialogId(ptr->peer_));

    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    if (status.message() == Slice("USERNAME_NOT_OCCUPIED")) {
      td->messages_manager_->drop_username(username_);
    }
    promise_.set_error(std::move(status));
  }
};

class GetChannelAdminLogQuery : public Td::ResultHandler {
  Promise<Unit> promise_;
  ChannelId channel_id_;
  int64 random_id_;

 public:
  explicit GetChannelAdminLogQuery(Promise<Unit> &&promise) : promise_(std::move(promise)) {
  }

  void send(ChannelId channel_id, const string &query, int64 from_event_id, int32 limit,
            tl_object_ptr<telegram_api::channelAdminLogEventsFilter> filter,
            vector<tl_object_ptr<telegram_api::InputUser>> input_users, int64 random_id) {
    channel_id_ = channel_id;
    random_id_ = random_id;

    auto input_channel = td->contacts_manager_->get_input_channel(channel_id);
    CHECK(input_channel != nullptr);

    int32 flags = 0;
    if (filter != nullptr) {
      flags |= telegram_api::channels_getAdminLog::EVENTS_FILTER_MASK;
    }
    if (!input_users.empty()) {
      flags |= telegram_api::channels_getAdminLog::ADMINS_MASK;
    }

    send_query(G()->net_query_creator().create(create_storer(telegram_api::channels_getAdminLog(
        flags, std::move(input_channel), query, std::move(filter), std::move(input_users), from_event_id, 0, limit))));
  }

  void on_result(uint64 id, BufferSlice packet) override {
    auto result_ptr = fetch_result<telegram_api::channels_getAdminLog>(packet);
    if (result_ptr.is_error()) {
      return on_error(id, result_ptr.move_as_error());
    }

    td->messages_manager_->on_get_event_log(channel_id_, random_id_, result_ptr.move_as_ok());
    promise_.set_value(Unit());
  }

  void on_error(uint64 id, Status status) override {
    td->contacts_manager_->on_get_channel_error(channel_id_, status, "GetChannelAdminLogQuery");
    td->messages_manager_->on_get_event_log(channel_id_, random_id_, nullptr);
    promise_.set_error(std::move(status));
  }
};

class MessagesManager::UploadMediaCallback : public FileManager::UploadCallback {
 public:
  void on_progress(FileId file_id) override {
  }
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media, file_id, std::move(input_file),
                       nullptr);
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media, file_id, nullptr,
                       std::move(input_file));
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) override {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_media_error, file_id, std::move(error));
  }
};

class MessagesManager::UploadThumbnailCallback : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_thumbnail, file_id, std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) override {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) override {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_thumbnail, file_id, nullptr);
  }
};

class MessagesManager::UploadDialogPhotoCallback : public FileManager::UploadCallback {
 public:
  void on_upload_ok(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_dialog_photo, file_id,
                       std::move(input_file));
  }
  void on_upload_encrypted_ok(FileId file_id, tl_object_ptr<telegram_api::InputEncryptedFile> input_file) override {
    UNREACHABLE();
  }
  void on_upload_secure_ok(FileId file_id, tl_object_ptr<telegram_api::InputSecureFile> input_file) override {
    UNREACHABLE();
  }
  void on_upload_error(FileId file_id, Status error) override {
    send_closure_later(G()->messages_manager(), &MessagesManager::on_upload_dialog_photo_error, file_id,
                       std::move(error));
  }
};

template <class StorerT>
void MessagesManager::Message::store(StorerT &storer) const {
  using td::store;
  bool has_sender = sender_user_id.is_valid();
  bool has_edit_date = edit_date > 0;
  bool has_random_id = random_id != 0;
  bool is_forwarded = forward_info != nullptr;
  bool is_reply = reply_to_message_id.is_valid();
  bool is_reply_to_random_id = reply_to_random_id != 0;
  bool is_via_bot = via_bot_user_id.is_valid();
  bool has_views = views > 0;
  bool has_reply_markup = reply_markup != nullptr;
  bool has_ttl = ttl != 0;
  bool has_author_signature = !author_signature.empty();
  bool has_forward_author_signature = is_forwarded && !forward_info->author_signature.empty();
  bool has_media_album_id = media_album_id != 0;
  bool has_forward_from =
      is_forwarded && (forward_info->from_dialog_id.is_valid() || forward_info->from_message_id.is_valid());
  bool has_send_date = message_id.is_yet_unsent() && send_date != 0;
  bool has_flags2 = true;
  bool has_notification_id = notification_id.is_valid();
  bool has_forward_sender_name = is_forwarded && !forward_info->sender_name.empty();
  bool has_send_error_code = send_error_code != 0;
  bool has_real_forward_from = real_forward_from_dialog_id.is_valid() && real_forward_from_message_id.is_valid();
  bool has_legacy_layer = legacy_layer != 0;
  bool has_restriction_reasons = !restriction_reasons.empty();
  BEGIN_STORE_FLAGS();
  STORE_FLAG(is_channel_post);
  STORE_FLAG(is_outgoing);
  STORE_FLAG(is_failed_to_send);
  STORE_FLAG(disable_notification);
  STORE_FLAG(contains_mention);
  STORE_FLAG(from_background);
  STORE_FLAG(disable_web_page_preview);
  STORE_FLAG(clear_draft);
  STORE_FLAG(have_previous);
  STORE_FLAG(have_next);
  STORE_FLAG(has_sender);
  STORE_FLAG(has_edit_date);
  STORE_FLAG(has_random_id);
  STORE_FLAG(is_forwarded);
  STORE_FLAG(is_reply);
  STORE_FLAG(is_reply_to_random_id);
  STORE_FLAG(is_via_bot);
  STORE_FLAG(has_views);
  STORE_FLAG(has_reply_markup);
  STORE_FLAG(has_ttl);
  STORE_FLAG(has_author_signature);
  STORE_FLAG(has_forward_author_signature);
  STORE_FLAG(had_reply_markup);
  STORE_FLAG(contains_unread_mention);
  STORE_FLAG(has_media_album_id);
  STORE_FLAG(has_forward_from);
  STORE_FLAG(in_game_share);
  STORE_FLAG(is_content_secret);
  STORE_FLAG(has_send_date);
  STORE_FLAG(has_flags2);
  END_STORE_FLAGS();
  if (has_flags2) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_notification_id);
    STORE_FLAG(is_mention_notification_disabled);
    STORE_FLAG(had_forward_info);
    STORE_FLAG(has_forward_sender_name);
    STORE_FLAG(has_send_error_code);
    STORE_FLAG(hide_via_bot);
    STORE_FLAG(is_bot_start_message);
    STORE_FLAG(has_real_forward_from);
    STORE_FLAG(has_legacy_layer);
    STORE_FLAG(hide_edit_date);
    STORE_FLAG(has_restriction_reasons);
    STORE_FLAG(is_from_scheduled);
    STORE_FLAG(is_copy);
    END_STORE_FLAGS();
  }

  store(message_id, storer);
  if (has_sender) {
    store(sender_user_id, storer);
  }
  store(date, storer);
  if (has_edit_date) {
    store(edit_date, storer);
  }
  if (has_send_date) {
    store(send_date, storer);
  }
  if (has_random_id) {
    store(random_id, storer);
  }
  if (is_forwarded) {
    store(forward_info->sender_user_id, storer);
    store(forward_info->date, storer);
    store(forward_info->dialog_id, storer);
    store(forward_info->message_id, storer);
    if (has_forward_author_signature) {
      store(forward_info->author_signature, storer);
    }
    if (has_forward_sender_name) {
      store(forward_info->sender_name, storer);
    }
    if (has_forward_from) {
      store(forward_info->from_dialog_id, storer);
      store(forward_info->from_message_id, storer);
    }
  }
  if (has_real_forward_from) {
    store(real_forward_from_dialog_id, storer);
    store(real_forward_from_message_id, storer);
  }
  if (is_reply) {
    store(reply_to_message_id, storer);
  }
  if (is_reply_to_random_id) {
    store(reply_to_random_id, storer);
  }
  if (is_via_bot) {
    store(via_bot_user_id, storer);
  }
  if (has_views) {
    store(views, storer);
  }
  if (has_ttl) {
    store(ttl, storer);
    store_time(ttl_expires_at, storer);
  }
  if (has_send_error_code) {
    store(send_error_code, storer);
    store(send_error_message, storer);
    if (send_error_code == 429) {
      store_time(try_resend_at, storer);
    }
  }
  if (has_author_signature) {
    store(author_signature, storer);
  }
  if (has_media_album_id) {
    store(media_album_id, storer);
  }
  if (has_notification_id) {
    store(notification_id, storer);
  }
  if (has_legacy_layer) {
    store(legacy_layer, storer);
  }
  if (has_restriction_reasons) {
    store(restriction_reasons, storer);
  }
  store_message_content(content.get(), storer);
  if (has_reply_markup) {
    store(reply_markup, storer);
  }
}

// do not forget to resolve message dependencies
template <class ParserT>
void MessagesManager::Message::parse(ParserT &parser) {
  using td::parse;
  bool has_sender;
  bool has_edit_date;
  bool has_random_id;
  bool is_forwarded;
  bool is_reply;
  bool is_reply_to_random_id;
  bool is_via_bot;
  bool has_views;
  bool has_reply_markup;
  bool has_ttl;
  bool has_author_signature;
  bool has_forward_author_signature;
  bool has_media_album_id;
  bool has_forward_from;
  bool has_send_date;
  bool has_flags2;
  bool has_notification_id = false;
  bool has_forward_sender_name = false;
  bool has_send_error_code = false;
  bool has_real_forward_from = false;
  bool has_legacy_layer = false;
  bool has_restriction_reasons = false;
  BEGIN_PARSE_FLAGS();
  PARSE_FLAG(is_channel_post);
  PARSE_FLAG(is_outgoing);
  PARSE_FLAG(is_failed_to_send);
  PARSE_FLAG(disable_notification);
  PARSE_FLAG(contains_mention);
  PARSE_FLAG(from_background);
  PARSE_FLAG(disable_web_page_preview);
  PARSE_FLAG(clear_draft);
  PARSE_FLAG(have_previous);
  PARSE_FLAG(have_next);
  PARSE_FLAG(has_sender);
  PARSE_FLAG(has_edit_date);
  PARSE_FLAG(has_random_id);
  PARSE_FLAG(is_forwarded);
  PARSE_FLAG(is_reply);
  PARSE_FLAG(is_reply_to_random_id);
  PARSE_FLAG(is_via_bot);
  PARSE_FLAG(has_views);
  PARSE_FLAG(has_reply_markup);
  PARSE_FLAG(has_ttl);
  PARSE_FLAG(has_author_signature);
  PARSE_FLAG(has_forward_author_signature);
  PARSE_FLAG(had_reply_markup);
  PARSE_FLAG(contains_unread_mention);
  PARSE_FLAG(has_media_album_id);
  PARSE_FLAG(has_forward_from);
  PARSE_FLAG(in_game_share);
  PARSE_FLAG(is_content_secret);
  PARSE_FLAG(has_send_date);
  PARSE_FLAG(has_flags2);
  END_PARSE_FLAGS();
  if (has_flags2) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_notification_id);
    PARSE_FLAG(is_mention_notification_disabled);
    PARSE_FLAG(had_forward_info);
    PARSE_FLAG(has_forward_sender_name);
    PARSE_FLAG(has_send_error_code);
    PARSE_FLAG(hide_via_bot);
    PARSE_FLAG(is_bot_start_message);
    PARSE_FLAG(has_real_forward_from);
    PARSE_FLAG(has_legacy_layer);
    PARSE_FLAG(hide_edit_date);
    PARSE_FLAG(has_restriction_reasons);
    PARSE_FLAG(is_from_scheduled);
    PARSE_FLAG(is_copy);
    END_PARSE_FLAGS();
  }

  parse(message_id, parser);
  random_y = get_random_y(message_id);
  if (has_sender) {
    parse(sender_user_id, parser);
  }
  parse(date, parser);
  if (has_edit_date) {
    parse(edit_date, parser);
  }
  if (has_send_date) {
    CHECK(message_id.is_valid() || message_id.is_valid_scheduled());
    CHECK(message_id.is_yet_unsent());
    parse(send_date, parser);
  } else if (message_id.is_valid() && message_id.is_yet_unsent()) {
    send_date = date;  // for backward compatibility
  }
  if (has_random_id) {
    parse(random_id, parser);
  }
  if (is_forwarded) {
    forward_info = make_unique<MessageForwardInfo>();
    parse(forward_info->sender_user_id, parser);
    parse(forward_info->date, parser);
    parse(forward_info->dialog_id, parser);
    parse(forward_info->message_id, parser);
    if (has_forward_author_signature) {
      parse(forward_info->author_signature, parser);
    }
    if (has_forward_sender_name) {
      parse(forward_info->sender_name, parser);
    }
    if (has_forward_from) {
      parse(forward_info->from_dialog_id, parser);
      parse(forward_info->from_message_id, parser);
    }
  }
  if (has_real_forward_from) {
    parse(real_forward_from_dialog_id, parser);
    parse(real_forward_from_message_id, parser);
  }
  if (is_reply) {
    parse(reply_to_message_id, parser);
  }
  if (is_reply_to_random_id) {
    parse(reply_to_random_id, parser);
  }
  if (is_via_bot) {
    parse(via_bot_user_id, parser);
  }
  if (has_views) {
    parse(views, parser);
  }
  if (has_ttl) {
    parse(ttl, parser);
    parse_time(ttl_expires_at, parser);
  }
  if (has_send_error_code) {
    parse(send_error_code, parser);
    parse(send_error_message, parser);
    if (send_error_code == 429) {
      parse_time(try_resend_at, parser);
    }
  }
  if (has_author_signature) {
    parse(author_signature, parser);
  }
  if (has_media_album_id) {
    parse(media_album_id, parser);
  }
  if (has_notification_id) {
    parse(notification_id, parser);
  }
  if (has_legacy_layer) {
    parse(legacy_layer, parser);
  }
  if (has_restriction_reasons) {
    parse(restriction_reasons, parser);
  }
  parse_message_content(content, parser);
  if (has_reply_markup) {
    parse(reply_markup, parser);
  }
  is_content_secret |=
      is_secret_message_content(ttl, content->get_type());  // repair is_content_secret for old messages
}

template <class StorerT>
void MessagesManager::NotificationGroupInfo::store(StorerT &storer) const {
  using td::store;
  store(group_id, storer);
  store(last_notification_date, storer);
  store(last_notification_id, storer);
  store(max_removed_notification_id, storer);
  store(max_removed_message_id, storer);
}

template <class ParserT>
void MessagesManager::NotificationGroupInfo::parse(ParserT &parser) {
  using td::parse;
  parse(group_id, parser);
  parse(last_notification_date, parser);
  parse(last_notification_id, parser);
  parse(max_removed_notification_id, parser);
  if (parser.version() >= static_cast<int32>(Version::AddNotificationGroupInfoMaxRemovedMessageId)) {
    parse(max_removed_message_id, parser);
  }
}

template <class StorerT>
void MessagesManager::Dialog::store(StorerT &storer) const {
  using td::store;
  const Message *last_database_message = nullptr;
  if (last_database_message_id.is_valid()) {
    last_database_message = get_message(this, last_database_message_id);
  }

  bool has_draft_message = draft_message != nullptr;
  bool has_last_database_message = last_database_message != nullptr;
  bool has_first_database_message_id = first_database_message_id.is_valid();
  bool is_pinned = pinned_order != DEFAULT_ORDER;
  bool has_first_database_message_id_by_index = true;
  bool has_message_count_by_index = true;
  bool has_client_data = !client_data.empty();
  bool has_last_read_all_mentions_message_id = last_read_all_mentions_message_id.is_valid();
  bool has_max_unavailable_message_id = max_unavailable_message_id.is_valid();
  bool has_local_unread_count = local_unread_count != 0;
  bool has_deleted_last_message = delete_last_message_date > 0;
  bool has_last_clear_history_message_id = last_clear_history_message_id.is_valid();
  bool has_last_database_message_id = !has_last_database_message && last_database_message_id.is_valid();
  bool has_message_notification_group =
      message_notification_group.group_id.is_valid() && !message_notification_group.try_reuse;
  bool has_mention_notification_group =
      mention_notification_group.group_id.is_valid() && !mention_notification_group.try_reuse;
  bool has_new_secret_chat_notification_id = new_secret_chat_notification_id.is_valid();
  bool has_pinned_message_notification = pinned_message_notification_message_id.is_valid();
  bool has_pinned_message_id = pinned_message_id.is_valid();
  bool has_flags2 = true;
  bool has_max_notification_message_id =
      max_notification_message_id.is_valid() && max_notification_message_id > last_new_message_id;
  bool has_folder_id = folder_id != FolderId();
  bool has_pending_read_channel_inbox = pending_read_channel_inbox_pts != 0;
  BEGIN_STORE_FLAGS();
  STORE_FLAG(has_draft_message);
  STORE_FLAG(has_last_database_message);
  STORE_FLAG(know_can_report_spam);
  STORE_FLAG(can_report_spam);
  STORE_FLAG(has_first_database_message_id);
  STORE_FLAG(is_pinned);
  STORE_FLAG(has_first_database_message_id_by_index);
  STORE_FLAG(has_message_count_by_index);
  STORE_FLAG(has_client_data);
  STORE_FLAG(need_restore_reply_markup);
  STORE_FLAG(have_full_history);
  STORE_FLAG(has_last_read_all_mentions_message_id);
  STORE_FLAG(has_max_unavailable_message_id);
  STORE_FLAG(is_last_read_inbox_message_id_inited);
  STORE_FLAG(is_last_read_outbox_message_id_inited);
  STORE_FLAG(has_local_unread_count);
  STORE_FLAG(has_deleted_last_message);
  STORE_FLAG(has_last_clear_history_message_id);
  STORE_FLAG(is_last_message_deleted_locally);
  STORE_FLAG(has_contact_registered_message);
  STORE_FLAG(has_last_database_message_id);
  STORE_FLAG(need_repair_server_unread_count);
  STORE_FLAG(is_marked_as_unread);
  STORE_FLAG(has_message_notification_group);
  STORE_FLAG(has_mention_notification_group);
  STORE_FLAG(has_new_secret_chat_notification_id);
  STORE_FLAG(has_pinned_message_notification);
  STORE_FLAG(has_pinned_message_id);
  STORE_FLAG(is_pinned_message_id_inited);
  STORE_FLAG(has_flags2);
  END_STORE_FLAGS();

  store(dialog_id, storer);  // must be stored at offset 4

  if (has_flags2) {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(has_max_notification_message_id);
    STORE_FLAG(has_folder_id);
    STORE_FLAG(is_folder_id_inited);
    STORE_FLAG(has_pending_read_channel_inbox);
    STORE_FLAG(know_action_bar);
    STORE_FLAG(can_add_contact);
    STORE_FLAG(can_block_user);
    STORE_FLAG(can_share_phone_number);
    STORE_FLAG(can_report_location);
    STORE_FLAG(has_scheduled_server_messages);
    STORE_FLAG(has_scheduled_database_messages);
    END_STORE_FLAGS();
  }

  store(last_new_message_id, storer);
  store(server_unread_count, storer);
  if (has_local_unread_count) {
    store(local_unread_count, storer);
  }
  store(last_read_inbox_message_id, storer);
  store(last_read_outbox_message_id, storer);
  store(reply_markup_message_id, storer);
  store(notification_settings, storer);
  if (has_draft_message) {
    store(draft_message, storer);
  }
  store(last_clear_history_date, storer);
  store(order, storer);
  if (has_last_database_message) {
    store(*last_database_message, storer);
  }
  if (has_first_database_message_id) {
    store(first_database_message_id, storer);
  }
  if (is_pinned) {
    store(pinned_order, storer);
  }
  if (has_deleted_last_message) {
    store(delete_last_message_date, storer);
    store(deleted_last_message_id, storer);
  }
  if (has_last_clear_history_message_id) {
    store(last_clear_history_message_id, storer);
  }

  if (has_first_database_message_id_by_index) {
    store(static_cast<int32>(first_database_message_id_by_index.size()), storer);
    for (auto first_message_id : first_database_message_id_by_index) {
      store(first_message_id, storer);
    }
  }
  if (has_message_count_by_index) {
    store(static_cast<int32>(message_count_by_index.size()), storer);
    for (auto message_count : message_count_by_index) {
      store(message_count, storer);
    }
  }
  if (has_client_data) {
    store(client_data, storer);
  }
  if (has_last_read_all_mentions_message_id) {
    store(last_read_all_mentions_message_id, storer);
  }
  if (has_max_unavailable_message_id) {
    store(max_unavailable_message_id, storer);
  }
  if (has_last_database_message_id) {
    store(last_database_message_id, storer);
  }
  if (has_message_notification_group) {
    store(message_notification_group, storer);
  }
  if (has_mention_notification_group) {
    store(mention_notification_group, storer);
  }
  if (has_new_secret_chat_notification_id) {
    store(new_secret_chat_notification_id, storer);
  }
  if (has_pinned_message_notification) {
    store(pinned_message_notification_message_id, storer);
  }
  if (has_pinned_message_id) {
    store(pinned_message_id, storer);
  }
  if (has_max_notification_message_id) {
    store(max_notification_message_id, storer);
  }
  if (has_folder_id) {
    store(folder_id, storer);
  }
  if (has_pending_read_channel_inbox) {
    store(pending_read_channel_inbox_pts, storer);
    store(pending_read_channel_inbox_max_message_id, storer);
    store(pending_read_channel_inbox_server_unread_count, storer);
  }
}

// do not forget to resolve dialog dependencies including dependencies of last_message
template <class ParserT>
void MessagesManager::Dialog::parse(ParserT &parser) {
  using td::parse;
  bool has_draft_message;
  bool has_last_database_message;
  bool has_first_database_message_id;
  bool is_pinned;
  bool has_first_database_message_id_by_index;
  bool has_message_count_by_index;
  bool has_client_data;
  bool has_last_read_all_mentions_message_id;
  bool has_max_unavailable_message_id;
  bool has_local_unread_count;
  bool has_deleted_last_message;
  bool has_last_clear_history_message_id;
  bool has_last_database_message_id;
  bool has_message_notification_group;
  bool has_mention_notification_group;
  bool has_new_secret_chat_notification_id;
  bool has_pinned_message_notification;
  bool has_pinned_message_id;
  bool has_flags2;
  bool has_max_notification_message_id = false;
  bool has_folder_id = false;
  bool has_pending_read_channel_inbox = false;
  BEGIN_PARSE_FLAGS();
  PARSE_FLAG(has_draft_message);
  PARSE_FLAG(has_last_database_message);
  PARSE_FLAG(know_can_report_spam);
  PARSE_FLAG(can_report_spam);
  PARSE_FLAG(has_first_database_message_id);
  PARSE_FLAG(is_pinned);
  PARSE_FLAG(has_first_database_message_id_by_index);
  PARSE_FLAG(has_message_count_by_index);
  PARSE_FLAG(has_client_data);
  PARSE_FLAG(need_restore_reply_markup);
  PARSE_FLAG(have_full_history);
  PARSE_FLAG(has_last_read_all_mentions_message_id);
  PARSE_FLAG(has_max_unavailable_message_id);
  PARSE_FLAG(is_last_read_inbox_message_id_inited);
  PARSE_FLAG(is_last_read_outbox_message_id_inited);
  PARSE_FLAG(has_local_unread_count);
  PARSE_FLAG(has_deleted_last_message);
  PARSE_FLAG(has_last_clear_history_message_id);
  PARSE_FLAG(is_last_message_deleted_locally);
  PARSE_FLAG(has_contact_registered_message);
  PARSE_FLAG(has_last_database_message_id);
  PARSE_FLAG(need_repair_server_unread_count);
  PARSE_FLAG(is_marked_as_unread);
  PARSE_FLAG(has_message_notification_group);
  PARSE_FLAG(has_mention_notification_group);
  PARSE_FLAG(has_new_secret_chat_notification_id);
  PARSE_FLAG(has_pinned_message_notification);
  PARSE_FLAG(has_pinned_message_id);
  PARSE_FLAG(is_pinned_message_id_inited);
  PARSE_FLAG(has_flags2);
  END_PARSE_FLAGS();

  parse(dialog_id, parser);  // must be stored at offset 4

  if (has_flags2) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(has_max_notification_message_id);
    PARSE_FLAG(has_folder_id);
    PARSE_FLAG(is_folder_id_inited);
    PARSE_FLAG(has_pending_read_channel_inbox);
    PARSE_FLAG(know_action_bar);
    PARSE_FLAG(can_add_contact);
    PARSE_FLAG(can_block_user);
    PARSE_FLAG(can_share_phone_number);
    PARSE_FLAG(can_report_location);
    PARSE_FLAG(has_scheduled_server_messages);
    PARSE_FLAG(has_scheduled_database_messages);
    END_PARSE_FLAGS();
  } else {
    is_folder_id_inited = false;
    know_action_bar = false;
    can_add_contact = false;
    can_block_user = false;
    can_share_phone_number = false;
    can_report_location = false;
  }

  parse(last_new_message_id, parser);
  parse(server_unread_count, parser);
  if (has_local_unread_count) {
    parse(local_unread_count, parser);
  }
  parse(last_read_inbox_message_id, parser);
  if (last_read_inbox_message_id.is_valid()) {
    is_last_read_inbox_message_id_inited = true;
  }
  parse(last_read_outbox_message_id, parser);
  if (last_read_outbox_message_id.is_valid()) {
    is_last_read_outbox_message_id_inited = true;
  }
  parse(reply_markup_message_id, parser);
  parse(notification_settings, parser);
  if (has_draft_message) {
    parse(draft_message, parser);
  }
  parse(last_clear_history_date, parser);
  parse(order, parser);
  if (has_last_database_message) {
    parse(messages, parser);
  }
  if (has_first_database_message_id) {
    parse(first_database_message_id, parser);
  }
  if (is_pinned) {
    parse(pinned_order, parser);
  }
  if (has_deleted_last_message) {
    parse(delete_last_message_date, parser);
    parse(deleted_last_message_id, parser);
  }
  if (has_last_clear_history_message_id) {
    parse(last_clear_history_message_id, parser);
  }

  if (has_first_database_message_id_by_index) {
    int32 size;
    parse(size, parser);
    if (size < 0) {
      // the logevent is broken
      // it should be impossible, but has happenned at least once
      parser.set_error("Wrong first_database_message_id_by_index table size");
      return;
    }
    LOG_CHECK(static_cast<size_t>(size) <= first_database_message_id_by_index.size())
        << size << " " << first_database_message_id_by_index.size();
    for (int32 i = 0; i < size; i++) {
      parse(first_database_message_id_by_index[i], parser);
    }
  }
  if (has_message_count_by_index) {
    int32 size;
    parse(size, parser);
    if (size < 0) {
      // the logevent is broken
      // it should be impossible, but has happenned at least once
      parser.set_error("Wrong message_count_by_index table size");
      return;
    }
    LOG_CHECK(static_cast<size_t>(size) <= message_count_by_index.size())
        << size << " " << message_count_by_index.size();
    for (int32 i = 0; i < size; i++) {
      parse(message_count_by_index[i], parser);
    }
  }
  unread_mention_count = message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)];
  LOG(INFO) << "Set unread mention message count in " << dialog_id << " to " << unread_mention_count;
  if (unread_mention_count < 0) {
    unread_mention_count = 0;
  }
  if (has_client_data) {
    parse(client_data, parser);
  }
  if (has_last_read_all_mentions_message_id) {
    parse(last_read_all_mentions_message_id, parser);
  }
  if (has_max_unavailable_message_id) {
    parse(max_unavailable_message_id, parser);
  }
  if (has_last_database_message_id) {
    parse(last_database_message_id, parser);
  }
  if (has_message_notification_group) {
    parse(message_notification_group, parser);
  }
  if (has_mention_notification_group) {
    parse(mention_notification_group, parser);
  }
  if (has_new_secret_chat_notification_id) {
    parse(new_secret_chat_notification_id, parser);
  }
  if (has_pinned_message_notification) {
    parse(pinned_message_notification_message_id, parser);
  }
  if (has_pinned_message_id) {
    parse(pinned_message_id, parser);
  }
  if (has_max_notification_message_id) {
    parse(max_notification_message_id, parser);
  }
  if (has_folder_id) {
    parse(folder_id, parser);
  }
  if (has_pending_read_channel_inbox) {
    parse(pending_read_channel_inbox_pts, parser);
    parse(pending_read_channel_inbox_max_message_id, parser);
    parse(pending_read_channel_inbox_server_unread_count, parser);
  }
}

template <class StorerT>
void MessagesManager::CallsDbState::store(StorerT &storer) const {
  using td::store;
  store(static_cast<int32>(first_calls_database_message_id_by_index.size()), storer);
  for (auto first_message_id : first_calls_database_message_id_by_index) {
    store(first_message_id, storer);
  }
  store(static_cast<int32>(message_count_by_index.size()), storer);
  for (auto message_count : message_count_by_index) {
    store(message_count, storer);
  }
}

template <class ParserT>
void MessagesManager::CallsDbState::parse(ParserT &parser) {
  using td::parse;
  int32 size;
  parse(size, parser);
  LOG_CHECK(static_cast<size_t>(size) <= first_calls_database_message_id_by_index.size())
      << size << " " << first_calls_database_message_id_by_index.size();
  for (int32 i = 0; i < size; i++) {
    parse(first_calls_database_message_id_by_index[i], parser);
  }
  parse(size, parser);
  LOG_CHECK(static_cast<size_t>(size) <= message_count_by_index.size()) << size << " " << message_count_by_index.size();
  for (int32 i = 0; i < size; i++) {
    parse(message_count_by_index[i], parser);
  }
}

void MessagesManager::load_calls_db_state() {
  if (!G()->parameters().use_message_db) {
    return;
  }
  std::fill(calls_db_state_.message_count_by_index.begin(), calls_db_state_.message_count_by_index.end(), -1);
  auto value = G()->td_db()->get_sqlite_sync_pmc()->get("calls_db_state");
  if (value.empty()) {
    return;
  }
  log_event_parse(calls_db_state_, value).ensure();
  LOG(INFO) << "Save calls database state " << calls_db_state_.first_calls_database_message_id_by_index[0] << " ("
            << calls_db_state_.message_count_by_index[0] << ") "
            << calls_db_state_.first_calls_database_message_id_by_index[1] << " ("
            << calls_db_state_.message_count_by_index[1] << ")";
}

void MessagesManager::save_calls_db_state() {
  if (!G()->parameters().use_message_db) {
    return;
  }

  LOG(INFO) << "Save calls database state " << calls_db_state_.first_calls_database_message_id_by_index[0] << " ("
            << calls_db_state_.message_count_by_index[0] << ") "
            << calls_db_state_.first_calls_database_message_id_by_index[1] << " ("
            << calls_db_state_.message_count_by_index[1] << ")";
  G()->td_db()->get_sqlite_pmc()->set("calls_db_state", log_event_store(calls_db_state_).as_slice().str(), Auto());
}

MessagesManager::Dialog::~Dialog() {
  if (!G()->close_flag()) {
    LOG(ERROR) << "Destroy " << dialog_id;
  }
}

MessagesManager::MessagesManager(Td *td, ActorShared<> parent) : td_(td), parent_(std::move(parent)) {
  upload_media_callback_ = std::make_shared<UploadMediaCallback>();
  upload_thumbnail_callback_ = std::make_shared<UploadThumbnailCallback>();
  upload_dialog_photo_callback_ = std::make_shared<UploadDialogPhotoCallback>();

  channel_get_difference_timeout_.set_callback(on_channel_get_difference_timeout_callback);
  channel_get_difference_timeout_.set_callback_data(static_cast<void *>(this));

  channel_get_difference_retry_timeout_.set_callback(on_channel_get_difference_timeout_callback);
  channel_get_difference_retry_timeout_.set_callback_data(static_cast<void *>(this));

  pending_message_views_timeout_.set_callback(on_pending_message_views_timeout_callback);
  pending_message_views_timeout_.set_callback_data(static_cast<void *>(this));

  pending_message_live_location_view_timeout_.set_callback(on_pending_message_live_location_view_timeout_callback);
  pending_message_live_location_view_timeout_.set_callback_data(static_cast<void *>(this));

  pending_draft_message_timeout_.set_callback(on_pending_draft_message_timeout_callback);
  pending_draft_message_timeout_.set_callback_data(static_cast<void *>(this));

  pending_read_history_timeout_.set_callback(on_pending_read_history_timeout_callback);
  pending_read_history_timeout_.set_callback_data(static_cast<void *>(this));

  pending_updated_dialog_timeout_.set_callback(on_pending_updated_dialog_timeout_callback);
  pending_updated_dialog_timeout_.set_callback_data(static_cast<void *>(this));

  pending_unload_dialog_timeout_.set_callback(on_pending_unload_dialog_timeout_callback);
  pending_unload_dialog_timeout_.set_callback_data(static_cast<void *>(this));

  dialog_unmute_timeout_.set_callback(on_dialog_unmute_timeout_callback);
  dialog_unmute_timeout_.set_callback_data(static_cast<void *>(this));

  pending_send_dialog_action_timeout_.set_callback(on_pending_send_dialog_action_timeout_callback);
  pending_send_dialog_action_timeout_.set_callback_data(static_cast<void *>(this));

  active_dialog_action_timeout_.set_callback(on_active_dialog_action_timeout_callback);
  active_dialog_action_timeout_.set_callback_data(static_cast<void *>(this));

  update_dialog_online_member_count_timeout_.set_callback(on_update_dialog_online_member_count_timeout_callback);
  update_dialog_online_member_count_timeout_.set_callback_data(static_cast<void *>(this));

  preload_dialog_list_timeout_.set_callback(on_preload_dialog_list_timeout_callback);
  preload_dialog_list_timeout_.set_callback_data(static_cast<void *>(this));

  sequence_dispatcher_ = create_actor<MultiSequenceDispatcher>("multi sequence dispatcher");
}

MessagesManager::~MessagesManager() = default;

void MessagesManager::on_channel_get_difference_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_channel_get_difference_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_message_views_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_pending_message_views_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_pending_message_live_location_view_timeout_callback(void *messages_manager_ptr,
                                                                             int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::view_message_live_location_on_server, task_id);
}

void MessagesManager::on_pending_draft_message_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::save_dialog_draft_message_on_server, DialogId(dialog_id_int));
}

void MessagesManager::on_pending_read_history_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::read_history_on_server_impl,
                     DialogId(dialog_id_int), MessageId());
}

void MessagesManager::on_pending_updated_dialog_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  // no check for G()->close_flag() to save dialogs even while closing

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  // TODO it is unsafe to save dialog to database before binlog is flushed

  // no send_closure_later, because messages_manager can be not an actor while closing
  messages_manager->save_dialog_to_database(DialogId(dialog_id_int));
}

void MessagesManager::on_pending_unload_dialog_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::unload_dialog,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_dialog_unmute_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  if (1 <= dialog_id_int && dialog_id_int <= 3) {
    send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_scope_unmute,
                       static_cast<NotificationSettingsScope>(dialog_id_int - 1));
  } else {
    send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_dialog_unmute,
                       DialogId(dialog_id_int));
  }
}

void MessagesManager::on_pending_send_dialog_action_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_send_dialog_action_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_active_dialog_action_timeout_callback(void *messages_manager_ptr, int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::on_active_dialog_action_timeout,
                     DialogId(dialog_id_int));
}

void MessagesManager::on_update_dialog_online_member_count_timeout_callback(void *messages_manager_ptr,
                                                                            int64 dialog_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager),
                     &MessagesManager::on_update_dialog_online_member_count_timeout, DialogId(dialog_id_int));
}

void MessagesManager::on_preload_dialog_list_timeout_callback(void *messages_manager_ptr, int64 folder_id_int) {
  if (G()->close_flag()) {
    return;
  }

  auto messages_manager = static_cast<MessagesManager *>(messages_manager_ptr);
  send_closure_later(messages_manager->actor_id(messages_manager), &MessagesManager::preload_dialog_list,
                     FolderId(narrow_cast<int32>(folder_id_int)));
}

BufferSlice MessagesManager::get_dialog_database_value(const Dialog *d) {
  // can't use log_event_store, because it tries to parse stored Dialog
  LogEventStorerCalcLength storer_calc_length;
  store(*d, storer_calc_length);

  BufferSlice value_buffer{storer_calc_length.get_length()};
  auto value = value_buffer.as_slice();

  LogEventStorerUnsafe storer_unsafe(value.ubegin());
  store(*d, storer_unsafe);
  return value_buffer;
}

void MessagesManager::save_dialog_to_database(DialogId dialog_id) {
  CHECK(G()->parameters().use_message_db);
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Save " << dialog_id << " to database";
  vector<NotificationGroupKey> changed_group_keys;
  bool can_reuse_notification_group = false;
  auto add_group_key = [&](auto &group_info) {
    if (group_info.is_changed) {
      can_reuse_notification_group |= group_info.try_reuse;
      changed_group_keys.emplace_back(group_info.group_id, group_info.try_reuse ? DialogId() : dialog_id,
                                      group_info.last_notification_date);
      group_info.is_changed = false;
    }
  };
  add_group_key(d->message_notification_group);
  add_group_key(d->mention_notification_group);
  G()->td_db()->get_dialog_db_async()->add_dialog(
      dialog_id, d->folder_id, d->is_folder_id_inited ? d->order : 0, get_dialog_database_value(d),
      std::move(changed_group_keys), PromiseCreator::lambda([dialog_id, can_reuse_notification_group](Result<> result) {
        send_closure(G()->messages_manager(), &MessagesManager::on_save_dialog_to_database, dialog_id,
                     can_reuse_notification_group, result.is_ok());
      }));
}

void MessagesManager::on_save_dialog_to_database(DialogId dialog_id, bool can_reuse_notification_group, bool success) {
  LOG(INFO) << "Successfully saved " << dialog_id << " to database";

  if (success && can_reuse_notification_group) {
    auto d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    try_reuse_notification_group(d->message_notification_group);
    try_reuse_notification_group(d->mention_notification_group);
  }

  // TODO erase some events from binlog
}

void MessagesManager::try_reuse_notification_group(NotificationGroupInfo &group_info) {
  if (!group_info.try_reuse) {
    return;
  }
  if (group_info.is_changed) {
    LOG(ERROR) << "Failed to reuse changed " << group_info.group_id;
    return;
  }
  group_info.try_reuse = false;
  if (!group_info.group_id.is_valid()) {
    LOG(ERROR) << "Failed to reuse invalid " << group_info.group_id;
    return;
  }
  CHECK(group_info.last_notification_id == NotificationId());
  CHECK(group_info.last_notification_date == 0);
  send_closure_later(G()->notification_manager(), &NotificationManager::try_reuse_notification_group_id,
                     group_info.group_id);
  notification_group_id_to_dialog_id_.erase(group_info.group_id);
  group_info.group_id = NotificationGroupId();
  group_info.max_removed_notification_id = NotificationId();
  group_info.max_removed_message_id = MessageId();
}

void MessagesManager::update_message_count_by_index(Dialog *d, int diff, const Message *m) {
  auto index_mask = get_message_index_mask(d->dialog_id, m);
  index_mask &= ~search_messages_filter_index_mask(
      SearchMessagesFilter::UnreadMention);  // unread mention count has been already manually updated

  update_message_count_by_index(d, diff, index_mask);
}

void MessagesManager::update_message_count_by_index(Dialog *d, int diff, int32 index_mask) {
  if (index_mask == 0) {
    return;
  }

  LOG(INFO) << "Update message count by " << diff << " in index mask " << index_mask;
  int i = 0;
  for (auto &message_count : d->message_count_by_index) {
    if (((index_mask >> i) & 1) != 0 && message_count != -1) {
      message_count += diff;
      if (message_count < 0) {
        if (d->dialog_id.get_type() == DialogType::SecretChat) {
          message_count = 0;
        } else {
          message_count = -1;
        }
      }
      on_dialog_updated(d->dialog_id, "update_message_count_by_index");
    }
    i++;
  }

  i = static_cast<int>(SearchMessagesFilter::Call) - 1;
  for (auto &message_count : calls_db_state_.message_count_by_index) {
    if (((index_mask >> i) & 1) != 0 && message_count != -1) {
      message_count += diff;
      if (message_count < 0) {
        if (d->dialog_id.get_type() == DialogType::SecretChat) {
          message_count = 0;
        } else {
          message_count = -1;
        }
      }
      save_calls_db_state();
    }
    i++;
  }
}

int32 MessagesManager::get_message_index_mask(DialogId dialog_id, const Message *m) const {
  CHECK(m != nullptr);
  if (m->message_id.is_scheduled() || m->message_id.is_yet_unsent() || m->is_failed_to_send) {
    return 0;
  }
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  if (!m->message_id.is_server() && !is_secret) {
    return 0;
  }
  // retain second condition just in case
  if (m->is_content_secret || (m->ttl > 0 && !is_secret)) {
    return 0;
  }
  int32 mentions_mask = get_message_content_index_mask(m->content.get(), td_, is_secret, m->is_outgoing);
  if (m->contains_mention) {
    mentions_mask |= search_messages_filter_index_mask(SearchMessagesFilter::Mention);
    if (m->contains_unread_mention) {
      mentions_mask |= search_messages_filter_index_mask(SearchMessagesFilter::UnreadMention);
    }
  }
  LOG(INFO) << "Have index mask " << mentions_mask << " for " << m->message_id << " in " << dialog_id;
  return mentions_mask;
}

vector<MessageId> MessagesManager::get_message_ids(const vector<int64> &input_message_ids) {
  vector<MessageId> message_ids;
  message_ids.reserve(input_message_ids.size());
  for (auto &input_message_id : input_message_ids) {
    message_ids.push_back(MessageId(input_message_id));
  }
  return message_ids;
}

vector<int32> MessagesManager::get_server_message_ids(const vector<MessageId> &message_ids) {
  return transform(message_ids, [](MessageId message_id) { return message_id.get_server_message_id().get(); });
}

vector<int32> MessagesManager::get_scheduled_server_message_ids(const vector<MessageId> &message_ids) {
  return transform(message_ids,
                   [](MessageId message_id) { return message_id.get_scheduled_server_message_id().get(); });
}

tl_object_ptr<telegram_api::InputMessage> MessagesManager::get_input_message(MessageId message_id) {
  return make_tl_object<telegram_api::inputMessageID>(message_id.get_server_message_id().get());
}

tl_object_ptr<telegram_api::InputPeer> MessagesManager::get_input_peer(DialogId dialog_id,
                                                                       AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->get_input_peer_user(user_id, access_rights);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->get_input_peer_chat(chat_id, access_rights);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->get_input_peer_channel(channel_id, access_rights);
    }
    case DialogType::SecretChat:
      return nullptr;
    case DialogType::None:
      return make_tl_object<telegram_api::inputPeerEmpty>();
    default:
      UNREACHABLE();
      return nullptr;
  }
}

vector<tl_object_ptr<telegram_api::InputPeer>> MessagesManager::get_input_peers(const vector<DialogId> &dialog_ids,
                                                                                AccessRights access_rights) const {
  vector<tl_object_ptr<telegram_api::InputPeer>> input_peers;
  input_peers.reserve(dialog_ids.size());
  for (auto &dialog_id : dialog_ids) {
    auto input_peer = get_input_peer(dialog_id, access_rights);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Have no access to " << dialog_id;
      continue;
    }
    input_peers.push_back(std::move(input_peer));
  }
  return input_peers;
}

tl_object_ptr<telegram_api::InputDialogPeer> MessagesManager::get_input_dialog_peer(DialogId dialog_id,
                                                                                    AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
    case DialogType::None:
      return make_tl_object<telegram_api::inputDialogPeer>(get_input_peer(dialog_id, access_rights));
    case DialogType::SecretChat:
      return nullptr;
    default:
      UNREACHABLE();
      return nullptr;
  }
}

vector<tl_object_ptr<telegram_api::InputDialogPeer>> MessagesManager::get_input_dialog_peers(
    const vector<DialogId> &dialog_ids, AccessRights access_rights) const {
  vector<tl_object_ptr<telegram_api::InputDialogPeer>> input_dialog_peers;
  input_dialog_peers.reserve(dialog_ids.size());
  for (auto &dialog_id : dialog_ids) {
    auto input_dialog_peer = get_input_dialog_peer(dialog_id, access_rights);
    if (input_dialog_peer == nullptr) {
      LOG(ERROR) << "Have no access to " << dialog_id;
      continue;
    }
    input_dialog_peers.push_back(std::move(input_dialog_peer));
  }
  return input_dialog_peers;
}

bool MessagesManager::have_input_peer(DialogId dialog_id, AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->have_input_peer_user(user_id, access_rights);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->have_input_peer_chat(chat_id, access_rights);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->have_input_peer_channel(channel_id, access_rights);
    }
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->have_input_encrypted_peer(secret_chat_id, access_rights);
    }
    case DialogType::None:
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

bool MessagesManager::have_dialog_info(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->have_user(user_id);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->have_chat(chat_id);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->have_channel(channel_id);
    }
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->have_secret_chat(secret_chat_id);
    }
    case DialogType::None:
    default:
      return false;
  }
}

bool MessagesManager::have_dialog_info_force(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      UserId user_id = dialog_id.get_user_id();
      return td_->contacts_manager_->have_user_force(user_id);
    }
    case DialogType::Chat: {
      ChatId chat_id = dialog_id.get_chat_id();
      return td_->contacts_manager_->have_chat_force(chat_id);
    }
    case DialogType::Channel: {
      ChannelId channel_id = dialog_id.get_channel_id();
      return td_->contacts_manager_->have_channel_force(channel_id);
    }
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->have_secret_chat_force(secret_chat_id);
    }
    case DialogType::None:
    default:
      return false;
  }
}

tl_object_ptr<telegram_api::inputEncryptedChat> MessagesManager::get_input_encrypted_chat(
    DialogId dialog_id, AccessRights access_rights) const {
  switch (dialog_id.get_type()) {
    case DialogType::SecretChat: {
      SecretChatId secret_chat_id = dialog_id.get_secret_chat_id();
      return td_->contacts_manager_->get_input_encrypted_chat(secret_chat_id, access_rights);
    }
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
    case DialogType::None:
    default:
      UNREACHABLE();
      return nullptr;
  }
}

bool MessagesManager::is_allowed_useless_update(const tl_object_ptr<telegram_api::Update> &update) const {
  auto constructor_id = update->get_id();
  if (constructor_id == dummyUpdate::ID) {
    // allow dummyUpdate just in case
    return true;
  }
  if (constructor_id == telegram_api::updateNewMessage::ID ||
      constructor_id == telegram_api::updateNewChannelMessage::ID) {
    // new outgoing messages are received again if random_id coincide
    return true;
  }

  return false;
}

bool MessagesManager::check_update_dialog_id(const tl_object_ptr<telegram_api::Update> &update, DialogId dialog_id) {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      return true;
    case DialogType::Channel:
    case DialogType::SecretChat:
    case DialogType::None:
      LOG(ERROR) << "Receive update in wrong " << dialog_id << ": " << oneline(to_string(update));
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::skip_old_pending_update(tl_object_ptr<telegram_api::Update> &&update, int32 new_pts,
                                              int32 old_pts, int32 pts_count, const char *source) {
  if (update->get_id() == telegram_api::updateNewMessage::ID) {
    auto update_new_message = static_cast<telegram_api::updateNewMessage *>(update.get());
    auto full_message_id = get_full_message_id(update_new_message->message_, false);
    if (update_message_ids_.find(full_message_id) != update_message_ids_.end()) {
      if (new_pts == old_pts) {  // otherwise message can be already deleted
        // apply sent message anyway
        on_get_message(std::move(update_new_message->message_), true, false, false, true, true,
                       "updateNewMessage with an awaited message");
        return;
      } else {
        LOG(ERROR) << "Receive awaited sent " << full_message_id << " from " << source << " with pts " << new_pts
                   << " and pts_count " << pts_count << ", but current pts is " << old_pts;
        dump_debug_message_op(get_dialog(full_message_id.get_dialog_id()), 3);
      }
    }
  }
  if (update->get_id() == updateSentMessage::ID) {
    auto update_sent_message = static_cast<updateSentMessage *>(update.get());
    if (being_sent_messages_.count(update_sent_message->random_id_) > 0) {
      if (new_pts == old_pts) {  // otherwise message can be already deleted
        // apply sent message anyway
        on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                                update_sent_message->date_, FileId(), "process old updateSentMessage");
        return;
      } else {
        LOG(ERROR) << "Receive awaited sent " << update_sent_message->message_id_ << " from " << source << " with pts "
                   << new_pts << " and pts_count " << pts_count << ", but current pts is " << old_pts;
        dump_debug_message_op(get_dialog(being_sent_messages_[update_sent_message->random_id_].get_dialog_id()), 3);
      }
    }
    return;
  }

  // very old or unuseful update
  LOG_IF(WARNING, new_pts == old_pts && pts_count == 0 && !is_allowed_useless_update(update))
      << "Receive useless update " << oneline(to_string(update)) << " from " << source;
}

void MessagesManager::add_pending_update(tl_object_ptr<telegram_api::Update> &&update, int32 new_pts, int32 pts_count,
                                         bool force_apply, const char *source) {
  // do not try to run getDifference from this function
  CHECK(update != nullptr);
  CHECK(source != nullptr);
  LOG(INFO) << "Receive from " << source << " pending " << to_string(update) << "new_pts = " << new_pts
            << ", pts_count = " << pts_count << ", force_apply = " << force_apply;
  if (pts_count < 0 || new_pts <= pts_count) {
    LOG(ERROR) << "Receive update with wrong pts = " << new_pts << " or pts_count = " << pts_count << " from " << source
               << ": " << oneline(to_string(update));
    return;
  }

  // TODO need to save all updates that can change result of running queries not associated with pts (for example
  // getHistory) and apply them to result of this queries

  switch (update->get_id()) {
    case dummyUpdate::ID:
    case updateSentMessage::ID:
    case telegram_api::updateReadMessagesContents::ID:
    case telegram_api::updateDeleteMessages::ID:
      // nothing to check
      break;
    case telegram_api::updateNewMessage::ID: {
      auto update_new_message = static_cast<const telegram_api::updateNewMessage *>(update.get());
      DialogId dialog_id = get_message_dialog_id(update_new_message->message_);
      if (!check_update_dialog_id(update, dialog_id)) {
        return;
      }
      break;
    }
    case telegram_api::updateReadHistoryInbox::ID: {
      auto update_read_history_inbox = static_cast<const telegram_api::updateReadHistoryInbox *>(update.get());
      auto dialog_id = DialogId(update_read_history_inbox->peer_);
      if (!check_update_dialog_id(update, dialog_id)) {
        return;
      }
      break;
    }
    case telegram_api::updateReadHistoryOutbox::ID: {
      auto update_read_history_outbox = static_cast<const telegram_api::updateReadHistoryOutbox *>(update.get());
      auto dialog_id = DialogId(update_read_history_outbox->peer_);
      if (!check_update_dialog_id(update, dialog_id)) {
        return;
      }
      break;
    }
    case telegram_api::updateEditMessage::ID: {
      auto update_edit_message = static_cast<const telegram_api::updateEditMessage *>(update.get());
      DialogId dialog_id = get_message_dialog_id(update_edit_message->message_);
      if (!check_update_dialog_id(update, dialog_id)) {
        return;
      }
      break;
    }
    default:
      LOG(ERROR) << "Receive unexpected update " << oneline(to_string(update)) << "from " << source;
      return;
  }

  if (force_apply) {
    CHECK(pending_updates_.empty());
    CHECK(accumulated_pts_ == -1);
    if (pts_count != 0) {
      LOG(ERROR) << "Receive forced update with pts_count = " << pts_count << " from " << source;
    }

    process_update(std::move(update));
    return;
  }
  if (DROP_UPDATES) {
    return set_get_difference_timeout(1.0);
  }

  int32 old_pts = td_->updates_manager_->get_pts();
  if (new_pts < old_pts - 19999) {
    // restore pts after delete_first_messages
    LOG(ERROR) << "Restore pts after delete_first_messages from " << old_pts << " to " << new_pts
               << " is temporarily disabled, pts_count = " << pts_count << ", update is from " << source << ": "
               << oneline(to_string(update));
    if (old_pts < 10000000 && update->get_id() == telegram_api::updateNewMessage::ID) {
      auto update_new_message = static_cast<telegram_api::updateNewMessage *>(update.get());
      auto dialog_id = get_message_dialog_id(update_new_message->message_);
      dump_debug_message_op(get_dialog(dialog_id), 6);
    }
    set_get_difference_timeout(0.001);

    /*
    LOG(WARNING) << "Restore pts after delete_first_messages";
    td_->updates_manager_->set_pts(new_pts - 1, "restore pts after delete_first_messages");
    old_pts = td_->updates_manager_->get_pts();
    CHECK(old_pts == new_pts - 1);
    */
  }

  if (new_pts <= old_pts) {
    skip_old_pending_update(std::move(update), new_pts, old_pts, pts_count, source);
    return;
  }

  if (td_->updates_manager_->running_get_difference()) {
    LOG(INFO) << "Save pending update got while running getDifference from " << source;
    CHECK(update->get_id() == dummyUpdate::ID || update->get_id() == updateSentMessage::ID);
    if (pts_count > 0) {
      postponed_pts_updates_.emplace(new_pts, PendingPtsUpdate(std::move(update), new_pts, pts_count));
    }
    return;
  }

  if (old_pts + pts_count > new_pts) {
    LOG(WARNING) << "Have old_pts (= " << old_pts << ") + pts_count (= " << pts_count << ") > new_pts (= " << new_pts
                 << "). Logged in " << G()->shared_config().get_option_integer("authorization_date") << ". Update from "
                 << source << " = " << oneline(to_string(update));
    set_get_difference_timeout(0.001);
    return;
  }

  accumulated_pts_count_ += pts_count;
  if (new_pts > accumulated_pts_) {
    accumulated_pts_ = new_pts;
  }

  if (old_pts + accumulated_pts_count_ > accumulated_pts_) {
    LOG(WARNING) << "Have old_pts (= " << old_pts << ") + accumulated_pts_count (= " << accumulated_pts_count_
                 << ") > accumulated_pts (= " << accumulated_pts_ << "). new_pts = " << new_pts
                 << ", pts_count = " << pts_count << ". Logged in "
                 << G()->shared_config().get_option_integer("authorization_date") << ". Update from " << source << " = "
                 << oneline(to_string(update));
    set_get_difference_timeout(0.001);
    return;
  }

  LOG_IF(INFO, pts_count == 0 && update->get_id() != dummyUpdate::ID) << "Skip useless update " << to_string(update);

  if (pending_updates_.empty() && old_pts + accumulated_pts_count_ == accumulated_pts_ &&
      !pts_gap_timeout_.has_timeout()) {
    if (pts_count > 0) {
      process_update(std::move(update));

      td_->updates_manager_->set_pts(accumulated_pts_, "process pending updates fast path")
          .set_value(Unit());  // TODO can't set until get messages really stored on persistent storage
      accumulated_pts_count_ = 0;
      accumulated_pts_ = -1;
    }
    return;
  }

  if (pts_count > 0) {
    pending_updates_.emplace(new_pts, PendingPtsUpdate(std::move(update), new_pts, pts_count));
  }

  if (old_pts + accumulated_pts_count_ < accumulated_pts_) {
    set_get_difference_timeout(UpdatesManager::MAX_UNFILLED_GAP_TIME);
    return;
  }

  CHECK(old_pts + accumulated_pts_count_ == accumulated_pts_);
  if (!pending_updates_.empty()) {
    process_pending_updates();
  }
}

MessagesManager::Dialog *MessagesManager::get_service_notifications_dialog() {
  UserId service_notifications_user_id = td_->contacts_manager_->get_service_notifications_user_id();
  DialogId service_notifications_dialog_id(service_notifications_user_id);
  force_create_dialog(service_notifications_dialog_id, "get_service_notifications_dialog");
  return get_dialog(service_notifications_dialog_id);
}

void MessagesManager::on_update_service_notification(tl_object_ptr<telegram_api::updateServiceNotification> &&update,
                                                     bool skip_new_entities, Promise<Unit> &&promise) {
  int32 ttl = 0;
  bool has_date = (update->flags_ & telegram_api::updateServiceNotification::INBOX_DATE_MASK) != 0;
  auto date = has_date ? update->inbox_date_ : G()->unix_time();
  auto message_text =
      get_message_text(td_->contacts_manager_.get(), std::move(update->message_), std::move(update->entities_),
                       skip_new_entities, date, false, "on_update_service_notification");
  auto content = get_message_content(
      td_, std::move(message_text), std::move(update->media_),
      td_->auth_manager_->is_bot() ? DialogId() : get_service_notifications_dialog()->dialog_id, false, UserId(), &ttl);
  bool is_content_secret = is_secret_message_content(ttl, content->get_type());
  if ((update->flags_ & telegram_api::updateServiceNotification::POPUP_MASK) != 0) {
    send_closure(G()->td(), &Td::send_update,
                 td_api::make_object<td_api::updateServiceNotification>(
                     update->type_, get_message_content_object(content.get(), td_, date, is_content_secret)));
  }
  if (has_date && !td_->auth_manager_->is_bot()) {
    Dialog *d = get_service_notifications_dialog();
    CHECK(d != nullptr);
    auto dialog_id = d->dialog_id;
    CHECK(dialog_id.get_type() == DialogType::User);
    auto new_message = make_unique<Message>();
    set_message_id(new_message, get_next_local_message_id(d));
    new_message->sender_user_id = dialog_id.get_user_id();
    new_message->date = date;
    new_message->ttl = ttl;
    new_message->is_content_secret = is_content_secret;
    new_message->content = std::move(content);
    new_message->have_previous = true;
    new_message->have_next = true;

    bool need_update = true;
    bool need_update_dialog_pos = false;

    const Message *m = add_message_to_dialog(d, std::move(new_message), true, &need_update, &need_update_dialog_pos,
                                             "on_update_service_notification");
    if (m != nullptr && need_update) {
      send_update_new_message(d, m);
    }

    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "on_update_service_notification");
    }
  }
  promise.set_value(Unit());
}

void MessagesManager::on_update_new_channel_message(tl_object_ptr<telegram_api::updateNewChannelMessage> &&update) {
  int new_pts = update->pts_;
  int pts_count = update->pts_count_;
  DialogId dialog_id = get_message_dialog_id(update->message_);
  switch (dialog_id.get_type()) {
    case DialogType::None:
      return;
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::SecretChat:
      LOG(ERROR) << "Receive updateNewChannelMessage in wrong " << dialog_id;
      return;
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      if (!td_->contacts_manager_->have_channel(channel_id)) {
        // if min channel was received
        if (td_->contacts_manager_->have_min_channel(channel_id)) {
          td_->updates_manager_->schedule_get_difference("on_update_new_channel_message");
          return;
        }
      }
      // Ok
      break;
    }
    default:
      UNREACHABLE();
      return;
  }

  if (pts_count < 0 || new_pts <= pts_count) {
    LOG(ERROR) << "Receive new channel message with wrong pts = " << new_pts << " or pts_count = " << pts_count << ": "
               << oneline(to_string(update));
    return;
  }

  add_pending_channel_update(dialog_id, std::move(update), new_pts, pts_count, "on_update_new_channel_message");
}

void MessagesManager::on_update_edit_channel_message(tl_object_ptr<telegram_api::updateEditChannelMessage> &&update) {
  int new_pts = update->pts_;
  int pts_count = update->pts_count_;
  DialogId dialog_id = get_message_dialog_id(update->message_);
  switch (dialog_id.get_type()) {
    case DialogType::None:
      return;
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::SecretChat:
      LOG(ERROR) << "Receive updateEditChannelMessage in wrong " << dialog_id;
      return;
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      if (!td_->contacts_manager_->have_channel(channel_id)) {
        // if min channel was received
        if (td_->contacts_manager_->have_min_channel(channel_id)) {
          td_->updates_manager_->schedule_get_difference("on_update_edit_channel_message");
          return;
        }
      }
      // Ok
      break;
    }
    default:
      UNREACHABLE();
      return;
  }

  if (pts_count < 0 || new_pts <= pts_count) {
    LOG(ERROR) << "Receive edited channel message with wrong pts = " << new_pts << " or pts_count = " << pts_count
               << ": " << oneline(to_string(update));
    return;
  }

  add_pending_channel_update(dialog_id, std::move(update), new_pts, pts_count, "on_update_edit_channel_message");
}

void MessagesManager::on_update_read_channel_inbox(tl_object_ptr<telegram_api::updateReadChannelInbox> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateReadChannelInbox";
    return;
  }

  FolderId folder_id;
  if ((update->flags_ & telegram_api::updateReadChannelInbox::FOLDER_ID_MASK) != 0) {
    folder_id = FolderId(update->folder_id_);
  }
  on_update_dialog_folder_id(DialogId(channel_id), folder_id);
  on_read_channel_inbox(channel_id, MessageId(ServerMessageId(update->max_id_)), update->still_unread_count_,
                        update->pts_, "updateReadChannelInbox");
}

void MessagesManager::on_update_read_channel_outbox(tl_object_ptr<telegram_api::updateReadChannelOutbox> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateReadChannelOutbox";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);
  read_history_outbox(dialog_id, MessageId(ServerMessageId(update->max_id_)));
}

void MessagesManager::on_update_read_channel_messages_contents(
    tl_object_ptr<telegram_api::updateChannelReadMessagesContents> &&update) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateChannelReadMessagesContents";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Receive read channel messages contents update in unknown " << dialog_id;
    return;
  }

  for (auto &server_message_id : update->messages_) {
    read_channel_message_content_from_updates(d, MessageId(ServerMessageId(server_message_id)));
  }
}

void MessagesManager::on_update_channel_too_long(tl_object_ptr<telegram_api::updateChannelTooLong> &&update,
                                                 bool force_apply) {
  ChannelId channel_id(update->channel_id_);
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive invalid " << channel_id << " in updateChannelTooLong";
    return;
  }

  DialogId dialog_id = DialogId(channel_id);
  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    auto pts = load_channel_pts(dialog_id);
    if (pts > 0) {
      d = add_dialog(dialog_id);
      CHECK(d != nullptr);
      CHECK(d->pts == pts);
      update_dialog_pos(d, false, "on_update_channel_too_long");
    }
  }

  int32 update_pts = (update->flags_ & UPDATE_CHANNEL_TO_LONG_FLAG_HAS_PTS) ? update->pts_ : 0;

  if (d != nullptr) {
    if (update_pts == 0 || update_pts > d->pts) {
      get_channel_difference(dialog_id, d->pts, true, "on_update_channel_too_long 1");
    }
  } else {
    if (force_apply) {
      get_channel_difference(dialog_id, -1, true, "on_update_channel_too_long 2");
    } else {
      td_->updates_manager_->schedule_get_difference("on_update_channel_too_long 3");
    }
  }
}

void MessagesManager::on_update_message_views(FullMessageId full_message_id, int32 views) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Ignore updateMessageViews in unknown " << dialog_id;
    return;
  }
  auto message_id = full_message_id.get_message_id();
  Message *m = get_message_force(d, message_id, "on_update_message_views");
  if (m == nullptr) {
    LOG(INFO) << "Ignore updateMessageViews about unknown " << full_message_id;
    if (!message_id.is_scheduled() && message_id > d->last_new_message_id &&
        dialog_id.get_type() == DialogType::Channel) {
      get_channel_difference(dialog_id, d->pts, true, "on_update_message_views");
    }
    return;
  }

  if (update_message_views(dialog_id, m, views)) {
    on_message_changed(d, m, true, "on_update_message_views");
  }
}

void MessagesManager::on_pending_message_views_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  const size_t MAX_MESSAGE_VIEWS = 100;  // server side limit
  vector<MessageId> message_ids;
  message_ids.reserve(min(d->pending_viewed_message_ids.size(), MAX_MESSAGE_VIEWS));
  for (auto message_id : d->pending_viewed_message_ids) {
    message_ids.push_back(message_id);
    if (message_ids.size() >= MAX_MESSAGE_VIEWS) {
      td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(message_ids), d->increment_view_counter);
      message_ids.clear();
    }
  }
  if (!message_ids.empty()) {
    td_->create_handler<GetMessagesViewsQuery>()->send(dialog_id, std::move(message_ids), d->increment_view_counter);
  }
  d->pending_viewed_message_ids.clear();
  d->increment_view_counter = false;
}

bool MessagesManager::update_message_views(DialogId dialog_id, Message *m, int32 views) {
  CHECK(m != nullptr);
  if (views > m->views) {
    LOG(DEBUG) << "Update views of " << FullMessageId{dialog_id, m->message_id} << " from " << m->views << " to "
               << views;
    m->views = views;
    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateMessageViews>(dialog_id.get(), m->message_id.get(), m->views));
    return true;
  }
  return false;
}

void MessagesManager::on_update_live_location_viewed(FullMessageId full_message_id) {
  LOG(DEBUG) << "Live location was viewed in " << full_message_id;
  if (!are_active_live_location_messages_loaded_) {
    get_active_live_location_messages(PromiseCreator::lambda([actor_id = actor_id(this), full_message_id](Unit result) {
      send_closure(actor_id, &MessagesManager::on_update_live_location_viewed, full_message_id);
    }));
    return;
  }

  auto active_live_location_message_ids = get_active_live_location_messages(Auto());
  if (!td::contains(active_live_location_message_ids, full_message_id)) {
    LOG(DEBUG) << "Can't find " << full_message_id << " in " << active_live_location_message_ids;
    return;
  }

  send_update_message_live_location_viewed(full_message_id);
}

void MessagesManager::on_update_some_live_location_viewed(Promise<Unit> &&promise) {
  LOG(DEBUG) << "Some live location was viewed";
  if (!are_active_live_location_messages_loaded_) {
    get_active_live_location_messages(
        PromiseCreator::lambda([actor_id = actor_id(this), promise = std::move(promise)](Unit result) mutable {
          send_closure(actor_id, &MessagesManager::on_update_some_live_location_viewed, std::move(promise));
        }));
    return;
  }

  // update all live locations, because it is unknown, which exactly was viewed
  auto active_live_location_message_ids = get_active_live_location_messages(Auto());
  for (auto full_message_id : active_live_location_message_ids) {
    send_update_message_live_location_viewed(full_message_id);
  }

  promise.set_value(Unit());
}

void MessagesManager::on_external_update_message_content(FullMessageId full_message_id) {
  const Dialog *d = get_dialog(full_message_id.get_dialog_id());
  CHECK(d != nullptr);
  const Message *m = get_message(d, full_message_id.get_message_id());
  CHECK(m != nullptr);
  auto live_location_date = m->is_failed_to_send ? 0 : m->date;
  send_update_message_content(full_message_id.get_dialog_id(), m->message_id, m->content.get(), live_location_date,
                              m->is_content_secret, "on_external_update_message_content");
  if (m->message_id == d->last_message_id) {
    send_update_chat_last_message_impl(d, "on_external_update_message_content");
  }
}

bool MessagesManager::update_message_contains_unread_mention(Dialog *d, Message *m, bool contains_unread_mention,
                                                             const char *source) {
  LOG_CHECK(m != nullptr) << source;
  CHECK(!m->message_id.is_scheduled());
  if (!contains_unread_mention && m->contains_unread_mention) {
    remove_message_notification_id(d, m, true, true);  // should be called before contains_unread_mention is updated

    m->contains_unread_mention = false;
    if (d->unread_mention_count == 0) {
      if (d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] != -1) {
        LOG(ERROR) << "Unread mention count of " << d->dialog_id << " became negative from " << source;
      }
    } else {
      d->unread_mention_count--;
      d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] =
          d->unread_mention_count;
      on_dialog_updated(d->dialog_id, "update_message_contains_unread_mention");
    }
    LOG(INFO) << "Update unread mention message count in " << d->dialog_id << " to " << d->unread_mention_count
              << " by reading " << m->message_id << " from " << source;

    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateMessageMentionRead>(d->dialog_id.get(), m->message_id.get(),
                                                                  d->unread_mention_count));
    return true;
  }
  return false;
}

void MessagesManager::on_read_channel_inbox(ChannelId channel_id, MessageId max_message_id, int32 server_unread_count,
                                            int32 pts, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!max_message_id.is_scheduled());
  if (!max_message_id.is_valid() && server_unread_count <= 0) {
    return;
  }

  DialogId dialog_id(channel_id);
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Receive read inbox in unknown " << dialog_id << " from " << source;
    return;
  }

  /*
  // dropping unread count can make things worse, so don't drop it
  if (server_unread_count > 0 && G()->parameters().use_message_db && d->is_last_read_inbox_message_id_inited) {
    server_unread_count = -1;
  }
  */

  if (d->pts == pts) {
    read_history_inbox(dialog_id, max_message_id, server_unread_count, source);
  } else if (d->pts > pts) {
    // outdated update, need to repair server_unread_count from the server
    repair_channel_server_unread_count(d);
  } else {
    // update from the future, keep it until it can be applied
    if (pts >= d->pending_read_channel_inbox_pts) {
      d->pending_read_channel_inbox_pts = pts;
      d->pending_read_channel_inbox_max_message_id = max_message_id;
      d->pending_read_channel_inbox_server_unread_count = server_unread_count;
      on_dialog_updated(dialog_id, "on_read_channel_inbox");
    }
  }
}

void MessagesManager::on_read_channel_outbox(ChannelId channel_id, MessageId max_message_id) {
  DialogId dialog_id(channel_id);
  CHECK(!max_message_id.is_scheduled());
  if (max_message_id.is_valid()) {
    read_history_outbox(dialog_id, max_message_id);
  }
}

void MessagesManager::on_update_channel_max_unavailable_message_id(ChannelId channel_id,
                                                                   MessageId max_unavailable_message_id) {
  if (!channel_id.is_valid()) {
    LOG(ERROR) << "Receive max_unavailable_message_id in invalid " << channel_id;
    return;
  }

  DialogId dialog_id(channel_id);
  CHECK(!max_unavailable_message_id.is_scheduled());
  if (!max_unavailable_message_id.is_valid() && max_unavailable_message_id != MessageId()) {
    LOG(ERROR) << "Receive wrong max_unavailable_message_id: " << max_unavailable_message_id;
    max_unavailable_message_id = MessageId();
  }
  set_dialog_max_unavailable_message_id(dialog_id, max_unavailable_message_id, true,
                                        "on_update_channel_max_unavailable_message_id");
}

void MessagesManager::on_update_dialog_online_member_count(DialogId dialog_id, int32 online_member_count,
                                                           bool is_from_server) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive online member count in invalid " << dialog_id;
    return;
  }

  if (is_broadcast_channel(dialog_id)) {
    LOG_IF(ERROR, online_member_count != 0)
        << "Receive online member count " << online_member_count << " in a broadcast " << dialog_id;
    return;
  }

  if (online_member_count < 0) {
    LOG(ERROR) << "Receive online member count " << online_member_count << " in a " << dialog_id;
    return;
  }

  set_dialog_online_member_count(dialog_id, online_member_count, is_from_server,
                                 "on_update_channel_online_member_count");
}

void MessagesManager::on_update_delete_scheduled_messages(DialogId dialog_id,
                                                          vector<ScheduledServerMessageId> &&server_message_ids) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive deleted scheduled messages in invalid " << dialog_id;
    return;
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Skip updateDeleteScheduledMessages in unknown " << dialog_id;
    return;
  }

  vector<int64> deleted_message_ids;
  for (auto server_message_id : server_message_ids) {
    if (!server_message_id.is_valid()) {
      LOG(ERROR) << "Incoming update tries to delete scheduled message " << server_message_id.get();
      continue;
    }

    auto message = do_delete_scheduled_message(d, MessageId(server_message_id, std::numeric_limits<int32>::max()), true,
                                               "on_update_delete_scheduled_messages");
    if (message != nullptr) {
      deleted_message_ids.push_back(message->message_id.get());
    }
  }

  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  send_update_chat_has_scheduled_messages(d);
}

void MessagesManager::on_update_include_sponsored_dialog_to_unread_count() {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  bool include_sponsored_dialog = G()->shared_config().get_option_boolean("include_sponsored_chat_to_unread_count");
  if (include_sponsored_dialog_to_unread_count_ == include_sponsored_dialog) {
    return;
  }
  if (sponsored_dialog_id_.is_valid()) {
    // preload sponsored dialog
    get_dialog_force(sponsored_dialog_id_);
  }

  include_sponsored_dialog_to_unread_count_ = include_sponsored_dialog;

  if (!sponsored_dialog_id_.is_valid()) {
    // nothing has changed
    return;
  }
  if (!G()->parameters().use_message_db) {
    // there is no support for unread count updates without message database
    return;
  }

  auto folder_id = FolderId::main();
  auto &list = get_dialog_list(folder_id);
  const Dialog *d = get_dialog(sponsored_dialog_id_);
  CHECK(d != nullptr);
  auto unread_count = d->server_unread_count + d->local_unread_count;
  if (unread_count != 0 && list.is_message_unread_count_inited_) {
    send_update_unread_message_count(folder_id, d->dialog_id, true,
                                     "on_update_include_sponsored_dialog_to_unread_count");
  }
  if ((unread_count != 0 || d->is_marked_as_unread) && list.is_dialog_unread_count_inited_) {
    send_update_unread_chat_count(folder_id, d->dialog_id, true, "on_update_include_sponsored_dialog_to_unread_count");
  }
}

bool MessagesManager::need_cancel_user_dialog_action(int32 action_id, MessageContentType message_content_type) {
  if (message_content_type == MessageContentType::None) {
    return true;
  }

  if (action_id == td_api::chatActionTyping::ID) {
    return message_content_type == MessageContentType::Text || message_content_type == MessageContentType::Game ||
           can_have_message_content_caption(message_content_type);
  }

  switch (message_content_type) {
    case MessageContentType::Animation:
    case MessageContentType::Audio:
    case MessageContentType::Document:
      return action_id == td_api::chatActionUploadingDocument::ID;
    case MessageContentType::ExpiredPhoto:
    case MessageContentType::Photo:
      return action_id == td_api::chatActionUploadingPhoto::ID;
    case MessageContentType::ExpiredVideo:
    case MessageContentType::Video:
      return action_id == td_api::chatActionRecordingVideo::ID || action_id == td_api::chatActionUploadingVideo::ID;
    case MessageContentType::VideoNote:
      return action_id == td_api::chatActionRecordingVideoNote::ID ||
             action_id == td_api::chatActionUploadingVideoNote::ID;
    case MessageContentType::VoiceNote:
      return action_id == td_api::chatActionRecordingVoiceNote::ID ||
             action_id == td_api::chatActionUploadingVoiceNote::ID;
    case MessageContentType::Contact:
      return action_id == td_api::chatActionChoosingContact::ID;
    case MessageContentType::LiveLocation:
    case MessageContentType::Location:
    case MessageContentType::Venue:
      return action_id == td_api::chatActionChoosingLocation::ID;
    case MessageContentType::Game:
    case MessageContentType::Invoice:
    case MessageContentType::Sticker:
    case MessageContentType::Text:
    case MessageContentType::Unsupported:
    case MessageContentType::ChatCreate:
    case MessageContentType::ChatChangeTitle:
    case MessageContentType::ChatChangePhoto:
    case MessageContentType::ChatDeletePhoto:
    case MessageContentType::ChatDeleteHistory:
    case MessageContentType::ChatAddUsers:
    case MessageContentType::ChatJoinedByLink:
    case MessageContentType::ChatDeleteUser:
    case MessageContentType::ChatMigrateTo:
    case MessageContentType::ChannelCreate:
    case MessageContentType::ChannelMigrateFrom:
    case MessageContentType::PinMessage:
    case MessageContentType::GameScore:
    case MessageContentType::ScreenshotTaken:
    case MessageContentType::ChatSetTtl:
    case MessageContentType::Call:
    case MessageContentType::PaymentSuccessful:
    case MessageContentType::ContactRegistered:
    case MessageContentType::CustomServiceAction:
    case MessageContentType::WebsiteConnected:
    case MessageContentType::PassportDataSent:
    case MessageContentType::PassportDataReceived:
    case MessageContentType::Poll:
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::on_user_dialog_action(DialogId dialog_id, UserId user_id,
                                            tl_object_ptr<td_api::ChatAction> &&action, int32 date,
                                            MessageContentType message_content_type) {
  if (td_->auth_manager_->is_bot() || !user_id.is_valid() || is_broadcast_channel(dialog_id)) {
    return;
  }

  if (action != nullptr || message_content_type != MessageContentType::None) {
    td_->contacts_manager_->on_update_user_local_was_online(user_id, date);
  }

  bool is_canceled = action == nullptr || action->get_id() == td_api::chatActionCancel::ID;
  if (is_canceled) {
    auto actions_it = active_dialog_actions_.find(dialog_id);
    if (actions_it == active_dialog_actions_.end()) {
      return;
    }

    auto &active_actions = actions_it->second;
    auto it = std::find_if(active_actions.begin(), active_actions.end(),
                           [user_id](const ActiveDialogAction &action) { return action.user_id == user_id; });
    if (it == active_actions.end()) {
      return;
    }

    if (!td_->contacts_manager_->is_user_bot(user_id) &&
        !need_cancel_user_dialog_action(it->action_id, message_content_type)) {
      return;
    }

    LOG(DEBUG) << "Cancel action of " << user_id << " in " << dialog_id;
    active_actions.erase(it);
    if (active_actions.empty()) {
      active_dialog_actions_.erase(dialog_id);
      LOG(DEBUG) << "Cancel action timeout in " << dialog_id;
      active_dialog_action_timeout_.cancel_timeout(dialog_id.get());
    }
    if (action == nullptr) {
      action = make_tl_object<td_api::chatActionCancel>();
    }
  } else {
    if (date < G()->unix_time_cached() - DIALOG_ACTION_TIMEOUT - 60) {
      LOG(DEBUG) << "Ignore too old action of " << user_id << " in " << dialog_id << " sent at " << date;
      return;
    }
    auto &active_actions = active_dialog_actions_[dialog_id];
    auto it = std::find_if(active_actions.begin(), active_actions.end(),
                           [user_id](const ActiveDialogAction &action) { return action.user_id == user_id; });
    int32 prev_action_id = 0;
    int32 prev_progress = 0;
    if (it != active_actions.end()) {
      LOG(DEBUG) << "Re-add action of " << user_id << " in " << dialog_id;
      prev_action_id = it->action_id;
      prev_progress = it->progress;
      active_actions.erase(it);
    } else {
      LOG(DEBUG) << "Add action of " << user_id << " in " << dialog_id;
    }

    auto action_id = action->get_id();
    auto progress = [&] {
      switch (action_id) {
        case td_api::chatActionUploadingVideo::ID:
          return static_cast<td_api::chatActionUploadingVideo &>(*action).progress_;
        case td_api::chatActionUploadingVoiceNote::ID:
          return static_cast<td_api::chatActionUploadingVoiceNote &>(*action).progress_;
        case td_api::chatActionUploadingPhoto::ID:
          return static_cast<td_api::chatActionUploadingPhoto &>(*action).progress_;
        case td_api::chatActionUploadingDocument::ID:
          return static_cast<td_api::chatActionUploadingDocument &>(*action).progress_;
        case td_api::chatActionUploadingVideoNote::ID:
          return static_cast<td_api::chatActionUploadingVideoNote &>(*action).progress_;
        default:
          return 0;
      }
    }();
    active_actions.emplace_back(user_id, action_id, Time::now());
    if (action_id == prev_action_id && progress <= prev_progress) {
      return;
    }
    if (active_actions.size() == 1u) {
      LOG(DEBUG) << "Set action timeout in " << dialog_id;
      active_dialog_action_timeout_.set_timeout_in(dialog_id.get(), DIALOG_ACTION_TIMEOUT);
    }
  }

  LOG(DEBUG) << "Send action of " << user_id << " in " << dialog_id << ": " << to_string(action);
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateUserChatAction>(
                   dialog_id.get(), td_->contacts_manager_->get_user_id_object(user_id, "on_user_dialog_action"),
                   std::move(action)));
}

void MessagesManager::cancel_user_dialog_action(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  if (m->forward_info != nullptr || m->had_forward_info || m->via_bot_user_id.is_valid() || m->hide_via_bot ||
      m->is_channel_post || m->message_id.is_scheduled()) {
    return;
  }

  on_user_dialog_action(dialog_id, m->sender_user_id, nullptr, m->date, m->content->get_type());
}

void MessagesManager::add_pending_channel_update(DialogId dialog_id, tl_object_ptr<telegram_api::Update> &&update,
                                                 int32 new_pts, int32 pts_count, const char *source,
                                                 bool is_postponed_udpate) {
  LOG(INFO) << "Receive from " << source << " pending " << to_string(update);
  CHECK(update != nullptr);
  CHECK(dialog_id.get_type() == DialogType::Channel);
  if (pts_count < 0 || new_pts <= pts_count) {
    LOG(ERROR) << "Receive channel update from " << source << " with wrong pts = " << new_pts
               << " or pts_count = " << pts_count << ": " << oneline(to_string(update));
    return;
  }

  // TODO need to save all updates that can change result of running queries not associated with pts (for example
  // getHistory) and apply them to result of this queries

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    auto pts = load_channel_pts(dialog_id);
    if (pts > 0) {
      auto channel_id = dialog_id.get_channel_id();
      if (!td_->contacts_manager_->have_channel(channel_id)) {
        // do not create dialog if there is no info about the channel
        LOG(WARNING) << "There is no info about " << channel_id << ", so ignore " << to_string(update);
        return;
      }

      d = add_dialog(dialog_id);
      CHECK(d != nullptr);
      CHECK(d->pts == pts);
      update_dialog_pos(d, false, "add_pending_channel_update");
    }
  }
  if (d == nullptr) {
    // if there is no dialog, it can be created by the update
    LOG(INFO) << "Receive pending update from " << source << " about unknown " << dialog_id;
    if (running_get_channel_difference(dialog_id)) {
      return;
    }
  } else {
    int32 old_pts = d->pts;
    if (new_pts <= old_pts) {  // very old or unuseful update
      if (new_pts < old_pts - 19999 && !is_postponed_udpate) {
        // restore channel pts after delete_first_messages
        LOG(ERROR) << "Restore pts in " << d->dialog_id << " from " << source << " after delete_first_messages from "
                   << old_pts << " to " << new_pts << " is temporarily disabled, pts_count = " << pts_count
                   << ", update is from " << source << ": " << oneline(to_string(update));
        if (old_pts < 10000000) {
          dump_debug_message_op(d, 6);
        }
        get_channel_difference(dialog_id, old_pts, true, "add_pending_channel_update old");
      }

      if (update->get_id() == telegram_api::updateNewChannelMessage::ID) {
        auto update_new_channel_message = static_cast<telegram_api::updateNewChannelMessage *>(update.get());
        auto message_id = get_message_id(update_new_channel_message->message_, false);
        FullMessageId full_message_id(dialog_id, message_id);
        if (update_message_ids_.find(full_message_id) != update_message_ids_.end()) {
          // apply sent channel message
          on_get_message(std::move(update_new_channel_message->message_), true, true, false, true, true,
                         "updateNewChannelMessage with an awaited message");
          return;
        }
      }
      if (update->get_id() == updateSentMessage::ID) {
        auto update_sent_message = static_cast<updateSentMessage *>(update.get());
        if (being_sent_messages_.count(update_sent_message->random_id_) > 0) {
          // apply sent channel message
          on_send_message_success(update_sent_message->random_id_, update_sent_message->message_id_,
                                  update_sent_message->date_, FileId(), "process old updateSentChannelMessage");
          return;
        }
      }

      LOG_IF(WARNING, new_pts == old_pts && pts_count == 0)
          << "Receive from " << source << " useless channel update " << oneline(to_string(update));
      LOG(INFO) << "Skip already applied channel update";
      return;
    }

    if (running_get_channel_difference(dialog_id)) {
      if (pts_count > 0) {
        d->postponed_channel_updates.emplace(new_pts, PendingPtsUpdate(std::move(update), new_pts, pts_count));
      }
      LOG(INFO) << "Postpone channel update, because getChannelDifference is run";
      return;
    }

    if (old_pts + pts_count != new_pts) {
      LOG(INFO) << "Found a gap in the " << dialog_id << " with pts = " << old_pts << ". new_pts = " << new_pts
                << ", pts_count = " << pts_count << " in update from " << source;

      if (pts_count > 0) {
        d->postponed_channel_updates.emplace(new_pts, PendingPtsUpdate(std::move(update), new_pts, pts_count));
      }

      get_channel_difference(dialog_id, old_pts, true, "add_pending_channel_update pts mismatch");
      return;
    }
  }

  if (d == nullptr || pts_count > 0) {
    process_channel_update(std::move(update));
    LOG_CHECK(!running_get_channel_difference(dialog_id)) << '"' << active_get_channel_differencies_[dialog_id] << '"';
  } else {
    LOG_IF(INFO, update->get_id() != dummyUpdate::ID)
        << "Skip useless channel update from " << source << ": " << to_string(update);
  }

  if (d == nullptr) {
    d = get_dialog(dialog_id);
    if (d == nullptr) {
      LOG(INFO) << "Update didn't created " << dialog_id;
      return;
    }
  }

  CHECK(new_pts > d->pts);
  set_channel_pts(d, new_pts, source);
}

void MessagesManager::set_get_difference_timeout(double timeout) {
  if (!pts_gap_timeout_.has_timeout()) {
    LOG(INFO) << "Gap in pts has found, current pts is " << td_->updates_manager_->get_pts();
    pts_gap_timeout_.set_callback(std::move(UpdatesManager::fill_pts_gap));
    pts_gap_timeout_.set_callback_data(static_cast<void *>(td_));
    pts_gap_timeout_.set_timeout_in(timeout);
  }
}

void MessagesManager::process_update(tl_object_ptr<telegram_api::Update> &&update) {
  switch (update->get_id()) {
    case dummyUpdate::ID:
      LOG(INFO) << "Process dummyUpdate";
      break;
    case telegram_api::updateNewMessage::ID:
      LOG(INFO) << "Process updateNewMessage";
      on_get_message(std::move(move_tl_object_as<telegram_api::updateNewMessage>(update)->message_), true, false, false,
                     true, true, "updateNewMessage");
      break;
    case updateSentMessage::ID: {
      auto send_message_success_update = move_tl_object_as<updateSentMessage>(update);
      LOG(INFO) << "Process updateSentMessage " << send_message_success_update->random_id_;
      on_send_message_success(send_message_success_update->random_id_, send_message_success_update->message_id_,
                              send_message_success_update->date_, FileId(), "process updateSentMessage");
      break;
    }
    case telegram_api::updateReadMessagesContents::ID: {
      auto read_contents_update = move_tl_object_as<telegram_api::updateReadMessagesContents>(update);
      LOG(INFO) << "Process updateReadMessageContents";
      for (auto &message_id : read_contents_update->messages_) {
        read_message_content_from_updates(MessageId(ServerMessageId(message_id)));
      }
      break;
    }
    case telegram_api::updateEditMessage::ID: {
      auto full_message_id =
          on_get_message(std::move(move_tl_object_as<telegram_api::updateEditMessage>(update)->message_), false, false,
                         false, false, false, "updateEditMessage");
      LOG(INFO) << "Process updateEditMessage";
      on_message_edited(full_message_id);
      break;
    }
    case telegram_api::updateDeleteMessages::ID: {
      auto delete_update = move_tl_object_as<telegram_api::updateDeleteMessages>(update);
      LOG(INFO) << "Process updateDeleteMessages";
      vector<MessageId> message_ids;
      for (auto &message : delete_update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }
      delete_messages_from_updates(message_ids);
      break;
    }
    case telegram_api::updateReadHistoryInbox::ID: {
      auto read_update = move_tl_object_as<telegram_api::updateReadHistoryInbox>(update);
      LOG(INFO) << "Process updateReadHistoryInbox";
      DialogId dialog_id(read_update->peer_);
      FolderId folder_id;
      if ((read_update->flags_ & telegram_api::updateReadHistoryInbox::FOLDER_ID_MASK) != 0) {
        folder_id = FolderId(read_update->folder_id_);
      }
      on_update_dialog_folder_id(dialog_id, folder_id);
      read_history_inbox(dialog_id, MessageId(ServerMessageId(read_update->max_id_)),
                         -1 /*read_update->still_unread_count*/, "updateReadHistoryInbox");
      break;
    }
    case telegram_api::updateReadHistoryOutbox::ID: {
      auto read_update = move_tl_object_as<telegram_api::updateReadHistoryOutbox>(update);
      LOG(INFO) << "Process updateReadHistoryOutbox";
      read_history_outbox(DialogId(read_update->peer_), MessageId(ServerMessageId(read_update->max_id_)));
      break;
    }
    default:
      UNREACHABLE();
  }
  CHECK(!td_->updates_manager_->running_get_difference());
}

void MessagesManager::process_channel_update(tl_object_ptr<telegram_api::Update> &&update) {
  switch (update->get_id()) {
    case dummyUpdate::ID:
      LOG(INFO) << "Process dummyUpdate";
      break;
    case updateSentMessage::ID: {
      auto send_message_success_update = move_tl_object_as<updateSentMessage>(update);
      LOG(INFO) << "Process updateSentMessage " << send_message_success_update->random_id_;
      on_send_message_success(send_message_success_update->random_id_, send_message_success_update->message_id_,
                              send_message_success_update->date_, FileId(), "process updateSentChannelMessage");
      break;
    }
    case telegram_api::updateNewChannelMessage::ID:
      LOG(INFO) << "Process updateNewChannelMessage";
      on_get_message(std::move(move_tl_object_as<telegram_api::updateNewChannelMessage>(update)->message_), true, true,
                     false, true, true, "updateNewChannelMessage");
      break;
    case telegram_api::updateDeleteChannelMessages::ID: {
      auto delete_channel_messages_update = move_tl_object_as<telegram_api::updateDeleteChannelMessages>(update);
      LOG(INFO) << "Process updateDeleteChannelMessages";
      ChannelId channel_id(delete_channel_messages_update->channel_id_);
      if (!channel_id.is_valid()) {
        LOG(ERROR) << "Receive invalid " << channel_id;
        break;
      }

      vector<MessageId> message_ids;
      for (auto &message : delete_channel_messages_update->messages_) {
        message_ids.push_back(MessageId(ServerMessageId(message)));
      }

      auto dialog_id = DialogId(channel_id);
      delete_dialog_messages_from_updates(dialog_id, message_ids);
      break;
    }
    case telegram_api::updateEditChannelMessage::ID: {
      LOG(INFO) << "Process updateEditChannelMessage";
      auto full_message_id =
          on_get_message(std::move(move_tl_object_as<telegram_api::updateEditChannelMessage>(update)->message_), false,
                         true, false, false, false, "updateEditChannelMessage");
      on_message_edited(full_message_id);
      break;
    }
    default:
      UNREACHABLE();
  }
}

void MessagesManager::on_message_edited(FullMessageId full_message_id) {
  if (full_message_id == FullMessageId()) {
    return;
  }

  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  const Message *m = get_message(d, full_message_id.get_message_id());
  CHECK(m != nullptr);
  if (td_->auth_manager_->is_bot()) {
    d->last_edited_message_id = m->message_id;
    send_update_message_edited(dialog_id, m);
  }
  update_used_hashtags(dialog_id, m);
}

void MessagesManager::process_pending_updates() {
  for (auto &update : pending_updates_) {
    process_update(std::move(update.second.update));
  }

  td_->updates_manager_->set_pts(accumulated_pts_, "process pending updates")
      .set_value(Unit());  // TODO can't set until get messages really stored on persistent storage
  drop_pending_updates();
}

void MessagesManager::drop_pending_updates() {
  accumulated_pts_count_ = 0;
  accumulated_pts_ = -1;
  pts_gap_timeout_.cancel_timeout();
  pending_updates_.clear();
}

string MessagesManager::get_notification_settings_scope_database_key(NotificationSettingsScope scope) {
  switch (scope) {
    case NotificationSettingsScope::Private:
      return "nsfpc";
    case NotificationSettingsScope::Group:
      return "nsfgc";
    case NotificationSettingsScope::Channel:
      return "nsfcc";
    default:
      UNREACHABLE();
      return "";
  }
}

void MessagesManager::save_scope_notification_settings(NotificationSettingsScope scope,
                                                       const ScopeNotificationSettings &new_settings) {
  string key = get_notification_settings_scope_database_key(scope);
  G()->td_db()->get_binlog_pmc()->set(key, log_event_store(new_settings).as_slice().str());
}

bool MessagesManager::update_dialog_notification_settings(DialogId dialog_id,
                                                          DialogNotificationSettings *current_settings,
                                                          const DialogNotificationSettings &new_settings) {
  bool need_update_server = current_settings->mute_until != new_settings.mute_until ||
                            current_settings->sound != new_settings.sound ||
                            current_settings->show_preview != new_settings.show_preview ||
                            current_settings->use_default_mute_until != new_settings.use_default_mute_until ||
                            current_settings->use_default_sound != new_settings.use_default_sound ||
                            current_settings->use_default_show_preview != new_settings.use_default_show_preview;
  bool need_update_local =
      current_settings->use_default_disable_pinned_message_notifications !=
          new_settings.use_default_disable_pinned_message_notifications ||
      current_settings->disable_pinned_message_notifications != new_settings.disable_pinned_message_notifications ||
      current_settings->use_default_disable_mention_notifications !=
          new_settings.use_default_disable_mention_notifications ||
      current_settings->disable_mention_notifications != new_settings.disable_mention_notifications;

  bool is_changed = need_update_server || need_update_local ||
                    current_settings->is_synchronized != new_settings.is_synchronized ||
                    current_settings->is_use_default_fixed != new_settings.is_use_default_fixed;

  if (is_changed) {
    Dialog *d = get_dialog(dialog_id);
    LOG_CHECK(d != nullptr) << "Wrong " << dialog_id << " in update_dialog_notification_settings";
    bool was_muted = is_dialog_muted(d);
    bool was_dialog_mentions_disabled = is_dialog_mention_notifications_disabled(d);
    update_dialog_unmute_timeout(d, current_settings->use_default_mute_until, current_settings->mute_until,
                                 new_settings.use_default_mute_until, new_settings.mute_until);
    on_dialog_updated(dialog_id, "update_dialog_notification_settings");

    VLOG(notifications) << "Update notification settings in " << dialog_id << " from " << *current_settings << " to "
                        << new_settings;
    *current_settings = new_settings;

    if (!was_muted && is_dialog_muted(d)) {
      remove_all_dialog_notifications(d, false, "save_scope_notification_settings");
    }
    if (is_dialog_pinned_message_notifications_disabled(d) && d->mention_notification_group.group_id.is_valid() &&
        d->pinned_message_notification_message_id.is_valid()) {
      remove_dialog_pinned_message_notification(d);
    }
    if (was_dialog_mentions_disabled != is_dialog_mention_notifications_disabled(d)) {
      if (was_dialog_mentions_disabled) {
        update_dialog_mention_notification_count(d);
      } else {
        remove_dialog_mention_notifications(d);
      }
    }

    if (need_update_server || need_update_local) {
      send_closure(G()->td(), &Td::send_update,
                   make_tl_object<td_api::updateChatNotificationSettings>(
                       dialog_id.get(), get_chat_notification_settings_object(current_settings)));
    }
  }
  return need_update_server;
}

bool MessagesManager::update_scope_notification_settings(NotificationSettingsScope scope,
                                                         ScopeNotificationSettings *current_settings,
                                                         const ScopeNotificationSettings &new_settings) {
  bool need_update_server = current_settings->mute_until != new_settings.mute_until ||
                            current_settings->sound != new_settings.sound ||
                            current_settings->show_preview != new_settings.show_preview;
  bool need_update_local =
      current_settings->disable_pinned_message_notifications != new_settings.disable_pinned_message_notifications ||
      current_settings->disable_mention_notifications != new_settings.disable_mention_notifications;
  bool was_inited = current_settings->is_synchronized;
  bool is_inited = new_settings.is_synchronized;
  if (was_inited && !is_inited) {
    return false;  // just in case
  }
  bool is_changed = need_update_server || need_update_local || was_inited != is_inited;
  if (is_changed) {
    save_scope_notification_settings(scope, new_settings);

    update_scope_unmute_timeout(scope, current_settings->mute_until, new_settings.mute_until);

    if (!current_settings->disable_pinned_message_notifications && new_settings.disable_pinned_message_notifications) {
      VLOG(notifications) << "Remove pinned message notifications in " << scope;
      for (auto &dialog : dialogs_) {
        Dialog *d = dialog.second.get();
        if (d->notification_settings.use_default_disable_pinned_message_notifications &&
            d->mention_notification_group.group_id.is_valid() && d->pinned_message_notification_message_id.is_valid() &&
            get_dialog_notification_setting_scope(d->dialog_id) == scope) {
          remove_dialog_pinned_message_notification(d);
        }
      }
    }
    if (current_settings->disable_mention_notifications != new_settings.disable_mention_notifications) {
      VLOG(notifications) << "Remove mention notifications in " << scope;
      for (auto &dialog : dialogs_) {
        Dialog *d = dialog.second.get();
        if (d->notification_settings.use_default_disable_mention_notifications &&
            get_dialog_notification_setting_scope(d->dialog_id) == scope) {
          if (current_settings->disable_mention_notifications) {
            update_dialog_mention_notification_count(d);
          } else {
            remove_dialog_mention_notifications(d);
          }
        }
      }
    }

    VLOG(notifications) << "Update notification settings in " << scope << " from " << *current_settings << " to "
                        << new_settings;
    *current_settings = new_settings;

    send_closure(G()->td(), &Td::send_update, get_update_scope_notification_settings_object(scope));
  }
  return need_update_server;
}

void MessagesManager::update_dialog_unmute_timeout(Dialog *d, bool old_use_default, int32 old_mute_until,
                                                   bool new_use_default, int32 new_mute_until) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  if (old_use_default == new_use_default && old_mute_until == new_mute_until) {
    return;
  }
  CHECK(d != nullptr);

  auto now = G()->unix_time_cached();
  if (!new_use_default && new_mute_until >= now && new_mute_until < now + 366 * 86400) {
    dialog_unmute_timeout_.set_timeout_in(d->dialog_id.get(), new_mute_until - now + 1);
  } else {
    dialog_unmute_timeout_.cancel_timeout(d->dialog_id.get());
  }

  auto &list = get_dialog_list(d->folder_id);
  if (old_mute_until != -1 && need_unread_counter(d->order) &&
      (list.is_message_unread_count_inited_ || list.is_dialog_unread_count_inited_)) {
    auto unread_count = d->server_unread_count + d->local_unread_count;
    if (unread_count != 0 || d->is_marked_as_unread) {
      if (old_use_default || new_use_default) {
        auto scope_mute_until = get_scope_mute_until(d->dialog_id);
        if (old_use_default) {
          old_mute_until = scope_mute_until;
        }
        if (new_use_default) {
          new_mute_until = scope_mute_until;
        }
      }
      if ((old_mute_until != 0) != (new_mute_until != 0)) {
        if (unread_count != 0 && list.is_message_unread_count_inited_) {
          int32 delta = old_mute_until != 0 ? -unread_count : unread_count;
          list.unread_message_muted_count_ += delta;
          send_update_unread_message_count(d->folder_id, d->dialog_id, true, "update_dialog_unmute_timeout");
        }
        if (list.is_dialog_unread_count_inited_) {
          int32 delta = old_mute_until != 0 ? -1 : 1;
          list.unread_dialog_muted_count_ += delta;
          if (unread_count == 0 && d->is_marked_as_unread) {
            list.unread_dialog_muted_marked_count_ += delta;
          }
          send_update_unread_chat_count(d->folder_id, d->dialog_id, true, "update_dialog_unmute_timeout");
        }
      }
    }
  }
}

void MessagesManager::update_scope_unmute_timeout(NotificationSettingsScope scope, int32 old_mute_until,
                                                  int32 new_mute_until) {
  LOG(INFO) << "Update " << scope << " unmute timeout from " << old_mute_until << " to " << new_mute_until;
  if (old_mute_until == new_mute_until) {
    return;
  }

  auto now = G()->unix_time_cached();
  if (new_mute_until >= now && new_mute_until < now + 366 * 86400) {
    dialog_unmute_timeout_.set_timeout_in(static_cast<int64>(scope) + 1, new_mute_until - now + 1);
  } else {
    dialog_unmute_timeout_.cancel_timeout(static_cast<int64>(scope) + 1);
  }

  if (old_mute_until != -1 && !td_->auth_manager_->is_bot() && G()->parameters().use_message_db) {
    auto was_muted = old_mute_until != 0;
    auto is_muted = new_mute_until != 0;
    if (was_muted != is_muted) {
      std::unordered_map<FolderId, int32, FolderIdHash> delta;
      std::unordered_map<FolderId, int32, FolderIdHash> total_count;
      std::unordered_map<FolderId, int32, FolderIdHash> marked_count;
      std::unordered_set<FolderId, FolderIdHash> folder_ids;
      for (auto &dialog : dialogs_) {
        Dialog *d = dialog.second.get();
        if (need_unread_counter(d->order) && d->notification_settings.use_default_mute_until &&
            get_dialog_notification_setting_scope(d->dialog_id) == scope) {
          int32 unread_count = d->server_unread_count + d->local_unread_count;
          if (unread_count != 0) {
            delta[d->folder_id] += unread_count;
            total_count[d->folder_id]++;
            folder_ids.insert(d->folder_id);
          } else if (d->is_marked_as_unread) {
            total_count[d->folder_id]++;
            marked_count[d->folder_id]++;
            folder_ids.insert(d->folder_id);
          }
        }
      }
      for (auto folder_id : folder_ids) {
        auto &list = get_dialog_list(folder_id);
        if (delta[folder_id] != 0 && list.is_message_unread_count_inited_) {
          if (was_muted) {
            list.unread_message_muted_count_ -= delta[folder_id];
          } else {
            list.unread_message_muted_count_ += delta[folder_id];
          }
          send_update_unread_message_count(folder_id, DialogId(), true, "update_scope_unmute_timeout");
        }
        if (total_count[folder_id] != 0 && list.is_dialog_unread_count_inited_) {
          if (was_muted) {
            list.unread_dialog_muted_count_ -= total_count[folder_id];
            list.unread_dialog_muted_marked_count_ -= marked_count[folder_id];
          } else {
            list.unread_dialog_muted_count_ += total_count[folder_id];
            list.unread_dialog_muted_marked_count_ += marked_count[folder_id];
          }
          send_update_unread_chat_count(folder_id, DialogId(), true, "update_scope_unmute_timeout");
        }
      }
    }
  }
}

void MessagesManager::on_dialog_unmute(DialogId dialog_id) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (d->notification_settings.use_default_mute_until) {
    return;
  }
  if (d->notification_settings.mute_until == 0) {
    return;
  }

  auto now = G()->unix_time();
  if (d->notification_settings.mute_until > now) {
    LOG(ERROR) << "Failed to unmute " << dialog_id << " in " << now << ", will be unmuted in "
               << d->notification_settings.mute_until;
    update_dialog_unmute_timeout(d, false, -1, false, d->notification_settings.mute_until);
    return;
  }

  LOG(INFO) << "Unmute " << dialog_id;
  update_dialog_unmute_timeout(d, false, d->notification_settings.mute_until, false, 0);
  d->notification_settings.mute_until = 0;
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatNotificationSettings>(
                   dialog_id.get(), get_chat_notification_settings_object(&d->notification_settings)));
  on_dialog_updated(dialog_id, "on_dialog_unmute");
}

void MessagesManager::on_scope_unmute(NotificationSettingsScope scope) {
  auto notification_settings = get_scope_notification_settings(scope);
  CHECK(notification_settings != nullptr);

  if (notification_settings->mute_until == 0) {
    return;
  }

  auto now = G()->unix_time();
  if (notification_settings->mute_until > now) {
    LOG(ERROR) << "Failed to unmute " << scope << " in " << now << ", will be unmuted in "
               << notification_settings->mute_until;
    update_scope_unmute_timeout(scope, -1, notification_settings->mute_until);
    return;
  }

  LOG(INFO) << "Unmute " << scope;
  update_scope_unmute_timeout(scope, notification_settings->mute_until, 0);
  notification_settings->mute_until = 0;
  send_closure(G()->td(), &Td::send_update, get_update_scope_notification_settings_object(scope));
  save_scope_notification_settings(scope, *notification_settings);
}

void MessagesManager::on_update_dialog_notify_settings(
    DialogId dialog_id, tl_object_ptr<telegram_api::peerNotifySettings> &&peer_notify_settings, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  VLOG(notifications) << "Receive notification settings for " << dialog_id << " from " << source << ": "
                      << to_string(peer_notify_settings);

  DialogNotificationSettings *current_settings = get_dialog_notification_settings(dialog_id, true);
  if (current_settings == nullptr) {
    return;
  }

  const DialogNotificationSettings notification_settings = ::td::get_dialog_notification_settings(
      std::move(peer_notify_settings), current_settings->use_default_disable_pinned_message_notifications,
      current_settings->disable_pinned_message_notifications,
      current_settings->use_default_disable_mention_notifications, current_settings->disable_mention_notifications);
  if (!notification_settings.is_synchronized) {
    return;
  }

  update_dialog_notification_settings(dialog_id, current_settings, notification_settings);
}

void MessagesManager::on_update_scope_notify_settings(
    NotificationSettingsScope scope, tl_object_ptr<telegram_api::peerNotifySettings> &&peer_notify_settings) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto old_notification_settings = get_scope_notification_settings(scope);
  CHECK(old_notification_settings != nullptr);

  const ScopeNotificationSettings notification_settings = ::td::get_scope_notification_settings(
      std::move(peer_notify_settings), old_notification_settings->disable_pinned_message_notifications,
      old_notification_settings->disable_mention_notifications);
  if (!notification_settings.is_synchronized) {
    return;
  }

  update_scope_notification_settings(scope, old_notification_settings, notification_settings);
}

bool MessagesManager::update_dialog_silent_send_message(Dialog *d, bool silent_send_message) {
  CHECK(d != nullptr);
  LOG_IF(WARNING, !d->notification_settings.is_synchronized)
      << "Have unknown notification settings in " << d->dialog_id;
  if (d->notification_settings.silent_send_message == silent_send_message) {
    return false;
  }

  LOG(INFO) << "Update silent send message in " << d->dialog_id << " to " << silent_send_message;
  d->notification_settings.silent_send_message = silent_send_message;

  on_dialog_updated(d->dialog_id, "update_dialog_silent_send_message");

  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatDefaultDisableNotification>(d->dialog_id.get(), silent_send_message));
  return true;
}

void MessagesManager::repair_dialog_action_bar(DialogId dialog_id, const char *source) {
  if (G()->close_flag() || !dialog_id.is_valid() || td_->auth_manager_->is_bot()) {
    return;
  }

  LOG(INFO) << "Repair action bar in " << dialog_id << " from " << source;
  switch (dialog_id.get_type()) {
    case DialogType::User:
      td_->contacts_manager_->reload_user_full(dialog_id.get_user_id());
      return;
    case DialogType::Chat:
    case DialogType::Channel:
      if (!have_input_peer(dialog_id, AccessRights::Read)) {
        return;
      }

      return td_->create_handler<GetPeerSettingsQuery>()->send(dialog_id);
    case DialogType::SecretChat:
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::hide_dialog_action_bar(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return;
  }
  hide_dialog_action_bar(d);
}

void MessagesManager::hide_dialog_action_bar(Dialog *d) {
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  if (!d->know_can_report_spam) {
    return;
  }
  if (!d->can_report_spam && !d->can_add_contact && !d->can_block_user && !d->can_share_phone_number &&
      !d->can_report_location) {
    return;
  }

  d->can_report_spam = false;
  d->can_add_contact = false;
  d->can_block_user = false;
  d->can_share_phone_number = false;
  d->can_report_location = false;
  send_update_chat_action_bar(d);
}

void MessagesManager::remove_dialog_action_bar(DialogId dialog_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(3, "Can't access the chat"));
  }

  if (dialog_id.get_type() == DialogType::SecretChat) {
    dialog_id = DialogId(td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id()));
    d = get_dialog_force(dialog_id);
    if (d == nullptr) {
      return promise.set_error(Status::Error(3, "Chat with the user not found"));
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      return promise.set_error(Status::Error(3, "Can't access the chat"));
    }
  }

  if (!d->know_can_report_spam) {
    return promise.set_error(Status::Error(3, "Can't update chat action bar"));
  }

  if (!d->can_report_spam && !d->can_add_contact && !d->can_block_user && !d->can_share_phone_number &&
      !d->can_report_location) {
    return promise.set_value(Unit());
  }

  hide_dialog_action_bar(d);

  change_dialog_report_spam_state_on_server(dialog_id, false, 0, std::move(promise));
}

class MessagesManager::ChangeDialogReportSpamStateOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_spam_dialog_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(is_spam_dialog_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(is_spam_dialog_, parser);
  }
};

uint64 MessagesManager::save_change_dialog_report_spam_state_on_server_logevent(DialogId dialog_id,
                                                                                bool is_spam_dialog) {
  ChangeDialogReportSpamStateOnServerLogEvent logevent{dialog_id, is_spam_dialog};
  auto storer = LogEventStorerImpl<ChangeDialogReportSpamStateOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ChangeDialogReportSpamStateOnServer, storer);
}

void MessagesManager::change_dialog_report_spam_state_on_server(DialogId dialog_id, bool is_spam_dialog,
                                                                uint64 logevent_id, Promise<Unit> &&promise) {
  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_change_dialog_report_spam_state_on_server_logevent(dialog_id, is_spam_dialog);
  }

  auto new_promise = get_erase_logevent_promise(logevent_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      return td_->create_handler<UpdatePeerSettingsQuery>(std::move(promise))->send(dialog_id, is_spam_dialog);
    case DialogType::SecretChat:
      if (is_spam_dialog) {
        return td_->create_handler<ReportEncryptedSpamQuery>(std::move(promise))->send(dialog_id);
      } else {
        auto user_id = td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
        if (!user_id.is_valid()) {
          return promise.set_error(Status::Error(400, "Peer user not found"));
        }
        return td_->create_handler<UpdatePeerSettingsQuery>(std::move(promise))->send(DialogId(user_id), false);
      }
    case DialogType::None:
    default:
      UNREACHABLE();
      return;
  }
}

bool MessagesManager::can_report_dialog(DialogId dialog_id) const {
  // doesn't include possibility of report from action bar
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->can_report_user(dialog_id.get_user_id());
    case DialogType::Chat:
      return false;
    case DialogType::Channel:
      return !td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id()).is_creator();
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::report_dialog(DialogId dialog_id, const tl_object_ptr<td_api::ChatReportReason> &reason,
                                    const vector<MessageId> &message_ids, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(3, "Can't access the chat"));
  }

  if (reason == nullptr) {
    return promise.set_error(Status::Error(3, "Reason shouldn't be empty"));
  }

  Dialog *user_d = d;
  bool is_dialog_spam_report = false;
  bool can_report_spam = d->can_report_spam;
  if (reason->get_id() == td_api::chatReportReasonSpam::ID && message_ids.empty()) {
    // report from action bar
    if (dialog_id.get_type() == DialogType::SecretChat) {
      auto user_dialog_id = DialogId(td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id()));
      user_d = get_dialog_force(user_dialog_id);
      if (user_d == nullptr) {
        return promise.set_error(Status::Error(3, "Chat with the user not found"));
      }

      is_dialog_spam_report = user_d->know_can_report_spam;
      can_report_spam = user_d->can_report_spam;
    } else {
      is_dialog_spam_report = d->know_can_report_spam;
    }
  }

  if (is_dialog_spam_report && can_report_spam) {
    hide_dialog_action_bar(user_d);
    return change_dialog_report_spam_state_on_server(dialog_id, true, 0, std::move(promise));
  }

  if (!can_report_dialog(dialog_id)) {
    if (is_dialog_spam_report) {
      return promise.set_value(Unit());
    }

    return promise.set_error(Status::Error(3, "Chat can't be reported"));
  }

  vector<MessageId> server_message_ids;
  for (auto message_id : message_ids) {
    if (message_id.is_scheduled()) {
      return promise.set_error(Status::Error(3, "Can't report scheduled messages"));
    }
    if (message_id.is_valid() && message_id.is_server()) {
      server_message_ids.push_back(message_id);
    }
  }

  tl_object_ptr<telegram_api::ReportReason> report_reason;
  switch (reason->get_id()) {
    case td_api::chatReportReasonSpam::ID:
      report_reason = make_tl_object<telegram_api::inputReportReasonSpam>();
      break;
    case td_api::chatReportReasonViolence::ID:
      report_reason = make_tl_object<telegram_api::inputReportReasonViolence>();
      break;
    case td_api::chatReportReasonPornography::ID:
      report_reason = make_tl_object<telegram_api::inputReportReasonPornography>();
      break;
    case td_api::chatReportReasonChildAbuse::ID:
      report_reason = make_tl_object<telegram_api::inputReportReasonChildAbuse>();
      break;
    case td_api::chatReportReasonCopyright::ID:
      report_reason = make_tl_object<telegram_api::inputReportReasonCopyright>();
      break;
    case td_api::chatReportReasonUnrelatedLocation::ID:
      report_reason = make_tl_object<telegram_api::inputReportReasonGeoIrrelevant>();
      if (dialog_id.get_type() == DialogType::Channel) {
        hide_dialog_action_bar(d);
      }
      break;
    case td_api::chatReportReasonCustom::ID: {
      auto other_reason = static_cast<const td_api::chatReportReasonCustom *>(reason.get());
      auto text = other_reason->text_;
      if (!clean_input_string(text)) {
        return promise.set_error(Status::Error(400, "Text must be encoded in UTF-8"));
      }

      report_reason = make_tl_object<telegram_api::inputReportReasonOther>(text);
      break;
    }
    default:
      UNREACHABLE();
  }
  CHECK(report_reason != nullptr);

  td_->create_handler<ReportPeerQuery>(std::move(promise))
      ->send(dialog_id, std::move(report_reason), server_message_ids);
}

void MessagesManager::on_get_peer_settings(DialogId dialog_id,
                                           tl_object_ptr<telegram_api::peerSettings> &&peer_settings,
                                           bool ignore_privacy_exception) {
  CHECK(peer_settings != nullptr);
  if (dialog_id.get_type() == DialogType::User && !ignore_privacy_exception) {
    auto need_phone_number_privacy_exception =
        (peer_settings->flags_ & telegram_api::peerSettings::NEED_CONTACTS_EXCEPTION_MASK) != 0;
    td_->contacts_manager_->on_update_user_need_phone_number_privacy_exception(dialog_id.get_user_id(),
                                                                               need_phone_number_privacy_exception);
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return;
  }

  auto can_report_spam = (peer_settings->flags_ & telegram_api::peerSettings::REPORT_SPAM_MASK) != 0;
  auto can_add_contact = (peer_settings->flags_ & telegram_api::peerSettings::ADD_CONTACT_MASK) != 0;
  auto can_block_user = (peer_settings->flags_ & telegram_api::peerSettings::BLOCK_CONTACT_MASK) != 0;
  auto can_share_phone_number = (peer_settings->flags_ & telegram_api::peerSettings::SHARE_CONTACT_MASK) != 0;
  auto can_report_location = (peer_settings->flags_ & telegram_api::peerSettings::REPORT_GEO_MASK) != 0;
  if (d->can_report_spam == can_report_spam && d->can_add_contact == can_add_contact &&
      d->can_block_user == can_block_user && d->can_share_phone_number == can_share_phone_number &&
      d->can_report_location == can_report_location) {
    if (!d->know_action_bar || !d->know_can_report_spam) {
      d->know_can_report_spam = true;
      d->know_action_bar = true;
      on_dialog_updated(d->dialog_id, "on_get_peer_settings");
    }
    return;
  }

  d->know_can_report_spam = true;
  d->know_action_bar = true;
  d->can_report_spam = can_report_spam;
  d->can_add_contact = can_add_contact;
  d->can_block_user = can_block_user;
  d->can_share_phone_number = can_share_phone_number;
  d->can_report_location = can_report_location;

  fix_dialog_action_bar(d);

  send_update_chat_action_bar(d);
}

void MessagesManager::fix_dialog_action_bar(Dialog *d) {
  CHECK(d != nullptr);
  if (!d->know_action_bar) {
    return;
  }

  if (d->can_report_location) {
    if (d->dialog_id.get_type() != DialogType::Channel) {
      LOG(ERROR) << "Receive can_report_location in " << d->dialog_id;
      d->can_report_location = false;
    } else if (d->can_report_spam || d->can_add_contact || d->can_block_user || d->can_share_phone_number) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/" << d->can_block_user
                 << "/" << d->can_share_phone_number << "/" << d->can_report_location;
      d->can_report_spam = false;
      d->can_add_contact = false;
      d->can_block_user = false;
      d->can_share_phone_number = false;
    }
  }
  if (d->dialog_id.get_type() == DialogType::User) {
    auto user_id = d->dialog_id.get_user_id();
    bool is_me = user_id == td_->contacts_manager_->get_my_id();
    bool is_contact = td_->contacts_manager_->is_user_contact(user_id);
    bool is_blocked = td_->contacts_manager_->is_user_blocked(user_id);
    bool is_deleted = td_->contacts_manager_->is_user_deleted(user_id);
    if (is_me || is_blocked) {
      d->can_report_spam = false;
    }
    if (is_me || is_blocked || is_deleted) {
      d->can_share_phone_number = false;
    }
    if (is_me || is_blocked || is_deleted || is_contact) {
      d->can_block_user = false;
      d->can_add_contact = false;
    }
  }
  if (d->can_share_phone_number) {
    CHECK(!d->can_report_location);
    if (d->dialog_id.get_type() != DialogType::User) {
      LOG(ERROR) << "Receive can_share_phone_number in " << d->dialog_id;
      d->can_share_phone_number = false;
    } else if (d->can_report_spam || d->can_add_contact || d->can_block_user) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/" << d->can_block_user
                 << "/" << d->can_share_phone_number;
      d->can_report_spam = false;
      d->can_add_contact = false;
      d->can_block_user = false;
    }
  }
  if (d->can_block_user) {
    CHECK(!d->can_report_location);
    CHECK(!d->can_share_phone_number);
    if (d->dialog_id.get_type() != DialogType::User) {
      LOG(ERROR) << "Receive can_block_user in " << d->dialog_id;
      d->can_block_user = false;
    } else if (!d->can_report_spam || !d->can_add_contact) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/"
                 << d->can_block_user;
      d->can_report_spam = true;
      d->can_add_contact = true;
    }
  }
  if (d->can_add_contact) {
    CHECK(!d->can_report_location);
    CHECK(!d->can_share_phone_number);
    if (d->dialog_id.get_type() != DialogType::User) {
      LOG(ERROR) << "Receive can_add_contact in " << d->dialog_id;
      d->can_add_contact = false;
    } else if (d->can_report_spam != d->can_block_user) {
      LOG(ERROR) << "Receive action bar " << d->can_report_spam << "/" << d->can_add_contact << "/"
                 << d->can_block_user;
      d->can_report_spam = false;
      d->can_block_user = false;
    }
  }
}

void MessagesManager::get_dialog_statistics_url(DialogId dialog_id, const string &parameters, bool is_dark,
                                                Promise<td_api::object_ptr<td_api::httpUrl>> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(3, "Can't access the chat"));
  }
  if (dialog_id.get_type() == DialogType::SecretChat) {
    return promise.set_error(Status::Error(500, "There is no statistics for secret chats"));
  }

  td_->create_handler<GetStatsUrlQuery>(std::move(promise))->send(dialog_id, parameters, is_dark);
}

Result<string> MessagesManager::get_login_button_url(DialogId dialog_id, MessageId message_id, int32 button_id) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(3, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(3, "Can't access the chat");
  }

  auto m = get_message_force(d, message_id, "get_login_button_url");
  if (m == nullptr) {
    return Status::Error(5, "Message not found");
  }
  if (m->reply_markup == nullptr || m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
    return Status::Error(5, "Message has no inline keyboard");
  }
  if (message_id.is_scheduled()) {
    return Status::Error(5, "Can't use login buttons from scheduled messages");
  }
  if (!message_id.is_server()) {
    // it shouldn't have UrlAuth buttons anyway
    return Status::Error(5, "Message is not server");
  }
  if (dialog_id.get_type() == DialogType::SecretChat) {
    // secret chat messages can't have reply markup, so this shouldn't happen now
    return Status::Error(5, "Message is in a secret chat");
  }

  for (auto &row : m->reply_markup->inline_keyboard) {
    for (auto &button : row) {
      if (button.type == InlineKeyboardButton::Type::UrlAuth && button.id == button_id) {
        return button.data;
      }
    }
  }

  return Status::Error(5, "Button not found");
}

void MessagesManager::get_login_url_info(DialogId dialog_id, MessageId message_id, int32 button_id,
                                         Promise<td_api::object_ptr<td_api::LoginUrlInfo>> &&promise) {
  auto r_url = get_login_button_url(dialog_id, message_id, button_id);
  if (r_url.is_error()) {
    return promise.set_error(r_url.move_as_error());
  }

  td_->create_handler<RequestUrlAuthQuery>(std::move(promise))
      ->send(r_url.move_as_ok(), dialog_id, message_id, button_id);
}

void MessagesManager::get_login_url(DialogId dialog_id, MessageId message_id, int32 button_id, bool allow_write_access,
                                    Promise<td_api::object_ptr<td_api::httpUrl>> &&promise) {
  auto r_url = get_login_button_url(dialog_id, message_id, button_id);
  if (r_url.is_error()) {
    return promise.set_error(r_url.move_as_error());
  }

  td_->create_handler<AcceptUrlAuthQuery>(std::move(promise))
      ->send(r_url.move_as_ok(), dialog_id, message_id, button_id, allow_write_access);
}

void MessagesManager::load_secret_thumbnail(FileId thumbnail_file_id) {
  class Callback : public FileManager::DownloadCallback {
   public:
    explicit Callback(Promise<> download_promise) : download_promise_(std::move(download_promise)) {
    }

    void on_download_ok(FileId file_id) override {
      download_promise_.set_value(Unit());
    }
    void on_download_error(FileId file_id, Status error) override {
      download_promise_.set_error(std::move(error));
    }

   private:
    Promise<> download_promise_;
  };

  auto thumbnail_promise = PromiseCreator::lambda([actor_id = actor_id(this),
                                                   thumbnail_file_id](Result<BufferSlice> r_thumbnail) {
    BufferSlice thumbnail_slice;
    if (r_thumbnail.is_ok()) {
      thumbnail_slice = r_thumbnail.move_as_ok();
    }
    send_closure(actor_id, &MessagesManager::on_load_secret_thumbnail, thumbnail_file_id, std::move(thumbnail_slice));
  });

  auto download_promise = PromiseCreator::lambda(
      [thumbnail_file_id, thumbnail_promise = std::move(thumbnail_promise)](Result<Unit> r_download) mutable {
        if (r_download.is_error()) {
          thumbnail_promise.set_error(r_download.move_as_error());
          return;
        }
        send_closure(G()->file_manager(), &FileManager::get_content, thumbnail_file_id, std::move(thumbnail_promise));
      });

  send_closure(G()->file_manager(), &FileManager::download, thumbnail_file_id,
               std::make_shared<Callback>(std::move(download_promise)), 1, -1, -1);
}

void MessagesManager::on_upload_media(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file,
                                      tl_object_ptr<telegram_api::InputEncryptedFile> input_encrypted_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = being_uploaded_files_.find(file_id);
  if (it == being_uploaded_files_.end()) {
    // callback may be called just before the file upload was cancelled
    return;
  }

  auto full_message_id = it->second.first;
  auto thumbnail_file_id = it->second.second;

  being_uploaded_files_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel, do not need to send or edit it
    // file upload should be already cancelled in cancel_send_message_query, it shouldn't happen
    LOG(ERROR) << "Message with a media has already been deleted";
    return;
  }

  bool is_edit = m->message_id.is_any_server();
  auto dialog_id = full_message_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (!is_edit && can_send_status.is_error()) {
    // user has left the chat during upload of the file or lost their privileges
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status.error();

    fail_send_message(full_message_id, can_send_status.move_as_error());
    return;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      if (input_file && thumbnail_file_id.is_valid()) {
        // TODO: download thumbnail if needed (like in secret chats)
        being_uploaded_thumbnails_[thumbnail_file_id] = {full_message_id, file_id, std::move(input_file)};
        LOG(INFO) << "Ask to upload thumbnail " << thumbnail_file_id;
        td_->file_manager_->upload(thumbnail_file_id, upload_thumbnail_callback_, 32, m->message_id.get());
      } else {
        do_send_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), nullptr);
      }
      break;
    case DialogType::SecretChat:
      if (thumbnail_file_id.is_valid()) {
        being_loaded_secret_thumbnails_[thumbnail_file_id] = {full_message_id, file_id,
                                                              std::move(input_encrypted_file)};
        LOG(INFO) << "Ask to load thumbnail " << thumbnail_file_id;

        load_secret_thumbnail(thumbnail_file_id);
      } else {
        do_send_secret_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_encrypted_file), BufferSlice());
      }
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
}

void MessagesManager::do_send_media(DialogId dialog_id, Message *m, FileId file_id, FileId thumbnail_file_id,
                                    tl_object_ptr<telegram_api::InputFile> input_file,
                                    tl_object_ptr<telegram_api::InputFile> input_thumbnail) {
  CHECK(m != nullptr);

  bool have_input_file = input_file != nullptr;
  bool have_input_thumbnail = input_thumbnail != nullptr;
  LOG(INFO) << "Do send media file " << file_id << " with thumbnail " << thumbnail_file_id
            << ", have_input_file = " << have_input_file << ", have_input_thumbnail = " << have_input_thumbnail
            << ", ttl = " << m->ttl;

  MessageContent *content = nullptr;
  if (m->message_id.is_any_server()) {
    content = m->edited_content.get();
    if (content == nullptr) {
      LOG(ERROR) << "Message has no edited content";
      return;
    }
  } else {
    content = m->content.get();
  }

  auto input_media = get_input_media(content, td_, std::move(input_file), std::move(input_thumbnail), file_id,
                                     thumbnail_file_id, m->ttl, true);
  LOG_CHECK(input_media != nullptr) << to_string(get_message_object(dialog_id, m)) << ' ' << have_input_file << ' '
                                    << have_input_thumbnail << ' ' << file_id << ' ' << thumbnail_file_id << ' '
                                    << m->ttl;

  on_message_media_uploaded(dialog_id, m, std::move(input_media), file_id, thumbnail_file_id);
}

void MessagesManager::do_send_secret_media(DialogId dialog_id, Message *m, FileId file_id, FileId thumbnail_file_id,
                                           tl_object_ptr<telegram_api::InputEncryptedFile> input_encrypted_file,
                                           BufferSlice thumbnail) {
  CHECK(dialog_id.get_type() == DialogType::SecretChat);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(m->message_id.is_yet_unsent());

  bool have_input_file = input_encrypted_file != nullptr;
  LOG(INFO) << "Do send secret media file " << file_id << " with thumbnail " << thumbnail_file_id
            << ", have_input_file = " << have_input_file;

  auto layer = td_->contacts_manager_->get_secret_chat_layer(dialog_id.get_secret_chat_id());
  on_secret_message_media_uploaded(
      dialog_id, m,
      get_secret_input_media(m->content.get(), td_, std::move(input_encrypted_file), std::move(thumbnail), layer),
      file_id, thumbnail_file_id);
}

void MessagesManager::on_upload_media_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(WARNING) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = being_uploaded_files_.find(file_id);
  if (it == being_uploaded_files_.end()) {
    // callback may be called just before the file upload was cancelled
    return;
  }

  auto full_message_id = it->second.first;

  being_uploaded_files_.erase(it);

  bool is_edit = full_message_id.get_message_id().is_any_server();
  if (is_edit) {
    fail_edit_message_media(full_message_id, Status::Error(status.code() > 0 ? status.code() : 500, status.message()));
  } else {
    fail_send_message(full_message_id, std::move(status));
  }
}

void MessagesManager::on_load_secret_thumbnail(FileId thumbnail_file_id, BufferSlice thumbnail) {
  if (G()->close_flag()) {
    // do not send secret media if closing, thumbnail may be wrong
    return;
  }

  LOG(INFO) << "SecretThumbnail " << thumbnail_file_id << " has been loaded with size " << thumbnail.size();

  auto it = being_loaded_secret_thumbnails_.find(thumbnail_file_id);
  if (it == being_loaded_secret_thumbnails_.end()) {
    // just in case, as in on_upload_thumbnail
    return;
  }

  auto full_message_id = it->second.full_message_id;
  auto file_id = it->second.file_id;
  auto input_file = std::move(it->second.input_file);

  being_loaded_secret_thumbnails_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user, do not need to send it
    // cancel file upload of the main file to allow next upload with the same file to succeed
    LOG(INFO) << "Message with a media has already been deleted";
    cancel_upload_file(file_id);
    return;
  }
  CHECK(m->message_id.is_yet_unsent());

  if (thumbnail.empty()) {
    delete_message_content_thumbnail(m->content.get(), td_);
  }

  auto dialog_id = full_message_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (can_send_status.is_error()) {
    // secret chat was closed during load of the file
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status.error();

    fail_send_message(full_message_id, can_send_status.move_as_error());
    return;
  }

  do_send_secret_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), std::move(thumbnail));
}

void MessagesManager::on_upload_thumbnail(FileId thumbnail_file_id,
                                          tl_object_ptr<telegram_api::InputFile> thumbnail_input_file) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "Thumbnail " << thumbnail_file_id << " has been uploaded as " << to_string(thumbnail_input_file);

  auto it = being_uploaded_thumbnails_.find(thumbnail_file_id);
  if (it == being_uploaded_thumbnails_.end()) {
    // callback may be called just before the thumbnail upload was cancelled
    return;
  }

  auto full_message_id = it->second.full_message_id;
  auto file_id = it->second.file_id;
  auto input_file = std::move(it->second.input_file);

  being_uploaded_thumbnails_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel, do not need to send or edit it
    // thumbnail file upload should be already cancelled in cancel_send_message_query
    LOG(ERROR) << "Message with a media has already been deleted";
    return;
  }

  bool is_edit = m->message_id.is_any_server();

  if (thumbnail_input_file == nullptr) {
    delete_message_content_thumbnail(is_edit ? m->edited_content.get() : m->content.get(), td_);
  }

  auto dialog_id = full_message_id.get_dialog_id();
  auto can_send_status = can_send_message(dialog_id);
  if (!is_edit && can_send_status.is_error()) {
    // user has left the chat during upload of the thumbnail or lost their privileges
    LOG(INFO) << "Can't send a message to " << dialog_id << ": " << can_send_status.error();

    fail_send_message(full_message_id, can_send_status.move_as_error());
    return;
  }

  do_send_media(dialog_id, m, file_id, thumbnail_file_id, std::move(input_file), std::move(thumbnail_input_file));
}

void MessagesManager::on_upload_dialog_photo(FileId file_id, tl_object_ptr<telegram_api::InputFile> input_file) {
  LOG(INFO) << "File " << file_id << " has been uploaded";

  auto it = uploaded_dialog_photos_.find(file_id);
  if (it == uploaded_dialog_photos_.end()) {
    // just in case, as in on_upload_media
    return;
  }

  Promise<Unit> promise = std::move(it->second.promise);
  DialogId dialog_id = it->second.dialog_id;

  uploaded_dialog_photos_.erase(it);

  tl_object_ptr<telegram_api::InputChatPhoto> input_chat_photo;
  FileView file_view = td_->file_manager_->get_file_view(file_id);
  CHECK(!file_view.is_encrypted());
  if (input_file == nullptr && file_view.has_remote_location()) {
    if (file_view.main_remote_location().is_web()) {
      // TODO reupload
      promise.set_error(Status::Error(400, "Can't use web photo as profile photo"));
      return;
    }
    auto input_photo = file_view.main_remote_location().as_input_photo();
    input_chat_photo = make_tl_object<telegram_api::inputChatPhoto>(std::move(input_photo));
  } else {
    input_chat_photo = make_tl_object<telegram_api::inputChatUploadedPhoto>(std::move(input_file));
  }

  send_edit_dialog_photo_query(dialog_id, file_id, std::move(input_chat_photo), std::move(promise));
}

void MessagesManager::on_upload_dialog_photo_error(FileId file_id, Status status) {
  if (G()->close_flag()) {
    // do not fail upload if closing
    return;
  }

  LOG(INFO) << "File " << file_id << " has upload error " << status;
  CHECK(status.is_error());

  auto it = uploaded_dialog_photos_.find(file_id);
  if (it == uploaded_dialog_photos_.end()) {
    // just in case, as in on_upload_media_error
    return;
  }

  Promise<Unit> promise = std::move(it->second.promise);

  uploaded_dialog_photos_.erase(it);

  promise.set_error(std::move(status));
}

void MessagesManager::before_get_difference() {
  running_get_difference_ = true;

  // scheduled messages are not returned in getDifference, so we must always reget them after it
  scheduled_messages_sync_generation_++;

  postponed_pts_updates_.insert(std::make_move_iterator(pending_updates_.begin()),
                                std::make_move_iterator(pending_updates_.end()));

  drop_pending_updates();
}

void MessagesManager::after_get_difference() {
  CHECK(!td_->updates_manager_->running_get_difference());

  if (postponed_pts_updates_.size()) {
    LOG(INFO) << "Begin to apply postponed pts updates";
    auto old_pts = td_->updates_manager_->get_pts();
    for (auto &update : postponed_pts_updates_) {
      auto new_pts = update.second.pts;
      if (new_pts <= old_pts) {
        skip_old_pending_update(std::move(update.second.update), new_pts, old_pts, update.second.pts_count,
                                "after get difference");
      } else {
        add_pending_update(std::move(update.second.update), update.second.pts, update.second.pts_count, false,
                           "after get difference");
      }
      CHECK(!td_->updates_manager_->running_get_difference());
    }
    postponed_pts_updates_.clear();
    LOG(INFO) << "Finish to apply postponed pts updates";
  }

  running_get_difference_ = false;

  if (!pending_on_get_dialogs_.empty()) {
    LOG(INFO) << "Apply postponed results of getDialogs";
    for (auto &res : pending_on_get_dialogs_) {
      on_get_dialogs(res.folder_id, std::move(res.dialogs), res.total_count, std::move(res.messages),
                     std::move(res.promise));
    }
    pending_on_get_dialogs_.clear();
  }

  if (!postponed_chat_read_inbox_updates_.empty()) {
    LOG(INFO) << "Send postponed chat read inbox updates";
    auto dialog_ids = std::move(postponed_chat_read_inbox_updates_);
    for (auto dialog_id : dialog_ids) {
      send_update_chat_read_inbox(get_dialog(dialog_id), false, "after_get_difference");
    }
  }
  while (!postponed_unread_message_count_updates_.empty()) {
    auto folder_id = *postponed_unread_message_count_updates_.begin();
    send_update_unread_message_count(folder_id, DialogId(), false, "after_get_difference");
  }
  while (!postponed_unread_chat_count_updates_.empty()) {
    auto folder_id = *postponed_unread_chat_count_updates_.begin();
    send_update_unread_chat_count(folder_id, DialogId(), false, "after_get_difference");
  }

  vector<FullMessageId> update_message_ids_to_delete;
  for (auto &it : update_message_ids_) {
    // this is impossible for ordinary chats because updates coming during getDifference have already been applied
    auto full_message_id = it.first;
    auto dialog_id = full_message_id.get_dialog_id();
    auto message_id = full_message_id.get_message_id();
    CHECK(message_id.is_valid());
    switch (dialog_id.get_type()) {
      case DialogType::Channel:
        // get channel difference may prevent updates from being applied
        if (running_get_channel_difference(dialog_id)) {
          break;
        }
      // fallthrough
      case DialogType::User:
      case DialogType::Chat: {
        if (!have_message_force({dialog_id, it.second}, "after get difference")) {
          // The sent message has already been deleted by the user or sent to inaccessible channel.
          // The sent message may never be received, but we will need updateMessageId in case the message is received
          // to delete it from the server and to not add to the chat.
          // But if the chat is inaccessible, then likely we will be unable to delete the message from server and
          // will delete it from the chat just after it is added. So we remove updateMessageId for such messages in
          // order to not check them over and over.
          if (!have_input_peer(dialog_id, AccessRights::Read)) {
            update_message_ids_to_delete.push_back(it.first);
          }
          break;
        }

        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        LOG(ERROR) << "Receive updateMessageId from " << it.second << " to " << full_message_id
                   << " but not receive corresponding message, last_new_message_id = " << d->last_new_message_id;
        if (dialog_id.get_type() != DialogType::Channel) {
          dump_debug_message_op(get_dialog(dialog_id));
        }
        if (message_id.is_scheduled() || message_id <= d->last_new_message_id) {
          get_message_from_server(it.first, PromiseCreator::lambda([this, full_message_id](Result<Unit> result) {
                                    if (result.is_error()) {
                                      LOG(WARNING)
                                          << "Failed to get missing " << full_message_id << ": " << result.error();
                                    } else {
                                      LOG(WARNING) << "Successfully get missing " << full_message_id << ": "
                                                   << to_string(get_message_object(full_message_id));
                                    }
                                  }));
        } else if (dialog_id.get_type() == DialogType::Channel) {
          LOG(INFO) << "Schedule getDifference in " << dialog_id.get_channel_id();
          channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
        }
        break;
      }
      case DialogType::SecretChat:
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
        break;
    }
  }
  for (auto full_message_id : update_message_ids_to_delete) {
    update_message_ids_.erase(full_message_id);
  }

  if (!G()->td_db()->get_binlog_pmc()->isset("fetched_marks_as_unread") && !td_->auth_manager_->is_bot()) {
    td_->create_handler<GetDialogUnreadMarksQuery>()->send();
  }

  load_notification_settings();

  if (!td_->auth_manager_->is_bot()) {
    auto folder_id = FolderId::archive();
    auto &list = get_dialog_list(folder_id);
    if (!list.is_dialog_unread_count_inited_) {
      get_dialogs(folder_id, MIN_DIALOG_DATE, 1, false, PromiseCreator::lambda([folder_id](Unit) {
                    LOG(INFO) << "Inited total chat count in " << folder_id;
                  }));
    }
  }
}

MessagesManager::MessagesInfo MessagesManager::on_get_messages(
    tl_object_ptr<telegram_api::messages_Messages> &&messages_ptr, const char *source) {
  CHECK(messages_ptr != nullptr);
  LOG(DEBUG) << "Receive result for " << source << ": " << to_string(messages_ptr);

  vector<tl_object_ptr<telegram_api::User>> users;
  vector<tl_object_ptr<telegram_api::Chat>> chats;
  MessagesInfo result;
  switch (messages_ptr->get_id()) {
    case telegram_api::messages_messages::ID: {
      auto messages = move_tl_object_as<telegram_api::messages_messages>(messages_ptr);

      users = std::move(messages->users_);
      chats = std::move(messages->chats_);
      result.total_count = narrow_cast<int32>(messages->messages_.size());
      result.messages = std::move(messages->messages_);
      break;
    }
    case telegram_api::messages_messagesSlice::ID: {
      auto messages = move_tl_object_as<telegram_api::messages_messagesSlice>(messages_ptr);

      users = std::move(messages->users_);
      chats = std::move(messages->chats_);
      result.total_count = messages->count_;
      result.messages = std::move(messages->messages_);
      break;
    }
    case telegram_api::messages_channelMessages::ID: {
      auto messages = move_tl_object_as<telegram_api::messages_channelMessages>(messages_ptr);

      users = std::move(messages->users_);
      chats = std::move(messages->chats_);
      result.total_count = messages->count_;
      result.messages = std::move(messages->messages_);
      result.is_channel_messages = true;
      break;
    }
    case telegram_api::messages_messagesNotModified::ID:
      LOG(ERROR) << "Server returned messagesNotModified in response to " << source;
      break;
    default:
      UNREACHABLE();
      break;
  }

  td_->contacts_manager_->on_get_users(std::move(users), source);
  td_->contacts_manager_->on_get_chats(std::move(chats), source);

  return result;
}

void MessagesManager::on_get_messages(vector<tl_object_ptr<telegram_api::Message>> &&messages, bool is_channel_message,
                                      bool is_scheduled, const char *source) {
  LOG(DEBUG) << "Receive " << messages.size() << " messages";
  for (auto &message : messages) {
    on_get_message(std::move(message), false, is_channel_message, is_scheduled, false, false, source);
  }
}

void MessagesManager::on_get_history(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit,
                                     bool from_the_end, vector<tl_object_ptr<telegram_api::Message>> &&messages) {
  LOG(INFO) << "Receive " << messages.size() << " history messages " << (from_the_end ? "from the end " : "") << "in "
            << dialog_id << " from " << from_message_id << " with offset " << offset << " and limit " << limit;
  CHECK(-limit < offset && offset <= 0);
  CHECK(offset < 0 || from_the_end);
  CHECK(!from_message_id.is_scheduled());

  // it is likely that there is no more history messages on the server
  bool have_full_history = from_the_end && narrow_cast<int32>(messages.size()) < limit;
  Dialog *d = get_dialog(dialog_id);

  if (messages.empty()) {
    if (d != nullptr) {
      if (have_full_history) {
        d->have_full_history = true;
        on_dialog_updated(dialog_id, "set have_full_history");
      }

      if (from_the_end && d->have_full_history && d->messages == nullptr && !d->last_database_message_id.is_valid()) {
        set_dialog_is_empty(d, "on_get_history empty");
      }
    }

    // be aware that in some cases an empty answer may be returned, because of the race of getHistory and deleteMessages
    // and not because there are no more messages
    return;
  }

  {
    // check that messages are received in decreasing message_id order
    MessageId cur_message_id = MessageId::max();
    for (const auto &message : messages) {
      MessageId message_id = get_message_id(message, false);
      if (message_id >= cur_message_id) {
        string error = PSTRING() << "Receive messages in the wrong order in history of " << dialog_id << " from "
                                 << from_message_id << " with offset " << offset << ", limit " << limit
                                 << ", from_the_end = " << from_the_end << ": ";
        for (const auto &debug_message : messages) {
          error += to_string(debug_message);
        }
        // TODO move to ERROR
        LOG(FATAL) << error;
        return;
      }
      cur_message_id = message_id;
    }
  }

  // be aware that dialog may not yet exist
  // be aware that returned messages are guaranteed to be consecutive messages, but if !from_the_end they
  // may be older (if some messages was deleted) or newer (if some messages were added) than an expected answer
  // be aware that any subset of the returned messages may be already deleted and returned as MessageEmpty
  bool is_channel_message = dialog_id.get_type() == DialogType::Channel;
  MessageId first_added_message_id;
  MessageId last_received_message_id = get_message_id(messages[0], false);
  MessageId last_added_message_id;
  bool have_next = false;

  if (narrow_cast<int32>(messages.size()) < limit + offset && d != nullptr) {
    MessageId first_received_message_id = get_message_id(messages.back(), false);
    if (first_received_message_id >= from_message_id && d->first_database_message_id.is_valid() &&
        first_received_message_id >= d->first_database_message_id) {
      // it is likely that there is no more history messages on the server
      have_full_history = true;
    }
  }

  bool prev_have_full_history = false;
  MessageId prev_last_new_message_id;
  MessageId prev_first_database_message_id;
  MessageId prev_last_database_message_id;
  MessageId prev_last_message_id;
  if (d != nullptr) {
    prev_last_new_message_id = d->last_new_message_id;
    prev_first_database_message_id = d->first_database_message_id;
    prev_last_database_message_id = d->last_database_message_id;
    prev_last_message_id = d->last_message_id;
    prev_have_full_history = d->have_full_history;
  }

  for (auto &message : messages) {
    if (!have_next && from_the_end && d != nullptr && get_message_id(message, false) < d->last_message_id) {
      // last message in the dialog should be attached to the next message if there is some
      have_next = true;
    }

    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive " << get_message_id(message, false) << " in wrong " << message_dialog_id << " instead of "
                 << dialog_id << ": " << oneline(to_string(message));
      continue;
    }

    auto full_message_id =
        on_get_message(std::move(message), false, is_channel_message, false, false, have_next, "get history");
    auto message_id = full_message_id.get_message_id();
    if (message_id.is_valid()) {
      if (!last_added_message_id.is_valid()) {
        last_added_message_id = message_id;
      }

      if (!have_next) {
        if (d == nullptr) {
          d = get_dialog(dialog_id);
          CHECK(d != nullptr);
        }
        have_next = true;
      } else if (first_added_message_id.is_valid()) {
        Message *next_message = get_message(d, first_added_message_id);
        CHECK(next_message != nullptr);
        if (!next_message->have_previous) {
          LOG(INFO) << "Fix have_previous for " << first_added_message_id;
          next_message->have_previous = true;
          attach_message_to_previous(d, first_added_message_id, "on_get_history");
        }
      }
      first_added_message_id = message_id;
      if (!message_id.is_yet_unsent()) {
        // message should be already saved to database in on_get_message
        // add_message_to_database(d, get_message(d, message_id), "on_get_history");
      }
    }
  }

  if (d == nullptr) {
    return;
  }

  if (have_full_history) {
    d->have_full_history = true;
    on_dialog_updated(dialog_id, "set have_full_history 2");
  }

  //  LOG_IF(ERROR, d->first_message_id.is_valid() && d->first_message_id > first_received_message_id)
  //      << "Receive message " << first_received_message_id << ", but first chat message is " << d->first_message_id;

  bool need_update_database_message_ids =
      last_added_message_id.is_valid() && (from_the_end || (last_added_message_id >= d->first_database_message_id &&
                                                            d->last_database_message_id >= first_added_message_id));
  if (from_the_end) {
    if (!d->last_new_message_id.is_valid()) {
      set_dialog_last_new_message_id(
          d, last_added_message_id.is_valid() ? last_added_message_id : last_received_message_id, "on_get_history");
    }
    if (last_added_message_id.is_valid()) {
      if (last_added_message_id > d->last_message_id) {
        CHECK(d->last_new_message_id.is_valid());
        set_dialog_last_message_id(d, last_added_message_id, "on_get_history");
        send_update_chat_last_message(d, "on_get_history");
      }
    }
  }

  if (need_update_database_message_ids) {
    bool is_dialog_updated = false;
    if (!d->last_database_message_id.is_valid()) {
      CHECK(d->last_message_id.is_valid());
      MessagesConstIterator it(d, d->last_message_id);
      while (*it != nullptr) {
        auto message_id = (*it)->message_id;
        if (message_id.is_server() || message_id.is_local()) {
          if (!d->last_database_message_id.is_valid()) {
            set_dialog_last_database_message_id(d, message_id, "on_get_history");
          }
          set_dialog_first_database_message_id(d, message_id, "on_get_history");
          try_restore_dialog_reply_markup(d, *it);
        }
        --it;
      }
      is_dialog_updated = true;
    } else {
      LOG_CHECK(d->last_new_message_id.is_valid())
          << dialog_id << " " << from_the_end << " " << d->first_database_message_id << " "
          << d->last_database_message_id << " " << first_added_message_id << " " << last_added_message_id << " "
          << d->last_message_id << " " << d->last_new_message_id << " " << d->have_full_history << " "
          << prev_last_new_message_id << " " << prev_first_database_message_id << " " << prev_last_database_message_id
          << " " << prev_last_message_id << " " << prev_have_full_history << " " << d->debug_last_new_message_id << " "
          << d->debug_first_database_message_id << " " << d->debug_last_database_message_id << " " << from_message_id
          << " " << offset << " " << limit << " " << messages.size() << " " << last_received_message_id << " "
          << d->debug_set_dialog_last_database_message_id;
      CHECK(d->first_database_message_id.is_valid());
      {
        MessagesConstIterator it(d, d->first_database_message_id);
        if (*it != nullptr && ((*it)->message_id == d->first_database_message_id || (*it)->have_next)) {
          while (*it != nullptr) {
            auto message_id = (*it)->message_id;
            if ((message_id.is_server() || message_id.is_local()) && message_id < d->first_database_message_id) {
              set_dialog_first_database_message_id(d, message_id, "on_get_history 2");
              try_restore_dialog_reply_markup(d, *it);
              is_dialog_updated = true;
            }
            --it;
          }
        }
      }
      {
        MessagesConstIterator it(d, d->last_database_message_id);
        if (*it != nullptr && ((*it)->message_id == d->last_database_message_id || (*it)->have_next)) {
          while (*it != nullptr) {
            auto message_id = (*it)->message_id;
            if ((message_id.is_server() || message_id.is_local()) && message_id > d->last_database_message_id) {
              set_dialog_last_database_message_id(d, message_id, "on_get_history 2");
              is_dialog_updated = true;
            }
            ++it;
          }
        }
      }
    }
    LOG_CHECK(d->first_database_message_id.is_valid())
        << dialog_id << " " << from_the_end << " " << d->first_database_message_id << " " << d->last_database_message_id
        << " " << first_added_message_id << " " << last_added_message_id << " " << d->last_message_id << " "
        << d->last_new_message_id << " " << d->have_full_history << " " << prev_last_new_message_id << " "
        << prev_first_database_message_id << " " << prev_last_database_message_id << " " << prev_last_message_id << " "
        << prev_have_full_history << " " << d->debug_last_new_message_id << " " << d->debug_first_database_message_id
        << " " << d->debug_last_database_message_id << " " << from_message_id << " " << offset << " " << limit << " "
        << messages.size() << " " << last_received_message_id << " " << d->debug_set_dialog_last_database_message_id;

    CHECK(d->last_database_message_id.is_valid());

    for (auto &first_message_id : d->first_database_message_id_by_index) {
      if (first_added_message_id < first_message_id && first_message_id <= last_added_message_id) {
        first_message_id = first_added_message_id;
      }
    }

    if (is_dialog_updated) {
      on_dialog_updated(dialog_id, "on_get_history");
    }
  }
}

vector<DialogId> MessagesManager::get_peers_dialog_ids(vector<tl_object_ptr<telegram_api::Peer>> &&peers) {
  vector<DialogId> result;
  result.reserve(peers.size());
  for (auto &peer : peers) {
    DialogId dialog_id(peer);
    if (dialog_id.is_valid()) {
      force_create_dialog(dialog_id, "get_peers_dialog_ids");
      result.push_back(dialog_id);
    }
  }
  return result;
}

void MessagesManager::on_get_public_dialogs_search_result(const string &query,
                                                          vector<tl_object_ptr<telegram_api::Peer>> &&my_peers,
                                                          vector<tl_object_ptr<telegram_api::Peer>> &&peers) {
  auto it = search_public_dialogs_queries_.find(query);
  CHECK(it != search_public_dialogs_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  search_public_dialogs_queries_.erase(it);

  found_public_dialogs_[query] = get_peers_dialog_ids(std::move(peers));
  found_on_server_dialogs_[query] = get_peers_dialog_ids(std::move(my_peers));

  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

void MessagesManager::on_failed_public_dialogs_search(const string &query, Status &&error) {
  auto it = search_public_dialogs_queries_.find(query);
  CHECK(it != search_public_dialogs_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  search_public_dialogs_queries_.erase(it);

  found_public_dialogs_[query];     // negative cache
  found_on_server_dialogs_[query];  // negative cache

  for (auto &promise : promises) {
    promise.set_error(error.clone());
  }
}

void MessagesManager::on_get_dialog_messages_search_result(DialogId dialog_id, const string &query,
                                                           UserId sender_user_id, MessageId from_message_id,
                                                           int32 offset, int32 limit, SearchMessagesFilter filter,
                                                           int64 random_id, int32 total_count,
                                                           vector<tl_object_ptr<telegram_api::Message>> &&messages) {
  LOG(INFO) << "Receive " << messages.size() << " found messages in " << dialog_id;
  if (!dialog_id.is_valid()) {
    CHECK(query.empty());
    CHECK(!sender_user_id.is_valid());
    auto it = found_call_messages_.find(random_id);
    CHECK(it != found_call_messages_.end());

    MessageId first_added_message_id;
    if (messages.empty()) {
      // messages may be empty because there is no more messages or they can't be found due to global limit
      // anyway pretend that there is no more messages
      first_added_message_id = MessageId::min();
    }

    auto &result = it->second.second;
    CHECK(result.empty());
    for (auto &message : messages) {
      auto new_full_message_id =
          on_get_message(std::move(message), false, false, false, false, false, "search call messages");
      if (new_full_message_id != FullMessageId()) {
        result.push_back(new_full_message_id);
      }

      auto message_id = new_full_message_id.get_message_id();
      if (message_id < first_added_message_id || !first_added_message_id.is_valid()) {
        first_added_message_id = message_id;
      }
    }
    if (G()->parameters().use_message_db) {
      bool update_state = false;

      auto &old_message_count = calls_db_state_.message_count_by_index[search_calls_filter_index(filter)];
      if (old_message_count != total_count) {
        LOG(INFO) << "Update calls database message count to " << total_count;
        old_message_count = total_count;
        update_state = true;
      }

      auto &old_first_db_message_id =
          calls_db_state_.first_calls_database_message_id_by_index[search_calls_filter_index(filter)];
      bool from_the_end = !from_message_id.is_valid() || from_message_id >= MessageId::max();
      LOG(INFO) << "Have from_the_end = " << from_the_end << ", old_first_db_message_id = " << old_first_db_message_id
                << ", first_added_message_id = " << first_added_message_id << ", from_message_id = " << from_message_id;
      if ((from_the_end || (old_first_db_message_id.is_valid() && old_first_db_message_id <= from_message_id)) &&
          (!old_first_db_message_id.is_valid() || first_added_message_id < old_first_db_message_id)) {
        LOG(INFO) << "Update calls database first message to " << first_added_message_id;
        old_first_db_message_id = first_added_message_id;
        update_state = true;
      }
      if (update_state) {
        save_calls_db_state();
      }
    }
    it->second.first = total_count;
    return;
  }

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());

  auto &result = it->second.second;
  CHECK(result.empty());
  MessageId first_added_message_id;
  if (messages.empty()) {
    // messages may be empty because there is no more messages or they can't be found due to global limit
    // anyway pretend that there is no more messages
    first_added_message_id = MessageId::min();
  }
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  for (auto &message : messages) {
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "SearchMessagesQuery");
    if (new_full_message_id == FullMessageId()) {
      total_count--;
      continue;
    }

    if (new_full_message_id.get_dialog_id() != dialog_id) {
      LOG(ERROR) << "Receive " << new_full_message_id << " instead of a message in " << dialog_id;
      total_count--;
      continue;
    }

    auto message_id = new_full_message_id.get_message_id();
    if (filter == SearchMessagesFilter::UnreadMention && message_id <= d->last_read_all_mentions_message_id) {
      total_count--;
      continue;
    }

    // TODO check that messages are returned in decreasing message_id order
    if (message_id < first_added_message_id || !first_added_message_id.is_valid()) {
      first_added_message_id = message_id;
    }
    result.push_back(message_id);
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  if (query.empty() && !sender_user_id.is_valid() && filter != SearchMessagesFilter::Empty &&
      G()->parameters().use_message_db) {
    bool update_dialog = false;

    auto &old_message_count = d->message_count_by_index[search_messages_filter_index(filter)];
    if (old_message_count != total_count) {
      old_message_count = total_count;
      if (filter == SearchMessagesFilter::UnreadMention) {
        d->unread_mention_count = old_message_count;
        update_dialog_mention_notification_count(d);
        send_update_chat_unread_mention_count(d);
      }
      update_dialog = true;
    }

    auto &old_first_db_message_id = d->first_database_message_id_by_index[search_messages_filter_index(filter)];
    bool from_the_end = !from_message_id.is_valid() ||
                        (d->last_message_id != MessageId() && from_message_id > d->last_message_id) ||
                        from_message_id >= MessageId::max();
    if ((from_the_end || (old_first_db_message_id.is_valid() && old_first_db_message_id <= from_message_id)) &&
        (!old_first_db_message_id.is_valid() || first_added_message_id < old_first_db_message_id)) {
      old_first_db_message_id = first_added_message_id;
      update_dialog = true;
    }
    if (update_dialog) {
      on_dialog_updated(dialog_id, "search results");
    }
  }
  it->second.first = total_count;
}

void MessagesManager::on_failed_dialog_messages_search(DialogId dialog_id, int64 random_id) {
  if (!dialog_id.is_valid()) {
    auto it = found_call_messages_.find(random_id);
    CHECK(it != found_call_messages_.end());
    found_call_messages_.erase(it);
    return;
  }

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());
  found_dialog_messages_.erase(it);
}

void MessagesManager::on_get_messages_search_result(const string &query, int32 offset_date, DialogId offset_dialog_id,
                                                    MessageId offset_message_id, int32 limit, int64 random_id,
                                                    int32 total_count,
                                                    vector<tl_object_ptr<telegram_api::Message>> &&messages) {
  LOG(INFO) << "Receive " << messages.size() << " found messages";
  auto it = found_messages_.find(random_id);
  CHECK(it != found_messages_.end());

  auto &result = it->second.second;
  CHECK(result.empty());
  for (auto &message : messages) {
    auto dialog_id = get_message_dialog_id(message);
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "search messages");
    if (new_full_message_id != FullMessageId()) {
      CHECK(dialog_id == new_full_message_id.get_dialog_id());
      result.push_back(new_full_message_id);
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  it->second.first = total_count;
}

void MessagesManager::on_failed_messages_search(int64 random_id) {
  auto it = found_messages_.find(random_id);
  CHECK(it != found_messages_.end());
  found_messages_.erase(it);
}

void MessagesManager::on_get_scheduled_server_messages(DialogId dialog_id, uint32 generation,
                                                       vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                                       bool is_not_modified) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (generation < d->scheduled_messages_sync_generation) {
    return;
  }
  d->scheduled_messages_sync_generation = generation;

  if (is_not_modified) {
    return;
  }

  vector<MessageId> old_message_ids;
  find_old_messages(d->scheduled_messages.get(),
                    MessageId(ScheduledServerMessageId(), std::numeric_limits<int32>::max(), true), old_message_ids);
  std::unordered_map<ScheduledServerMessageId, MessageId, ScheduledServerMessageIdHash> old_server_message_ids;
  for (auto &message_id : old_message_ids) {
    if (message_id.is_scheduled_server()) {
      old_server_message_ids[message_id.get_scheduled_server_message_id()] = message_id;
    }
  }

  bool is_channel_message = dialog_id.get_type() == DialogType::Channel;
  for (auto &message : messages) {
    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      if (dialog_id.is_valid()) {
        LOG(ERROR) << "Receive " << get_message_id(message, true) << " in wrong " << message_dialog_id << " instead of "
                   << dialog_id << ": " << oneline(to_string(message));
      }
      continue;
    }

    auto full_message_id = on_get_message(std::move(message), false, is_channel_message, true, false, false,
                                          "on_get_scheduled_server_messages");
    auto message_id = full_message_id.get_message_id();
    if (message_id.is_valid_scheduled()) {
      CHECK(message_id.is_scheduled_server());
      old_server_message_ids.erase(message_id.get_scheduled_server_message_id());
    }
  }

  for (auto it : old_server_message_ids) {
    auto message_id = it.second;
    auto message = do_delete_scheduled_message(d, message_id, true, "on_get_scheduled_server_messages");
    CHECK(message != nullptr);
    send_update_delete_messages(dialog_id, {message->message_id.get()}, true, false);
  }

  send_update_chat_has_scheduled_messages(d);
}

void MessagesManager::on_get_recent_locations(DialogId dialog_id, int32 limit, int64 random_id, int32 total_count,
                                              vector<tl_object_ptr<telegram_api::Message>> &&messages) {
  LOG(INFO) << "Receive " << messages.size() << " recent locations in " << dialog_id;
  auto it = found_dialog_recent_location_messages_.find(random_id);
  CHECK(it != found_dialog_recent_location_messages_.end());

  auto &result = it->second.second;
  CHECK(result.empty());
  for (auto &message : messages) {
    auto new_full_message_id = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel,
                                              false, false, false, "get recent locations");
    if (new_full_message_id != FullMessageId()) {
      if (new_full_message_id.get_dialog_id() != dialog_id) {
        LOG(ERROR) << "Receive " << new_full_message_id << " instead of a message in " << dialog_id;
        total_count--;
        continue;
      }
      auto m = get_message(new_full_message_id);
      CHECK(m != nullptr);
      if (m->content->get_type() != MessageContentType::LiveLocation) {
        LOG(ERROR) << "Receive a message of wrong type " << m->content->get_type() << " in on_get_recent_locations in "
                   << dialog_id;
        total_count--;
        continue;
      }

      result.push_back(m->message_id);
    } else {
      total_count--;
    }
  }
  if (total_count < static_cast<int32>(result.size())) {
    LOG(ERROR) << "Receive " << result.size() << " valid messages out of " << total_count << " in " << messages.size()
               << " messages";
    total_count = static_cast<int32>(result.size());
  }
  it->second.first = total_count;
}

void MessagesManager::on_get_recent_locations_failed(int64 random_id) {
  auto it = found_dialog_recent_location_messages_.find(random_id);
  CHECK(it != found_dialog_recent_location_messages_.end());
  found_dialog_recent_location_messages_.erase(it);
}

void MessagesManager::delete_messages_from_updates(const vector<MessageId> &message_ids) {
  std::unordered_map<DialogId, vector<int64>, DialogIdHash> deleted_message_ids;
  std::unordered_map<DialogId, bool, DialogIdHash> need_update_dialog_pos;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() || !message_id.is_server()) {
      LOG(ERROR) << "Incoming update tries to delete " << message_id;
      continue;
    }

    Dialog *d = get_dialog_by_message_id(message_id);
    if (d != nullptr) {
      auto message = delete_message(d, message_id, true, &need_update_dialog_pos[d->dialog_id], "updates");
      CHECK(message != nullptr);
      LOG_CHECK(message->message_id == message_id) << message_id << " " << message->message_id << " " << d->dialog_id;
      deleted_message_ids[d->dialog_id].push_back(message->message_id.get());
    }
    if (last_clear_history_message_id_to_dialog_id_.count(message_id)) {
      d = get_dialog(last_clear_history_message_id_to_dialog_id_[message_id]);
      CHECK(d != nullptr);
      auto message = delete_message(d, message_id, true, &need_update_dialog_pos[d->dialog_id], "updates");
      CHECK(message == nullptr);
    }
  }
  for (auto &it : need_update_dialog_pos) {
    if (it.second) {
      auto dialog_id = it.first;
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      send_update_chat_last_message(d, "delete_messages_from_updates");
    }
  }
  for (auto &it : deleted_message_ids) {
    auto dialog_id = it.first;
    send_update_delete_messages(dialog_id, std::move(it.second), true, false);
  }
}

void MessagesManager::delete_dialog_messages_from_updates(DialogId dialog_id, const vector<MessageId> &message_ids) {
  CHECK(dialog_id.get_type() == DialogType::Channel || dialog_id.get_type() == DialogType::SecretChat);
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Ignore deleteChannelMessages for unknown " << dialog_id;
    CHECK(dialog_id.get_type() == DialogType::Channel);
    return;
  }

  vector<int64> deleted_message_ids;
  bool need_update_dialog_pos = false;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() || (!message_id.is_server() && dialog_id.get_type() != DialogType::SecretChat)) {
      LOG(ERROR) << "Incoming update tries to delete " << message_id;
      continue;
    }

    auto message = delete_message(d, message_id, true, &need_update_dialog_pos, "updates");
    deleted_message_ids.push_back(message == nullptr ? message_id.get() : message->message_id.get());
  }
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "delete_dialog_messages_from_updates");
  }
  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);
}

string MessagesManager::get_search_text(const Message *m) const {
  if (m->is_content_secret) {
    return string();
  }
  return get_message_content_search_text(td_, m->content.get());
}

bool MessagesManager::can_forward_message(DialogId from_dialog_id, const Message *m) {
  if (m == nullptr) {
    return false;
  }
  if (m->ttl > 0) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  switch (from_dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }

  return can_forward_message_content(m->content.get());
}

bool MessagesManager::can_delete_channel_message(DialogParticipantStatus status, const Message *m, bool is_bot) {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local() || m->message_id.is_yet_unsent()) {
    return true;
  }
  if (m->message_id.is_scheduled()) {
    if (m->is_channel_post) {
      return status.can_post_messages();
    }
    return true;
  }

  if (is_bot && G()->unix_time_cached() >= m->date + 2 * 86400) {
    // bots can't delete messages older than 2 days
    return false;
  }

  CHECK(m->message_id.is_server());
  if (m->message_id.get_server_message_id().get() == 1) {
    return false;
  }
  auto content_type = m->content->get_type();
  if (content_type == MessageContentType::ChannelMigrateFrom || content_type == MessageContentType::ChannelCreate) {
    return false;
  }

  if (status.can_delete_messages()) {
    return true;
  }

  if (!m->is_outgoing) {
    return false;
  }

  if (m->is_channel_post || is_service_message_content(content_type)) {
    return status.can_post_messages();
  }

  return true;
}

bool MessagesManager::can_revoke_message(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return true;
  }
  if (m->message_id.is_local()) {
    return false;
  }
  if (dialog_id == get_my_dialog_id()) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  if (m->message_id.is_yet_unsent()) {
    return true;
  }
  CHECK(m->message_id.is_server());

  const int32 DEFAULT_REVOKE_TIME_LIMIT = td_->auth_manager_->is_bot() ? 2 * 86400 : std::numeric_limits<int32>::max();
  auto content_type = m->content->get_type();
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      bool can_revoke_incoming = G()->shared_config().get_option_boolean("revoke_pm_inbox", true);
      int32 revoke_time_limit =
          G()->shared_config().get_option_integer("revoke_pm_time_limit", DEFAULT_REVOKE_TIME_LIMIT);

      return ((m->is_outgoing && !is_service_message_content(content_type)) ||
              (can_revoke_incoming && content_type != MessageContentType::ScreenshotTaken)) &&
             G()->unix_time_cached() - m->date <= revoke_time_limit;
    }
    case DialogType::Chat: {
      bool is_appointed_administrator =
          td_->contacts_manager_->is_appointed_chat_administrator(dialog_id.get_chat_id());
      int32 revoke_time_limit = G()->shared_config().get_option_integer("revoke_time_limit", DEFAULT_REVOKE_TIME_LIMIT);

      return ((m->is_outgoing && !is_service_message_content(content_type)) || is_appointed_administrator) &&
             G()->unix_time_cached() - m->date <= revoke_time_limit;
    }
    case DialogType::Channel:
      return true;  // any server message that can be deleted will be deleted for all participants
    case DialogType::SecretChat:
      // all non-service messages will be deleted for everyone if secret chat is active
      return td_->contacts_manager_->get_secret_chat_state(dialog_id.get_secret_chat_id()) == SecretChatState::Active &&
             !is_service_message_content(content_type);
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::delete_messages(DialogId dialog_id, const vector<MessageId> &input_message_ids, bool revoke,
                                      Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(6, "Chat is not found"));
  }

  if (input_message_ids.empty()) {
    return promise.set_value(Unit());
  }

  auto dialog_type = dialog_id.get_type();
  bool is_secret = dialog_type == DialogType::SecretChat;

  vector<MessageId> message_ids;
  message_ids.reserve(input_message_ids.size());
  vector<MessageId> deleted_server_message_ids;

  vector<MessageId> deleted_scheduled_server_message_ids;
  for (auto message_id : input_message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      return promise.set_error(Status::Error(6, "Invalid message identifier"));
    }

    message_id = get_persistent_message_id(d, message_id);
    message_ids.push_back(message_id);
    auto message = get_message_force(d, message_id, "delete_messages");
    if (message != nullptr) {
      if (message->message_id.is_scheduled()) {
        if (message->message_id.is_scheduled_server()) {
          deleted_scheduled_server_message_ids.push_back(message->message_id);
        }
      } else {
        if (message->message_id.is_server() || is_secret) {
          deleted_server_message_ids.push_back(message->message_id);
        }
      }
    }
  }

  bool is_bot = td_->auth_manager_->is_bot();
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      if (is_bot) {
        for (auto message_id : message_ids) {
          if (!message_id.is_scheduled() && message_id.is_server() &&
              !can_revoke_message(dialog_id, get_message(d, message_id))) {
            return promise.set_error(Status::Error(6, "Message can't be deleted"));
          }
        }
      }
      break;
    case DialogType::Channel: {
      auto dialog_status = td_->contacts_manager_->get_channel_permissions(dialog_id.get_channel_id());
      for (auto message_id : message_ids) {
        if (!can_delete_channel_message(dialog_status, get_message(d, message_id), is_bot)) {
          return promise.set_error(Status::Error(6, "Message can't be deleted"));
        }
      }
      break;
    }
    case DialogType::SecretChat:
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  MultiPromiseActorSafe mpas{"DeleteMessagesFromServerMultiPromiseActor"};
  mpas.add_promise(std::move(promise));

  auto lock = mpas.get_promise();
  delete_messages_from_server(dialog_id, std::move(deleted_server_message_ids), revoke, 0, mpas.get_promise());
  delete_scheduled_messages_from_server(dialog_id, std::move(deleted_scheduled_server_message_ids), 0,
                                        mpas.get_promise());
  lock.set_value(Unit());

  bool need_update_dialog_pos = false;
  vector<int64> deleted_message_ids;
  for (auto message_id : message_ids) {
    auto m = delete_message(d, message_id, true, &need_update_dialog_pos, DELETE_MESSAGE_USER_REQUEST_SOURCE);
    if (m == nullptr) {
      LOG(INFO) << "Can't delete " << message_id << " because it is not found";
    } else {
      deleted_message_ids.push_back(m->message_id.get());
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "delete_messages");
  }
  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  send_update_chat_has_scheduled_messages(d);
}

void MessagesManager::delete_message_from_server(DialogId dialog_id, MessageId message_id, bool revoke) {
  if (message_id.is_valid()) {
    CHECK(message_id.is_server());
    delete_messages_from_server(dialog_id, {message_id}, revoke, 0, Auto());
  } else {
    CHECK(message_id.is_scheduled_server());
    delete_scheduled_messages_from_server(dialog_id, {message_id}, 0, Auto());
  }
}

class MessagesManager::DeleteMessagesFromServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_delete_messages_from_server_logevent(DialogId dialog_id,
                                                                  const vector<MessageId> &message_ids, bool revoke) {
  DeleteMessagesFromServerLogEvent logevent{dialog_id, message_ids, revoke};
  auto storer = LogEventStorerImpl<DeleteMessagesFromServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteMessagesFromServer, storer);
}

void MessagesManager::delete_messages_from_server(DialogId dialog_id, vector<MessageId> message_ids, bool revoke,
                                                  uint64 logevent_id, Promise<Unit> &&promise) {
  if (message_ids.empty()) {
    return promise.set_value(Unit());
  }
  LOG(INFO) << (revoke ? "Revoke " : "Delete ") << format::as_array(message_ids) << " in " << dialog_id
            << " from server";

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_delete_messages_from_server_logevent(dialog_id, message_ids, revoke);
  }

  auto new_promise = get_erase_logevent_promise(logevent_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<DeleteMessagesQuery>(std::move(promise))->send(std::move(message_ids), revoke);
      break;
    case DialogType::Channel:
      td_->create_handler<DeleteChannelMessagesQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), std::move(message_ids));
      break;
    case DialogType::SecretChat: {
      vector<int64> random_ids;
      auto d = get_dialog_force(dialog_id);
      CHECK(d != nullptr);
      for (auto &message_id : message_ids) {
        auto *m = get_message(d, message_id);
        if (m != nullptr) {
          random_ids.push_back(m->random_id);
        }
      }
      if (!random_ids.empty()) {
        send_closure(G()->secret_chats_manager(), &SecretChatsManager::delete_messages, dialog_id.get_secret_chat_id(),
                     std::move(random_ids), std::move(promise));
      } else {
        promise.set_value(Unit());
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

class MessagesManager::DeleteScheduledMessagesFromServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_delete_scheduled_messages_from_server_logevent(DialogId dialog_id,
                                                                            const vector<MessageId> &message_ids) {
  DeleteScheduledMessagesFromServerLogEvent logevent{dialog_id, message_ids};
  auto storer = LogEventStorerImpl<DeleteScheduledMessagesFromServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteScheduledMessagesFromServer, storer);
}

void MessagesManager::delete_scheduled_messages_from_server(DialogId dialog_id, vector<MessageId> message_ids,
                                                            uint64 logevent_id, Promise<Unit> &&promise) {
  if (message_ids.empty()) {
    return promise.set_value(Unit());
  }
  LOG(INFO) << "Delete " << format::as_array(message_ids) << " in " << dialog_id << " from server";

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_delete_scheduled_messages_from_server_logevent(dialog_id, message_ids);
  }

  auto new_promise = get_erase_logevent_promise(logevent_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  td_->create_handler<DeleteScheduledMessagesQuery>(std::move(promise))->send(dialog_id, std::move(message_ids));
}

void MessagesManager::delete_dialog_history(DialogId dialog_id, bool remove_from_dialog_list, bool revoke,
                                            Promise<Unit> &&promise) {
  LOG(INFO) << "Receive deleteChatHistory request to delete all messages in " << dialog_id
            << ", remove_from_chat_list is " << remove_from_dialog_list << ", revoke is " << revoke;

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(3, "Chat info not found"));
  }

  auto dialog_type = dialog_id.get_type();
  bool is_secret = false;
  switch (dialog_type) {
    case DialogType::User:
    case DialogType::Chat:
      // ok
      break;
    case DialogType::Channel:
      if (is_broadcast_channel(dialog_id)) {
        return promise.set_error(Status::Error(3, "Can't delete chat history in a channel"));
      }
      if (td_->contacts_manager_->is_channel_public(dialog_id.get_channel_id())) {
        return promise.set_error(Status::Error(3, "Can't delete chat history in a public supergroup"));
      }
      break;
    case DialogType::SecretChat:
      is_secret = true;
      // ok
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }

  auto last_new_message_id = d->last_new_message_id;
  if (!is_secret && last_new_message_id == MessageId()) {
    // TODO get dialog from the server and delete history from last message id
  }

  bool allow_error = d->messages == nullptr;

  delete_all_dialog_messages(d, remove_from_dialog_list, true);

  if (last_new_message_id.is_valid() && last_new_message_id == d->max_unavailable_message_id && !revoke) {
    // history has already been cleared, nothing to do
    promise.set_value(Unit());
    return;
  }

  set_dialog_max_unavailable_message_id(dialog_id, last_new_message_id, false, "delete_dialog_history");

  delete_dialog_history_from_server(dialog_id, last_new_message_id, remove_from_dialog_list, revoke, allow_error, 0,
                                    std::move(promise));
}

class MessagesManager::DeleteDialogHistoryFromServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId max_message_id_;
  bool remove_from_dialog_list_;
  bool revoke_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(remove_from_dialog_list_);
    STORE_FLAG(revoke_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(remove_from_dialog_list_);
    PARSE_FLAG(revoke_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

uint64 MessagesManager::save_delete_dialog_history_from_server_logevent(DialogId dialog_id, MessageId max_message_id,
                                                                        bool remove_from_dialog_list, bool revoke) {
  DeleteDialogHistoryFromServerLogEvent logevent{dialog_id, max_message_id, remove_from_dialog_list, revoke};
  auto storer = LogEventStorerImpl<DeleteDialogHistoryFromServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteDialogHistoryFromServer, storer);
}

void MessagesManager::delete_dialog_history_from_server(DialogId dialog_id, MessageId max_message_id,
                                                        bool remove_from_dialog_list, bool revoke, bool allow_error,
                                                        uint64 logevent_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Delete history in " << dialog_id << " up to " << max_message_id << " from server";

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id =
        save_delete_dialog_history_from_server_logevent(dialog_id, max_message_id, remove_from_dialog_list, revoke);
  }

  auto new_promise = get_erase_logevent_promise(logevent_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<DeleteHistoryQuery>(std::move(promise))
          ->send(dialog_id, max_message_id, remove_from_dialog_list, revoke);
      break;
    case DialogType::Channel:
      td_->create_handler<DeleteChannelHistoryQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), max_message_id, allow_error);
      break;
    case DialogType::SecretChat:
      send_closure(G()->secret_chats_manager(), &SecretChatsManager::delete_all_messages,
                   dialog_id.get_secret_chat_id(), std::move(promise));
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
}

void MessagesManager::find_messages_from_user(const Message *m, UserId user_id, vector<MessageId> &message_ids) {
  if (m == nullptr) {
    return;
  }

  find_messages_from_user(m->left.get(), user_id, message_ids);

  if (m->sender_user_id == user_id) {
    message_ids.push_back(m->message_id);
  }

  find_messages_from_user(m->right.get(), user_id, message_ids);
}

void MessagesManager::find_unread_mentions(const Message *m, vector<MessageId> &message_ids) {
  if (m == nullptr) {
    return;
  }

  find_unread_mentions(m->left.get(), message_ids);

  if (m->contains_unread_mention) {
    message_ids.push_back(m->message_id);
  }

  find_unread_mentions(m->right.get(), message_ids);
}

void MessagesManager::find_old_messages(const Message *m, MessageId max_message_id, vector<MessageId> &message_ids) {
  if (m == nullptr) {
    return;
  }

  find_old_messages(m->left.get(), max_message_id, message_ids);

  if (m->message_id <= max_message_id) {
    message_ids.push_back(m->message_id);

    find_old_messages(m->right.get(), max_message_id, message_ids);
  }
}

void MessagesManager::find_unloadable_messages(const Dialog *d, int32 unload_before_date, const Message *m,
                                               vector<MessageId> &message_ids, int32 &left_to_unload) const {
  if (m == nullptr) {
    return;
  }

  find_unloadable_messages(d, unload_before_date, m->left.get(), message_ids, left_to_unload);

  if (can_unload_message(d, m)) {
    if (m->last_access_date <= unload_before_date) {
      message_ids.push_back(m->message_id);
    } else {
      left_to_unload++;
    }
  }

  find_unloadable_messages(d, unload_before_date, m->right.get(), message_ids, left_to_unload);
}

void MessagesManager::delete_dialog_messages_from_user(DialogId dialog_id, UserId user_id, Promise<Unit> &&promise) {
  bool is_bot = td_->auth_manager_->is_bot();
  if (is_bot) {
    return promise.set_error(Status::Error(3, "Method is not available for bots"));
  }

  LOG(INFO) << "Receive deleteChatMessagesFromUser request to delete all messages in " << dialog_id << " from the user "
            << user_id;
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Write)) {
    return promise.set_error(Status::Error(3, "Not enough rights"));
  }

  if (!td_->contacts_manager_->have_input_user(user_id)) {
    return promise.set_error(Status::Error(3, "User not found"));
  }

  ChannelId channel_id;
  DialogParticipantStatus channel_status = DialogParticipantStatus::Left();
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::SecretChat:
      return promise.set_error(Status::Error(3, "All messages from a user can be deleted only in supergroup chats"));
    case DialogType::Channel: {
      channel_id = dialog_id.get_channel_id();
      auto channel_type = td_->contacts_manager_->get_channel_type(channel_id);
      if (channel_type != ChannelType::Megagroup) {
        return promise.set_error(Status::Error(3, "The method is available only for supergroup chats"));
      }
      channel_status = td_->contacts_manager_->get_channel_permissions(channel_id);
      if (!channel_status.can_delete_messages()) {
        return promise.set_error(Status::Error(5, "Need delete messages administator right in the supergroup chat"));
      }
      channel_id = dialog_id.get_channel_id();
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
      break;
  }
  CHECK(channel_id.is_valid());

  if (G()->parameters().use_message_db) {
    LOG(INFO) << "Delete all messages from " << user_id << " in " << dialog_id << " from database";
    G()->td_db()->get_messages_db_async()->delete_dialog_messages_from_user(dialog_id, user_id,
                                                                            Auto());  // TODO Promise
  }

  vector<MessageId> message_ids;
  find_messages_from_user(d->messages.get(), user_id, message_ids);

  vector<int64> deleted_message_ids;
  bool need_update_dialog_pos = false;
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->sender_user_id == user_id);
    CHECK(m->message_id == message_id);
    if (can_delete_channel_message(channel_status, m, is_bot)) {
      auto p = delete_message(d, message_id, true, &need_update_dialog_pos, "delete messages from user");
      CHECK(p.get() == m);
      deleted_message_ids.push_back(p->message_id.get());
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "delete_messages_from_user");
  }

  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);

  delete_all_channel_messages_from_user_on_server(channel_id, user_id, 0, std::move(promise));
}

class MessagesManager::DeleteAllChannelMessagesFromUserOnServerLogEvent {
 public:
  ChannelId channel_id_;
  UserId user_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(channel_id_, storer);
    td::store(user_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(channel_id_, parser);
    td::parse(user_id_, parser);
  }
};

uint64 MessagesManager::save_delete_all_channel_messages_from_user_on_server_logevent(ChannelId channel_id,
                                                                                      UserId user_id) {
  DeleteAllChannelMessagesFromUserOnServerLogEvent logevent{channel_id, user_id};
  auto storer = LogEventStorerImpl<DeleteAllChannelMessagesFromUserOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::DeleteAllChannelMessagesFromUserOnServer,
                    storer);
}

void MessagesManager::delete_all_channel_messages_from_user_on_server(ChannelId channel_id, UserId user_id,
                                                                      uint64 logevent_id, Promise<Unit> &&promise) {
  if (logevent_id == 0 && G()->parameters().use_chat_info_db) {
    logevent_id = save_delete_all_channel_messages_from_user_on_server_logevent(channel_id, user_id);
  }

  td_->create_handler<DeleteUserHistoryQuery>(get_erase_logevent_promise(logevent_id, std::move(promise)))
      ->send(channel_id, user_id);
}

int32 MessagesManager::get_unload_dialog_delay() const {
  constexpr int32 DIALOG_UNLOAD_DELAY = 60;       // seconds
  constexpr int32 DIALOG_UNLOAD_BOT_DELAY = 600;  // seconds
  return td_->auth_manager_->is_bot() ? DIALOG_UNLOAD_BOT_DELAY : DIALOG_UNLOAD_DELAY;
}

void MessagesManager::unload_dialog(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  vector<MessageId> to_unload_message_ids;
  int32 left_to_unload = 0;
  find_unloadable_messages(d, G()->unix_time_cached() - get_unload_dialog_delay() + 2, d->messages.get(),
                           to_unload_message_ids, left_to_unload);

  vector<int64> unloaded_message_ids;
  for (auto message_id : to_unload_message_ids) {
    unload_message(d, message_id);
    unloaded_message_ids.push_back(message_id.get());
  }

  if (!unloaded_message_ids.empty()) {
    if (!G()->parameters().use_message_db) {
      d->have_full_history = false;
    }

    send_closure_later(
        G()->td(), &Td::send_update,
        make_tl_object<td_api::updateDeleteMessages>(dialog_id.get(), std::move(unloaded_message_ids), false, true));
  }

  if (left_to_unload > 0) {
    LOG(DEBUG) << "Need to unload " << left_to_unload << " messages more in " << dialog_id;
    pending_unload_dialog_timeout_.add_timeout_in(d->dialog_id.get(), get_unload_dialog_delay());
  }
}

void MessagesManager::delete_all_dialog_messages(Dialog *d, bool remove_from_dialog_list, bool is_permanently_deleted) {
  CHECK(d != nullptr);
  LOG(INFO) << "Delete all messages in " << d->dialog_id
            << " with remove_from_dialog_list = " << remove_from_dialog_list
            << " and is_permanently_deleted = " << is_permanently_deleted;
  if (is_debug_message_op_enabled()) {
    d->debug_message_op.emplace_back(Dialog::MessageOp::DeleteAll, MessageId(), MessageContentType::None,
                                     remove_from_dialog_list, false, false, "");
  }

  if (d->server_unread_count + d->local_unread_count > 0) {
    MessageId max_message_id =
        d->last_database_message_id.is_valid() ? d->last_database_message_id : d->last_new_message_id;
    if (max_message_id.is_valid()) {
      read_history_inbox(d->dialog_id, max_message_id, -1, "delete_all_dialog_messages");
    }
    if (d->server_unread_count != 0 || d->local_unread_count != 0) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), 0, 0, true, "delete_all_dialog_messages");
    }
  }

  if (d->unread_mention_count > 0) {
    d->unread_mention_count = 0;
    d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] = 0;
    send_update_chat_unread_mention_count(d);
  }

  bool has_last_message_id = d->last_message_id != MessageId();
  int32 last_message_date = 0;
  MessageId last_clear_history_message_id;
  if (!remove_from_dialog_list) {
    if (has_last_message_id) {
      auto m = get_message(d, d->last_message_id);
      CHECK(m != nullptr);
      last_message_date = m->date;
      last_clear_history_message_id = d->last_message_id;
    } else {
      last_message_date = d->last_clear_history_date;
      last_clear_history_message_id = d->last_clear_history_message_id;
    }
  }

  vector<int64> deleted_message_ids;
  do_delete_all_dialog_messages(d, d->messages, is_permanently_deleted, deleted_message_ids);
  delete_all_dialog_messages_from_database(d, MessageId::max(), "delete_all_dialog_messages");
  if (is_permanently_deleted) {
    for (auto id : deleted_message_ids) {
      d->deleted_message_ids.insert(MessageId{id});
    }
  }

  if (d->reply_markup_message_id != MessageId()) {
    set_dialog_reply_markup(d, MessageId());
  }

  set_dialog_first_database_message_id(d, MessageId(), "delete_all_dialog_messages");
  set_dialog_last_database_message_id(d, MessageId(), "delete_all_dialog_messages");
  set_dialog_last_clear_history_date(d, last_message_date, last_clear_history_message_id, "delete_all_dialog_messages");
  d->last_read_all_mentions_message_id = MessageId();                            // it is not needed anymore
  d->message_notification_group.max_removed_notification_id = NotificationId();  // it is not needed anymore
  d->message_notification_group.max_removed_message_id = MessageId();            // it is not needed anymore
  d->mention_notification_group.max_removed_notification_id = NotificationId();  // it is not needed anymore
  d->mention_notification_group.max_removed_message_id = MessageId();            // it is not needed anymore
  std::fill(d->message_count_by_index.begin(), d->message_count_by_index.end(), 0);
  d->notification_id_to_message_id.clear();

  if (has_last_message_id) {
    set_dialog_last_message_id(d, MessageId(), "delete_all_dialog_messages");
    send_update_chat_last_message(d, "delete_all_dialog_messages");
  }
  if (remove_from_dialog_list && d->pinned_order != DEFAULT_ORDER) {
    set_dialog_is_pinned(d, false);
  }
  update_dialog_pos(d, remove_from_dialog_list, "delete_all_dialog_messages");

  on_dialog_updated(d->dialog_id, "delete_all_dialog_messages");

  send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), is_permanently_deleted, false);
}

void MessagesManager::delete_dialog(DialogId dialog_id) {
  LOG(INFO) << "Delete " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return;
  }

  delete_all_dialog_messages(d, true, false);
  if (dialog_id.get_type() != DialogType::SecretChat) {
    d->have_full_history = false;
    d->need_restore_reply_markup = true;
  }
  if (remove_recently_found_dialog_internal(dialog_id)) {
    save_recently_found_dialogs();
  }

  close_dialog(d);
}

void MessagesManager::read_all_dialog_mentions(DialogId dialog_id, Promise<Unit> &&promise) {
  bool is_bot = td_->auth_manager_->is_bot();
  if (is_bot) {
    return promise.set_error(Status::Error(3, "Method is not available for bots"));
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  LOG(INFO) << "Receive readAllChatMentions request in " << dialog_id << " with " << d->unread_mention_count
            << " unread mentions";
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(3, "Chat is not accessible"));
  }
  if (dialog_id.get_type() == DialogType::SecretChat) {
    CHECK(d->unread_mention_count == 0);
    return promise.set_value(Unit());
  }

  if (d->last_new_message_id > d->last_read_all_mentions_message_id) {
    d->last_read_all_mentions_message_id = d->last_new_message_id;
    on_dialog_updated(dialog_id, "read_all_dialog_mentions");
  }

  vector<MessageId> message_ids;
  find_unread_mentions(d->messages.get(), message_ids);

  LOG(INFO) << "Found " << message_ids.size() << " messages with unread mentions in memory";
  bool is_update_sent = false;
  for (auto message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->contains_unread_mention);
    CHECK(m->message_id == message_id);
    CHECK(m->message_id.is_valid());
    remove_message_notification_id(d, m, true, false);  // should be called before contains_unread_mention is updated
    m->contains_unread_mention = false;

    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateMessageMentionRead>(dialog_id.get(), m->message_id.get(), 0));
    is_update_sent = true;
    on_message_changed(d, m, true, "read_all_dialog_mentions");
  }

  if (d->unread_mention_count != 0) {
    d->unread_mention_count = 0;
    d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] = 0;
    if (!is_update_sent) {
      send_update_chat_unread_mention_count(d);
    } else {
      LOG(INFO) << "Update unread mention message count in " << dialog_id << " to " << d->unread_mention_count;
      on_dialog_updated(dialog_id, "read_all_dialog_mentions");
    }
  }
  remove_message_dialog_notifications(d, MessageId::max(), true, "read_all_dialog_mentions");

  read_all_dialog_mentions_on_server(dialog_id, 0, std::move(promise));
}

class MessagesManager::ReadAllDialogMentionsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_read_all_dialog_mentions_on_server_logevent(DialogId dialog_id) {
  ReadAllDialogMentionsOnServerLogEvent logevent{dialog_id};
  auto storer = LogEventStorerImpl<ReadAllDialogMentionsOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadAllDialogMentionsOnServer, storer);
}

void MessagesManager::read_all_dialog_mentions_on_server(DialogId dialog_id, uint64 logevent_id,
                                                         Promise<Unit> &&promise) {
  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_read_all_dialog_mentions_on_server_logevent(dialog_id);
  }

  LOG(INFO) << "Read all mentions on server in " << dialog_id;
  td_->create_handler<ReadAllMentionsQuery>(get_erase_logevent_promise(logevent_id, std::move(promise)))
      ->send(dialog_id);
}

void MessagesManager::read_message_content_from_updates(MessageId message_id) {
  if (!message_id.is_valid() || !message_id.is_server()) {
    LOG(ERROR) << "Incoming update tries to read content of " << message_id;
    return;
  }

  Dialog *d = get_dialog_by_message_id(message_id);
  if (d != nullptr) {
    Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    read_message_content(d, m, false, "read_message_content_from_updates");
  }
}

void MessagesManager::read_channel_message_content_from_updates(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  if (!message_id.is_valid() || !message_id.is_server()) {
    LOG(ERROR) << "Incoming update tries to read content of " << message_id << " in " << d->dialog_id;
    return;
  }

  Message *m = get_message_force(d, message_id, "read_channel_message_content_from_updates");
  if (m != nullptr) {
    read_message_content(d, m, false, "read_channel_message_content_from_updates");
  } else if (message_id > d->last_new_message_id) {
    get_channel_difference(d->dialog_id, d->pts, true, "read_channel_message_content_from_updates");
  }
}

bool MessagesManager::read_message_content(Dialog *d, Message *m, bool is_local_read, const char *source) {
  LOG_CHECK(m != nullptr) << source;
  CHECK(!m->message_id.is_scheduled());
  bool is_mention_read = update_message_contains_unread_mention(d, m, false, "read_message_content");
  bool is_content_read =
      update_opened_message_content(m->content.get()) | ttl_on_open(d, m, Time::now(), is_local_read);

  LOG(INFO) << "Read message content of " << m->message_id << " in " << d->dialog_id
            << ": is_mention_read = " << is_mention_read << ", is_content_read = " << is_content_read;
  if (is_mention_read || is_content_read) {
    on_message_changed(d, m, true, "read_message_content");
    if (is_content_read) {
      send_closure(G()->td(), &Td::send_update,
                   make_tl_object<td_api::updateMessageContentOpened>(d->dialog_id.get(), m->message_id.get()));
    }
    return true;
  }
  return false;
}

bool MessagesManager::has_incoming_notification(DialogId dialog_id, const Message *m) const {
  if (m->is_from_scheduled) {
    return true;
  }
  return !m->is_outgoing && dialog_id != get_my_dialog_id();
}

int32 MessagesManager::calc_new_unread_count_from_last_unread(Dialog *d, MessageId max_message_id,
                                                              MessageType type) const {
  CHECK(!max_message_id.is_scheduled());
  MessagesConstIterator it(d, max_message_id);
  if (*it == nullptr || (*it)->message_id != max_message_id) {
    return -1;
  }

  int32 unread_count = type == MessageType::Server ? d->server_unread_count : d->local_unread_count;
  while (*it != nullptr && (*it)->message_id > d->last_read_inbox_message_id) {
    if (has_incoming_notification(d->dialog_id, *it) && (*it)->message_id.get_type() == type) {
      unread_count--;
    }
    --it;
  }
  if (*it == nullptr || (*it)->message_id != d->last_read_inbox_message_id) {
    return -1;
  }

  LOG(INFO) << "Found " << unread_count << " unread messages in " << d->dialog_id << " from last unread message";
  return unread_count;
}

int32 MessagesManager::calc_new_unread_count_from_the_end(Dialog *d, MessageId max_message_id, MessageType type,
                                                          int32 hint_unread_count) const {
  CHECK(!max_message_id.is_scheduled());
  int32 unread_count = 0;
  MessagesConstIterator it(d, MessageId::max());
  while (*it != nullptr && (*it)->message_id > max_message_id) {
    if (has_incoming_notification(d->dialog_id, *it) && (*it)->message_id.get_type() == type) {
      unread_count++;
    }
    --it;
  }

  bool is_count_exact = d->last_message_id.is_valid() && *it != nullptr;
  if (hint_unread_count >= 0) {
    if (is_count_exact) {
      if (hint_unread_count == unread_count) {
        return hint_unread_count;
      }
    } else {
      if (hint_unread_count >= unread_count) {
        return hint_unread_count;
      }
    }

    // hint_unread_count is definitely wrong, ignore it
    LOG(ERROR) << "Receive hint_unread_count = " << hint_unread_count << ", but found " << unread_count
               << " unread messages in " << d->dialog_id;
  }

  if (!is_count_exact) {
    // unread count is likely to be calculated wrong, so ignore it
    return -1;
  }

  LOG(INFO) << "Found " << unread_count << " unread messages in " << d->dialog_id << " from the end";
  return unread_count;
}

int32 MessagesManager::calc_new_unread_count(Dialog *d, MessageId max_message_id, MessageType type,
                                             int32 hint_unread_count) const {
  CHECK(!max_message_id.is_scheduled());
  if (d->is_empty) {
    return 0;
  }

  if (!d->last_read_inbox_message_id.is_valid()) {
    return calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
  }

  if (!d->last_message_id.is_valid() ||
      (d->last_message_id.get() - max_message_id.get() > max_message_id.get() - d->last_read_inbox_message_id.get())) {
    int32 unread_count = calc_new_unread_count_from_last_unread(d, max_message_id, type);
    return unread_count >= 0 ? unread_count
                             : calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
  } else {
    int32 unread_count = calc_new_unread_count_from_the_end(d, max_message_id, type, hint_unread_count);
    return unread_count >= 0 ? unread_count : calc_new_unread_count_from_last_unread(d, max_message_id, type);
  }
}

void MessagesManager::repair_server_unread_count(DialogId dialog_id, int32 unread_count) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  LOG(INFO) << "Repair server unread count in " << dialog_id << " from " << unread_count;
  create_actor<SleepActor>("RepairServerUnreadCountSleepActor", 0.2,
                           PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](Result<Unit> result) {
                             send_closure(actor_id, &MessagesManager::send_get_dialog_query, dialog_id, Promise<Unit>(),
                                          0);
                           }))
      .release();
}

void MessagesManager::repair_channel_server_unread_count(Dialog *d) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::Channel);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (d->last_read_inbox_message_id >= d->last_new_message_id) {
    // all messages are already read
    return;
  }
  if (d->order == 0) {
    // there is no unread count in left channels
    return;
  }

  LOG(INFO) << "Reload ChannelFull for " << d->dialog_id << " to repair unread message counts";
  // TODO logevent?
  td_->contacts_manager_->get_channel_full(d->dialog_id.get_channel_id(), Promise<Unit>());
}

void MessagesManager::read_history_inbox(DialogId dialog_id, MessageId max_message_id, int32 unread_count,
                                         const char *source) {
  CHECK(!max_message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d != nullptr) {
    // there can be updateReadHistoryInbox up to message 0, if messages where read and then all messages where deleted
    if (!max_message_id.is_valid() && max_message_id != MessageId()) {
      LOG(ERROR) << "Receive read inbox update in " << dialog_id << " up to " << max_message_id << " from " << source;
      return;
    }
    if (d->is_last_read_inbox_message_id_inited && max_message_id <= d->last_read_inbox_message_id) {
      LOG(INFO) << "Receive read inbox update in " << dialog_id << " up to " << max_message_id << " from " << source
                << ", but all messages have already been read up to " << d->last_read_inbox_message_id;
      return;
    }

    if (max_message_id != MessageId() && max_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Try to update last read inbox message in " << dialog_id << " with " << max_message_id << " from "
                 << source;
      return;
    }

    if (max_message_id != MessageId() && unread_count > 0 && max_message_id >= d->last_new_message_id &&
        max_message_id >= d->last_message_id && max_message_id >= d->last_database_message_id) {
      LOG(ERROR) << "Have unknown " << unread_count << " unread messages up to " << max_message_id << " in "
                 << dialog_id << " with last_new_message_id = " << d->last_new_message_id
                 << ", last_message_id = " << d->last_message_id
                 << ", last_database_message_id = " << d->last_database_message_id;
      unread_count = 0;
    }

    LOG_IF(INFO, d->last_new_message_id.is_valid() && max_message_id > d->last_new_message_id &&
                     max_message_id > d->max_notification_message_id && max_message_id.is_server() &&
                     dialog_id.get_type() != DialogType::Channel && !running_get_difference_)
        << "Receive read inbox update up to unknown " << max_message_id << " in " << dialog_id << " from " << source
        << ". Last new is " << d->last_new_message_id << ", unread_count = " << unread_count
        << ". Possible only for deleted incoming message";

    if (dialog_id.get_type() == DialogType::SecretChat) {
      ttl_read_history(d, false, max_message_id, d->last_read_inbox_message_id, Time::now());
    }

    if (max_message_id > d->last_new_message_id && dialog_id.get_type() == DialogType::Channel) {
      LOG(INFO) << "Schedule getDifference in " << dialog_id.get_channel_id();
      channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
    }

    int32 server_unread_count = calc_new_unread_count(d, max_message_id, MessageType::Server, unread_count);
    int32 local_unread_count =
        d->local_unread_count == 0 ? 0 : calc_new_unread_count(d, max_message_id, MessageType::Local, -1);

    if (server_unread_count < 0) {
      server_unread_count = unread_count >= 0 ? unread_count : d->server_unread_count;
      if (dialog_id.get_type() != DialogType::SecretChat && have_input_peer(dialog_id, AccessRights::Read) &&
          d->order > 0) {
        d->need_repair_server_unread_count = true;
        repair_server_unread_count(dialog_id, server_unread_count);
      }
    }
    if (local_unread_count < 0) {
      // TODO repair local unread count
      local_unread_count = d->local_unread_count;
    }

    set_dialog_last_read_inbox_message_id(d, max_message_id, server_unread_count, local_unread_count, true, source);

    if (d->is_marked_as_unread && max_message_id != MessageId()) {
      set_dialog_is_marked_as_unread(d, false);
    }
  } else {
    LOG(INFO) << "Receive read inbox about unknown " << dialog_id << " from " << source;
  }
}

void MessagesManager::read_history_outbox(DialogId dialog_id, MessageId max_message_id, int32 read_date) {
  CHECK(!max_message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d != nullptr) {
    if (!max_message_id.is_valid()) {
      LOG(ERROR) << "Receive read outbox update in " << dialog_id << " with " << max_message_id;
      return;
    }
    if (max_message_id <= d->last_read_outbox_message_id) {
      LOG(INFO) << "Receive read outbox update up to " << max_message_id
                << ", but all messages have already been read up to " << d->last_read_outbox_message_id;
      return;
    }

    if (max_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Try to update last read outbox message with " << max_message_id;
      return;
    }

    // it is impossible for just sent outgoing messages because updates are ordered by pts
    LOG_IF(INFO, d->last_new_message_id.is_valid() && max_message_id > d->last_new_message_id &&
                     dialog_id.get_type() != DialogType::Channel)
        << "Receive read outbox update about unknown " << max_message_id << " in " << dialog_id << " with last new "
        << d->last_new_message_id << ". Possible only for deleted outgoing message";

    if (dialog_id.get_type() == DialogType::SecretChat) {
      double server_time = Time::now();
      double read_time = server_time;
      if (read_date <= 0) {
        LOG(ERROR) << "Receive wrong read date " << read_date << " in " << dialog_id;
      } else if (read_date < server_time) {
        read_time = read_date;
      }
      ttl_read_history(d, true, max_message_id, d->last_read_outbox_message_id, read_time);
    }

    set_dialog_last_read_outbox_message_id(d, max_message_id);
  } else {
    LOG(INFO) << "Receive read outbox update about unknown " << dialog_id;
  }
}

bool MessagesManager::need_unread_counter(int64 dialog_order) {
  return dialog_order != DEFAULT_ORDER;
}

int32 MessagesManager::get_dialog_total_count(const DialogList &list) {
  if (list.server_dialog_total_count_ != -1 && list.secret_chat_total_count_ != -1) {
    return std::max(list.server_dialog_total_count_ + list.secret_chat_total_count_,
                    list.in_memory_dialog_total_count_);
  }
  if (list.last_dialog_date_ == MAX_DIALOG_DATE) {
    return list.in_memory_dialog_total_count_;
  }
  return list.in_memory_dialog_total_count_ + 1;
}

void MessagesManager::repair_server_dialog_total_count(FolderId folder_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  send_closure(td_->create_net_actor<GetDialogListActor>(Promise<Unit>()), &GetDialogListActor::send, folder_id,
               2147483647, ServerMessageId(), DialogId(), 1,
               get_sequence_dispatcher_id(DialogId(), MessageContentType::None));
}

void MessagesManager::repair_secret_chat_total_count(FolderId folder_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  if (G()->parameters().use_message_db) {
    // race-prone
    G()->td_db()->get_dialog_db_async()->get_secret_chat_count(
        folder_id, PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Result<int32> result) {
          if (result.is_error()) {
            return;
          }
          send_closure(actor_id, &MessagesManager::on_get_secret_chat_total_count, folder_id, result.move_as_ok());
        }));
  } else {
    int32 total_count = 0;
    auto &list = get_dialog_list(folder_id);
    for (const auto &dialog_date : list.ordered_server_dialogs_) {
      auto dialog_id = dialog_date.get_dialog_id();
      if (dialog_id.get_type() == DialogType::SecretChat && dialog_date.get_order() != DEFAULT_ORDER) {
        total_count++;
      }
    }
    on_get_secret_chat_total_count(folder_id, total_count);
  }
}

void MessagesManager::on_get_secret_chat_total_count(FolderId folder_id, int32 total_count) {
  CHECK(!td_->auth_manager_->is_bot());
  auto &list = get_dialog_list(folder_id);
  CHECK(total_count >= 0);
  if (list.secret_chat_total_count_ != total_count) {
    auto old_dialog_total_count = get_dialog_total_count(list);
    list.secret_chat_total_count_ = total_count;
    if (list.is_dialog_unread_count_inited_ && old_dialog_total_count != get_dialog_total_count(list)) {
      send_update_unread_chat_count(folder_id, DialogId(), true, "on_get_secret_chat_total_count");
    }
  }
}

void MessagesManager::recalc_unread_count(FolderId folder_id) {
  if (td_->auth_manager_->is_bot() || !G()->parameters().use_message_db) {
    return;
  }

  auto &list = get_dialog_list(folder_id);
  if (!list.need_unread_count_recalc_) {
    return;
  }
  LOG(INFO) << "Recalculate unread counts in " << folder_id;
  list.need_unread_count_recalc_ = false;
  list.is_message_unread_count_inited_ = true;
  list.is_dialog_unread_count_inited_ = true;

  int32 message_total_count = 0;
  int32 message_muted_count = 0;
  int32 dialog_total_count = 0;
  int32 dialog_muted_count = 0;
  int32 dialog_marked_count = 0;
  int32 dialog_muted_marked_count = 0;
  int32 server_dialog_total_count = 0;
  int32 secret_chat_total_count = 0;
  for (const auto &dialog_date : list.ordered_server_dialogs_) {
    auto dialog_id = dialog_date.get_dialog_id();
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    int unread_count = d->server_unread_count + d->local_unread_count;
    if (need_unread_counter(d->order) && (unread_count > 0 || d->is_marked_as_unread)) {
      message_total_count += unread_count;
      dialog_total_count++;
      if (unread_count == 0 && d->is_marked_as_unread) {
        dialog_marked_count++;
      }

      LOG(DEBUG) << "Have " << unread_count << " messages in " << dialog_id;
      if (is_dialog_muted(d)) {
        message_muted_count += unread_count;
        dialog_muted_count++;
        if (unread_count == 0 && d->is_marked_as_unread) {
          dialog_muted_marked_count++;
        }
      }
    }
    if (d->order != DEFAULT_ORDER) {
      if (dialog_id.get_type() == DialogType::SecretChat) {
        secret_chat_total_count++;
      } else {
        server_dialog_total_count++;
      }
    }
  }

  if (list.unread_message_total_count_ != message_total_count ||
      list.unread_message_muted_count_ != message_muted_count) {
    list.unread_message_total_count_ = message_total_count;
    list.unread_message_muted_count_ = message_muted_count;
    send_update_unread_message_count(folder_id, DialogId(), true, "recalc_unread_count");
  }

  auto old_dialog_total_count = get_dialog_total_count(list);
  if (list.last_dialog_date_ == MAX_DIALOG_DATE) {
    if (server_dialog_total_count != list.server_dialog_total_count_ ||
        secret_chat_total_count != list.secret_chat_total_count_) {
      list.server_dialog_total_count_ = server_dialog_total_count;
      list.secret_chat_total_count_ = secret_chat_total_count;
    }
  } else {
    repair_server_dialog_total_count(folder_id);

    if (list.secret_chat_total_count_ == -1) {
      repair_secret_chat_total_count(folder_id);
    }
  }
  if (list.unread_dialog_total_count_ != dialog_total_count || list.unread_dialog_muted_count_ != dialog_muted_count ||
      list.unread_dialog_marked_count_ != dialog_marked_count ||
      list.unread_dialog_muted_marked_count_ != dialog_muted_marked_count ||
      old_dialog_total_count != get_dialog_total_count(list)) {
    list.unread_dialog_total_count_ = dialog_total_count;
    list.unread_dialog_muted_count_ = dialog_muted_count;
    list.unread_dialog_marked_count_ = dialog_marked_count;
    list.unread_dialog_muted_marked_count_ = dialog_muted_marked_count;
    send_update_unread_chat_count(folder_id, DialogId(), true, "recalc_unread_count");
  }
}

void MessagesManager::set_dialog_last_read_inbox_message_id(Dialog *d, MessageId message_id, int32 server_unread_count,
                                                            int32 local_unread_count, bool force_update,
                                                            const char *source) {
  CHECK(!message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG(INFO) << "Update last read inbox message in " << d->dialog_id << " from " << d->last_read_inbox_message_id
            << " to " << message_id << " and update unread message count from " << d->server_unread_count << " + "
            << d->local_unread_count << " to " << server_unread_count << " + " << local_unread_count << " from "
            << source;
  if (message_id != MessageId::min()) {
    d->last_read_inbox_message_id = message_id;
    d->is_last_read_inbox_message_id_inited = true;
  }
  int32 old_unread_count = d->server_unread_count + d->local_unread_count;
  d->server_unread_count = server_unread_count;
  d->local_unread_count = local_unread_count;
  int32 new_unread_count = d->server_unread_count + d->local_unread_count;
  int32 delta = new_unread_count - old_unread_count;
  auto &list = get_dialog_list(d->folder_id);
  if (delta != 0 && need_unread_counter(d->order) && list.is_message_unread_count_inited_) {
    list.unread_message_total_count_ += delta;
    if (is_dialog_muted(d)) {
      list.unread_message_muted_count_ += delta;
    }
    send_update_unread_message_count(d->folder_id, d->dialog_id, force_update, source);
  }
  delta = static_cast<int32>(new_unread_count != 0) - static_cast<int32>(old_unread_count != 0);
  if (delta != 0 && need_unread_counter(d->order) && list.is_dialog_unread_count_inited_) {
    if (d->is_marked_as_unread) {
      list.unread_dialog_marked_count_ -= delta;
    } else {
      list.unread_dialog_total_count_ += delta;
    }
    if (is_dialog_muted(d)) {
      if (d->is_marked_as_unread) {
        list.unread_dialog_muted_marked_count_ -= delta;
      } else {
        list.unread_dialog_muted_count_ += delta;
      }
    }
    send_update_unread_chat_count(d->folder_id, d->dialog_id, force_update, source);
  }
  if (message_id != MessageId::min() && d->last_read_inbox_message_id.is_valid() && d->order != DEFAULT_ORDER &&
      d->order != SPONSORED_DIALOG_ORDER) {
    VLOG(notifications) << "Remove some notifications in " << d->dialog_id
                        << " after updating last read inbox message to " << message_id
                        << " and unread message count to " << server_unread_count << " + " << local_unread_count
                        << " from " << source;
    if (d->message_notification_group.group_id.is_valid()) {
      auto total_count = get_dialog_pending_notification_count(d, false);
      if (total_count == 0) {
        set_dialog_last_notification(d->dialog_id, d->message_notification_group, 0, NotificationId(),
                                     "set_dialog_last_read_inbox_message_id");
      }
      if (!d->pending_new_message_notifications.empty()) {
        for (auto &it : d->pending_new_message_notifications) {
          if (it.second <= message_id) {
            it.first = DialogId();
          }
        }
        flush_pending_new_message_notifications(d->dialog_id, false, DialogId(UserId(1)));
      }
      total_count -= static_cast<int32>(d->pending_new_message_notifications.size());
      if (total_count < 0) {
        LOG(ERROR) << "Total message notification count is " << total_count << " in " << d->dialog_id << " with "
                   << d->pending_new_message_notifications.size() << " pending new message notifications";
        total_count = 0;
      }
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification_group,
                         d->message_notification_group.group_id, NotificationId(), d->last_read_inbox_message_id,
                         total_count, Slice(source) == Slice("view_messages"), Promise<Unit>());
    }

    if (d->mention_notification_group.group_id.is_valid() && d->pinned_message_notification_message_id.is_valid() &&
        d->pinned_message_notification_message_id <= d->last_read_inbox_message_id) {
      // remove pinned message notification when it is read
      remove_dialog_pinned_message_notification(d);
    }
  }

  send_update_chat_read_inbox(d, force_update, source);
}

void MessagesManager::set_dialog_last_read_outbox_message_id(Dialog *d, MessageId message_id) {
  CHECK(!message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG(INFO) << "Update last read outbox message in " << d->dialog_id << " from " << d->last_read_outbox_message_id
            << " to " << message_id;
  d->last_read_outbox_message_id = message_id;
  d->is_last_read_outbox_message_id_inited = true;
  send_update_chat_read_outbox(d);
}

void MessagesManager::set_dialog_max_unavailable_message_id(DialogId dialog_id, MessageId max_unavailable_message_id,
                                                            bool from_update, const char *source) {
  CHECK(!max_unavailable_message_id.is_scheduled());

  Dialog *d = get_dialog_force(dialog_id);
  if (d != nullptr) {
    if (max_unavailable_message_id > d->last_new_message_id && from_update) {
      if (d->last_new_message_id.is_valid()) {
        if (!td_->auth_manager_->is_bot()) {
          LOG(ERROR) << "Tried to set " << dialog_id << " max unavailable message to " << max_unavailable_message_id
                     << " from " << source << ", but last new message is " << d->last_new_message_id;
        }
        max_unavailable_message_id = d->last_new_message_id;
      } else if (max_unavailable_message_id.is_valid() && max_unavailable_message_id.is_server()) {
        set_dialog_last_new_message_id(d, max_unavailable_message_id, source);
      }
    }

    if (d->max_unavailable_message_id == max_unavailable_message_id) {
      return;
    }

    if (max_unavailable_message_id.is_valid() && max_unavailable_message_id.is_yet_unsent()) {
      LOG(ERROR) << "Try to update " << dialog_id << " last read outbox message with " << max_unavailable_message_id
                 << " from " << source;
      return;
    }
    LOG(INFO) << "Set max unavailable message to " << max_unavailable_message_id << " in " << dialog_id << " from "
              << source;

    on_dialog_updated(dialog_id, "set_dialog_max_unavailable_message_id");

    if (d->max_unavailable_message_id > max_unavailable_message_id) {
      d->max_unavailable_message_id = max_unavailable_message_id;
      return;
    }

    d->max_unavailable_message_id = max_unavailable_message_id;

    vector<MessageId> message_ids;
    find_old_messages(d->messages.get(), max_unavailable_message_id, message_ids);

    vector<int64> deleted_message_ids;
    bool need_update_dialog_pos = false;
    for (auto message_id : message_ids) {
      if (message_id.is_yet_unsent()) {
        continue;
      }

      auto m = get_message(d, message_id);
      CHECK(m != nullptr);
      CHECK(m->message_id <= max_unavailable_message_id);
      CHECK(m->message_id == message_id);
      auto p =
          delete_message(d, message_id, !from_update, &need_update_dialog_pos, "set_dialog_max_unavailable_message_id");
      CHECK(p.get() == m);
      deleted_message_ids.push_back(p->message_id.get());
    }

    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "set_dialog_max_unavailable_message_id");
    }

    send_update_delete_messages(dialog_id, std::move(deleted_message_ids), !from_update, false);

    if (d->server_unread_count + d->local_unread_count > 0) {
      read_history_inbox(dialog_id, max_unavailable_message_id, -1, "set_dialog_max_unavailable_message_id");
    }
  } else {
    LOG(INFO) << "Receive max unavailable message in unknown " << dialog_id << " from " << source;
  }
}

void MessagesManager::set_dialog_online_member_count(DialogId dialog_id, int32 online_member_count, bool is_from_server,
                                                     const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  if (d == nullptr) {
    return;
  }

  auto &info = dialog_online_member_counts_[dialog_id];
  LOG(INFO) << "Change online member count from " << info.online_member_count << " to " << online_member_count << " in "
            << dialog_id << " from " << source;
  bool need_update = d->is_opened && (!info.is_update_sent || info.online_member_count != online_member_count);
  info.online_member_count = online_member_count;
  info.updated_time = Time::now();

  if (need_update) {
    info.is_update_sent = true;
    send_update_chat_online_member_count(dialog_id, online_member_count);
  }
  if (d->is_opened) {
    if (is_from_server) {
      update_dialog_online_member_count_timeout_.set_timeout_in(dialog_id.get(), ONLINE_MEMBER_COUNT_UPDATE_TIME);
    } else {
      update_dialog_online_member_count_timeout_.add_timeout_in(dialog_id.get(), ONLINE_MEMBER_COUNT_UPDATE_TIME);
    }
  }
}

void MessagesManager::on_update_dialog_online_member_count_timeout(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  LOG(INFO) << "Expired timeout for online member count for " << dialog_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (!d->is_opened) {
    send_update_chat_online_member_count(dialog_id, 0);
    return;
  }

  if (dialog_id.get_type() == DialogType::Channel && !is_broadcast_channel(dialog_id)) {
    auto participant_count = td_->contacts_manager_->get_channel_participant_count(dialog_id.get_channel_id());
    if (participant_count == 0 || participant_count >= 195) {
      td_->create_handler<GetOnlinesQuery>()->send(dialog_id);
    } else {
      td_->contacts_manager_->send_get_channel_participants_query(
          dialog_id.get_channel_id(),
          ChannelParticipantsFilter(td_api::make_object<td_api::supergroupMembersFilterRecent>()), 0, 200, 0, Auto());
    }
    return;
  }
  if (dialog_id.get_type() == DialogType::Chat) {
    // we need actual online status state, so we need to reget chat participants
    td_->contacts_manager_->repair_chat_participants(dialog_id.get_chat_id());
    return;
  }
}

MessageId MessagesManager::get_message_id(const tl_object_ptr<telegram_api::Message> &message_ptr, bool is_scheduled) {
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID: {
      auto message = static_cast<const telegram_api::messageEmpty *>(message_ptr.get());
      return is_scheduled ? MessageId() : MessageId(ServerMessageId(message->id_));
    }
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      return is_scheduled ? MessageId(ScheduledServerMessageId(message->id_), message->date_)
                          : MessageId(ServerMessageId(message->id_));
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      return is_scheduled ? MessageId(ScheduledServerMessageId(message->id_), message->date_)
                          : MessageId(ServerMessageId(message->id_));
    }
    default:
      UNREACHABLE();
      return MessageId();
  }
}

DialogId MessagesManager::get_message_dialog_id(const tl_object_ptr<telegram_api::Message> &message_ptr) const {
  DialogId dialog_id;
  UserId sender_user_id;
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      return DialogId();
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      dialog_id = DialogId(message->to_id_);
      if (message->flags_ & MESSAGE_FLAG_HAS_FROM_ID) {
        sender_user_id = UserId(message->from_id_);
      }
      break;
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      dialog_id = DialogId(message->to_id_);
      if (message->flags_ & MESSAGE_FLAG_HAS_FROM_ID) {
        sender_user_id = UserId(message->from_id_);
      }
      break;
    }
    default:
      UNREACHABLE();
      break;
  }

  if (dialog_id == get_my_dialog_id()) {
    LOG_IF(ERROR, !sender_user_id.is_valid()) << "Receive invalid " << sender_user_id;
    return DialogId(sender_user_id);
  }
  return dialog_id;
}

FullMessageId MessagesManager::get_full_message_id(const tl_object_ptr<telegram_api::Message> &message_ptr,
                                                   bool is_scheduled) const {
  return {get_message_dialog_id(message_ptr), get_message_id(message_ptr, is_scheduled)};
}

int32 MessagesManager::get_message_date(const tl_object_ptr<telegram_api::Message> &message_ptr) {
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      return 0;
    case telegram_api::message::ID: {
      auto message = static_cast<const telegram_api::message *>(message_ptr.get());
      return message->date_;
    }
    case telegram_api::messageService::ID: {
      auto message = static_cast<const telegram_api::messageService *>(message_ptr.get());
      return message->date_;
    }
    default:
      UNREACHABLE();
      return 0;
  }
}

void MessagesManager::ttl_read_history(Dialog *d, bool is_outgoing, MessageId from_message_id,
                                       MessageId till_message_id, double view_date) {
  CHECK(!from_message_id.is_scheduled());
  CHECK(!till_message_id.is_scheduled());

  // TODO: protect with logevent
  suffix_load_till_message_id(d, till_message_id,
                              PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = d->dialog_id, is_outgoing,
                                                      from_message_id, till_message_id, view_date](Result<Unit>) {
                                send_closure(actor_id, &MessagesManager::ttl_read_history_impl, dialog_id, is_outgoing,
                                             from_message_id, till_message_id, view_date);
                              }));
}

void MessagesManager::ttl_read_history_impl(DialogId dialog_id, bool is_outgoing, MessageId from_message_id,
                                            MessageId till_message_id, double view_date) {
  CHECK(!from_message_id.is_scheduled());
  CHECK(!till_message_id.is_scheduled());

  auto *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto now = Time::now();
  for (auto it = MessagesIterator(d, from_message_id); *it && (*it)->message_id >= till_message_id; --it) {
    auto *m = *it;
    if (m->is_outgoing == is_outgoing) {
      ttl_on_view(d, m, view_date, now);
    }
  }
}

void MessagesManager::ttl_on_view(const Dialog *d, Message *m, double view_date, double now) {
  if (m->ttl > 0 && m->ttl_expires_at == 0 && !m->message_id.is_scheduled() && !m->message_id.is_yet_unsent() &&
      !m->is_failed_to_send && !m->is_content_secret) {
    m->ttl_expires_at = m->ttl + view_date;
    ttl_register_message(d->dialog_id, m, now);
    on_message_changed(d, m, true, "ttl_on_view");
  }
}

bool MessagesManager::ttl_on_open(Dialog *d, Message *m, double now, bool is_local_read) {
  CHECK(!m->message_id.is_scheduled());
  if (m->ttl > 0 && m->ttl_expires_at == 0) {
    if (!is_local_read && d->dialog_id.get_type() != DialogType::SecretChat) {
      on_message_ttl_expired(d, m);
    } else {
      m->ttl_expires_at = m->ttl + now;
      ttl_register_message(d->dialog_id, m, now);
    }
    return true;
  }
  return false;
}

void MessagesManager::ttl_register_message(DialogId dialog_id, const Message *m, double now) {
  if (m->ttl_expires_at == 0) {
    return;
  }
  CHECK(!m->message_id.is_scheduled());

  auto it_flag = ttl_nodes_.insert(TtlNode(dialog_id, m->message_id));
  CHECK(it_flag.second);
  auto it = it_flag.first;

  ttl_heap_.insert(m->ttl_expires_at, it->as_heap_node());
  ttl_update_timeout(now);
}

void MessagesManager::ttl_unregister_message(DialogId dialog_id, const Message *m, double now, const char *source) {
  if (m->ttl_expires_at == 0) {
    return;
  }
  CHECK(!m->message_id.is_scheduled());

  TtlNode ttl_node(dialog_id, m->message_id);
  auto it = ttl_nodes_.find(ttl_node);

  // expect m->ttl == 0, but m->ttl_expires_at > 0 from binlog
  LOG_CHECK(it != ttl_nodes_.end()) << dialog_id << " " << m->message_id << " " << source << " " << G()->close_flag()
                                    << " " << m->ttl << " " << m->ttl_expires_at << " " << Time::now() << " "
                                    << m->from_database;

  auto *heap_node = it->as_heap_node();
  if (heap_node->in_heap()) {
    ttl_heap_.erase(heap_node);
  }
  ttl_nodes_.erase(it);
  ttl_update_timeout(now);
}

void MessagesManager::ttl_loop(double now) {
  std::unordered_map<DialogId, std::vector<MessageId>, DialogIdHash> to_delete;
  while (!ttl_heap_.empty() && ttl_heap_.top_key() < now) {
    auto full_message_id = TtlNode::from_heap_node(ttl_heap_.pop())->full_message_id;
    auto dialog_id = full_message_id.get_dialog_id();
    if (dialog_id.get_type() == DialogType::SecretChat) {
      to_delete[dialog_id].push_back(full_message_id.get_message_id());
    } else {
      auto d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      auto m = get_message(d, full_message_id.get_message_id());
      CHECK(m != nullptr);
      on_message_ttl_expired(d, m);
      on_message_changed(d, m, true, "ttl_loop");
    }
  }
  for (auto &it : to_delete) {
    delete_dialog_messages_from_updates(it.first, it.second);
  }
  ttl_update_timeout(now);
}

void MessagesManager::ttl_update_timeout(double now) {
  if (ttl_heap_.empty()) {
    if (!ttl_slot_.empty()) {
      ttl_slot_.cancel_timeout();
    }
    return;
  }
  ttl_slot_.set_event(EventCreator::yield(actor_id()));
  ttl_slot_.set_timeout_in(ttl_heap_.top_key() - now);
}

void MessagesManager::on_message_ttl_expired(Dialog *d, Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->ttl > 0);
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  ttl_unregister_message(d->dialog_id, m, Time::now(), "on_message_ttl_expired");
  unregister_message_content(td_, m->content.get(), {d->dialog_id, m->message_id});
  remove_message_file_sources(d->dialog_id, m);
  on_message_ttl_expired_impl(d, m);
  register_message_content(td_, m->content.get(), {d->dialog_id, m->message_id});
  send_update_message_content(d->dialog_id, m->message_id, m->content.get(), m->date, m->is_content_secret,
                              "on_message_ttl_expired");
}

void MessagesManager::on_message_ttl_expired_impl(Dialog *d, Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(m->ttl > 0);
  CHECK(d->dialog_id.get_type() != DialogType::SecretChat);
  delete_message_files(d->dialog_id, m);
  update_expired_message_content(m->content);
  m->ttl = 0;
  m->ttl_expires_at = 0;
  if (m->reply_markup != nullptr) {
    if (m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
      if (!td_->auth_manager_->is_bot()) {
        if (d->reply_markup_message_id == m->message_id) {
          set_dialog_reply_markup(d, MessageId());
        }
      }
      m->had_reply_markup = true;
    }
    m->reply_markup = nullptr;
  }
  remove_message_notification_id(d, m, true, true);
  update_message_contains_unread_mention(d, m, false, "on_message_ttl_expired_impl");
  m->contains_mention = false;
  m->reply_to_message_id = MessageId();
  m->is_content_secret = false;
}

void MessagesManager::loop() {
  auto token = get_link_token();
  if (token == YieldType::TtlDb) {
    ttl_db_loop(G()->server_time());
  } else {
    ttl_loop(Time::now());
  }
}

void MessagesManager::tear_down() {
  parent_.reset();
}

void MessagesManager::start_up() {
  init();
}

void MessagesManager::init() {
  if (is_inited_) {
    return;
  }
  is_inited_ = true;

  always_wait_for_mailbox();

  start_time_ = Time::now();

  include_sponsored_dialog_to_unread_count_ =
      G()->shared_config().get_option_boolean("include_sponsored_chat_to_unread_count");

  if (G()->parameters().use_message_db) {
    // erase old keys
    G()->td_db()->get_binlog_pmc()->erase("last_server_dialog_date");
    G()->td_db()->get_binlog_pmc()->erase("unread_message_count");
    G()->td_db()->get_binlog_pmc()->erase("unread_dialog_count");

    auto last_database_server_dialog_dates = G()->td_db()->get_binlog_pmc()->prefix_get("last_server_dialog_date");
    for (auto &it : last_database_server_dialog_dates) {
      auto r_folder_id = to_integer_safe<int32>(Slice(it.first).substr(Slice("last_server_dialog_date").size()));
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }

      string order_str;
      string dialog_id_str;
      std::tie(order_str, dialog_id_str) = split(it.second);

      auto r_order = to_integer_safe<int64>(order_str);
      auto r_dialog_id = to_integer_safe<int64>(dialog_id_str);
      if (r_order.is_error() || r_dialog_id.is_error()) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        FolderId folder_id(r_folder_id.ok());
        auto &list = get_dialog_list(folder_id);
        list.last_database_server_dialog_date_ = DialogDate(r_order.ok(), DialogId(r_dialog_id.ok()));
        LOG(INFO) << "Loaded last_database_server_dialog_date_ " << list.last_database_server_dialog_date_ << " in "
                  << folder_id;
      }
    }

    auto sponsored_dialog_id_string = G()->td_db()->get_binlog_pmc()->get("sponsored_dialog_id");
    if (sponsored_dialog_id_string.empty()) {
      sponsored_dialog_id_string = G()->td_db()->get_binlog_pmc()->get("promoted_dialog_id");
      if (!sponsored_dialog_id_string.empty()) {
        G()->td_db()->get_binlog_pmc()->erase("promoted_dialog_id");
        G()->td_db()->get_binlog_pmc()->set("sponsored_dialog_id", sponsored_dialog_id_string);
      }
    }
    if (!sponsored_dialog_id_string.empty()) {
      auto r_dialog_id = to_integer_safe<int64>(sponsored_dialog_id_string);
      if (r_dialog_id.is_error()) {
        LOG(ERROR) << "Can't parse " << sponsored_dialog_id_string;
      } else {
        sponsored_dialog_id_ = DialogId(r_dialog_id.ok());
        if (!sponsored_dialog_id_.is_valid()) {
          LOG(ERROR) << "Have invalid chat ID " << sponsored_dialog_id_string;
          sponsored_dialog_id_ = DialogId();
        } else {
          Dialog *d = get_dialog_force(sponsored_dialog_id_);
          if (d == nullptr) {
            LOG(ERROR) << "Can't load " << sponsored_dialog_id_;
            sponsored_dialog_id_ = DialogId();
          }
        }
      }
    }

    auto unread_message_counts = G()->td_db()->get_binlog_pmc()->prefix_get("unread_message_count");
    for (auto &it : unread_message_counts) {
      auto r_folder_id = to_integer_safe<int32>(Slice(it.first).substr(Slice("unread_message_count").size()));
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }
      string total_count;
      string muted_count;
      std::tie(total_count, muted_count) = split(it.second);

      auto r_total_count = to_integer_safe<int32>(total_count);
      auto r_muted_count = to_integer_safe<int32>(muted_count);
      if (r_total_count.is_error() || r_muted_count.is_error()) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        FolderId folder_id(r_folder_id.ok());
        auto &list = get_dialog_list(folder_id);
        list.unread_message_total_count_ = r_total_count.ok();
        list.unread_message_muted_count_ = r_muted_count.ok();
        list.is_message_unread_count_inited_ = true;
        send_update_unread_message_count(folder_id, DialogId(), true, "load unread_message_count");
      }
    }

    auto unread_dialog_counts = G()->td_db()->get_binlog_pmc()->prefix_get("unread_dialog_count");
    for (auto &it : unread_dialog_counts) {
      auto r_folder_id = to_integer_safe<int32>(Slice(it.first).substr(Slice("unread_dialog_count").size()));
      if (r_folder_id.is_error()) {
        LOG(ERROR) << "Can't parse folder ID from " << it.first;
        continue;
      }

      auto counts = transform(full_split(it.second), [](Slice str) { return to_integer_safe<int32>(str); });
      if ((counts.size() != 4 && counts.size() != 6) ||
          std::any_of(counts.begin(), counts.end(), [](auto &c) { return c.is_error(); })) {
        LOG(ERROR) << "Can't parse " << it.second;
      } else {
        FolderId folder_id(r_folder_id.ok());
        auto &list = get_dialog_list(folder_id);
        list.unread_dialog_total_count_ = counts[0].ok();
        list.unread_dialog_muted_count_ = counts[1].ok();
        list.unread_dialog_marked_count_ = counts[2].ok();
        list.unread_dialog_muted_marked_count_ = counts[3].ok();
        if (counts.size() == 6) {
          list.server_dialog_total_count_ = counts[4].ok();
          list.secret_chat_total_count_ = counts[5].ok();
        }
        if (list.server_dialog_total_count_ == -1) {
          repair_server_dialog_total_count(folder_id);
        }
        if (list.secret_chat_total_count_ == -1) {
          repair_secret_chat_total_count(folder_id);
        }
        list.is_dialog_unread_count_inited_ = true;
        send_update_unread_chat_count(folder_id, DialogId(), true, "load unread_dialog_count");
      }
    }

    ttl_db_loop_start(G()->server_time());
  } else {
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("last_server_dialog_date");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("unread_message_count");
    G()->td_db()->get_binlog_pmc()->erase_by_prefix("unread_dialog_count");
    G()->td_db()->get_binlog_pmc()->erase("promoted_dialog_id");
    G()->td_db()->get_binlog_pmc()->erase("sponsored_dialog_id");
  }

  load_calls_db_state();

  vector<NotificationSettingsScope> scopes{NotificationSettingsScope::Private, NotificationSettingsScope::Group,
                                           NotificationSettingsScope::Channel};
  for (auto scope : scopes) {
    auto notification_settings_string =
        G()->td_db()->get_binlog_pmc()->get(get_notification_settings_scope_database_key(scope));
    if (!notification_settings_string.empty()) {
      ScopeNotificationSettings notification_settings;
      log_event_parse(notification_settings, notification_settings_string).ensure();

      auto current_settings = get_scope_notification_settings(scope);
      CHECK(current_settings != nullptr);
      current_settings->mute_until = -1;
      update_scope_notification_settings(scope, current_settings, notification_settings);
    }
  }
  if (!channels_notification_settings_.is_synchronized) {
    channels_notification_settings_ = chats_notification_settings_;
    channels_notification_settings_.disable_pinned_message_notifications = false;
    channels_notification_settings_.disable_mention_notifications = false;
    channels_notification_settings_.is_synchronized = false;
    if (td_->auth_manager_->is_authorized() && !td_->auth_manager_->is_bot()) {
      send_get_scope_notification_settings_query(NotificationSettingsScope::Channel, Promise<>());
    }
  }
  G()->td_db()->get_binlog_pmc()->erase("nsfac");

  /*
  FI LE *f = std::f open("error.txt", "r");
  if (f != nullptr) {
    DialogId dialog_id(ChannelId(123456));
    force_create_dialog(dialog_id, "test");
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);

    delete_all_dialog_messages(d, true, false);

    d->last_new_message_id = MessageId();
    d->last_read_inbox_message_id = MessageId();
    d->last_read_outbox_message_id = MessageId();
    d->is_last_read_inbox_message_id_inited = false;
    d->is_last_read_outbox_message_id_inited = false;

    struct MessageBasicInfo {
      MessageId message_id;
      bool have_previous;
      bool have_next;
    };
    vector<MessageBasicInfo> messages_info;
    std::function<void(Message *m)> get_messages_info = [&](Message *m) {
      if (m == nullptr) {
        return;
      }
      get_messages_info(m->left.get());
      messages_info.push_back(MessageBasicInfo{m->message_id, m->have_previous, m->have_next});
      get_messages_info(m->right.get());
    };

    char buf[1280];
    while (std::f gets(buf, sizeof(buf), f) != nullptr) {
      Slice log_string(buf, std::strlen(buf));
      Slice op = log_string.substr(0, log_string.find(' '));
      if (op != "MessageOpAdd" && op != "MessageOpDelete") {
        LOG(ERROR) << "Unsupported op " << op;
        continue;
      }
      log_string.remove_prefix(log_string.find(' ') + 1);

      if (!begins_with(log_string, "at ")) {
        LOG(ERROR) << "Date expected, found " << log_string;
        continue;
      }
      log_string.remove_prefix(3);
      auto date_slice = log_string.substr(0, log_string.find(' '));
      log_string.remove_prefix(date_slice.size());

      bool is_server = false;
      if (begins_with(log_string, " server message ")) {
        log_string.remove_prefix(16);
        is_server = true;
      } else if (begins_with(log_string, " yet unsent message ")) {
        log_string.remove_prefix(20);
      } else if (begins_with(log_string, " local message ")) {
        log_string.remove_prefix(15);
      } else {
        LOG(ERROR) << "Message id expected, found " << log_string;
        continue;
      }

      auto server_message_id = to_integer<int32>(log_string);
      auto add = 0;
      if (!is_server) {
        log_string.remove_prefix(log_string.find('.') + 1);
        add = to_integer<int32>(log_string);
      }
      log_string.remove_prefix(log_string.find(' ') + 1);

      auto message_id = MessageId(MessageId(ServerMessageId(server_message_id)).get() + add);

      auto content_type = log_string.substr(0, log_string.find(' '));
      log_string.remove_prefix(log_string.find(' ') + 1);

      auto read_bool = [](Slice &str) {
        if (begins_with(str, "true ")) {
          str.remove_prefix(5);
          return true;
        }
        if (begins_with(str, "false ")) {
          str.remove_prefix(6);
          return false;
        }
        LOG(ERROR) << "Bool expected, found " << str;
        return false;
      };

      bool from_update = read_bool(log_string);
      bool have_previous = read_bool(log_string);
      bool have_next = read_bool(log_string);

      if (op == "MessageOpAdd") {
        auto m = make_unique<Message>();
        set_message_id(m, message_id);
        m->date = G()->unix_time();
        m->content = create_text_message_content("text", {}, {});

        m->have_previous = have_previous;
        m->have_next = have_next;

        bool need_update = from_update;
        bool need_update_dialog_pos = false;
        if (add_message_to_dialog(dialog_id, std::move(m), from_update, &need_update, &need_update_dialog_pos,
                                  "Unknown source") == nullptr) {
          LOG(ERROR) << "Can't add message " << message_id;
        }
      } else {
        bool need_update_dialog_pos = false;
        auto m = delete_message(d, message_id, true, &need_update_dialog_pos, "Unknown source");
        CHECK(m != nullptr);
      }

      messages_info.clear();
      get_messages_info(d->messages.get());

      for (size_t i = 0; i + 1 < messages_info.size(); i++) {
        if (messages_info[i].have_next != messages_info[i + 1].have_previous) {
          LOG(ERROR) << messages_info[i].message_id << " has have_next = " << messages_info[i].have_next << ", but "
                     << messages_info[i + 1].message_id
                     << " has have_previous = " << messages_info[i + 1].have_previous;
        }
      }
      if (!messages_info.empty()) {
        if (messages_info.back().have_next != false) {
          LOG(ERROR) << messages_info.back().message_id << " has have_next = true, but there is no next message";
        }
        if (messages_info[0].have_previous != false) {
          LOG(ERROR) << messages_info[0].message_id << " has have_previous = true, but there is no previous message";
        }
      }
    }

    messages_info.clear();
    get_messages_info(d->messages.get());
    for (auto &info : messages_info) {
      bool need_update_dialog_pos = false;
      auto m = delete_message(d, info.message_id, true, &need_update_dialog_pos, "Unknown source");
      CHECK(m != nullptr);
    }

    std::f close(f);
  }
  */
}

void MessagesManager::ttl_db_loop_start(double server_now) {
  ttl_db_expires_from_ = 0;
  ttl_db_expires_till_ = static_cast<int32>(server_now) + 15 /* 15 seconds */;
  ttl_db_has_query_ = false;

  ttl_db_loop(server_now);
}

void MessagesManager::ttl_db_loop(double server_now) {
  LOG(INFO) << "Begin ttl_db loop: " << tag("expires_from", ttl_db_expires_from_)
            << tag("expires_till", ttl_db_expires_till_) << tag("has_query", ttl_db_has_query_);
  if (ttl_db_has_query_) {
    return;
  }

  auto now = static_cast<int32>(server_now);

  if (ttl_db_expires_till_ < 0) {
    LOG(INFO) << "Finish ttl_db loop";
    return;
  }

  if (now < ttl_db_expires_from_) {
    ttl_db_slot_.set_event(EventCreator::yield(actor_shared(this, YieldType::TtlDb)));
    auto wakeup_in = ttl_db_expires_from_ - server_now;
    ttl_db_slot_.set_timeout_in(wakeup_in);
    LOG(INFO) << "Set ttl_db timeout in " << wakeup_in;
    return;
  }

  ttl_db_has_query_ = true;
  int32 limit = 50;
  LOG(INFO) << "Send ttl_db query " << tag("expires_from", ttl_db_expires_from_)
            << tag("expires_till", ttl_db_expires_till_) << tag("limit", limit);
  G()->td_db()->get_messages_db_async()->get_expiring_messages(
      ttl_db_expires_from_, ttl_db_expires_till_, limit,
      PromiseCreator::lambda(
          [actor_id = actor_id(this)](Result<std::pair<std::vector<std::pair<DialogId, BufferSlice>>, int32>> result) {
            send_closure(actor_id, &MessagesManager::ttl_db_on_result, std::move(result), false);
          }));
}

void MessagesManager::ttl_db_on_result(Result<std::pair<std::vector<std::pair<DialogId, BufferSlice>>, int32>> r_result,
                                       bool dummy) {
  auto result = r_result.move_as_ok();
  ttl_db_has_query_ = false;
  ttl_db_expires_from_ = ttl_db_expires_till_;
  ttl_db_expires_till_ = result.second;

  LOG(INFO) << "Receive ttl_db query result " << tag("new expires_till", ttl_db_expires_till_)
            << tag("got messages", result.first.size());
  for (auto &dialog_message : result.first) {
    on_get_message_from_database(dialog_message.first, get_dialog_force(dialog_message.first), dialog_message.second,
                                 false, "ttl_db_on_result");
  }
  ttl_db_loop(G()->server_time());
}

void MessagesManager::on_send_secret_message_error(int64 random_id, Status error, Promise<> promise) {
  promise.set_value(Unit());  // TODO: set after error is saved
  LOG(INFO) << "Receive error for SecretChatsManager::send_message: " << error;

  auto it = being_sent_messages_.find(random_id);
  if (it != being_sent_messages_.end()) {
    auto full_message_id = it->second;
    auto *m = get_message(full_message_id);
    if (m != nullptr) {
      auto file_id = get_message_content_upload_file_id(m->content.get());
      if (file_id.is_valid()) {
        if (G()->close_flag() && G()->parameters().use_message_db) {
          // do not send error, message will be re-sent
          return;
        }
        if (begins_with(error.message(), "FILE_PART_") && ends_with(error.message(), "_MISSING")) {
          on_send_message_file_part_missing(random_id, to_integer<int32>(error.message().substr(10)));
          return;
        }

        if (error.code() != 429 && error.code() < 500 && !G()->close_flag()) {
          td_->file_manager_->delete_partial_remote_location(file_id);
        }
      }
    }
  }

  on_send_message_fail(random_id, std::move(error));
}

void MessagesManager::on_send_secret_message_success(int64 random_id, MessageId message_id, int32 date,
                                                     tl_object_ptr<telegram_api::EncryptedFile> file_ptr,
                                                     Promise<> promise) {
  promise.set_value(Unit());  // TODO: set after message is saved

  FileId new_file_id;
  if (file_ptr != nullptr && file_ptr->get_id() == telegram_api::encryptedFile::ID) {
    auto file = move_tl_object_as<telegram_api::encryptedFile>(file_ptr);
    if (!DcId::is_valid(file->dc_id_)) {
      LOG(ERROR) << "Wrong dc_id = " << file->dc_id_ << " in file " << to_string(file);
    } else {
      DialogId owner_dialog_id;
      auto it = being_sent_messages_.find(random_id);
      if (it != being_sent_messages_.end()) {
        owner_dialog_id = it->second.get_dialog_id();
      }

      new_file_id = td_->file_manager_->register_remote(
          FullRemoteFileLocation(FileType::Encrypted, file->id_, file->access_hash_, DcId::internal(file->dc_id_), ""),
          FileLocationSource::FromServer, owner_dialog_id, 0, file->size_, to_string(static_cast<uint64>(file->id_)));
    }
  }

  on_send_message_success(random_id, message_id, date, new_file_id, "process send_secret_message_success");
}

void MessagesManager::delete_secret_messages(SecretChatId secret_chat_id, std::vector<int64> random_ids,
                                             Promise<> promise) {
  LOG(DEBUG) << "On delete messages in " << secret_chat_id << " with random_ids " << random_ids;
  CHECK(secret_chat_id.is_valid());

  DialogId dialog_id(secret_chat_id);
  if (!have_dialog_force(dialog_id)) {
    LOG(ERROR) << "Ignore delete secret messages in unknown " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  pending_secret_message->type = PendingSecretMessage::Type::DeleteMessages;
  pending_secret_message->dialog_id = dialog_id;
  pending_secret_message->random_ids = std::move(random_ids);

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::finish_delete_secret_messages(DialogId dialog_id, std::vector<int64> random_ids,
                                                    Promise<> promise) {
  LOG(INFO) << "Delete messages with random_ids " << random_ids << " in " << dialog_id;
  promise.set_value(Unit());  // TODO: set after event is saved

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  vector<MessageId> to_delete_message_ids;
  for (auto &random_id : random_ids) {
    auto message_id = get_message_id_by_random_id(d, random_id, "delete_secret_messages");
    if (!message_id.is_valid()) {
      LOG(INFO) << "Can't find message with random_id " << random_id;
      continue;
    }
    const Message *m = get_message(d, message_id);
    CHECK(m != nullptr);
    if (!is_service_message_content(m->content->get_type())) {
      to_delete_message_ids.push_back(message_id);
    } else {
      LOG(INFO) << "Skip deletion of service " << message_id;
    }
  }
  delete_dialog_messages_from_updates(dialog_id, to_delete_message_ids);
}

void MessagesManager::delete_secret_chat_history(SecretChatId secret_chat_id, MessageId last_message_id,
                                                 Promise<> promise) {
  LOG(DEBUG) << "On delete history in " << secret_chat_id << " up to " << last_message_id;
  CHECK(secret_chat_id.is_valid());
  CHECK(!last_message_id.is_scheduled());

  DialogId dialog_id(secret_chat_id);
  if (!have_dialog_force(dialog_id)) {
    LOG(ERROR) << "Ignore delete history in unknown " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  pending_secret_message->type = PendingSecretMessage::Type::DeleteHistory;
  pending_secret_message->dialog_id = dialog_id;
  pending_secret_message->last_message_id = last_message_id;

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::finish_delete_secret_chat_history(DialogId dialog_id, MessageId last_message_id,
                                                        Promise<> promise) {
  LOG(DEBUG) << "Delete history in " << dialog_id << " up to " << last_message_id;
  promise.set_value(Unit());  // TODO: set after event is saved
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  // TODO: probably last_message_id is not needed
  delete_all_dialog_messages(d, false, true);
}

void MessagesManager::read_secret_chat_outbox(SecretChatId secret_chat_id, int32 up_to_date, int32 read_date) {
  if (!secret_chat_id.is_valid()) {
    LOG(ERROR) << "Receive read secret chat outbox in the invalid " << secret_chat_id;
    return;
  }
  auto dialog_id = DialogId(secret_chat_id);
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return;
  }

  if (read_date > 0) {
    auto user_id = td_->contacts_manager_->get_secret_chat_user_id(secret_chat_id);
    if (user_id.is_valid()) {
      td_->contacts_manager_->on_update_user_local_was_online(user_id, read_date);
    }
  }

  // TODO: protect with logevent
  suffix_load_till_date(
      d, up_to_date,
      PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, up_to_date, read_date](Result<Unit> result) {
        send_closure(actor_id, &MessagesManager::read_secret_chat_outbox_inner, dialog_id, up_to_date, read_date);
      }));
}

void MessagesManager::read_secret_chat_outbox_inner(DialogId dialog_id, int32 up_to_date, int32 read_date) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto end = MessagesConstIterator(d, MessageId::max());
  while (*end && (*end)->date > up_to_date) {
    --end;
  }
  if (!*end) {
    LOG(INFO) << "Ignore read_secret_chat_outbox in " << dialog_id << " at " << up_to_date
              << ": no messages with such date are known";
    return;
  }
  auto max_message_id = (*end)->message_id;
  read_history_outbox(dialog_id, max_message_id, read_date);
}

void MessagesManager::open_secret_message(SecretChatId secret_chat_id, int64 random_id, Promise<> promise) {
  promise.set_value(Unit());  // TODO: set after event is saved
  DialogId dialog_id(secret_chat_id);
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(ERROR) << "Ignore opening secret chat message in unknown " << dialog_id;
    return;
  }

  auto message_id = get_message_id_by_random_id(d, random_id, "open_secret_message");
  if (!message_id.is_valid()) {
    return;
  }
  Message *m = get_message(d, message_id);
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent() || m->is_failed_to_send || !m->is_outgoing) {
    LOG(ERROR) << "Peer has opened wrong " << message_id << " in " << dialog_id;
    return;
  }

  read_message_content(d, m, false, "open_secret_message");
}

void MessagesManager::on_update_secret_chat_state(SecretChatId secret_chat_id, SecretChatState state) {
  if (state == SecretChatState::Closed) {
    DialogId dialog_id(secret_chat_id);
    Dialog *d = get_dialog_force(dialog_id);
    if (d != nullptr) {
      if (d->new_secret_chat_notification_id.is_valid()) {
        remove_new_secret_chat_notification(d, true);
      }
      if (d->message_notification_group.group_id.is_valid() && get_dialog_pending_notification_count(d, false) == 0 &&
          !d->message_notification_group.last_notification_id.is_valid()) {
        CHECK(d->message_notification_group.last_notification_date == 0);
        d->message_notification_group.try_reuse = true;
        d->message_notification_group.is_changed = true;
        on_dialog_updated(d->dialog_id, "on_update_secret_chat_state");
      }
      CHECK(!d->mention_notification_group.group_id.is_valid());  // there can't be unread mentions in secret chats
    }
  }
}

void MessagesManager::on_get_secret_message(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                            int32 date, tl_object_ptr<telegram_api::encryptedFile> file,
                                            tl_object_ptr<secret_api::decryptedMessage> message, Promise<> promise) {
  LOG(DEBUG) << "On get " << to_string(message);
  CHECK(message != nullptr);
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = message->random_id_;
  message_info.ttl = message->ttl_;

  Dialog *d = get_dialog_force(message_info.dialog_id);
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  pending_secret_message->load_data_multipromise.add_promise(Auto());
  auto lock_promise = pending_secret_message->load_data_multipromise.get_promise();

  int32 flags = MESSAGE_FLAG_HAS_UNREAD_CONTENT | MESSAGE_FLAG_HAS_FROM_ID;
  if ((message->flags_ & secret_api::decryptedMessage::REPLY_TO_RANDOM_ID_MASK) != 0) {
    message_info.reply_to_message_id = get_message_id_by_random_id(
        get_dialog(message_info.dialog_id), message->reply_to_random_id_, "on_get_secret_message");
    if (message_info.reply_to_message_id.is_valid()) {
      flags |= MESSAGE_FLAG_IS_REPLY;
    }
  }
  if ((message->flags_ & secret_api::decryptedMessage::ENTITIES_MASK) != 0) {
    flags |= MESSAGE_FLAG_HAS_ENTITIES;
  }
  if ((message->flags_ & secret_api::decryptedMessage::MEDIA_MASK) != 0) {
    flags |= MESSAGE_FLAG_HAS_MEDIA;
  }

  if (!clean_input_string(message->via_bot_name_)) {
    LOG(WARNING) << "Receive invalid bot username " << message->via_bot_name_;
    message->via_bot_name_.clear();
  }
  if ((message->flags_ & secret_api::decryptedMessage::VIA_BOT_NAME_MASK) != 0 && !message->via_bot_name_.empty()) {
    pending_secret_message->load_data_multipromise.add_promise(
        PromiseCreator::lambda([this, via_bot_name = message->via_bot_name_, &flags = message_info.flags,
                                &via_bot_user_id = message_info.via_bot_user_id](Unit) mutable {
          auto dialog_id = resolve_dialog_username(via_bot_name);
          if (dialog_id.is_valid() && dialog_id.get_type() == DialogType::User) {
            flags |= MESSAGE_FLAG_IS_SENT_VIA_BOT;
            via_bot_user_id = dialog_id.get_user_id();
          }
        }));
    search_public_dialog(message->via_bot_name_, false, pending_secret_message->load_data_multipromise.get_promise());
  }
  if ((message->flags_ & secret_api::decryptedMessage::GROUPED_ID_MASK) != 0 && message->grouped_id_ != 0) {
    message_info.media_album_id = message->grouped_id_;
    flags |= MESSAGE_FLAG_HAS_MEDIA_ALBUM_ID;
  }

  message_info.flags = flags;
  message_info.content = get_secret_message_content(
      td_, std::move(message->message_), std::move(file), std::move(message->media_), std::move(message->entities_),
      message_info.dialog_id, pending_secret_message->load_data_multipromise);

  add_secret_message(std::move(pending_secret_message), std::move(lock_promise));
}

void MessagesManager::on_secret_chat_screenshot_taken(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                                      int32 date, int64 random_id, Promise<> promise) {
  LOG(DEBUG) << "On screenshot taken in " << secret_chat_id;
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = random_id;
  message_info.flags = MESSAGE_FLAG_HAS_FROM_ID;
  message_info.content = create_screenshot_taken_message_content();

  Dialog *d = get_dialog_force(message_info.dialog_id);
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::on_secret_chat_ttl_changed(SecretChatId secret_chat_id, UserId user_id, MessageId message_id,
                                                 int32 date, int32 ttl, int64 random_id, Promise<> promise) {
  LOG(DEBUG) << "On ttl set in " << secret_chat_id << " to " << ttl;
  CHECK(secret_chat_id.is_valid());
  CHECK(user_id.is_valid());
  CHECK(message_id.is_valid());
  CHECK(date > 0);
  if (ttl < 0) {
    LOG(WARNING) << "Receive wrong ttl = " << ttl;
    promise.set_value(Unit());
    return;
  }

  auto pending_secret_message = make_unique<PendingSecretMessage>();
  pending_secret_message->success_promise = std::move(promise);
  MessageInfo &message_info = pending_secret_message->message_info;
  message_info.dialog_id = DialogId(secret_chat_id);
  message_info.message_id = message_id;
  message_info.sender_user_id = user_id;
  message_info.date = date;
  message_info.random_id = random_id;
  message_info.flags = MESSAGE_FLAG_HAS_FROM_ID;
  message_info.content = create_chat_set_ttl_message_content(ttl);

  Dialog *d = get_dialog_force(message_info.dialog_id);
  if (d == nullptr) {
    LOG(ERROR) << "Ignore secret message in unknown " << message_info.dialog_id;
    pending_secret_message->success_promise.set_error(Status::Error(500, "Chat not found"));
    return;
  }

  add_secret_message(std::move(pending_secret_message));
}

void MessagesManager::add_secret_message(unique_ptr<PendingSecretMessage> pending_secret_message,
                                         Promise<Unit> lock_promise) {
  auto &multipromise = pending_secret_message->load_data_multipromise;
  multipromise.set_ignore_errors(true);
  int64 token = pending_secret_messages_.add(std::move(pending_secret_message));

  multipromise.add_promise(PromiseCreator::lambda([token, actor_id = actor_id(this),
                                                   this](Result<Unit> result) mutable {
    if (result.is_ok() && !G()->close_flag()) {  // if we aren't closing
      this->pending_secret_messages_.finish(token, [actor_id](unique_ptr<PendingSecretMessage> pending_secret_message) {
        send_closure_later(actor_id, &MessagesManager::finish_add_secret_message, std::move(pending_secret_message));
      });
    }
  }));

  if (!lock_promise) {
    lock_promise = multipromise.get_promise();
  }
  lock_promise.set_value(Unit());
}

void MessagesManager::finish_add_secret_message(unique_ptr<PendingSecretMessage> pending_secret_message) {
  if (G()->close_flag()) {
    return;
  }

  if (pending_secret_message->type == PendingSecretMessage::Type::DeleteMessages) {
    return finish_delete_secret_messages(pending_secret_message->dialog_id,
                                         std::move(pending_secret_message->random_ids),
                                         std::move(pending_secret_message->success_promise));
  }
  if (pending_secret_message->type == PendingSecretMessage::Type::DeleteHistory) {
    return finish_delete_secret_chat_history(pending_secret_message->dialog_id, pending_secret_message->last_message_id,
                                             std::move(pending_secret_message->success_promise));
  }

  auto d = get_dialog(pending_secret_message->message_info.dialog_id);
  CHECK(d != nullptr);
  auto random_id = pending_secret_message->message_info.random_id;
  auto message_id = get_message_id_by_random_id(d, random_id, "finish_add_secret_message");
  if (message_id.is_valid()) {
    if (message_id != pending_secret_message->message_info.message_id) {
      LOG(WARNING) << "Ignore duplicate " << pending_secret_message->message_info.message_id
                   << " received earlier with " << message_id << " and random_id " << random_id;
    }
  } else {
    on_get_message(std::move(pending_secret_message->message_info), true, false, true, true,
                   "finish add secret message");
  }
  pending_secret_message->success_promise.set_value(Unit());  // TODO: set after message is saved
}

void MessagesManager::fix_message_info_dialog_id(MessageInfo &message_info) const {
  if (message_info.dialog_id != get_my_dialog_id()) {
    return;
  }

  UserId sender_user_id = message_info.sender_user_id;
  if (!sender_user_id.is_valid()) {
    LOG(ERROR) << "Receive invalid sender user id in private chat";
    return;
  }

  message_info.dialog_id = DialogId(sender_user_id);
  LOG_IF(ERROR, !message_info.message_id.is_scheduled() && (message_info.flags & MESSAGE_FLAG_IS_OUT) != 0)
      << "Receive message out flag for incoming " << message_info.message_id << " in " << message_info.dialog_id;
}

MessagesManager::MessageInfo MessagesManager::parse_telegram_api_message(
    tl_object_ptr<telegram_api::Message> message_ptr, bool is_scheduled, const char *source) const {
  LOG(DEBUG) << "Receive from " << source << " " << to_string(message_ptr);
  LOG_CHECK(message_ptr != nullptr) << source;

  MessageInfo message_info;
  message_info.message_id = get_message_id(message_ptr, is_scheduled);
  switch (message_ptr->get_id()) {
    case telegram_api::messageEmpty::ID:
      message_info.message_id = MessageId();
      break;
    case telegram_api::message::ID: {
      auto message = move_tl_object_as<telegram_api::message>(message_ptr);

      message_info.dialog_id = DialogId(message->to_id_);
      if (message->flags_ & MESSAGE_FLAG_HAS_FROM_ID) {
        message_info.sender_user_id = UserId(message->from_id_);
      }
      message_info.date = message->date_;
      message_info.forward_header = std::move(message->fwd_from_);
      message_info.reply_to_message_id = MessageId(ServerMessageId(
          message->flags_ & MESSAGE_FLAG_IS_REPLY ? message->reply_to_msg_id_ : 0));  // TODO zero init in fetch
      if (message->flags_ & MESSAGE_FLAG_IS_SENT_VIA_BOT) {
        message_info.via_bot_user_id = UserId(message->via_bot_id_);
        if (!message_info.via_bot_user_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << message_info.via_bot_user_id << " from " << source;
          message_info.via_bot_user_id = UserId();
        }
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_VIEWS) {
        message_info.views = message->views_;
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_EDIT_DATE) {
        message_info.edit_date = message->edit_date_;
      }
      if (message->flags_ & MESSAGE_FLAG_HAS_MEDIA_ALBUM_ID) {
        message_info.media_album_id = message->grouped_id_;
      }
      message_info.flags = message->flags_;
      fix_message_info_dialog_id(message_info);
      bool is_content_read = (message->flags_ & MESSAGE_FLAG_HAS_UNREAD_CONTENT) == 0;
      if (is_message_auto_read(message_info.dialog_id, (message->flags_ & MESSAGE_FLAG_IS_OUT) != 0)) {
        is_content_read = true;
      }
      if (is_scheduled) {
        is_content_read = false;
      }
      auto new_source = PSTRING() << FullMessageId(message_info.dialog_id, message_info.message_id) << " from "
                                  << source;
      message_info.content = get_message_content(
          td_,
          get_message_text(td_->contacts_manager_.get(), std::move(message->message_), std::move(message->entities_),
                           true, message_info.forward_header ? message_info.forward_header->date_ : message_info.date,
                           message_info.media_album_id != 0, new_source.c_str()),
          std::move(message->media_), message_info.dialog_id, is_content_read, message_info.via_bot_user_id,
          &message_info.ttl);
      message_info.reply_markup =
          message->flags_ & MESSAGE_FLAG_HAS_REPLY_MARKUP ? std::move(message->reply_markup_) : nullptr;
      message_info.restriction_reasons = get_restriction_reasons(std::move(message->restriction_reason_));
      message_info.author_signature = std::move(message->post_author_);
      break;
    }
    case telegram_api::messageService::ID: {
      auto message = move_tl_object_as<telegram_api::messageService>(message_ptr);

      message_info.dialog_id = DialogId(message->to_id_);
      if (message->flags_ & MESSAGE_FLAG_HAS_FROM_ID) {
        message_info.sender_user_id = UserId(message->from_id_);
      }
      message_info.date = message->date_;
      message_info.flags = message->flags_;
      fix_message_info_dialog_id(message_info);
      MessageId reply_to_message_id = MessageId(ServerMessageId(
          message->flags_ & MESSAGE_FLAG_IS_REPLY ? message->reply_to_msg_id_ : 0));  // TODO zero init in fetch
      message_info.content =
          get_action_message_content(td_, std::move(message->action_), message_info.dialog_id, reply_to_message_id);
      break;
    }
    default:
      UNREACHABLE();
      break;
  }
  return message_info;
}

std::pair<DialogId, unique_ptr<MessagesManager::Message>> MessagesManager::create_message(MessageInfo &&message_info,
                                                                                          bool is_channel_message) {
  DialogId dialog_id = message_info.dialog_id;
  MessageId message_id = message_info.message_id;
  if ((!message_id.is_valid() && !message_id.is_valid_scheduled()) || !dialog_id.is_valid()) {
    if (message_id != MessageId() || dialog_id != DialogId()) {
      LOG(ERROR) << "Receive " << message_id << " in " << dialog_id;
    }
    return {DialogId(), nullptr};
  }
  if (message_id.is_yet_unsent() || message_id.is_local()) {
    LOG(ERROR) << "Receive " << message_id;
    return {DialogId(), nullptr};
  }

  CHECK(message_info.content != nullptr);

  auto dialog_type = dialog_id.get_type();
  UserId sender_user_id = message_info.sender_user_id;
  if (!sender_user_id.is_valid()) {
    if (!is_broadcast_channel(dialog_id) && td_->auth_manager_->is_bot()) {
      sender_user_id = td_->contacts_manager_->get_service_notifications_user_id();
    } else if (sender_user_id != UserId()) {
      LOG(ERROR) << "Receive invalid " << sender_user_id;
      sender_user_id = UserId();
    }
  }
  if (message_id.is_scheduled()) {
    is_channel_message = (dialog_type == DialogType::Channel);
  }

  int32 flags = message_info.flags;
  if (flags &
      ~(MESSAGE_FLAG_IS_OUT | MESSAGE_FLAG_IS_FORWARDED | MESSAGE_FLAG_IS_REPLY | MESSAGE_FLAG_HAS_MENTION |
        MESSAGE_FLAG_HAS_UNREAD_CONTENT | MESSAGE_FLAG_HAS_REPLY_MARKUP | MESSAGE_FLAG_HAS_ENTITIES |
        MESSAGE_FLAG_HAS_FROM_ID | MESSAGE_FLAG_HAS_MEDIA | MESSAGE_FLAG_HAS_VIEWS | MESSAGE_FLAG_IS_SENT_VIA_BOT |
        MESSAGE_FLAG_IS_SILENT | MESSAGE_FLAG_IS_POST | MESSAGE_FLAG_HAS_EDIT_DATE | MESSAGE_FLAG_HAS_AUTHOR_SIGNATURE |
        MESSAGE_FLAG_HAS_MEDIA_ALBUM_ID | MESSAGE_FLAG_IS_FROM_SCHEDULED | MESSAGE_FLAG_IS_LEGACY |
        MESSAGE_FLAG_HIDE_EDIT_DATE | MESSAGE_FLAG_IS_RESTRICTED)) {
    LOG(ERROR) << "Unsupported message flags = " << flags << " received";
  }

  bool is_outgoing = (flags & MESSAGE_FLAG_IS_OUT) != 0;
  bool is_silent = (flags & MESSAGE_FLAG_IS_SILENT) != 0;
  bool is_channel_post = (flags & MESSAGE_FLAG_IS_POST) != 0;
  bool is_legacy = (flags & MESSAGE_FLAG_IS_LEGACY) != 0;
  bool hide_edit_date = (flags & MESSAGE_FLAG_HIDE_EDIT_DATE) != 0;
  bool is_from_scheduled = (flags & MESSAGE_FLAG_IS_FROM_SCHEDULED) != 0;

  LOG_IF(ERROR, is_channel_message != (dialog_type == DialogType::Channel))
      << "is_channel_message is wrong for message received in the " << dialog_id;
  LOG_IF(ERROR, is_channel_post && !is_broadcast_channel(dialog_id))
      << "is_channel_post is true for message received in the " << dialog_id;

  UserId my_id = td_->contacts_manager_->get_my_id();
  DialogId my_dialog_id = DialogId(my_id);
  if (dialog_id == my_dialog_id) {
    // dialog_id should be already fixed
    CHECK(sender_user_id == my_id);
  }

  bool supposed_to_be_outgoing = sender_user_id == my_id && !(dialog_id == my_dialog_id && !message_id.is_scheduled());
  if (sender_user_id.is_valid() && supposed_to_be_outgoing != is_outgoing) {
    LOG(ERROR) << "Receive wrong message out flag: me is " << my_id << ", message is from " << sender_user_id
               << ", flags = " << flags << " for " << message_id << " in " << dialog_id;
    is_outgoing = supposed_to_be_outgoing;

    if (dialog_type == DialogType::Channel && !running_get_difference_ && !running_get_channel_difference(dialog_id) &&
        get_channel_difference_to_logevent_id_.count(dialog_id) == 0) {
      // it is safer to completely ignore the message and re-get it through getChannelsDifference
      Dialog *d = get_dialog(dialog_id);
      if (d != nullptr) {
        channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
        return {DialogId(), nullptr};
      }
    }
  }

  MessageId reply_to_message_id = message_info.reply_to_message_id;
  CHECK(!reply_to_message_id.is_scheduled());
  if (!message_id.is_scheduled() && reply_to_message_id != MessageId() &&
      (!reply_to_message_id.is_valid() || reply_to_message_id >= message_id)) {
    if (!reply_to_message_id.is_valid() ||
        reply_to_message_id.get() - message_id.get() <= MessageId(ServerMessageId(2000000000)).get()) {
      LOG(ERROR) << "Receive reply to wrong " << reply_to_message_id << " in " << message_id;
    }
    reply_to_message_id = MessageId();
  }

  UserId via_bot_user_id = message_info.via_bot_user_id;
  if (!via_bot_user_id.is_valid()) {
    via_bot_user_id = UserId();
  }

  int32 date = message_info.date;
  if (date <= 0) {
    LOG(ERROR) << "Wrong date = " << date << " received in " << message_id << " in " << dialog_id;
    date = 1;
  }

  int32 edit_date = message_info.edit_date;
  if (edit_date < 0) {
    LOG(ERROR) << "Wrong edit_date = " << edit_date << " received in " << message_id << " in " << dialog_id;
    edit_date = 0;
  }

  if (hide_edit_date && td_->auth_manager_->is_bot()) {
    hide_edit_date = false;
  }

  int32 ttl = message_info.ttl;
  auto content_type = message_info.content->get_type();
  bool is_content_secret = is_secret_message_content(ttl, content_type);  // should be calculated before TTL is adjusted
  if (ttl < 0) {
    LOG(ERROR) << "Wrong ttl = " << ttl << " received in " << message_id << " in " << dialog_id;
    ttl = 0;
  } else if (ttl > 0) {
    ttl = max(ttl, get_message_content_duration(message_info.content.get(), td_) + 1);
  }

  int32 views = message_info.views;
  if (views < 0) {
    LOG(ERROR) << "Wrong views = " << views << " received in " << message_id << " in " << dialog_id;
    views = 0;
  }

  bool has_forward_info = message_info.forward_header != nullptr;

  LOG(INFO) << "Receive " << message_id << " in " << dialog_id << " from " << sender_user_id;

  auto message = make_unique<Message>();
  set_message_id(message, message_id);
  message->sender_user_id = sender_user_id;
  message->date = date;
  message->ttl = ttl;
  message->edit_date = edit_date;
  message->random_id = message_info.random_id;
  message->forward_info = get_message_forward_info(std::move(message_info.forward_header));
  message->reply_to_message_id = reply_to_message_id;
  message->via_bot_user_id = via_bot_user_id;
  message->restriction_reasons = std::move(message_info.restriction_reasons);
  message->author_signature = std::move(message_info.author_signature);
  message->is_outgoing = is_outgoing;
  message->is_channel_post = is_channel_post;
  message->contains_mention =
      !is_outgoing && dialog_type != DialogType::User &&
      ((flags & MESSAGE_FLAG_HAS_MENTION) != 0 || content_type == MessageContentType::PinMessage);
  message->contains_unread_mention =
      !message_id.is_scheduled() && message_id.is_server() && message->contains_mention &&
      (flags & MESSAGE_FLAG_HAS_UNREAD_CONTENT) != 0 &&
      (dialog_type == DialogType::Chat || (dialog_type == DialogType::Channel && !is_broadcast_channel(dialog_id)));
  message->disable_notification = is_silent;
  message->is_content_secret = is_content_secret;
  message->hide_edit_date = hide_edit_date;
  message->is_from_scheduled = is_from_scheduled;
  message->views = views;
  message->legacy_layer = (is_legacy ? MTPROTO_LAYER : 0);
  message->content = std::move(message_info.content);
  message->reply_markup = get_reply_markup(std::move(message_info.reply_markup), td_->auth_manager_->is_bot(), false,
                                           message->contains_mention || dialog_id.get_type() == DialogType::User);

  if (content_type == MessageContentType::ExpiredPhoto || content_type == MessageContentType::ExpiredVideo) {
    CHECK(message->ttl == 0);  // ttl is ignored/set to 0 if the message has already been expired
    if (message->reply_markup != nullptr) {
      if (message->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
        message->had_reply_markup = true;
      }
      message->reply_markup = nullptr;
    }
    message->reply_to_message_id = MessageId();
  }

  if (message_info.media_album_id != 0) {
    if (!is_allowed_media_group_content(content_type)) {
      LOG(ERROR) << "Receive media group id " << message_info.media_album_id << " in " << message_id << " from "
                 << dialog_id << " with content "
                 << oneline(to_string(
                        get_message_content_object(message->content.get(), td_, message->date, is_content_secret)));
    } else {
      message->media_album_id = message_info.media_album_id;
    }
  }

  if (message->forward_info == nullptr && has_forward_info) {
    message->had_forward_info = true;
  }

  return {dialog_id, std::move(message)};
}

MessageId MessagesManager::find_old_message_id(DialogId dialog_id, MessageId message_id) const {
  if (message_id.is_scheduled()) {
    CHECK(message_id.is_scheduled_server());
    auto dialog_it = update_scheduled_message_ids_.find(dialog_id);
    if (dialog_it != update_scheduled_message_ids_.end()) {
      auto it = dialog_it->second.find(message_id.get_scheduled_server_message_id());
      if (it != dialog_it->second.end()) {
        return it->second;
      }
    }
  } else {
    CHECK(message_id.is_server());
    auto it = update_message_ids_.find(FullMessageId(dialog_id, message_id));
    if (it != update_message_ids_.end()) {
      return it->second;
    }
  }
  return MessageId();
}

FullMessageId MessagesManager::on_get_message(tl_object_ptr<telegram_api::Message> message_ptr, bool from_update,
                                              bool is_channel_message, bool is_scheduled, bool have_previous,
                                              bool have_next, const char *source) {
  return on_get_message(parse_telegram_api_message(std::move(message_ptr), is_scheduled, source), from_update,
                        is_channel_message, have_previous, have_next, source);
}

FullMessageId MessagesManager::on_get_message(MessageInfo &&message_info, bool from_update, bool is_channel_message,
                                              bool have_previous, bool have_next, const char *source) {
  DialogId dialog_id;
  unique_ptr<Message> new_message;
  std::tie(dialog_id, new_message) = create_message(std::move(message_info), is_channel_message);
  if (new_message == nullptr) {
    return FullMessageId();
  }
  MessageId message_id = new_message->message_id;

  new_message->have_previous = have_previous;
  new_message->have_next = have_next;

  bool need_update = from_update || message_id.is_scheduled();
  bool need_update_dialog_pos = false;

  MessageId old_message_id = find_old_message_id(dialog_id, message_id);
  bool need_add_active_live_location = false;
  LOG(INFO) << "Found old " << old_message_id << " by " << FullMessageId{dialog_id, message_id};
  if (old_message_id.is_valid() || old_message_id.is_valid_scheduled()) {
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);

    if (!from_update && !message_id.is_scheduled()) {
      if (message_id <= d->last_new_message_id) {
        if (get_message_force(d, message_id, "receive missed unsent message not from update") != nullptr) {
          LOG(ERROR) << "New " << old_message_id << "/" << message_id << " in " << dialog_id << " from " << source
                     << " has id less than last_new_message_id = " << d->last_new_message_id;
          return FullMessageId();
        }
        // if there is no message yet, then it is likely was missed because of a server bug and is being repaired via
        // get_message_from_server from after_get_difference
        // TODO move to INFO
        LOG(ERROR) << "Receive " << old_message_id << "/" << message_id << " in " << dialog_id << " from " << source
                   << " with id less than last_new_message_id = " << d->last_new_message_id
                   << " and trying to add it anyway";
      } else {
        LOG(ERROR) << "Ignore " << old_message_id << "/" << message_id << " received not through update from " << source
                   << ": "
                   << oneline(to_string(get_message_object(dialog_id, new_message.get())));  // TODO move to INFO
        dump_debug_message_op(d, 3);                                                         // TODO remove
        if (dialog_id.get_type() == DialogType::Channel && have_input_peer(dialog_id, AccessRights::Read)) {
          channel_get_difference_retry_timeout_.add_timeout_in(dialog_id.get(), 0.001);
        }
        return FullMessageId();
      }
    }

    if (message_id.is_scheduled()) {
      auto dialog_it = update_scheduled_message_ids_.find(dialog_id);
      CHECK(dialog_it != update_scheduled_message_ids_.end());
      dialog_it->second.erase(message_id.get_scheduled_server_message_id());
      if (dialog_it->second.empty()) {
        update_scheduled_message_ids_.erase(dialog_it);
      }
    } else {
      update_message_ids_.erase(FullMessageId(dialog_id, message_id));
    }

    if (!new_message->is_outgoing && dialog_id != get_my_dialog_id()) {
      // sent message is not from me
      LOG(ERROR) << "Sent in " << dialog_id << " " << message_id << " is sent by " << new_message->sender_user_id;
      return FullMessageId();
    }

    unique_ptr<Message> old_message =
        delete_message(d, old_message_id, false, &need_update_dialog_pos, "add sent message");
    if (old_message == nullptr) {
      // message has already been deleted by the user or sent to inaccessible channel
      // don't need to send update to the user, because the message has already been deleted
      LOG(INFO) << "Delete already deleted sent " << new_message->message_id << " from server";
      delete_message_from_server(dialog_id, new_message->message_id, true);
      return FullMessageId();
    }

    need_update = false;

    set_message_id(new_message, old_message_id);
    new_message->from_database = false;
    new_message->have_previous = false;
    new_message->have_next = false;
    update_message(d, old_message.get(), std::move(new_message), &need_update_dialog_pos);
    new_message = std::move(old_message);

    set_message_id(new_message, message_id);
    send_update_message_send_succeeded(d, old_message_id, new_message.get());

    if (!message_id.is_scheduled()) {
      need_add_active_live_location = true;

      // add_message_to_dialog will not update counts, because need_update == false
      update_message_count_by_index(d, +1, new_message.get());
    }

    if (!from_update) {
      new_message->have_previous = have_previous;
      new_message->have_next = have_next;
    } else {
      new_message->have_previous = true;
      new_message->have_next = true;
    }
  }

  const Message *m = add_message_to_dialog(dialog_id, std::move(new_message), from_update, &need_update,
                                           &need_update_dialog_pos, source);
  Dialog *d = get_dialog(dialog_id);
  if (m == nullptr) {
    if (need_update_dialog_pos && d != nullptr) {
      send_update_chat_last_message(d, "on_get_message");
    }

    return FullMessageId();
  }

  CHECK(d != nullptr);

  if (need_add_active_live_location) {
    try_add_active_live_location(dialog_id, m);
  }

  auto pcc_it = pending_created_dialogs_.find(dialog_id);
  if (from_update && pcc_it != pending_created_dialogs_.end()) {
    pcc_it->second.set_value(Unit());

    pending_created_dialogs_.erase(pcc_it);
  }

  if (need_update) {
    send_update_new_message(d, m);
  }

  if (dialog_id.get_type() == DialogType::Channel && !have_input_peer(dialog_id, AccessRights::Read)) {
    auto p = delete_message(d, message_id, false, &need_update_dialog_pos, "get a message in inaccessible chat");
    CHECK(p.get() == m);
    // CHECK(d->messages == nullptr);
    send_update_delete_messages(dialog_id, {p->message_id.get()}, false, false);
    // don't need to update dialog pos
    return FullMessageId();
  }

  send_update_chat_has_scheduled_messages(d);

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_get_message");
  }

  // set dialog reply markup only after updateNewMessage and updateChatLastMessage are sent
  if (need_update && m->reply_markup != nullptr && !m->message_id.is_scheduled() &&
      m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard && m->reply_markup->is_personal &&
      !td_->auth_manager_->is_bot()) {
    set_dialog_reply_markup(d, message_id);
  }

  return FullMessageId(dialog_id, message_id);
}

void MessagesManager::set_dialog_last_message_id(Dialog *d, MessageId last_message_id, const char *source) {
  CHECK(!last_message_id.is_scheduled());

  LOG(INFO) << "Set " << d->dialog_id << " last message to " << last_message_id << " from " << source;
  d->last_message_id = last_message_id;

  if (!last_message_id.is_valid()) {
    d->suffix_load_first_message_id_ = MessageId();
    d->suffix_load_done_ = false;
  }
  if (last_message_id.is_valid() && d->delete_last_message_date != 0) {
    d->delete_last_message_date = 0;
    d->deleted_last_message_id = MessageId();
    d->is_last_message_deleted_locally = false;
    on_dialog_updated(d->dialog_id, "update_delete_last_message_date");
  }
  if (d->pending_last_message_date != 0) {
    d->pending_last_message_date = 0;
    d->pending_last_message_id = MessageId();
  }
}

void MessagesManager::set_dialog_first_database_message_id(Dialog *d, MessageId first_database_message_id,
                                                           const char *source) {
  CHECK(!first_database_message_id.is_scheduled());

  LOG(INFO) << "Set " << d->dialog_id << " first database message to " << first_database_message_id << " from "
            << source;
  d->first_database_message_id = first_database_message_id;
  on_dialog_updated(d->dialog_id, "set_dialog_first_database_message_id");
}

void MessagesManager::set_dialog_last_database_message_id(Dialog *d, MessageId last_database_message_id,
                                                          const char *source, bool is_loaded_from_database) {
  CHECK(!last_database_message_id.is_scheduled());

  LOG(INFO) << "Set " << d->dialog_id << " last database message to " << last_database_message_id << " from " << source;
  d->debug_set_dialog_last_database_message_id = source;
  d->last_database_message_id = last_database_message_id;
  if (!is_loaded_from_database) {
    on_dialog_updated(d->dialog_id, "set_dialog_last_database_message_id");
  }
}

void MessagesManager::set_dialog_last_new_message_id(Dialog *d, MessageId last_new_message_id, const char *source) {
  CHECK(!last_new_message_id.is_scheduled());

  LOG_CHECK(last_new_message_id > d->last_new_message_id)
      << last_new_message_id << " " << d->last_new_message_id << " " << source;
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat || last_new_message_id.is_server());
  if (!d->last_new_message_id.is_valid()) {
    delete_all_dialog_messages_from_database(d, MessageId::max(), "set_dialog_last_new_message_id");
    set_dialog_first_database_message_id(d, MessageId(), "set_dialog_last_new_message_id");
    set_dialog_last_database_message_id(d, MessageId(), source);
    if (d->dialog_id.get_type() != DialogType::SecretChat) {
      d->have_full_history = false;
    }
    for (auto &first_message_id : d->first_database_message_id_by_index) {
      first_message_id = last_new_message_id;
    }

    MessagesConstIterator it(d, MessageId::max());
    vector<MessageId> to_delete_message_ids;
    while (*it != nullptr) {
      auto message_id = (*it)->message_id;
      if (message_id <= last_new_message_id) {
        break;
      }
      if (!message_id.is_yet_unsent()) {
        to_delete_message_ids.push_back(message_id);
      }
      --it;
    }
    if (!to_delete_message_ids.empty()) {
      LOG(ERROR) << "Delete " << format::as_array(to_delete_message_ids) << " because of received last new "
                 << last_new_message_id << " in " << d->dialog_id << " from " << source;

      vector<int64> deleted_message_ids;
      bool need_update_dialog_pos = false;
      for (auto message_id : to_delete_message_ids) {
        auto message = delete_message(d, message_id, false, &need_update_dialog_pos, "set_dialog_last_new_message_id");
        if (message != nullptr) {
          deleted_message_ids.push_back(message->message_id.get());
        }
      }
      if (need_update_dialog_pos) {
        send_update_chat_last_message(d, "set_dialog_last_new_message_id");
      }
      send_update_delete_messages(d->dialog_id, std::move(deleted_message_ids), false, false);
    }

    auto last_new_message = get_message(d, last_new_message_id);
    if (last_new_message != nullptr) {
      add_message_to_database(d, last_new_message, "set_dialog_last_new_message_id");
      set_dialog_first_database_message_id(d, last_new_message_id, "set_dialog_last_new_message_id");
      set_dialog_last_database_message_id(d, last_new_message_id, "set_dialog_last_new_message_id");
      try_restore_dialog_reply_markup(d, last_new_message);
    }
  }

  LOG(INFO) << "Set " << d->dialog_id << " last new message to " << last_new_message_id << " from " << source;
  d->last_new_message_id = last_new_message_id;
  on_dialog_updated(d->dialog_id, source);
}

void MessagesManager::set_dialog_last_clear_history_date(Dialog *d, int32 date, MessageId last_clear_history_message_id,
                                                         const char *source, bool is_loaded_from_database) {
  CHECK(!last_clear_history_message_id.is_scheduled());

  LOG(INFO) << "Set " << d->dialog_id << " last clear history date to " << date << " of "
            << last_clear_history_message_id << " from " << source;
  if (d->last_clear_history_message_id.is_valid()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        last_clear_history_message_id_to_dialog_id_.erase(d->last_clear_history_message_id);
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        // nothing to do
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }

  d->last_clear_history_date = date;
  d->last_clear_history_message_id = last_clear_history_message_id;
  if (!is_loaded_from_database) {
    on_dialog_updated(d->dialog_id, "set_dialog_last_clear_history_date");
  }

  if (d->last_clear_history_message_id.is_valid()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        last_clear_history_message_id_to_dialog_id_[d->last_clear_history_message_id] = d->dialog_id;
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        // nothing to do
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }
}

void MessagesManager::set_dialog_is_empty(Dialog *d, const char *source) {
  LOG(INFO) << "Set " << d->dialog_id << " is_empty to true from " << source;
  d->is_empty = true;

  if (d->server_unread_count + d->local_unread_count > 0) {
    MessageId max_message_id =
        d->last_database_message_id.is_valid() ? d->last_database_message_id : d->last_new_message_id;
    if (max_message_id.is_valid()) {
      read_history_inbox(d->dialog_id, max_message_id, -1, "set_dialog_is_empty");
    }
    if (d->server_unread_count != 0 || d->local_unread_count != 0) {
      set_dialog_last_read_inbox_message_id(d, MessageId::min(), 0, 0, true, "set_dialog_is_empty");
    }
  }
  if (d->unread_mention_count > 0) {
    d->unread_mention_count = 0;
    d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] = 0;
    send_update_chat_unread_mention_count(d);
  }
  if (d->reply_markup_message_id != MessageId()) {
    set_dialog_reply_markup(d, MessageId());
  }
  std::fill(d->message_count_by_index.begin(), d->message_count_by_index.end(), 0);
  d->notification_id_to_message_id.clear();

  if (d->delete_last_message_date != 0) {
    if (d->is_last_message_deleted_locally && d->last_clear_history_date == 0) {
      set_dialog_last_clear_history_date(d, d->delete_last_message_date, d->deleted_last_message_id,
                                         "set_dialog_is_empty");
    }
    d->delete_last_message_date = 0;
    d->deleted_last_message_id = MessageId();
    d->is_last_message_deleted_locally = false;

    on_dialog_updated(d->dialog_id, "set_dialog_is_empty");
  }
  if (d->pending_last_message_date != 0) {
    d->pending_last_message_date = 0;
    d->pending_last_message_id = MessageId();
  }
  if (d->last_database_message_id.is_valid()) {
    set_dialog_first_database_message_id(d, MessageId(), "set_dialog_is_empty");
    set_dialog_last_database_message_id(d, MessageId(), "set_dialog_is_empty");
  }

  update_dialog_pos(d, false, source);
}

void MessagesManager::set_dialog_is_pinned(DialogId dialog_id, bool is_pinned) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (!is_pinned && d->pinned_order == DEFAULT_ORDER) {
    return;
  }
  set_dialog_is_pinned(d, is_pinned);
  update_dialog_pos(d, false, "set_dialog_is_pinned");
}

void MessagesManager::set_dialog_is_pinned(Dialog *d, bool is_pinned) {
  CHECK(d != nullptr);
  bool was_pinned = d->pinned_order != DEFAULT_ORDER;
  d->pinned_order = is_pinned ? get_next_pinned_dialog_order() : DEFAULT_ORDER;
  on_dialog_updated(d->dialog_id, "set_dialog_is_pinned");

  if (is_pinned != was_pinned) {
    LOG(INFO) << "Set " << d->dialog_id << " is pinned to " << is_pinned;
    LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in set_dialog_is_pinned";
    update_dialog_pos(d, false, "set_dialog_is_pinned", false);
    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateChatIsPinned>(d->dialog_id.get(), is_pinned, get_dialog_public_order(d)));
  }
  // there is no need to call update_dialog_pos, it will be called by the caller
}

void MessagesManager::set_dialog_reply_markup(Dialog *d, MessageId message_id) {
  CHECK(!message_id.is_scheduled());

  if (d->reply_markup_message_id != message_id) {
    on_dialog_updated(d->dialog_id, "set_dialog_reply_markup");
  }

  d->need_restore_reply_markup = false;

  if (d->reply_markup_message_id.is_valid() || message_id.is_valid()) {
    LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in set_dialog_reply_markup";
    d->reply_markup_message_id = message_id;
    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateChatReplyMarkup>(d->dialog_id.get(), message_id.get()));
  }
}

void MessagesManager::try_restore_dialog_reply_markup(Dialog *d, const Message *m) {
  if (!d->need_restore_reply_markup || td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(!m->message_id.is_scheduled());
  if (m->had_reply_markup) {
    LOG(INFO) << "Restore deleted reply markup in " << d->dialog_id;
    set_dialog_reply_markup(d, MessageId());
  } else if (m->reply_markup != nullptr && m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard &&
             m->reply_markup->is_personal) {
    LOG(INFO) << "Restore reply markup in " << d->dialog_id << " to " << m->message_id;
    set_dialog_reply_markup(d, m->message_id);
  }
}

void MessagesManager::set_dialog_pinned_message_notification(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  auto old_message_id = d->pinned_message_notification_message_id;
  if (old_message_id == message_id) {
    return;
  }
  VLOG(notifications) << "Change pinned message notification in " << d->dialog_id << " from " << old_message_id
                      << " to " << message_id;
  if (old_message_id.is_valid()) {
    auto m = get_message_force(d, old_message_id, "set_dialog_pinned_message_notification");
    if (m != nullptr && m->notification_id.is_valid() && is_message_notification_active(d, m)) {
      // Can't remove pinned_message_notification_message_id  before the call,
      // because the notification needs to be still active inside remove_message_notification_id
      remove_message_notification_id(d, m, true, false, true);
      on_message_changed(d, m, false, "set_dialog_pinned_message_notification");
    } else {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_temporary_notification_by_message_id,
                         d->mention_notification_group.group_id, old_message_id, false,
                         "set_dialog_pinned_message_notification 2");
    }
  }
  d->pinned_message_notification_message_id = message_id;
  on_dialog_updated(d->dialog_id, "set_dialog_pinned_message_notification");
}

void MessagesManager::remove_dialog_pinned_message_notification(Dialog *d) {
  set_dialog_pinned_message_notification(d, MessageId());
}

void MessagesManager::remove_dialog_mention_notifications(Dialog *d) {
  auto notification_group_id = d->mention_notification_group.group_id;
  if (!notification_group_id.is_valid()) {
    return;
  }
  if (d->unread_mention_count == 0) {
    return;
  }
  CHECK(!d->being_added_message_id.is_valid());

  VLOG(notifications) << "Remove mention notifications in " << d->dialog_id;

  vector<MessageId> message_ids;
  std::unordered_set<NotificationId, NotificationIdHash> removed_notification_ids_set;
  find_unread_mentions(d->messages.get(), message_ids);
  VLOG(notifications) << "Found unread mentions in " << message_ids;
  for (auto &message_id : message_ids) {
    auto m = get_message(d, message_id);
    CHECK(m != nullptr);
    CHECK(m->message_id.is_valid());
    if (m->notification_id.is_valid() && is_message_notification_active(d, m) &&
        is_from_mention_notification_group(d, m)) {
      removed_notification_ids_set.insert(m->notification_id);
    }
  }

  message_ids = td_->notification_manager_->get_notification_group_message_ids(notification_group_id);
  VLOG(notifications) << "Found active mention notifications in " << message_ids;
  for (auto &message_id : message_ids) {
    CHECK(!message_id.is_scheduled());
    if (message_id != d->pinned_message_notification_message_id) {
      auto m = get_message_force(d, message_id, "remove_dialog_mention_notifications");
      if (m != nullptr && m->notification_id.is_valid() && is_message_notification_active(d, m)) {
        CHECK(is_from_mention_notification_group(d, m));
        removed_notification_ids_set.insert(m->notification_id);
      }
    }
  }

  vector<NotificationId> removed_notification_ids(removed_notification_ids_set.begin(),
                                                  removed_notification_ids_set.end());
  for (size_t i = 0; i < removed_notification_ids.size(); i++) {
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification, notification_group_id,
                       removed_notification_ids[i], false, i + 1 == removed_notification_ids.size(), Promise<Unit>(),
                       "remove_dialog_mention_notifications");
  }
}

bool MessagesManager::set_dialog_last_notification(DialogId dialog_id, NotificationGroupInfo &group_info,
                                                   int32 last_notification_date, NotificationId last_notification_id,
                                                   const char *source) {
  if (group_info.last_notification_date != last_notification_date ||
      group_info.last_notification_id != last_notification_id) {
    VLOG(notifications) << "Set " << group_info.group_id << '/' << dialog_id << " last notification to "
                        << last_notification_id << " sent at " << last_notification_date << " from " << source;
    group_info.last_notification_date = last_notification_date;
    group_info.last_notification_id = last_notification_id;
    group_info.is_changed = true;
    on_dialog_updated(dialog_id, "set_dialog_last_notification");
    return true;
  }
  return false;
}

void MessagesManager::on_update_sent_text_message(int64 random_id,
                                                  tl_object_ptr<telegram_api::MessageMedia> message_media,
                                                  vector<tl_object_ptr<telegram_api::MessageEntity>> &&entities) {
  int32 message_media_id = message_media == nullptr ? telegram_api::messageMediaEmpty::ID : message_media->get_id();
  LOG_IF(ERROR, message_media_id != telegram_api::messageMediaWebPage::ID &&
                    message_media_id != telegram_api::messageMediaEmpty::ID)
      << "Receive non web-page media for text message: " << oneline(to_string(message_media));

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // result of sending message has already been received through getDifference
    return;
  }

  auto full_message_id = it->second;
  auto dialog_id = full_message_id.get_dialog_id();
  auto m = get_message_force(full_message_id, "on_update_sent_text_message");
  if (m == nullptr) {
    // message has already been deleted
    return;
  }
  full_message_id = FullMessageId(dialog_id, m->message_id);

  if (m->content->get_type() != MessageContentType::Text) {
    LOG(ERROR) << "Text message content has been already changed to " << m->content->get_type();
    return;
  }

  const FormattedText *old_message_text = get_message_content_text(m->content.get());
  CHECK(old_message_text != nullptr);
  FormattedText new_message_text = get_message_text(
      td_->contacts_manager_.get(), old_message_text->text, std::move(entities), true,
      m->forward_info ? m->forward_info->date : m->date, m->media_album_id != 0, "on_update_sent_text_message");
  auto new_content = get_message_content(td_, std::move(new_message_text), std::move(message_media), dialog_id,
                                         true /*likely ignored*/, UserId() /*likely ignored*/, nullptr /*ignored*/);
  if (new_content->get_type() != MessageContentType::Text) {
    LOG(ERROR) << "Text message content has changed to " << new_content->get_type();
    return;
  }

  bool need_update = false;
  bool is_content_changed = false;
  merge_message_contents(td_, m->content.get(), new_content.get(), need_message_changed_warning(m), dialog_id, false,
                         is_content_changed, need_update);

  if (is_content_changed || need_update) {
    reregister_message_content(td_, m->content.get(), new_content.get(), full_message_id);
    m->content = std::move(new_content);
    m->is_content_secret = is_secret_message_content(m->ttl, MessageContentType::Text);
  }
  if (need_update) {
    send_update_message_content(dialog_id, m->message_id, m->content.get(), m->date, m->is_content_secret,
                                "on_update_sent_text_message");
  }
}

void MessagesManager::delete_pending_message_web_page(FullMessageId full_message_id) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  Message *m = get_message(d, full_message_id.get_message_id());
  CHECK(m != nullptr);

  MessageContent *content = m->content.get();
  unregister_message_content(td_, content, full_message_id);
  remove_message_content_web_page(content);
  register_message_content(td_, content, full_message_id);

  // don't need to send an updateMessageContent, because the web page was pending

  on_message_changed(d, m, false, "delete_pending_message_web_page");
}

void MessagesManager::on_get_dialogs(FolderId folder_id, vector<tl_object_ptr<telegram_api::Dialog>> &&dialog_folders,
                                     int32 total_count, vector<tl_object_ptr<telegram_api::Message>> &&messages,
                                     Promise<Unit> &&promise) {
  if (td_->updates_manager_->running_get_difference()) {
    LOG(INFO) << "Postpone result of getDialogs";
    pending_on_get_dialogs_.push_back(PendingOnGetDialogs{folder_id, std::move(dialog_folders), total_count,
                                                          std::move(messages), std::move(promise)});
    return;
  }
  bool from_dialog_list = total_count >= 0;
  bool from_get_dialog = total_count == -1;
  bool from_pinned_dialog_list = total_count == -2;

  if (from_get_dialog && dialog_folders.size() == 1 && dialog_folders[0]->get_id() == telegram_api::dialog::ID) {
    DialogId dialog_id(static_cast<const telegram_api::dialog *>(dialog_folders[0].get())->peer_);
    if (running_get_channel_difference(dialog_id)) {
      LOG(INFO) << "Postpone result of channels getDialogs for " << dialog_id;
      pending_channel_on_get_dialogs_.emplace(
          dialog_id, PendingOnGetDialogs{folder_id, std::move(dialog_folders), total_count, std::move(messages),
                                         std::move(promise)});
      return;
    }
  }

  vector<tl_object_ptr<telegram_api::dialog>> dialogs;
  for (auto &dialog_folder : dialog_folders) {
    switch (dialog_folder->get_id()) {
      case telegram_api::dialog::ID:
        dialogs.push_back(telegram_api::move_object_as<telegram_api::dialog>(dialog_folder));
        break;
      case telegram_api::dialogFolder::ID: {
        auto folder = telegram_api::move_object_as<telegram_api::dialogFolder>(dialog_folder);
        if (from_pinned_dialog_list) {
          // TODO update unread_muted_peers_count:int unread_unmuted_peers_count:int
          // unread_muted_messages_count:int unread_unmuted_messages_count:int
          FolderId folder_folder_id(folder->folder_->id_);
          if (folder_folder_id == FolderId::archive()) {
            // archive is expected in pinned dialogs list
            break;
          }
        }
        LOG(ERROR) << "Receive unexpected " << to_string(folder);
        break;
      }
      default:
        UNREACHABLE();
    }
  }

  LOG(INFO) << "Receive " << dialogs.size() << " dialogs out of " << total_count << " in result of GetDialogsQuery";
  std::unordered_map<FullMessageId, DialogDate, FullMessageIdHash> full_message_id_to_dialog_date;
  std::unordered_map<FullMessageId, tl_object_ptr<telegram_api::Message>, FullMessageIdHash> full_message_id_to_message;
  for (auto &message : messages) {
    auto full_message_id = get_full_message_id(message, false);
    if (from_dialog_list) {
      auto message_date = get_message_date(message);
      int64 order = get_dialog_order(full_message_id.get_message_id(), message_date);
      full_message_id_to_dialog_date.emplace(full_message_id, DialogDate(order, full_message_id.get_dialog_id()));
    }
    full_message_id_to_message[full_message_id] = std::move(message);
  }

  DialogDate max_dialog_date = MIN_DIALOG_DATE;
  for (auto &dialog : dialogs) {
    //    LOG(INFO) << to_string(dialog);
    DialogId dialog_id(dialog->peer_);
    bool has_pts = (dialog->flags_ & DIALOG_FLAG_HAS_PTS) != 0;

    if (!dialog_id.is_valid()) {
      LOG(ERROR) << "Receive wrong " << dialog_id;
      return promise.set_error(Status::Error(500, "Wrong query result returned: receive wrong chat identifier"));
    }
    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        if (has_pts) {
          LOG(ERROR) << "Receive user or group " << dialog_id << " with pts";
          return promise.set_error(
              Status::Error(500, "Wrong query result returned: receive user or basic group chat with pts"));
        }
        break;
      case DialogType::Channel:
        if (!has_pts) {
          LOG(ERROR) << "Receive channel " << dialog_id << " without pts";
          return promise.set_error(
              Status::Error(500, "Wrong query result returned: receive supergroup chat without pts"));
        }
        break;
      case DialogType::SecretChat:
      case DialogType::None:
      default:
        UNREACHABLE();
        return promise.set_error(Status::Error(500, "UNREACHABLE"));
    }

    if (from_dialog_list) {
      MessageId last_message_id(ServerMessageId(dialog->top_message_));
      if (last_message_id.is_valid()) {
        FullMessageId full_message_id(dialog_id, last_message_id);
        auto it = full_message_id_to_dialog_date.find(full_message_id);
        if (it == full_message_id_to_dialog_date.end()) {
          LOG(ERROR) << "Last " << last_message_id << " in " << dialog_id << " not found";
          return promise.set_error(Status::Error(500, "Wrong query result returned: last message not found"));
        }
        FolderId dialog_folder_id((dialog->flags_ & DIALOG_FLAG_HAS_FOLDER_ID) != 0 ? dialog->folder_id_ : 0);
        if (dialog_folder_id != folder_id) {
          LOG(ERROR) << "Receive " << dialog_id << " in " << dialog_folder_id << " instead of " << folder_id;
          continue;
        }

        DialogDate dialog_date = it->second;
        CHECK(dialog_date.get_dialog_id() == dialog_id);

        if (max_dialog_date < dialog_date) {
          max_dialog_date = dialog_date;
        }
      } else {
        LOG(ERROR) << "Receive " << last_message_id << " as last chat message";
        continue;
      }
    }
  }

  auto &list = get_dialog_list(folder_id);
  bool need_recalc_unread_count =
      list.last_server_dialog_date_ == MIN_DIALOG_DATE && (from_dialog_list || from_pinned_dialog_list);
  if (from_dialog_list) {
    if (dialogs.empty()) {
      // if there are no more dialogs on the server
      max_dialog_date = MAX_DIALOG_DATE;
    }
    if (list.last_server_dialog_date_ < max_dialog_date) {
      list.last_server_dialog_date_ = max_dialog_date;
      update_last_dialog_date(folder_id);
    } else if (promise) {
      LOG(ERROR) << "Last server dialog date didn't increased from " << list.last_server_dialog_date_ << " to "
                 << max_dialog_date << " after receiving " << dialogs.size() << " chats from " << total_count << " in "
                 << folder_id << ". Know about order of " << list.ordered_dialogs_.size()
                 << " chats, last_dialog_date = " << list.last_dialog_date_
                 << ", last_loaded_database_dialog_date = " << list.last_loaded_database_dialog_date_;
    }
    if (total_count < narrow_cast<int32>(dialogs.size())) {
      LOG(ERROR) << "Receive chat total_count = " << total_count << ", but " << dialogs.size() << " chats";
      total_count = narrow_cast<int32>(dialogs.size());
    }
  }
  if (from_pinned_dialog_list) {
    max_dialog_date = DialogDate(get_dialog_order(MessageId(), MIN_PINNED_DIALOG_DATE - 1), DialogId());
    if (list.last_server_dialog_date_ < max_dialog_date) {
      list.last_server_dialog_date_ = max_dialog_date;
      update_last_dialog_date(folder_id);
    }
  }

  vector<DialogId> added_dialog_ids;
  for (auto &dialog : dialogs) {
    MessageId last_message_id(ServerMessageId(dialog->top_message_));
    if (!last_message_id.is_valid() && from_dialog_list) {
      // skip dialogs without messages
      total_count--;
      continue;
    }

    DialogId dialog_id(dialog->peer_);
    if (td::contains(added_dialog_ids, dialog_id)) {
      LOG(ERROR) << "Receive " << dialog_id << " twice in result of getChats with total_count = " << total_count;
      continue;
    }
    added_dialog_ids.push_back(dialog_id);
    Dialog *d = get_dialog_force(dialog_id);
    bool need_update_dialog_pos = false;
    being_added_dialog_id_ = dialog_id;
    if (d == nullptr) {
      d = add_dialog(dialog_id);
      need_update_dialog_pos = true;
    } else {
      LOG(INFO) << "Receive already created " << dialog_id;
      CHECK(d->dialog_id == dialog_id);
    }
    bool is_new = d->last_new_message_id == MessageId();

    set_dialog_folder_id(d, FolderId((dialog->flags_ & DIALOG_FLAG_HAS_FOLDER_ID) != 0 ? dialog->folder_id_ : 0));

    on_update_dialog_notify_settings(dialog_id, std::move(dialog->notify_settings_), "on_get_dialogs");
    if (!d->notification_settings.is_synchronized) {
      LOG(ERROR) << "Failed to synchronize settings in " << dialog_id;
      d->notification_settings.is_synchronized = true;
      on_dialog_updated(dialog_id, "set notification_settings.is_synchronized");
    }

    if (dialog->unread_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_count_ << " as number of unread messages in " << dialog_id;
      dialog->unread_count_ = 0;
    }
    MessageId read_inbox_max_message_id = MessageId(ServerMessageId(dialog->read_inbox_max_id_));
    if (!read_inbox_max_message_id.is_valid() && read_inbox_max_message_id != MessageId()) {
      LOG(ERROR) << "Receive " << read_inbox_max_message_id << " as last read inbox message in " << dialog_id;
      read_inbox_max_message_id = MessageId();
    }
    MessageId read_outbox_max_message_id = MessageId(ServerMessageId(dialog->read_outbox_max_id_));
    if (!read_outbox_max_message_id.is_valid() && read_outbox_max_message_id != MessageId()) {
      LOG(ERROR) << "Receive " << read_outbox_max_message_id << " as last read outbox message in " << dialog_id;
      read_outbox_max_message_id = MessageId();
    }
    if (dialog->unread_mentions_count_ < 0) {
      LOG(ERROR) << "Receive " << dialog->unread_mentions_count_ << " as number of unread mention messages in "
                 << dialog_id;
      dialog->unread_mentions_count_ = 0;
    }
    if (!d->is_pinned_message_id_inited && !td_->auth_manager_->is_bot()) {
      // asynchronously get dialog pinned message from the server
      // TODO add pinned_message_id to telegram_api::dialog
      get_dialog_pinned_message(dialog_id, Auto());
    }

    need_update_dialog_pos |= update_dialog_draft_message(
        d, get_draft_message(td_->contacts_manager_.get(), std::move(dialog->draft_)), true, false);
    if (is_new) {
      bool has_pts = (dialog->flags_ & DIALOG_FLAG_HAS_PTS) != 0;
      if (last_message_id.is_valid()) {
        FullMessageId full_message_id(dialog_id, last_message_id);
        auto last_message = std::move(full_message_id_to_message[full_message_id]);
        if (last_message == nullptr) {
          LOG(ERROR) << "Last " << full_message_id << " not found";
        } else {
          auto added_full_message_id =
              on_get_message(std::move(last_message), false, has_pts, false, false, false, "get chats");
          CHECK(d->last_new_message_id == MessageId());
          set_dialog_last_new_message_id(d, last_message_id, "on_get_dialogs");
          if (d->last_new_message_id > d->last_message_id && added_full_message_id.get_message_id().is_valid()) {
            CHECK(added_full_message_id.get_message_id() == d->last_new_message_id);
            set_dialog_last_message_id(d, d->last_new_message_id, "on_get_dialogs");
            send_update_chat_last_message(d, "on_get_dialogs");
          }
        }
      }

      if (has_pts && !running_get_channel_difference(dialog_id)) {
        int32 channel_pts = dialog->pts_;
        LOG_IF(ERROR, channel_pts < d->pts) << "In new " << dialog_id << " pts = " << d->pts
                                            << ", but pts = " << channel_pts << " received in GetChats";
        set_channel_pts(d, channel_pts, "get channel");
      }
    }
    bool is_pinned = (dialog->flags_ & DIALOG_FLAG_IS_PINNED) != 0;
    bool was_pinned = d->pinned_order != DEFAULT_ORDER;
    if (is_pinned != was_pinned) {
      set_dialog_is_pinned(d, is_pinned);
      need_update_dialog_pos = false;
    }
    bool is_marked_as_unread = (dialog->flags_ & telegram_api::dialog::UNREAD_MARK_MASK) != 0;
    if (is_marked_as_unread != d->is_marked_as_unread) {
      set_dialog_is_marked_as_unread(d, is_marked_as_unread);
    }

    if (need_update_dialog_pos) {
      update_dialog_pos(d, false, "on_get_dialogs");
    }

    if (!G()->parameters().use_message_db || is_new || !d->is_last_read_inbox_message_id_inited ||
        d->need_repair_server_unread_count) {
      if (d->need_repair_server_unread_count) {
        LOG(INFO) << "Repaired server unread count in " << dialog_id << " from " << d->last_read_inbox_message_id << "/"
                  << d->server_unread_count << " to " << read_inbox_max_message_id << "/" << dialog->unread_count_;
        d->need_repair_server_unread_count = false;
        on_dialog_updated(dialog_id, "repair dialog server unread count");
      }
      if (d->server_unread_count != dialog->unread_count_ ||
          d->last_read_inbox_message_id < read_inbox_max_message_id) {
        set_dialog_last_read_inbox_message_id(d, read_inbox_max_message_id, dialog->unread_count_,
                                              d->local_unread_count, true, "on_get_dialogs");
      }
      if (!d->is_last_read_inbox_message_id_inited) {
        d->is_last_read_inbox_message_id_inited = true;
        on_dialog_updated(dialog_id, "set is_last_read_inbox_message_id_inited");
      }
    }

    if (!G()->parameters().use_message_db || is_new || !d->is_last_read_outbox_message_id_inited) {
      if (d->last_read_outbox_message_id < read_outbox_max_message_id) {
        set_dialog_last_read_outbox_message_id(d, read_outbox_max_message_id);
      }
      if (!d->is_last_read_outbox_message_id_inited) {
        d->is_last_read_outbox_message_id_inited = true;
        on_dialog_updated(dialog_id, "set is_last_read_outbox_message_id_inited");
      }
    }

    if (!G()->parameters().use_message_db || is_new) {
      if (d->unread_mention_count != dialog->unread_mentions_count_) {
        d->unread_mention_count = dialog->unread_mentions_count_;
        update_dialog_mention_notification_count(d);
        d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] =
            d->unread_mention_count;
        send_update_chat_unread_mention_count(d);
      }
    }
    being_added_dialog_id_ = DialogId();
  }
  if (from_dialog_list) {
    CHECK(total_count >= 0);
    if (list.server_dialog_total_count_ != total_count) {
      auto old_dialog_total_count = get_dialog_total_count(list);
      list.server_dialog_total_count_ = total_count;
      if (list.is_dialog_unread_count_inited_ && old_dialog_total_count != get_dialog_total_count(list)) {
        send_update_unread_chat_count(folder_id, DialogId(), true, "on_get_dialogs");
      }
    }
  }
  if (from_pinned_dialog_list) {
    auto pinned_dialog_ids = remove_secret_chat_dialog_ids(get_pinned_dialogs(folder_id));
    std::reverse(pinned_dialog_ids.begin(), pinned_dialog_ids.end());
    if (pinned_dialog_ids != added_dialog_ids) {
      LOG(INFO) << "Repair pinned dialogs order from " << format::as_array(pinned_dialog_ids) << " to "
                << format::as_array(added_dialog_ids);

      std::unordered_set<DialogId, DialogIdHash> old_pinned_dialog_ids(pinned_dialog_ids.begin(),
                                                                       pinned_dialog_ids.end());

      auto old_it = pinned_dialog_ids.begin();
      for (auto dialog_id : added_dialog_ids) {
        old_pinned_dialog_ids.erase(dialog_id);
        while (old_it < pinned_dialog_ids.end()) {
          if (*old_it == dialog_id) {
            break;
          }
          old_it++;
        }
        if (old_it < pinned_dialog_ids.end()) {
          // leave dialog where it is
          continue;
        }
        set_dialog_is_pinned(dialog_id, true);
      }
      for (auto dialog_id : old_pinned_dialog_ids) {
        set_dialog_is_pinned(dialog_id, false);
      }
    }
  }
  promise.set_value(Unit());

  if (need_recalc_unread_count) {
    recalc_unread_count(folder_id);
  }
}

void MessagesManager::dump_debug_message_op(const Dialog *d, int priority) {
  if (!is_debug_message_op_enabled()) {
    return;
  }
  if (d == nullptr) {
    LOG(ERROR) << "Chat not found";
    return;
  }
  static int last_dumped_priority = -1;
  if (priority <= last_dumped_priority) {
    LOG(ERROR) << "Skip dump " << d->dialog_id;
    return;
  }
  last_dumped_priority = priority;

  for (auto &op : d->debug_message_op) {
    switch (op.type) {
      case Dialog::MessageOp::Add: {
        LOG(ERROR) << "MessageOpAdd at " << op.date << " " << op.message_id << " " << op.content_type << " "
                   << op.from_update << " " << op.have_previous << " " << op.have_next << " " << op.source;
        break;
      }
      case Dialog::MessageOp::SetPts: {
        LOG(ERROR) << "MessageOpSetPts at " << op.date << " " << op.pts << " " << op.source;
        break;
      }
      case Dialog::MessageOp::Delete: {
        LOG(ERROR) << "MessageOpDelete at " << op.date << " " << op.message_id << " " << op.content_type << " "
                   << op.from_update << " " << op.have_previous << " " << op.have_next << " " << op.source;
        break;
      }
      case Dialog::MessageOp::DeleteAll: {
        LOG(ERROR) << "MessageOpDeleteAll at " << op.date << " " << op.from_update;
        break;
      }
      default:
        UNREACHABLE();
    }
  }
}

bool MessagesManager::is_message_unload_enabled() const {
  return G()->parameters().use_message_db || td_->auth_manager_->is_bot();
}

bool MessagesManager::can_unload_message(const Dialog *d, const Message *m) const {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  // don't want to unload messages from opened dialogs
  // don't want to unload messages to which there are replies in yet unsent messages
  // don't want to unload message with active reply markup
  // don't want to unload pinned message
  // don't want to unload last edited message, because server can send updateEditChannelMessage again
  // can't unload from memory last dialog, last database messages, yet unsent messages, being edited media messages and active live locations
  // can't unload messages in dialog with active suffix load query
  FullMessageId full_message_id{d->dialog_id, m->message_id};
  return !d->is_opened && m->message_id != d->last_message_id && m->message_id != d->last_database_message_id &&
         !m->message_id.is_yet_unsent() && active_live_location_full_message_ids_.count(full_message_id) == 0 &&
         replied_by_yet_unsent_messages_.count(full_message_id) == 0 && m->edited_content == nullptr &&
         d->suffix_load_queries_.empty() && m->message_id != d->reply_markup_message_id &&
         m->message_id != d->pinned_message_id && m->message_id != d->last_edited_message_id;
}

void MessagesManager::unload_message(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(message_id.is_valid());
  bool need_update_dialog_pos = false;
  auto m = do_delete_message(d, message_id, false, true, &need_update_dialog_pos, "unload_message");
  CHECK(!need_update_dialog_pos);
}

unique_ptr<MessagesManager::Message> MessagesManager::delete_message(Dialog *d, MessageId message_id,
                                                                     bool is_permanently_deleted,
                                                                     bool *need_update_dialog_pos, const char *source) {
  return do_delete_message(d, message_id, is_permanently_deleted, false, need_update_dialog_pos, source);
}

void MessagesManager::add_random_id_to_message_id_correspondence(Dialog *d, int64 random_id, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat);
  CHECK(message_id.is_valid());
  auto it = d->random_id_to_message_id.find(random_id);
  if (it == d->random_id_to_message_id.end() || it->second < message_id) {
    LOG(INFO) << "Add correspondence from random_id " << random_id << " to " << message_id << " in " << d->dialog_id;
    d->random_id_to_message_id[random_id] = message_id;
  }
}

void MessagesManager::delete_random_id_to_message_id_correspondence(Dialog *d, int64 random_id, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(d->dialog_id.get_type() == DialogType::SecretChat);
  CHECK(message_id.is_valid());
  auto it = d->random_id_to_message_id.find(random_id);
  if (it != d->random_id_to_message_id.end() && it->second == message_id) {
    LOG(INFO) << "Delete correspondence from random_id " << random_id << " to " << message_id << " in " << d->dialog_id;
    d->random_id_to_message_id.erase(it);
  }
}

void MessagesManager::add_notification_id_to_message_id_correspondence(Dialog *d, NotificationId notification_id,
                                                                       MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(notification_id.is_valid());
  CHECK(message_id.is_valid());
  auto it = d->notification_id_to_message_id.find(notification_id);
  if (it == d->notification_id_to_message_id.end()) {
    VLOG(notifications) << "Add correspondence from " << notification_id << " to " << message_id << " in "
                        << d->dialog_id;
    d->notification_id_to_message_id.emplace(notification_id, message_id);
  } else if (it->second != message_id) {
    LOG(ERROR) << "Have duplicated " << notification_id << " in " << d->dialog_id << " in " << message_id << " and "
               << it->second;
    if (it->second < message_id) {
      it->second = message_id;
    }
  }
}

void MessagesManager::delete_notification_id_to_message_id_correspondence(Dialog *d, NotificationId notification_id,
                                                                          MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(notification_id.is_valid());
  CHECK(message_id.is_valid());
  auto it = d->notification_id_to_message_id.find(notification_id);
  if (it != d->notification_id_to_message_id.end() && it->second == message_id) {
    VLOG(notifications) << "Delete correspondence from " << notification_id << " to " << message_id << " in "
                        << d->dialog_id;
    d->notification_id_to_message_id.erase(it);
  } else {
    LOG(ERROR) << "Can't find " << notification_id << " in " << d->dialog_id << " with " << message_id;
  }
}

void MessagesManager::remove_message_notification_id(Dialog *d, Message *m, bool is_permanent, bool force_update,
                                                     bool ignore_pinned_message_notification_removal) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  if (!m->notification_id.is_valid()) {
    return;
  }

  auto from_mentions = is_from_mention_notification_group(d, m);
  auto &group_info = get_notification_group_info(d, m);
  if (!group_info.group_id.is_valid()) {
    return;
  }

  bool had_active_notification = is_message_notification_active(d, m);

  auto notification_id = m->notification_id;
  VLOG(notifications) << "Remove " << notification_id << " from " << m->message_id << " in " << group_info.group_id
                      << '/' << d->dialog_id << " from database, was_active = " << had_active_notification
                      << ", is_permanent = " << is_permanent;
  delete_notification_id_to_message_id_correspondence(d, notification_id, m->message_id);
  m->removed_notification_id = m->notification_id;
  m->notification_id = NotificationId();
  if (d->pinned_message_notification_message_id == m->message_id && is_permanent &&
      !ignore_pinned_message_notification_removal) {
    remove_dialog_pinned_message_notification(d);  // must be called after notification_id is removed
  }
  if (group_info.last_notification_id == notification_id) {
    // last notification is deleted, need to find new last notification
    fix_dialog_last_notification_id(d, from_mentions, m->message_id);
  }

  if (is_permanent) {
    if (had_active_notification) {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification, group_info.group_id,
                         notification_id, is_permanent, force_update, Promise<Unit>(),
                         "remove_message_notification_id");
    }

    // on_message_changed will be called by the caller
    // don't need to call there to not save twice/or to save just deleted message
  } else {
    on_message_changed(d, m, false, "remove_message_notification_id");
  }
}

void MessagesManager::remove_new_secret_chat_notification(Dialog *d, bool is_permanent) {
  CHECK(d != nullptr);
  auto notification_id = d->new_secret_chat_notification_id;
  CHECK(notification_id.is_valid());
  VLOG(notifications) << "Remove " << notification_id << " about new secret " << d->dialog_id << " from "
                      << d->message_notification_group.group_id;
  d->new_secret_chat_notification_id = NotificationId();
  bool is_fixed = set_dialog_last_notification(d->dialog_id, d->message_notification_group, 0, NotificationId(),
                                               "remove_new_secret_chat_notification");
  CHECK(is_fixed);
  if (is_permanent) {
    CHECK(d->message_notification_group.group_id.is_valid());
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification,
                       d->message_notification_group.group_id, notification_id, true, true, Promise<Unit>(),
                       "remove_new_secret_chat_notification");
  }
}

void MessagesManager::fix_dialog_last_notification_id(Dialog *d, bool from_mentions, MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  MessagesConstIterator it(d, message_id);
  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  VLOG(notifications) << "Trying to fix last notification id in " << group_info.group_id << " from " << d->dialog_id
                      << " from " << message_id << "/" << group_info.last_notification_id;
  if (*it != nullptr && ((*it)->message_id == message_id || (*it)->have_next)) {
    while (*it != nullptr) {
      const Message *m = *it;
      if (is_from_mention_notification_group(d, m) == from_mentions && m->notification_id.is_valid() &&
          is_message_notification_active(d, m) && m->message_id != message_id) {
        bool is_fixed = set_dialog_last_notification(d->dialog_id, group_info, m->date, m->notification_id,
                                                     "fix_dialog_last_notification_id");
        CHECK(is_fixed);
        return;
      }
      --it;
    }
  }
  if (G()->parameters().use_message_db) {
    get_message_notifications_from_database(
        d->dialog_id, group_info.group_id, group_info.last_notification_id, message_id, 1,
        PromiseCreator::lambda(
            [actor_id = actor_id(this), dialog_id = d->dialog_id, from_mentions,
             prev_last_notification_id = group_info.last_notification_id](Result<vector<Notification>> result) {
              send_closure(actor_id, &MessagesManager::do_fix_dialog_last_notification_id, dialog_id, from_mentions,
                           prev_last_notification_id, std::move(result));
            }));
  }
}

void MessagesManager::do_fix_dialog_last_notification_id(DialogId dialog_id, bool from_mentions,
                                                         NotificationId prev_last_notification_id,
                                                         Result<vector<Notification>> result) {
  if (result.is_error()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  VLOG(notifications) << "Receive " << result.ok().size() << " message notifications in " << group_info.group_id << '/'
                      << dialog_id << " from " << prev_last_notification_id;
  if (group_info.last_notification_id != prev_last_notification_id) {
    // last_notification_id was changed
    return;
  }

  auto notifications = result.move_as_ok();
  CHECK(notifications.size() <= 1);

  int32 last_notification_date = 0;
  NotificationId last_notification_id;
  if (!notifications.empty()) {
    last_notification_date = notifications[0].date;
    last_notification_id = notifications[0].notification_id;
  }

  bool is_fixed = set_dialog_last_notification(dialog_id, group_info, last_notification_date, last_notification_id,
                                               "do_fix_dialog_last_notification_id");
  CHECK(is_fixed);
}

// DO NOT FORGET TO ADD ALL CHANGES OF THIS FUNCTION AS WELL TO do_delete_all_dialog_messages
unique_ptr<MessagesManager::Message> MessagesManager::do_delete_message(Dialog *d, MessageId message_id,
                                                                        bool is_permanently_deleted,
                                                                        bool only_from_memory,
                                                                        bool *need_update_dialog_pos,
                                                                        const char *source) {
  CHECK(d != nullptr);
  if (!message_id.is_valid()) {
    if (message_id.is_valid_scheduled()) {
      return do_delete_scheduled_message(d, message_id, is_permanently_deleted, source);
    }

    LOG(ERROR) << "Trying to delete " << message_id << " in " << d->dialog_id << " from " << source;
    return nullptr;
  }

  FullMessageId full_message_id(d->dialog_id, message_id);
  unique_ptr<Message> *v = treap_find_message(&d->messages, message_id);
  if (*v == nullptr) {
    LOG(INFO) << message_id << " is not found in " << d->dialog_id << " to be deleted from " << source;
    if (only_from_memory) {
      return nullptr;
    }

    if (get_message_force(d, message_id, "do_delete_message") == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted);

      if (is_permanently_deleted && d->last_clear_history_message_id == message_id) {
        set_dialog_last_clear_history_date(d, 0, MessageId(), "do_delete_message");
        *need_update_dialog_pos = true;
      }

      /*
      can't do this because the message may be never received in the dialog, unread count will became negative
      // if last_read_inbox_message_id is not known, we can't be sure whether unread_count should be decreased or not
      if (message_id.is_valid() && !message_id.is_yet_unsent() && d->is_last_read_inbox_message_id_inited &&
          message_id > d->last_read_inbox_message_id && !td_->auth_manager_->is_bot()) {
        int32 server_unread_count = d->server_unread_count;
        int32 local_unread_count = d->local_unread_count;
        int32 &unread_count = message_id.is_server() ? server_unread_count : local_unread_count;
        if (unread_count == 0) {
          LOG(ERROR) << "Unread count became negative in " << d->dialog_id << " after deletion of " << message_id
                     << ". Last read is " << d->last_read_inbox_message_id;
          dump_debug_message_op(d, 3);
        } else {
          unread_count--;
          set_dialog_last_read_inbox_message_id(d, MessageId::min(), server_unread_count, local_unread_count, false,
                                                source);
        }
      }
      */
      return nullptr;
    }
    v = treap_find_message(&d->messages, message_id);
    CHECK(*v != nullptr);
  }

  const Message *m = v->get();
  CHECK(m->message_id == message_id);

  if (only_from_memory && !can_unload_message(d, m)) {
    return nullptr;
  }

  LOG_CHECK(!d->being_deleted_message_id.is_valid())
      << d->being_deleted_message_id << " " << message_id << " " << source;
  d->being_deleted_message_id = message_id;

  if (is_debug_message_op_enabled()) {
    d->debug_message_op.emplace_back(Dialog::MessageOp::Delete, m->message_id, m->content->get_type(), false,
                                     m->have_previous, m->have_next, source);
  }

  bool need_get_history = false;
  if (!only_from_memory) {
    LOG(INFO) << "Deleting " << full_message_id << " with have_previous = " << m->have_previous
              << " and have_next = " << m->have_next << " from " << source;

    delete_message_from_database(d, message_id, m, is_permanently_deleted);

    delete_active_live_location(d->dialog_id, m);
    remove_message_file_sources(d->dialog_id, m);

    if (message_id == d->last_message_id) {
      MessagesConstIterator it(d, message_id);
      CHECK(*it == m);
      if ((*it)->have_previous) {
        --it;
        if (*it != nullptr) {
          set_dialog_last_message_id(d, (*it)->message_id, "do_delete_message");
        } else {
          LOG(ERROR) << "Have have_previous is true, but there is no previous for " << full_message_id << " from "
                     << source;
          dump_debug_message_op(d);
          set_dialog_last_message_id(d, MessageId(), "do_delete_message");
        }
      } else {
        need_get_history = true;
        set_dialog_last_message_id(d, MessageId(), "do_delete_message");
        d->delete_last_message_date = m->date;
        d->deleted_last_message_id = message_id;
        d->is_last_message_deleted_locally = Slice(source) == Slice(DELETE_MESSAGE_USER_REQUEST_SOURCE);
        on_dialog_updated(d->dialog_id, "do delete last message");
      }
      *need_update_dialog_pos = true;
    }

    if (message_id == d->last_database_message_id) {
      MessagesConstIterator it(d, message_id);
      CHECK(*it == m);
      while ((*it)->have_previous) {
        --it;
        if (*it == nullptr || !(*it)->message_id.is_yet_unsent()) {
          break;
        }
      }

      if (*it != nullptr) {
        if (!(*it)->message_id.is_yet_unsent() && (*it)->message_id != d->last_database_message_id) {
          set_dialog_last_database_message_id(d, (*it)->message_id, "do_delete_message");
          if (d->last_database_message_id < d->first_database_message_id) {
            LOG(ERROR) << "Last database " << d->last_database_message_id << " became less than first database "
                       << d->first_database_message_id << " in " << d->dialog_id;
            set_dialog_first_database_message_id(d, d->last_database_message_id, "do_delete_message 2");
          }
        } else {
          need_get_history = true;
        }
      } else {
        LOG(ERROR) << "Have have_previous is true, but there is no previous";
        dump_debug_message_op(d);
      }
    }
    if (d->last_database_message_id.is_valid()) {
      CHECK(d->first_database_message_id.is_valid());
    } else {
      set_dialog_first_database_message_id(d, MessageId(), "do_delete_message");
    }

    if (message_id == d->suffix_load_first_message_id_) {
      MessagesConstIterator it(d, message_id);
      CHECK(*it == m);
      if ((*it)->have_previous) {
        --it;
        if (*it != nullptr) {
          d->suffix_load_first_message_id_ = (*it)->message_id;
        } else {
          LOG(ERROR) << "Have have_previous is true, but there is no previous for " << full_message_id << " from "
                     << source;
          dump_debug_message_op(d);
          d->suffix_load_first_message_id_ = MessageId();
          d->suffix_load_done_ = false;
        }
      } else {
        d->suffix_load_first_message_id_ = MessageId();
        d->suffix_load_done_ = false;
      }
    }
  }
  if (only_from_memory && message_id >= d->suffix_load_first_message_id_) {
    d->suffix_load_first_message_id_ = MessageId();
    d->suffix_load_done_ = false;
  }

  if (m->have_previous && (only_from_memory || !m->have_next)) {
    MessagesIterator it(d, message_id);
    CHECK(*it == m);
    --it;
    Message *prev_m = *it;
    if (prev_m != nullptr) {
      prev_m->have_next = false;
    } else {
      LOG(ERROR) << "Have have_previous is true, but there is no previous for " << full_message_id << " from "
                 << source;
      dump_debug_message_op(d);
    }
  }
  if ((*v)->have_next && (only_from_memory || !(*v)->have_previous)) {
    MessagesIterator it(d, message_id);
    CHECK(*it == m);
    ++it;
    Message *next_m = *it;
    if (next_m != nullptr) {
      next_m->have_previous = false;
    } else {
      LOG(ERROR) << "Have have_next is true, but there is no next for " << full_message_id << " from " << source;
      dump_debug_message_op(d);
    }
  }

  auto result = treap_delete_message(v);

  d->being_deleted_message_id = MessageId();

  if (!only_from_memory) {
    if (need_get_history && !td_->auth_manager_->is_bot() && have_input_peer(d->dialog_id, AccessRights::Read)) {
      get_history_from_the_end(d->dialog_id, true, false, Auto());
    }

    if (d->reply_markup_message_id == message_id) {
      set_dialog_reply_markup(d, MessageId());
    }
    // if last_read_inbox_message_id is not known, we can't be sure whether unread_count should be decreased or not
    if (has_incoming_notification(d->dialog_id, result.get()) && message_id > d->last_read_inbox_message_id &&
        d->is_last_read_inbox_message_id_inited && !td_->auth_manager_->is_bot()) {
      int32 server_unread_count = d->server_unread_count;
      int32 local_unread_count = d->local_unread_count;
      int32 &unread_count = message_id.is_server() ? server_unread_count : local_unread_count;
      if (unread_count == 0) {
        if (d->order > 0) {
          LOG(ERROR) << "Unread count became negative in " << d->dialog_id << " after deletion of " << message_id
                     << ". Last read is " << d->last_read_inbox_message_id;
          dump_debug_message_op(d, 3);
        }
      } else {
        unread_count--;
        set_dialog_last_read_inbox_message_id(d, MessageId::min(), server_unread_count, local_unread_count, false,
                                              source);
      }
    }
    if (result->contains_unread_mention) {
      if (d->unread_mention_count == 0) {
        LOG_IF(ERROR,
               d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] != -1)
            << "Unread mention count became negative in " << d->dialog_id << " after deletion of " << message_id;
      } else {
        d->unread_mention_count--;
        d->message_count_by_index[search_messages_filter_index(SearchMessagesFilter::UnreadMention)] =
            d->unread_mention_count;
        send_update_chat_unread_mention_count(d);
      }
    }

    update_message_count_by_index(d, -1, result.get());
  }

  on_message_deleted(d, result.get(), is_permanently_deleted, source);

  return result;
}

void MessagesManager::on_message_deleted(Dialog *d, Message *m, bool is_permanently_deleted, const char *source) {
  // also called for unloaded messages

  cancel_send_deleted_message(d->dialog_id, m, is_permanently_deleted);

  CHECK(m->message_id.is_valid());
  switch (d->dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      if (m->message_id.is_server()) {
        message_id_to_dialog_id_.erase(m->message_id);
      }
      break;
    case DialogType::Channel:
      // nothing to do
      break;
    case DialogType::SecretChat:
      delete_random_id_to_message_id_correspondence(d, m->random_id, m->message_id);
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }
  ttl_unregister_message(d->dialog_id, m, Time::now(), source);
  unregister_message_content(td_, m->content.get(), {d->dialog_id, m->message_id});
  if (m->notification_id.is_valid()) {
    delete_notification_id_to_message_id_correspondence(d, m->notification_id, m->message_id);
  }
}

unique_ptr<MessagesManager::Message> MessagesManager::do_delete_scheduled_message(Dialog *d, MessageId message_id,
                                                                                  bool is_permanently_deleted,
                                                                                  const char *source) {
  CHECK(d != nullptr);
  CHECK(message_id.is_valid_scheduled());

  unique_ptr<Message> *v = treap_find_message(&d->scheduled_messages, message_id);
  if (*v == nullptr) {
    LOG(INFO) << message_id << " is not found in " << d->dialog_id << " to be deleted from " << source;
    auto message = get_message_force(d, message_id, "do_delete_scheduled_message");
    if (message == nullptr) {
      // currently there may be a race between add_message_to_database and get_message_force,
      // so delete a message from database just in case
      delete_message_from_database(d, message_id, nullptr, is_permanently_deleted);
      return nullptr;
    }

    message_id = message->message_id;
    v = treap_find_message(&d->scheduled_messages, message_id);
    CHECK(*v != nullptr);
  }

  const Message *m = v->get();
  CHECK(m->message_id == message_id);

  LOG(INFO) << "Deleting " << FullMessageId{d->dialog_id, message_id} << " from " << source;

  delete_message_from_database(d, message_id, m, is_permanently_deleted);

  remove_message_file_sources(d->dialog_id, m);

  auto result = treap_delete_message(v);

  if (message_id.is_scheduled_server()) {
    size_t erased = d->scheduled_message_date.erase(message_id.get_scheduled_server_message_id());
    CHECK(erased != 0);
  }

  cancel_send_deleted_message(d->dialog_id, result.get(), is_permanently_deleted);

  unregister_message_content(td_, result->content.get(), {d->dialog_id, message_id});

  return result;
}

void MessagesManager::do_delete_all_dialog_messages(Dialog *d, unique_ptr<Message> &message,
                                                    bool is_permanently_deleted, vector<int64> &deleted_message_ids) {
  if (message == nullptr) {
    return;
  }
  const Message *m = message.get();
  MessageId message_id = m->message_id;

  if (is_debug_message_op_enabled()) {
    d->debug_message_op.emplace_back(Dialog::MessageOp::Delete, m->message_id, m->content->get_type(), false,
                                     m->have_previous, m->have_next, "delete all messages");
  }

  LOG(INFO) << "Delete " << message_id;
  deleted_message_ids.push_back(message_id.get());

  do_delete_all_dialog_messages(d, message->right, is_permanently_deleted, deleted_message_ids);
  do_delete_all_dialog_messages(d, message->left, is_permanently_deleted, deleted_message_ids);

  delete_active_live_location(d->dialog_id, m);
  remove_message_file_sources(d->dialog_id, m);

  on_message_deleted(d, message.get(), is_permanently_deleted, "do_delete_all_dialog_messages");

  message = nullptr;
}

bool MessagesManager::have_dialog(DialogId dialog_id) const {
  return dialogs_.count(dialog_id) > 0;
}

void MessagesManager::load_dialogs(vector<DialogId> dialog_ids, Promise<Unit> &&promise) {
  LOG(INFO) << "Load dialogs " << format::as_array(dialog_ids);

  Dependencies dependencies;
  for (auto dialog_id : dialog_ids) {
    if (dialog_id.is_valid() && !have_dialog(dialog_id)) {
      add_dialog_dependencies(dependencies, dialog_id);
    }
  }
  resolve_dependencies_force(td_, dependencies);

  for (auto dialog_id : dialog_ids) {
    if (dialog_id.is_valid()) {
      force_create_dialog(dialog_id, "load_dialogs");
    }
  }

  promise.set_value(Unit());
}

bool MessagesManager::load_dialog(DialogId dialog_id, int left_tries, Promise<Unit> &&promise) {
  if (!dialog_id.is_valid()) {
    promise.set_error(Status::Error(6, "Invalid chat identifier"));
    return false;
  }

  if (!have_dialog_force(dialog_id)) {  // TODO remove _force
    if (G()->parameters().use_message_db) {
      //      TODO load dialog from database, DialogLoader
      //      send_closure_later(actor_id(this), &MessagesManager::load_dialog_from_database, dialog_id,
      //      std::move(promise));
      //      return false;
    }
    if (td_->auth_manager_->is_bot()) {
      switch (dialog_id.get_type()) {
        case DialogType::User: {
          auto user_id = dialog_id.get_user_id();
          auto have_user = td_->contacts_manager_->get_user(user_id, left_tries, std::move(promise));
          if (!have_user) {
            return false;
          }
          break;
        }
        case DialogType::Chat: {
          auto have_chat = td_->contacts_manager_->get_chat(dialog_id.get_chat_id(), left_tries, std::move(promise));
          if (!have_chat) {
            return false;
          }
          break;
        }
        case DialogType::Channel: {
          auto have_channel =
              td_->contacts_manager_->get_channel(dialog_id.get_channel_id(), left_tries, std::move(promise));
          if (!have_channel) {
            return false;
          }
          break;
        }
        case DialogType::SecretChat:
          promise.set_error(Status::Error(6, "Chat not found"));
          return false;
        case DialogType::None:
        default:
          UNREACHABLE();
      }
      if (!have_input_peer(dialog_id, AccessRights::Read)) {
        return false;
      }

      add_dialog(dialog_id);
      return true;
    }

    promise.set_error(Status::Error(6, "Chat not found"));
    return false;
  }

  promise.set_value(Unit());
  return true;
}

vector<DialogId> MessagesManager::get_dialogs(FolderId folder_id, DialogDate offset, int32 limit, bool force,
                                              Promise<Unit> &&promise) {
  auto &list = get_dialog_list(folder_id);
  LOG(INFO) << "Get chats in " << folder_id << " with offset " << offset << " and limit " << limit
            << ". Know about order of " << list.ordered_dialogs_.size()
            << " chat(s). last_dialog_date = " << list.last_dialog_date_
            << ", last_server_dialog_date = " << list.last_server_dialog_date_
            << ", last_loaded_database_dialog_date = " << list.last_loaded_database_dialog_date_;

  vector<DialogId> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit in getChats must be positive"));
    return result;
  }

  if (limit > MAX_GET_DIALOGS) {
    limit = MAX_GET_DIALOGS;
  }

  auto it = list.ordered_dialogs_.upper_bound(offset);
  auto end = list.ordered_dialogs_.end();
  while (it != end && limit-- > 0) {
    result.push_back(it->get_dialog_id());
    ++it;
  }

  if (limit <= 0 || force) {
    promise.set_value(Unit());
    return result;
  }

  load_dialog_list(folder_id, limit, false, std::move(promise));
  return result;
}

void MessagesManager::load_dialog_list(FolderId folder_id, int32 limit, bool only_local, Promise<Unit> &&promise) {
  auto &list = get_dialog_list(folder_id);
  if (list.last_dialog_date_ == MAX_DIALOG_DATE) {
    return promise.set_value(Unit());
  }

  bool use_database = G()->parameters().use_message_db &&
                      list.last_loaded_database_dialog_date_ < list.last_database_server_dialog_date_;
  if (only_local && !use_database) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Load dialog list in " << folder_id << " with limit " << limit;
  auto &multipromise = list.load_dialog_list_multipromise_;
  multipromise.add_promise(std::move(promise));
  if (multipromise.promise_count() != 1) {
    // queries have already been sent, just wait for the result
    if (use_database && list.load_dialog_list_limit_max_ != 0) {
      list.load_dialog_list_limit_max_ = max(list.load_dialog_list_limit_max_, limit);
    }
    return;
  }

  bool is_query_sent = false;
  if (use_database) {
    load_dialog_list_from_database(folder_id, limit, multipromise.get_promise());
    is_query_sent = true;
  } else {
    LOG(INFO) << "Get dialogs from " << list.last_server_dialog_date_;
    reload_pinned_dialogs(folder_id, multipromise.get_promise());
    if (list.last_dialog_date_ == list.last_server_dialog_date_) {
      send_closure(td_->create_net_actor<GetDialogListActor>(multipromise.get_promise()), &GetDialogListActor::send,
                   folder_id, list.last_server_dialog_date_.get_date(),
                   list.last_server_dialog_date_.get_message_id().get_next_server_message_id().get_server_message_id(),
                   list.last_server_dialog_date_.get_dialog_id(), int32{MAX_GET_DIALOGS},
                   get_sequence_dispatcher_id(DialogId(), MessageContentType::None));
      is_query_sent = true;
    }
    if (folder_id == FolderId::main() && list.last_server_dialog_date_ == MIN_DIALOG_DATE) {
      // do not pass promise to not wait for drafts before showing chat list
      td_->create_handler<GetAllDraftsQuery>()->send();
    }
  }
  CHECK(is_query_sent);
}

void MessagesManager::load_dialog_list_from_database(FolderId folder_id, int32 limit, Promise<Unit> &&promise) {
  auto &list = get_dialog_list(folder_id);
  LOG(INFO) << "Load " << limit << " chats in " << folder_id << " from database from "
            << list.last_loaded_database_dialog_date_
            << ", last database server dialog date = " << list.last_database_server_dialog_date_;

  CHECK(list.load_dialog_list_limit_max_ == 0);
  list.load_dialog_list_limit_max_ = limit;
  G()->td_db()->get_dialog_db_async()->get_dialogs(
      folder_id, list.last_loaded_database_dialog_date_.get_order(),
      list.last_loaded_database_dialog_date_.get_dialog_id(), limit,
      PromiseCreator::lambda([actor_id = actor_id(this), folder_id, limit,
                              promise = std::move(promise)](DialogDbGetDialogsResult result) mutable {
        send_closure(actor_id, &MessagesManager::on_get_dialogs_from_database, folder_id, limit, std::move(result),
                     std::move(promise));
      }));
}

void MessagesManager::on_get_dialogs_from_database(FolderId folder_id, int32 limit, DialogDbGetDialogsResult &&dialogs,
                                                   Promise<Unit> &&promise) {
  auto &list = get_dialog_list(folder_id);
  LOG(INFO) << "Receive " << dialogs.dialogs.size() << " from expected " << limit << " chats in " << folder_id
            << " in from database with next order " << dialogs.next_order << " and next " << dialogs.next_dialog_id;
  int32 new_get_dialogs_limit = 0;
  int32 have_more_dialogs_in_database = (limit == static_cast<int32>(dialogs.dialogs.size()));
  if (have_more_dialogs_in_database && limit < list.load_dialog_list_limit_max_) {
    new_get_dialogs_limit = list.load_dialog_list_limit_max_ - limit;
  }
  list.load_dialog_list_limit_max_ = 0;

  size_t dialogs_skipped = 0;
  for (auto &dialog : dialogs.dialogs) {
    Dialog *d = on_load_dialog_from_database(DialogId(), std::move(dialog));
    if (d == nullptr) {
      dialogs_skipped++;
      continue;
    }
    if (d->folder_id != folder_id) {
      LOG(WARNING) << "Skip " << d->dialog_id << " received from database, because it is in " << d->folder_id
                   << " instead of " << folder_id;
      dialogs_skipped++;
      continue;
    }

    LOG(INFO) << "Chat " << d->dialog_id << " with order " << d->order << " is loaded from database";
  }

  DialogDate max_dialog_date(dialogs.next_order, dialogs.next_dialog_id);
  if (!have_more_dialogs_in_database) {
    list.last_loaded_database_dialog_date_ = MAX_DIALOG_DATE;
    LOG(INFO) << "Set last loaded database dialog date to " << list.last_loaded_database_dialog_date_;
    list.last_server_dialog_date_ = max(list.last_server_dialog_date_, list.last_database_server_dialog_date_);
    LOG(INFO) << "Set last server dialog date to " << list.last_server_dialog_date_;
    update_last_dialog_date(folder_id);
  } else if (list.last_loaded_database_dialog_date_ < max_dialog_date) {
    list.last_loaded_database_dialog_date_ = min(max_dialog_date, list.last_database_server_dialog_date_);
    LOG(INFO) << "Set last loaded database dialog date to " << list.last_loaded_database_dialog_date_;
    list.last_server_dialog_date_ = max(list.last_server_dialog_date_, list.last_loaded_database_dialog_date_);
    LOG(INFO) << "Set last server dialog date to " << list.last_server_dialog_date_;
    update_last_dialog_date(folder_id);
  } else {
    LOG(ERROR) << "Last loaded database dialog date didn't increased, skipped " << dialogs_skipped << " chats out of "
               << dialogs.dialogs.size();
  }

  if (!(list.last_loaded_database_dialog_date_ < list.last_database_server_dialog_date_)) {
    // have_more_dialogs_in_database = false;
    new_get_dialogs_limit = 0;
  }

  if (new_get_dialogs_limit == 0) {
    preload_dialog_list_timeout_.add_timeout_in(folder_id.get(), 0.2);
    promise.set_value(Unit());
  } else {
    load_dialog_list_from_database(folder_id, new_get_dialogs_limit, std::move(promise));
  }
}

void MessagesManager::preload_dialog_list(FolderId folder_id) {
  if (G()->close_flag()) {
    LOG(INFO) << "Skip chat list preload because of closing";
    return;
  }

  auto &list = get_dialog_list(folder_id);
  CHECK(G()->parameters().use_message_db);
  if (list.load_dialog_list_multipromise_.promise_count() != 0) {
    LOG(INFO) << "Skip chat list preload, because there is a pending load chat list request";
    return;
  }

  if (list.ordered_dialogs_.size() > MAX_PRELOADED_DIALOGS) {
    // do nothing if there are more than MAX_PRELOADED_DIALOGS dialogs already loaded
    recalc_unread_count(folder_id);
    return;
  }

  if (list.last_loaded_database_dialog_date_ < list.last_database_server_dialog_date_) {
    // if there are some dialogs in database, preload some of them
    load_dialog_list(folder_id, 20, true, Auto());
  } else if (list.last_dialog_date_ != MAX_DIALOG_DATE) {
    // otherwise load more dialogs from the server
    load_dialog_list(folder_id, MAX_GET_DIALOGS, false,
                     PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Result<Unit> result) {
                       if (result.is_ok()) {
                         send_closure(actor_id, &MessagesManager::recalc_unread_count, folder_id);
                       }
                     }));
  } else {
    recalc_unread_count(folder_id);
  }
}

vector<DialogId> MessagesManager::get_pinned_dialogs(FolderId folder_id) const {
  vector<DialogId> result;

  auto *list = get_dialog_list(folder_id);
  if (list != nullptr) {
    for (const DialogDate &dialog_date : list->ordered_dialogs_) {
      if (dialog_date.get_date() < MIN_PINNED_DIALOG_DATE) {
        break;
      }
      if (dialog_date.get_order() != SPONSORED_DIALOG_ORDER) {
        result.push_back(dialog_date.get_dialog_id());
      }
    }
  }

  return result;
}

void MessagesManager::reload_pinned_dialogs(FolderId folder_id, Promise<Unit> &&promise) {
  if (G()->close_flag()) {
    return promise.set_error(Status::Error(500, "Request aborted"));
  }
  send_closure(td_->create_net_actor<GetPinnedDialogsActor>(std::move(promise)), &GetPinnedDialogsActor::send,
               folder_id, get_sequence_dispatcher_id(DialogId(), MessageContentType::None));
}

vector<DialogId> MessagesManager::search_public_dialogs(const string &query, Promise<Unit> &&promise) {
  LOG(INFO) << "Search public chats with query = \"" << query << '"';

  if (utf8_length(query) < MIN_SEARCH_PUBLIC_DIALOG_PREFIX_LEN) {
    string username = clean_username(query);
    if (username[0] == '@') {
      username = username.substr(1);
    }

    for (auto &short_username : get_valid_short_usernames()) {
      if (2 * username.size() > short_username.size() && begins_with(short_username, username)) {
        username = short_username.str();
        auto it = resolved_usernames_.find(username);
        if (it == resolved_usernames_.end()) {
          td_->create_handler<ResolveUsernameQuery>(std::move(promise))->send(username);
          return {};
        }

        if (it->second.expires_at < Time::now()) {
          td_->create_handler<ResolveUsernameQuery>(Promise<>())->send(username);
        }

        auto dialog_id = it->second.dialog_id;
        force_create_dialog(dialog_id, "public dialogs search");
        promise.set_value(Unit());
        return {dialog_id};
      }
    }
    promise.set_value(Unit());
    return {};
  }

  auto it = found_public_dialogs_.find(query);
  if (it != found_public_dialogs_.end()) {
    promise.set_value(Unit());
    return it->second;
  }

  send_search_public_dialogs_query(query, std::move(promise));
  return vector<DialogId>();
}

void MessagesManager::send_search_public_dialogs_query(const string &query, Promise<Unit> &&promise) {
  auto &promises = search_public_dialogs_queries_[query];
  promises.push_back(std::move(promise));
  if (promises.size() != 1) {
    // query has already been sent, just wait for the result
    return;
  }

  td_->create_handler<SearchPublicDialogsQuery>()->send(query);
}

std::pair<size_t, vector<DialogId>> MessagesManager::search_dialogs(const string &query, int32 limit,
                                                                    Promise<Unit> &&promise) {
  LOG(INFO) << "Search chats with query \"" << query << "\" and limit " << limit;

  if (limit < 0) {
    promise.set_error(Status::Error(400, "Limit must be non-negative"));
    return {};
  }
  if (query.empty()) {
    if (!load_recently_found_dialogs(promise)) {
      return {};
    }

    promise.set_value(Unit());
    size_t result_size = min(static_cast<size_t>(limit), recently_found_dialog_ids_.size());
    return {recently_found_dialog_ids_.size(),
            vector<DialogId>(recently_found_dialog_ids_.begin(), recently_found_dialog_ids_.begin() + result_size)};
  }

  auto result = dialogs_hints_.search(query, limit);
  vector<DialogId> dialog_ids;
  dialog_ids.reserve(result.second.size());
  for (auto key : result.second) {
    dialog_ids.push_back(DialogId(-key));
  }

  promise.set_value(Unit());
  return {result.first, std::move(dialog_ids)};
}

vector<DialogId> MessagesManager::sort_dialogs_by_order(const vector<DialogId> &dialog_ids, int32 limit) const {
  int64 fake_order = static_cast<int64>(dialog_ids.size()) + 1;
  auto dialog_dates = transform(dialog_ids, [this, &fake_order](DialogId dialog_id) {
    const Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);
    if (is_dialog_inited(d) || d->order != DEFAULT_ORDER) {
      return DialogDate(d->order, dialog_id);
    }
    // if the dialog is not inited yet, we need to assume that server knows better and the dialog needs to be returned
    return DialogDate(fake_order--, dialog_id);
  });
  if (static_cast<size_t>(limit) >= dialog_dates.size()) {
    std::sort(dialog_dates.begin(), dialog_dates.end());
  } else {
    std::partial_sort(dialog_dates.begin(), dialog_dates.begin() + limit, dialog_dates.end());
    dialog_dates.resize(limit, MAX_DIALOG_DATE);
  }
  while (!dialog_dates.empty() && dialog_dates.back().get_order() == DEFAULT_ORDER) {
    dialog_dates.pop_back();
  }
  return transform(dialog_dates, [](auto dialog_date) { return dialog_date.get_dialog_id(); });
}

vector<DialogId> MessagesManager::search_dialogs_on_server(const string &query, int32 limit, Promise<Unit> &&promise) {
  LOG(INFO) << "Search chats on server with query \"" << query << "\" and limit " << limit;

  if (limit < 0) {
    promise.set_error(Status::Error(400, "Limit must be non-negative"));
    return {};
  }
  if (limit > MAX_GET_DIALOGS) {
    limit = MAX_GET_DIALOGS;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return {};
  }

  auto it = found_on_server_dialogs_.find(query);
  if (it != found_on_server_dialogs_.end()) {
    promise.set_value(Unit());
    return sort_dialogs_by_order(it->second, limit);
  }

  send_search_public_dialogs_query(query, std::move(promise));
  return vector<DialogId>();
}

void MessagesManager::drop_common_dialogs_cache(UserId user_id) {
  auto it = found_common_dialogs_.find(user_id);
  if (it != found_common_dialogs_.end()) {
    it->second.is_outdated = true;
  }
}

vector<DialogId> MessagesManager::get_common_dialogs(UserId user_id, DialogId offset_dialog_id, int32 limit, bool force,
                                                     Promise<Unit> &&promise) {
  if (!td_->contacts_manager_->have_input_user(user_id)) {
    promise.set_error(Status::Error(6, "Have no access to the user"));
    return vector<DialogId>();
  }

  if (user_id == td_->contacts_manager_->get_my_id()) {
    promise.set_error(Status::Error(6, "Can't get common chats with self"));
    return vector<DialogId>();
  }
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit must be positive"));
    return vector<DialogId>();
  }
  if (limit > MAX_GET_DIALOGS) {
    limit = MAX_GET_DIALOGS;
  }

  int32 offset_chat_id = 0;
  switch (offset_dialog_id.get_type()) {
    case DialogType::Chat:
      offset_chat_id = offset_dialog_id.get_chat_id().get();
      break;
    case DialogType::Channel:
      offset_chat_id = offset_dialog_id.get_channel_id().get();
      break;
    case DialogType::None:
      if (offset_dialog_id == DialogId()) {
        break;
      }
    // fallthrough
    case DialogType::User:
    case DialogType::SecretChat:
      promise.set_error(Status::Error(6, "Wrong offset_chat_id"));
      return vector<DialogId>();
    default:
      UNREACHABLE();
      break;
  }

  auto it = found_common_dialogs_.find(user_id);
  if (it != found_common_dialogs_.end() && !it->second.dialog_ids.empty()) {
    vector<DialogId> &common_dialog_ids = it->second.dialog_ids;
    bool use_cache = (!it->second.is_outdated && it->second.received_date >= Time::now() - 3600) || force ||
                     offset_chat_id != 0 || common_dialog_ids.size() >= static_cast<size_t>(MAX_GET_DIALOGS);
    // use cache if it's up to date, or we required to use it or we can't update it
    if (use_cache) {
      auto offset_it = common_dialog_ids.begin();
      if (offset_dialog_id != DialogId()) {
        offset_it = std::find(common_dialog_ids.begin(), common_dialog_ids.end(), offset_dialog_id);
        if (offset_it == common_dialog_ids.end()) {
          promise.set_error(Status::Error(6, "Wrong offset_chat_id"));
          return vector<DialogId>();
        }
        ++offset_it;
      }
      vector<DialogId> result;
      while (result.size() < static_cast<size_t>(limit)) {
        if (offset_it == common_dialog_ids.end()) {
          break;
        }
        auto dialog_id = *offset_it++;
        if (dialog_id == DialogId()) {  // end of the list
          promise.set_value(Unit());
          return result;
        }
        result.push_back(dialog_id);
      }
      if (result.size() == static_cast<size_t>(limit) || force) {
        promise.set_value(Unit());
        return result;
      }
    }
  }

  td_->create_handler<GetCommonDialogsQuery>(std::move(promise))->send(user_id, offset_chat_id, MAX_GET_DIALOGS);
  return vector<DialogId>();
}

void MessagesManager::on_get_common_dialogs(UserId user_id, int32 offset_chat_id,
                                            vector<tl_object_ptr<telegram_api::Chat>> &&chats, int32 total_count) {
  td_->contacts_manager_->on_update_user_common_chat_count(user_id, total_count);

  auto &common_dialogs = found_common_dialogs_[user_id];
  if (common_dialogs.is_outdated && offset_chat_id == 0 &&
      common_dialogs.dialog_ids.size() < static_cast<size_t>(MAX_GET_DIALOGS)) {
    // drop outdated cache if possible
    common_dialogs = CommonDialogs();
  }
  if (common_dialogs.received_date == 0) {
    common_dialogs.received_date = Time::now();
  }
  common_dialogs.is_outdated = false;
  auto &result = common_dialogs.dialog_ids;
  if (!result.empty() && result.back() == DialogId()) {
    return;
  }
  bool is_last = chats.empty() && offset_chat_id == 0;
  for (auto &chat : chats) {
    DialogId dialog_id;
    switch (chat->get_id()) {
      case telegram_api::chatEmpty::ID: {
        auto c = static_cast<const telegram_api::chatEmpty *>(chat.get());
        ChatId chat_id(c->id_);
        if (!chat_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << chat_id;
          continue;
        }
        dialog_id = DialogId(chat_id);
        break;
      }
      case telegram_api::chat::ID: {
        auto c = static_cast<const telegram_api::chat *>(chat.get());
        ChatId chat_id(c->id_);
        if (!chat_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << chat_id;
          continue;
        }
        dialog_id = DialogId(chat_id);
        break;
      }
      case telegram_api::chatForbidden::ID: {
        auto c = static_cast<const telegram_api::chatForbidden *>(chat.get());
        ChatId chat_id(c->id_);
        if (!chat_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << chat_id;
          continue;
        }
        dialog_id = DialogId(chat_id);
        break;
      }
      case telegram_api::channel::ID: {
        auto c = static_cast<const telegram_api::channel *>(chat.get());
        ChannelId channel_id(c->id_);
        if (!channel_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << channel_id;
          continue;
        }
        dialog_id = DialogId(channel_id);
        break;
      }
      case telegram_api::channelForbidden::ID: {
        auto c = static_cast<const telegram_api::channelForbidden *>(chat.get());
        ChannelId channel_id(c->id_);
        if (!channel_id.is_valid()) {
          LOG(ERROR) << "Receive invalid " << channel_id;
          continue;
        }
        dialog_id = DialogId(channel_id);
        break;
      }
      default:
        UNREACHABLE();
    }
    CHECK(dialog_id.is_valid());
    td_->contacts_manager_->on_get_chat(std::move(chat), "on_get_common_dialogs");

    if (!td::contains(result, dialog_id)) {
      force_create_dialog(dialog_id, "get common dialogs");
      result.push_back(dialog_id);
    }
  }
  if (result.size() >= static_cast<size_t>(total_count) || is_last) {
    result.push_back(DialogId());
  }
}

bool MessagesManager::have_message_force(FullMessageId full_message_id, const char *source) {
  return get_message_force(full_message_id, source) != nullptr;
}

MessagesManager::Message *MessagesManager::get_message(FullMessageId full_message_id) {
  Dialog *d = get_dialog(full_message_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message(d, full_message_id.get_message_id());
}

const MessagesManager::Message *MessagesManager::get_message(FullMessageId full_message_id) const {
  const Dialog *d = get_dialog(full_message_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message(d, full_message_id.get_message_id());
}

MessagesManager::Message *MessagesManager::get_message_force(FullMessageId full_message_id, const char *source) {
  Dialog *d = get_dialog_force(full_message_id.get_dialog_id());
  if (d == nullptr) {
    return nullptr;
  }

  return get_message_force(d, full_message_id.get_message_id(), source);
}

MessageId MessagesManager::get_replied_message_id(const Message *m) {
  auto message_id = get_message_content_replied_message_id(m->content.get());
  if (message_id.is_valid()) {
    CHECK(!m->reply_to_message_id.is_valid());
    return message_id;
  }
  return m->reply_to_message_id;
}

void MessagesManager::get_message_force_from_server(Dialog *d, MessageId message_id, Promise<Unit> &&promise,
                                                    tl_object_ptr<telegram_api::InputMessage> input_message) {
  LOG(INFO) << "Get " << message_id << " in " << d->dialog_id << " using " << to_string(input_message);
  auto dialog_type = d->dialog_id.get_type();
  auto m = get_message_force(d, message_id, "get_message_force_from_server");
  if (m == nullptr && message_id.is_valid() && message_id.is_server()) {
    if (d->last_new_message_id != MessageId() && message_id > d->last_new_message_id) {
      // message will not be added to the dialog anyway
      if (dialog_type == DialogType::Channel) {
        // so we try to force channel difference first

        // replied message can't be older than already added original message, but pinned message can be
        CHECK(input_message == nullptr || input_message->get_id() == telegram_api::inputMessagePinned::ID);
        postponed_get_message_requests_[d->dialog_id].emplace_back(message_id, std::move(promise),
                                                                   std::move(input_message));
        get_channel_difference(d->dialog_id, d->pts, true, "get_message");
      } else {
        promise.set_value(Unit());
      }
      return;
    }

    if (d->deleted_message_ids.count(message_id) == 0 && dialog_type != DialogType::SecretChat) {
      return get_message_from_server({d->dialog_id, message_id}, std::move(promise), std::move(input_message));
    }
  } else if (m == nullptr && message_id.is_valid_scheduled() && message_id.is_scheduled_server()) {
    if (d->deleted_scheduled_server_message_ids.count(message_id.get_scheduled_server_message_id()) == 0 &&
        dialog_type != DialogType::SecretChat && input_message == nullptr) {
      return get_message_from_server({d->dialog_id, message_id}, std::move(promise));
    }
  }

  promise.set_value(Unit());
}

void MessagesManager::get_message(FullMessageId full_message_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(full_message_id.get_dialog_id());
  if (d == nullptr) {
    return promise.set_error(Status::Error(6, "Chat not found"));
  }

  get_message_force_from_server(d, full_message_id.get_message_id(), std::move(promise));
}

MessageId MessagesManager::get_replied_message(DialogId dialog_id, MessageId message_id, bool force,
                                               Promise<Unit> &&promise) {
  LOG(INFO) << "Get replied message to " << message_id << " in " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return MessageId();
  }

  auto m = get_message_force(d, message_id, "get_replied_message");
  if (m == nullptr) {
    if (force) {
      promise.set_value(Unit());
    } else {
      get_message_force_from_server(d, message_id, std::move(promise));
    }
    return MessageId();
  }

  tl_object_ptr<telegram_api::InputMessage> input_message;
  if (m->message_id.is_valid() && m->message_id.is_server()) {
    input_message = make_tl_object<telegram_api::inputMessageReplyTo>(m->message_id.get_server_message_id().get());
  }
  auto replied_message_id = get_replied_message_id(m);
  get_message_force_from_server(d, replied_message_id, std::move(promise), std::move(input_message));

  return replied_message_id;
}

void MessagesManager::get_dialog_info_full(DialogId dialog_id, Promise<Unit> &&promise) {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      td_->contacts_manager_->get_user_full(dialog_id.get_user_id(), std::move(promise));
      return;
    case DialogType::Chat:
      td_->contacts_manager_->get_chat_full(dialog_id.get_chat_id(), std::move(promise));
      return;
    case DialogType::Channel:
      td_->contacts_manager_->get_channel_full(dialog_id.get_channel_id(), std::move(promise));
      return;
    case DialogType::SecretChat:
      return promise.set_value(Unit());
    case DialogType::None:
    default:
      UNREACHABLE();
      return promise.set_error(Status::Error(500, "Wrong chat type"));
  }
}

MessageId MessagesManager::get_dialog_pinned_message(DialogId dialog_id, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return MessageId();
  }

  LOG(INFO) << "Get pinned message in " << dialog_id << " with "
            << (d->is_pinned_message_id_inited ? "inited" : "unknown") << " pinned " << d->pinned_message_id;

  if (!d->is_pinned_message_id_inited) {
    get_dialog_info_full(dialog_id, std::move(promise));
    return MessageId();
  }

  get_dialog_info_full(dialog_id, Auto());

  if (d->pinned_message_id.is_valid()) {
    tl_object_ptr<telegram_api::InputMessage> input_message;
    if (dialog_id.get_type() == DialogType::Channel) {
      input_message = make_tl_object<telegram_api::inputMessagePinned>();
    }
    get_message_force_from_server(d, d->pinned_message_id, std::move(promise), std::move(input_message));
  }

  return d->pinned_message_id;
}

bool MessagesManager::get_messages(DialogId dialog_id, const vector<MessageId> &message_ids, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return false;
  }

  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  vector<FullMessageId> missed_message_ids;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      promise.set_error(Status::Error(6, "Invalid message identifier"));
      return false;
    }

    auto *m = get_message_force(d, message_id, "get_messages");
    if (m == nullptr && message_id.is_any_server() && !is_secret) {
      missed_message_ids.emplace_back(dialog_id, message_id);
      continue;
    }
  }

  if (!missed_message_ids.empty()) {
    get_messages_from_server(std::move(missed_message_ids), std::move(promise));
    return false;
  }

  promise.set_value(Unit());
  return true;
}

void MessagesManager::get_message_from_server(FullMessageId full_message_id, Promise<Unit> &&promise,
                                              tl_object_ptr<telegram_api::InputMessage> input_message) {
  get_messages_from_server({full_message_id}, std::move(promise), std::move(input_message));
}

void MessagesManager::get_messages_from_server(vector<FullMessageId> &&message_ids, Promise<Unit> &&promise,
                                               tl_object_ptr<telegram_api::InputMessage> input_message) {
  if (G()->close_flag()) {
    return promise.set_error(Status::Error(500, "Request aborted"));
  }
  if (message_ids.empty()) {
    LOG(ERROR) << "Empty message_ids";
    return promise.set_error(Status::Error(500, "There are no messages specified to fetch"));
  }

  if (input_message != nullptr) {
    CHECK(message_ids.size() == 1);
  }

  vector<tl_object_ptr<telegram_api::InputMessage>> ordinary_message_ids;
  std::unordered_map<ChannelId, vector<tl_object_ptr<telegram_api::InputMessage>>, ChannelIdHash> channel_message_ids;
  std::unordered_map<DialogId, vector<int32>, DialogIdHash> scheduled_message_ids;
  for (auto &full_message_id : message_ids) {
    auto dialog_id = full_message_id.get_dialog_id();
    auto message_id = full_message_id.get_message_id();
    if (!message_id.is_valid() || !message_id.is_server()) {
      if (message_id.is_valid_scheduled()) {
        scheduled_message_ids[dialog_id].push_back(message_id.get_scheduled_server_message_id().get());
      }
      continue;
    }

    switch (dialog_id.get_type()) {
      case DialogType::User:
      case DialogType::Chat:
        ordinary_message_ids.push_back(input_message == nullptr ? get_input_message(message_id)
                                                                : std::move(input_message));
        break;
      case DialogType::Channel:
        channel_message_ids[dialog_id.get_channel_id()].push_back(
            input_message == nullptr ? get_input_message(message_id) : std::move(input_message));
        break;
      case DialogType::SecretChat:
        LOG(ERROR) << "Can't get secret chat message from server";
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
        break;
    }
  }

  MultiPromiseActorSafe mpas{"GetMessagesFromServerMultiPromiseActor"};
  mpas.add_promise(std::move(promise));
  auto lock = mpas.get_promise();

  if (!ordinary_message_ids.empty()) {
    td_->create_handler<GetMessagesQuery>(mpas.get_promise())->send(std::move(ordinary_message_ids));
  }

  for (auto &it : scheduled_message_ids) {
    auto dialog_id = it.first;
    have_dialog_force(dialog_id);
    auto input_peer = get_input_peer(dialog_id, AccessRights::Read);
    if (input_peer == nullptr) {
      LOG(ERROR) << "Can't find info about " << dialog_id << " to get a message from it";
      mpas.get_promise().set_error(Status::Error(6, "Can't access the chat"));
      continue;
    }
    td_->create_handler<GetScheduledMessagesQuery>(mpas.get_promise())
        ->send(dialog_id, std::move(input_peer), std::move(it.second));
  }

  for (auto &it : channel_message_ids) {
    td_->contacts_manager_->have_channel_force(it.first);
    auto input_channel = td_->contacts_manager_->get_input_channel(it.first);
    if (input_channel == nullptr) {
      LOG(ERROR) << "Can't find info about " << it.first << " to get a message from it";
      mpas.get_promise().set_error(Status::Error(6, "Can't access the chat"));
      continue;
    }
    td_->create_handler<GetChannelMessagesQuery>(mpas.get_promise())
        ->send(it.first, std::move(input_channel), std::move(it.second));
  }
  lock.set_value(Unit());
}

bool MessagesManager::is_message_edited_recently(FullMessageId full_message_id, int32 seconds) {
  if (seconds < 0) {
    return false;
  }
  if (!full_message_id.get_message_id().is_valid()) {
    return false;
  }

  auto m = get_message_force(full_message_id, "is_message_edited_recently");
  if (m == nullptr) {
    return true;
  }

  return m->edit_date >= G()->unix_time() - seconds;
}

std::pair<string, string> MessagesManager::get_public_message_link(FullMessageId full_message_id, bool for_group,
                                                                   Promise<Unit> &&promise) {
  auto dialog_id = full_message_id.get_dialog_id();
  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(6, "Can't access the chat"));
    return {};
  }
  if (dialog_id.get_type() != DialogType::Channel ||
      td_->contacts_manager_->get_channel_username(dialog_id.get_channel_id()).empty()) {
    promise.set_error(Status::Error(
        6, "Public message links are available only for messages in supergroups and channel chats with a username"));
    return {};
  }

  auto *m = get_message_force(d, full_message_id.get_message_id(), "get_public_message_link");
  if (m == nullptr) {
    promise.set_error(Status::Error(6, "Message not found"));
    return {};
  }
  if (m->message_id.is_yet_unsent()) {
    promise.set_error(Status::Error(6, "Message is yet unsent"));
    return {};
  }
  if (m->message_id.is_scheduled()) {
    promise.set_error(Status::Error(6, "Message is scheduled"));
    return {};
  }
  if (!m->message_id.is_server()) {
    promise.set_error(Status::Error(6, "Message is local"));
    return {};
  }

  auto it = public_message_links_[for_group].find(full_message_id);
  if (it == public_message_links_[for_group].end()) {
    td_->create_handler<ExportChannelMessageLinkQuery>(std::move(promise))
        ->send(dialog_id.get_channel_id(), m->message_id, for_group, false);
    return {};
  }

  promise.set_value(Unit());
  return it->second;
}

void MessagesManager::on_get_public_message_link(FullMessageId full_message_id, bool for_group, string url,
                                                 string html) {
  LOG_IF(ERROR, url.empty() && html.empty()) << "Receive empty public link for " << full_message_id;
  public_message_links_[for_group][full_message_id] = {std::move(url), std::move(html)};
}

string MessagesManager::get_message_link(FullMessageId full_message_id, Promise<Unit> &&promise) {
  auto dialog_id = full_message_id.get_dialog_id();
  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(6, "Can't access the chat"));
    return {};
  }
  if (dialog_id.get_type() != DialogType::Channel) {
    promise.set_error(
        Status::Error(6, "Message links are available only for messages in supergroups and channel chats"));
    return {};
  }

  auto *m = get_message_force(d, full_message_id.get_message_id(), "get_message_link");
  if (m == nullptr) {
    promise.set_error(Status::Error(6, "Message not found"));
    return {};
  }
  if (m->message_id.is_scheduled()) {
    promise.set_error(Status::Error(6, "Message is scheduled"));
    return {};
  }
  if (!m->message_id.is_server()) {
    promise.set_error(Status::Error(6, "Message is local"));
    return {};
  }

  td_->create_handler<ExportChannelMessageLinkQuery>(Promise<Unit>())
      ->send(dialog_id.get_channel_id(), m->message_id, false, true);

  promise.set_value(Unit());
  return PSTRING() << G()->shared_config().get_option_string("t_me_url", "https://t.me/") << "c/"
                   << dialog_id.get_channel_id().get() << "/" << m->message_id.get_server_message_id().get();
}

Result<MessagesManager::MessageLinkInfo> MessagesManager::get_message_link_info(Slice url) {
  if (url.empty()) {
    return Status::Error("URL must be non-empty");
  }

  url.truncate(url.find('#'));

  string lower_cased_url = to_lower(url);
  url = lower_cased_url;

  Slice username;
  Slice channel_id_slice;
  Slice message_id_slice;
  bool is_single = false;
  if (begins_with(url, "tg:")) {
    url = url.substr(3);
    if (begins_with(url, "//")) {
      url = url.substr(2);
    }

    // resolve?domain=username&post=12345&single
    // privatepost?channel=123456789&msg_id=12345

    bool is_resolve = false;
    if (begins_with(url, "resolve")) {
      url = url.substr(7);
      is_resolve = true;
    } else if (begins_with(url, "privatepost")) {
      url = url.substr(11);
    } else {
      return Status::Error("Wrong message link URL");
    }

    if (begins_with(url, "/")) {
      url = url.substr(1);
    }
    if (!begins_with(url, "?")) {
      return Status::Error("Wrong message link URL");
    }
    url = url.substr(1);

    auto args = full_split(url, '&');
    for (auto arg : args) {
      auto key_value = split(arg, '=');
      if (is_resolve) {
        if (key_value.first == "domain") {
          username = key_value.second;
        }
        if (key_value.first == "post") {
          message_id_slice = key_value.second;
        }
      } else {
        if (key_value.first == "channel") {
          channel_id_slice = key_value.second;
        }
        if (key_value.first == "msg_id") {
          message_id_slice = key_value.second;
        }
      }
      if (key_value.first == "single") {
        is_single = true;
      }
    }
  } else {
    if (begins_with(url, "http://") || begins_with(url, "https://")) {
      url = url.substr(url[4] == 's' ? 8 : 7);
    }

    // t.me/c/123456789/12345
    // t.me/username/12345?single

    vector<Slice> t_me_urls{Slice("t.me/"), Slice("telegram.me/"), Slice("telegram.dog/")};
    string cur_t_me_url = G()->shared_config().get_option_string("t_me_url");
    if (begins_with(cur_t_me_url, "http://") || begins_with(cur_t_me_url, "https://")) {
      Slice t_me_url = cur_t_me_url;
      t_me_url = t_me_url.substr(url[4] == 's' ? 8 : 7);
      if (!td::contains(t_me_urls, t_me_url)) {
        t_me_urls.push_back(t_me_url);
      }
    }

    for (auto t_me_url : t_me_urls) {
      if (begins_with(url, t_me_url)) {
        url = url.substr(t_me_url.size());
        auto username_end_pos = url.find('/');
        if (username_end_pos == Slice::npos) {
          return Status::Error("Wrong message link URL");
        }
        username = url.substr(0, username_end_pos);
        url = url.substr(username_end_pos + 1);
        if (username == "c") {
          username = Slice();
          auto channel_id_end_pos = url.find('/');
          if (channel_id_end_pos == Slice::npos) {
            return Status::Error("Wrong message link URL");
          }
          channel_id_slice = url.substr(0, channel_id_end_pos);
          url = url.substr(channel_id_end_pos + 1);
        }

        auto query_pos = url.find('?');
        message_id_slice = url.substr(0, query_pos);
        is_single = query_pos != Slice::npos && url.substr(query_pos + 1) == "single";
        break;
      }
    }
  }

  ChannelId channel_id;
  if (username.empty()) {
    auto r_channel_id = to_integer_safe<int32>(channel_id_slice);
    if (r_channel_id.is_error() || !ChannelId(r_channel_id.ok()).is_valid()) {
      return Status::Error("Wrong channel ID");
    }
    channel_id = ChannelId(r_channel_id.ok());
  }

  auto r_message_id = to_integer_safe<int32>(message_id_slice);
  if (r_message_id.is_error() || !ServerMessageId(r_message_id.ok()).is_valid()) {
    return Status::Error("Wrong message ID");
  }

  MessageLinkInfo info;
  info.username = username.str();
  info.channel_id = channel_id;
  info.message_id = MessageId(ServerMessageId(r_message_id.ok()));
  info.is_single = is_single;
  LOG(INFO) << "Have link to " << info.message_id << " in chat @" << info.username << "/" << channel_id.get();
  return std::move(info);
}

void MessagesManager::get_message_link_info(Slice url, Promise<MessageLinkInfo> &&promise) {
  auto r_message_link_info = get_message_link_info(url);
  if (r_message_link_info.is_error()) {
    return promise.set_error(Status::Error(400, r_message_link_info.error().message()));
  }

  auto info = r_message_link_info.move_as_ok();
  CHECK(info.username.empty() == info.channel_id.is_valid());

  bool have_dialog = info.username.empty() ? td_->contacts_manager_->have_channel_force(info.channel_id)
                                           : resolve_dialog_username(info.username).is_valid();
  if (!have_dialog) {
    auto query_promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), info, promise = std::move(promise)](Result<Unit> &&result) mutable {
          if (result.is_error()) {
            return promise.set_value(std::move(info));
          }
          send_closure(actor_id, &MessagesManager::on_get_message_link_dialog, std::move(info), std::move(promise));
        });
    if (info.username.empty()) {
      td_->contacts_manager_->reload_channel(info.channel_id, std::move(query_promise));
    } else {
      td_->create_handler<ResolveUsernameQuery>(std::move(query_promise))->send(info.username);
    }
    return;
  }

  return on_get_message_link_dialog(std::move(info), std::move(promise));
}

void MessagesManager::on_get_message_link_dialog(MessageLinkInfo &&info, Promise<MessageLinkInfo> &&promise) {
  DialogId dialog_id;
  if (info.username.empty()) {
    if (!td_->contacts_manager_->have_channel(info.channel_id)) {
      return promise.set_error(Status::Error(500, "Chat info not found"));
    }

    dialog_id = DialogId(info.channel_id);
    force_create_dialog(dialog_id, "on_get_message_link_dialog");
  } else {
    dialog_id = resolve_dialog_username(info.username);
    if (dialog_id.is_valid()) {
      force_create_dialog(dialog_id, "on_get_message_link_dialog", true);
    }
  }
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(500, "Chat not found"));
  }

  get_message_force_from_server(
      d, info.message_id,
      PromiseCreator::lambda([info = std::move(info), promise = std::move(promise)](Result<Unit> &&result) mutable {
        promise.set_value(std::move(info));
      }));
}

td_api::object_ptr<td_api::messageLinkInfo> MessagesManager::get_message_link_info_object(
    const MessageLinkInfo &info) const {
  CHECK(info.username.empty() == info.channel_id.is_valid());

  bool is_public = !info.username.empty();
  DialogId dialog_id = is_public ? resolve_dialog_username(info.username) : DialogId(info.channel_id);
  td_api::object_ptr<td_api::message> message;
  bool for_album = false;

  const Dialog *d = get_dialog(dialog_id);
  if (d == nullptr) {
    dialog_id = DialogId();
  } else {
    const Message *m = get_message(d, info.message_id);
    if (m != nullptr) {
      message = get_message_object(dialog_id, m);
      for_album = !info.is_single && m->media_album_id != 0;
    }
  }

  return td_api::make_object<td_api::messageLinkInfo>(is_public, dialog_id.get(), std::move(message), for_album);
}

Status MessagesManager::delete_dialog_reply_markup(DialogId dialog_id, MessageId message_id) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(6, "Bots can't delete chat reply markup");
  }
  if (message_id.is_scheduled()) {
    return Status::Error(6, "Wrong message identifier specified");
  }
  if (!message_id.is_valid()) {
    return Status::Error(6, "Invalid message identifier specified");
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(6, "Chat not found");
  }
  if (d->reply_markup_message_id != message_id) {
    return Status::OK();
  }

  Message *m = get_message_force(d, message_id, "delete_dialog_reply_markup");
  CHECK(m != nullptr);
  CHECK(m->reply_markup != nullptr);

  if (m->reply_markup->type == ReplyMarkup::Type::ForceReply) {
    set_dialog_reply_markup(d, MessageId());
  } else if (m->reply_markup->type == ReplyMarkup::Type::ShowKeyboard) {
    if (!m->reply_markup->is_one_time_keyboard) {
      return Status::Error(6, "Do not need to delete non one-time keyboard");
    }
    if (m->reply_markup->is_personal) {
      m->reply_markup->is_personal = false;
      set_dialog_reply_markup(d, message_id);

      on_message_changed(d, m, true, "delete_dialog_reply_markup");
    }
  } else {
    // non-bots can't have messages with RemoveKeyboard
    UNREACHABLE();
  }
  return Status::OK();
}

class MessagesManager::SaveDialogDraftMessageOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

Status MessagesManager::set_dialog_draft_message(DialogId dialog_id,
                                                 tl_object_ptr<td_api::draftMessage> &&draft_message) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(6, "Bots can't change chat draft message");
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(6, "Chat not found");
  }
  TRY_STATUS(can_send_message(dialog_id));

  unique_ptr<DraftMessage> new_draft_message;
  if (draft_message != nullptr) {
    new_draft_message = make_unique<DraftMessage>();
    new_draft_message->date = G()->unix_time();
    new_draft_message->reply_to_message_id = get_reply_to_message_id(d, MessageId(draft_message->reply_to_message_id_));

    auto input_message_content = std::move(draft_message->input_message_text_);
    if (input_message_content != nullptr) {
      int32 draft_message_content_type = input_message_content->get_id();
      if (draft_message_content_type != td_api::inputMessageText::ID) {
        return Status::Error(5, "Input message content type must be InputMessageText");
      }
      TRY_RESULT(message_content, process_input_message_text(td_->contacts_manager_.get(), dialog_id,
                                                             std::move(input_message_content), false, true));
      new_draft_message->input_message_text = std::move(message_content);
    }

    if (!new_draft_message->reply_to_message_id.is_valid() && new_draft_message->input_message_text.text.text.empty()) {
      new_draft_message = nullptr;
    }
  }

  if (update_dialog_draft_message(d, std::move(new_draft_message), false, true)) {
    if (dialog_id.get_type() != DialogType::SecretChat) {
      if (G()->parameters().use_message_db) {
        LOG(INFO) << "Save draft of " << dialog_id << " to binlog";
        SaveDialogDraftMessageOnServerLogEvent logevent;
        logevent.dialog_id_ = dialog_id;
        auto storer = LogEventStorerImpl<SaveDialogDraftMessageOnServerLogEvent>(logevent);
        if (d->save_draft_message_logevent_id == 0) {
          d->save_draft_message_logevent_id =
              binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SaveDialogDraftMessageOnServer, storer);
          LOG(INFO) << "Add draft logevent " << d->save_draft_message_logevent_id;
        } else {
          auto new_logevent_id = binlog_rewrite(G()->td_db()->get_binlog(), d->save_draft_message_logevent_id,
                                                LogEvent::HandlerType::SaveDialogDraftMessageOnServer, storer);
          LOG(INFO) << "Rewrite draft logevent " << d->save_draft_message_logevent_id << " with " << new_logevent_id;
        }
        d->save_draft_message_logevent_id_generation++;
      }

      pending_draft_message_timeout_.set_timeout_in(dialog_id.get(), d->is_opened ? MIN_SAVE_DRAFT_DELAY : 0);
    }
  }
  return Status::OK();
}

void MessagesManager::save_dialog_draft_message_on_server(DialogId dialog_id) {
  if (G()->close_flag()) {
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  Promise<> promise;
  if (d->save_draft_message_logevent_id != 0) {
    d->save_draft_message_logevent_id_generation++;
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id,
                                      generation = d->save_draft_message_logevent_id_generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_saved_dialog_draft_message, dialog_id, generation);
      }
    });
  }

  // TODO do not send two queries simultaneously or use SequenceDispatcher
  td_->create_handler<SaveDraftMessageQuery>(std::move(promise))->send(dialog_id, d->draft_message);
}

void MessagesManager::on_saved_dialog_draft_message(DialogId dialog_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Saved draft in " << dialog_id << " with logevent " << d->save_draft_message_logevent_id;
  if (d->save_draft_message_logevent_id_generation == generation) {
    CHECK(d->save_draft_message_logevent_id != 0);
    LOG(INFO) << "Delete draft logevent " << d->save_draft_message_logevent_id;
    binlog_erase(G()->td_db()->get_binlog(), d->save_draft_message_logevent_id);
    d->save_draft_message_logevent_id = 0;
  }
}

void MessagesManager::clear_all_draft_messages(bool exclude_secret_chats, Promise<Unit> &&promise) {
  if (!exclude_secret_chats) {
    for (auto &dialog : dialogs_) {
      Dialog *d = dialog.second.get();
      if (d->dialog_id.get_type() == DialogType::SecretChat) {
        update_dialog_draft_message(d, nullptr, false, true);
      }
    }
  }
  td_->create_handler<ClearAllDraftsQuery>(std::move(promise))->send();
}

int32 MessagesManager::get_pinned_dialogs_limit(FolderId folder_id) {
  Slice key{"pinned_chat_count_max"};
  int32 default_limit = 5;
  if (folder_id != FolderId::main()) {
    key = Slice("pinned_archived_chat_count_max");
    default_limit = 100;
  }
  int32 limit = clamp(G()->shared_config().get_option_integer(key), 0, 1000000);
  if (limit <= 0) {
    return default_limit;
  }
  return limit;
}

vector<DialogId> MessagesManager::remove_secret_chat_dialog_ids(vector<DialogId> dialog_ids) {
  td::remove_if(dialog_ids, [](DialogId dialog_id) { return dialog_id.get_type() == DialogType::SecretChat; });
  return dialog_ids;
}

Status MessagesManager::toggle_dialog_is_pinned(DialogId dialog_id, bool is_pinned) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(6, "Bots can't change chat pin state");
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(6, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(6, "Can't access the chat");
  }

  bool was_pinned = d->pinned_order != DEFAULT_ORDER;
  if (is_pinned == was_pinned) {
    return Status::OK();
  }

  if (is_pinned) {
    auto pinned_dialog_ids = get_pinned_dialogs(d->folder_id);
    auto pinned_dialog_count = pinned_dialog_ids.size();
    auto secret_pinned_dialog_count =
        std::count_if(pinned_dialog_ids.begin(), pinned_dialog_ids.end(),
                      [](DialogId dialog_id) { return dialog_id.get_type() == DialogType::SecretChat; });
    size_t dialog_count = dialog_id.get_type() == DialogType::SecretChat
                              ? secret_pinned_dialog_count
                              : pinned_dialog_count - secret_pinned_dialog_count;

    if (dialog_count >= static_cast<size_t>(get_pinned_dialogs_limit(d->folder_id))) {
      return Status::Error(400, "Maximum number of pinned chats exceeded");
    }
  }

  set_dialog_is_pinned(d, is_pinned);
  update_dialog_pos(d, false, "toggle_dialog_is_pinned");

  toggle_dialog_is_pinned_on_server(dialog_id, is_pinned, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsPinnedOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_pinned_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_pinned_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_pinned_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_pinned_on_server_logevent(DialogId dialog_id, bool is_pinned) {
  ToggleDialogIsPinnedOnServerLogEvent logevent{dialog_id, is_pinned};
  auto storer = LogEventStorerImpl<ToggleDialogIsPinnedOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsPinnedOnServer, storer);
}

void MessagesManager::toggle_dialog_is_pinned_on_server(DialogId dialog_id, bool is_pinned, uint64 logevent_id) {
  if (logevent_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_toggle_dialog_is_pinned_on_server_logevent(dialog_id, is_pinned);
  }

  td_->create_handler<ToggleDialogPinQuery>(get_erase_logevent_promise(logevent_id))->send(dialog_id, is_pinned);
}

Status MessagesManager::set_pinned_dialogs(FolderId folder_id, vector<DialogId> dialog_ids) {
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(6, "Bots can't reorder pinned chats");
  }

  int32 dialog_count = 0;
  int32 secret_dialog_count = 0;
  auto dialog_count_limit = get_pinned_dialogs_limit(folder_id);
  for (auto dialog_id : dialog_ids) {
    Dialog *d = get_dialog_force(dialog_id);
    if (d == nullptr) {
      return Status::Error(6, "Chat not found");
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      return Status::Error(6, "Can't access the chat");
    }
    if (dialog_id.get_type() == DialogType::SecretChat) {
      secret_dialog_count++;
    } else {
      dialog_count++;
    }

    if (dialog_count > dialog_count_limit || secret_dialog_count > dialog_count_limit) {
      return Status::Error(400, "Maximum number of pinned chats exceeded");
    }
  }
  std::unordered_set<DialogId, DialogIdHash> new_pinned_dialog_ids(dialog_ids.begin(), dialog_ids.end());
  if (new_pinned_dialog_ids.size() != dialog_ids.size()) {
    return Status::Error(400, "Duplicate chats in the list of pinned chats");
  }

  auto pinned_dialog_ids = get_pinned_dialogs(folder_id);
  if (pinned_dialog_ids == dialog_ids) {
    return Status::OK();
  }
  LOG(INFO) << "Reorder pinned chats order in " << folder_id << " from " << format::as_array(pinned_dialog_ids)
            << " to " << format::as_array(dialog_ids);

  auto server_old_dialog_ids = remove_secret_chat_dialog_ids(pinned_dialog_ids);
  auto server_new_dialog_ids = remove_secret_chat_dialog_ids(dialog_ids);

  std::reverse(pinned_dialog_ids.begin(), pinned_dialog_ids.end());
  std::reverse(dialog_ids.begin(), dialog_ids.end());

  std::unordered_set<DialogId, DialogIdHash> old_pinned_dialog_ids(pinned_dialog_ids.begin(), pinned_dialog_ids.end());
  auto old_it = pinned_dialog_ids.begin();
  for (auto dialog_id : dialog_ids) {
    old_pinned_dialog_ids.erase(dialog_id);
    while (old_it < pinned_dialog_ids.end()) {
      if (*old_it == dialog_id) {
        break;
      }
      old_it++;
    }
    if (old_it < pinned_dialog_ids.end()) {
      // leave dialog where it is
      continue;
    }
    set_dialog_is_pinned(dialog_id, true);
  }
  for (auto dialog_id : old_pinned_dialog_ids) {
    set_dialog_is_pinned(dialog_id, false);
  }

  if (server_old_dialog_ids != server_new_dialog_ids) {
    reorder_pinned_dialogs_on_server(folder_id, server_new_dialog_ids, 0);
  }
  return Status::OK();
}

class MessagesManager::ReorderPinnedDialogsOnServerLogEvent {
 public:
  FolderId folder_id_;
  vector<DialogId> dialog_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(folder_id_, storer);
    td::store(dialog_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    if (parser.version() >= static_cast<int32>(Version::AddFolders)) {
      td::parse(folder_id_, parser);
    } else {
      folder_id_ = FolderId();
    }
    td::parse(dialog_ids_, parser);
  }
};

uint64 MessagesManager::save_reorder_pinned_dialogs_on_server_logevent(FolderId folder_id,
                                                                       const vector<DialogId> &dialog_ids) {
  ReorderPinnedDialogsOnServerLogEvent logevent{folder_id, dialog_ids};
  auto storer = LogEventStorerImpl<ReorderPinnedDialogsOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReorderPinnedDialogsOnServer, storer);
}

void MessagesManager::reorder_pinned_dialogs_on_server(FolderId folder_id, const vector<DialogId> &dialog_ids,
                                                       uint64 logevent_id) {
  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_reorder_pinned_dialogs_on_server_logevent(folder_id, dialog_ids);
  }

  td_->create_handler<ReorderPinnedDialogsQuery>(get_erase_logevent_promise(logevent_id))->send(folder_id, dialog_ids);
}

Status MessagesManager::toggle_dialog_is_marked_as_unread(DialogId dialog_id, bool is_marked_as_unread) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(6, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(6, "Can't access the chat");
  }

  if (is_marked_as_unread == d->is_marked_as_unread) {
    return Status::OK();
  }

  set_dialog_is_marked_as_unread(d, is_marked_as_unread);

  toggle_dialog_is_marked_as_unread_on_server(dialog_id, is_marked_as_unread, 0);
  return Status::OK();
}

class MessagesManager::ToggleDialogIsMarkedAsUnreadOnServerLogEvent {
 public:
  DialogId dialog_id_;
  bool is_marked_as_unread_;

  template <class StorerT>
  void store(StorerT &storer) const {
    BEGIN_STORE_FLAGS();
    STORE_FLAG(is_marked_as_unread_);
    END_STORE_FLAGS();

    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    BEGIN_PARSE_FLAGS();
    PARSE_FLAG(is_marked_as_unread_);
    END_PARSE_FLAGS();

    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_toggle_dialog_is_marked_as_unread_on_server_logevent(DialogId dialog_id,
                                                                                  bool is_marked_as_unread) {
  ToggleDialogIsMarkedAsUnreadOnServerLogEvent logevent{dialog_id, is_marked_as_unread};
  auto storer = LogEventStorerImpl<ToggleDialogIsMarkedAsUnreadOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ToggleDialogIsMarkedAsUnreadOnServer, storer);
}

void MessagesManager::toggle_dialog_is_marked_as_unread_on_server(DialogId dialog_id, bool is_marked_as_unread,
                                                                  uint64 logevent_id) {
  if (logevent_id == 0 && dialog_id.get_type() == DialogType::SecretChat) {
    // don't even create new binlog events
    return;
  }

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_toggle_dialog_is_marked_as_unread_on_server_logevent(dialog_id, is_marked_as_unread);
  }

  td_->create_handler<ToggleDialogUnreadMarkQuery>(get_erase_logevent_promise(logevent_id))
      ->send(dialog_id, is_marked_as_unread);
}

Status MessagesManager::toggle_dialog_silent_send_message(DialogId dialog_id, bool silent_send_message) {
  CHECK(!td_->auth_manager_->is_bot());

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(6, "Chat not found");
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(6, "Can't access the chat");
  }

  if (update_dialog_silent_send_message(d, silent_send_message)) {
    update_dialog_notification_settings_on_server(dialog_id, false);
  }

  return Status::OK();
}

class MessagesManager::UpdateDialogNotificationSettingsOnServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

void MessagesManager::update_dialog_notification_settings_on_server(DialogId dialog_id, bool from_binlog) {
  if (!from_binlog && get_input_notify_peer(dialog_id) == nullptr) {
    // don't even create new binlog events
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (!from_binlog && G()->parameters().use_message_db) {
    LOG(INFO) << "Save notification settings of " << dialog_id << " to binlog";
    UpdateDialogNotificationSettingsOnServerLogEvent logevent;
    logevent.dialog_id_ = dialog_id;
    auto storer = LogEventStorerImpl<UpdateDialogNotificationSettingsOnServerLogEvent>(logevent);
    if (d->save_notification_settings_logevent_id == 0) {
      d->save_notification_settings_logevent_id = binlog_add(
          G()->td_db()->get_binlog(), LogEvent::HandlerType::UpdateDialogNotificationSettingsOnServer, storer);
      LOG(INFO) << "Add notification settings logevent " << d->save_notification_settings_logevent_id;
    } else {
      auto new_logevent_id = binlog_rewrite(G()->td_db()->get_binlog(), d->save_notification_settings_logevent_id,
                                            LogEvent::HandlerType::UpdateDialogNotificationSettingsOnServer, storer);
      LOG(INFO) << "Rewrite notification settings logevent " << d->save_notification_settings_logevent_id << " with "
                << new_logevent_id;
    }
    d->save_notification_settings_logevent_id_generation++;
  }

  Promise<> promise;
  if (d->save_notification_settings_logevent_id != 0) {
    d->save_notification_settings_logevent_id_generation++;
    promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id,
         generation = d->save_notification_settings_logevent_id_generation](Result<Unit> result) {
          if (!G()->close_flag()) {
            send_closure(actor_id, &MessagesManager::on_updated_dialog_notification_settings, dialog_id, generation);
          }
        });
  }

  send_update_dialog_notification_settings_query(dialog_id, std::move(promise));
}

void MessagesManager::send_update_dialog_notification_settings_query(DialogId dialog_id, Promise<Unit> &&promise) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  // TODO do not send two queries simultaneously or use SequenceDispatcher
  td_->create_handler<UpdateDialogNotifySettingsQuery>(std::move(promise))->send(dialog_id, d->notification_settings);
}

void MessagesManager::on_updated_dialog_notification_settings(DialogId dialog_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Saved notification settings in " << dialog_id << " with logevent "
            << d->save_notification_settings_logevent_id;
  if (d->save_notification_settings_logevent_id_generation == generation) {
    CHECK(d->save_notification_settings_logevent_id != 0);
    LOG(INFO) << "Delete notification settings logevent " << d->save_notification_settings_logevent_id;
    binlog_erase(G()->td_db()->get_binlog(), d->save_notification_settings_logevent_id);
    d->save_notification_settings_logevent_id = 0;
  }
}

Status MessagesManager::set_dialog_client_data(DialogId dialog_id, string &&client_data) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(6, "Chat not found");
  }

  d->client_data = std::move(client_data);
  on_dialog_updated(d->dialog_id, "set_dialog_client_data");
  return Status::OK();
}

bool MessagesManager::is_dialog_inited(const Dialog *d) {
  return d != nullptr && d->notification_settings.is_synchronized && d->is_last_read_inbox_message_id_inited &&
         d->is_last_read_outbox_message_id_inited;
}

int32 MessagesManager::get_dialog_mute_until(const Dialog *d) const {
  CHECK(d != nullptr);
  return d->notification_settings.use_default_mute_until ? get_scope_mute_until(d->dialog_id)
                                                         : d->notification_settings.mute_until;
}

bool MessagesManager::is_dialog_muted(const Dialog *d) const {
  return get_dialog_mute_until(d) != 0;
}

bool MessagesManager::is_dialog_pinned_message_notifications_disabled(const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_disable_pinned_message_notifications) {
    auto scope = get_dialog_notification_setting_scope(d->dialog_id);
    return get_scope_notification_settings(scope)->disable_pinned_message_notifications;
  }

  return d->notification_settings.disable_pinned_message_notifications;
}

bool MessagesManager::is_dialog_mention_notifications_disabled(const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->notification_settings.use_default_disable_mention_notifications) {
    auto scope = get_dialog_notification_setting_scope(d->dialog_id);
    return get_scope_notification_settings(scope)->disable_mention_notifications;
  }

  return d->notification_settings.disable_mention_notifications;
}

void MessagesManager::create_dialog(DialogId dialog_id, bool force, Promise<Unit> &&promise) {
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    if (!have_dialog_info_force(dialog_id)) {
      return promise.set_error(Status::Error(6, "Chat info not found"));
    }
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      return promise.set_error(Status::Error(6, "Can't access the chat"));
    }
  }

  if (force || td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    force_create_dialog(dialog_id, "create dialog");
  } else {
    const Dialog *d = get_dialog_force(dialog_id);
    if (!is_dialog_inited(d)) {
      return send_get_dialog_query(dialog_id, std::move(promise));
    }
  }

  promise.set_value(Unit());
}

DialogId MessagesManager::create_new_group_chat(const vector<UserId> &user_ids, const string &title, int64 &random_id,
                                                Promise<Unit> &&promise) {
  LOG(INFO) << "Trying to create group chat \"" << title << "\" with members " << format::as_array(user_ids);

  if (random_id != 0) {
    // request has already been sent before
    auto it = created_dialogs_.find(random_id);
    CHECK(it != created_dialogs_.end());
    auto dialog_id = it->second;
    CHECK(dialog_id.get_type() == DialogType::Chat);
    CHECK(have_dialog(dialog_id));

    created_dialogs_.erase(it);

    // set default notification settings to newly created chat
    on_update_dialog_notify_settings(
        dialog_id, make_tl_object<telegram_api::peerNotifySettings>(0, false, false, 0, ""), "create_new_group_chat");

    promise.set_value(Unit());
    return dialog_id;
  }

  if (user_ids.empty()) {
    promise.set_error(Status::Error(3, "Too few users to create basic group chat"));
    return DialogId();
  }

  auto new_title = clean_name(title, MAX_TITLE_LENGTH);
  if (new_title.empty()) {
    promise.set_error(Status::Error(3, "Title can't be empty"));
    return DialogId();
  }

  vector<tl_object_ptr<telegram_api::InputUser>> input_users;
  for (auto user_id : user_ids) {
    auto input_user = td_->contacts_manager_->get_input_user(user_id);
    if (input_user == nullptr) {
      promise.set_error(Status::Error(3, "User not found"));
      return DialogId();
    }
    input_users.push_back(std::move(input_user));
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || created_dialogs_.find(random_id) != created_dialogs_.end());
  created_dialogs_[random_id];  // reserve place for result

  td_->create_handler<CreateChatQuery>(std::move(promise))->send(std::move(input_users), new_title, random_id);
  return DialogId();
}

DialogId MessagesManager::create_new_channel_chat(const string &title, bool is_megagroup, const string &description,
                                                  const DialogLocation &location, int64 &random_id,
                                                  Promise<Unit> &&promise) {
  LOG(INFO) << "Trying to create " << (is_megagroup ? "supergroup" : "broadcast") << " with title \"" << title
            << "\", description \"" << description << "\" and " << location;

  if (random_id != 0) {
    // request has already been sent before
    auto it = created_dialogs_.find(random_id);
    CHECK(it != created_dialogs_.end());
    auto dialog_id = it->second;
    CHECK(dialog_id.get_type() == DialogType::Channel);
    CHECK(have_dialog(dialog_id));

    created_dialogs_.erase(it);

    // set default notification settings to newly created chat
    on_update_dialog_notify_settings(
        dialog_id, make_tl_object<telegram_api::peerNotifySettings>(0, false, false, 0, ""), "create_new_channel_chat");

    promise.set_value(Unit());
    return dialog_id;
  }

  auto new_title = clean_name(title, MAX_TITLE_LENGTH);
  if (new_title.empty()) {
    promise.set_error(Status::Error(3, "Title can't be empty"));
    return DialogId();
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || created_dialogs_.find(random_id) != created_dialogs_.end());
  created_dialogs_[random_id];  // reserve place for result

  td_->create_handler<CreateChannelQuery>(std::move(promise))
      ->send(new_title, is_megagroup, strip_empty_characters(description, MAX_DESCRIPTION_LENGTH), location, random_id);
  return DialogId();
}

void MessagesManager::create_new_secret_chat(UserId user_id, Promise<SecretChatId> &&promise) {
  auto user_base = td_->contacts_manager_->get_input_user(user_id);
  if (user_base == nullptr || user_base->get_id() != telegram_api::inputUser::ID) {
    return promise.set_error(Status::Error(6, "User not found"));
  }
  auto user = move_tl_object_as<telegram_api::inputUser>(user_base);

  send_closure(G()->secret_chats_manager(), &SecretChatsManager::create_chat, user->user_id_, user->access_hash_,
               std::move(promise));
}

DialogId MessagesManager::migrate_dialog_to_megagroup(DialogId dialog_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Trying to convert " << dialog_id << " to supergroup";

  if (dialog_id.get_type() != DialogType::Chat) {
    promise.set_error(Status::Error(3, "Only basic group chats can be converted to supergroup"));
    return DialogId();
  }

  auto channel_id = td_->contacts_manager_->migrate_chat_to_megagroup(dialog_id.get_chat_id(), promise);
  if (!channel_id.is_valid()) {
    return DialogId();
  }

  if (!td_->contacts_manager_->have_channel(channel_id)) {
    LOG(ERROR) << "Can't find info about supergroup to which the group has migrated";
    promise.set_error(Status::Error(6, "Supergroup is not found"));
    return DialogId();
  }

  auto new_dialog_id = DialogId(channel_id);
  Dialog *d = get_dialog_force(new_dialog_id);
  if (d == nullptr) {
    d = add_dialog(new_dialog_id);
    if (d->pts == 0) {
      d->pts = 1;
      if (is_debug_message_op_enabled()) {
        d->debug_message_op.emplace_back(Dialog::MessageOp::SetPts, d->pts, "migrate");
      }
    }
    update_dialog_pos(d, false, "migrate_dialog_to_megagroup");
  }

  promise.set_value(Unit());
  return new_dialog_id;
}

Status MessagesManager::open_dialog(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(3, "Chat not found");
  }

  open_dialog(d);
  return Status::OK();
}

Status MessagesManager::close_dialog(DialogId dialog_id) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(3, "Chat not found");
  }

  close_dialog(d);
  return Status::OK();
}

DialogId MessagesManager::get_my_dialog_id() const {
  return DialogId(td_->contacts_manager_->get_my_id());
}

Status MessagesManager::view_messages(DialogId dialog_id, const vector<MessageId> &message_ids, bool force_read) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(3, "Chat not found");
  }
  for (auto message_id : message_ids) {
    if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
      return Status::Error(3, "Invalid message identifier");
    }
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(5, "Can't access the chat");
  }

  bool need_read = force_read || d->is_opened;
  MessageId max_message_id;  // max server or local viewed message_id
  vector<MessageId> read_content_message_ids;
  for (auto message_id : message_ids) {
    if (!message_id.is_valid()) {
      continue;
    }

    auto *m = get_message_force(d, message_id, "view_messages");
    if (m != nullptr) {
      if (m->message_id.is_server() && m->views > 0) {
        d->pending_viewed_message_ids.insert(m->message_id);
      }

      if (!m->message_id.is_yet_unsent() && m->message_id > max_message_id) {
        max_message_id = m->message_id;
      }

      auto message_content_type = m->content->get_type();
      if (message_content_type == MessageContentType::LiveLocation) {
        on_message_live_location_viewed(d, m);
      }

      if (need_read && message_content_type != MessageContentType::VoiceNote &&
          message_content_type != MessageContentType::VideoNote &&
          update_message_contains_unread_mention(d, m, false, "view_messages")) {
        CHECK(m->message_id.is_server());
        read_content_message_ids.push_back(m->message_id);
        on_message_changed(d, m, true, "view_messages");
      }
    } else if (!message_id.is_yet_unsent() && message_id > max_message_id &&
               message_id <= d->max_notification_message_id) {
      max_message_id = message_id;
    }
  }
  if (!d->pending_viewed_message_ids.empty()) {
    pending_message_views_timeout_.add_timeout_in(dialog_id.get(), MAX_MESSAGE_VIEW_DELAY);
    d->increment_view_counter |= d->is_opened;
  }
  if (!read_content_message_ids.empty()) {
    read_message_contents_on_server(dialog_id, std::move(read_content_message_ids), 0, Auto());
  }

  if (need_read && max_message_id > d->last_read_inbox_message_id) {
    MessageId last_read_message_id = max_message_id;
    MessageId prev_last_read_inbox_message_id = d->last_read_inbox_message_id;
    read_history_inbox(d->dialog_id, last_read_message_id, -1, "view_messages");

    if (dialog_id.get_type() != DialogType::SecretChat) {
      if (last_read_message_id.get_prev_server_message_id().get() >
          prev_last_read_inbox_message_id.get_prev_server_message_id().get()) {
        read_history_on_server(d, last_read_message_id.get_prev_server_message_id());
      }
    } else {
      if (last_read_message_id > prev_last_read_inbox_message_id) {
        read_history_on_server(d, last_read_message_id);
      }
    }
  }
  if (need_read && d->is_marked_as_unread) {
    set_dialog_is_marked_as_unread(d, false);
  }

  return Status::OK();
}

Status MessagesManager::open_message_content(FullMessageId full_message_id) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(3, "Chat not found");
  }

  auto *m = get_message_force(d, full_message_id.get_message_id(), "open_message_content");
  if (m == nullptr) {
    return Status::Error(4, "Message not found");
  }

  if (m->message_id.is_scheduled() || m->message_id.is_yet_unsent() || m->is_outgoing) {
    return Status::OK();
  }

  if (read_message_content(d, m, true, "open_message_content") &&
      (m->message_id.is_server() || dialog_id.get_type() == DialogType::SecretChat)) {
    read_message_contents_on_server(dialog_id, {m->message_id}, 0, Auto());
  }

  if (m->content->get_type() == MessageContentType::LiveLocation) {
    on_message_live_location_viewed(d, m);
  }

  return Status::OK();
}

class MessagesManager::ReadMessageContentsOnServerLogEvent {
 public:
  DialogId dialog_id_;
  vector<MessageId> message_ids_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(message_ids_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(message_ids_, parser);
  }
};

uint64 MessagesManager::save_read_message_contents_on_server_logevent(DialogId dialog_id,
                                                                      const vector<MessageId> &message_ids) {
  ReadMessageContentsOnServerLogEvent logevent{dialog_id, message_ids};
  auto storer = LogEventStorerImpl<ReadMessageContentsOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadMessageContentsOnServer, storer);
}

void MessagesManager::read_message_contents_on_server(DialogId dialog_id, vector<MessageId> message_ids,
                                                      uint64 logevent_id, Promise<Unit> &&promise, bool skip_logevent) {
  CHECK(!message_ids.empty());

  LOG(INFO) << "Read contents of " << format::as_array(message_ids) << " in " << dialog_id << " on server";

  if (logevent_id == 0 && G()->parameters().use_message_db && !skip_logevent) {
    logevent_id = save_read_message_contents_on_server_logevent(dialog_id, message_ids);
  }

  auto new_promise = get_erase_logevent_promise(logevent_id, std::move(promise));
  promise = std::move(new_promise);  // to prevent self-move

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<ReadMessagesContentsQuery>(std::move(promise))->send(std::move(message_ids));
      break;
    case DialogType::Channel:
      td_->create_handler<ReadChannelMessagesContentsQuery>(std::move(promise))
          ->send(dialog_id.get_channel_id(), std::move(message_ids));
      break;
    case DialogType::SecretChat: {
      CHECK(message_ids.size() == 1);
      auto m = get_message_force({dialog_id, message_ids[0]}, "read_message_contents_on_server");
      if (m != nullptr) {
        send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_open_message,
                     dialog_id.get_secret_chat_id(), m->random_id, std::move(promise));
      } else {
        promise.set_error(Status::Error(400, "Message not found"));
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::open_dialog(Dialog *d) {
  DialogId dialog_id = d->dialog_id;
  if (d->is_opened || !have_input_peer(dialog_id, AccessRights::Read)) {
    return;
  }
  d->is_opened = true;

  auto min_message_id = MessageId(ServerMessageId(1));
  if (d->last_message_id == MessageId() && d->last_read_outbox_message_id < min_message_id && d->messages != nullptr &&
      d->messages->message_id < min_message_id) {
    Message *m = d->messages.get();
    while (m->right != nullptr) {
      m = m->right.get();
    }
    if (m->message_id < min_message_id) {
      read_history_inbox(dialog_id, m->message_id, -1, "open_dialog");
    }
  }

  LOG(INFO) << "Cancel unload timeout for " << dialog_id;
  pending_unload_dialog_timeout_.cancel_timeout(dialog_id.get());

  if (d->new_secret_chat_notification_id.is_valid()) {
    remove_new_secret_chat_notification(d, true);
  }

  get_dialog_pinned_message(dialog_id, Auto());

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      td_->contacts_manager_->repair_chat_participants(dialog_id.get_chat_id());
      repair_dialog_action_bar(dialog_id, "open_dialog");
      break;
    case DialogType::Channel:
      if (!is_broadcast_channel(dialog_id)) {
        auto participant_count = td_->contacts_manager_->get_channel_participant_count(dialog_id.get_channel_id());
        if (participant_count < 195) {  // include unknown participant_count
          td_->contacts_manager_->send_get_channel_participants_query(
              dialog_id.get_channel_id(),
              ChannelParticipantsFilter(td_api::make_object<td_api::supergroupMembersFilterRecent>()), 0, 200, 0,
              Auto());
        }
      }
      get_channel_difference(dialog_id, d->pts, true, "open_dialog");
      repair_dialog_action_bar(dialog_id, "open_dialog");
      break;
    case DialogType::SecretChat: {
      // to repair dialog action bar
      auto user_id = td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
      if (user_id.is_valid()) {
        td_->contacts_manager_->reload_user_full(user_id);
      }
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  if (!td_->auth_manager_->is_bot()) {
    auto online_count_it = dialog_online_member_counts_.find(dialog_id);
    if (online_count_it != dialog_online_member_counts_.end()) {
      auto &info = online_count_it->second;
      CHECK(!info.is_update_sent);
      if (Time::now() - info.updated_time < ONLINE_MEMBER_COUNT_CACHE_EXPIRE_TIME) {
        info.is_update_sent = true;
        send_update_chat_online_member_count(dialog_id, info.online_member_count);
      }
    }

    if (d->has_scheduled_database_messages && !d->is_has_scheduled_database_messages_checked) {
      CHECK(G()->parameters().use_message_db);

      LOG(INFO) << "Send check has_scheduled_database_messages request";
      d->is_has_scheduled_database_messages_checked = true;
      G()->td_db()->get_messages_db_async()->get_scheduled_messages(
          dialog_id, 1,
          PromiseCreator::lambda([dialog_id, actor_id = actor_id(this)](std::vector<BufferSlice> messages) {
            if (messages.empty()) {
              send_closure(actor_id, &MessagesManager::set_dialog_has_scheduled_database_messages, dialog_id, false);
            }
          }));
    }
  }
}

void MessagesManager::close_dialog(Dialog *d) {
  if (!d->is_opened) {
    return;
  }
  d->is_opened = false;

  auto dialog_id = d->dialog_id;
  if (have_input_peer(dialog_id, AccessRights::Write)) {
    if (pending_draft_message_timeout_.has_timeout(dialog_id.get())) {
      pending_draft_message_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
  } else {
    pending_draft_message_timeout_.cancel_timeout(dialog_id.get());
  }

  if (have_input_peer(dialog_id, AccessRights::Read)) {
    if (pending_message_views_timeout_.has_timeout(dialog_id.get())) {
      pending_message_views_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
    if (pending_read_history_timeout_.has_timeout(dialog_id.get())) {
      pending_read_history_timeout_.set_timeout_in(dialog_id.get(), 0.0);
    }
  } else {
    pending_message_views_timeout_.cancel_timeout(dialog_id.get());
    d->pending_viewed_message_ids.clear();
    d->increment_view_counter = false;

    pending_read_history_timeout_.cancel_timeout(dialog_id.get());
  }

  if (is_message_unload_enabled()) {
    LOG(INFO) << "Schedule unload of " << dialog_id;
    pending_unload_dialog_timeout_.set_timeout_in(dialog_id.get(), get_unload_dialog_delay());
  }

  for (auto &it : d->pending_viewed_live_locations) {
    auto live_location_task_id = it.second;
    auto erased_count = viewed_live_location_tasks_.erase(live_location_task_id);
    CHECK(erased_count > 0);
  }
  d->pending_viewed_live_locations.clear();

  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      break;
    case DialogType::Channel:
      channel_get_difference_timeout_.cancel_timeout(dialog_id.get());
      break;
    case DialogType::SecretChat:
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  if (!td_->auth_manager_->is_bot()) {
    auto online_count_it = dialog_online_member_counts_.find(dialog_id);
    if (online_count_it != dialog_online_member_counts_.end()) {
      auto &info = online_count_it->second;
      info.is_update_sent = false;
    }
    update_dialog_online_member_count_timeout_.set_timeout_in(dialog_id.get(), ONLINE_MEMBER_COUNT_CACHE_EXPIRE_TIME);
  }
}

td_api::object_ptr<td_api::ChatType> MessagesManager::get_chat_type_object(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_api::make_object<td_api::chatTypePrivate>(
          td_->contacts_manager_->get_user_id_object(dialog_id.get_user_id(), "chatTypePrivate"));
    case DialogType::Chat:
      return td_api::make_object<td_api::chatTypeBasicGroup>(
          td_->contacts_manager_->get_basic_group_id_object(dialog_id.get_chat_id(), "chatTypeBasicGroup"));
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      auto channel_type = td_->contacts_manager_->get_channel_type(channel_id);
      return td_api::make_object<td_api::chatTypeSupergroup>(
          td_->contacts_manager_->get_supergroup_id_object(channel_id, "chatTypeSupergroup"),
          channel_type != ChannelType::Megagroup);
    }
    case DialogType::SecretChat: {
      auto secret_chat_id = dialog_id.get_secret_chat_id();
      auto user_id = td_->contacts_manager_->get_secret_chat_user_id(secret_chat_id);
      return td_api::make_object<td_api::chatTypeSecret>(
          td_->contacts_manager_->get_secret_chat_id_object(secret_chat_id, "chatTypeSecret"),
          td_->contacts_manager_->get_user_id_object(user_id, "chatTypeSecret"));
    }
    case DialogType::None:
    default:
      UNREACHABLE();
      return nullptr;
  }
}

td_api::object_ptr<td_api::ChatList> MessagesManager::get_chat_list_object(const Dialog *d) {
  if (d->order == DEFAULT_ORDER) {
    return nullptr;
  }
  if (d->folder_id != FolderId::main() && d->folder_id != FolderId::archive()) {
    LOG(ERROR) << "Have " << d->dialog_id << " in unknown " << d->folder_id;
  }
  return get_chat_list_object(d->folder_id);
}

td_api::object_ptr<td_api::ChatList> MessagesManager::get_chat_list_object(FolderId folder_id) {
  if (folder_id == FolderId::archive()) {
    return td_api::make_object<td_api::chatListArchive>();
  }
  if (folder_id == FolderId::main()) {
    return td_api::make_object<td_api::chatListMain>();
  }
  return td_api::make_object<td_api::chatListMain>();
}

td_api::object_ptr<td_api::ChatActionBar> MessagesManager::get_chat_action_bar_object(const Dialog *d) const {
  CHECK(d != nullptr);
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    auto user_id = td_->contacts_manager_->get_secret_chat_user_id(d->dialog_id.get_secret_chat_id());
    if (!user_id.is_valid()) {
      return nullptr;
    }
    const Dialog *user_d = get_dialog(DialogId(user_id));
    if (user_d == nullptr) {
      return nullptr;
    }
    return get_chat_action_bar_object(user_d);
  }

  if (!d->know_action_bar) {
    if (d->know_can_report_spam && d->dialog_id.get_type() != DialogType::SecretChat && d->can_report_spam) {
      return td_api::make_object<td_api::chatActionBarReportSpam>();
    }
    return nullptr;
  }

  if (d->can_report_location) {
    CHECK(d->dialog_id.get_type() == DialogType::Channel);
    CHECK(!d->can_share_phone_number && !d->can_block_user && !d->can_add_contact && !d->can_report_spam);
    return td_api::make_object<td_api::chatActionBarReportUnrelatedLocation>();
  }
  if (d->can_share_phone_number) {
    CHECK(d->dialog_id.get_type() == DialogType::User);
    CHECK(!d->can_block_user && !d->can_add_contact && !d->can_report_spam);
    return td_api::make_object<td_api::chatActionBarSharePhoneNumber>();
  }
  if (d->can_block_user) {
    CHECK(d->dialog_id.get_type() == DialogType::User);
    CHECK(d->can_report_spam && d->can_add_contact);
    return td_api::make_object<td_api::chatActionBarReportAddBlock>();
  }
  if (d->can_add_contact) {
    CHECK(d->dialog_id.get_type() == DialogType::User);
    CHECK(!d->can_report_spam);
    return td_api::make_object<td_api::chatActionBarAddContact>();
  }
  if (d->can_report_spam) {
    return td_api::make_object<td_api::chatActionBarReportSpam>();
  }
  return nullptr;
}

td_api::object_ptr<td_api::chat> MessagesManager::get_chat_object(const Dialog *d) const {
  CHECK(d != nullptr);

  bool can_delete_for_self = false;
  bool can_delete_for_all_users = false;
  if (!td_->auth_manager_->is_bot()) {
    switch (d->dialog_id.get_type()) {
      case DialogType::User:
        can_delete_for_self = true;
        can_delete_for_all_users = G()->shared_config().get_option_boolean("revoke_pm_inbox", true);
        if (d->dialog_id == get_my_dialog_id() || td_->contacts_manager_->is_user_deleted(d->dialog_id.get_user_id()) ||
            td_->contacts_manager_->is_user_bot(d->dialog_id.get_user_id())) {
          can_delete_for_all_users = false;
        }
        break;
      case DialogType::Chat:
        // chats can't be deleted only for self with deleteChatHistory
        can_delete_for_self = true;
        break;
      case DialogType::Channel:
        if (is_broadcast_channel(d->dialog_id) ||
            td_->contacts_manager_->is_channel_public(d->dialog_id.get_channel_id())) {
          // deleteChatHistory can't be used in channels and public supergroups
        } else {
          // private supergroups can be deleted for self
          can_delete_for_self = true;
        }
        break;
      case DialogType::SecretChat:
        if (td_->contacts_manager_->get_secret_chat_state(d->dialog_id.get_secret_chat_id()) ==
            SecretChatState::Closed) {
          // in a closed secret chats there is no way to delete messages for both users
          can_delete_for_self = true;
        } else {
          // active secret chats can be deleted only for both users
          can_delete_for_all_users = true;
        }
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }

  bool has_scheduled_messages =
      d->has_scheduled_server_messages || d->has_scheduled_database_messages || d->scheduled_messages != nullptr;
  return make_tl_object<td_api::chat>(
      d->dialog_id.get(), get_chat_type_object(d->dialog_id), get_chat_list_object(d), get_dialog_title(d->dialog_id),
      get_chat_photo_object(td_->file_manager_.get(), get_dialog_photo(d->dialog_id)),
      get_dialog_permissions(d->dialog_id).get_chat_permissions_object(),
      get_message_object(d->dialog_id, get_message(d, d->last_message_id)), get_dialog_public_order(d),
      d->pinned_order != DEFAULT_ORDER, d->is_marked_as_unread, d->order == SPONSORED_DIALOG_ORDER,
      has_scheduled_messages, can_delete_for_self, can_delete_for_all_users, can_report_dialog(d->dialog_id),
      d->notification_settings.silent_send_message, d->server_unread_count + d->local_unread_count,
      d->last_read_inbox_message_id.get(), d->last_read_outbox_message_id.get(), d->unread_mention_count,
      get_chat_notification_settings_object(&d->notification_settings), get_chat_action_bar_object(d),
      d->pinned_message_id.get(), d->reply_markup_message_id.get(), get_draft_message_object(d->draft_message),
      d->client_data);
}

tl_object_ptr<td_api::chat> MessagesManager::get_chat_object(DialogId dialog_id) const {
  return get_chat_object(get_dialog(dialog_id));
}

tl_object_ptr<td_api::chats> MessagesManager::get_chats_object(const vector<DialogId> &dialogs) {
  return td_api::make_object<td_api::chats>(transform(dialogs, [](DialogId dialog_id) { return dialog_id.get(); }));
}

td_api::object_ptr<td_api::updateScopeNotificationSettings>
MessagesManager::get_update_scope_notification_settings_object(NotificationSettingsScope scope) const {
  auto notification_settings = get_scope_notification_settings(scope);
  CHECK(notification_settings != nullptr);
  return td_api::make_object<td_api::updateScopeNotificationSettings>(
      get_notification_settings_scope_object(scope), get_scope_notification_settings_object(notification_settings));
}

std::pair<bool, int32> MessagesManager::get_dialog_mute_until(DialogId dialog_id, const Dialog *d) const {
  if (d == nullptr || !d->notification_settings.is_synchronized) {
    return {false, get_scope_mute_until(dialog_id)};
  }

  return {d->notification_settings.is_use_default_fixed, get_dialog_mute_until(d)};
}

NotificationSettingsScope MessagesManager::get_dialog_notification_setting_scope(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::SecretChat:
      return NotificationSettingsScope::Private;
    case DialogType::Chat:
      return NotificationSettingsScope::Group;
    case DialogType::Channel:
      return is_broadcast_channel(dialog_id) ? NotificationSettingsScope::Channel : NotificationSettingsScope::Group;
    case DialogType::None:
    default:
      UNREACHABLE();
      return NotificationSettingsScope::Private;
  }
}

int32 MessagesManager::get_scope_mute_until(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::SecretChat:
      return users_notification_settings_.mute_until;
    case DialogType::Chat:
      return chats_notification_settings_.mute_until;
    case DialogType::Channel:
      return is_broadcast_channel(dialog_id) ? channels_notification_settings_.mute_until
                                             : chats_notification_settings_.mute_until;
    case DialogType::None:
    default:
      UNREACHABLE();
      return 0;
  }
}

vector<DialogId> MessagesManager::get_dialog_notification_settings_exceptions(NotificationSettingsScope scope,
                                                                              bool filter_scope, bool compare_sound,
                                                                              bool force, Promise<Unit> &&promise) {
  bool have_all_dialogs = true;
  bool have_main_list = false;
  bool have_archive_list = false;
  for (auto &list : dialog_lists_) {
    if (list.second.last_dialog_date_ != MAX_DIALOG_DATE) {
      have_all_dialogs = false;
    }
    have_main_list |= list.first == FolderId::main();
    have_archive_list |= list.first == FolderId::archive();
  }
  if (!have_main_list || !have_archive_list) {
    have_all_dialogs = false;
  }

  if (have_all_dialogs || force) {
    vector<DialogDate> ordered_dialogs;
    auto my_dialog_id = get_my_dialog_id();
    for (auto &list : dialog_lists_) {
      for (const auto &it : list.second.ordered_server_dialogs_) {
        auto dialog_id = it.get_dialog_id();
        if (filter_scope && get_dialog_notification_setting_scope(dialog_id) != scope) {
          continue;
        }
        if (dialog_id == my_dialog_id) {
          continue;
        }

        Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        if (d->order == DEFAULT_ORDER) {
          break;
        }
        if (are_default_dialog_notification_settings(d->notification_settings, compare_sound)) {
          continue;
        }
        if (is_dialog_message_notification_disabled(dialog_id, std::numeric_limits<int32>::max())) {
          continue;
        }
        ordered_dialogs.push_back(DialogDate(d->order, dialog_id));
      }
    }
    std::sort(ordered_dialogs.begin(), ordered_dialogs.end());

    vector<DialogId> result;
    for (auto &it : ordered_dialogs) {
      CHECK(result.empty() || result.back() != it.get_dialog_id());
      result.push_back(it.get_dialog_id());
    }
    promise.set_value(Unit());
    return result;
  }

  load_dialog_list(FolderId::main(), MAX_GET_DIALOGS, true, Auto());
  load_dialog_list(FolderId::archive(), MAX_GET_DIALOGS, true, Auto());
  for (auto &list : dialog_lists_) {
    if (list.first != FolderId::main() && list.first != FolderId::archive()) {
      load_dialog_list(list.first, MAX_GET_DIALOGS, true, Auto());
    }
  }

  td_->create_handler<GetNotifySettingsExceptionsQuery>(std::move(promise))->send(scope, filter_scope, compare_sound);
  return {};
}

const ScopeNotificationSettings *MessagesManager::get_scope_notification_settings(NotificationSettingsScope scope,
                                                                                  Promise<Unit> &&promise) {
  const ScopeNotificationSettings *notification_settings = get_scope_notification_settings(scope);
  CHECK(notification_settings != nullptr);
  if (!notification_settings->is_synchronized && !td_->auth_manager_->is_bot()) {
    send_get_scope_notification_settings_query(scope, std::move(promise));
    return nullptr;
  }

  promise.set_value(Unit());
  return notification_settings;
}

DialogNotificationSettings *MessagesManager::get_dialog_notification_settings(DialogId dialog_id, bool force) {
  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return nullptr;
  }
  if (!force && !have_input_peer(dialog_id, AccessRights::Read)) {
    return nullptr;
  }
  return &d->notification_settings;
}

ScopeNotificationSettings *MessagesManager::get_scope_notification_settings(NotificationSettingsScope scope) {
  switch (scope) {
    case NotificationSettingsScope::Private:
      return &users_notification_settings_;
    case NotificationSettingsScope::Group:
      return &chats_notification_settings_;
    case NotificationSettingsScope::Channel:
      return &channels_notification_settings_;
    default:
      UNREACHABLE();
      return nullptr;
  }
}

const ScopeNotificationSettings *MessagesManager::get_scope_notification_settings(
    NotificationSettingsScope scope) const {
  switch (scope) {
    case NotificationSettingsScope::Private:
      return &users_notification_settings_;
    case NotificationSettingsScope::Group:
      return &chats_notification_settings_;
    case NotificationSettingsScope::Channel:
      return &channels_notification_settings_;
    default:
      UNREACHABLE();
      return nullptr;
  }
}

tl_object_ptr<telegram_api::InputNotifyPeer> MessagesManager::get_input_notify_peer(DialogId dialog_id) const {
  if (get_dialog(dialog_id) == nullptr) {
    return nullptr;
  }
  auto input_peer = get_input_peer(dialog_id, AccessRights::Read);
  if (input_peer == nullptr) {
    return nullptr;
  }
  return make_tl_object<telegram_api::inputNotifyPeer>(std::move(input_peer));
}

Status MessagesManager::set_dialog_notification_settings(
    DialogId dialog_id, tl_object_ptr<td_api::chatNotificationSettings> &&notification_settings) {
  auto current_settings = get_dialog_notification_settings(dialog_id, false);
  if (current_settings == nullptr) {
    return Status::Error(6, "Wrong chat identifier specified");
  }
  if (dialog_id == get_my_dialog_id()) {
    return Status::Error(6, "Notification settings of the Saved Messages chat can't be changed");
  }

  TRY_RESULT(new_settings, ::td::get_dialog_notification_settings(std::move(notification_settings),
                                                                  current_settings->silent_send_message));
  if (update_dialog_notification_settings(dialog_id, current_settings, new_settings)) {
    update_dialog_notification_settings_on_server(dialog_id, false);
  }
  return Status::OK();
}

Status MessagesManager::set_scope_notification_settings(
    NotificationSettingsScope scope, tl_object_ptr<td_api::scopeNotificationSettings> &&notification_settings) {
  TRY_RESULT(new_settings, ::td::get_scope_notification_settings(std::move(notification_settings)));
  if (update_scope_notification_settings(scope, get_scope_notification_settings(scope), new_settings)) {
    update_scope_notification_settings_on_server(scope, 0);
  }
  return Status::OK();
}

class MessagesManager::UpdateScopeNotificationSettingsOnServerLogEvent {
 public:
  NotificationSettingsScope scope_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(scope_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(scope_, parser);
  }
};

uint64 MessagesManager::save_update_scope_notification_settings_on_server_logevent(NotificationSettingsScope scope) {
  UpdateScopeNotificationSettingsOnServerLogEvent logevent{scope};
  auto storer = LogEventStorerImpl<UpdateScopeNotificationSettingsOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::UpdateScopeNotificationSettingsOnServer, storer);
}

void MessagesManager::update_scope_notification_settings_on_server(NotificationSettingsScope scope,
                                                                   uint64 logevent_id) {
  if (logevent_id == 0) {
    logevent_id = save_update_scope_notification_settings_on_server_logevent(scope);
  }

  LOG(INFO) << "Update " << scope << " notification settings on server with logevent " << logevent_id;
  td_->create_handler<UpdateScopeNotifySettingsQuery>(get_erase_logevent_promise(logevent_id))
      ->send(scope, *get_scope_notification_settings(scope));
}

void MessagesManager::reset_all_notification_settings() {
  DialogNotificationSettings new_dialog_settings;
  ScopeNotificationSettings new_scope_settings;
  new_dialog_settings.is_synchronized = true;
  new_scope_settings.is_synchronized = true;

  update_scope_notification_settings(NotificationSettingsScope::Private, &users_notification_settings_,
                                     new_scope_settings);
  update_scope_notification_settings(NotificationSettingsScope::Group, &chats_notification_settings_,
                                     new_scope_settings);
  update_scope_notification_settings(NotificationSettingsScope::Channel, &channels_notification_settings_,
                                     new_scope_settings);

  for (auto &dialog : dialogs_) {
    Dialog *d = dialog.second.get();
    update_dialog_notification_settings(d->dialog_id, &d->notification_settings, new_dialog_settings);
  }
  reset_all_notification_settings_on_server(0);
}

class MessagesManager::ResetAllNotificationSettingsOnServerLogEvent {
 public:
  template <class StorerT>
  void store(StorerT &storer) const {
  }

  template <class ParserT>
  void parse(ParserT &parser) {
  }
};

uint64 MessagesManager::save_reset_all_notification_settings_on_server_logevent() {
  ResetAllNotificationSettingsOnServerLogEvent logevent;
  auto storer = LogEventStorerImpl<ResetAllNotificationSettingsOnServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ResetAllNotificationSettingsOnServer, storer);
}

void MessagesManager::reset_all_notification_settings_on_server(uint64 logevent_id) {
  if (logevent_id == 0) {
    logevent_id = save_reset_all_notification_settings_on_server_logevent();
  }

  LOG(INFO) << "Reset all notification settings";
  td_->create_handler<ResetNotifySettingsQuery>(get_erase_logevent_promise(logevent_id))->send();
}

tl_object_ptr<td_api::messages> MessagesManager::get_dialog_history(DialogId dialog_id, MessageId from_message_id,
                                                                    int32 offset, int32 limit, int left_tries,
                                                                    bool only_local, Promise<Unit> &&promise) {
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit must be positive"));
    return nullptr;
  }
  if (limit > MAX_GET_HISTORY) {
    limit = MAX_GET_HISTORY;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(5, "Parameter offset must be non-positive"));
    return nullptr;
  }
  if (offset <= -MAX_GET_HISTORY) {
    promise.set_error(Status::Error(5, "Parameter offset must be greater than -100"));
    return nullptr;
  }
  if (offset < -limit) {
    promise.set_error(Status::Error(5, "Parameter offset must not be less than -limit"));
    return nullptr;
  }
  bool is_limit_increased = false;
  if (limit == -offset) {
    limit++;
    is_limit_increased = true;
  }
  CHECK(0 < limit && limit <= MAX_GET_HISTORY);
  CHECK(-limit < offset && offset <= 0);

  if (from_message_id == MessageId() || from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }
  if (!from_message_id.is_valid()) {
    promise.set_error(Status::Error(3, "Invalid value of parameter from_message_id specified"));
    return nullptr;
  }

  const Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return nullptr;
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(5, "Can't access the chat"));
    return nullptr;
  }

  LOG(INFO) << "Get " << (only_local ? "local " : "") << "history in " << dialog_id << " from " << from_message_id
            << " with offset " << offset << " and limit " << limit << ", " << left_tries
            << " tries left. Last read inbox message is " << d->last_read_inbox_message_id
            << ", last read outbox message is " << d->last_read_outbox_message_id
            << ", have_full_history = " << d->have_full_history;

  MessagesConstIterator p(d, from_message_id);
  LOG(DEBUG) << "Iterator points to " << (*p ? (*p)->message_id : MessageId());
  bool from_the_end = (d->last_message_id != MessageId() && from_message_id > d->last_message_id) ||
                      from_message_id >= MessageId::max();

  if (from_the_end) {
    limit += offset;
    offset = 0;
    if (d->last_message_id == MessageId()) {
      p = MessagesConstIterator();
    }
  } else {
    bool have_a_gap = false;
    if (*p == nullptr) {
      // there is no gap if from_message_id is less than first message in the dialog
      if (left_tries == 0 && d->messages != nullptr && offset < 0) {
        const Message *cur = d->messages.get();
        while (cur->left != nullptr) {
          cur = cur->left.get();
        }
        CHECK(cur->message_id > from_message_id);
        from_message_id = cur->message_id;
        p = MessagesConstIterator(d, from_message_id);
      } else {
        have_a_gap = true;
      }
    } else if ((*p)->message_id != from_message_id) {
      CHECK((*p)->message_id < from_message_id);
      if (!(*p)->have_next && (d->last_message_id == MessageId() || (*p)->message_id < d->last_message_id)) {
        have_a_gap = true;
      }
    }

    if (have_a_gap) {
      LOG(INFO) << "Have a gap near message to get chat history from";
      p = MessagesConstIterator();
    }
    if (*p != nullptr && (*p)->message_id == from_message_id) {
      if (offset < 0) {
        offset++;
      } else {
        --p;
      }
    }

    while (*p != nullptr && offset < 0) {
      ++p;
      if (*p) {
        ++offset;
        from_message_id = (*p)->message_id;
      }
    }

    if (offset < 0 && ((d->last_message_id != MessageId() && from_message_id >= d->last_message_id) ||
                       (!have_a_gap && left_tries == 0))) {
      CHECK(!have_a_gap);
      limit += offset;
      offset = 0;
      p = MessagesConstIterator(d, from_message_id);
    }

    if (!have_a_gap && offset < 0) {
      offset--;
    }
  }

  LOG(INFO) << "Iterator after applying offset points to " << (*p ? (*p)->message_id : MessageId())
            << ", offset = " << offset << ", limit = " << limit << ", from_the_end = " << from_the_end;
  vector<tl_object_ptr<td_api::message>> messages;
  if (*p != nullptr && offset == 0) {
    while (*p != nullptr && messages.size() < static_cast<size_t>(limit)) {
      messages.push_back(get_message_object(dialog_id, *p));
      from_message_id = (*p)->message_id;
      --p;
    }
  }

  if (messages.size()) {
    // maybe need some messages
    CHECK(offset == 0);
    preload_newer_messages(d, MessageId(messages[0]->id_));
    preload_older_messages(d, MessageId(messages.back()->id_));
  } else if (messages.size() < static_cast<size_t>(limit) && left_tries != 0 &&
             !(d->is_empty && d->have_full_history && left_tries < 3)) {
    // there can be more messages in the database or on the server, need to load them
    if (from_the_end) {
      from_message_id = MessageId();
    }
    send_closure_later(actor_id(this), &MessagesManager::load_messages, d->dialog_id, from_message_id, offset,
                       limit - static_cast<int32>(messages.size()), left_tries, only_local, std::move(promise));
    return nullptr;
  }

  LOG(INFO) << "Have " << messages.size() << " messages out of requested "
            << (is_limit_increased ? "increased " : "exact ") << limit;
  if (is_limit_increased && static_cast<size_t>(limit) == messages.size()) {
    messages.pop_back();
  }

  LOG(INFO) << "Return " << messages.size() << " messages in result to getChatHistory";
  promise.set_value(Unit());                            // can return some messages
  return get_messages_object(-1, std::move(messages));  // TODO return real total_count of messages in the dialog
}

class MessagesManager::ReadHistoryOnServerLogEvent {
 public:
  DialogId dialog_id_;
  MessageId max_message_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(max_message_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(max_message_id_, parser);
  }
};

class MessagesManager::ReadHistoryInSecretChatLogEvent {
 public:
  DialogId dialog_id_;
  int32 max_date_ = 0;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(max_date_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(max_date_, parser);
  }
};

void MessagesManager::read_history_on_server(Dialog *d, MessageId max_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  CHECK(!max_message_id.is_scheduled());

  auto dialog_id = d->dialog_id;
  LOG(INFO) << "Read history in " << dialog_id << " on server up to " << max_message_id;

  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;
  if (is_secret) {
    auto *m = get_message_force(d, max_message_id, "read_history_on_server");
    if (m == nullptr) {
      LOG(ERROR) << "Failed to read history in " << dialog_id << " up to " << max_message_id;
      return;
    }

    ReadHistoryInSecretChatLogEvent logevent;
    logevent.dialog_id_ = dialog_id;
    logevent.max_date_ = m->date;

    d->last_read_inbox_message_date = m->date;

    auto storer = LogEventStorerImpl<ReadHistoryInSecretChatLogEvent>(logevent);
    if (d->read_history_logevent_id == 0) {
      d->read_history_logevent_id =
          binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadHistoryInSecretChat, storer);
      LOG(INFO) << "Add read history logevent " << d->read_history_logevent_id;
    } else {
      auto new_logevent_id = binlog_rewrite(G()->td_db()->get_binlog(), d->read_history_logevent_id,
                                            LogEvent::HandlerType::ReadHistoryInSecretChat, storer);
      LOG(INFO) << "Rewrite read history logevent " << d->read_history_logevent_id << " with " << new_logevent_id;
    }
    d->read_history_logevent_id_generation++;
  } else if (G()->parameters().use_message_db) {
    ReadHistoryOnServerLogEvent logevent;
    logevent.dialog_id_ = dialog_id;
    logevent.max_message_id_ = max_message_id;

    auto storer = LogEventStorerImpl<ReadHistoryOnServerLogEvent>(logevent);
    if (d->read_history_logevent_id == 0) {
      d->read_history_logevent_id =
          binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ReadHistoryOnServer, storer);
      LOG(INFO) << "Add read history logevent " << d->read_history_logevent_id;
    } else {
      auto new_logevent_id = binlog_rewrite(G()->td_db()->get_binlog(), d->read_history_logevent_id,
                                            LogEvent::HandlerType::ReadHistoryOnServer, storer);
      LOG(INFO) << "Rewrite read history logevent " << d->read_history_logevent_id << " with " << new_logevent_id;
    }
    d->read_history_logevent_id_generation++;
  }

  bool need_delay = d->is_opened && !is_secret && d->server_unread_count > 0;
  pending_read_history_timeout_.set_timeout_in(dialog_id.get(), need_delay ? MIN_READ_HISTORY_DELAY : 0);
}

void MessagesManager::read_history_on_server_impl(DialogId dialog_id, MessageId max_message_id) {
  if (G()->close_flag()) {
    return;
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto message_id = d->last_read_inbox_message_id;
  if (dialog_id.get_type() != DialogType::SecretChat) {
    message_id = message_id.get_prev_server_message_id();
  }
  if (message_id > max_message_id) {
    max_message_id = message_id;
  }

  Promise<> promise;
  if (d->read_history_logevent_id != 0) {
    d->read_history_logevent_id_generation++;
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id,
                                      generation = d->read_history_logevent_id_generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_read_history_finished, dialog_id, generation);
      }
    });
  }

  if (!max_message_id.is_valid()) {
    return promise.set_value(Unit());
  }

  LOG(INFO) << "Send read history request in " << dialog_id << " up to " << max_message_id;
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      td_->create_handler<ReadHistoryQuery>(std::move(promise))->send(dialog_id, max_message_id);
      break;
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      td_->create_handler<ReadChannelHistoryQuery>(std::move(promise))->send(channel_id, max_message_id);
      break;
    }
    case DialogType::SecretChat: {
      auto secret_chat_id = dialog_id.get_secret_chat_id();
      auto date = d->last_read_inbox_message_date;
      auto *m = get_message_force(d, max_message_id, "read_history_on_server_impl");
      if (m != nullptr && m->date > date) {
        date = m->date;
      }
      if (date == 0) {
        LOG(ERROR) << "Don't know last read inbox message date in " << dialog_id;
        return promise.set_value(Unit());
      }
      send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_read_history, secret_chat_id, date,
                   std::move(promise));
      break;
    }
    case DialogType::None:
    default:
      UNREACHABLE();
  }
}

void MessagesManager::on_read_history_finished(DialogId dialog_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Finished reading history in " << dialog_id << " with logevent " << d->read_history_logevent_id;
  if (d->read_history_logevent_id_generation == generation) {
    CHECK(d->read_history_logevent_id != 0);
    LOG(INFO) << "Delete read history logevent " << d->read_history_logevent_id;
    binlog_erase(G()->td_db()->get_binlog(), d->read_history_logevent_id);
    d->read_history_logevent_id = 0;
  }
}

std::pair<int32, vector<MessageId>> MessagesManager::search_dialog_messages(
    DialogId dialog_id, const string &query, UserId sender_user_id, MessageId from_message_id, int32 offset,
    int32 limit, const tl_object_ptr<td_api::SearchMessagesFilter> &filter, int64 &random_id, bool use_db,
    Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_messages_.find(random_id);
    if (it != found_dialog_messages_.end()) {
      auto result = std::move(it->second);
      found_dialog_messages_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Search messages with query \"" << query << "\" in " << dialog_id << " sent by " << sender_user_id
            << " filtered by " << to_string(filter) << " from " << from_message_id << " with offset " << offset
            << " and limit " << limit;

  std::pair<int32, vector<MessageId>> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }
  if (limit <= -offset) {
    promise.set_error(Status::Error(5, "Parameter limit must be greater than -offset"));
    return result;
  }
  if (offset > 0) {
    promise.set_error(Status::Error(5, "Parameter offset must be non-positive"));
    return result;
  }

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(3, "Parameter from_message_id must be identifier of the chat message or 0"));
    return result;
  }
  from_message_id = from_message_id.get_next_server_message_id();

  const Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return result;
  }

  auto input_user = td_->contacts_manager_->get_input_user(sender_user_id);
  if (sender_user_id.is_valid() && input_user == nullptr) {
    promise.set_error(Status::Error(6, "Wrong sender user identifier specified"));
    return result;
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_messages_.find(random_id) != found_dialog_messages_.end());
  found_dialog_messages_[random_id];  // reserve place for result

  auto filter_type = get_search_messages_filter(filter);
  if (filter_type == SearchMessagesFilter::UnreadMention) {
    if (!query.empty()) {
      promise.set_error(Status::Error(6, "Non-empty query is unsupported with the specified filter"));
      return result;
    }
    if (input_user != nullptr) {
      promise.set_error(Status::Error(6, "Non-empty sender user is unsupported with the specified filter"));
      return result;
    }
  }

  // Trying to use database
  if (use_db && query.empty() && G()->parameters().use_message_db && filter_type != SearchMessagesFilter::Empty &&
      input_user == nullptr) {  // TODO support filter by users in the database
    MessageId first_db_message_id = get_first_database_message_id_by_index(d, filter_type);
    int32 message_count = d->message_count_by_index[search_messages_filter_index(filter_type)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    LOG(INFO) << "Search messages in " << dialog_id << " from " << fixed_from_message_id << ", have up to "
              << first_db_message_id << ", message_count = " << message_count;
    if ((first_db_message_id < fixed_from_message_id || (first_db_message_id == fixed_from_message_id && offset < 0)) &&
        message_count != -1) {
      LOG(INFO) << "Search messages in database in " << dialog_id << " from " << fixed_from_message_id
                << " and with limit " << limit;
      auto new_promise = PromiseCreator::lambda(
          [random_id, dialog_id, fixed_from_message_id, first_db_message_id, filter_type, offset, limit,
           promise = std::move(promise)](Result<std::vector<BufferSlice>> r_messages) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_search_dialog_messages_db_result, random_id,
                         dialog_id, fixed_from_message_id, first_db_message_id, filter_type, offset, limit,
                         std::move(r_messages), std::move(promise));
          });
      MessagesDbMessagesQuery db_query;
      db_query.dialog_id = dialog_id;
      db_query.index_mask = search_messages_filter_index_mask(filter_type);
      db_query.from_message_id = fixed_from_message_id;
      db_query.offset = offset;
      db_query.limit = limit;
      G()->td_db()->get_messages_db_async()->get_messages(db_query, std::move(new_promise));
      return result;
    }
  }

  LOG(DEBUG) << "Search messages on server in " << dialog_id << " with query \"" << query << "\" from user "
             << sender_user_id << " from " << from_message_id << " and with limit " << limit;

  switch (dialog_id.get_type()) {
    case DialogType::None:
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<SearchMessagesQuery>(std::move(promise))
          ->send(dialog_id, query, sender_user_id, std::move(input_user), from_message_id, offset, limit, filter_type,
                 random_id);
      break;
    case DialogType::SecretChat:
      if (filter_type == SearchMessagesFilter::UnreadMention) {
        promise.set_value(Unit());
      } else {
        promise.set_error(Status::Error(500, "Search messages in secret chats is not supported"));
      }
      break;
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Search messages is not supported"));
  }
  return result;
}

std::pair<int32, vector<FullMessageId>> MessagesManager::search_call_messages(MessageId from_message_id, int32 limit,
                                                                              bool only_missed, int64 &random_id,
                                                                              bool use_db, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_call_messages_.find(random_id);
    if (it != found_call_messages_.end()) {
      auto result = std::move(it->second);
      found_call_messages_.erase(it);
      promise.set_value(Unit());
      return result;
    }
    random_id = 0;
  }
  LOG(INFO) << "Search call messages from " << from_message_id << " with limit " << limit;

  std::pair<int32, vector<FullMessageId>> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  if (from_message_id.get() > MessageId::max().get()) {
    from_message_id = MessageId::max();
  }

  if (!from_message_id.is_valid() && from_message_id != MessageId()) {
    promise.set_error(Status::Error(3, "Parameter from_message_id must be identifier of the chat message or 0"));
    return result;
  }
  from_message_id = from_message_id.get_next_server_message_id();

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_call_messages_.find(random_id) != found_call_messages_.end());
  found_call_messages_[random_id];  // reserve place for result

  auto filter_type = only_missed ? SearchMessagesFilter::MissedCall : SearchMessagesFilter::Call;

  if (use_db && G()->parameters().use_message_db) {
    // Try to use database
    MessageId first_db_message_id =
        calls_db_state_.first_calls_database_message_id_by_index[search_calls_filter_index(filter_type)];
    int32 message_count = calls_db_state_.message_count_by_index[search_calls_filter_index(filter_type)];
    auto fixed_from_message_id = from_message_id;
    if (fixed_from_message_id == MessageId()) {
      fixed_from_message_id = MessageId::max();
    }
    CHECK(fixed_from_message_id.is_valid() && fixed_from_message_id.is_server());
    LOG(INFO) << "Search call messages from " << fixed_from_message_id << ", have up to " << first_db_message_id
              << ", message_count = " << message_count;
    if (first_db_message_id < fixed_from_message_id && message_count != -1) {
      LOG(INFO) << "Search messages in database from " << fixed_from_message_id << " and with limit " << limit;

      MessagesDbCallsQuery db_query;
      db_query.index_mask = search_messages_filter_index_mask(filter_type);
      db_query.from_unique_message_id = fixed_from_message_id.get_server_message_id().get();
      db_query.limit = limit;
      G()->td_db()->get_messages_db_async()->get_calls(
          db_query, PromiseCreator::lambda([random_id, first_db_message_id, filter_type, promise = std::move(promise)](
                                               Result<MessagesDbCallsResult> calls_result) mutable {
            send_closure(G()->messages_manager(), &MessagesManager::on_messages_db_calls_result,
                         std::move(calls_result), random_id, first_db_message_id, filter_type, std::move(promise));
          }));
      return result;
    }
  }

  LOG(DEBUG) << "Search call messages on server from " << from_message_id << " and with limit " << limit;
  td_->create_handler<SearchMessagesQuery>(std::move(promise))
      ->send(DialogId(), "", UserId(), nullptr, from_message_id, 0, limit, filter_type, random_id);
  return result;
}

std::pair<int32, vector<MessageId>> MessagesManager::search_dialog_recent_location_messages(DialogId dialog_id,
                                                                                            int32 limit,
                                                                                            int64 &random_id,
                                                                                            Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_recent_location_messages_.find(random_id);
    CHECK(it != found_dialog_recent_location_messages_.end());
    auto result = std::move(it->second);
    found_dialog_recent_location_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }
  LOG(INFO) << "Search recent location messages in " << dialog_id << " with limit " << limit;

  std::pair<int32, vector<MessageId>> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  const Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return result;
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 ||
           found_dialog_recent_location_messages_.find(random_id) != found_dialog_recent_location_messages_.end());
  found_dialog_recent_location_messages_[random_id];  // reserve place for result

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<GetRecentLocationsQuery>(std::move(promise))->send(dialog_id, limit, random_id);
      break;
    case DialogType::SecretChat:
      promise.set_value(Unit());
      break;
    default:
      UNREACHABLE();
      promise.set_error(Status::Error(500, "Search messages is not supported"));
  }
  return result;
}

vector<FullMessageId> MessagesManager::get_active_live_location_messages(Promise<Unit> &&promise) {
  if (!G()->parameters().use_message_db) {
    are_active_live_location_messages_loaded_ = true;
  }

  if (!are_active_live_location_messages_loaded_) {
    load_active_live_location_messages_queries_.push_back(std::move(promise));
    if (load_active_live_location_messages_queries_.size() == 1u) {
      LOG(INFO) << "Trying to load active live location messages from database";
      G()->td_db()->get_sqlite_pmc()->get(
          "di_active_live_location_messages", PromiseCreator::lambda([](string value) {
            send_closure(G()->messages_manager(),
                         &MessagesManager::on_load_active_live_location_full_message_ids_from_database,
                         std::move(value));
          }));
    }
    return {};
  }

  promise.set_value(Unit());
  vector<FullMessageId> result;
  for (auto &full_message_id : active_live_location_full_message_ids_) {
    auto m = get_message(full_message_id);
    CHECK(m != nullptr);
    CHECK(m->content->get_type() == MessageContentType::LiveLocation);
    CHECK(!m->message_id.is_scheduled());

    if (m->is_failed_to_send) {
      continue;
    }

    auto live_period = get_message_content_live_location_period(m->content.get());
    if (live_period <= G()->unix_time() - m->date) {  // bool is_expired flag?
      // live location is expired
      continue;
    }
    result.push_back(full_message_id);
  }

  return result;
}

void MessagesManager::on_load_active_live_location_full_message_ids_from_database(string value) {
  if (value.empty()) {
    LOG(INFO) << "Active live location messages aren't found in the database";
    on_load_active_live_location_messages_finished();

    if (!active_live_location_full_message_ids_.empty()) {
      save_active_live_locations();
    }
    return;
  }

  LOG(INFO) << "Successfully loaded active live location messages list of size " << value.size() << " from database";

  auto new_full_message_ids = std::move(active_live_location_full_message_ids_);
  vector<FullMessageId> old_full_message_ids;
  log_event_parse(old_full_message_ids, value).ensure();

  // TODO asynchronously load messages from database
  active_live_location_full_message_ids_.clear();
  for (auto full_message_id : old_full_message_ids) {
    Message *m = get_message_force(full_message_id, "on_load_active_live_location_full_message_ids_from_database");
    if (m != nullptr) {
      try_add_active_live_location(full_message_id.get_dialog_id(), m);
    }
  }

  for (auto full_message_id : new_full_message_ids) {
    add_active_live_location(full_message_id);
  }

  on_load_active_live_location_messages_finished();

  if (!new_full_message_ids.empty() || old_full_message_ids.size() != active_live_location_full_message_ids_.size()) {
    save_active_live_locations();
  }
}

void MessagesManager::on_load_active_live_location_messages_finished() {
  are_active_live_location_messages_loaded_ = true;
  auto promises = std::move(load_active_live_location_messages_queries_);
  load_active_live_location_messages_queries_.clear();
  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

void MessagesManager::try_add_active_live_location(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);

  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (m->content->get_type() != MessageContentType::LiveLocation || m->message_id.is_scheduled() ||
      m->message_id.is_local() || m->via_bot_user_id.is_valid() || m->forward_info != nullptr) {
    return;
  }

  auto live_period = get_message_content_live_location_period(m->content.get());
  if (live_period <= G()->unix_time() - m->date + 1) {  // bool is_expired flag?
    // live location is expired
    return;
  }
  add_active_live_location({dialog_id, m->message_id});
}

void MessagesManager::add_active_live_location(FullMessageId full_message_id) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }
  if (!active_live_location_full_message_ids_.insert(full_message_id).second) {
    return;
  }

  // TODO add timer for live location expiration

  if (!G()->parameters().use_message_db) {
    return;
  }

  if (are_active_live_location_messages_loaded_) {
    save_active_live_locations();
  } else if (load_active_live_location_messages_queries_.empty()) {
    // load active live locations and save after that
    get_active_live_location_messages(Auto());
  }
}

bool MessagesManager::delete_active_live_location(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  return active_live_location_full_message_ids_.erase(FullMessageId{dialog_id, m->message_id}) != 0;
}

void MessagesManager::save_active_live_locations() {
  CHECK(are_active_live_location_messages_loaded_);
  LOG(INFO) << "Save active live locations of size " << active_live_location_full_message_ids_.size() << " to database";
  if (G()->parameters().use_message_db) {
    G()->td_db()->get_sqlite_pmc()->set("di_active_live_location_messages",
                                        log_event_store(active_live_location_full_message_ids_).as_slice().str(),
                                        Auto());
  }
}

void MessagesManager::on_message_live_location_viewed(Dialog *d, const Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->content->get_type() == MessageContentType::LiveLocation);
  CHECK(!m->message_id.is_scheduled());

  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  switch (d->dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      // ok
      break;
    case DialogType::SecretChat:
      return;
    default:
      UNREACHABLE();
      return;
  }
  if (!d->is_opened) {
    return;
  }

  if (m->is_outgoing || !m->message_id.is_server() || m->via_bot_user_id.is_valid() || !m->sender_user_id.is_valid() ||
      td_->contacts_manager_->is_user_bot(m->sender_user_id) || m->forward_info != nullptr) {
    return;
  }

  auto live_period = get_message_content_live_location_period(m->content.get());
  if (live_period <= G()->unix_time() - m->date + 1) {
    // live location is expired
    return;
  }

  auto &live_location_task_id = d->pending_viewed_live_locations[m->message_id];
  if (live_location_task_id != 0) {
    return;
  }

  live_location_task_id = ++viewed_live_location_task_id_;
  auto &full_message_id = viewed_live_location_tasks_[live_location_task_id];
  full_message_id = FullMessageId(d->dialog_id, m->message_id);
  view_message_live_location_on_server_impl(live_location_task_id, full_message_id);
}

void MessagesManager::view_message_live_location_on_server(int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = viewed_live_location_tasks_.find(task_id);
  if (it == viewed_live_location_tasks_.end()) {
    return;
  }

  auto full_message_id = it->second;
  Dialog *d = get_dialog(full_message_id.get_dialog_id());
  const Message *m = get_message_force(d, full_message_id.get_message_id(), "view_message_live_location_on_server");
  if (m == nullptr || get_message_content_live_location_period(m->content.get()) <= G()->unix_time() - m->date + 1) {
    // the message was deleted or live location is expired
    viewed_live_location_tasks_.erase(it);
    auto erased_count = d->pending_viewed_live_locations.erase(full_message_id.get_message_id());
    CHECK(erased_count > 0);
    return;
  }

  view_message_live_location_on_server_impl(task_id, full_message_id);
}

void MessagesManager::view_message_live_location_on_server_impl(int64 task_id, FullMessageId full_message_id) {
  auto promise = PromiseCreator::lambda([actor_id = actor_id(this), task_id](Unit result) {
    send_closure(actor_id, &MessagesManager::on_message_live_location_viewed_on_server, task_id);
  });
  read_message_contents_on_server(full_message_id.get_dialog_id(), {full_message_id.get_message_id()}, 0,
                                  std::move(promise), true);
}

void MessagesManager::on_message_live_location_viewed_on_server(int64 task_id) {
  if (G()->close_flag()) {
    return;
  }

  auto it = viewed_live_location_tasks_.find(task_id);
  if (it == viewed_live_location_tasks_.end()) {
    return;
  }

  pending_message_live_location_view_timeout_.add_timeout_in(task_id, LIVE_LOCATION_VIEW_PERIOD);
}

FileSourceId MessagesManager::get_message_file_source_id(FullMessageId full_message_id) {
  auto dialog_id = full_message_id.get_dialog_id();
  auto message_id = full_message_id.get_message_id();
  if (!dialog_id.is_valid() || !(message_id.is_valid() || message_id.is_valid_scheduled()) ||
      dialog_id.get_type() == DialogType::SecretChat || !message_id.is_any_server()) {
    return FileSourceId();
  }

  auto &file_source_id = full_message_id_to_file_source_id_[full_message_id];
  if (!file_source_id.is_valid()) {
    file_source_id = td_->file_reference_manager_->create_message_file_source(full_message_id);
  }
  return file_source_id;
}

void MessagesManager::add_message_file_sources(DialogId dialog_id, const Message *m) {
  auto file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (file_ids.empty()) {
    return;
  }

  // do not create file_source_id for messages without file_ids
  auto file_source_id = get_message_file_source_id(FullMessageId(dialog_id, m->message_id));
  if (file_source_id.is_valid()) {
    for (auto file_id : file_ids) {
      td_->file_manager_->add_file_source(file_id, file_source_id);
    }
  }
}

void MessagesManager::remove_message_file_sources(DialogId dialog_id, const Message *m) {
  auto file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (file_ids.empty()) {
    return;
  }

  // do not create file_source_id for messages without file_ids
  auto file_source_id = get_message_file_source_id(FullMessageId(dialog_id, m->message_id));
  if (file_source_id.is_valid()) {
    for (auto file_id : file_ids) {
      td_->file_manager_->remove_file_source(file_id, file_source_id);
    }
  }
}

void MessagesManager::change_message_files(DialogId dialog_id, const Message *m, const vector<FileId> &old_file_ids) {
  auto new_file_ids = get_message_content_file_ids(m->content.get(), td_);
  if (new_file_ids == old_file_ids) {
    return;
  }

  FullMessageId full_message_id{dialog_id, m->message_id};
  if (need_delete_message_files(dialog_id, m)) {
    for (auto file_id : old_file_ids) {
      if (!td::contains(new_file_ids, file_id) && need_delete_file(full_message_id, file_id)) {
        send_closure(G()->file_manager(), &FileManager::delete_file, file_id, Promise<>(), "change_message_files");
      }
    }
  }

  auto file_source_id = get_message_file_source_id(full_message_id);
  if (file_source_id.is_valid()) {
    td_->file_manager_->change_files_source(file_source_id, old_file_ids, new_file_ids);
  }
}

MessageId MessagesManager::get_first_database_message_id_by_index(const Dialog *d, SearchMessagesFilter filter) {
  CHECK(d != nullptr);
  auto message_id = filter == SearchMessagesFilter::Empty
                        ? d->first_database_message_id
                        : d->first_database_message_id_by_index[search_messages_filter_index(filter)];
  CHECK(!message_id.is_scheduled());
  if (!message_id.is_valid()) {
    if (d->dialog_id.get_type() == DialogType::SecretChat) {
      LOG(ERROR) << "Invalid first_database_message_id_by_index in " << d->dialog_id;
      return MessageId::min();
    }
    return MessageId::max();
  }
  return message_id;
}

void MessagesManager::on_search_dialog_messages_db_result(int64 random_id, DialogId dialog_id,
                                                          MessageId from_message_id, MessageId first_db_message_id,
                                                          SearchMessagesFilter filter_type, int32 offset, int32 limit,
                                                          Result<std::vector<BufferSlice>> r_messages,
                                                          Promise<> promise) {
  if (r_messages.is_error()) {
    LOG(ERROR) << r_messages.error();
    if (first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat) {
      found_dialog_messages_.erase(random_id);
    }
    return promise.set_value(Unit());
  }
  CHECK(!from_message_id.is_scheduled());
  CHECK(!first_db_message_id.is_scheduled());

  auto messages = r_messages.move_as_ok();

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto it = found_dialog_messages_.find(random_id);
  CHECK(it != found_dialog_messages_.end());
  auto &res = it->second.second;

  res.reserve(messages.size());
  for (auto &message : messages) {
    auto m = on_get_message_from_database(dialog_id, d, message, false, "on_search_dialog_messages_db_result");
    if (m != nullptr && first_db_message_id <= m->message_id) {
      if (filter_type == SearchMessagesFilter::UnreadMention && !m->contains_unread_mention) {
        // skip already read by d->last_read_all_mentions_message_id mentions
      } else {
        CHECK(!m->message_id.is_scheduled());
        res.push_back(m->message_id);
      }
    }
  }

  auto &message_count = d->message_count_by_index[search_messages_filter_index(filter_type)];
  int32 result_size = narrow_cast<int32>(res.size());
  bool from_the_end =
      from_message_id == MessageId::max() || (offset < 0 && (result_size == 0 || res[0] < from_message_id));
  if (message_count < result_size || (message_count > result_size && from_the_end &&
                                      first_db_message_id == MessageId::min() && result_size < limit + offset)) {
    LOG(INFO) << "Fix found message count in " << dialog_id << " from " << message_count << " to " << result_size;
    message_count = result_size;
    if (filter_type == SearchMessagesFilter::UnreadMention) {
      d->unread_mention_count = message_count;
      update_dialog_mention_notification_count(d);
      send_update_chat_unread_mention_count(d);
    }
    on_dialog_updated(dialog_id, "on_search_dialog_messages_db_result");
  }
  it->second.first = message_count;
  if (res.empty() && first_db_message_id != MessageId::min() && dialog_id.get_type() != DialogType::SecretChat) {
    LOG(INFO) << "No messages in database found";
    found_dialog_messages_.erase(it);
  } else {
    LOG(INFO) << "Found " << res.size() << " messages out of " << message_count << " in database";
  }
  promise.set_value(Unit());
}

std::pair<int64, vector<FullMessageId>> MessagesManager::offline_search_messages(
    DialogId dialog_id, const string &query, int64 from_search_id, int32 limit,
    const tl_object_ptr<td_api::SearchMessagesFilter> &filter, int64 &random_id, Promise<> &&promise) {
  if (!G()->parameters().use_message_db) {
    promise.set_error(Status::Error(400, "Message database is required to search messages in secret chats"));
    return {};
  }

  if (random_id != 0) {
    // request has already been sent before
    auto it = found_fts_messages_.find(random_id);
    CHECK(it != found_fts_messages_.end());
    auto result = std::move(it->second);
    found_fts_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return {};
  }
  if (dialog_id != DialogId() && !have_dialog_force(dialog_id)) {
    promise.set_error(Status::Error(400, "Chat not found"));
    return {};
  }
  if (limit <= 0) {
    promise.set_error(Status::Error(400, "Limit must be positive"));
    return {};
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  MessagesDbFtsQuery fts_query;
  fts_query.query = query;
  fts_query.dialog_id = dialog_id;
  fts_query.index_mask = search_messages_filter_index_mask(get_search_messages_filter(filter));
  fts_query.from_search_id = from_search_id;
  fts_query.limit = limit;

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_fts_messages_.find(random_id) != found_fts_messages_.end());
  found_fts_messages_[random_id];  // reserve place for result

  G()->td_db()->get_messages_db_async()->get_messages_fts(
      std::move(fts_query),
      PromiseCreator::lambda([random_id, promise = std::move(promise)](Result<MessagesDbFtsResult> fts_result) mutable {
        send_closure(G()->messages_manager(), &MessagesManager::on_messages_db_fts_result, std::move(fts_result),
                     random_id, std::move(promise));
      }));

  return {};
}

void MessagesManager::on_messages_db_fts_result(Result<MessagesDbFtsResult> result, int64 random_id,
                                                Promise<> &&promise) {
  if (result.is_error()) {
    found_fts_messages_.erase(random_id);
    return promise.set_error(result.move_as_error());
  }
  auto fts_result = result.move_as_ok();

  auto it = found_fts_messages_.find(random_id);
  CHECK(it != found_fts_messages_.end());
  auto &res = it->second.second;

  res.reserve(fts_result.messages.size());
  for (auto &message : fts_result.messages) {
    auto m = on_get_message_from_database(message.dialog_id, get_dialog_force(message.dialog_id), message.data, false,
                                          "on_messages_db_fts_result");
    if (m != nullptr) {
      res.push_back(FullMessageId(message.dialog_id, m->message_id));
    }
  }

  it->second.first = fts_result.next_search_id;

  promise.set_value(Unit());
}

void MessagesManager::on_messages_db_calls_result(Result<MessagesDbCallsResult> result, int64 random_id,
                                                  MessageId first_db_message_id, SearchMessagesFilter filter,
                                                  Promise<> &&promise) {
  CHECK(!first_db_message_id.is_scheduled());
  if (result.is_error()) {
    found_call_messages_.erase(random_id);
    return promise.set_error(result.move_as_error());
  }
  auto calls_result = result.move_as_ok();

  auto it = found_call_messages_.find(random_id);
  CHECK(it != found_call_messages_.end());
  auto &res = it->second.second;

  res.reserve(calls_result.messages.size());
  for (auto &message : calls_result.messages) {
    auto m = on_get_message_from_database(message.dialog_id, get_dialog_force(message.dialog_id), message.data, false,
                                          "on_messages_db_calls_result");

    if (m != nullptr && first_db_message_id <= m->message_id) {
      res.push_back(FullMessageId(message.dialog_id, m->message_id));
    }
  }
  it->second.first = calls_db_state_.message_count_by_index[search_calls_filter_index(filter)];

  if (res.empty() && first_db_message_id != MessageId::min()) {
    LOG(INFO) << "No messages in database found";
    found_call_messages_.erase(it);
  }

  promise.set_value(Unit());
}

std::pair<int32, vector<FullMessageId>> MessagesManager::search_messages(FolderId folder_id, bool ignore_folder_id,
                                                                         const string &query, int32 offset_date,
                                                                         DialogId offset_dialog_id,
                                                                         MessageId offset_message_id, int32 limit,
                                                                         int64 &random_id, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_messages_.find(random_id);
    CHECK(it != found_messages_.end());
    auto result = std::move(it->second);
    found_messages_.erase(it);
    promise.set_value(Unit());
    return result;
  }

  std::pair<int32, vector<FullMessageId>> result;
  if (limit <= 0) {
    promise.set_error(Status::Error(3, "Parameter limit must be positive"));
    return result;
  }
  if (limit > MAX_SEARCH_MESSAGES) {
    limit = MAX_SEARCH_MESSAGES;
  }

  if (offset_date <= 0) {
    offset_date = std::numeric_limits<int32>::max();
  }
  if (!offset_message_id.is_valid()) {
    if (offset_message_id.is_valid_scheduled()) {
      promise.set_error(Status::Error(3, "Parameter offset_message_id can't be a scheduled message identifier"));
      return result;
    }
    offset_message_id = MessageId();
  }
  if (offset_message_id != MessageId() && !offset_message_id.is_server()) {
    promise.set_error(
        Status::Error(3, "Parameter offset_message_id must be identifier of the last found message or 0"));
    return result;
  }

  if (query.empty()) {
    promise.set_value(Unit());
    return result;
  }

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_messages_.find(random_id) != found_messages_.end());
  found_messages_[random_id];  // reserve place for result

  LOG(DEBUG) << "Search messages globally with query = \"" << query << "\" from date " << offset_date << ", "
             << offset_dialog_id << ", " << offset_message_id << " and with limit " << limit;

  td_->create_handler<SearchMessagesGlobalQuery>(std::move(promise))
      ->send(folder_id, ignore_folder_id, query, offset_date, offset_dialog_id, offset_message_id, limit, random_id);
  return result;
}

int64 MessagesManager::get_dialog_message_by_date(DialogId dialog_id, int32 date, Promise<Unit> &&promise) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(5, "Chat not found"));
    return 0;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(5, "Can't access the chat"));
    return 0;
  }

  if (date <= 0) {
    date = 1;
  }

  int64 random_id = 0;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 ||
           get_dialog_message_by_date_results_.find(random_id) != get_dialog_message_by_date_results_.end());
  get_dialog_message_by_date_results_[random_id];  // reserve place for result

  auto message_id = find_message_by_date(d->messages.get(), date);
  if (message_id.is_valid() && (message_id == d->last_message_id || get_message(d, message_id)->have_next)) {
    get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
    promise.set_value(Unit());
    return random_id;
  }

  if (G()->parameters().use_message_db && d->last_database_message_id != MessageId()) {
    CHECK(d->first_database_message_id != MessageId());
    G()->td_db()->get_messages_db_async()->get_dialog_message_by_date(
        dialog_id, d->first_database_message_id, d->last_database_message_id, date,
        PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, date, random_id,
                                promise = std::move(promise)](Result<BufferSlice> result) mutable {
          send_closure(actor_id, &MessagesManager::on_get_dialog_message_by_date_from_database, dialog_id, date,
                       random_id, std::move(result), std::move(promise));
        }));
  } else {
    get_dialog_message_by_date_from_server(d, date, random_id, false, std::move(promise));
  }
  return random_id;
}

MessageId MessagesManager::find_message_by_date(const Message *m, int32 date) {
  if (m == nullptr) {
    return MessageId();
  }

  if (m->date > date) {
    return find_message_by_date(m->left.get(), date);
  }

  auto message_id = find_message_by_date(m->right.get(), date);
  if (message_id.is_valid()) {
    return message_id;
  }

  return m->message_id;
}

void MessagesManager::on_get_dialog_message_by_date_from_database(DialogId dialog_id, int32 date, int64 random_id,
                                                                  Result<BufferSlice> result, Promise<Unit> promise) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (result.is_ok()) {
    Message *m =
        on_get_message_from_database(dialog_id, d, result.ok(), false, "on_get_dialog_message_by_date_from_database");
    if (m != nullptr) {
      auto message_id = find_message_by_date(d->messages.get(), date);
      if (!message_id.is_valid()) {
        LOG(ERROR) << "Failed to find " << m->message_id << " in " << dialog_id << " by date " << date;
        message_id = m->message_id;
      }
      get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
      promise.set_value(Unit());
      return;
    }
    // TODO if m == nullptr, we need to just adjust it to the next non-nullptr message, not get from server
  }

  return get_dialog_message_by_date_from_server(d, date, random_id, true, std::move(promise));
}

void MessagesManager::get_dialog_message_by_date_from_server(const Dialog *d, int32 date, int64 random_id,
                                                             bool after_database_search, Promise<Unit> &&promise) {
  CHECK(d != nullptr);
  if (d->have_full_history) {
    // request can be always done locally/in memory. There is no need to send request to the server
    if (after_database_search) {
      return promise.set_value(Unit());
    }

    auto message_id = find_message_by_date(d->messages.get(), date);
    if (message_id.is_valid()) {
      get_dialog_message_by_date_results_[random_id] = {d->dialog_id, message_id};
    }
    promise.set_value(Unit());
    return;
  }
  if (d->dialog_id.get_type() == DialogType::SecretChat) {
    // there is no way to send request to the server
    return promise.set_value(Unit());
  }

  td_->create_handler<GetDialogMessageByDateQuery>(std::move(promise))->send(d->dialog_id, date, random_id);
}

void MessagesManager::on_get_dialog_message_by_date_success(DialogId dialog_id, int32 date, int64 random_id,
                                                            vector<tl_object_ptr<telegram_api::Message>> &&messages) {
  auto it = get_dialog_message_by_date_results_.find(random_id);
  CHECK(it != get_dialog_message_by_date_results_.end());
  auto &result = it->second;
  CHECK(result == FullMessageId());

  for (auto &message : messages) {
    auto message_date = get_message_date(message);
    auto message_dialog_id = get_message_dialog_id(message);
    if (message_dialog_id != dialog_id) {
      LOG(ERROR) << "Receive message in wrong " << message_dialog_id << " instead of " << dialog_id;
      continue;
    }
    if (message_date != 0 && message_date <= date) {
      result = on_get_message(std::move(message), false, dialog_id.get_type() == DialogType::Channel, false, false,
                              false, "on_get_dialog_message_by_date_success");
      if (result != FullMessageId()) {
        const Dialog *d = get_dialog(dialog_id);
        CHECK(d != nullptr);
        auto message_id = find_message_by_date(d->messages.get(), date);
        if (!message_id.is_valid()) {
          LOG(ERROR) << "Failed to find " << result.get_message_id() << " in " << dialog_id << " by date " << date;
          message_id = result.get_message_id();
        }
        get_dialog_message_by_date_results_[random_id] = {dialog_id, message_id};
        // TODO result must be adjusted by local messages
        return;
      }
    }
  }
}

void MessagesManager::on_get_dialog_message_by_date_fail(int64 random_id) {
  auto erased = get_dialog_message_by_date_results_.erase(random_id);
  CHECK(erased > 0);
}

tl_object_ptr<td_api::message> MessagesManager::get_dialog_message_by_date_object(int64 random_id) {
  auto it = get_dialog_message_by_date_results_.find(random_id);
  CHECK(it != get_dialog_message_by_date_results_.end());
  auto full_message_id = std::move(it->second);
  get_dialog_message_by_date_results_.erase(it);
  return get_message_object(full_message_id);
}

int32 MessagesManager::get_dialog_message_count(DialogId dialog_id,
                                                const tl_object_ptr<td_api::SearchMessagesFilter> &filter,
                                                bool return_local, int64 &random_id, Promise<Unit> &&promise) {
  if (random_id != 0) {
    // request has already been sent before
    auto it = found_dialog_messages_.find(random_id);
    CHECK(it != found_dialog_messages_.end());
    auto result = std::move(it->second);
    found_dialog_messages_.erase(it);
    promise.set_value(Unit());
    return result.first;
  }

  LOG(INFO) << "Get " << (return_local ? "local " : "") << "number of messages in " << dialog_id << " filtered by "
            << to_string(filter);

  const Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return -1;
  }

  auto filter_type = get_search_messages_filter(filter);
  if (filter_type == SearchMessagesFilter::Empty) {
    promise.set_error(Status::Error(6, "SearchMessagesFilterEmpty is not supported"));
    return -1;
  }

  auto dialog_type = dialog_id.get_type();
  int32 message_count = d->message_count_by_index[search_messages_filter_index(filter_type)];
  if (message_count == -1) {
    if (filter_type == SearchMessagesFilter::UnreadMention) {
      message_count = d->unread_mention_count;
    }
  }
  if (message_count != -1 || return_local || dialog_type == DialogType::SecretChat) {
    promise.set_value(Unit());
    return message_count;
  }

  LOG(INFO) << "Get number of messages in " << dialog_id << " filtered by " << to_string(filter) << " from the server";

  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || found_dialog_messages_.find(random_id) != found_dialog_messages_.end());
  found_dialog_messages_[random_id];  // reserve place for result

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::Channel:
      td_->create_handler<SearchMessagesQuery>(std::move(promise))
          ->send(dialog_id, "", UserId(), nullptr, MessageId(), 0, 1, filter_type, random_id);
      break;
    case DialogType::None:
    case DialogType::SecretChat:
    default:
      UNREACHABLE();
  }
  return -1;
}

void MessagesManager::preload_newer_messages(const Dialog *d, MessageId max_message_id) {
  CHECK(d != nullptr);
  CHECK(max_message_id.is_valid());
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  MessagesConstIterator p(d, max_message_id);
  int32 limit = MAX_GET_HISTORY * 3 / 10;
  while (*p != nullptr && limit-- > 0) {
    ++p;
    if (*p) {
      max_message_id = (*p)->message_id;
    }
  }
  if (limit > 0 && (d->last_message_id == MessageId() || max_message_id < d->last_message_id)) {
    // need to preload some new messages
    LOG(INFO) << "Preloading newer after " << max_message_id;
    load_messages(d->dialog_id, max_message_id, -MAX_GET_HISTORY + 1, MAX_GET_HISTORY, 3, false, Promise<Unit>());
  }
}

void MessagesManager::preload_older_messages(const Dialog *d, MessageId min_message_id) {
  CHECK(d != nullptr);
  CHECK(min_message_id.is_valid());
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  /*
    if (d->first_remote_message_id == -1) {
      // nothing left to preload from server
      return;
    }
  */
  MessagesConstIterator p(d, min_message_id);
  int32 limit = MAX_GET_HISTORY * 3 / 10 + 1;
  while (*p != nullptr && limit-- > 0) {
    min_message_id = (*p)->message_id;
    --p;
  }
  if (limit > 0) {
    // need to preload some old messages
    LOG(INFO) << "Preloading older before " << min_message_id;
    load_messages(d->dialog_id, min_message_id, 0, MAX_GET_HISTORY / 2, 3, false, Promise<Unit>());
  }
}

unique_ptr<MessagesManager::Message> MessagesManager::parse_message(DialogId dialog_id, const BufferSlice &value,
                                                                    bool is_scheduled) {
  LOG(INFO) << "Loaded message of size " << value.size() << " from database";
  auto m = make_unique<Message>();

  auto status = log_event_parse(*m, value.as_slice());
  bool is_message_id_valid = is_scheduled ? m->message_id.is_valid_scheduled() : m->message_id.is_valid();
  if (status.is_error() || !is_message_id_valid) {
    // can't happen unless database is broken, but has been seen in the wild
    LOG(ERROR) << "Receive invalid message from database: " << m->message_id << ' ' << status << ' '
               << format::as_hex_dump<4>(value.as_slice());
    if (!is_scheduled && dialog_id.get_type() != DialogType::SecretChat && m->message_id.is_valid() &&
        m->message_id.is_server()) {
      // trying to repair the message
      get_message_from_server({dialog_id, m->message_id}, Auto());
    }
    return nullptr;
  }

  return m;
}

void MessagesManager::on_get_history_from_database(DialogId dialog_id, MessageId from_message_id, int32 offset,
                                                   int32 limit, bool from_the_end, bool only_local,
                                                   vector<BufferSlice> &&messages, Promise<Unit> &&promise) {
  CHECK(-limit < offset && offset <= 0);
  CHECK(offset < 0 || from_the_end);
  CHECK(!from_message_id.is_scheduled());

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    LOG(WARNING) << "Ignore result of get_history_from_database in " << dialog_id;
    promise.set_value(Unit());
    return;
  }

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  LOG(INFO) << "Receive " << messages.size() << " history messages from database "
            << (from_the_end ? "from the end " : "") << "in " << dialog_id << " from " << from_message_id
            << " with offset " << offset << " and limit " << limit << ". First database message is "
            << d->first_database_message_id << ", have_full_history = " << d->have_full_history;

  if (messages.empty() && from_the_end && d->messages == nullptr) {
    if (d->have_full_history) {
      set_dialog_is_empty(d, "on_get_history_from_database empty");
    } else if (d->last_database_message_id.is_valid()) {
      set_dialog_first_database_message_id(d, MessageId(), "on_get_history_from_database empty");
      set_dialog_last_database_message_id(d, MessageId(), "on_get_history_from_database empty");
    }
  }

  bool have_next = false;
  bool need_update = false;
  bool need_update_dialog_pos = false;
  bool added_new_message = false;
  MessageId last_added_message_id;
  Message *next_message = nullptr;
  Dependencies dependencies;
  bool is_first = true;
  bool had_full_history = d->have_full_history;
  auto debug_first_database_message_id = d->first_database_message_id;
  auto debug_last_message_id = d->last_message_id;
  auto debug_last_new_message_id = d->last_new_message_id;
  auto debug_cur_message_id = MessageId::max();
  size_t pos = 0;
  for (auto &message_slice : messages) {
    if (!d->first_database_message_id.is_valid() && !d->have_full_history) {
      break;
    }
    auto message = parse_message(dialog_id, std::move(message_slice), false);
    if (message == nullptr) {
      if (d->have_full_history) {
        d->have_full_history = false;
        on_dialog_updated(dialog_id, "drop have_full_history in on_get_history_from_database");
      }
      break;
    }
    if (message->message_id >= debug_cur_message_id) {
      // TODO move to ERROR
      LOG(FATAL) << "Receive message " << message->message_id << " after " << debug_cur_message_id
                 << " from database in the history of " << dialog_id << " from " << from_message_id << " with offset "
                 << offset << ", limit " << limit << ", from_the_end = " << from_the_end;
      break;
    }
    debug_cur_message_id = message->message_id;

    if (message->message_id < d->first_database_message_id) {
      if (d->have_full_history) {
        LOG(ERROR) << "Have full history in the " << dialog_id << " and receive " << message->message_id
                   << " from database, but first database message is " << d->first_database_message_id;
      } else {
        break;
      }
    }
    if (!have_next && (from_the_end || (is_first && offset < -1 && message->message_id <= from_message_id)) &&
        message->message_id < d->last_message_id) {
      // last message in the dialog must be attached to the next local message
      have_next = true;
    }

    message->have_previous = false;
    message->have_next = have_next;
    message->from_database = true;

    auto old_message = get_message(d, message->message_id);
    Message *m = old_message ? old_message
                             : add_message_to_dialog(d, std::move(message), false, &need_update,
                                                     &need_update_dialog_pos, "on_get_history_from_database");
    if (m != nullptr) {
      if (!have_next) {
        last_added_message_id = m->message_id;
      }
      if (old_message == nullptr) {
        add_message_dependencies(dependencies, dialog_id, m);
        added_new_message = true;
      } else if (m->message_id != from_message_id) {
        added_new_message = true;
      }
      if (next_message != nullptr && !next_message->have_previous) {
        LOG_CHECK(m->message_id < next_message->message_id)
            << m->message_id << ' ' << next_message->message_id << ' ' << debug_cur_message_id << ' ' << dialog_id
            << ' ' << from_message_id << ' ' << offset << ' ' << limit << ' ' << from_the_end << ' ' << only_local
            << ' ' << messages.size() << ' ' << debug_first_database_message_id << ' ' << last_added_message_id << ' '
            << added_new_message << ' ' << pos << ' ' << m << ' ' << next_message << ' ' << old_message << ' '
            << to_string(get_message_object(dialog_id, m)) << to_string(get_message_object(dialog_id, next_message));
        LOG(INFO) << "Fix have_previous for " << next_message->message_id;
        next_message->have_previous = true;
        attach_message_to_previous(
            d, next_message->message_id,
            (PSLICE() << "on_get_history_from_database 1 " << m->message_id << ' ' << from_message_id << ' ' << offset
                      << ' ' << limit << ' ' << d->first_database_message_id << ' ' << d->have_full_history << ' '
                      << pos)
                .c_str());
      }

      have_next = true;
      next_message = m;
    }
    is_first = false;
    pos++;
  }
  resolve_dependencies_force(td_, dependencies);

  if (!added_new_message && !only_local && dialog_id.get_type() != DialogType::SecretChat) {
    if (from_the_end) {
      from_message_id = MessageId();
    }
    load_messages(dialog_id, from_message_id, offset, limit, 1, false, std::move(promise));
    return;
  }

  if (from_the_end && last_added_message_id.is_valid()) {
    // CHECK(d->first_database_message_id.is_valid());
    // CHECK(last_added_message_id >= d->first_database_message_id);
    if ((had_full_history || d->have_full_history) && !d->last_new_message_id.is_valid() &&
        (last_added_message_id.is_server() || d->dialog_id.get_type() == DialogType::SecretChat)) {
      LOG(ERROR) << "Trying to hard fix " << d->dialog_id << " last new message to " << last_added_message_id
                 << " from on_get_history_from_database 2";
      d->last_new_message_id = last_added_message_id;
      on_dialog_updated(d->dialog_id, "on_get_history_from_database 3");
    }
    if (last_added_message_id > d->last_message_id && d->last_new_message_id.is_valid()) {
      set_dialog_last_message_id(d, last_added_message_id, "on_get_history_from_database 4");
      need_update_dialog_pos = true;
    }
    if (last_added_message_id != d->last_database_message_id && d->last_new_message_id.is_valid()) {
      auto debug_last_database_message_id = d->last_database_message_id;
      set_dialog_last_database_message_id(d, last_added_message_id, "on_get_history_from_database 5");
      if (last_added_message_id < d->first_database_message_id || !d->first_database_message_id.is_valid()) {
        CHECK(next_message != nullptr);
        LOG_CHECK(had_full_history || d->have_full_history)
            << had_full_history << ' ' << d->have_full_history << ' ' << next_message->message_id << ' '
            << last_added_message_id << ' ' << d->first_database_message_id << ' ' << debug_first_database_message_id
            << ' ' << d->last_database_message_id << ' ' << debug_last_database_message_id << ' ' << dialog_id << ' '
            << d->last_new_message_id << ' ' << debug_last_new_message_id << ' ' << d->last_message_id << ' '
            << debug_last_message_id;
        CHECK(next_message->message_id <= d->last_database_message_id);
        LOG(ERROR) << "Fix first database message in " << dialog_id << " from " << d->first_database_message_id
                   << " to " << next_message->message_id;
        set_dialog_first_database_message_id(d, next_message->message_id, "on_get_history_from_database 6");
      }
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_get_history_from_database 7");
  }

  promise.set_value(Unit());
}

void MessagesManager::get_history_from_the_end(DialogId dialog_id, bool from_database, bool only_local,
                                               Promise<Unit> &&promise) {
  CHECK(dialog_id.is_valid());
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // can't get history in dialogs without read access
    return promise.set_value(Unit());
  }
  if (G()->close_flag()) {
    return promise.set_error(Status::Error(500, "Request aborted"));
  }
  const int32 limit = MAX_GET_HISTORY;
  if (from_database && G()->parameters().use_message_db) {
    LOG(INFO) << "Get history from the end of " << dialog_id << " from database";
    MessagesDbMessagesQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.from_message_id = MessageId::max();
    db_query.limit = limit;
    G()->td_db()->get_messages_db_async()->get_messages(
        db_query, PromiseCreator::lambda([dialog_id, only_local, limit, actor_id = actor_id(this),
                                          promise = std::move(promise)](std::vector<BufferSlice> messages) mutable {
          send_closure(actor_id, &MessagesManager::on_get_history_from_database, dialog_id, MessageId::max(), 0, limit,
                       true, only_local, std::move(messages), std::move(promise));
        }));
  } else {
    if (only_local || dialog_id.get_type() == DialogType::SecretChat) {
      promise.set_value(Unit());
      return;
    }

    LOG(INFO) << "Get history from the end of " << dialog_id << " from server";
    td_->create_handler<GetHistoryQuery>(std::move(promise))->send_get_from_the_end(dialog_id, limit);
  }
}

void MessagesManager::get_history(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit,
                                  bool from_database, bool only_local, Promise<Unit> &&promise) {
  CHECK(dialog_id.is_valid());
  CHECK(from_message_id.is_valid());
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // can't get history in dialogs without read access
    return promise.set_value(Unit());
  }
  if (G()->close_flag()) {
    return promise.set_error(Status::Error(500, "Request aborted"));
  }
  if (from_database && G()->parameters().use_message_db) {
    LOG(INFO) << "Get history in " << dialog_id << " from " << from_message_id << " with offset " << offset
              << " and limit " << limit << " from database";
    MessagesDbMessagesQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.from_message_id = from_message_id;
    db_query.offset = offset;
    db_query.limit = limit;
    G()->td_db()->get_messages_db_async()->get_messages(
        db_query,
        PromiseCreator::lambda([dialog_id, from_message_id, offset, limit, only_local, actor_id = actor_id(this),
                                promise = std::move(promise)](std::vector<BufferSlice> messages) mutable {
          send_closure(actor_id, &MessagesManager::on_get_history_from_database, dialog_id, from_message_id, offset,
                       limit, false, only_local, std::move(messages), std::move(promise));
        }));
  } else {
    if (only_local || dialog_id.get_type() == DialogType::SecretChat) {
      return promise.set_value(Unit());
    }

    LOG(INFO) << "Get history in " << dialog_id << " from " << from_message_id << " with offset " << offset
              << " and limit " << limit << " from server";
    td_->create_handler<GetHistoryQuery>(std::move(promise))
        ->send(dialog_id, from_message_id.get_next_server_message_id(), offset, limit);
  }
}

void MessagesManager::load_messages(DialogId dialog_id, MessageId from_message_id, int32 offset, int32 limit,
                                    int left_tries, bool only_local, Promise<Unit> &&promise) {
  LOG(INFO) << "Load " << (only_local ? "local " : "") << "messages in " << dialog_id << " from " << from_message_id
            << " with offset = " << offset << " and limit = " << limit << ". " << left_tries << " tries left";
  CHECK(offset <= 0);
  CHECK(left_tries > 0);
  only_local |= dialog_id.get_type() == DialogType::SecretChat;
  if (!only_local) {
    Dialog *d = get_dialog(dialog_id);
    if (d != nullptr && d->have_full_history) {
      LOG(INFO) << "Have full history in " << dialog_id << ", so don't need to get chat history from server";
      only_local = true;
    }
  }
  bool from_database = (left_tries > 2 || only_local) && G()->parameters().use_message_db;
  // TODO do not send requests to database if (from_message_id < d->first_database_message_id ||
  // !d->first_database_message_id.is_valid()) && !d->have_full_history

  if (from_message_id == MessageId()) {
    get_history_from_the_end(dialog_id, from_database, only_local, std::move(promise));
    return;
  }
  if (offset >= -1) {
    // get history before some server or local message
    limit = min(max(limit + offset + 1, MAX_GET_HISTORY / 2), MAX_GET_HISTORY);
    offset = -1;
  } else {
    // get history around some server or local message
    int32 messages_to_load = max(MAX_GET_HISTORY, limit);
    int32 max_add = max(messages_to_load - limit - 2, 0);
    offset -= max_add;
    limit = MAX_GET_HISTORY;
  }
  get_history(dialog_id, from_message_id, offset, limit, from_database, only_local, std::move(promise));
}

vector<MessageId> MessagesManager::get_dialog_scheduled_messages(DialogId dialog_id, Promise<Unit> &&promise) {
  if (G()->close_flag()) {
    promise.set_error(Status::Error(500, "Request aborted"));
    return {};
  }

  const Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(6, "Chat not found"));
    return {};
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(5, "Can't access the chat"));
    return {};
  }
  if (is_broadcast_channel(dialog_id) &&
      !td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id()).can_post_messages()) {
    promise.set_error(Status::Error(3, "Not enough rights to get scheduled messages"));
    return {};
  }

  if (!d->has_loaded_scheduled_messages_from_database) {
    load_dialog_scheduled_messages(dialog_id, true, 0, std::move(promise));
    return {};
  }

  vector<MessageId> message_ids;
  find_old_messages(d->scheduled_messages.get(),
                    MessageId(ScheduledServerMessageId(), std::numeric_limits<int32>::max(), true), message_ids);
  std::reverse(message_ids.begin(), message_ids.end());

  if (G()->parameters().use_message_db) {
    bool has_scheduled_database_messages = false;
    for (auto &message_id : message_ids) {
      CHECK(message_id.is_valid_scheduled());
      if (!message_id.is_yet_unsent()) {
        has_scheduled_database_messages = true;
        break;
      }
    }
    set_dialog_has_scheduled_database_messages(d->dialog_id, has_scheduled_database_messages);
  }

  if (d->scheduled_messages_sync_generation != scheduled_messages_sync_generation_) {
    vector<uint32> numbers;
    for (auto &message_id : message_ids) {
      if (!message_id.is_scheduled_server()) {
        continue;
      }

      numbers.push_back(message_id.get_scheduled_server_message_id().get());
      const Message *m = get_message(d, message_id);
      CHECK(m != nullptr);
      CHECK(m->message_id.get_scheduled_server_message_id() == message_id.get_scheduled_server_message_id());
      numbers.push_back(m->edit_date);
      numbers.push_back(m->date);
    }
    auto hash = get_vector_hash(numbers);

    if (d->has_scheduled_server_messages ||
        (d->scheduled_messages_sync_generation == 0 && !G()->parameters().use_message_db)) {
      load_dialog_scheduled_messages(dialog_id, false, hash, std::move(promise));
      return {};
    }
    load_dialog_scheduled_messages(dialog_id, false, hash, Promise<Unit>());
  }

  promise.set_value(Unit());
  return message_ids;
}

void MessagesManager::load_dialog_scheduled_messages(DialogId dialog_id, bool from_database, int32 hash,
                                                     Promise<Unit> &&promise) {
  if (G()->parameters().use_message_db && from_database) {
    LOG(INFO) << "Load scheduled messages from database in " << dialog_id;
    auto &queries = load_scheduled_messages_from_database_queries_[dialog_id];
    queries.push_back(std::move(promise));
    if (queries.size() == 1) {
      G()->td_db()->get_messages_db_async()->get_scheduled_messages(
          dialog_id, 1000,
          PromiseCreator::lambda([dialog_id, actor_id = actor_id(this)](std::vector<BufferSlice> messages) {
            send_closure(actor_id, &MessagesManager::on_get_scheduled_messages_from_database, dialog_id,
                         std::move(messages));
          }));
    }
  } else {
    td_->create_handler<GetAllScheduledMessagesQuery>(std::move(promise))
        ->send(dialog_id, hash, scheduled_messages_sync_generation_);
  }
}

void MessagesManager::on_get_scheduled_messages_from_database(DialogId dialog_id, vector<BufferSlice> &&messages) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  d->has_loaded_scheduled_messages_from_database = true;

  LOG(INFO) << "Receive " << messages.size() << " scheduled messages from database in " << dialog_id;

  Dependencies dependencies;
  vector<MessageId> added_message_ids;
  for (auto &message_slice : messages) {
    auto message = parse_message(dialog_id, std::move(message_slice), true);
    if (message == nullptr) {
      continue;
    }
    message->from_database = true;

    if (get_message(d, message->message_id) != nullptr) {
      continue;
    }

    bool need_update = false;
    Message *m = add_scheduled_message_to_dialog(d, std::move(message), false, &need_update,
                                                 "on_get_scheduled_messages_from_database");
    if (m != nullptr) {
      add_message_dependencies(dependencies, dialog_id, m);
      added_message_ids.push_back(m->message_id);
    }
  }
  resolve_dependencies_force(td_, dependencies);

  for (auto message_id : added_message_ids) {
    send_update_new_message(d, get_message(d, message_id));
  }
  send_update_chat_has_scheduled_messages(d);

  auto it = load_scheduled_messages_from_database_queries_.find(dialog_id);
  CHECK(it != load_scheduled_messages_from_database_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  load_scheduled_messages_from_database_queries_.erase(it);

  for (auto &promise : promises) {
    promise.set_value(Unit());
  }
}

Result<int32> MessagesManager::get_message_schedule_date(
    td_api::object_ptr<td_api::MessageSchedulingState> &&scheduling_state) {
  if (scheduling_state == nullptr) {
    return 0;
  }

  switch (scheduling_state->get_id()) {
    case td_api::messageSchedulingStateSendWhenOnline::ID: {
      auto send_date = SCHEDULE_WHEN_ONLINE_DATE;
      return send_date;
    }
    case td_api::messageSchedulingStateSendAtDate::ID: {
      auto send_at_date = td_api::move_object_as<td_api::messageSchedulingStateSendAtDate>(scheduling_state);
      auto send_date = send_at_date->send_date_;
      if (send_date <= 0) {
        return Status::Error(400, "Invalid send date specified");
      }
      if (send_date <= G()->unix_time() + 10) {
        return 0;
      }
      if (send_date - G()->unix_time() > 367 * 86400) {
        return Status::Error(400, "Send date is too far in the future");
      }
      return send_date;
    }
    default:
      UNREACHABLE();
      return 0;
  }
}

tl_object_ptr<td_api::MessageSendingState> MessagesManager::get_message_sending_state_object(const Message *m) const {
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent()) {
    return td_api::make_object<td_api::messageSendingStatePending>();
  }
  if (m->is_failed_to_send) {
    return td_api::make_object<td_api::messageSendingStateFailed>(
        m->send_error_code, m->send_error_message, can_resend_message(m), max(m->try_resend_at - Time::now(), 0.0));
  }
  return nullptr;
}

tl_object_ptr<td_api::MessageSchedulingState> MessagesManager::get_message_scheduling_state_object(int32 send_date) {
  if (send_date == SCHEDULE_WHEN_ONLINE_DATE) {
    return td_api::make_object<td_api::messageSchedulingStateSendWhenOnline>();
  }
  return td_api::make_object<td_api::messageSchedulingStateSendAtDate>(send_date);
}

tl_object_ptr<td_api::message> MessagesManager::get_message_object(FullMessageId full_message_id) {
  return get_message_object(full_message_id.get_dialog_id(), get_message_force(full_message_id, "get_message_object"));
}

tl_object_ptr<td_api::message> MessagesManager::get_message_object(DialogId dialog_id, const Message *m,
                                                                   bool for_event_log) const {
  if (m == nullptr) {
    return nullptr;
  }

  auto sending_state = get_message_sending_state_object(m);

  if (for_event_log) {
    CHECK(m->message_id.is_server());
    CHECK(sending_state == nullptr);
  }

  bool can_delete = true;
  auto dialog_type = dialog_id.get_type();
  auto is_bot = td_->auth_manager_->is_bot();
  if (dialog_type == DialogType::Channel) {
    auto dialog_status = td_->contacts_manager_->get_channel_permissions(dialog_id.get_channel_id());
    can_delete = can_delete_channel_message(dialog_status, m, is_bot);
  }

  bool is_scheduled = m->message_id.is_scheduled();
  DialogId my_dialog_id = get_my_dialog_id();
  bool can_delete_for_self = false;
  bool can_delete_for_all_users = can_delete && can_revoke_message(dialog_id, m);
  if (can_delete) {
    switch (dialog_type) {
      case DialogType::User:
      case DialogType::Chat:
        // TODO allow to delete yet unsent message just for self
        can_delete_for_self = !m->message_id.is_yet_unsent() || dialog_id == my_dialog_id;
        break;
      case DialogType::Channel:
      case DialogType::SecretChat:
        can_delete_for_self = !can_delete_for_all_users;
        break;
      case DialogType::None:
      default:
        UNREACHABLE();
    }
  }
  if (for_event_log) {
    can_delete_for_self = false;
    can_delete_for_all_users = false;
  } else if (is_scheduled) {
    can_delete_for_self = (dialog_id == my_dialog_id);
    can_delete_for_all_users = !can_delete_for_self;
  }

  bool is_outgoing = m->is_outgoing;
  if (dialog_id == my_dialog_id) {
    // in Saved Messages all non-forwarded messages must be outgoing
    // a forwarded message is outgoing, only if it doesn't have from_dialog_id and its sender isn't hidden
    // i.e. a message is incoming only if it's a forwarded message with known from_dialog_id or with a hidden sender
    auto forward_info = m->forward_info.get();
    is_outgoing = is_scheduled || forward_info == nullptr ||
                  (!forward_info->from_dialog_id.is_valid() && !is_forward_info_sender_hidden(forward_info));
  }

  int32 ttl = m->ttl;
  double ttl_expires_in = 0;
  if (!for_event_log) {
    if (m->ttl_expires_at != 0) {
      ttl_expires_in = max(m->ttl_expires_at - Time::now(), 1e-3);
    } else {
      ttl_expires_in = m->ttl;
    }
  } else {
    ttl = 0;
  }
  auto scheduling_state = is_scheduled ? get_message_scheduling_state_object(m->date) : nullptr;
  bool can_be_edited = for_event_log ? false : can_edit_message(dialog_id, m, false, is_bot);
  bool can_be_forwarded = for_event_log ? false : can_forward_message(dialog_id, m);
  auto media_album_id = for_event_log ? static_cast<int64>(0) : m->media_album_id;
  auto reply_to_message_id = for_event_log ? static_cast<int64>(0) : m->reply_to_message_id.get();
  bool contains_unread_mention = for_event_log ? false : m->contains_unread_mention;
  auto live_location_date = m->is_failed_to_send ? 0 : m->date;
  auto date = is_scheduled ? 0 : m->date;
  auto edit_date = m->hide_edit_date ? 0 : m->edit_date;
  return make_tl_object<td_api::message>(
      m->message_id.get(), td_->contacts_manager_->get_user_id_object(m->sender_user_id, "sender_user_id"),
      dialog_id.get(), std::move(sending_state), std::move(scheduling_state), is_outgoing, can_be_edited,
      can_be_forwarded, can_delete_for_self, can_delete_for_all_users, m->is_channel_post, contains_unread_mention,
      date, edit_date, get_message_forward_info_object(m->forward_info), reply_to_message_id, ttl, ttl_expires_in,
      td_->contacts_manager_->get_user_id_object(m->via_bot_user_id, "via_bot_user_id"), m->author_signature, m->views,
      media_album_id, get_restriction_reason_description(m->restriction_reasons),
      get_message_content_object(m->content.get(), td_, live_location_date, m->is_content_secret),
      get_reply_markup_object(m->reply_markup));
}

tl_object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count, DialogId dialog_id,
                                                                     const vector<MessageId> &message_ids) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  return get_messages_object(total_count, transform(message_ids, [this, dialog_id, d](MessageId message_id) {
                               return get_message_object(dialog_id,
                                                         get_message_force(d, message_id, "get_messages_object"));
                             }));
}

tl_object_ptr<td_api::messages> MessagesManager::get_messages_object(int32 total_count,
                                                                     const vector<FullMessageId> &full_message_ids) {
  return get_messages_object(total_count, transform(full_message_ids, [this](FullMessageId full_message_id) {
                               return get_message_object(full_message_id);
                             }));
}

tl_object_ptr<td_api::messages> MessagesManager::get_messages_object(
    int32 total_count, vector<tl_object_ptr<td_api::message>> &&messages) {
  if (total_count == -1) {
    total_count = narrow_cast<int32>(messages.size());
  }
  return td_api::make_object<td_api::messages>(total_count, std::move(messages));
}

MessagesManager::Message *MessagesManager::get_message_to_send(
    Dialog *d, MessageId reply_to_message_id, const SendMessageOptions &options, unique_ptr<MessageContent> &&content,
    bool *need_update_dialog_pos, unique_ptr<MessageForwardInfo> forward_info, bool is_copy) {
  CHECK(d != nullptr);
  CHECK(!reply_to_message_id.is_scheduled());
  CHECK(content != nullptr);

  bool is_scheduled = options.schedule_date != 0;
  DialogId dialog_id = d->dialog_id;
  MessageId message_id = is_scheduled ? get_next_yet_unsent_scheduled_message_id(d, options.schedule_date)
                                      : get_next_yet_unsent_message_id(d);
  LOG(INFO) << "Create " << message_id << " in " << dialog_id;

  auto dialog_type = dialog_id.get_type();
  auto my_id = td_->contacts_manager_->get_my_id();

  auto m = make_unique<Message>();
  set_message_id(m, message_id);
  bool is_channel_post = is_broadcast_channel(dialog_id);
  if (is_channel_post) {
    // sender of the post can be hidden
    if (!is_scheduled && td_->contacts_manager_->get_channel_sign_messages(dialog_id.get_channel_id())) {
      m->author_signature = td_->contacts_manager_->get_user_title(my_id);
    }
  } else {
    m->sender_user_id = my_id;
  }
  m->send_date = G()->unix_time();
  m->date = is_scheduled ? options.schedule_date : m->send_date;
  m->reply_to_message_id = reply_to_message_id;
  m->is_channel_post = is_channel_post;
  m->is_outgoing = is_scheduled || dialog_id != DialogId(my_id);
  m->from_background = options.from_background;
  m->views = is_channel_post ? 1 : 0;
  m->content = std::move(content);
  m->forward_info = std::move(forward_info);
  m->is_copy = is_copy || forward_info != nullptr;

  if (td_->auth_manager_->is_bot() || options.disable_notification) {
    m->disable_notification = options.disable_notification;
  } else {
    auto notification_settings = get_dialog_notification_settings(dialog_id, true);
    CHECK(notification_settings != nullptr);
    m->disable_notification = notification_settings->silent_send_message;
  }

  if (dialog_type == DialogType::SecretChat) {
    CHECK(!is_scheduled);
    m->ttl = td_->contacts_manager_->get_secret_chat_ttl(dialog_id.get_secret_chat_id());
    if (is_service_message_content(m->content->get_type())) {
      m->ttl = 0;
    }
    m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());
    if (reply_to_message_id.is_valid()) {
      auto *reply_to_message = get_message_force(d, reply_to_message_id, "get_message_to_send");
      if (reply_to_message != nullptr) {
        m->reply_to_random_id = reply_to_message->random_id;
      } else {
        m->reply_to_message_id = MessageId();
      }
    }
  }

  m->have_previous = true;
  m->have_next = true;

  do {
    m->random_id = Random::secure_int64();
  } while (m->random_id == 0 || message_random_ids_.find(m->random_id) != message_random_ids_.end());
  message_random_ids_.insert(m->random_id);

  bool need_update = false;
  CHECK(have_input_peer(dialog_id, AccessRights::Read));
  auto result = add_message_to_dialog(d, std::move(m), true, &need_update, need_update_dialog_pos, "send message");
  CHECK(result != nullptr);
  send_update_chat_has_scheduled_messages(d);
  return result;
}

int64 MessagesManager::begin_send_message(DialogId dialog_id, const Message *m) {
  LOG(INFO) << "Begin to send " << FullMessageId(dialog_id, m->message_id) << " with random_id = " << m->random_id;
  CHECK(m->random_id != 0 && being_sent_messages_.find(m->random_id) == being_sent_messages_.end());
  CHECK(m->message_id.is_yet_unsent());
  being_sent_messages_[m->random_id] = FullMessageId(dialog_id, m->message_id);
  debug_being_sent_messages_[m->random_id] = dialog_id;
  return m->random_id;
}

Status MessagesManager::can_send_message(DialogId dialog_id) const {
  if (!have_input_peer(dialog_id, AccessRights::Write)) {
    return Status::Error(400, "Have no write access to the chat");
  }

  if (dialog_id.get_type() == DialogType::Channel) {
    auto channel_id = dialog_id.get_channel_id();
    auto channel_type = td_->contacts_manager_->get_channel_type(channel_id);
    auto channel_status = td_->contacts_manager_->get_channel_permissions(channel_id);

    switch (channel_type) {
      case ChannelType::Unknown:
      case ChannelType::Megagroup:
        if (!channel_status.can_send_messages()) {
          return Status::Error(400, "Have no rights to send a message");
        }
        break;
      case ChannelType::Broadcast: {
        if (!channel_status.can_post_messages()) {
          return Status::Error(400, "Need administrator rights in the channel chat");
        }
        break;
      }
      default:
        UNREACHABLE();
    }
  }
  return Status::OK();
}

Status MessagesManager::can_send_message_content(DialogId dialog_id, const MessageContent *content,
                                                 bool is_forward) const {
  auto dialog_type = dialog_id.get_type();
  int32 secret_chat_layer = std::numeric_limits<int32>::max();
  if (dialog_type == DialogType::SecretChat) {
    auto secret_chat_id = dialog_id.get_secret_chat_id();
    secret_chat_layer = td_->contacts_manager_->get_secret_chat_layer(secret_chat_id);
  }

  bool can_send_messages = true;
  bool can_send_media = true;
  bool can_send_stickers = true;
  bool can_send_animations = true;
  bool can_send_games = true;
  bool can_send_polls = true;

  auto content_type = content->get_type();
  switch (dialog_type) {
    case DialogType::User:
      if (content_type == MessageContentType::Poll && !is_forward && !td_->auth_manager_->is_bot() &&
          !td_->contacts_manager_->is_user_bot(dialog_id.get_user_id())) {
        return Status::Error(400, "Polls can't be sent to the private chat");
      }
      break;
    case DialogType::Chat:
      // ok
      break;
    case DialogType::Channel: {
      auto channel_status = td_->contacts_manager_->get_channel_permissions(dialog_id.get_channel_id());
      can_send_messages = channel_status.can_send_messages();
      can_send_media = channel_status.can_send_media();
      can_send_stickers = channel_status.can_send_stickers();
      can_send_animations = channel_status.can_send_animations();
      can_send_games = channel_status.can_send_games();
      can_send_polls = channel_status.can_send_polls();
      break;
    }
    case DialogType::SecretChat:
      if (content_type == MessageContentType::Game) {
        return Status::Error(400, "Games can't be sent to secret chats");
      }
      if (content_type == MessageContentType::Poll) {
        return Status::Error(400, "Polls can't be sent to secret chats");
      }
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  switch (content_type) {
    case MessageContentType::Animation:
      if (!can_send_animations) {
        return Status::Error(400, "Not enough rights to send animations to the chat");
      }
      break;
    case MessageContentType::Audio:
      if (!can_send_media) {
        return Status::Error(400, "Not enough rights to send audios to the chat");
      }
      break;
    case MessageContentType::Contact:
      if (!can_send_messages) {
        return Status::Error(400, "Not enough rights to send contacts to the chat");
      }
      break;
    case MessageContentType::Document:
      if (!can_send_media) {
        return Status::Error(400, "Not enough rights to send documents to the chat");
      }
      break;
    case MessageContentType::Game:
      if (is_broadcast_channel(dialog_id)) {
        // return Status::Error(400, "Games can't be sent to channel chats");
      }

      if (!can_send_games) {
        return Status::Error(400, "Not enough rights to send games to the chat");
      }
      break;
    case MessageContentType::Invoice:
      if (!is_forward) {
        switch (dialog_type) {
          case DialogType::User:
            // ok
            break;
          case DialogType::Chat:
          case DialogType::Channel:
          case DialogType::SecretChat:
            return Status::Error(400, "Invoices can be sent only to private chats");
          case DialogType::None:
          default:
            UNREACHABLE();
        }
      }
      break;
    case MessageContentType::LiveLocation:
      if (!can_send_messages) {
        return Status::Error(400, "Not enough rights to send live locations to the chat");
      }
      break;
    case MessageContentType::Location:
      if (!can_send_messages) {
        return Status::Error(400, "Not enough rights to send locations to the chat");
      }
      break;
    case MessageContentType::Photo:
      if (!can_send_media) {
        return Status::Error(400, "Not enough rights to send photos to the chat");
      }
      break;
    case MessageContentType::Poll:
      if (!can_send_polls) {
        return Status::Error(400, "Not enough rights to send polls to the chat");
      }
      if (!get_message_content_poll_is_anonymous(td_, content) && is_broadcast_channel(dialog_id)) {
        return Status::Error(400, "Non-anonymous polls can't be sent to channel chats");
      }
      break;
    case MessageContentType::Sticker:
      if (!can_send_stickers) {
        return Status::Error(400, "Not enough rights to send stickers to the chat");
      }
      break;
    case MessageContentType::Text:
      if (!can_send_messages) {
        return Status::Error(400, "Not enough rights to send text messages to the chat");
      }
      break;
    case MessageContentType::Venue:
      if (!can_send_messages) {
        return Status::Error(400, "Not enough rights to send venues to the chat");
      }
      break;
    case MessageContentType::Video:
      if (!can_send_media) {
        return Status::Error(400, "Not enough rights to send videos to the chat");
      }
      break;
    case MessageContentType::VideoNote:
      if (!can_send_media) {
        return Status::Error(400, "Not enough rights to send video notes to the chat");
      }
      if (secret_chat_layer < SecretChatActor::VIDEO_NOTES_LAYER) {
        return Status::Error(400, PSLICE()
                                      << "Video notes can't be sent to secret chat with layer " << secret_chat_layer);
      }
      break;
    case MessageContentType::VoiceNote:
      if (!can_send_media) {
        return Status::Error(400, "Not enough rights to send voice notes to the chat");
      }
      break;
    case MessageContentType::None:
    case MessageContentType::ChatCreate:
    case MessageContentType::ChatChangeTitle:
    case MessageContentType::ChatChangePhoto:
    case MessageContentType::ChatDeletePhoto:
    case MessageContentType::ChatDeleteHistory:
    case MessageContentType::ChatAddUsers:
    case MessageContentType::ChatJoinedByLink:
    case MessageContentType::ChatDeleteUser:
    case MessageContentType::ChatMigrateTo:
    case MessageContentType::ChannelCreate:
    case MessageContentType::ChannelMigrateFrom:
    case MessageContentType::PinMessage:
    case MessageContentType::GameScore:
    case MessageContentType::ScreenshotTaken:
    case MessageContentType::ChatSetTtl:
    case MessageContentType::Unsupported:
    case MessageContentType::Call:
    case MessageContentType::PaymentSuccessful:
    case MessageContentType::ContactRegistered:
    case MessageContentType::ExpiredPhoto:
    case MessageContentType::ExpiredVideo:
    case MessageContentType::CustomServiceAction:
    case MessageContentType::WebsiteConnected:
    case MessageContentType::PassportDataSent:
    case MessageContentType::PassportDataReceived:
      UNREACHABLE();
  }
  return Status::OK();
}

MessageId MessagesManager::get_persistent_message_id(const Dialog *d, MessageId message_id) const {
  if (!message_id.is_valid() && !message_id.is_valid_scheduled()) {
    return MessageId();
  }
  if (message_id.is_yet_unsent()) {
    // it is possible that user tries to do something with an already sent message by its temporary id
    // we need to use real message in this case and transparently replace message_id
    auto it = d->yet_unsent_message_id_to_persistent_message_id.find(message_id);
    if (it != d->yet_unsent_message_id_to_persistent_message_id.end()) {
      return it->second;
    }
  }

  return message_id;
}

MessageId MessagesManager::get_reply_to_message_id(Dialog *d, MessageId message_id) {
  CHECK(d != nullptr);
  if (!message_id.is_valid()) {
    return MessageId();
  }
  message_id = get_persistent_message_id(d, message_id);
  const Message *m = get_message_force(d, message_id, "get_reply_to_message_id");
  if (m == nullptr || m->message_id.is_yet_unsent() ||
      (m->message_id.is_local() && d->dialog_id.get_type() != DialogType::SecretChat)) {
    if (message_id.is_server() && d->dialog_id.get_type() != DialogType::SecretChat &&
        message_id > d->last_new_message_id && message_id <= d->max_notification_message_id) {
      // allow to reply yet unreceived server message
      return message_id;
    }

    // TODO local replies to local messages can be allowed
    // TODO replies to yet unsent messages can be allowed with special handling of them on application restart
    return MessageId();
  }
  return m->message_id;
}

vector<FileId> MessagesManager::get_message_file_ids(const Message *m) const {
  CHECK(m != nullptr);
  return get_message_content_file_ids(m->content.get(), td_);
}

void MessagesManager::cancel_upload_message_content_files(const MessageContent *content) {
  auto file_id = get_message_content_upload_file_id(content);
  // always cancel file upload, it should be a no-op in the worst case
  if (being_uploaded_files_.erase(file_id) || file_id.is_valid()) {
    cancel_upload_file(file_id);
  }
  file_id = get_message_content_thumbnail_file_id(content, td_);
  if (being_uploaded_thumbnails_.erase(file_id) || file_id.is_valid()) {
    cancel_upload_file(file_id);
  }
}

void MessagesManager::cancel_upload_file(FileId file_id) {
  // send the request later so they doesn't interfere with other actions
  // for example merge, supposed to happen soon, can auto-cancel the upload
  LOG(INFO) << "Cancel upload of file " << file_id;
  send_closure_later(G()->file_manager(), &FileManager::cancel_upload, file_id);
}

void MessagesManager::cancel_send_message_query(DialogId dialog_id, Message *m) {
  CHECK(m != nullptr);
  CHECK(m->content != nullptr);
  CHECK(m->message_id.is_valid() || m->message_id.is_valid_scheduled());
  CHECK(m->message_id.is_yet_unsent());
  LOG(INFO) << "Cancel send message query for " << m->message_id;

  cancel_upload_message_content_files(m->content.get());

  CHECK(m->edited_content == nullptr);

  if (!m->send_query_ref.empty()) {
    LOG(INFO) << "Cancel send query for " << m->message_id;
    cancel_query(m->send_query_ref);
    m->send_query_ref = NetQueryRef();
  }

  if (m->send_message_logevent_id != 0) {
    LOG(INFO) << "Delete send message log event for " << m->message_id;
    binlog_erase(G()->td_db()->get_binlog(), m->send_message_logevent_id);
    m->send_message_logevent_id = 0;
  }

  if (m->reply_to_message_id.is_valid() && !m->reply_to_message_id.is_yet_unsent()) {
    auto it = replied_by_yet_unsent_messages_.find({dialog_id, m->reply_to_message_id});
    CHECK(it != replied_by_yet_unsent_messages_.end());
    it->second--;
    CHECK(it->second >= 0);
    if (it->second == 0) {
      replied_by_yet_unsent_messages_.erase(it);
    }
  }

  if (m->media_album_id != 0) {
    send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id, dialog_id,
                       m->message_id, Status::OK());
  }

  if (!m->message_id.is_scheduled() && G()->parameters().use_file_db &&
      !m->is_copy) {  // ResourceManager::Mode::Baseline
    auto queue_id = get_sequence_dispatcher_id(dialog_id, m->content->get_type());
    if (queue_id & 1) {
      auto queue_it = yet_unsent_media_queues_.find(queue_id);
      if (queue_it != yet_unsent_media_queues_.end()) {
        auto &queue = queue_it->second;
        LOG(INFO) << "Delete " << m->message_id << " from queue " << queue_id;
        if (queue.erase(m->message_id.get()) != 0) {
          if (queue.empty()) {
            yet_unsent_media_queues_.erase(queue_it);
          } else {
            on_yet_unsent_media_queue_updated(dialog_id);
          }
        }
      }
    }
  }
}

void MessagesManager::cancel_send_deleted_message(DialogId dialog_id, Message *m, bool is_permanently_deleted) {
  CHECK(m != nullptr);
  if (m->message_id.is_yet_unsent()) {
    cancel_send_message_query(dialog_id, m);
  } else if (is_permanently_deleted || !m->message_id.is_scheduled()) {
    cancel_edit_message_media(dialog_id, m, "Message was deleted");
  }
}

bool MessagesManager::is_message_auto_read(DialogId dialog_id, bool is_outgoing) const {
  switch (dialog_id.get_type()) {
    case DialogType::User: {
      auto user_id = dialog_id.get_user_id();
      if (user_id == td_->contacts_manager_->get_my_id()) {
        return true;
      }
      if (is_outgoing && td_->contacts_manager_->is_user_bot(user_id)) {
        return true;
      }
      return false;
    }
    case DialogType::Chat:
      // TODO auto_read message content and messages sent to group with bots only
      return false;
    case DialogType::Channel:
      return is_outgoing && is_broadcast_channel(dialog_id);
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
      return false;
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::add_message_dependencies(Dependencies &dependencies, DialogId dialog_id, const Message *m) {
  dependencies.user_ids.insert(m->sender_user_id);
  dependencies.user_ids.insert(m->via_bot_user_id);
  if (m->forward_info != nullptr) {
    dependencies.user_ids.insert(m->forward_info->sender_user_id);
    if (m->forward_info->dialog_id.is_valid() && dependencies.dialog_ids.insert(m->forward_info->dialog_id).second) {
      add_dialog_dependencies(dependencies, m->forward_info->dialog_id);
    }
    if (m->forward_info->from_dialog_id.is_valid() &&
        dependencies.dialog_ids.insert(m->forward_info->from_dialog_id).second) {
      add_dialog_dependencies(dependencies, m->forward_info->from_dialog_id);
    }
  }
  add_message_content_dependencies(dependencies, m->content.get());
}

void MessagesManager::add_dialog_dependencies(Dependencies &dependencies, DialogId dialog_id) {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      dependencies.user_ids.insert(dialog_id.get_user_id());
      break;
    case DialogType::Chat:
      dependencies.chat_ids.insert(dialog_id.get_chat_id());
      break;
    case DialogType::Channel:
      dependencies.channel_ids.insert(dialog_id.get_channel_id());
      break;
    case DialogType::SecretChat:
      dependencies.secret_chat_ids.insert(dialog_id.get_secret_chat_id());
      break;
    case DialogType::None:
      break;
    default:
      UNREACHABLE();
  }
}

class MessagesManager::SendMessageLogEvent {
 public:
  DialogId dialog_id;
  const Message *m_in;
  unique_ptr<Message> m_out;

  SendMessageLogEvent() : dialog_id(), m_in(nullptr) {
  }

  SendMessageLogEvent(DialogId dialog_id, const Message *m) : dialog_id(dialog_id), m_in(m) {
  }

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id, storer);
    td::store(*m_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id, parser);
    td::parse(m_out, parser);
  }
};

Result<MessageId> MessagesManager::send_message(DialogId dialog_id, MessageId reply_to_message_id,
                                                tl_object_ptr<td_api::sendMessageOptions> &&options,
                                                tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                tl_object_ptr<td_api::InputMessageContent> &&input_message_content) {
  if (input_message_content == nullptr) {
    return Status::Error(5, "Can't send message without content");
  }

  LOG(INFO) << "Begin to send message to " << dialog_id << " in reply to " << reply_to_message_id;
  if (input_message_content->get_id() == td_api::inputMessageForwarded::ID) {
    auto input_message = static_cast<const td_api::inputMessageForwarded *>(input_message_content.get());
    return forward_message(dialog_id, DialogId(input_message->from_chat_id_), MessageId(input_message->message_id_),
                           std::move(options), input_message->in_game_share_, input_message->send_copy_,
                           input_message->remove_caption_);
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(message_reply_markup, get_dialog_reply_markup(dialog_id, std::move(reply_markup)));
  TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
  TRY_RESULT(send_message_options, process_send_message_options(dialog_id, std::move(options)));
  TRY_STATUS(can_use_send_message_options(send_message_options, message_content));

  // there must be no errors after get_message_to_send call

  bool need_update_dialog_pos = false;
  Message *m = get_message_to_send(
      d, get_reply_to_message_id(d, reply_to_message_id), send_message_options,
      dup_message_content(td_, dialog_id, message_content.content.get(), MessageContentDupType::Send),
      &need_update_dialog_pos, nullptr, message_content.via_bot_user_id.is_valid());
  m->reply_markup = std::move(message_reply_markup);
  m->via_bot_user_id = message_content.via_bot_user_id;
  m->disable_web_page_preview = message_content.disable_web_page_preview;
  m->clear_draft = message_content.clear_draft;
  if (message_content.ttl > 0) {
    m->ttl = message_content.ttl;
    m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());
  }

  if (message_content.clear_draft) {
    update_dialog_draft_message(d, nullptr, false, !need_update_dialog_pos);
  }

  save_send_message_logevent(dialog_id, m);
  do_send_message(dialog_id, m);

  send_update_new_message(d, m);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_message");
  }

  return m->message_id;
}

Result<InputMessageContent> MessagesManager::process_input_message_content(
    DialogId dialog_id, tl_object_ptr<td_api::InputMessageContent> &&input_message_content) {
  if (input_message_content == nullptr) {
    return Status::Error(400, "Can't send message without content");
  }

  if (input_message_content->get_id() == td_api::inputMessageForwarded::ID) {
    auto input_message = static_cast<const td_api::inputMessageForwarded *>(input_message_content.get());
    if (!input_message->send_copy_) {
      return Status::Error(400, "Can't use forwarded message");
    }

    DialogId from_dialog_id(input_message->from_chat_id_);
    Dialog *from_dialog = get_dialog_force(from_dialog_id);
    if (from_dialog == nullptr) {
      return Status::Error(400, "Chat to copy message from not found");
    }
    if (!have_input_peer(from_dialog_id, AccessRights::Read)) {
      return Status::Error(400, "Can't access the chat to copy message from");
    }
    if (from_dialog_id.get_type() == DialogType::SecretChat) {
      return Status::Error(400, "Can't copy message from secret chats");
    }
    MessageId message_id = get_persistent_message_id(from_dialog, MessageId(input_message->message_id_));

    const Message *copied_message = get_message_force(from_dialog, message_id, "process_input_message_content");
    if (copied_message == nullptr) {
      return Status::Error(400, "Can't find message to copy");
    }
    if (!can_forward_message(from_dialog_id, copied_message)) {
      return Status::Error(400, "Can't copy message");
    }

    unique_ptr<MessageContent> content = dup_message_content(
        td_, dialog_id, copied_message->content.get(),
        input_message->remove_caption_ ? MessageContentDupType::CopyWithoutCaption : MessageContentDupType::Copy);
    if (content == nullptr) {
      return Status::Error(400, "Can't copy message content");
    }

    return InputMessageContent(std::move(content), copied_message->disable_web_page_preview, false, 0, UserId());
  }

  TRY_RESULT(content, get_input_message_content(dialog_id, std::move(input_message_content), td_));

  if (content.ttl < 0 || content.ttl > MAX_PRIVATE_MESSAGE_TTL) {
    return Status::Error(10, "Wrong message TTL specified");
  }
  if (content.ttl > 0 && dialog_id.get_type() != DialogType::User) {
    return Status::Error(10, "Message TTL can be specified only in private chats");
  }

  if (dialog_id != DialogId()) {
    TRY_STATUS(can_send_message_content(dialog_id, content.content.get(), false));
  }

  return std::move(content);
}

Result<MessagesManager::SendMessageOptions> MessagesManager::process_send_message_options(
    DialogId dialog_id, tl_object_ptr<td_api::sendMessageOptions> &&options) const {
  SendMessageOptions result;
  if (options != nullptr) {
    result.disable_notification = options->disable_notification_;
    result.from_background = options->from_background_;
    TRY_RESULT_ASSIGN(result.schedule_date, get_message_schedule_date(std::move(options->scheduling_state_)));
  }

  auto dialog_type = dialog_id.get_type();
  bool is_secret = dialog_type == DialogType::SecretChat;
  if (result.disable_notification && is_secret) {
    return Status::Error(400, "Can't send messages with silent notifications to secret chats");
  }
  if (result.schedule_date != 0) {
    if (is_secret) {
      return Status::Error(400, "Can't schedule messages in secret chats");
    }
    if (td_->auth_manager_->is_bot()) {
      return Status::Error(400, "Bots can't send scheduled messages");
    }
  }
  if (result.schedule_date == SCHEDULE_WHEN_ONLINE_DATE) {
    if (dialog_type != DialogType::User) {
      return Status::Error(400, "Messages can be scheduled till online only in private chats");
    }
    if (dialog_id == get_my_dialog_id()) {
      return Status::Error(400, "Can't scheduled till online messages in chat with self");
    }
  }

  return result;
}

Status MessagesManager::can_use_send_message_options(const SendMessageOptions &options,
                                                     const unique_ptr<MessageContent> &content, int32 ttl) {
  if (options.schedule_date != 0) {
    if (ttl > 0) {
      return Status::Error(400, "Can't send scheduled self-destructing messages");
    }
    if (content->get_type() == MessageContentType::LiveLocation) {
      return Status::Error(400, "Can't send scheduled live location messages");
    }
  }

  return Status::OK();
}

Status MessagesManager::can_use_send_message_options(const SendMessageOptions &options,
                                                     const InputMessageContent &content) {
  return can_use_send_message_options(options, content.content, content.ttl);
}

int64 MessagesManager::generate_new_media_album_id() {
  int64 media_album_id = 0;
  do {
    media_album_id = Random::secure_int64();
  } while (media_album_id >= 0 || pending_message_group_sends_.count(media_album_id) != 0);
  return media_album_id;
}

Result<vector<MessageId>> MessagesManager::send_message_group(
    DialogId dialog_id, MessageId reply_to_message_id, tl_object_ptr<td_api::sendMessageOptions> &&options,
    vector<tl_object_ptr<td_api::InputMessageContent>> &&input_message_contents) {
  if (input_message_contents.size() > MAX_GROUPED_MESSAGES) {
    return Status::Error(4, "Too much messages to send as an album");
  }
  if (input_message_contents.empty()) {
    return Status::Error(4, "There are no messages to send");
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(send_message_options, process_send_message_options(dialog_id, std::move(options)));

  vector<std::pair<unique_ptr<MessageContent>, int32>> message_contents;
  for (auto &input_message_content : input_message_contents) {
    TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
    TRY_STATUS(can_use_send_message_options(send_message_options, message_content));
    if (!is_allowed_media_group_content(message_content.content->get_type())) {
      return Status::Error(5, "Wrong message content type");
    }

    message_contents.emplace_back(std::move(message_content.content), message_content.ttl);
  }

  reply_to_message_id = get_reply_to_message_id(d, reply_to_message_id);

  int64 media_album_id = 0;
  if (message_contents.size() > 1) {
    media_album_id = generate_new_media_album_id();
  }

  // there must be no errors after get_message_to_send calls

  vector<MessageId> result;
  bool need_update_dialog_pos = false;
  for (auto &message_content : message_contents) {
    Message *m = get_message_to_send(
        d, reply_to_message_id, send_message_options,
        dup_message_content(td_, dialog_id, message_content.first.get(), MessageContentDupType::Send),
        &need_update_dialog_pos);
    result.push_back(m->message_id);
    auto ttl = message_content.second;
    if (ttl > 0) {
      m->ttl = ttl;
      m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());
    }
    m->media_album_id = media_album_id;

    save_send_message_logevent(dialog_id, m);
    do_send_message(dialog_id, m);

    send_update_new_message(d, m);
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_message_group");
  }

  return result;
}

void MessagesManager::save_send_message_logevent(DialogId dialog_id, const Message *m) {
  if (!G()->parameters().use_message_db) {
    return;
  }

  CHECK(m != nullptr);
  LOG(INFO) << "Save " << FullMessageId(dialog_id, m->message_id) << " to binlog";
  auto logevent = SendMessageLogEvent(dialog_id, m);
  auto storer = LogEventStorerImpl<SendMessageLogEvent>(logevent);
  CHECK(m->send_message_logevent_id == 0);
  m->send_message_logevent_id = binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SendMessage, storer);
}

void MessagesManager::do_send_message(DialogId dialog_id, const Message *m, vector<int> bad_parts) {
  bool is_edit = m->message_id.is_any_server();
  LOG(INFO) << "Do " << (is_edit ? "edit" : "send") << ' ' << FullMessageId(dialog_id, m->message_id);
  bool is_secret = dialog_id.get_type() == DialogType::SecretChat;

  if (m->media_album_id != 0 && bad_parts.empty() && !is_secret && !is_edit) {
    auto &request = pending_message_group_sends_[m->media_album_id];
    request.dialog_id = dialog_id;
    request.message_ids.push_back(m->message_id);
    request.is_finished.push_back(false);

    request.results.push_back(Status::OK());
  }

  auto content = is_edit ? m->edited_content.get() : m->content.get();
  CHECK(content != nullptr);
  auto content_type = content->get_type();
  if (content_type == MessageContentType::Text) {
    CHECK(!is_edit);
    const FormattedText *message_text = get_message_content_text(content);
    CHECK(message_text != nullptr);

    int64 random_id = begin_send_message(dialog_id, m);
    if (is_secret) {
      CHECK(!m->message_id.is_scheduled());
      auto layer = td_->contacts_manager_->get_secret_chat_layer(dialog_id.get_secret_chat_id());
      send_closure(td_->create_net_actor<SendSecretMessageActor>(), &SendSecretMessageActor::send, dialog_id,
                   m->reply_to_random_id, m->ttl, message_text->text,
                   get_secret_input_media(content, td_, nullptr, BufferSlice(), layer),
                   get_input_secret_message_entities(message_text->entities, layer), m->via_bot_user_id,
                   m->media_album_id, random_id);
    } else {
      send_closure(td_->create_net_actor<SendMessageActor>(), &SendMessageActor::send, get_message_flags(m), dialog_id,
                   m->reply_to_message_id, get_message_schedule_date(m), get_input_reply_markup(m->reply_markup),
                   get_input_message_entities(td_->contacts_manager_.get(), message_text->entities, "do_send_message"),
                   message_text->text, random_id, &m->send_query_ref,
                   get_sequence_dispatcher_id(dialog_id, content_type));
    }
    return;
  }

  FileId file_id = get_message_content_any_file_id(content);  // any_file_id, because it could be a photo sent by ID
  FileView file_view = td_->file_manager_->get_file_view(file_id);
  FileId thumbnail_file_id = get_message_content_thumbnail_file_id(content, td_);
  LOG(DEBUG) << "Need to send file " << file_id << " with thumbnail " << thumbnail_file_id;
  if (is_secret) {
    CHECK(!is_edit);
    auto layer = td_->contacts_manager_->get_secret_chat_layer(dialog_id.get_secret_chat_id());
    auto secret_input_media = get_secret_input_media(content, td_, nullptr, BufferSlice(), layer);
    if (secret_input_media.empty()) {
      CHECK(file_view.is_encrypted_secret());
      CHECK(file_id.is_valid());
      being_uploaded_files_[file_id] = {FullMessageId(dialog_id, m->message_id), thumbnail_file_id};
      LOG(INFO) << "Ask to upload encrypted file " << file_id;
      // need to call resume_upload synchronously to make upload process consistent with being_uploaded_files_
      td_->file_manager_->resume_upload(file_id, std::move(bad_parts), upload_media_callback_, 1, m->message_id.get());
    } else {
      on_secret_message_media_uploaded(dialog_id, m, std::move(secret_input_media), file_id, thumbnail_file_id);
    }
  } else {
    auto input_media = get_input_media(content, td_, m->ttl, false);
    if (input_media == nullptr) {
      if (content_type == MessageContentType::Game || content_type == MessageContentType::Poll) {
        return;
      }
      if (content_type == MessageContentType::Photo) {
        thumbnail_file_id = FileId();
      }

      CHECK(file_id.is_valid());
      being_uploaded_files_[file_id] = {FullMessageId(dialog_id, m->message_id), thumbnail_file_id};
      LOG(INFO) << "Ask to upload file " << file_id;
      // need to call resume_upload synchronously to make upload process consistent with being_uploaded_files_
      td_->file_manager_->resume_upload(file_id, std::move(bad_parts), upload_media_callback_, 1, m->message_id.get());
    } else {
      on_message_media_uploaded(dialog_id, m, std::move(input_media), file_id, thumbnail_file_id);
    }
  }
}

void MessagesManager::on_message_media_uploaded(DialogId dialog_id, const Message *m,
                                                tl_object_ptr<telegram_api::InputMedia> &&input_media, FileId file_id,
                                                FileId thumbnail_file_id) {
  CHECK(m != nullptr);
  CHECK(input_media != nullptr);

  auto message_id = m->message_id;
  if (message_id.is_any_server()) {
    const FormattedText *caption = get_message_content_caption(m->edited_content.get());
    auto input_reply_markup = get_input_reply_markup(m->edited_reply_markup);
    bool was_uploaded = FileManager::extract_was_uploaded(input_media);
    bool was_thumbnail_uploaded = FileManager::extract_was_thumbnail_uploaded(input_media);

    LOG(INFO) << "Edit media from " << message_id << " in " << dialog_id;
    auto schedule_date = get_message_schedule_date(m);
    auto promise = PromiseCreator::lambda(
        [actor_id = actor_id(this), dialog_id, message_id, file_id, thumbnail_file_id, schedule_date,
         generation = m->edit_generation, was_uploaded, was_thumbnail_uploaded,
         file_reference = FileManager::extract_file_reference(input_media)](Result<Unit> result) mutable {
          send_closure(actor_id, &MessagesManager::on_message_media_edited, dialog_id, message_id, file_id,
                       thumbnail_file_id, was_uploaded, was_thumbnail_uploaded, std::move(file_reference),
                       schedule_date, generation, std::move(result));
        });
    send_closure(td_->create_net_actor<EditMessageActor>(std::move(promise)), &EditMessageActor::send, 1 << 11,
                 dialog_id, message_id, caption == nullptr ? "" : caption->text,
                 get_input_message_entities(td_->contacts_manager_.get(), caption, "edit_message_media"),
                 std::move(input_media), std::move(input_reply_markup), schedule_date,
                 get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
    return;
  }

  if (m->media_album_id == 0) {
    on_media_message_ready_to_send(
        dialog_id, message_id,
        PromiseCreator::lambda([this, dialog_id, input_media = std::move(input_media), file_id,
                                thumbnail_file_id](Result<Message *> result) mutable {
          if (result.is_error() || G()->close_flag()) {
            return;
          }

          auto m = result.move_as_ok();
          CHECK(m != nullptr);
          CHECK(input_media != nullptr);

          const FormattedText *caption = get_message_content_caption(m->content.get());
          LOG(INFO) << "Send media from " << m->message_id << " in " << dialog_id << " in reply to "
                    << m->reply_to_message_id;
          int64 random_id = begin_send_message(dialog_id, m);
          send_closure(
              td_->create_net_actor<SendMediaActor>(), &SendMediaActor::send, file_id, thumbnail_file_id,
              get_message_flags(m), dialog_id, m->reply_to_message_id, get_message_schedule_date(m),
              get_input_reply_markup(m->reply_markup),
              get_input_message_entities(td_->contacts_manager_.get(), caption, "on_message_media_uploaded"),
              caption == nullptr ? "" : caption->text, std::move(input_media), random_id, &m->send_query_ref,
              get_sequence_dispatcher_id(dialog_id, m->is_copy ? MessageContentType::None : m->content->get_type()));
        }));
  } else {
    switch (input_media->get_id()) {
      case telegram_api::inputMediaUploadedDocument::ID:
        static_cast<telegram_api::inputMediaUploadedDocument *>(input_media.get())->flags_ |=
            telegram_api::inputMediaUploadedDocument::NOSOUND_VIDEO_MASK;
      // fallthrough
      case telegram_api::inputMediaUploadedPhoto::ID:
      case telegram_api::inputMediaDocumentExternal::ID:
      case telegram_api::inputMediaPhotoExternal::ID:
        LOG(INFO) << "Upload media from " << message_id << " in " << dialog_id;
        td_->create_handler<UploadMediaQuery>()->send(dialog_id, message_id, file_id, thumbnail_file_id,
                                                      std::move(input_media));
        break;
      case telegram_api::inputMediaDocument::ID:
      case telegram_api::inputMediaPhoto::ID:
        send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id,
                           dialog_id, message_id, Status::OK());
        break;
      default:
        LOG(ERROR) << "Have wrong input media " << to_string(input_media);
        send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id,
                           dialog_id, message_id, Status::Error(400, "Wrong input media"));
    }
  }
}

void MessagesManager::on_secret_message_media_uploaded(DialogId dialog_id, const Message *m,
                                                       SecretInputMedia &&secret_input_media, FileId file_id,
                                                       FileId thumbnail_file_id) {
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());
  CHECK(!secret_input_media.empty());
  /*
  if (m->media_album_id != 0) {
    switch (secret_input_media->input_file_->get_id()) {
      case telegram_api::inputEncryptedFileUploaded::ID:
      case telegram_api::inputEncryptedFileBigUploaded::ID:
        LOG(INFO) << "Upload media from " << m->message_id << " in " << dialog_id;
        return td_->create_handler<UploadEncryptedMediaQuery>()->send(dialog_id, m->message_id, std::move(secret_input_media));
      case telegram_api::inputEncryptedFile::ID:
        return send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id,
                           dialog_id, m->message_id, Status::OK());
      default:
        LOG(ERROR) << "Have wrong secret input media " << to_string(secret_input_media->input_file_);
        return send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id,
                           dialog_id, m->message_id, Status::Error(400, "Wrong input media"));
  }
  */
  // TODO use file_id, thumbnail_file_id, was_uploaded, was_thumbnail_uploaded,
  // invalidate partial remote location for file_id in case of failed upload even message has already been deleted
  on_media_message_ready_to_send(
      dialog_id, m->message_id,
      PromiseCreator::lambda(
          [this, dialog_id, secret_input_media = std::move(secret_input_media)](Result<Message *> result) mutable {
            if (result.is_error() || G()->close_flag()) {
              return;
            }

            auto m = result.move_as_ok();
            CHECK(m != nullptr);
            CHECK(!secret_input_media.empty());
            LOG(INFO) << "Send secret media from " << m->message_id << " in " << dialog_id << " in reply to "
                      << m->reply_to_message_id;
            int64 random_id = begin_send_message(dialog_id, m);
            send_closure(td_->create_net_actor<SendSecretMessageActor>(), &SendSecretMessageActor::send, dialog_id,
                         m->reply_to_random_id, m->ttl, "", std::move(secret_input_media),
                         vector<tl_object_ptr<secret_api::MessageEntity>>(), m->via_bot_user_id, m->media_album_id,
                         random_id);
          }));
}

void MessagesManager::on_upload_message_media_success(DialogId dialog_id, MessageId message_id,
                                                      tl_object_ptr<telegram_api::MessageMedia> &&media) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  CHECK(message_id.is_valid() || message_id.is_valid_scheduled());
  CHECK(message_id.is_yet_unsent());
  Message *m = get_message(d, message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send error to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat "
              << FullMessageId{dialog_id, message_id};
    return;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return;  // the message should be deleted soon
  }

  auto caption = get_message_content_caption(m->content.get());
  auto content = get_message_content(td_, caption == nullptr ? FormattedText() : *caption, std::move(media), dialog_id,
                                     false, UserId(), nullptr);

  update_message_content(dialog_id, m, std::move(content), true, true, true);

  auto input_media = get_input_media(m->content.get(), td_, m->ttl, true);
  Status result;
  if (input_media == nullptr) {
    result = Status::Error(400, "Failed to upload file");
  }

  send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id, dialog_id,
                     m->message_id, std::move(result));
}

void MessagesManager::on_upload_message_media_file_part_missing(DialogId dialog_id, MessageId message_id,
                                                                int bad_part) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  Message *m = get_message(d, message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send error to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat "
              << FullMessageId{dialog_id, message_id};
    return;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << m->message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(get_dialog(dialog_id), 5);
    return;  // the message should be deleted soon
  }

  CHECK(dialog_id.get_type() != DialogType::SecretChat);

  do_send_message(dialog_id, m, {bad_part});
}

void MessagesManager::on_upload_message_media_fail(DialogId dialog_id, MessageId message_id, Status error) {
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  Message *m = get_message(d, message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send error to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat "
              << FullMessageId{dialog_id, message_id};
    return;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << m->message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(get_dialog(dialog_id), 5);
    return;  // the message should be deleted soon
  }

  CHECK(dialog_id.get_type() != DialogType::SecretChat);

  send_closure_later(actor_id(this), &MessagesManager::on_upload_message_media_finished, m->media_album_id, dialog_id,
                     m->message_id, std::move(error));
}

void MessagesManager::on_upload_message_media_finished(int64 media_album_id, DialogId dialog_id, MessageId message_id,
                                                       Status result) {
  CHECK(media_album_id < 0);
  auto it = pending_message_group_sends_.find(media_album_id);
  if (it == pending_message_group_sends_.end()) {
    // the group may be already sent or failed to be sent
    return;
  }
  auto &request = it->second;
  CHECK(request.dialog_id == dialog_id);
  auto message_it = std::find(request.message_ids.begin(), request.message_ids.end(), message_id);
  if (message_it == request.message_ids.end()) {
    // the message may be already deleted and the album is recreated without it
    CHECK(message_id.is_yet_unsent());
    LOG_CHECK(get_message({dialog_id, message_id}) == nullptr)
        << dialog_id << ' ' << request.message_ids << ' ' << message_id << ' ' << request.finished_count << ' '
        << request.is_finished << ' ' << request.results;
    return;
  }
  auto pos = static_cast<size_t>(message_it - request.message_ids.begin());

  if (request.is_finished[pos]) {
    LOG(INFO) << "Upload media of " << message_id << " in " << dialog_id << " from group " << media_album_id
              << " at pos " << pos << " was already finished";
    return;
  }
  LOG(INFO) << "Finish to upload media of " << message_id << " in " << dialog_id << " from group " << media_album_id
            << " at pos " << pos << " with result " << result
            << " and previous finished_count = " << request.finished_count;

  request.results[pos] = std::move(result);
  request.is_finished[pos] = true;
  request.finished_count++;

  if (request.finished_count == request.message_ids.size() || request.results[pos].is_error()) {
    // send later, because some messages may be being deleted now
    for (auto request_message_id : request.message_ids) {
      LOG(INFO) << "Send on_media_message_ready_to_send for " << request_message_id << " in " << dialog_id;
      send_closure_later(actor_id(this), &MessagesManager::on_media_message_ready_to_send, dialog_id,
                         request_message_id,
                         PromiseCreator::lambda([actor_id = actor_id(this), media_album_id](Result<Message *> result) {
                           if (result.is_error() || G()->close_flag()) {
                             return;
                           }

                           auto m = result.move_as_ok();
                           CHECK(m != nullptr);
                           CHECK(m->media_album_id == media_album_id);
                           send_closure_later(actor_id, &MessagesManager::do_send_message_group, media_album_id);
                         }));
    }
  }
}

void MessagesManager::do_send_message_group(int64 media_album_id) {
  CHECK(media_album_id < 0);
  auto it = pending_message_group_sends_.find(media_album_id);
  if (it == pending_message_group_sends_.end()) {
    // the group may be already sent or failed to be sent
    return;
  }
  auto &request = it->second;

  auto dialog_id = request.dialog_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto default_status = can_send_message(dialog_id);
  bool success = default_status.is_ok();
  vector<FileId> file_ids;
  vector<int64> random_ids;
  vector<tl_object_ptr<telegram_api::inputSingleMedia>> input_single_media;
  MessageId reply_to_message_id;
  int32 flags = 0;
  int32 schedule_date = 0;
  bool is_copy = false;
  for (size_t i = 0; i < request.message_ids.size(); i++) {
    auto *m = get_message(d, request.message_ids[i]);
    if (m == nullptr) {
      // skip deleted messages
      random_ids.push_back(0);
      continue;
    }

    reply_to_message_id = m->reply_to_message_id;
    flags = get_message_flags(m);
    schedule_date = get_message_schedule_date(m);
    is_copy = m->is_copy;

    file_ids.push_back(get_message_content_any_file_id(m->content.get()));
    random_ids.push_back(begin_send_message(dialog_id, m));

    LOG(INFO) << "Have file " << file_ids.back() << " in " << m->message_id << " with result " << request.results[i]
              << " and is_finished = " << static_cast<bool>(request.is_finished[i]);

    if (request.results[i].is_error() || !request.is_finished[i]) {
      success = false;
      continue;
    }

    const FormattedText *caption = get_message_content_caption(m->content.get());
    auto input_media = get_input_media(m->content.get(), td_, m->ttl, true);
    if (input_media == nullptr) {
      // TODO return CHECK
      auto file_id = get_message_content_any_file_id(m->content.get());
      auto file_view = td_->file_manager_->get_file_view(file_id);
      bool has_remote = file_view.has_remote_location();
      bool is_web = has_remote ? file_view.remote_location().is_web() : false;
      LOG(FATAL) << request.dialog_id << " " << request.finished_count << " " << i << " " << request.message_ids << " "
                 << request.is_finished << " " << request.results << " " << m->ttl << " " << has_remote << " "
                 << file_view.has_alive_remote_location() << " " << file_view.has_active_upload_remote_location() << " "
                 << file_view.has_active_download_remote_location() << " " << file_view.is_encrypted() << " " << is_web
                 << " " << file_view.has_url() << " "
                 << to_string(get_message_content_object(m->content.get(), td_, m->date, m->is_content_secret));
    }
    auto entities = get_input_message_entities(td_->contacts_manager_.get(), caption, "do_send_message_group");
    int32 input_single_media_flags = 0;
    if (!entities.empty()) {
      input_single_media_flags |= telegram_api::inputSingleMedia::ENTITIES_MASK;
    }

    input_single_media.push_back(make_tl_object<telegram_api::inputSingleMedia>(
        input_single_media_flags, std::move(input_media), random_ids.back(), caption == nullptr ? "" : caption->text,
        std::move(entities)));
  }

  if (!success) {
    if (default_status.is_ok()) {
      default_status = Status::Error(400, "Group send failed");
    }
    for (size_t i = 0; i < random_ids.size(); i++) {
      if (random_ids[i] != 0) {
        on_send_message_fail(random_ids[i],
                             request.results[i].is_error() ? std::move(request.results[i]) : default_status.clone());
      }
    }
    pending_message_group_sends_.erase(it);
    return;
  }
  LOG_CHECK(request.finished_count == request.message_ids.size())
      << request.finished_count << " " << request.message_ids.size();
  pending_message_group_sends_.erase(it);

  LOG(INFO) << "Begin to send media group " << media_album_id << " to " << dialog_id;

  if (input_single_media.empty()) {
    LOG(INFO) << "Media group " << media_album_id << " from " << dialog_id << " is empty";
  }
  send_closure(td_->create_net_actor<SendMultiMediaActor>(), &SendMultiMediaActor::send, flags, dialog_id,
               reply_to_message_id, schedule_date, std::move(file_ids), std::move(input_single_media),
               get_sequence_dispatcher_id(dialog_id, is_copy ? MessageContentType::None : MessageContentType::Photo));
}

void MessagesManager::on_media_message_ready_to_send(DialogId dialog_id, MessageId message_id,
                                                     Promise<Message *> &&promise) {
  LOG(INFO) << "Ready to send " << message_id << " to " << dialog_id;
  CHECK(promise);
  if (!G()->parameters().use_file_db || message_id.is_scheduled()) {  // ResourceManager::Mode::Greedy
    auto m = get_message({dialog_id, message_id});
    if (m != nullptr) {
      promise.set_value(std::move(m));
    }
    return;
  }

  auto queue_id = get_sequence_dispatcher_id(dialog_id, MessageContentType::Photo);
  CHECK(queue_id & 1);
  auto &queue = yet_unsent_media_queues_[queue_id];
  auto it = queue.find(message_id.get());
  if (it == queue.end()) {
    if (queue.empty()) {
      yet_unsent_media_queues_.erase(queue_id);
    }

    LOG(INFO) << "Can't find " << message_id << " in the queue of " << dialog_id;
    auto m = get_message({dialog_id, message_id});
    if (m != nullptr) {
      promise.set_value(std::move(m));
    }
    return;
  }
  if (it->second) {
    promise.set_error(Status::Error(500, "Duplicate promise"));
    return;
  }
  it->second = std::move(promise);

  on_yet_unsent_media_queue_updated(dialog_id);
}

void MessagesManager::on_yet_unsent_media_queue_updated(DialogId dialog_id) {
  auto queue_id = get_sequence_dispatcher_id(dialog_id, MessageContentType::Photo);
  CHECK(queue_id & 1);
  auto &queue = yet_unsent_media_queues_[queue_id];
  LOG(INFO) << "Queue for " << dialog_id << " is updated to size of " << queue.size();
  while (!queue.empty()) {
    auto first_it = queue.begin();
    if (!first_it->second) {
      break;
    }

    auto m = get_message({dialog_id, MessageId(first_it->first)});
    if (m != nullptr) {
      LOG(INFO) << "Can send " << FullMessageId{dialog_id, m->message_id};
      first_it->second.set_value(std::move(m));
    }
    queue.erase(first_it);
  }
  LOG(INFO) << "Queue for " << dialog_id << " now has size " << queue.size();
  if (queue.empty()) {
    yet_unsent_media_queues_.erase(queue_id);
  }
}

Result<MessageId> MessagesManager::send_bot_start_message(UserId bot_user_id, DialogId dialog_id,
                                                          const string &parameter) {
  LOG(INFO) << "Begin to send bot start message to " << dialog_id;
  if (td_->auth_manager_->is_bot()) {
    return Status::Error(5, "Bot can't send start message to another bot");
  }

  TRY_RESULT(bot_data, td_->contacts_manager_->get_bot_data(bot_user_id));

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  bool is_chat_with_bot = false;
  switch (dialog_id.get_type()) {
    case DialogType::User:
      if (dialog_id.get_user_id() != bot_user_id) {
        return Status::Error(5, "Can't send start message to a private chat other than chat with the bot");
      }
      is_chat_with_bot = true;
      break;
    case DialogType::Chat: {
      if (!bot_data.can_join_groups) {
        return Status::Error(5, "Bot can't join groups");
      }

      auto chat_id = dialog_id.get_chat_id();
      if (!td_->contacts_manager_->have_input_peer_chat(chat_id, AccessRights::Write)) {
        return Status::Error(3, "Can't access the chat");
      }
      auto status = td_->contacts_manager_->get_chat_permissions(chat_id);
      if (!status.can_invite_users()) {
        return Status::Error(3, "Need administrator rights to invite a bot to the group chat");
      }
      break;
    }
    case DialogType::Channel: {
      auto channel_id = dialog_id.get_channel_id();
      if (!td_->contacts_manager_->have_input_peer_channel(channel_id, AccessRights::Write)) {
        return Status::Error(3, "Can't access the chat");
      }
      switch (td_->contacts_manager_->get_channel_type(channel_id)) {
        case ChannelType::Megagroup:
          if (!bot_data.can_join_groups) {
            return Status::Error(5, "The bot can't join groups");
          }
          break;
        case ChannelType::Broadcast:
          return Status::Error(3, "Bots can't be invited to channel chats. Add them as administrators instead");
        case ChannelType::Unknown:
        default:
          UNREACHABLE();
      }
      auto status = td_->contacts_manager_->get_channel_permissions(channel_id);
      if (!status.can_invite_users()) {
        return Status::Error(3, "Need administrator rights to invite a bot to the supergroup chat");
      }
      break;
    }
    case DialogType::SecretChat:
      return Status::Error(5, "Can't send bot start message to a secret chat");
    case DialogType::None:
    default:
      UNREACHABLE();
  }
  string text = "/start";
  if (!is_chat_with_bot) {
    text += '@';
    text += bot_data.username;
  }

  vector<MessageEntity> text_entities;
  text_entities.emplace_back(MessageEntity::Type::BotCommand, 0, narrow_cast<int32>(text.size()));
  bool need_update_dialog_pos = false;
  Message *m = get_message_to_send(d, MessageId(), SendMessageOptions(),
                                   create_text_message_content(text, std::move(text_entities), WebPageId()),
                                   &need_update_dialog_pos);
  m->is_bot_start_message = true;

  send_update_new_message(d, m);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_bot_start_message");
  }

  save_send_bot_start_message_logevent(bot_user_id, dialog_id, parameter, m);
  do_send_bot_start_message(bot_user_id, dialog_id, parameter, m);
  return m->message_id;
}

class MessagesManager::SendBotStartMessageLogEvent {
 public:
  UserId bot_user_id;
  DialogId dialog_id;
  string parameter;
  const Message *m_in = nullptr;
  unique_ptr<Message> m_out;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(bot_user_id, storer);
    td::store(dialog_id, storer);
    td::store(parameter, storer);
    td::store(*m_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(bot_user_id, parser);
    td::parse(dialog_id, parser);
    td::parse(parameter, parser);
    td::parse(m_out, parser);
  }
};

void MessagesManager::save_send_bot_start_message_logevent(UserId bot_user_id, DialogId dialog_id,
                                                           const string &parameter, const Message *m) {
  if (!G()->parameters().use_message_db) {
    return;
  }

  CHECK(m != nullptr);
  LOG(INFO) << "Save " << FullMessageId(dialog_id, m->message_id) << " to binlog";
  SendBotStartMessageLogEvent logevent;
  logevent.bot_user_id = bot_user_id;
  logevent.dialog_id = dialog_id;
  logevent.parameter = parameter;
  logevent.m_in = m;
  auto storer = LogEventStorerImpl<SendBotStartMessageLogEvent>(logevent);
  CHECK(m->send_message_logevent_id == 0);
  m->send_message_logevent_id =
      binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SendBotStartMessage, storer);
}

void MessagesManager::do_send_bot_start_message(UserId bot_user_id, DialogId dialog_id, const string &parameter,
                                                const Message *m) {
  LOG(INFO) << "Do send bot start " << FullMessageId(dialog_id, m->message_id) << " to bot " << bot_user_id;

  int64 random_id = begin_send_message(dialog_id, m);
  telegram_api::object_ptr<telegram_api::InputPeer> input_peer = dialog_id.get_type() == DialogType::User
                                                                     ? make_tl_object<telegram_api::inputPeerEmpty>()
                                                                     : get_input_peer(dialog_id, AccessRights::Write);
  if (input_peer == nullptr) {
    return on_send_message_fail(random_id, Status::Error(400, "Have no info about the chat"));
  }
  auto bot_input_user = td_->contacts_manager_->get_input_user(bot_user_id);
  if (bot_input_user == nullptr) {
    return on_send_message_fail(random_id, Status::Error(400, "Have no info about the bot"));
  }

  m->send_query_ref = td_->create_handler<StartBotQuery>()->send(std::move(bot_input_user), dialog_id,
                                                                 std::move(input_peer), parameter, random_id);
}

Result<MessageId> MessagesManager::send_inline_query_result_message(DialogId dialog_id, MessageId reply_to_message_id,
                                                                    tl_object_ptr<td_api::sendMessageOptions> &&options,
                                                                    int64 query_id, const string &result_id,
                                                                    bool hide_via_bot) {
  LOG(INFO) << "Begin to send inline query result message to " << dialog_id << " in reply to " << reply_to_message_id;

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));
  TRY_RESULT(send_message_options, process_send_message_options(dialog_id, std::move(options)));
  bool to_secret = false;
  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
      // ok
      break;
    case DialogType::Channel: {
      auto channel_status = td_->contacts_manager_->get_channel_permissions(dialog_id.get_channel_id());
      if (!channel_status.can_use_inline_bots()) {
        return Status::Error(400, "Can't use inline bots in the chat");
      }
      break;
    }
    case DialogType::SecretChat:
      to_secret = true;
      // ok
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  const InlineMessageContent *content = td_->inline_queries_manager_->get_inline_message_content(query_id, result_id);
  if (content == nullptr) {
    return Status::Error(5, "Inline query result not found");
  }

  TRY_STATUS(can_use_send_message_options(send_message_options, content->message_content, 0));
  TRY_STATUS(can_send_message_content(dialog_id, content->message_content.get(), false));

  bool need_update_dialog_pos = false;
  Message *m = get_message_to_send(
      d, get_reply_to_message_id(d, reply_to_message_id), send_message_options,
      dup_message_content(td_, dialog_id, content->message_content.get(), MessageContentDupType::SendViaBot),
      &need_update_dialog_pos, nullptr, true);
  m->hide_via_bot = hide_via_bot;
  if (!hide_via_bot) {
    m->via_bot_user_id = td_->inline_queries_manager_->get_inline_bot_user_id(query_id);
  }
  if (content->message_reply_markup != nullptr && !to_secret) {
    m->reply_markup = make_unique<ReplyMarkup>(*content->message_reply_markup);
  }
  m->disable_web_page_preview = content->disable_web_page_preview;
  m->clear_draft = true;

  update_dialog_draft_message(d, nullptr, false, !need_update_dialog_pos);

  send_update_new_message(d, m);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_inline_query_result_message");
  }

  if (to_secret) {
    save_send_message_logevent(dialog_id, m);
    do_send_message(dialog_id, m);
    return m->message_id;
  }

  save_send_inline_query_result_message_logevent(dialog_id, m, query_id, result_id);
  do_send_inline_query_result_message(dialog_id, m, query_id, result_id);
  return m->message_id;
}

class MessagesManager::SendInlineQueryResultMessageLogEvent {
 public:
  DialogId dialog_id;
  int64 query_id;
  string result_id;
  const Message *m_in = nullptr;
  unique_ptr<Message> m_out;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id, storer);
    td::store(query_id, storer);
    td::store(result_id, storer);
    td::store(*m_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id, parser);
    td::parse(query_id, parser);
    td::parse(result_id, parser);
    td::parse(m_out, parser);
  }
};

void MessagesManager::save_send_inline_query_result_message_logevent(DialogId dialog_id, const Message *m,
                                                                     int64 query_id, const string &result_id) {
  if (!G()->parameters().use_message_db) {
    return;
  }

  CHECK(m != nullptr);
  LOG(INFO) << "Save " << FullMessageId(dialog_id, m->message_id) << " to binlog";
  SendInlineQueryResultMessageLogEvent logevent;
  logevent.dialog_id = dialog_id;
  logevent.query_id = query_id;
  logevent.result_id = result_id;
  logevent.m_in = m;
  auto storer = LogEventStorerImpl<SendInlineQueryResultMessageLogEvent>(logevent);
  CHECK(m->send_message_logevent_id == 0);
  m->send_message_logevent_id =
      binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SendInlineQueryResultMessage, storer);
}

void MessagesManager::do_send_inline_query_result_message(DialogId dialog_id, const Message *m, int64 query_id,
                                                          const string &result_id) {
  LOG(INFO) << "Do send inline query result " << FullMessageId(dialog_id, m->message_id);

  int64 random_id = begin_send_message(dialog_id, m);
  auto flags = get_message_flags(m);
  if (!m->via_bot_user_id.is_valid() || m->hide_via_bot) {
    flags |= telegram_api::messages_sendInlineBotResult::HIDE_VIA_MASK;
  }
  m->send_query_ref = td_->create_handler<SendInlineBotResultQuery>()->send(
      flags, dialog_id, m->reply_to_message_id, get_message_schedule_date(m), random_id, query_id, result_id);
}

bool MessagesManager::can_edit_message(DialogId dialog_id, const Message *m, bool is_editing,
                                       bool only_reply_markup) const {
  if (m == nullptr) {
    return false;
  }
  if (m->message_id.is_yet_unsent()) {
    return false;
  }
  if (m->message_id.is_local()) {
    return false;
  }
  if (m->forward_info != nullptr || m->had_forward_info) {
    return false;
  }

  if (m->had_reply_markup) {
    return false;
  }
  if (m->reply_markup != nullptr && m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
    return false;
  }

  auto my_id = td_->contacts_manager_->get_my_id();
  if (m->via_bot_user_id.is_valid() && (m->via_bot_user_id != my_id || m->message_id.is_scheduled())) {
    return false;
  }

  bool is_bot = td_->auth_manager_->is_bot();
  auto content_type = m->content->get_type();
  DialogId my_dialog_id(my_id);
  bool has_edit_time_limit = !(is_bot && m->is_outgoing) && dialog_id != my_dialog_id &&
                             content_type != MessageContentType::Poll &&
                             content_type != MessageContentType::LiveLocation && !m->message_id.is_scheduled();
  switch (dialog_id.get_type()) {
    case DialogType::User:
      if (!m->is_outgoing && dialog_id != my_dialog_id && !m->via_bot_user_id.is_valid()) {
        return false;
      }
      break;
    case DialogType::Chat:
      if (!m->is_outgoing && !m->via_bot_user_id.is_valid()) {
        return false;
      }
      break;
    case DialogType::Channel: {
      if (m->via_bot_user_id.is_valid()) {
        // outgoing via_bot messages can always be edited
        break;
      }

      auto channel_id = dialog_id.get_channel_id();
      auto channel_status = td_->contacts_manager_->get_channel_permissions(channel_id);
      if (m->is_channel_post) {
        if (m->message_id.is_scheduled()) {
          if (!channel_status.can_post_messages()) {
            return false;
          }
        } else {
          if (!channel_status.can_edit_messages() && !(channel_status.can_post_messages() && m->is_outgoing)) {
            return false;
          }
        }
        if (is_bot && only_reply_markup) {
          has_edit_time_limit = false;
        }
      } else {
        if (!m->is_outgoing) {
          return false;
        }
        if (channel_status.can_pin_messages()) {
          has_edit_time_limit = false;
        }
      }
      break;
    }
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }

  if (has_edit_time_limit) {
    const int32 DEFAULT_EDIT_TIME_LIMIT = 2 * 86400;
    int32 edit_time_limit = G()->shared_config().get_option_integer("edit_time_limit", DEFAULT_EDIT_TIME_LIMIT);
    if (G()->unix_time_cached() - m->date - (is_editing ? 300 : 0) >= edit_time_limit) {
      return false;
    }
  }

  switch (content_type) {
    case MessageContentType::Animation:
    case MessageContentType::Audio:
    case MessageContentType::Document:
    case MessageContentType::Game:
    case MessageContentType::Photo:
    case MessageContentType::Text:
    case MessageContentType::Video:
    case MessageContentType::VoiceNote:
      return true;
    case MessageContentType::LiveLocation: {
      if (is_bot && only_reply_markup) {
        // there is no caption to edit, but bot can edit inline reply_markup
        return true;
      }
      return G()->unix_time_cached() - m->date < get_message_content_live_location_period(m->content.get());
    }
    case MessageContentType::Poll: {
      if (is_bot && only_reply_markup) {
        // there is no caption to edit, but bot can edit inline reply_markup
        return true;
      }
      if (m->message_id.is_scheduled()) {
        return false;
      }
      return !get_message_content_poll_is_closed(td_, m->content.get());
    }
    case MessageContentType::Contact:
    case MessageContentType::Location:
    case MessageContentType::Sticker:
    case MessageContentType::Venue:
    case MessageContentType::VideoNote:
      // there is no caption to edit, but bot can edit inline reply_markup
      return is_bot && only_reply_markup;
    case MessageContentType::Invoice:
    case MessageContentType::Unsupported:
    case MessageContentType::ChatCreate:
    case MessageContentType::ChatChangeTitle:
    case MessageContentType::ChatChangePhoto:
    case MessageContentType::ChatDeletePhoto:
    case MessageContentType::ChatDeleteHistory:
    case MessageContentType::ChatAddUsers:
    case MessageContentType::ChatJoinedByLink:
    case MessageContentType::ChatDeleteUser:
    case MessageContentType::ChatMigrateTo:
    case MessageContentType::ChannelCreate:
    case MessageContentType::ChannelMigrateFrom:
    case MessageContentType::PinMessage:
    case MessageContentType::GameScore:
    case MessageContentType::ScreenshotTaken:
    case MessageContentType::ChatSetTtl:
    case MessageContentType::Call:
    case MessageContentType::PaymentSuccessful:
    case MessageContentType::ContactRegistered:
    case MessageContentType::ExpiredPhoto:
    case MessageContentType::ExpiredVideo:
    case MessageContentType::CustomServiceAction:
    case MessageContentType::WebsiteConnected:
    case MessageContentType::PassportDataSent:
    case MessageContentType::PassportDataReceived:
      return false;
    default:
      UNREACHABLE();
  }

  return false;
}

bool MessagesManager::can_resend_message(const Message *m) const {
  if (m->send_error_code != 429 && m->send_error_message != "Message is too old to be re-sent automatically" &&
      m->send_error_message != "SCHEDULE_TOO_MUCH") {
    return false;
  }
  if (m->is_bot_start_message) {
    return false;
  }
  if (m->forward_info != nullptr || m->real_forward_from_dialog_id.is_valid()) {
    // TODO implement resending of forwarded messages
    return false;
  }
  auto content_type = m->content->get_type();
  if (m->via_bot_user_id.is_valid() || m->hide_via_bot) {
    // via bot message
    if (!can_have_input_media(td_, m->content.get())) {
      return false;
    }

    // resend via_bot message as an ordinary message if error code is 429
    // TODO support other error codes
  }

  if (content_type == MessageContentType::ChatSetTtl || content_type == MessageContentType::ScreenshotTaken) {
    // TODO implement resending of ChatSetTtl and ScreenshotTaken messages
    return false;
  }
  return true;
}

bool MessagesManager::is_broadcast_channel(DialogId dialog_id) const {
  if (dialog_id.get_type() != DialogType::Channel) {
    return false;
  }

  return td_->contacts_manager_->get_channel_type(dialog_id.get_channel_id()) == ChannelType::Broadcast;
}

int32 MessagesManager::get_message_schedule_date(const Message *m) {
  if (!m->message_id.is_scheduled()) {
    return 0;
  }
  if (m->edited_schedule_date != 0) {
    return m->edited_schedule_date;
  }
  return m->date;
}

void MessagesManager::edit_message_text(FullMessageId full_message_id,
                                        tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                        tl_object_ptr<td_api::InputMessageContent> &&input_message_content,
                                        Promise<Unit> &&promise) {
  if (input_message_content == nullptr) {
    return promise.set_error(Status::Error(5, "Can't edit message without new content"));
  }
  int32 new_message_content_type = input_message_content->get_id();
  if (new_message_content_type != td_api::inputMessageText::ID) {
    return promise.set_error(Status::Error(5, "Input message content type must be InputMessageText"));
  }

  LOG(INFO) << "Begin to edit text of " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "edit_message_text");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  if (!can_edit_message(dialog_id, m, true)) {
    return promise.set_error(Status::Error(5, "Message can't be edited"));
  }

  MessageContentType old_message_content_type = m->content->get_type();
  if (old_message_content_type != MessageContentType::Text && old_message_content_type != MessageContentType::Game) {
    return promise.set_error(Status::Error(5, "There is no text in the message to edit"));
  }

  auto r_input_message_text = process_input_message_text(
      td_->contacts_manager_.get(), dialog_id, std::move(input_message_content), td_->auth_manager_->is_bot());
  if (r_input_message_text.is_error()) {
    return promise.set_error(r_input_message_text.move_as_error());
  }
  InputMessageText input_message_text = r_input_message_text.move_as_ok();

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false,
                                             !is_broadcast_channel(dialog_id));
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }
  auto input_reply_markup = get_input_reply_markup(r_new_reply_markup.ok());
  int32 flags = 0;
  if (input_message_text.disable_web_page_preview) {
    flags |= SEND_MESSAGE_FLAG_DISABLE_WEB_PAGE_PREVIEW;
  }

  send_closure(
      td_->create_net_actor<EditMessageActor>(std::move(promise)), &EditMessageActor::send, flags, dialog_id,
      m->message_id, input_message_text.text.text,
      get_input_message_entities(td_->contacts_manager_.get(), input_message_text.text.entities, "edit_message_text"),
      nullptr, std::move(input_reply_markup), get_message_schedule_date(m),
      get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
}

void MessagesManager::edit_message_live_location(FullMessageId full_message_id,
                                                 tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                 tl_object_ptr<td_api::location> &&input_location,
                                                 Promise<Unit> &&promise) {
  LOG(INFO) << "Begin to edit live location of " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "edit_message_live_location");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  if (!can_edit_message(dialog_id, m, true)) {
    return promise.set_error(Status::Error(5, "Message can't be edited"));
  }

  MessageContentType old_message_content_type = m->content->get_type();
  if (old_message_content_type != MessageContentType::LiveLocation) {
    return promise.set_error(Status::Error(5, "There is no live location in the message to edit"));
  }
  if (m->message_id.is_scheduled()) {
    LOG(ERROR) << "Have " << full_message_id << " with live location";
    return promise.set_error(Status::Error(5, "Can't edit live location in scheduled message"));
  }

  Location location(input_location);
  if (location.empty() && input_location != nullptr) {
    return promise.set_error(Status::Error(400, "Wrong location specified"));
  }

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false,
                                             !is_broadcast_channel(dialog_id));
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }
  auto input_reply_markup = get_input_reply_markup(r_new_reply_markup.ok());

  int32 flags = 0;
  if (location.empty()) {
    flags |= telegram_api::inputMediaGeoLive::STOPPED_MASK;
  }
  auto input_media = telegram_api::make_object<telegram_api::inputMediaGeoLive>(flags, false /*ignored*/,
                                                                                location.get_input_geo_point(), 0);
  send_closure(td_->create_net_actor<EditMessageActor>(std::move(promise)), &EditMessageActor::send, 0, dialog_id,
               m->message_id, string(), vector<tl_object_ptr<telegram_api::MessageEntity>>(), std::move(input_media),
               std::move(input_reply_markup), get_message_schedule_date(m),
               get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
}

void MessagesManager::cancel_edit_message_media(DialogId dialog_id, Message *m, Slice error_message) {
  if (m->edited_content == nullptr) {
    return;
  }

  cancel_upload_message_content_files(m->edited_content.get());

  m->edited_content = nullptr;
  m->edited_reply_markup = nullptr;
  m->edit_generation = 0;
  m->edit_promise.set_error(Status::Error(400, error_message));
}

void MessagesManager::on_message_media_edited(DialogId dialog_id, MessageId message_id, FileId file_id,
                                              FileId thumbnail_file_id, bool was_uploaded, bool was_thumbnail_uploaded,
                                              string file_reference, int32 schedule_date, uint64 generation,
                                              Result<Unit> &&result) {
  CHECK(message_id.is_any_server());
  auto m = get_message({dialog_id, message_id});
  if (m == nullptr || m->edit_generation != generation) {
    // message is already deleted or was edited again
    return;
  }

  CHECK(m->edited_content != nullptr);
  if (result.is_ok()) {
    // message content has already been replaced from updateEdit{Channel,}Message
    // TODO check that it really was replaced
    // need only merge files from edited_content with their uploaded counterparts
    // updateMessageContent was already sent and needs to be sent again,
    // only if 'i' and 't' sizes from edited_content was added to the photo
    std::swap(m->content, m->edited_content);
    bool need_send_update_message_content = m->edited_content->get_type() == MessageContentType::Photo &&
                                            m->content->get_type() == MessageContentType::Photo;
    update_message_content(dialog_id, m, std::move(m->edited_content), need_send_update_message_content, true, true);
  } else {
    if (was_uploaded) {
      if (was_thumbnail_uploaded) {
        CHECK(thumbnail_file_id.is_valid());
        // always delete partial remote location for the thumbnail, because it can't be reused anyway
        td_->file_manager_->delete_partial_remote_location(thumbnail_file_id);
      }
      CHECK(file_id.is_valid());
      auto error_message = result.error().message();
      if (begins_with(error_message, "FILE_PART_") && ends_with(error_message, "_MISSING")) {
        do_send_message(dialog_id, m, {to_integer<int32>(error_message.substr(10))});
        return;
      }

      if (result.error().code() != 429 && result.error().code() < 500 && !G()->close_flag()) {
        td_->file_manager_->delete_partial_remote_location(file_id);
      }
    } else if (FileReferenceManager::is_file_reference_error(result.error())) {
      if (file_id.is_valid()) {
        VLOG(file_references) << "Receive " << result.error() << " for " << file_id;
        td_->file_manager_->delete_file_reference(file_id, file_reference);
        do_send_message(dialog_id, m, {-1});
        return;
      } else {
        LOG(ERROR) << "Receive file reference error, but have no file_id";
      }
    }

    cancel_upload_message_content_files(m->edited_content.get());

    if (dialog_id.get_type() != DialogType::SecretChat) {
      get_message_from_server({dialog_id, m->message_id}, Auto());
    }
  }

  if (m->edited_schedule_date == schedule_date) {
    m->edited_schedule_date = 0;
  }
  m->edited_content = nullptr;
  m->edited_reply_markup = nullptr;
  m->edit_generation = 0;
  if (result.is_ok()) {
    m->edit_promise.set_value(Unit());
  } else {
    m->edit_promise.set_error(result.move_as_error());
  }
}

void MessagesManager::edit_message_media(FullMessageId full_message_id,
                                         tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                         tl_object_ptr<td_api::InputMessageContent> &&input_message_content,
                                         Promise<Unit> &&promise) {
  if (input_message_content == nullptr) {
    return promise.set_error(Status::Error(5, "Can't edit message without new content"));
  }
  int32 new_message_content_type = input_message_content->get_id();
  if (new_message_content_type != td_api::inputMessageAnimation::ID &&
      new_message_content_type != td_api::inputMessageAudio::ID &&
      new_message_content_type != td_api::inputMessageDocument::ID &&
      new_message_content_type != td_api::inputMessagePhoto::ID &&
      new_message_content_type != td_api::inputMessageVideo::ID) {
    return promise.set_error(Status::Error(5, "Unsupported input message content type"));
  }

  LOG(INFO) << "Begin to edit media of " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  Message *m = get_message_force(d, full_message_id.get_message_id(), "edit_message_media");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  if (!can_edit_message(dialog_id, m, true)) {
    return promise.set_error(Status::Error(5, "Message can't be edited"));
  }
  CHECK(m->message_id.is_any_server());

  MessageContentType old_message_content_type = m->content->get_type();
  if (old_message_content_type != MessageContentType::Animation &&
      old_message_content_type != MessageContentType::Audio &&
      old_message_content_type != MessageContentType::Document &&
      old_message_content_type != MessageContentType::Photo && old_message_content_type != MessageContentType::Video) {
    return promise.set_error(Status::Error(5, "There is no media in the message to edit"));
  }
  if (m->media_album_id != 0 && new_message_content_type != td_api::inputMessagePhoto::ID &&
      new_message_content_type != td_api::inputMessageVideo::ID) {
    return promise.set_error(Status::Error(5, "Message can be edit only to Photo or Video"));
  }
  if (m->ttl > 0) {
    return promise.set_error(Status::Error(5, "Can't edit media in self-destructing message"));
  }

  auto r_input_message_content = process_input_message_content(dialog_id, std::move(input_message_content));
  if (r_input_message_content.is_error()) {
    return promise.set_error(r_input_message_content.move_as_error());
  }
  InputMessageContent content = r_input_message_content.move_as_ok();
  if (content.ttl > 0) {
    return promise.set_error(Status::Error(5, "Can't enable self-destruction for media"));
  }

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false,
                                             !is_broadcast_channel(dialog_id));
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }

  cancel_edit_message_media(dialog_id, m, "Cancelled by new editMessageMedia request");

  m->edited_content = dup_message_content(td_, dialog_id, content.content.get(), MessageContentDupType::Send);
  CHECK(m->edited_content != nullptr);
  m->edited_reply_markup = r_new_reply_markup.move_as_ok();
  m->edit_generation = ++current_message_edit_generation_;
  m->edit_promise = std::move(promise);

  do_send_message(dialog_id, m);
}

void MessagesManager::edit_message_caption(FullMessageId full_message_id,
                                           tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                           tl_object_ptr<td_api::formattedText> &&input_caption,
                                           Promise<Unit> &&promise) {
  LOG(INFO) << "Begin to edit caption of " << full_message_id;

  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "edit_message_caption");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  if (!can_edit_message(dialog_id, m, true)) {
    return promise.set_error(Status::Error(5, "Message can't be edited"));
  }

  if (!can_have_message_content_caption(m->content->get_type())) {
    return promise.set_error(Status::Error(400, "There is no caption in the message to edit"));
  }

  auto r_caption = process_input_caption(td_->contacts_manager_.get(), dialog_id, std::move(input_caption),
                                         td_->auth_manager_->is_bot());
  if (r_caption.is_error()) {
    return promise.set_error(r_caption.move_as_error());
  }
  auto caption = r_caption.move_as_ok();

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false,
                                             !is_broadcast_channel(dialog_id));
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }
  auto input_reply_markup = get_input_reply_markup(r_new_reply_markup.ok());

  send_closure(td_->create_net_actor<EditMessageActor>(std::move(promise)), &EditMessageActor::send, 1 << 11, dialog_id,
               m->message_id, caption.text,
               get_input_message_entities(td_->contacts_manager_.get(), caption.entities, "edit_message_caption"),
               nullptr, std::move(input_reply_markup), get_message_schedule_date(m),
               get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
}

void MessagesManager::edit_message_reply_markup(FullMessageId full_message_id,
                                                tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  LOG(INFO) << "Begin to edit reply markup of " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "edit_message_reply_markup");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  if (!can_edit_message(dialog_id, m, true, true)) {
    return promise.set_error(Status::Error(5, "Message can't be edited"));
  }

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false,
                                             !is_broadcast_channel(dialog_id));
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }
  auto input_reply_markup = get_input_reply_markup(r_new_reply_markup.ok());
  send_closure(td_->create_net_actor<EditMessageActor>(std::move(promise)), &EditMessageActor::send, 0, dialog_id,
               m->message_id, string(), vector<tl_object_ptr<telegram_api::MessageEntity>>(), nullptr,
               std::move(input_reply_markup), get_message_schedule_date(m),
               get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
}

void MessagesManager::edit_inline_message_text(const string &inline_message_id,
                                               tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                               tl_object_ptr<td_api::InputMessageContent> &&input_message_content,
                                               Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  if (input_message_content == nullptr) {
    return promise.set_error(Status::Error(5, "Can't edit message without new content"));
  }
  int32 new_message_content_type = input_message_content->get_id();
  if (new_message_content_type != td_api::inputMessageText::ID) {
    return promise.set_error(Status::Error(5, "Input message content type must be InputMessageText"));
  }

  auto r_input_message_text = process_input_message_text(
      td_->contacts_manager_.get(), DialogId(), std::move(input_message_content), td_->auth_manager_->is_bot());
  if (r_input_message_text.is_error()) {
    return promise.set_error(r_input_message_text.move_as_error());
  }
  InputMessageText input_message_text = r_input_message_text.move_as_ok();

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false, true);
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
  }

  int32 flags = 0;
  if (input_message_text.disable_web_page_preview) {
    flags |= SEND_MESSAGE_FLAG_DISABLE_WEB_PAGE_PREVIEW;
  }
  td_->create_handler<EditInlineMessageQuery>(std::move(promise))
      ->send(flags, std::move(input_bot_inline_message_id), input_message_text.text.text,
             get_input_message_entities(td_->contacts_manager_.get(), input_message_text.text.entities,
                                        "edit_inline_message_text"),
             nullptr, get_input_reply_markup(r_new_reply_markup.ok()));
}

void MessagesManager::edit_inline_message_live_location(const string &inline_message_id,
                                                        tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                        tl_object_ptr<td_api::location> &&input_location,
                                                        Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false, true);
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
  }

  Location location(input_location);
  if (location.empty() && input_location != nullptr) {
    return promise.set_error(Status::Error(400, "Wrong location specified"));
  }

  int32 flags = 0;
  if (location.empty()) {
    flags |= telegram_api::inputMediaGeoLive::STOPPED_MASK;
  }
  auto input_media = telegram_api::make_object<telegram_api::inputMediaGeoLive>(flags, false /*ignored*/,
                                                                                location.get_input_geo_point(), 0);
  td_->create_handler<EditInlineMessageQuery>(std::move(promise))
      ->send(0, std::move(input_bot_inline_message_id), "", vector<tl_object_ptr<telegram_api::MessageEntity>>(),
             std::move(input_media), get_input_reply_markup(r_new_reply_markup.ok()));
}

void MessagesManager::edit_inline_message_media(const string &inline_message_id,
                                                tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                tl_object_ptr<td_api::InputMessageContent> &&input_message_content,
                                                Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  if (input_message_content == nullptr) {
    return promise.set_error(Status::Error(5, "Can't edit message without new content"));
  }
  int32 new_message_content_type = input_message_content->get_id();
  if (new_message_content_type != td_api::inputMessageAnimation::ID &&
      new_message_content_type != td_api::inputMessageAudio::ID &&
      new_message_content_type != td_api::inputMessageDocument::ID &&
      new_message_content_type != td_api::inputMessagePhoto::ID &&
      new_message_content_type != td_api::inputMessageVideo::ID) {
    return promise.set_error(Status::Error(5, "Unsupported input message content type"));
  }

  auto r_input_message_content = process_input_message_content(DialogId(), std::move(input_message_content));
  if (r_input_message_content.is_error()) {
    return promise.set_error(r_input_message_content.move_as_error());
  }
  InputMessageContent content = r_input_message_content.move_as_ok();
  if (content.ttl > 0) {
    LOG(ERROR) << "Have message content with ttl " << content.ttl;
    return promise.set_error(Status::Error(5, "Can't enable self-destruction for media"));
  }

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false, true);
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
  }

  auto input_media = get_input_media(content.content.get(), td_, 0, true);
  if (input_media == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong message content specified"));
  }

  const FormattedText *caption = get_message_content_caption(content.content.get());
  td_->create_handler<EditInlineMessageQuery>(std::move(promise))
      ->send(1 << 11, std::move(input_bot_inline_message_id), caption == nullptr ? "" : caption->text,
             get_input_message_entities(td_->contacts_manager_.get(), caption, "edit_inline_message_media"),
             std::move(input_media), get_input_reply_markup(r_new_reply_markup.ok()));
}

void MessagesManager::edit_inline_message_caption(const string &inline_message_id,
                                                  tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                  tl_object_ptr<td_api::formattedText> &&input_caption,
                                                  Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  auto r_caption = process_input_caption(td_->contacts_manager_.get(), DialogId(), std::move(input_caption),
                                         td_->auth_manager_->is_bot());
  if (r_caption.is_error()) {
    return promise.set_error(r_caption.move_as_error());
  }
  auto caption = r_caption.move_as_ok();

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false, true);
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
  }

  td_->create_handler<EditInlineMessageQuery>(std::move(promise))
      ->send(1 << 11, std::move(input_bot_inline_message_id), caption.text,
             get_input_message_entities(td_->contacts_manager_.get(), caption.entities, "edit_inline_message_caption"),
             nullptr, get_input_reply_markup(r_new_reply_markup.ok()));
}

void MessagesManager::edit_inline_message_reply_markup(const string &inline_message_id,
                                                       tl_object_ptr<td_api::ReplyMarkup> &&reply_markup,
                                                       Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  auto r_new_reply_markup = get_reply_markup(std::move(reply_markup), td_->auth_manager_->is_bot(), true, false, true);
  if (r_new_reply_markup.is_error()) {
    return promise.set_error(r_new_reply_markup.move_as_error());
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
  }

  td_->create_handler<EditInlineMessageQuery>(std::move(promise))
      ->send(0, std::move(input_bot_inline_message_id), string(), vector<tl_object_ptr<telegram_api::MessageEntity>>(),
             nullptr, get_input_reply_markup(r_new_reply_markup.ok()));
}

void MessagesManager::edit_message_scheduling_state(
    FullMessageId full_message_id, td_api::object_ptr<td_api::MessageSchedulingState> &&scheduling_state,
    Promise<Unit> &&promise) {
  auto r_schedule_date = get_message_schedule_date(std::move(scheduling_state));
  if (r_schedule_date.is_error()) {
    return promise.set_error(r_schedule_date.move_as_error());
  }
  auto schedule_date = r_schedule_date.move_as_ok();

  LOG(INFO) << "Begin to reschedule " << full_message_id << " to " << schedule_date;

  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  Message *m = get_message_force(d, full_message_id.get_message_id(), "edit_message_scheduling_state");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  if (!m->message_id.is_scheduled()) {
    return promise.set_error(Status::Error(5, "Message is not scheduled"));
  }
  if (!m->message_id.is_scheduled_server()) {
    return promise.set_error(Status::Error(5, "Can't reschedule the message"));
  }

  if (get_message_schedule_date(m) == schedule_date) {
    return promise.set_value(Unit());
  }
  m->edited_schedule_date = schedule_date;

  if (schedule_date > 0) {
    send_closure(td_->create_net_actor<EditMessageActor>(std::move(promise)), &EditMessageActor::send, 0, dialog_id,
                 m->message_id, string(), vector<tl_object_ptr<telegram_api::MessageEntity>>(), nullptr, nullptr,
                 schedule_date, get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
  } else {
    send_closure(td_->create_net_actor<SendScheduledMessageActor>(std::move(promise)), &SendScheduledMessageActor::send,
                 dialog_id, m->message_id, get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
  }
}

int32 MessagesManager::get_message_flags(const Message *m) {
  int32 flags = 0;
  if (m->reply_to_message_id.is_valid()) {
    flags |= SEND_MESSAGE_FLAG_IS_REPLY;
  }
  if (m->disable_web_page_preview) {
    flags |= SEND_MESSAGE_FLAG_DISABLE_WEB_PAGE_PREVIEW;
  }
  if (m->reply_markup != nullptr) {
    flags |= SEND_MESSAGE_FLAG_HAS_REPLY_MARKUP;
  }
  if (m->disable_notification) {
    flags |= SEND_MESSAGE_FLAG_DISABLE_NOTIFICATION;
  }
  if (m->from_background) {
    flags |= SEND_MESSAGE_FLAG_FROM_BACKGROUND;
  }
  if (m->clear_draft) {
    flags |= SEND_MESSAGE_FLAG_CLEAR_DRAFT;
  }
  if (m->message_id.is_scheduled()) {
    flags |= SEND_MESSAGE_FLAG_HAS_SCHEDULE_DATE;
  }
  return flags;
}

bool MessagesManager::can_set_game_score(DialogId dialog_id, const Message *m) const {
  if (m == nullptr) {
    return false;
  }
  if (m->message_id.is_scheduled()) {
    return false;
  }
  if (m->message_id.is_yet_unsent()) {
    return false;
  }
  if (m->message_id.is_local()) {
    return false;
  }
  if (m->via_bot_user_id.is_valid() && m->via_bot_user_id != td_->contacts_manager_->get_my_id()) {
    return false;
  }

  if (!td_->auth_manager_->is_bot()) {
    return false;
  }
  if (m->reply_markup == nullptr || m->reply_markup->type != ReplyMarkup::Type::InlineKeyboard ||
      m->reply_markup->inline_keyboard.empty()) {
    return false;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
      if (!m->is_outgoing && dialog_id != get_my_dialog_id()) {
        return false;
      }
      break;
    case DialogType::Chat:
      if (!m->is_outgoing) {
        return false;
      }
      break;
    case DialogType::Channel: {
      if (m->via_bot_user_id.is_valid()) {
        // outgoing via_bot messages can always be edited
        break;
      }
      auto channel_id = dialog_id.get_channel_id();
      auto channel_status = td_->contacts_manager_->get_channel_permissions(channel_id);
      if (m->is_channel_post) {
        if (!channel_status.can_edit_messages() && !(channel_status.can_post_messages() && m->is_outgoing)) {
          return false;
        }
      } else {
        if (!m->is_outgoing) {
          return false;
        }
      }
      break;
    }
    case DialogType::SecretChat:
      return false;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }

  return m->content->get_type() == MessageContentType::Game;
}

void MessagesManager::set_game_score(FullMessageId full_message_id, bool edit_message, UserId user_id, int32 score,
                                     bool force, Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  LOG(INFO) << "Begin to set game score of " << user_id << " in " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  if (!have_input_peer(dialog_id, AccessRights::Edit)) {
    return promise.set_error(Status::Error(5, "Can't access the chat"));
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "set_game_score");
  if (m == nullptr) {
    return promise.set_error(Status::Error(5, "Message not found"));
  }

  auto input_user = td_->contacts_manager_->get_input_user(user_id);
  if (input_user == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong user identifier specified"));
  }

  if (!can_set_game_score(dialog_id, m)) {
    return promise.set_error(Status::Error(5, "Game score can't be set"));
  }

  send_closure(td_->create_net_actor<SetGameScoreActor>(std::move(promise)), &SetGameScoreActor::send, dialog_id,
               m->message_id, edit_message, std::move(input_user), score, force,
               get_sequence_dispatcher_id(dialog_id, MessageContentType::None));
}

void MessagesManager::set_inline_game_score(const string &inline_message_id, bool edit_message, UserId user_id,
                                            int32 score, bool force, Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    return promise.set_error(Status::Error(3, "Method is available only for bots"));
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
  }

  auto input_user = td_->contacts_manager_->get_input_user(user_id);
  if (input_user == nullptr) {
    return promise.set_error(Status::Error(400, "Wrong user identifier specified"));
  }

  td_->create_handler<SetInlineGameScoreQuery>(std::move(promise))
      ->send(std::move(input_bot_inline_message_id), edit_message, std::move(input_user), score, force);
}

int64 MessagesManager::get_game_high_scores(FullMessageId full_message_id, UserId user_id, Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    promise.set_error(Status::Error(3, "Method is available only for bots"));
    return 0;
  }

  LOG(INFO) << "Begin to get game high scores of " << user_id << " in " << full_message_id;
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    promise.set_error(Status::Error(5, "Chat not found"));
    return 0;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    promise.set_error(Status::Error(5, "Can't access the chat"));
    return 0;
  }

  const Message *m = get_message_force(d, full_message_id.get_message_id(), "get_game_high_scores");
  if (m == nullptr) {
    promise.set_error(Status::Error(5, "Message not found"));
    return 0;
  }
  if (m->message_id.is_scheduled() || !m->message_id.is_server()) {
    promise.set_error(Status::Error(5, "Wrong message identifier specified"));
    return 0;
  }

  auto input_user = td_->contacts_manager_->get_input_user(user_id);
  if (input_user == nullptr) {
    promise.set_error(Status::Error(400, "Wrong user identifier specified"));
    return 0;
  }

  int64 random_id = 0;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || game_high_scores_.find(random_id) != game_high_scores_.end());
  game_high_scores_[random_id];  // reserve place for result

  td_->create_handler<GetGameHighScoresQuery>(std::move(promise))
      ->send(dialog_id, m->message_id, std::move(input_user), random_id);
  return random_id;
}

int64 MessagesManager::get_inline_game_high_scores(const string &inline_message_id, UserId user_id,
                                                   Promise<Unit> &&promise) {
  if (!td_->auth_manager_->is_bot()) {
    promise.set_error(Status::Error(3, "Method is available only for bots"));
    return 0;
  }

  auto input_bot_inline_message_id = td_->inline_queries_manager_->get_input_bot_inline_message_id(inline_message_id);
  if (input_bot_inline_message_id == nullptr) {
    promise.set_error(Status::Error(400, "Wrong inline message identifier specified"));
    return 0;
  }

  auto input_user = td_->contacts_manager_->get_input_user(user_id);
  if (input_user == nullptr) {
    promise.set_error(Status::Error(400, "Wrong user identifier specified"));
    return 0;
  }

  int64 random_id = 0;
  do {
    random_id = Random::secure_int64();
  } while (random_id == 0 || game_high_scores_.find(random_id) != game_high_scores_.end());
  game_high_scores_[random_id];  // reserve place for result

  td_->create_handler<GetInlineGameHighScoresQuery>(std::move(promise))
      ->send(std::move(input_bot_inline_message_id), std::move(input_user), random_id);
  return random_id;
}

void MessagesManager::on_get_game_high_scores(int64 random_id,
                                              tl_object_ptr<telegram_api::messages_highScores> &&high_scores) {
  auto it = game_high_scores_.find(random_id);
  CHECK(it != game_high_scores_.end());
  auto &result = it->second;
  CHECK(result == nullptr);

  if (high_scores == nullptr) {
    game_high_scores_.erase(it);
    return;
  }

  td_->contacts_manager_->on_get_users(std::move(high_scores->users_), "on_get_game_high_scores");

  result = make_tl_object<td_api::gameHighScores>();

  for (auto &high_score : high_scores->scores_) {
    int32 position = high_score->pos_;
    if (position <= 0) {
      LOG(ERROR) << "Receive wrong position = " << position;
      continue;
    }
    UserId user_id(high_score->user_id_);
    LOG_IF(ERROR, !td_->contacts_manager_->have_user(user_id)) << "Have no info about " << user_id;
    int32 score = high_score->score_;
    if (score < 0) {
      LOG(ERROR) << "Receive wrong score = " << score;
      continue;
    }
    result->scores_.push_back(make_tl_object<td_api::gameHighScore>(
        position, td_->contacts_manager_->get_user_id_object(user_id, "gameHighScore"), score));
  }
}

tl_object_ptr<td_api::gameHighScores> MessagesManager::get_game_high_scores_object(int64 random_id) {
  auto it = game_high_scores_.find(random_id);
  CHECK(it != game_high_scores_.end());
  auto result = std::move(it->second);
  game_high_scores_.erase(it);
  return result;
}

bool MessagesManager::is_forward_info_sender_hidden(const MessageForwardInfo *forward_info) {
  if (!forward_info->sender_name.empty()) {
    return true;
  }
  DialogId hidden_sender_dialog_id(static_cast<int64>(G()->is_test_dc() ? -1000010460537ll : -1001228946795ll));
  return forward_info->dialog_id == hidden_sender_dialog_id && !forward_info->author_signature.empty() &&
         !forward_info->message_id.is_valid();
}

unique_ptr<MessagesManager::MessageForwardInfo> MessagesManager::get_message_forward_info(
    tl_object_ptr<telegram_api::messageFwdHeader> &&forward_header) {
  if (forward_header == nullptr) {
    return nullptr;
  }

  if (forward_header->date_ <= 0) {
    LOG(ERROR) << "Wrong date in message forward header: " << oneline(to_string(forward_header));
    return nullptr;
  }

  auto flags = forward_header->flags_;
  UserId sender_user_id;
  ChannelId channel_id;
  MessageId message_id;
  string author_signature;
  DialogId from_dialog_id;
  MessageId from_message_id;
  string sender_name;
  if ((flags & telegram_api::messageFwdHeader::FROM_ID_MASK) != 0) {
    sender_user_id = UserId(forward_header->from_id_);
    if (!sender_user_id.is_valid()) {
      LOG(ERROR) << "Receive invalid sender id in message forward header: " << oneline(to_string(forward_header));
      sender_user_id = UserId();
    }
  }
  if ((flags & telegram_api::messageFwdHeader::CHANNEL_ID_MASK) != 0) {
    channel_id = ChannelId(forward_header->channel_id_);
    if (!channel_id.is_valid()) {
      LOG(ERROR) << "Receive invalid channel id in message forward header: " << oneline(to_string(forward_header));
    }
  }
  constexpr int32 MESSAGE_FORWARD_HEADER_FLAG_HAS_MESSAGE_ID = telegram_api::messageFwdHeader::CHANNEL_POST_MASK;
  if ((flags & MESSAGE_FORWARD_HEADER_FLAG_HAS_MESSAGE_ID) != 0) {
    message_id = MessageId(ServerMessageId(forward_header->channel_post_));
    if (!message_id.is_valid()) {
      LOG(ERROR) << "Receive " << message_id << " in message forward header: " << oneline(to_string(forward_header));
      message_id = MessageId();
    }
  }
  constexpr int32 MESSAGE_FORWARD_HEADER_FLAG_HAS_AUTHOR_SIGNATURE = telegram_api::messageFwdHeader::POST_AUTHOR_MASK;
  if ((flags & MESSAGE_FORWARD_HEADER_FLAG_HAS_AUTHOR_SIGNATURE) != 0) {
    author_signature = std::move(forward_header->post_author_);
  }
  if ((flags & telegram_api::messageFwdHeader::SAVED_FROM_PEER_MASK) != 0) {
    from_dialog_id = DialogId(forward_header->saved_from_peer_);
    from_message_id = MessageId(ServerMessageId(forward_header->saved_from_msg_id_));
    if (!from_dialog_id.is_valid() || !from_message_id.is_valid()) {
      LOG(ERROR) << "Receive " << from_message_id << " in " << from_dialog_id
                 << " in message forward header: " << oneline(to_string(forward_header));
      from_dialog_id = DialogId();
      from_message_id = MessageId();
    }
  }
  if ((flags & telegram_api::messageFwdHeader::FROM_NAME_MASK) != 0) {
    sender_name = std::move(forward_header->from_name_);
  }

  DialogId dialog_id;
  if (!channel_id.is_valid()) {
    if (sender_user_id.is_valid()) {
      if (message_id.is_valid()) {
        LOG(ERROR) << "Receive non-empty message identifier in message forward header: "
                   << oneline(to_string(forward_header));
        message_id = MessageId();
      }
    } else if (sender_name.empty()) {
      LOG(ERROR) << "Receive wrong message forward header: " << oneline(to_string(forward_header));
      return nullptr;
    }
  } else {
    LOG_IF(ERROR, td_->contacts_manager_->have_min_channel(channel_id)) << "Receive forward from min channel";
    dialog_id = DialogId(channel_id);
    force_create_dialog(dialog_id, "message forward info", true);
    if (sender_user_id.is_valid()) {
      LOG(ERROR) << "Receive valid sender user id in message forward header: " << oneline(to_string(forward_header));
      sender_user_id = UserId();
    }
  }
  if (from_dialog_id.is_valid()) {
    force_create_dialog(from_dialog_id, "message forward from info", true);
  }

  return td::make_unique<MessageForwardInfo>(sender_user_id, forward_header->date_, dialog_id, message_id,
                                             std::move(author_signature), std::move(sender_name), from_dialog_id,
                                             from_message_id);
}

td_api::object_ptr<td_api::messageForwardInfo> MessagesManager::get_message_forward_info_object(
    const unique_ptr<MessageForwardInfo> &forward_info) const {
  if (forward_info == nullptr) {
    return nullptr;
  }

  auto origin = [&]() -> td_api::object_ptr<td_api::MessageForwardOrigin> {
    if (is_forward_info_sender_hidden(forward_info.get())) {
      return td_api::make_object<td_api::messageForwardOriginHiddenUser>(
          forward_info->sender_name.empty() ? forward_info->author_signature : forward_info->sender_name);
    }
    if (forward_info->dialog_id.is_valid()) {
      return td_api::make_object<td_api::messageForwardOriginChannel>(
          forward_info->dialog_id.get(), forward_info->message_id.get(), forward_info->author_signature);
    }
    return td_api::make_object<td_api::messageForwardOriginUser>(
        td_->contacts_manager_->get_user_id_object(forward_info->sender_user_id, "messageForwardOriginUser"));
  }();

  return td_api::make_object<td_api::messageForwardInfo>(
      std::move(origin), forward_info->date, forward_info->from_dialog_id.get(), forward_info->from_message_id.get());
}

Result<unique_ptr<ReplyMarkup>> MessagesManager::get_dialog_reply_markup(
    DialogId dialog_id, tl_object_ptr<td_api::ReplyMarkup> &&reply_markup_ptr) const {
  if (reply_markup_ptr == nullptr) {
    return nullptr;
  }

  auto dialog_type = dialog_id.get_type();
  bool is_broadcast = is_broadcast_channel(dialog_id);

  bool only_inline_keyboard = is_broadcast;
  bool request_buttons_allowed = dialog_type == DialogType::User;
  bool switch_inline_buttons_allowed = !is_broadcast;

  TRY_RESULT(reply_markup,
             get_reply_markup(std::move(reply_markup_ptr), td_->auth_manager_->is_bot(), only_inline_keyboard,
                              request_buttons_allowed, switch_inline_buttons_allowed));
  if (reply_markup == nullptr) {
    return nullptr;
  }

  switch (dialog_type) {
    case DialogType::User:
      if (reply_markup->type != ReplyMarkup::Type::InlineKeyboard) {
        reply_markup->is_personal = false;
      }
      break;
    case DialogType::Channel:
    case DialogType::Chat:
    case DialogType::SecretChat:
    case DialogType::None:
      // nothing special
      break;
    default:
      UNREACHABLE();
  }

  return std::move(reply_markup);
}

class MessagesManager::ForwardMessagesLogEvent {
 public:
  DialogId to_dialog_id;
  DialogId from_dialog_id;
  vector<MessageId> message_ids;
  vector<Message *> messages_in;
  vector<unique_ptr<Message>> messages_out;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(to_dialog_id, storer);
    td::store(from_dialog_id, storer);
    td::store(message_ids, storer);

    td::store(narrow_cast<int32>(messages_in.size()), storer);
    for (auto m : messages_in) {
      td::store(*m, storer);
    }
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(to_dialog_id, parser);
    td::parse(from_dialog_id, parser);
    td::parse(message_ids, parser);

    CHECK(messages_out.empty());
    int32 size = parser.fetch_int();
    messages_out.resize(size);
    for (auto &m_out : messages_out) {
      td::parse(m_out, parser);
    }
  }
};

uint64 MessagesManager::save_forward_messages_logevent(DialogId to_dialog_id, DialogId from_dialog_id,
                                                       const vector<Message *> &messages,
                                                       const vector<MessageId> &message_ids) {
  ForwardMessagesLogEvent logevent{to_dialog_id, from_dialog_id, message_ids, messages, Auto()};
  auto storer = LogEventStorerImpl<ForwardMessagesLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::ForwardMessages, storer);
}

void MessagesManager::do_forward_messages(DialogId to_dialog_id, DialogId from_dialog_id,
                                          const vector<Message *> &messages, const vector<MessageId> &message_ids,
                                          uint64 logevent_id) {
  CHECK(messages.size() == message_ids.size());
  if (messages.empty()) {
    return;
  }

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_forward_messages_logevent(to_dialog_id, from_dialog_id, messages, message_ids);
  }

  auto schedule_date = get_message_schedule_date(messages[0]);

  int32 flags = 0;
  if (messages[0]->disable_notification) {
    flags |= SEND_MESSAGE_FLAG_DISABLE_NOTIFICATION;
  }
  if (messages[0]->from_background) {
    flags |= SEND_MESSAGE_FLAG_FROM_BACKGROUND;
  }
  if (messages[0]->media_album_id != 0) {
    flags |= SEND_MESSAGE_FLAG_GROUP_MEDIA;
  }
  if (messages[0]->in_game_share) {
    flags |= SEND_MESSAGE_FLAG_WITH_MY_SCORE;
  }
  if (schedule_date != 0) {
    flags |= SEND_MESSAGE_FLAG_HAS_SCHEDULE_DATE;
  }

  vector<int64> random_ids =
      transform(messages, [this, to_dialog_id](const Message *m) { return begin_send_message(to_dialog_id, m); });
  send_closure(td_->create_net_actor<ForwardMessagesActor>(get_erase_logevent_promise(logevent_id)),
               &ForwardMessagesActor::send, flags, to_dialog_id, from_dialog_id, message_ids, std::move(random_ids),
               schedule_date, get_sequence_dispatcher_id(to_dialog_id, MessageContentType::None));
}

Result<MessageId> MessagesManager::forward_message(DialogId to_dialog_id, DialogId from_dialog_id, MessageId message_id,
                                                   tl_object_ptr<td_api::sendMessageOptions> &&options,
                                                   bool in_game_share, bool send_copy, bool remove_caption) {
  TRY_RESULT(result, forward_messages(to_dialog_id, from_dialog_id, {message_id}, std::move(options), in_game_share,
                                      false, send_copy, remove_caption));
  CHECK(result.size() == 1);
  auto sent_message_id = result[0];
  if (sent_message_id == MessageId()) {
    return Status::Error(11, "Message can't be forwarded");
  }
  return sent_message_id;
}

Result<vector<MessageId>> MessagesManager::forward_messages(DialogId to_dialog_id, DialogId from_dialog_id,
                                                            vector<MessageId> message_ids,
                                                            tl_object_ptr<td_api::sendMessageOptions> &&options,
                                                            bool in_game_share, bool as_album, bool send_copy,
                                                            bool remove_caption) {
  if (message_ids.size() > 100) {  // TODO replace with const from config or implement mass-forward
    return Status::Error(4, "Too much messages to forward");
  }
  if (message_ids.empty()) {
    return Status::Error(4, "There are no messages to forward");
  }

  Dialog *from_dialog = get_dialog_force(from_dialog_id);
  if (from_dialog == nullptr) {
    return Status::Error(5, "Chat to forward messages from not found");
  }
  if (!have_input_peer(from_dialog_id, AccessRights::Read)) {
    return Status::Error(5, "Can't access the chat to forward messages from");
  }
  if (from_dialog_id.get_type() == DialogType::SecretChat) {
    return Status::Error(7, "Can't forward messages from secret chats");
  }

  Dialog *to_dialog = get_dialog_force(to_dialog_id);
  if (to_dialog == nullptr) {
    return Status::Error(5, "Chat to forward messages to not found");
  }

  TRY_STATUS(can_send_message(to_dialog_id));
  TRY_RESULT(send_message_options, process_send_message_options(to_dialog_id, std::move(options)));

  for (auto message_id : message_ids) {
    if (message_id.is_valid_scheduled()) {
      return Status::Error(5, "Can't forward scheduled messages");
    }
    if (!message_id.is_valid()) {
      return Status::Error(5, "Invalid message identifier");
    }
    CHECK(!message_id.is_scheduled());
  }

  bool to_secret = to_dialog_id.get_type() == DialogType::SecretChat;

  vector<MessageId> result(message_ids.size());
  vector<Message *> forwarded_messages;
  vector<MessageId> forwarded_message_ids;

  struct CopiedMessage {
    unique_ptr<MessageContent> content;
    bool disable_web_page_preview;
    size_t index;
  };
  vector<CopiedMessage> copied_messages;

  auto my_id = td_->contacts_manager_->get_my_id();
  bool need_update_dialog_pos = false;
  for (size_t i = 0; i < message_ids.size(); i++) {
    MessageId message_id = get_persistent_message_id(from_dialog, message_ids[i]);

    const Message *forwarded_message = get_message_force(from_dialog, message_id, "forward_messages");
    if (forwarded_message == nullptr) {
      LOG(INFO) << "Can't find " << message_id << " to forward";
      continue;
    }
    CHECK(message_id.is_valid());
    CHECK(message_id == forwarded_message->message_id);

    if (!can_forward_message(from_dialog_id, forwarded_message)) {
      LOG(INFO) << "Can't forward " << message_id;
      continue;
    }

    bool need_copy = !message_id.is_server() || to_secret || send_copy;
    auto type = need_copy ? (remove_caption ? MessageContentDupType::CopyWithoutCaption : MessageContentDupType::Copy)
                          : MessageContentDupType::Forward;
    unique_ptr<MessageContent> content = dup_message_content(td_, to_dialog_id, forwarded_message->content.get(), type);
    if (content == nullptr) {
      LOG(INFO) << "Can't forward " << message_id;
      continue;
    }

    auto can_send_status = can_send_message_content(to_dialog_id, content.get(), !need_copy);
    if (can_send_status.is_error()) {
      LOG(INFO) << "Can't forward " << message_id << ": " << can_send_status.message();
      continue;
    }

    auto can_use_options_status = can_use_send_message_options(send_message_options, content, 0);
    if (can_use_options_status.is_error()) {
      LOG(INFO) << "Can't forward " << message_id << ": " << can_send_status.message();
      continue;
    }

    if (need_copy) {
      copied_messages.push_back({std::move(content), forwarded_message->disable_web_page_preview, i});
      continue;
    }

    auto content_type = content->get_type();
    bool is_game = content_type == MessageContentType::Game;
    unique_ptr<MessageForwardInfo> forward_info;
    if (!is_game && content_type != MessageContentType::Audio) {
      DialogId saved_from_dialog_id;
      MessageId saved_from_message_id;
      if (to_dialog_id == DialogId(my_id)) {
        saved_from_dialog_id = from_dialog_id;
        saved_from_message_id = message_id;
      }

      if (forwarded_message->forward_info != nullptr) {
        forward_info = make_unique<MessageForwardInfo>(*forwarded_message->forward_info);
        forward_info->from_dialog_id = saved_from_dialog_id;
        forward_info->from_message_id = saved_from_message_id;
      } else {
        if (from_dialog_id != DialogId(my_id)) {
          if (forwarded_message->is_channel_post) {
            if (is_broadcast_channel(from_dialog_id)) {
              auto author_signature = forwarded_message->sender_user_id.is_valid()
                                          ? td_->contacts_manager_->get_user_title(forwarded_message->sender_user_id)
                                          : forwarded_message->author_signature;
              forward_info = td::make_unique<MessageForwardInfo>(
                  UserId(), forwarded_message->date, from_dialog_id, forwarded_message->message_id,
                  std::move(author_signature), "", saved_from_dialog_id, saved_from_message_id);
            } else {
              LOG(ERROR) << "Don't know how to forward a channel post not from a channel";
            }
          } else if (forwarded_message->sender_user_id.is_valid()) {
            forward_info =
                make_unique<MessageForwardInfo>(forwarded_message->sender_user_id, forwarded_message->date, DialogId(),
                                                MessageId(), "", "", saved_from_dialog_id, saved_from_message_id);
          } else {
            LOG(ERROR) << "Don't know how to forward a non-channel post message without forward info and sender";
          }
        }
      }
    }

    Message *m = get_message_to_send(to_dialog, MessageId(), send_message_options, std::move(content),
                                     &need_update_dialog_pos, std::move(forward_info));
    m->real_forward_from_dialog_id = from_dialog_id;
    m->real_forward_from_message_id = message_id;
    m->via_bot_user_id = forwarded_message->via_bot_user_id;
    m->in_game_share = in_game_share;
    if (forwarded_message->views > 0 && m->forward_info != nullptr) {
      m->views = forwarded_message->views;
    }

    if (is_game) {
      if (m->via_bot_user_id == my_id) {
        m->via_bot_user_id = UserId();
      } else if (m->via_bot_user_id == UserId()) {
        m->via_bot_user_id = forwarded_message->sender_user_id;
      }
    }
    if (!to_secret && forwarded_message->reply_markup != nullptr &&
        forwarded_message->reply_markup->type == ReplyMarkup::Type::InlineKeyboard) {
      bool need_reply_markup = true;
      for (auto &row : forwarded_message->reply_markup->inline_keyboard) {
        for (auto &button : row) {
          if (button.type == InlineKeyboardButton::Type::Url || button.type == InlineKeyboardButton::Type::UrlAuth) {
            // ok
            continue;
          }
          if (m->via_bot_user_id.is_valid() && (button.type == InlineKeyboardButton::Type::SwitchInline ||
                                                button.type == InlineKeyboardButton::Type::SwitchInlineCurrentDialog)) {
            // ok
            continue;
          }

          need_reply_markup = false;
        }
      }
      if (need_reply_markup) {
        m->reply_markup = make_unique<ReplyMarkup>(*forwarded_message->reply_markup);
        for (auto &row : m->reply_markup->inline_keyboard) {
          for (auto &button : row) {
            if (button.type == InlineKeyboardButton::Type::SwitchInlineCurrentDialog) {
              button.type = InlineKeyboardButton::Type::SwitchInline;
            }
            if (!button.forward_text.empty()) {
              button.text = std::move(button.forward_text);
              button.forward_text.clear();
            }
          }
        }
      }
    }

    result[i] = m->message_id;
    forwarded_messages.push_back(m);
    forwarded_message_ids.push_back(message_id);
  }

  if (!forwarded_messages.empty()) {
    if (as_album && forwarded_messages.size() > 1 && forwarded_messages.size() <= MAX_GROUPED_MESSAGES) {
      bool allow_album = true;
      for (auto m : forwarded_messages) {
        if (!is_allowed_media_group_content(m->content->get_type())) {
          allow_album = false;
          break;
        }
      }

      if (allow_album) {
        int64 media_album_id = generate_new_media_album_id();
        for (auto m : forwarded_messages) {
          m->media_album_id = media_album_id;
        }
      }
    }

    for (auto m : forwarded_messages) {
      send_update_new_message(to_dialog, m);
    }

    do_forward_messages(to_dialog_id, from_dialog_id, forwarded_messages, forwarded_message_ids, 0);
  }

  if (!copied_messages.empty()) {
    int64 media_album_id = 0;
    if (as_album && copied_messages.size() > 1 && copied_messages.size() <= MAX_GROUPED_MESSAGES) {
      bool allow_album = true;
      for (auto &copied_message : copied_messages) {
        if (!is_allowed_media_group_content(copied_message.content->get_type())) {
          allow_album = false;
          break;
        }
      }

      if (allow_album) {
        media_album_id = generate_new_media_album_id();
      }
    }

    for (auto &copied_message : copied_messages) {
      Message *m = get_message_to_send(to_dialog, MessageId(), send_message_options, std::move(copied_message.content),
                                       &need_update_dialog_pos, nullptr, true);
      m->disable_web_page_preview = copied_message.disable_web_page_preview;
      m->media_album_id = media_album_id;

      save_send_message_logevent(to_dialog_id, m);
      do_send_message(to_dialog_id, m);
      result[copied_message.index] = m->message_id;

      send_update_new_message(to_dialog, m);
    }
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(to_dialog, "forward_messages");
  }

  return result;
}

Result<vector<MessageId>> MessagesManager::resend_messages(DialogId dialog_id, vector<MessageId> message_ids) {
  if (message_ids.empty()) {
    return Status::Error(4, "There are no messages to resend");
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(400, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));

  MessageId last_message_id;
  for (auto &message_id : message_ids) {
    message_id = get_persistent_message_id(d, message_id);
    const Message *m = get_message_force(d, message_id, "resend_messages");
    if (m == nullptr) {
      return Status::Error(400, "Message not found");
    }
    if (!m->is_failed_to_send) {
      return Status::Error(400, "Message is not failed to send");
    }
    if (!can_resend_message(m)) {
      return Status::Error(400, "Message can't be re-sent");
    }
    if (m->try_resend_at > Time::now()) {
      return Status::Error(400, "Message can't be re-sent yet");
    }
    if (last_message_id != MessageId()) {
      if (m->message_id.is_scheduled() != last_message_id.is_scheduled()) {
        return Status::Error(400, "Messages must be all scheduled or ordinary");
      }
      if (m->message_id <= last_message_id) {
        return Status::Error(400, "Message identifiers must be in a strictly increasing order");
      }
    }
    last_message_id = m->message_id;
  }

  vector<unique_ptr<MessageContent>> new_contents(message_ids.size());
  std::unordered_map<int64, std::pair<int64, int32>> new_media_album_ids;
  for (size_t i = 0; i < message_ids.size(); i++) {
    MessageId message_id = message_ids[i];
    const Message *m = get_message(d, message_id);
    CHECK(m != nullptr);

    unique_ptr<MessageContent> content =
        dup_message_content(td_, dialog_id, m->content.get(), MessageContentDupType::Send);
    if (content == nullptr) {
      LOG(INFO) << "Can't resend " << m->message_id;
      continue;
    }

    auto can_send_status = can_send_message_content(dialog_id, content.get(), false);
    if (can_send_status.is_error()) {
      LOG(INFO) << "Can't resend " << m->message_id << ": " << can_send_status.message();
      continue;
    }

    new_contents[i] = std::move(content);

    if (m->media_album_id != 0) {
      auto &new_media_album_id = new_media_album_ids[m->media_album_id];
      new_media_album_id.second++;
      if (new_media_album_id.second == 2) {  // have at least 2 messages in the new album
        CHECK(new_media_album_id.first == 0);
        new_media_album_id.first = generate_new_media_album_id();
      }
      if (new_media_album_id.second == MAX_GROUPED_MESSAGES + 1) {
        CHECK(new_media_album_id.first != 0);
        new_media_album_id.first = 0;  // just in case
      }
    }
  }

  vector<MessageId> result(message_ids.size());
  bool need_update_dialog_pos = false;
  for (size_t i = 0; i < message_ids.size(); i++) {
    if (new_contents[i] == nullptr) {
      continue;
    }

    unique_ptr<Message> message = delete_message(d, message_ids[i], true, &need_update_dialog_pos, "resend_messages");
    CHECK(message != nullptr);
    send_update_delete_messages(dialog_id, {message->message_id.get()}, true, false);

    SendMessageOptions options(message->disable_notification, message->from_background,
                               get_message_schedule_date(message.get()));
    Message *m = get_message_to_send(d, get_reply_to_message_id(d, message->reply_to_message_id), options,
                                     std::move(new_contents[i]), &need_update_dialog_pos, nullptr, message->is_copy);
    m->reply_markup = std::move(message->reply_markup);
    m->via_bot_user_id = message->via_bot_user_id;
    m->disable_web_page_preview = message->disable_web_page_preview;
    m->clear_draft = false;  // never clear draft in resend
    m->ttl = message->ttl;
    m->is_content_secret = message->is_content_secret;
    m->media_album_id = new_media_album_ids[message->media_album_id].first;

    save_send_message_logevent(dialog_id, m);
    do_send_message(dialog_id, m);

    send_update_new_message(d, m);

    result[i] = m->message_id;
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "resend_messages");
  }

  return result;
}

Result<MessageId> MessagesManager::send_dialog_set_ttl_message(DialogId dialog_id, int32 ttl) {
  if (dialog_id.get_type() != DialogType::SecretChat) {
    return Status::Error(5, "Can't set chat ttl in non-secret chat");
  }

  if (ttl < 0) {
    return Status::Error(5, "Message ttl can't be negative");
  }

  LOG(INFO) << "Begin to set ttl in " << dialog_id << " to " << ttl;

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));
  bool need_update_dialog_pos = false;
  Message *m = get_message_to_send(d, MessageId(), SendMessageOptions(), create_chat_set_ttl_message_content(ttl),
                                   &need_update_dialog_pos);

  send_update_new_message(d, m);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "send_dialog_set_ttl_message");
  }

  int64 random_id = begin_send_message(dialog_id, m);

  send_closure(td_->secret_chats_manager_, &SecretChatsManager::send_set_ttl_message, dialog_id.get_secret_chat_id(),
               ttl, random_id, Promise<>());  // TODO Promise

  return m->message_id;
}

Status MessagesManager::send_screenshot_taken_notification_message(DialogId dialog_id) {
  auto dialog_type = dialog_id.get_type();
  if (dialog_type != DialogType::User && dialog_type != DialogType::SecretChat) {
    return Status::Error(5, "Notification about taken screenshot can be sent only in private and secret chats");
  }

  LOG(INFO) << "Begin to send notification about taken screenshot in " << dialog_id;

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  TRY_STATUS(can_send_message(dialog_id));

  if (dialog_type == DialogType::User) {
    bool need_update_dialog_pos = false;
    const Message *m = get_message_to_send(d, MessageId(), SendMessageOptions(),
                                           create_screenshot_taken_message_content(), &need_update_dialog_pos);

    do_send_screenshot_taken_notification_message(dialog_id, m, 0);

    send_update_new_message(d, m);
    if (need_update_dialog_pos) {
      send_update_chat_last_message(d, "send_screenshot_taken_notification_message");
    }
  } else {
    send_closure(td_->secret_chats_manager_, &SecretChatsManager::notify_screenshot_taken,
                 dialog_id.get_secret_chat_id(),
                 Promise<>());  // TODO Promise
  }

  return Status::OK();
}

class MessagesManager::SendScreenshotTakenNotificationMessageLogEvent {
 public:
  DialogId dialog_id;
  const Message *m_in = nullptr;
  unique_ptr<Message> m_out;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id, storer);
    td::store(*m_in, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id, parser);
    td::parse(m_out, parser);
  }
};

uint64 MessagesManager::save_send_screenshot_taken_notification_message_logevent(DialogId dialog_id, const Message *m) {
  if (!G()->parameters().use_message_db) {
    return 0;
  }

  CHECK(m != nullptr);
  LOG(INFO) << "Save " << FullMessageId(dialog_id, m->message_id) << " to binlog";
  SendScreenshotTakenNotificationMessageLogEvent logevent;
  logevent.dialog_id = dialog_id;
  logevent.m_in = m;
  auto storer = LogEventStorerImpl<SendScreenshotTakenNotificationMessageLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SendScreenshotTakenNotificationMessage, storer);
}

void MessagesManager::do_send_screenshot_taken_notification_message(DialogId dialog_id, const Message *m,
                                                                    uint64 logevent_id) {
  LOG(INFO) << "Do send screenshot taken notification " << FullMessageId(dialog_id, m->message_id);
  CHECK(dialog_id.get_type() == DialogType::User);

  if (logevent_id == 0) {
    logevent_id = save_send_screenshot_taken_notification_message_logevent(dialog_id, m);
  }

  int64 random_id = begin_send_message(dialog_id, m);
  td_->create_handler<SendScreenshotNotificationQuery>(get_erase_logevent_promise(logevent_id))
      ->send(dialog_id, random_id);
}

Result<MessageId> MessagesManager::add_local_message(
    DialogId dialog_id, UserId sender_user_id, MessageId reply_to_message_id, bool disable_notification,
    tl_object_ptr<td_api::InputMessageContent> &&input_message_content) {
  if (input_message_content == nullptr) {
    return Status::Error(5, "Can't add local message without content");
  }

  LOG(INFO) << "Begin to add local message to " << dialog_id << " in reply to " << reply_to_message_id;
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(5, "Chat not found");
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return Status::Error(400, "Can't access the chat");
  }
  TRY_RESULT(message_content, process_input_message_content(dialog_id, std::move(input_message_content)));
  if (message_content.content->get_type() == MessageContentType::Poll) {
    return Status::Error(400, "Can't add local poll message");
  }
  if (message_content.content->get_type() == MessageContentType::Game) {
    return Status::Error(400, "Can't add local game message");
  }

  bool is_channel_post = is_broadcast_channel(dialog_id);
  if (sender_user_id != UserId() && !td_->contacts_manager_->have_user_force(sender_user_id)) {
    return Status::Error(400, "User not found");
  }

  auto dialog_type = dialog_id.get_type();
  auto my_id = td_->contacts_manager_->get_my_id();
  if (sender_user_id != my_id) {
    if (dialog_type == DialogType::User && DialogId(sender_user_id) != dialog_id) {
      return Status::Error(400, "Wrong sender user");
    }
    if (dialog_type == DialogType::SecretChat) {
      auto peer_user_id = td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
      if (!peer_user_id.is_valid() || sender_user_id != peer_user_id) {
        return Status::Error(400, "Wrong sender user");
      }
    }
  }

  MessageId message_id = get_next_local_message_id(d);

  auto m = make_unique<Message>();
  set_message_id(m, message_id);
  if (is_channel_post) {
    // sender of the post can be hidden
    if (td_->contacts_manager_->get_channel_sign_messages(dialog_id.get_channel_id())) {
      m->author_signature = td_->contacts_manager_->get_user_title(sender_user_id);
    }
  } else {
    m->sender_user_id = sender_user_id;
  }
  m->date = G()->unix_time();
  m->reply_to_message_id = get_reply_to_message_id(d, reply_to_message_id);
  m->is_channel_post = is_channel_post;
  m->is_outgoing = dialog_id != DialogId(my_id) && sender_user_id == my_id;
  m->disable_notification = disable_notification;
  m->from_background = false;
  m->views = 0;
  m->content = std::move(message_content.content);
  m->disable_web_page_preview = message_content.disable_web_page_preview;
  m->clear_draft = message_content.clear_draft;
  if (dialog_type == DialogType::SecretChat) {
    m->ttl = td_->contacts_manager_->get_secret_chat_ttl(dialog_id.get_secret_chat_id());
    if (is_service_message_content(m->content->get_type())) {
      m->ttl = 0;
    }
  } else if (message_content.ttl > 0) {
    m->ttl = message_content.ttl;
  }
  m->is_content_secret = is_secret_message_content(m->ttl, m->content->get_type());

  m->have_previous = true;
  m->have_next = true;

  bool need_update = true;
  bool need_update_dialog_pos = false;
  auto result =
      add_message_to_dialog(d, std::move(m), true, &need_update, &need_update_dialog_pos, "add local message");
  LOG_CHECK(result != nullptr) << message_id << " " << debug_add_message_to_dialog_fail_reason_;

  if (is_message_auto_read(dialog_id, result->is_outgoing)) {
    if (result->is_outgoing) {
      read_history_outbox(dialog_id, message_id);
    } else {
      read_history_inbox(dialog_id, message_id, 0, "add_local_message");
    }
  }

  if (message_content.clear_draft) {
    update_dialog_draft_message(d, nullptr, false, !need_update_dialog_pos);
  }

  send_update_new_message(d, result);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "add_local_message");
  }

  return message_id;
}

bool MessagesManager::on_update_message_id(int64 random_id, MessageId new_message_id, const string &source) {
  if (!new_message_id.is_valid()) {
    LOG(ERROR) << "Receive " << new_message_id << " in updateMessageId with random_id " << random_id << " from "
               << source;
    auto it = debug_being_sent_messages_.find(random_id);
    if (it == debug_being_sent_messages_.end()) {
      LOG(ERROR) << "Message with random_id " << random_id << " was not sent";
      return false;
    }
    auto dialog_id = it->second;
    if (!dialog_id.is_valid()) {
      LOG(ERROR) << "Sent message is in invalid " << dialog_id;
      return false;
    }
    if (!have_dialog(dialog_id)) {
      LOG(ERROR) << "Sent message is in not found " << dialog_id;
      return false;
    }
    LOG(ERROR) << "Receive " << new_message_id << " in updateMessageId with random_id " << random_id << " in "
               << dialog_id;
    return false;
  }

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // update about new message sent from other device or service message
    LOG(INFO) << "Receive not send outgoing " << new_message_id << " with random_id = " << random_id;
    return true;
  }

  auto dialog_id = it->second.get_dialog_id();
  auto old_message_id = it->second.get_message_id();

  being_sent_messages_.erase(it);

  LOG(INFO) << "Save correspondence from " << new_message_id << " in " << dialog_id << " to " << old_message_id;
  update_message_ids_[FullMessageId(dialog_id, new_message_id)] = old_message_id;
  return true;
}

bool MessagesManager::on_update_scheduled_message_id(int64 random_id, ScheduledServerMessageId new_message_id,
                                                     const string &source) {
  if (!new_message_id.is_valid()) {
    LOG(ERROR) << "Receive " << new_message_id << " in updateMessageId with random_id " << random_id << " from "
               << source;
    return false;
  }

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    LOG(ERROR) << "Receive not send outgoing " << new_message_id << " with random_id = " << random_id;
    return false;
  }

  auto dialog_id = it->second.get_dialog_id();
  auto old_message_id = it->second.get_message_id();

  being_sent_messages_.erase(it);

  LOG(INFO) << "Save correspondence from " << new_message_id << " in " << dialog_id << " to " << old_message_id;
  update_scheduled_message_ids_[dialog_id][new_message_id] = old_message_id;
  return true;
}

bool MessagesManager::on_get_dialog_error(DialogId dialog_id, const Status &status, const string &source) {
  if (status.code() == 401) {
    // authorization is lost
    return true;
  }
  if (status.code() == 420 || status.code() == 429) {
    // flood wait
    return true;
  }
  if (status.message() == CSlice("BOT_METHOD_INVALID")) {
    LOG(ERROR) << "Receive BOT_METHOD_INVALID from " << source;
    return true;
  }
  if (G()->close_flag()) {
    return true;
  }

  switch (dialog_id.get_type()) {
    case DialogType::User:
    case DialogType::Chat:
    case DialogType::SecretChat:
      // to be implemented if necessary
      break;
    case DialogType::Channel:
      return td_->contacts_manager_->on_get_channel_error(dialog_id.get_channel_id(), status, source);
    case DialogType::None:
      // to be implemented if necessary
      break;
    default:
      UNREACHABLE();
  }
  return false;
}

void MessagesManager::on_dialog_updated(DialogId dialog_id, const char *source) {
  if (G()->parameters().use_message_db) {
    LOG(INFO) << "Update " << dialog_id << " from " << source;
    pending_updated_dialog_timeout_.add_timeout_in(dialog_id.get(), MAX_SAVE_DIALOG_DELAY);
  }
}

void MessagesManager::send_update_new_message(const Dialog *d, const Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);

  LOG(INFO) << "Send updateNewMessage for " << m->message_id << " in " << d->dialog_id;
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateNewMessage>(get_message_object(d->dialog_id, m)));
}

MessagesManager::NotificationGroupInfo &MessagesManager::get_notification_group_info(Dialog *d, const Message *m) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  return is_from_mention_notification_group(d, m) ? d->mention_notification_group : d->message_notification_group;
}

NotificationGroupId MessagesManager::get_dialog_notification_group_id(DialogId dialog_id,
                                                                      NotificationGroupInfo &group_info) {
  if (!group_info.group_id.is_valid()) {
    NotificationGroupId next_notification_group_id;
    do {
      next_notification_group_id = td_->notification_manager_->get_next_notification_group_id();
      if (!next_notification_group_id.is_valid()) {
        return NotificationGroupId();
      }
    } while (get_message_notification_group_force(next_notification_group_id).dialog_id.is_valid());
    group_info.group_id = next_notification_group_id;
    group_info.is_changed = true;
    VLOG(notifications) << "Assign " << next_notification_group_id << " to " << dialog_id;
    on_dialog_updated(dialog_id, "get_dialog_notification_group_id");

    notification_group_id_to_dialog_id_.emplace(next_notification_group_id, dialog_id);

    if (running_get_channel_difference(dialog_id) || get_channel_difference_to_logevent_id_.count(dialog_id) != 0) {
      send_closure_later(G()->notification_manager(), &NotificationManager::before_get_chat_difference,
                         next_notification_group_id);
    }
  }

  CHECK(group_info.group_id.is_valid());

  // notification group must be preloaded to guarantee that there is no race between
  // get_message_notifications_from_database_force and new notifications added right now
  td_->notification_manager_->load_group_force(group_info.group_id);

  return group_info.group_id;
}

Result<MessagesManager::MessagePushNotificationInfo> MessagesManager::get_message_push_notification_info(
    DialogId dialog_id, MessageId message_id, int64 random_id, UserId sender_user_id, int32 date,
    bool is_from_scheduled, bool contains_mention, bool is_pinned, bool is_from_binlog) {
  init();

  if (!is_from_scheduled && dialog_id == get_my_dialog_id()) {
    return Status::Error("Ignore notification in chat with self");
  }
  if (td_->auth_manager_->is_bot()) {
    return Status::Error("Ignore notification sent to bot");
  }

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return Status::Error(406, "Ignore notification in unknown chat");
  }

  bool is_new_pinned = is_pinned && message_id.is_valid() && message_id > d->max_notification_message_id;
  CHECK(!message_id.is_scheduled());
  if (message_id.is_valid()) {
    if (message_id <= d->last_new_message_id) {
      return Status::Error("Ignore notification about known message");
    }
    if (!is_from_binlog && message_id == d->max_notification_message_id) {
      return Status::Error("Ignore previously added message push notification");
    }
    if (!is_from_binlog && message_id < d->max_notification_message_id) {
      return Status::Error("Ignore out of order message push notification");
    }
    if (message_id <= d->last_read_inbox_message_id) {
      return Status::Error("Ignore notification about read message");
    }
  }
  if (random_id != 0) {
    CHECK(dialog_id.get_type() == DialogType::SecretChat);
    if (get_message_id_by_random_id(d, random_id, "get_message_push_notification_info").is_valid()) {
      return Status::Error(406, "Ignore notification about known secret message");
    }
  }

  if (is_pinned) {
    contains_mention = !is_dialog_pinned_message_notifications_disabled(d);
  } else if (contains_mention && is_dialog_mention_notifications_disabled(d)) {
    contains_mention = false;
  }

  DialogId settings_dialog_id = dialog_id;
  Dialog *settings_dialog = d;
  if (contains_mention && sender_user_id.is_valid()) {
    settings_dialog_id = DialogId(sender_user_id);
    settings_dialog = get_dialog_force(settings_dialog_id);
  }

  bool have_settings;
  int32 mute_until;
  std::tie(have_settings, mute_until) = get_dialog_mute_until(settings_dialog_id, settings_dialog);
  if (have_settings && mute_until > date) {
    if (is_new_pinned) {
      remove_dialog_pinned_message_notification(d);
    }
    return Status::Error("Ignore notification in muted chat");
  }

  if (is_dialog_message_notification_disabled(settings_dialog_id, date)) {
    if (is_new_pinned) {
      remove_dialog_pinned_message_notification(d);
    }
    return Status::Error("Ignore notification in chat, because notifications are disabled in the chat");
  }

  auto group_id = get_dialog_notification_group_id(
      dialog_id, contains_mention ? d->mention_notification_group : d->message_notification_group);
  if (!group_id.is_valid()) {
    return Status::Error("Can't assign notification group ID");
  }

  if (message_id.is_valid() && message_id > d->max_notification_message_id) {
    if (is_new_pinned) {
      set_dialog_pinned_message_notification(d, contains_mention ? message_id : MessageId());
    }
    d->max_notification_message_id = message_id;
    on_dialog_updated(dialog_id, "set_max_notification_message_id");
  }

  MessagePushNotificationInfo result;
  result.group_id = group_id;
  result.group_type = contains_mention ? NotificationGroupType::Mentions : NotificationGroupType::Messages;
  result.settings_dialog_id = settings_dialog_id;
  return result;
}

NotificationId MessagesManager::get_next_notification_id(Dialog *d, NotificationGroupId notification_group_id,
                                                         MessageId message_id) {
  CHECK(d != nullptr);
  CHECK(!message_id.is_scheduled());
  NotificationId notification_id;
  do {
    notification_id = td_->notification_manager_->get_next_notification_id();
    if (!notification_id.is_valid()) {
      return NotificationId();
    }
  } while (d->notification_id_to_message_id.count(notification_id) != 0 ||
           d->new_secret_chat_notification_id == notification_id ||
           notification_id.get() <= d->message_notification_group.last_notification_id.get() ||
           notification_id.get() <= d->message_notification_group.max_removed_notification_id.get() ||
           notification_id.get() <= d->mention_notification_group.last_notification_id.get() ||
           notification_id.get() <= d->mention_notification_group.max_removed_notification_id.get());  // just in case
  if (message_id.is_valid()) {
    add_notification_id_to_message_id_correspondence(d, notification_id, message_id);
  }
  return notification_id;
}

MessagesManager::MessageNotificationGroup MessagesManager::get_message_notification_group_force(
    NotificationGroupId group_id) {
  CHECK(group_id.is_valid());
  Dialog *d = nullptr;
  auto it = notification_group_id_to_dialog_id_.find(group_id);
  if (it != notification_group_id_to_dialog_id_.end()) {
    d = get_dialog(it->second);
    CHECK(d != nullptr);
  } else if (G()->parameters().use_message_db) {
    G()->td_db()->get_dialog_db_sync()->begin_transaction().ensure();
    auto r_value = G()->td_db()->get_dialog_db_sync()->get_notification_group(group_id);
    if (r_value.is_ok()) {
      VLOG(notifications) << "Loaded " << r_value.ok() << " from database by " << group_id;
      d = get_dialog_force(r_value.ok().dialog_id);
    } else {
      VLOG(notifications) << "Failed to load " << group_id << " from database";
    }
    G()->td_db()->get_dialog_db_sync()->commit_transaction().ensure();
  }

  if (d == nullptr) {
    return MessageNotificationGroup();
  }
  if (d->message_notification_group.group_id != group_id && d->mention_notification_group.group_id != group_id) {
    if (d->dialog_id.get_type() == DialogType::SecretChat && !d->message_notification_group.group_id.is_valid() &&
        !d->mention_notification_group.group_id.is_valid()) {
      // the group was reused, but wasn't deleted from the database, trying to resave it
      auto &group_info = d->message_notification_group;
      group_info.group_id = group_id;
      group_info.is_changed = true;
      group_info.try_reuse = true;
      save_dialog_to_database(d->dialog_id);
      group_info.group_id = NotificationGroupId();
      group_info.is_changed = false;
      group_info.try_reuse = false;
    }
  }

  LOG_CHECK(d->message_notification_group.group_id == group_id || d->mention_notification_group.group_id == group_id)
      << group_id << " " << d->message_notification_group.group_id << " " << d->mention_notification_group.group_id
      << " " << d->dialog_id << " " << notification_group_id_to_dialog_id_[group_id] << " "
      << notification_group_id_to_dialog_id_[d->message_notification_group.group_id] << " "
      << notification_group_id_to_dialog_id_[d->mention_notification_group.group_id];

  bool from_mentions = d->mention_notification_group.group_id == group_id;
  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;

  MessageNotificationGroup result;
  VLOG(notifications) << "Found " << (from_mentions ? "Mentions " : "Messages ") << group_info.group_id << '/'
                      << d->dialog_id << " by " << group_id << " with " << d->unread_mention_count
                      << " unread mentions, pinned " << d->pinned_message_notification_message_id
                      << ", new secret chat " << d->new_secret_chat_notification_id << " and "
                      << d->server_unread_count + d->local_unread_count << " unread messages";
  result.dialog_id = d->dialog_id;
  result.total_count = get_dialog_pending_notification_count(d, from_mentions);
  auto pending_notification_count =
      from_mentions ? d->pending_new_mention_notifications.size() : d->pending_new_message_notifications.size();
  result.total_count -= static_cast<int32>(pending_notification_count);
  if (result.total_count < 0) {
    LOG(ERROR) << "Total notification count is " << result.total_count << " in " << d->dialog_id << " with "
               << pending_notification_count << " pending new notifications";
    result.total_count = 0;
  }
  if (d->new_secret_chat_notification_id.is_valid()) {
    CHECK(d->dialog_id.get_type() == DialogType::SecretChat);
    result.type = NotificationGroupType::SecretChat;
    result.notifications.emplace_back(d->new_secret_chat_notification_id,
                                      td_->contacts_manager_->get_secret_chat_date(d->dialog_id.get_secret_chat_id()),
                                      false, create_new_secret_chat_notification());
  } else {
    result.type = from_mentions ? NotificationGroupType::Mentions : NotificationGroupType::Messages;
    result.notifications = get_message_notifications_from_database_force(
        d, from_mentions, static_cast<int32>(td_->notification_manager_->get_max_notification_group_size()));
  }

  int32 last_notification_date = 0;
  NotificationId last_notification_id;
  if (!result.notifications.empty()) {
    last_notification_date = result.notifications[0].date;
    last_notification_id = result.notifications[0].notification_id;
  }
  if (last_notification_date != group_info.last_notification_date ||
      last_notification_id != group_info.last_notification_id) {
    LOG(ERROR) << "Fix last notification date in " << d->dialog_id << " from " << group_info.last_notification_date
               << " to " << last_notification_date << " and last notification id from "
               << group_info.last_notification_id << " to " << last_notification_id;
    set_dialog_last_notification(d->dialog_id, group_info, last_notification_date, last_notification_id,
                                 "get_message_notification_group_force");
  }

  std::reverse(result.notifications.begin(), result.notifications.end());

  return result;
}

bool MessagesManager::is_from_mention_notification_group(const Dialog *d, const Message *m) {
  return m->contains_mention && !m->is_mention_notification_disabled;
}

bool MessagesManager::is_message_notification_active(const Dialog *d, const Message *m) {
  CHECK(!m->message_id.is_scheduled());
  if (is_from_mention_notification_group(d, m)) {
    return m->notification_id.get() > d->mention_notification_group.max_removed_notification_id.get() &&
           m->message_id > d->mention_notification_group.max_removed_message_id &&
           (m->contains_unread_mention || m->message_id == d->pinned_message_notification_message_id);
  } else {
    return m->notification_id.get() > d->message_notification_group.max_removed_notification_id.get() &&
           m->message_id > d->message_notification_group.max_removed_message_id &&
           m->message_id > d->last_read_inbox_message_id;
  }
}

void MessagesManager::try_add_pinned_message_notification(Dialog *d, vector<Notification> &res,
                                                          NotificationId max_notification_id, int32 limit) {
  CHECK(d != nullptr);
  auto message_id = d->pinned_message_notification_message_id;
  if (!message_id.is_valid() || message_id > d->last_new_message_id) {
    CHECK(!message_id.is_scheduled());
    return;
  }

  auto m = get_message_force(d, message_id, "try_add_pinned_message_notification");
  if (m != nullptr && m->notification_id.get() > d->mention_notification_group.max_removed_notification_id.get() &&
      m->message_id > d->mention_notification_group.max_removed_message_id &&
      m->message_id > d->last_read_inbox_message_id && !is_dialog_pinned_message_notifications_disabled(d)) {
    if (m->notification_id.get() < max_notification_id.get()) {
      VLOG(notifications) << "Add " << m->notification_id << " about pinned " << message_id << " in " << d->dialog_id;
      auto pinned_message_id = get_message_content_pinned_message_id(m->content.get());
      if (pinned_message_id.is_valid()) {
        get_message_force(d, pinned_message_id, "try_add_pinned_message_notification 2");  // preload pinned message
      }

      auto pos = res.size();
      res.emplace_back(m->notification_id, m->date, m->disable_notification,
                       create_new_message_notification(message_id));
      while (pos > 0 && res[pos - 1].type->get_message_id() < message_id) {
        std::swap(res[pos - 1], res[pos]);
        pos--;
      }
      if (pos > 0 && res[pos - 1].type->get_message_id() == message_id) {
        res.erase(res.begin() + pos);  // notification was already there
      }
      if (res.size() > static_cast<size_t>(limit)) {
        res.pop_back();
        CHECK(res.size() == static_cast<size_t>(limit));
      }
    }
  } else {
    remove_dialog_pinned_message_notification(d);
  }
}

vector<Notification> MessagesManager::get_message_notifications_from_database_force(Dialog *d, bool from_mentions,
                                                                                    int32 limit) {
  CHECK(d != nullptr);
  if (!G()->parameters().use_message_db) {
    return {};
  }

  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  auto from_notification_id = NotificationId::max();
  auto from_message_id = MessageId::max();
  vector<Notification> res;
  if (!from_mentions && from_message_id <= d->last_read_inbox_message_id) {
    return res;
  }
  while (true) {
    auto result = do_get_message_notifications_from_database_force(d, from_mentions, from_notification_id,
                                                                   from_message_id, limit);
    if (result.is_error()) {
      break;
    }
    auto messages = result.move_as_ok();
    if (messages.empty()) {
      break;
    }

    bool is_found = false;
    VLOG(notifications) << "Loaded " << messages.size() << (from_mentions ? " mention" : "")
                        << " messages with notifications from database in " << group_info.group_id << '/'
                        << d->dialog_id;
    for (auto &message : messages) {
      auto m = on_get_message_from_database(d->dialog_id, d, std::move(message), false,
                                            "get_message_notifications_from_database_force");
      if (m == nullptr) {
        VLOG(notifications) << "Receive from database a broken message";
        continue;
      }

      auto notification_id = m->notification_id.is_valid() ? m->notification_id : m->removed_notification_id;
      if (!notification_id.is_valid()) {
        VLOG(ERROR) << "Can't find notification ID for " << m->message_id << " in " << d->dialog_id;
        continue;
      }
      CHECK(m->message_id.is_valid());

      bool is_correct = true;
      if (notification_id.get() >= from_notification_id.get()) {
        // possible if two messages has the same notification_id
        LOG(ERROR) << "Have nonmonotoic notification ids: " << d->dialog_id << " " << m->message_id << " "
                   << notification_id << " " << from_message_id << " " << from_notification_id;
        is_correct = false;
      } else {
        from_notification_id = notification_id;
        is_found = true;
      }
      if (m->message_id >= from_message_id) {
        LOG(ERROR) << "Have nonmonotoic message ids: " << d->dialog_id << " " << m->message_id << " " << notification_id
                   << " " << from_message_id << " " << from_notification_id;
        is_correct = false;
      } else {
        from_message_id = m->message_id;
        is_found = true;
      }

      if (notification_id.get() <= group_info.max_removed_notification_id.get() ||
          m->message_id <= group_info.max_removed_message_id ||
          (!from_mentions && m->message_id <= d->last_read_inbox_message_id)) {
        // if message still has notification_id, but it was removed via max_removed_notification_id,
        // or max_removed_message_id, or last_read_inbox_message_id,
        // then there will be no more messages with active notifications
        is_found = false;
        break;
      }

      if (!m->notification_id.is_valid()) {
        // notification_id can be empty if it is deleted in memory, but not in the database
        VLOG(notifications) << "Receive from database " << m->message_id << " with removed "
                            << m->removed_notification_id;
        continue;
      }

      if (is_from_mention_notification_group(d, m) != from_mentions) {
        VLOG(notifications) << "Receive from database " << m->message_id << " with " << m->notification_id
                            << " from another group";
        continue;
      }

      if (!is_message_notification_active(d, m)) {
        CHECK(from_mentions);
        CHECK(!m->contains_unread_mention);
        CHECK(m->message_id != d->pinned_message_notification_message_id);
        // skip read mentions
        continue;
      }

      if (is_correct) {
        // skip mention messages returned among unread messages
        res.emplace_back(m->notification_id, m->date, m->disable_notification,
                         create_new_message_notification(m->message_id));
      } else {
        remove_message_notification_id(d, m, true, false);
        on_message_changed(d, m, false, "get_message_notifications_from_database_force");
      }
    }
    if (!res.empty() || !is_found) {
      break;
    }
  }
  if (from_mentions) {
    try_add_pinned_message_notification(d, res, NotificationId::max(), limit);
  }
  return res;
}

Result<vector<BufferSlice>> MessagesManager::do_get_message_notifications_from_database_force(
    Dialog *d, bool from_mentions, NotificationId from_notification_id, MessageId from_message_id, int32 limit) {
  CHECK(G()->parameters().use_message_db);
  CHECK(!from_message_id.is_scheduled());

  auto *db = G()->td_db()->get_messages_db_sync();
  if (!from_mentions) {
    CHECK(from_message_id > d->last_read_inbox_message_id);
    VLOG(notifications) << "Trying to load " << limit << " messages with notifications in "
                        << d->message_notification_group.group_id << '/' << d->dialog_id << " from "
                        << from_notification_id;
    return db->get_messages_from_notification_id(d->dialog_id, from_notification_id, limit);
  } else {
    VLOG(notifications) << "Trying to load " << limit << " messages with unread mentions in "
                        << d->mention_notification_group.group_id << '/' << d->dialog_id << " from " << from_message_id;

    // ignore first_db_message_id, notifications can be nonconsecutive
    MessagesDbMessagesQuery db_query;
    db_query.dialog_id = d->dialog_id;
    db_query.index_mask = search_messages_filter_index_mask(SearchMessagesFilter::UnreadMention);
    db_query.from_message_id = from_message_id;
    db_query.offset = 0;
    db_query.limit = limit;
    return db->get_messages(db_query);
  }
}

vector<NotificationGroupKey> MessagesManager::get_message_notification_group_keys_from_database(
    NotificationGroupKey from_group_key, int32 limit) {
  if (!G()->parameters().use_message_db) {
    return {};
  }

  init();

  VLOG(notifications) << "Trying to load " << limit << " message notification groups from database from "
                      << from_group_key;

  G()->td_db()->get_dialog_db_sync()->begin_transaction().ensure();
  Result<vector<NotificationGroupKey>> r_notification_group_keys =
      G()->td_db()->get_dialog_db_sync()->get_notification_groups_by_last_notification_date(from_group_key, limit);
  r_notification_group_keys.ensure();
  auto group_keys = r_notification_group_keys.move_as_ok();

  vector<NotificationGroupKey> result;
  for (auto &group_key : group_keys) {
    CHECK(group_key.dialog_id.is_valid());
    const Dialog *d = get_dialog_force(group_key.dialog_id);
    if (d == nullptr || (d->message_notification_group.group_id != group_key.group_id &&
                         d->mention_notification_group.group_id != group_key.group_id)) {
      continue;
    }

    CHECK(d->dialog_id == group_key.dialog_id);
    CHECK(notification_group_id_to_dialog_id_[group_key.group_id] == d->dialog_id);

    VLOG(notifications) << "Loaded " << group_key << " from database";
    result.push_back(group_key);
  }
  G()->td_db()->get_dialog_db_sync()->commit_transaction().ensure();
  return result;
}

void MessagesManager::get_message_notifications_from_database(DialogId dialog_id, NotificationGroupId group_id,
                                                              NotificationId from_notification_id,
                                                              MessageId from_message_id, int32 limit,
                                                              Promise<vector<Notification>> promise) {
  if (!G()->parameters().use_message_db) {
    return promise.set_error(Status::Error(500, "There is no message database"));
  }

  CHECK(dialog_id.is_valid());
  CHECK(group_id.is_valid());
  CHECK(!from_message_id.is_scheduled());
  CHECK(limit > 0);

  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  if (d->message_notification_group.group_id != group_id && d->mention_notification_group.group_id != group_id) {
    return promise.set_value(vector<Notification>());
  }

  VLOG(notifications) << "Get " << limit << " message notifications from database in " << group_id << " from "
                      << dialog_id << " from " << from_notification_id << "/" << from_message_id;
  bool from_mentions = d->mention_notification_group.group_id == group_id;
  if (d->new_secret_chat_notification_id.is_valid()) {
    CHECK(d->dialog_id.get_type() == DialogType::SecretChat);
    vector<Notification> notifications;
    if (!from_mentions && d->new_secret_chat_notification_id.get() < from_notification_id.get()) {
      notifications.emplace_back(d->new_secret_chat_notification_id,
                                 td_->contacts_manager_->get_secret_chat_date(d->dialog_id.get_secret_chat_id()), false,
                                 create_new_secret_chat_notification());
    }
    return promise.set_value(std::move(notifications));
  }

  do_get_message_notifications_from_database(d, from_mentions, from_notification_id, from_notification_id,
                                             from_message_id, limit, std::move(promise));
}

void MessagesManager::do_get_message_notifications_from_database(Dialog *d, bool from_mentions,
                                                                 NotificationId initial_from_notification_id,
                                                                 NotificationId from_notification_id,
                                                                 MessageId from_message_id, int32 limit,
                                                                 Promise<vector<Notification>> promise) {
  CHECK(G()->parameters().use_message_db);
  CHECK(!from_message_id.is_scheduled());

  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  if (from_notification_id.get() <= group_info.max_removed_notification_id.get() ||
      from_message_id <= group_info.max_removed_message_id ||
      (!from_mentions && from_message_id <= d->last_read_inbox_message_id)) {
    return promise.set_value(vector<Notification>());
  }

  auto dialog_id = d->dialog_id;
  auto new_promise =
      PromiseCreator::lambda([actor_id = actor_id(this), dialog_id, from_mentions, initial_from_notification_id, limit,
                              promise = std::move(promise)](Result<vector<BufferSlice>> result) mutable {
        send_closure(actor_id, &MessagesManager::on_get_message_notifications_from_database, dialog_id, from_mentions,
                     initial_from_notification_id, limit, std::move(result), std::move(promise));
      });

  auto *db = G()->td_db()->get_messages_db_async();
  if (!from_mentions) {
    VLOG(notifications) << "Trying to load " << limit << " messages with notifications in " << group_info.group_id
                        << '/' << dialog_id << " from " << from_notification_id;
    return db->get_messages_from_notification_id(d->dialog_id, from_notification_id, limit, std::move(new_promise));
  } else {
    VLOG(notifications) << "Trying to load " << limit << " messages with unread mentions in " << group_info.group_id
                        << '/' << dialog_id << " from " << from_message_id;

    // ignore first_db_message_id, notifications can be nonconsecutive
    MessagesDbMessagesQuery db_query;
    db_query.dialog_id = dialog_id;
    db_query.index_mask = search_messages_filter_index_mask(SearchMessagesFilter::UnreadMention);
    db_query.from_message_id = from_message_id;
    db_query.offset = 0;
    db_query.limit = limit;
    return db->get_messages(db_query, std::move(new_promise));
  }
}

void MessagesManager::on_get_message_notifications_from_database(DialogId dialog_id, bool from_mentions,
                                                                 NotificationId initial_from_notification_id,
                                                                 int32 limit, Result<vector<BufferSlice>> result,
                                                                 Promise<vector<Notification>> promise) {
  if (result.is_error()) {
    return promise.set_error(result.move_as_error());
  }

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  if (!group_info.group_id.is_valid()) {
    return promise.set_error(Status::Error("Notification group was deleted"));
  }

  auto messages = result.move_as_ok();
  vector<Notification> res;
  res.reserve(messages.size());
  NotificationId from_notification_id;
  MessageId from_message_id;
  VLOG(notifications) << "Loaded " << messages.size() << " messages with notifications in " << group_info.group_id
                      << '/' << dialog_id << " from database";
  for (auto &message : messages) {
    auto m = on_get_message_from_database(dialog_id, d, std::move(message), false,
                                          "on_get_message_notifications_from_database");
    if (m == nullptr) {
      VLOG(notifications) << "Receive from database a broken message";
      continue;
    }

    auto notification_id = m->notification_id.is_valid() ? m->notification_id : m->removed_notification_id;
    if (!notification_id.is_valid()) {
      VLOG(ERROR) << "Can't find notification ID for " << m->message_id << " in " << d->dialog_id;
      continue;
    }
    CHECK(m->message_id.is_valid());

    bool is_correct = true;
    if (from_notification_id.is_valid() && notification_id.get() >= from_notification_id.get()) {
      LOG(ERROR) << "Receive " << m->message_id << "/" << notification_id << " after " << from_message_id << "/"
                 << from_notification_id;
      is_correct = false;
    } else {
      from_notification_id = notification_id;
    }
    if (from_message_id.is_valid() && m->message_id >= from_message_id) {
      LOG(ERROR) << "Receive " << m->message_id << "/" << notification_id << " after " << from_message_id << "/"
                 << from_notification_id;
      is_correct = false;
    } else {
      from_message_id = m->message_id;
    }

    if (notification_id.get() <= group_info.max_removed_notification_id.get() ||
        m->message_id <= group_info.max_removed_message_id ||
        (!from_mentions && m->message_id <= d->last_read_inbox_message_id)) {
      // if message still has notification_id, but it was removed via max_removed_notification_id,
      // or max_removed_message_id, or last_read_inbox_message_id,
      // then there will be no more messages with active notifications
      from_notification_id = NotificationId();  // stop requesting database
      break;
    }

    if (!m->notification_id.is_valid()) {
      // notification_id can be empty if it is deleted in memory, but not in the database
      VLOG(notifications) << "Receive from database " << m->message_id << " with removed "
                          << m->removed_notification_id;
      continue;
    }

    if (is_from_mention_notification_group(d, m) != from_mentions) {
      VLOG(notifications) << "Receive from database " << m->message_id << " with " << m->notification_id
                          << " from another category";
      continue;
    }

    if (!is_message_notification_active(d, m)) {
      CHECK(from_mentions);
      CHECK(!m->contains_unread_mention);
      CHECK(m->message_id != d->pinned_message_notification_message_id);
      // skip read mentions
      continue;
    }

    if (is_correct) {
      // skip mention messages returned among unread messages
      res.emplace_back(m->notification_id, m->date, m->disable_notification,
                       create_new_message_notification(m->message_id));
    } else {
      remove_message_notification_id(d, m, true, false);
      on_message_changed(d, m, false, "on_get_message_notifications_from_database");
    }
  }
  if (!res.empty() || !from_notification_id.is_valid() || static_cast<size_t>(limit) > messages.size()) {
    if (from_mentions) {
      try_add_pinned_message_notification(d, res, initial_from_notification_id, limit);
    }

    std::reverse(res.begin(), res.end());
    return promise.set_value(std::move(res));
  }

  // try again from adjusted from_notification_id and from_message_id
  do_get_message_notifications_from_database(d, from_mentions, initial_from_notification_id, from_notification_id,
                                             from_message_id, limit, std::move(promise));
}

void MessagesManager::remove_message_notification(DialogId dialog_id, NotificationGroupId group_id,
                                                  NotificationId notification_id) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(ERROR) << "Can't find " << dialog_id;
    return;
  }
  if (d->message_notification_group.group_id != group_id && d->mention_notification_group.group_id != group_id) {
    LOG(ERROR) << "There is no " << group_id << " in " << dialog_id;
    return;
  }
  if (notification_id == NotificationId::max() || !notification_id.is_valid()) {
    return;  // there can be no notification with this ID
  }

  bool from_mentions = d->mention_notification_group.group_id == group_id;
  if (d->new_secret_chat_notification_id.is_valid()) {
    if (!from_mentions && d->new_secret_chat_notification_id == notification_id) {
      return remove_new_secret_chat_notification(d, false);
    }
    return;
  }

  auto it = d->notification_id_to_message_id.find(notification_id);
  if (it != d->notification_id_to_message_id.end()) {
    auto m = get_message(d, it->second);
    CHECK(m != nullptr);
    CHECK(m->notification_id == notification_id);
    CHECK(!m->message_id.is_scheduled());
    if (is_from_mention_notification_group(d, m) == from_mentions && is_message_notification_active(d, m)) {
      remove_message_notification_id(d, m, false, false);
    }
    return;
  }

  if (G()->parameters().use_message_db) {
    G()->td_db()->get_messages_db_async()->get_messages_from_notification_id(
        dialog_id, NotificationId(notification_id.get() + 1), 1,
        PromiseCreator::lambda(
            [dialog_id, from_mentions, notification_id, actor_id = actor_id(this)](vector<BufferSlice> result) {
              send_closure(actor_id, &MessagesManager::do_remove_message_notification, dialog_id, from_mentions,
                           notification_id, std::move(result));
            }));
  }
}

void MessagesManager::remove_message_notifications_by_message_ids(DialogId dialog_id,
                                                                  const vector<MessageId> &message_ids) {
  VLOG(notifications) << "Trying to remove notification about " << message_ids << " in " << dialog_id;
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return;
  }

  bool need_update_dialog_pos = false;
  vector<int64> deleted_message_ids;
  for (auto message_id : message_ids) {
    CHECK(!message_id.is_scheduled());
    // can't remove just notification_id, because total_count will stay wrong after restart
    // delete whole message
    auto message =
        delete_message(d, message_id, true, &need_update_dialog_pos, "remove_message_notifications_by_message_ids");
    if (message == nullptr) {
      LOG(INFO) << "Can't delete " << message_id << " because it is not found";
      // call synchronously to remove them before ProcessPush returns
      td_->notification_manager_->remove_temporary_notification_by_message_id(
          d->message_notification_group.group_id, message_id, true, "remove_message_notifications_by_message_ids");
      td_->notification_manager_->remove_temporary_notification_by_message_id(
          d->mention_notification_group.group_id, message_id, true, "remove_message_notifications_by_message_ids");
      continue;
    }
    deleted_message_ids.push_back(message->message_id.get());
  }

  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "remove_message_notifications_by_message_ids");
  }
  send_update_delete_messages(dialog_id, std::move(deleted_message_ids), true, false);
}

void MessagesManager::do_remove_message_notification(DialogId dialog_id, bool from_mentions,
                                                     NotificationId notification_id, vector<BufferSlice> result) {
  if (result.empty()) {
    return;
  }
  CHECK(result.size() == 1);

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  auto m = on_get_message_from_database(dialog_id, d, std::move(result[0]), false, "do_remove_message_notification");
  if (m != nullptr && m->notification_id == notification_id &&
      is_from_mention_notification_group(d, m) == from_mentions && is_message_notification_active(d, m)) {
    remove_message_notification_id(d, m, false, false);
  }
}

void MessagesManager::remove_message_notifications(DialogId dialog_id, NotificationGroupId group_id,
                                                   NotificationId max_notification_id, MessageId max_message_id) {
  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(ERROR) << "Can't find " << dialog_id;
    return;
  }
  if (d->message_notification_group.group_id != group_id && d->mention_notification_group.group_id != group_id) {
    LOG(ERROR) << "There is no " << group_id << " in " << dialog_id;
    return;
  }
  if (!max_notification_id.is_valid()) {
    return;
  }
  CHECK(!max_message_id.is_scheduled());

  bool from_mentions = d->mention_notification_group.group_id == group_id;
  if (d->new_secret_chat_notification_id.is_valid()) {
    if (!from_mentions && d->new_secret_chat_notification_id.get() <= max_notification_id.get()) {
      return remove_new_secret_chat_notification(d, false);
    }
    return;
  }
  auto &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  if (max_notification_id.get() <= group_info.max_removed_notification_id.get()) {
    return;
  }
  if (max_message_id > group_info.max_removed_message_id) {
    VLOG(notifications) << "Set max_removed_message_id in " << group_info.group_id << '/' << dialog_id << " to "
                        << max_message_id;
    group_info.max_removed_message_id = max_message_id.get_prev_server_message_id();
  }

  VLOG(notifications) << "Set max_removed_notification_id in " << group_info.group_id << '/' << dialog_id << " to "
                      << max_notification_id;
  group_info.max_removed_notification_id = max_notification_id;
  on_dialog_updated(dialog_id, "remove_message_notifications");

  if (group_info.last_notification_id.is_valid() &&
      max_notification_id.get() >= group_info.last_notification_id.get()) {
    bool is_changed =
        set_dialog_last_notification(dialog_id, group_info, 0, NotificationId(), "remove_message_notifications");
    CHECK(is_changed);
  }
}

int32 MessagesManager::get_dialog_pending_notification_count(const Dialog *d, bool from_mentions) const {
  CHECK(d != nullptr);
  if (from_mentions) {
    bool has_pinned_message = d->pinned_message_notification_message_id.is_valid() &&
                              d->pinned_message_notification_message_id <= d->last_new_message_id;
    return d->unread_mention_count + static_cast<int32>(has_pinned_message);
  } else {
    if (d->new_secret_chat_notification_id.is_valid()) {
      return 1;
    }
    if (is_dialog_muted(d)) {
      return 0;
    }

    return d->server_unread_count + d->local_unread_count;
  }
}

void MessagesManager::update_dialog_mention_notification_count(const Dialog *d) {
  CHECK(d != nullptr);
  if (!d->mention_notification_group.group_id.is_valid()) {
    return;
  }
  auto total_count =
      get_dialog_pending_notification_count(d, true) - static_cast<int32>(d->pending_new_mention_notifications.size());
  if (total_count < 0) {
    LOG(ERROR) << "Total mention notification count is " << total_count << " in " << d->dialog_id << " with "
               << d->pending_new_mention_notifications << " pending new mention notifications";
    total_count = 0;
  }
  send_closure_later(G()->notification_manager(), &NotificationManager::set_notification_total_count,
                     d->mention_notification_group.group_id, total_count);
}

bool MessagesManager::is_message_notification_disabled(const Dialog *d, const Message *m) const {
  CHECK(d != nullptr);
  CHECK(m != nullptr);

  if (!has_incoming_notification(d->dialog_id, m) || td_->auth_manager_->is_bot()) {
    return true;
  }

  switch (m->content->get_type()) {
    case MessageContentType::ChatDeleteHistory:
    case MessageContentType::ChatMigrateTo:
    case MessageContentType::Unsupported:
    case MessageContentType::ExpiredPhoto:
    case MessageContentType::ExpiredVideo:
    case MessageContentType::PassportDataSent:
    case MessageContentType::PassportDataReceived:
      VLOG(notifications) << "Disable notification for " << m->message_id << " in " << d->dialog_id
                          << " with content of type " << m->content->get_type();
      return true;
    case MessageContentType::ContactRegistered:
      if (m->disable_notification) {
        return true;
      }
      break;
    default:
      break;
  }

  return is_dialog_message_notification_disabled(d->dialog_id, m->date);
}

bool MessagesManager::is_dialog_message_notification_disabled(DialogId dialog_id, int32 message_date) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      break;
    case DialogType::Chat:
      if (!td_->contacts_manager_->get_chat_is_active(dialog_id.get_chat_id())) {
        return true;
      }
      break;
    case DialogType::Channel:
      if (!td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id()).is_member() ||
          message_date < td_->contacts_manager_->get_channel_date(dialog_id.get_channel_id())) {
        return true;
      }
      break;
    case DialogType::SecretChat:
      if (td_->contacts_manager_->get_secret_chat_state(dialog_id.get_secret_chat_id()) == SecretChatState::Closed) {
        return true;
      }
      break;
    case DialogType::None:
    default:
      UNREACHABLE();
  }

  return false;
}

bool MessagesManager::may_need_message_notification(const Dialog *d, const Message *m) const {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());

  if (is_message_notification_disabled(d, m)) {
    return false;
  }

  if (is_from_mention_notification_group(d, m)) {
    return true;
  }

  bool have_settings;
  int32 mute_until;
  std::tie(have_settings, mute_until) = get_dialog_mute_until(d->dialog_id, d);
  return !have_settings || mute_until <= m->date;
}

bool MessagesManager::add_new_message_notification(Dialog *d, Message *m, bool force) {
  CHECK(d != nullptr);
  CHECK(m != nullptr);
  CHECK(m->message_id.is_valid());

  if (!force) {
    if (d->message_notification_group.group_id.is_valid()) {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_temporary_notifications,
                         d->message_notification_group.group_id, "add_new_message_notification 1");
    }
    if (d->mention_notification_group.group_id.is_valid()) {
      send_closure_later(G()->notification_manager(), &NotificationManager::remove_temporary_notifications,
                         d->mention_notification_group.group_id, "add_new_message_notification 2");
    }
  }

  CHECK(!m->notification_id.is_valid());
  if (is_message_notification_disabled(d, m)) {
    return false;
  }

  auto from_mentions = is_from_mention_notification_group(d, m);
  bool is_pinned = m->content->get_type() == MessageContentType::PinMessage;
  bool is_active =
      from_mentions ? m->contains_unread_mention || is_pinned : m->message_id > d->last_read_inbox_message_id;
  if (is_active) {
    auto &group = from_mentions ? d->mention_notification_group : d->message_notification_group;
    if (group.max_removed_message_id >= m->message_id) {
      is_active = false;
    }
  }
  if (!is_active) {
    VLOG(notifications) << "Disable inactive notification for " << m->message_id << " in " << d->dialog_id;
    if (is_pinned) {
      remove_dialog_pinned_message_notification(d);
    }
    return false;
  }

  VLOG(notifications) << "Trying to " << (force ? "forcely " : "") << "add new message notification for "
                      << m->message_id << " in " << d->dialog_id
                      << (m->disable_notification ? " silently" : " with sound");

  DialogId settings_dialog_id = d->dialog_id;
  Dialog *settings_dialog = d;
  if (m->contains_mention && !m->is_mention_notification_disabled && m->sender_user_id.is_valid()) {
    // have a mention, so use notification settings from the dialog with the sender
    settings_dialog_id = DialogId(m->sender_user_id);
    settings_dialog = get_dialog_force(settings_dialog_id);
  }

  bool have_settings;
  int32 mute_until;
  std::tie(have_settings, mute_until) = get_dialog_mute_until(settings_dialog_id, settings_dialog);
  if (mute_until > m->date && (have_settings || force)) {
    VLOG(notifications) << "Disable notification, because " << settings_dialog_id << " is muted";
    if (is_pinned) {
      remove_dialog_pinned_message_notification(d);
    }
    return false;
  }

  MessageId missing_pinned_message_id;
  if (is_pinned) {
    auto message_id = get_message_content_pinned_message_id(m->content.get());
    if (message_id.is_valid() &&
        !have_message_force({d->dialog_id, message_id},
                            force ? "add_new_message_notification force" : "add_new_message_notification not force")) {
      missing_pinned_message_id = message_id;
    }
  }

  auto &pending_notifications =
      from_mentions ? d->pending_new_mention_notifications : d->pending_new_message_notifications;
  if (!force && (!have_settings || !pending_notifications.empty() || missing_pinned_message_id.is_valid())) {
    VLOG(notifications) << "Delay new message notification for " << m->message_id << " in " << d->dialog_id << " with "
                        << pending_notifications.size() << " already waiting messages";
    if (pending_notifications.empty()) {
      VLOG(notifications) << "Create FlushPendingNewMessageNotificationsSleepActor for " << d->dialog_id;
      create_actor<SleepActor>("FlushPendingNewMessageNotificationsSleepActor", 5.0,
                               PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = d->dialog_id,
                                                       from_mentions](Result<Unit> result) {
                                 VLOG(notifications)
                                     << "Pending notifications timeout in " << dialog_id << " has expired";
                                 send_closure(actor_id, &MessagesManager::flush_pending_new_message_notifications,
                                              dialog_id, from_mentions, DialogId());
                               }))
          .release();
    }
    auto last_settings_dialog_id = (pending_notifications.empty() ? DialogId() : pending_notifications.back().first);
    pending_notifications.emplace_back((have_settings ? DialogId() : settings_dialog_id), m->message_id);
    if (!have_settings && last_settings_dialog_id != settings_dialog_id) {
      VLOG(notifications) << "Fetch notification settings for " << settings_dialog_id;
      auto promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id = d->dialog_id, from_mentions,
                                             settings_dialog_id](Result<Unit> result) {
        send_closure(actor_id, &MessagesManager::flush_pending_new_message_notifications, dialog_id, from_mentions,
                     settings_dialog_id);
      });
      if (settings_dialog == nullptr && have_input_peer(settings_dialog_id, AccessRights::Read)) {
        force_create_dialog(settings_dialog_id, "add_new_message_notification 2");
        settings_dialog = get_dialog(settings_dialog_id);
      }
      if (settings_dialog != nullptr) {
        send_get_dialog_notification_settings_query(settings_dialog_id, std::move(promise));
      } else {
        send_get_dialog_query(settings_dialog_id, std::move(promise));
      }
    }
    if (missing_pinned_message_id.is_valid()) {
      VLOG(notifications) << "Fetch pinned " << missing_pinned_message_id;
      auto promise = PromiseCreator::lambda(
          [actor_id = actor_id(this), dialog_id = d->dialog_id, from_mentions](Result<Unit> result) {
            send_closure(actor_id, &MessagesManager::flush_pending_new_message_notifications, dialog_id, from_mentions,
                         dialog_id);
          });
      get_message_from_server({d->dialog_id, missing_pinned_message_id}, std::move(promise));
    }
    return false;
  }

  LOG_IF(WARNING, !have_settings) << "Have no notification settings for " << settings_dialog_id
                                  << ", but forced to send notification about " << m->message_id << " in "
                                  << d->dialog_id;
  auto &group_info = get_notification_group_info(d, m);
  auto notification_group_id = get_dialog_notification_group_id(d->dialog_id, group_info);
  if (!notification_group_id.is_valid()) {
    return false;
  }
  // if !force, then add_message_to_dialog will add the correspondence
  m->notification_id = get_next_notification_id(d, notification_group_id, force ? m->message_id : MessageId());
  if (!m->notification_id.is_valid()) {
    return false;
  }
  bool is_changed = set_dialog_last_notification(d->dialog_id, group_info, m->date, m->notification_id,
                                                 "add_new_message_notification 3");
  CHECK(is_changed);
  if (is_pinned) {
    set_dialog_pinned_message_notification(d, from_mentions ? m->message_id : MessageId());
  }
  if (!m->notification_id.is_valid()) {
    // protection from accidental notification_id removal in set_dialog_pinned_message_notification
    return false;
  }
  VLOG(notifications) << "Create " << m->notification_id << " with " << m->message_id << " in " << group_info.group_id
                      << '/' << d->dialog_id;
  int32 min_delay_ms = 0;
  if (need_delay_message_content_notification(m->content.get(), td_->contacts_manager_->get_my_id())) {
    min_delay_ms = 3000;  // 3 seconds
  } else if (td_->is_online() && d->is_opened) {
    min_delay_ms = 1000;  // 1 second
  }
  bool is_silent = m->disable_notification || m->message_id <= d->max_notification_message_id;
  send_closure_later(G()->notification_manager(), &NotificationManager::add_notification, notification_group_id,
                     from_mentions ? NotificationGroupType::Mentions : NotificationGroupType::Messages, d->dialog_id,
                     m->date, settings_dialog_id, m->disable_notification, is_silent, min_delay_ms, m->notification_id,
                     create_new_message_notification(m->message_id), "add_new_message_notification");
  return true;
}

void MessagesManager::flush_pending_new_message_notifications(DialogId dialog_id, bool from_mentions,
                                                              DialogId settings_dialog_id) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  auto &pending_notifications =
      from_mentions ? d->pending_new_mention_notifications : d->pending_new_message_notifications;
  if (pending_notifications.empty()) {
    VLOG(notifications) << "Have no pending notifications in " << dialog_id << " to flush";
    return;
  }
  for (auto &it : pending_notifications) {
    if (it.first == settings_dialog_id || !settings_dialog_id.is_valid()) {
      it.first = DialogId();
    }
  }

  VLOG(notifications) << "Flush pending notifications in " << dialog_id
                      << " because of received notification settings in " << settings_dialog_id;
  auto it = pending_notifications.begin();
  while (it != pending_notifications.end() && it->first == DialogId()) {
    auto m = get_message(d, it->second);
    if (m != nullptr) {
      if (add_new_message_notification(d, m, true)) {
        on_message_changed(d, m, false, "flush_pending_new_message_notifications");
      }
    }
    ++it;
  }

  if (it == pending_notifications.end()) {
    reset_to_empty(pending_notifications);
  } else {
    pending_notifications.erase(pending_notifications.begin(), it);
  }
}

void MessagesManager::remove_all_dialog_notifications(Dialog *d, bool from_mentions, const char *source) {
  // removes up to group_info.last_notification_id
  NotificationGroupInfo &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  if (group_info.group_id.is_valid() && group_info.last_notification_id.is_valid() &&
      group_info.max_removed_notification_id != group_info.last_notification_id) {
    VLOG(notifications) << "Set max_removed_notification_id in " << group_info.group_id << '/' << d->dialog_id << " to "
                        << group_info.last_notification_id << " from " << source;
    group_info.max_removed_notification_id = group_info.last_notification_id;
    if (d->max_notification_message_id > group_info.max_removed_message_id) {
      group_info.max_removed_message_id = d->max_notification_message_id.get_prev_server_message_id();
    }
    if (!d->pending_new_message_notifications.empty()) {
      for (auto &it : d->pending_new_message_notifications) {
        it.first = DialogId();
      }
      flush_pending_new_message_notifications(d->dialog_id, from_mentions, DialogId(UserId(2)));
    }
    // remove_message_notifications will be called by NotificationManager
    send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification_group,
                       group_info.group_id, group_info.last_notification_id, MessageId(), 0, true, Promise<Unit>());
    if (d->new_secret_chat_notification_id.is_valid() && &group_info == &d->message_notification_group) {
      remove_new_secret_chat_notification(d, false);
    } else {
      bool is_changed = set_dialog_last_notification(d->dialog_id, group_info, 0, NotificationId(), source);
      CHECK(is_changed);
    }
  }
}

void MessagesManager::remove_message_dialog_notifications(Dialog *d, MessageId max_message_id, bool from_mentions,
                                                          const char *source) {
  // removes up to max_message_id
  CHECK(!max_message_id.is_scheduled());
  NotificationGroupInfo &group_info = from_mentions ? d->mention_notification_group : d->message_notification_group;
  if (!group_info.group_id.is_valid()) {
    return;
  }

  VLOG(notifications) << "Remove message dialog notifications in " << group_info.group_id << '/' << d->dialog_id
                      << " up to " << max_message_id << " from " << source;

  if (!d->pending_new_message_notifications.empty()) {
    for (auto &it : d->pending_new_message_notifications) {
      if (it.second <= max_message_id) {
        it.first = DialogId();
      }
    }
    flush_pending_new_message_notifications(d->dialog_id, from_mentions, DialogId(UserId(3)));
  }

  auto max_notification_message_id = max_message_id;
  if (d->last_message_id.is_valid() && max_notification_message_id >= d->last_message_id) {
    max_notification_message_id = d->last_message_id;
    set_dialog_last_notification(d->dialog_id, group_info, 0, NotificationId(),
                                 "remove_message_dialog_notifications 1");
  } else if (max_notification_message_id == MessageId::max()) {
    max_notification_message_id = get_next_local_message_id(d);
    set_dialog_last_notification(d->dialog_id, group_info, 0, NotificationId(),
                                 "remove_message_dialog_notifications 2");
  } else {
    LOG(FATAL) << "TODO support deleting up to " << max_message_id << " if ever will be needed";
  }

  send_closure_later(G()->notification_manager(), &NotificationManager::remove_notification_group, group_info.group_id,
                     NotificationId(), max_notification_message_id, 0, true, Promise<Unit>());
}

void MessagesManager::send_update_message_send_succeeded(Dialog *d, MessageId old_message_id, const Message *m) const {
  CHECK(m != nullptr);
  d->yet_unsent_message_id_to_persistent_message_id.emplace(old_message_id, m->message_id);
  send_closure(
      G()->td(), &Td::send_update,
      make_tl_object<td_api::updateMessageSendSucceeded>(get_message_object(d->dialog_id, m), old_message_id.get()));
}

void MessagesManager::send_update_message_content(DialogId dialog_id, MessageId message_id,
                                                  const MessageContent *content, int32 message_date,
                                                  bool is_content_secret, const char *source) const {
  LOG(INFO) << "Send updateMessageContent for " << message_id << " in " << dialog_id << " from " << source;
  LOG_CHECK(have_dialog(dialog_id)) << "Send updateMessageContent in unknown " << dialog_id << " from " << source
                                    << " with load count " << loaded_dialogs_.count(dialog_id);
  auto content_object = get_message_content_object(content, td_, message_date, is_content_secret);
  send_closure(
      G()->td(), &Td::send_update,
      td_api::make_object<td_api::updateMessageContent>(dialog_id.get(), message_id.get(), std::move(content_object)));
}

void MessagesManager::send_update_message_edited(DialogId dialog_id, const Message *m) {
  CHECK(m != nullptr);
  cancel_user_dialog_action(dialog_id, m);
  auto edit_date = m->hide_edit_date ? 0 : m->edit_date;
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateMessageEdited>(dialog_id.get(), m->message_id.get(), edit_date,
                                                           get_reply_markup_object(m->reply_markup)));
}

void MessagesManager::send_update_message_live_location_viewed(FullMessageId full_message_id) {
  CHECK(get_message(full_message_id) != nullptr);
  send_closure(G()->td(), &Td::send_update,
               td_api::make_object<td_api::updateMessageLiveLocationViewed>(full_message_id.get_dialog_id().get(),
                                                                            full_message_id.get_message_id().get()));
}

void MessagesManager::send_update_delete_messages(DialogId dialog_id, vector<int64> &&message_ids, bool is_permanent,
                                                  bool from_cache) const {
  if (message_ids.empty()) {
    return;
  }

  LOG_CHECK(have_dialog(dialog_id)) << "Wrong " << dialog_id << " in send_update_delete_messages";
  send_closure(
      G()->td(), &Td::send_update,
      make_tl_object<td_api::updateDeleteMessages>(dialog_id.get(), std::move(message_ids), is_permanent, from_cache));
}

void MessagesManager::send_update_new_chat(Dialog *d) {
  CHECK(d != nullptr);
  CHECK(d->messages == nullptr);
  auto chat_object = get_chat_object(d);
  bool has_action_bar = chat_object->action_bar_ != nullptr;
  d->last_sent_has_scheduled_messages = chat_object->has_scheduled_messages_;
  send_closure(G()->td(), &Td::send_update, make_tl_object<td_api::updateNewChat>(std::move(chat_object)));
  d->is_update_new_chat_sent = true;

  if (has_action_bar) {
    send_update_secret_chats_with_user_action_bar(d);
  }
}

void MessagesManager::send_update_chat_draft_message(const Dialog *d) {
  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_draft_message";
  on_dialog_updated(d->dialog_id, "send_update_chat_draft_message");
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatDraftMessage>(
                   d->dialog_id.get(), get_draft_message_object(d->draft_message), get_dialog_public_order(d)));
}

void MessagesManager::send_update_chat_last_message(Dialog *d, const char *source) {
  update_dialog_pos(d, false, source, false);
  send_update_chat_last_message_impl(d, source);
}

void MessagesManager::send_update_chat_last_message_impl(const Dialog *d, const char *source) const {
  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_last_message from "
                                        << source;
  LOG(INFO) << "Send updateChatLastMessage in " << d->dialog_id << " to " << d->last_message_id << " from " << source;
  auto update = make_tl_object<td_api::updateChatLastMessage>(
      d->dialog_id.get(), get_message_object(d->dialog_id, get_message(d, d->last_message_id)),
      get_dialog_public_order(d));
  send_closure(G()->td(), &Td::send_update, std::move(update));
}

void MessagesManager::send_update_unread_message_count(FolderId folder_id, DialogId dialog_id, bool force,
                                                       const char *source) {
  if (td_->auth_manager_->is_bot() || !G()->parameters().use_message_db) {
    return;
  }

  auto &list = get_dialog_list(folder_id);
  CHECK(list.is_message_unread_count_inited_);
  if (list.unread_message_muted_count_ < 0 || list.unread_message_muted_count_ > list.unread_message_total_count_) {
    LOG(ERROR) << "Unread message count became invalid: " << list.unread_message_total_count_ << '/'
               << list.unread_message_total_count_ - list.unread_message_muted_count_ << " from " << source << " and "
               << dialog_id;
    if (list.unread_message_muted_count_ < 0) {
      list.unread_message_muted_count_ = 0;
    }
    if (list.unread_message_muted_count_ > list.unread_message_total_count_) {
      list.unread_message_total_count_ = list.unread_message_muted_count_;
    }
  }
  G()->td_db()->get_binlog_pmc()->set(
      PSTRING() << "unread_message_count" << folder_id.get(),
      PSTRING() << list.unread_message_total_count_ << ' ' << list.unread_message_muted_count_);
  int32 unread_unmuted_count = list.unread_message_total_count_ - list.unread_message_muted_count_;
  if (!force && running_get_difference_) {
    LOG(INFO) << "Postpone updateUnreadMessageCount in " << folder_id << " to " << list.unread_message_total_count_
              << '/' << unread_unmuted_count << " from " << source << " and " << dialog_id;
    postponed_unread_message_count_updates_.insert(folder_id);
  } else {
    postponed_unread_message_count_updates_.erase(folder_id);
    LOG(INFO) << "Send updateUnreadMessageCount in " << folder_id << " to " << list.unread_message_total_count_ << '/'
              << unread_unmuted_count << " from " << source << " and " << dialog_id;
    send_closure(G()->td(), &Td::send_update, get_update_unread_message_count_object(folder_id, list));
  }
}

void MessagesManager::send_update_unread_chat_count(FolderId folder_id, DialogId dialog_id, bool force,
                                                    const char *source) {
  if (td_->auth_manager_->is_bot() || !G()->parameters().use_message_db) {
    return;
  }

  auto &list = get_dialog_list(folder_id);
  CHECK(list.is_dialog_unread_count_inited_);
  if (list.unread_dialog_muted_marked_count_ < 0 ||
      list.unread_dialog_marked_count_ < list.unread_dialog_muted_marked_count_ ||
      list.unread_dialog_muted_count_ < list.unread_dialog_muted_marked_count_ ||
      list.unread_dialog_total_count_ + list.unread_dialog_muted_marked_count_ <
          list.unread_dialog_muted_count_ + list.unread_dialog_marked_count_) {
    LOG(ERROR) << "Unread chat count became invalid: " << list.unread_dialog_total_count_ << '/'
               << list.unread_dialog_total_count_ - list.unread_dialog_muted_count_ << '/'
               << list.unread_dialog_marked_count_ << '/'
               << list.unread_dialog_marked_count_ - list.unread_dialog_muted_marked_count_ << " from " << source
               << " and " << dialog_id;
    if (list.unread_dialog_muted_marked_count_ < 0) {
      list.unread_dialog_muted_marked_count_ = 0;
    }
    if (list.unread_dialog_marked_count_ < list.unread_dialog_muted_marked_count_) {
      list.unread_dialog_marked_count_ = list.unread_dialog_muted_marked_count_;
    }
    if (list.unread_dialog_muted_count_ < list.unread_dialog_muted_marked_count_) {
      list.unread_dialog_muted_count_ = list.unread_dialog_muted_marked_count_;
    }
    if (list.unread_dialog_total_count_ + list.unread_dialog_muted_marked_count_ <
        list.unread_dialog_muted_count_ + list.unread_dialog_marked_count_) {
      list.unread_dialog_total_count_ =
          list.unread_dialog_muted_count_ + list.unread_dialog_marked_count_ - list.unread_dialog_muted_marked_count_;
    }
  }
  G()->td_db()->get_binlog_pmc()->set(
      PSTRING() << "unread_dialog_count" << folder_id.get(),
      PSTRING() << list.unread_dialog_total_count_ << ' ' << list.unread_dialog_muted_count_ << ' '
                << list.unread_dialog_marked_count_ << ' ' << list.unread_dialog_muted_marked_count_ << ' '
                << list.server_dialog_total_count_ << ' ' << list.secret_chat_total_count_);
  bool need_postpone = !force && running_get_difference_;
  int32 unread_unmuted_count = list.unread_dialog_total_count_ - list.unread_dialog_muted_count_;
  int32 unread_unmuted_marked_count = list.unread_dialog_marked_count_ - list.unread_dialog_muted_marked_count_;
  LOG(INFO) << (need_postpone ? "Postpone" : "Send") << " updateUnreadChatCount in " << folder_id << " to "
            << list.in_memory_dialog_total_count_ << '/' << list.server_dialog_total_count_ << '+'
            << list.secret_chat_total_count_ << '/' << list.unread_dialog_total_count_ << '/' << unread_unmuted_count
            << '/' << list.unread_dialog_marked_count_ << '/' << unread_unmuted_marked_count << " from " << source
            << " and " << dialog_id;
  if (need_postpone) {
    postponed_unread_chat_count_updates_.insert(folder_id);
  } else {
    postponed_unread_chat_count_updates_.erase(folder_id);
    send_closure(G()->td(), &Td::send_update, get_update_unread_chat_count_object(folder_id, list));
  }
}

void MessagesManager::send_update_chat_read_inbox(const Dialog *d, bool force, const char *source) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_read_inbox from "
                                        << source;
  on_dialog_updated(d->dialog_id, source);
  if (!force && (running_get_difference_ || running_get_channel_difference(d->dialog_id) ||
                 get_channel_difference_to_logevent_id_.count(d->dialog_id) != 0)) {
    LOG(INFO) << "Postpone updateChatReadInbox in " << d->dialog_id << "(" << get_dialog_title(d->dialog_id) << ") to "
              << d->server_unread_count << " + " << d->local_unread_count << " from " << source;
    postponed_chat_read_inbox_updates_.insert(d->dialog_id);
  } else {
    postponed_chat_read_inbox_updates_.erase(d->dialog_id);
    LOG(INFO) << "Send updateChatReadInbox in " << d->dialog_id << "(" << get_dialog_title(d->dialog_id) << ") to "
              << d->server_unread_count << " + " << d->local_unread_count << " from " << source;
    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateChatReadInbox>(d->dialog_id.get(), d->last_read_inbox_message_id.get(),
                                                             d->server_unread_count + d->local_unread_count));
  }
}

void MessagesManager::send_update_chat_read_outbox(const Dialog *d) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_read_outbox";
  on_dialog_updated(d->dialog_id, "send_update_chat_read_outbox");
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatReadOutbox>(d->dialog_id.get(), d->last_read_outbox_message_id.get()));
}

void MessagesManager::send_update_chat_unread_mention_count(const Dialog *d) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_unread_mention_count";
  LOG(INFO) << "Update unread mention message count in " << d->dialog_id << " to " << d->unread_mention_count;
  on_dialog_updated(d->dialog_id, "send_update_chat_unread_mention_count");
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatUnreadMentionCount>(d->dialog_id.get(), d->unread_mention_count));
}

void MessagesManager::send_update_chat_is_sponsored(const Dialog *d) const {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_is_sponsored";
  bool is_sponsored = d->order == SPONSORED_DIALOG_ORDER;
  LOG(INFO) << "Update chat is sponsored for " << d->dialog_id;
  send_closure(
      G()->td(), &Td::send_update,
      make_tl_object<td_api::updateChatIsSponsored>(d->dialog_id.get(), is_sponsored, get_dialog_public_order(d)));
}

void MessagesManager::send_update_chat_online_member_count(DialogId dialog_id, int32 online_member_count) const {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatOnlineMemberCount>(dialog_id.get(), online_member_count));
}

void MessagesManager::send_update_chat_chat_list(const Dialog *d) const {
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_chat_list";
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatChatList>(d->dialog_id.get(), get_chat_list_object(d)));
}

void MessagesManager::send_update_secret_chats_with_user_action_bar(const Dialog *d) const {
  if (d->dialog_id.get_type() != DialogType::User) {
    return;
  }

  td_->contacts_manager_->for_each_secret_chat_with_user(
      d->dialog_id.get_user_id(), [this, user_d = d](SecretChatId secret_chat_id) {
        DialogId dialog_id(secret_chat_id);
        auto secret_chat_d = get_dialog(dialog_id);  // must not create the dialog
        if (secret_chat_d != nullptr && secret_chat_d->is_update_new_chat_sent) {
          send_closure(
              G()->td(), &Td::send_update,
              td_api::make_object<td_api::updateChatActionBar>(dialog_id.get(), get_chat_action_bar_object(user_d)));
        }
      });
}

void MessagesManager::send_update_chat_action_bar(const Dialog *d) {
  CHECK(d != nullptr);
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_action_bar";
  on_dialog_updated(d->dialog_id, "send_update_chat_action_bar");
  send_closure(G()->td(), &Td::send_update,
               td_api::make_object<td_api::updateChatActionBar>(d->dialog_id.get(), get_chat_action_bar_object(d)));

  send_update_secret_chats_with_user_action_bar(d);
}

void MessagesManager::send_update_chat_has_scheduled_messages(Dialog *d) {
  if (d->scheduled_messages == nullptr && d->has_loaded_scheduled_messages_from_database) {
    set_dialog_has_scheduled_database_messages_impl(d, false);
  }

  bool has_scheduled_messages =
      d->has_scheduled_server_messages || d->has_scheduled_database_messages || d->scheduled_messages != nullptr;
  if (has_scheduled_messages == d->last_sent_has_scheduled_messages) {
    return;
  }
  d->last_sent_has_scheduled_messages = has_scheduled_messages;

  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in send_update_chat_has_scheduled_messages";
  send_closure(G()->td(), &Td::send_update,
               td_api::make_object<td_api::updateChatHasScheduledMessages>(d->dialog_id.get(), has_scheduled_messages));
}

void MessagesManager::on_send_message_get_quick_ack(int64 random_id) {
  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    LOG(ERROR) << "Receive quick ack about unknown message with random_id = " << random_id;
    return;
  }

  auto dialog_id = it->second.get_dialog_id();
  auto message_id = it->second.get_message_id();

  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateMessageSendAcknowledged>(dialog_id.get(), message_id.get()));
}

void MessagesManager::check_send_message_result(int64 random_id, DialogId dialog_id,
                                                const telegram_api::Updates *updates_ptr, const char *source) {
  CHECK(updates_ptr != nullptr);
  CHECK(source != nullptr);
  auto sent_messages = UpdatesManager::get_new_messages(updates_ptr);
  auto sent_messages_random_ids = UpdatesManager::get_sent_messages_random_ids(updates_ptr);
  if (sent_messages.size() != 1u || sent_messages_random_ids.size() != 1u ||
      *sent_messages_random_ids.begin() != random_id || get_message_dialog_id(*sent_messages[0]) != dialog_id) {
    LOG(ERROR) << "Receive wrong result for sending message with random_id " << random_id << " from " << source
               << " to " << dialog_id << ": " << oneline(to_string(*updates_ptr));
    if (dialog_id.get_type() == DialogType::Channel) {
      Dialog *d = get_dialog(dialog_id);
      CHECK(d != nullptr);
      get_channel_difference(dialog_id, d->pts, true, "check_send_message_result");
    } else {
      td_->updates_manager_->schedule_get_difference("check_send_message_result");
    }
    repair_dialog_scheduled_messages(dialog_id);
  }
}

FullMessageId MessagesManager::on_send_message_success(int64 random_id, MessageId new_message_id, int32 date,
                                                       FileId new_file_id, const char *source) {
  CHECK(source != nullptr);
  // do not try to run getDifference from this function
  if (DROP_UPDATES) {
    return {};
  }
  if (!new_message_id.is_valid()) {
    LOG(ERROR) << "Receive " << new_message_id << " as sent message from " << source;
    on_send_message_fail(
        random_id,
        Status::Error(500, "Internal server error: receive invalid message identifier as sent message identifier"));
    return {};
  }
  if (new_message_id.is_yet_unsent()) {
    LOG(ERROR) << "Receive " << new_message_id << " as sent message from " << source;
    on_send_message_fail(random_id,
                         Status::Error(500, "Internal server error: receive yet unsent message as sent message"));
    return {};
  }

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    LOG(ERROR) << "Result from sendMessage for " << new_message_id << " with random_id " << random_id << " sent at "
               << date << " comes from " << source << " after updateNewMessageId, but was not discarded by pts";
    if (debug_being_sent_messages_.count(random_id) == 0) {
      LOG(ERROR) << "Message with random_id " << random_id << " was mot sent";
      return {};
    }
    auto dialog_id = debug_being_sent_messages_[random_id];
    if (!dialog_id.is_valid()) {
      LOG(ERROR) << "Sent message is in invalid " << dialog_id;
      return {};
    }
    auto d = get_dialog(dialog_id);
    if (d == nullptr) {
      LOG(ERROR) << "Sent message is in not found " << dialog_id;
      return {};
    }
    dump_debug_message_op(d, 7);
    auto m = get_message_force(d, new_message_id, "on_send_message_success");
    if (m == nullptr) {
      LOG(ERROR) << new_message_id << " in " << dialog_id << " not found";
      return {};
    }
    LOG(ERROR) << "Result from sent " << (m->is_outgoing ? "outgoing" : "incoming")
               << (m->forward_info == nullptr ? " not" : "") << " forwarded " << new_message_id
               << " with content of the type " << m->content->get_type() << " in " << dialog_id
               << " comes after updateNewMessageId, current last new is " << d->last_new_message_id << ", last is "
               << d->last_message_id;
    return {};
  }

  auto dialog_id = it->second.get_dialog_id();
  auto old_message_id = it->second.get_message_id();

  if (new_message_id.is_local() && dialog_id.get_type() != DialogType::SecretChat) {
    LOG(ERROR) << "Receive " << new_message_id << " as sent message from " << source;
    on_send_message_fail(random_id, Status::Error(500, "Internal server error: receive local as sent message"));
    return {};
  }

  being_sent_messages_.erase(it);

  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  bool need_update_dialog_pos = false;
  unique_ptr<Message> sent_message = delete_message(d, old_message_id, false, &need_update_dialog_pos, source);
  if (sent_message == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send update to the user, because the message has already been deleted
    LOG(INFO) << "Delete already deleted sent " << new_message_id << " from server";
    delete_message_from_server(dialog_id, new_message_id, true);
    return {};
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << old_message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(d, 5);
  }

  // imitation of update_message(d, sent_message.get(), std::move(new_message), &need_update_dialog_pos);
  if (date <= 0) {
    LOG(ERROR) << "Receive " << new_message_id << " in " << dialog_id << " with wrong date " << date;
  } else {
    LOG_CHECK(sent_message->date > 0) << old_message_id << ' ' << sent_message->message_id << ' ' << new_message_id
                                      << ' ' << sent_message->date << ' ' << date;
    sent_message->date = date;
    CHECK(d->last_message_id != old_message_id);
  }

  // reply_to message may be already deleted
  // but can't use get_message_force for check, because the message can be already unloaded from the memory
  // if (get_message_force(d, sent_message->reply_to_message_id, "on_send_message_success 2") == nullptr) {
  //   sent_message->reply_to_message_id = MessageId();
  // }

  if (merge_message_content_file_id(td_, sent_message->content.get(), new_file_id)) {
    send_update_message_content(dialog_id, old_message_id, sent_message->content.get(), sent_message->date,
                                sent_message->is_content_secret, source);
  }

  set_message_id(sent_message, new_message_id);

  sent_message->from_database = false;
  sent_message->have_previous = true;
  sent_message->have_next = true;

  bool need_update = true;
  Message *m = add_message_to_dialog(d, std::move(sent_message), true, &need_update, &need_update_dialog_pos, source);
  LOG_CHECK(m != nullptr) << td_->contacts_manager_->get_my_id() << " " << dialog_id << " " << old_message_id << " "
                          << new_message_id << " " << d->last_clear_history_message_id << " "
                          << d->max_unavailable_message_id << " " << d->last_message_id << " " << d->last_new_message_id
                          << " " << d->last_assigned_message_id << " " << have_input_peer(dialog_id, AccessRights::Read)
                          << " " << debug_add_message_to_dialog_fail_reason_ << " " << source;

  send_update_message_send_succeeded(d, old_message_id, m);
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "on_send_message_success");
  }
  try_add_active_live_location(dialog_id, m);
  return {dialog_id, new_message_id};
}

void MessagesManager::on_send_message_file_part_missing(int64 random_id, int bad_part) {
  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // we can't receive fail more than once
    // but message can be successfully sent before
    LOG(WARNING) << "Receive FILE_PART_" << bad_part
                 << "_MISSING about successfully sent message with random_id = " << random_id;
    return;
  }

  auto full_message_id = it->second;

  being_sent_messages_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send error to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat " << full_message_id;
    return;
  }

  auto dialog_id = full_message_id.get_dialog_id();
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << m->message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(get_dialog(dialog_id), 5);
  }

  if (dialog_id.get_type() == DialogType::SecretChat) {
    CHECK(!m->message_id.is_scheduled());
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);

    // need to change message random_id before resending
    do {
      m->random_id = Random::secure_int64();
    } while (m->random_id == 0 || message_random_ids_.find(m->random_id) != message_random_ids_.end());
    message_random_ids_.insert(m->random_id);

    delete_random_id_to_message_id_correspondence(d, random_id, m->message_id);
    add_random_id_to_message_id_correspondence(d, m->random_id, m->message_id);

    auto logevent = SendMessageLogEvent(dialog_id, m);
    auto storer = LogEventStorerImpl<SendMessageLogEvent>(logevent);
    CHECK(m->send_message_logevent_id != 0);
    binlog_rewrite(G()->td_db()->get_binlog(), m->send_message_logevent_id, LogEvent::HandlerType::SendMessage, storer);
  }

  do_send_message(dialog_id, m, {bad_part});
}

void MessagesManager::on_send_message_file_reference_error(int64 random_id) {
  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // we can't receive fail more than once
    // but message can be successfully sent before
    LOG(WARNING) << "Receive file reference invalid error about successfully sent message with random_id = "
                 << random_id;
    return;
  }

  auto full_message_id = it->second;

  being_sent_messages_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send error to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat " << full_message_id;
    return;
  }

  auto dialog_id = full_message_id.get_dialog_id();
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << m->message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(get_dialog(dialog_id), 5);
  }

  if (dialog_id.get_type() == DialogType::SecretChat) {
    CHECK(!m->message_id.is_scheduled());
    Dialog *d = get_dialog(dialog_id);
    CHECK(d != nullptr);

    // need to change message random_id before resending
    do {
      m->random_id = Random::secure_int64();
    } while (m->random_id == 0 || message_random_ids_.find(m->random_id) != message_random_ids_.end());
    message_random_ids_.insert(m->random_id);

    delete_random_id_to_message_id_correspondence(d, random_id, m->message_id);
    add_random_id_to_message_id_correspondence(d, m->random_id, m->message_id);

    auto logevent = SendMessageLogEvent(dialog_id, m);
    auto storer = LogEventStorerImpl<SendMessageLogEvent>(logevent);
    CHECK(m->send_message_logevent_id != 0);
    binlog_rewrite(G()->td_db()->get_binlog(), m->send_message_logevent_id, LogEvent::HandlerType::SendMessage, storer);
  }

  do_send_message(dialog_id, m, {-1});
}

void MessagesManager::on_send_media_group_file_reference_error(DialogId dialog_id, vector<int64> random_ids) {
  int64 media_album_id = 0;
  vector<MessageId> message_ids;
  vector<Message *> messages;
  for (auto &random_id : random_ids) {
    auto it = being_sent_messages_.find(random_id);
    if (it == being_sent_messages_.end()) {
      // we can't receive fail more than once
      // but message can be successfully sent before
      LOG(ERROR) << "Receive file reference invalid error about successfully sent message with random_id = "
                 << random_id;
      continue;
    }

    auto full_message_id = it->second;

    being_sent_messages_.erase(it);

    Message *m = get_message(full_message_id);
    if (m == nullptr) {
      // message has already been deleted by the user or sent to inaccessible channel
      // don't need to send error to the user, because the message has already been deleted
      // and there is nothing to be deleted from the server
      LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat " << full_message_id;
      continue;
    }

    CHECK(m->media_album_id != 0);
    CHECK(media_album_id == 0 || media_album_id == m->media_album_id);
    media_album_id = m->media_album_id;

    CHECK(dialog_id == full_message_id.get_dialog_id());
    message_ids.push_back(m->message_id);
    messages.push_back(m);
  }

  CHECK(dialog_id.get_type() != DialogType::SecretChat);

  if (message_ids.empty()) {
    // all messages was deleted, nothing to do
    return;
  }

  auto &request = pending_message_group_sends_[media_album_id];
  CHECK(!request.dialog_id.is_valid());
  CHECK(request.finished_count == 0);
  CHECK(request.results.empty());
  request.dialog_id = dialog_id;
  request.message_ids = std::move(message_ids);
  request.is_finished.resize(request.message_ids.size(), false);
  for (size_t i = 0; i < request.message_ids.size(); i++) {
    request.results.push_back(Status::OK());
  }

  for (auto m : messages) {
    do_send_message(dialog_id, m, {-1});
  }
}

void MessagesManager::on_send_message_fail(int64 random_id, Status error) {
  CHECK(error.is_error());

  auto it = being_sent_messages_.find(random_id);
  if (it == being_sent_messages_.end()) {
    // we can't receive fail more than once
    // but message can be successfully sent before
    if (error.code() != NetQuery::Cancelled) {
      auto debug_it = debug_being_sent_messages_.find(random_id);
      if (debug_it == debug_being_sent_messages_.end()) {
        LOG(ERROR) << "Message with random_id " << random_id << " was not sent";
        return;
      }
      auto dialog_id = debug_it->second;
      if (!dialog_id.is_valid()) {
        LOG(ERROR) << "Sent message is in invalid " << dialog_id;
        return;
      }
      if (!have_dialog(dialog_id)) {
        LOG(ERROR) << "Sent message is in not found " << dialog_id;
        return;
      }
      LOG(ERROR) << "Receive error " << error << " about successfully sent message with random_id = " << random_id
                 << " in " << dialog_id;
      dump_debug_message_op(get_dialog(dialog_id), 7);
    }
    return;
  }

  auto full_message_id = it->second;

  being_sent_messages_.erase(it);

  Message *m = get_message(full_message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send error to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    LOG(INFO) << "Fail to send already deleted by the user or sent to inaccessible chat " << full_message_id;
    return;
  }
  LOG_IF(ERROR, error.code() == NetQuery::Cancelled)
      << "Receive error " << error << " about sent message with random_id = " << random_id;

  auto dialog_id = full_message_id.get_dialog_id();
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << m->message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(get_dialog(dialog_id), 5);
  }

  int error_code = error.code();
  string error_message = error.message().str();
  switch (error_code) {
    case 420:
      error_code = 429;
      LOG(ERROR) << "Receive error 420: " << error_message;
      break;
    case 429:
      // nothing special, error description has already been changed
      LOG_IF(ERROR, !begins_with(error_message, "Too Many Requests: retry after "))
          << "Wrong error message: " << error_message;
      break;
    case 400:
      if (error.message() == "MESSAGE_TOO_LONG") {
        error_message = "Message is too long";
        // TODO move check to send_message
      } else if (error.message() == "INPUT_USER_DEACTIVATED") {
        error_code = 403;
        error_message = "User is deactivated";
      } else if (error.message() == "USER_IS_BLOCKED") {
        error_code = 403;
        if (td_->auth_manager_->is_bot()) {
          switch (dialog_id.get_type()) {
            case DialogType::User:
              error_message = "Bot was blocked by the user";
              break;
            case DialogType::Chat:
            case DialogType::Channel:
              error_message = "Bot was kicked from the chat";
              break;
            case DialogType::SecretChat:
              break;
            case DialogType::None:
            default:
              UNREACHABLE();
          }
        } else {
          switch (dialog_id.get_type()) {
            case DialogType::User:
              error_message = "User was blocked by the other user";
              break;
            case DialogType::Chat:
            case DialogType::Channel:
              error_message = "User is not in the chat";
              break;
            case DialogType::SecretChat:
              break;
            case DialogType::None:
            default:
              UNREACHABLE();
          }
        }
        // TODO add check to send_message
      } else if (error.message() == "USER_IS_BOT") {
        if (td_->auth_manager_->is_bot() && dialog_id.get_type() == DialogType::User) {
          error_code = 403;
          if (td_->contacts_manager_->is_user_bot(dialog_id.get_user_id())) {
            error_message = "Bot can't send messages to bots";
          } else {
            error_message = "Bot can't send messages to the user";
          }
          // TODO move check to send_message
        }
      } else if (error.message() == "PEER_ID_INVALID") {
        error_code = 403;
        if (td_->auth_manager_->is_bot()) {
          error_message = "Bot can't initiate conversation with a user";
        }
      } else if (error.message() == "WC_CONVERT_URL_INVALID" || error.message() == "EXTERNAL_URL_INVALID") {
        error_message = "Wrong HTTP URL specified";
      } else if (error.message() == "WEBPAGE_CURL_FAILED") {
        error_message = "Failed to get HTTP URL content";
      } else if (error.message() == "WEBPAGE_MEDIA_EMPTY") {
        error_message = "Wrong type of the web page content";
      } else if (error.message() == "MEDIA_EMPTY") {
        auto content_type = m->content->get_type();
        if (content_type == MessageContentType::Game) {
          error_message = "Wrong game short name specified";
        } else if (content_type == MessageContentType::Invoice) {
          error_message = "Wrong invoice information specified";
        } else if (content_type == MessageContentType::Poll) {
          error_message = "Wrong poll data specified";
        } else {
          error_message = "Wrong file identifier/HTTP URL specified";
        }
      } else if (error.message() == "PHOTO_EXT_INVALID") {
        error_message = "Photo has unsupported extension. Use one of .jpg, .jpeg, .gif, .png, .tif or .bmp";
      }
      break;
    case 403:
      if (error.message() == "MESSAGE_DELETE_FORBIDDEN") {
        error_code = 400;
        error_message = "Message can't be deleted";
      } else if (error.message() != "CHANNEL_PUBLIC_GROUP_NA" && error.message() != "USER_IS_BLOCKED" &&
                 error.message() != "USER_BOT_INVALID" && error.message() != "USER_DELETED") {
        error_code = 400;
      }
      break;
    // TODO other codes
    default:
      break;
  }
  if (error.message() == "REPLY_MARKUP_INVALID") {
    if (m->reply_markup == nullptr) {
      LOG(ERROR) << "Receive " << error.message() << " for " << oneline(to_string(get_message_object(dialog_id, m)));
    } else {
      LOG(ERROR) << "Receive " << error.message() << " for " << full_message_id << " with keyboard "
                 << *m->reply_markup;
    }
  }
  LOG_IF(WARNING, error_code != 403) << "Fail to send " << full_message_id << " with the error " << error;
  if (error_code <= 0) {
    error_code = 500;
  }
  fail_send_message(full_message_id, error_code, error_message);
}

MessageId MessagesManager::get_next_message_id(Dialog *d, MessageType type) {
  CHECK(d != nullptr);
  MessageId last_message_id =
      std::max({d->last_message_id, d->last_new_message_id, d->last_database_message_id, d->last_assigned_message_id,
                d->last_clear_history_message_id, d->deleted_last_message_id, d->max_unavailable_message_id,
                d->max_added_message_id});
  if (last_message_id < d->last_read_inbox_message_id &&
      d->last_read_inbox_message_id < d->last_new_message_id.get_next_server_message_id()) {
    last_message_id = d->last_read_inbox_message_id;
  }
  if (last_message_id < d->last_read_outbox_message_id &&
      d->last_read_outbox_message_id < d->last_new_message_id.get_next_server_message_id()) {
    last_message_id = d->last_read_outbox_message_id;
  }

  d->last_assigned_message_id = last_message_id.get_next_message_id(type);
  if (d->last_assigned_message_id > MessageId::max()) {
    LOG(FATAL) << "Force restart because of message_id overflow: " << d->last_assigned_message_id;
  }
  CHECK(d->last_assigned_message_id.is_valid());
  return d->last_assigned_message_id;
}

MessageId MessagesManager::get_next_yet_unsent_message_id(Dialog *d) {
  return get_next_message_id(d, MessageType::YetUnsent);
}

MessageId MessagesManager::get_next_local_message_id(Dialog *d) {
  return get_next_message_id(d, MessageType::Local);
}

MessageId MessagesManager::get_next_yet_unsent_scheduled_message_id(const Dialog *d, int32 date) {
  CHECK(date > 0);
  auto it = MessagesConstIterator(d, MessageId(ScheduledServerMessageId(), date + 1, true));
  int32 prev_date = 0;
  if (*it != nullptr) {
    prev_date = (*it)->message_id.get_scheduled_message_date();
  }
  if (prev_date < date) {
    return MessageId(ScheduledServerMessageId(1), date).get_next_message_id(MessageType::YetUnsent);
  }
  CHECK(*it != nullptr);
  return (*it)->message_id.get_next_message_id(MessageType::YetUnsent);
}

void MessagesManager::fail_send_message(FullMessageId full_message_id, int error_code, const string &error_message) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  MessageId old_message_id = full_message_id.get_message_id();
  CHECK(old_message_id.is_valid() || old_message_id.is_valid_scheduled());
  CHECK(old_message_id.is_yet_unsent());

  bool need_update_dialog_pos = false;
  unique_ptr<Message> message = delete_message(d, old_message_id, false, &need_update_dialog_pos, "fail send message");
  if (message == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    // don't need to send update to the user, because the message has already been deleted
    // and there is nothing to be deleted from the server
    return;
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    // LOG(ERROR) << "Found " << old_message_id << " in inaccessible " << dialog_id;
    // dump_debug_message_op(d, 5);
  }

  MessageId new_message_id =
      old_message_id.get_next_message_id(MessageType::Local);  // trying to not change message place
  if (!old_message_id.is_scheduled()) {
    if (get_message_force(d, new_message_id, "fail_send_message") != nullptr ||
        d->deleted_message_ids.count(new_message_id) || new_message_id <= d->last_clear_history_message_id) {
      new_message_id = get_next_local_message_id(d);
    } else if (new_message_id > d->last_assigned_message_id) {
      d->last_assigned_message_id = new_message_id;
    }
  } else {
    while (get_message_force(d, new_message_id, "fail_send_message") != nullptr ||
           d->deleted_message_ids.count(new_message_id)) {
      new_message_id = new_message_id.get_next_message_id(MessageType::Local);
    }
  }

  set_message_id(message, new_message_id);
  if (old_message_id.is_scheduled()) {
    CHECK(message->message_id.is_valid_scheduled());
  } else {
    CHECK(message->message_id.is_valid());
  }
  message->is_failed_to_send = true;
  message->send_error_code = error_code;
  message->send_error_message = error_message;
  message->try_resend_at = 0.0;
  Slice retry_after_prefix("Too Many Requests: retry after ");
  if (error_code == 429 && begins_with(error_message, retry_after_prefix)) {
    auto r_retry_after = to_integer_safe<int32>(error_message.substr(retry_after_prefix.size()));
    if (r_retry_after.is_ok() && r_retry_after.ok() > 0) {
      message->try_resend_at = Time::now() + r_retry_after.ok();
    }
  }
  update_failed_to_send_message_content(td_, message->content);

  message->from_database = false;
  message->have_previous = true;
  message->have_next = true;

  bool need_update = false;
  Message *m = add_message_to_dialog(dialog_id, std::move(message), false, &need_update, &need_update_dialog_pos,
                                     "fail_send_message");
  LOG_CHECK(m != nullptr) << "Failed to add failed to send " << new_message_id << " to " << dialog_id << " due to "
                          << debug_add_message_to_dialog_fail_reason_;

  LOG(INFO) << "Send updateMessageSendFailed for " << full_message_id;
  d->yet_unsent_message_id_to_persistent_message_id.emplace(old_message_id, m->message_id);
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateMessageSendFailed>(get_message_object(dialog_id, m), old_message_id.get(),
                                                               error_code, error_message));
  if (need_update_dialog_pos) {
    send_update_chat_last_message(d, "fail_send_message");
  }
}

void MessagesManager::fail_send_message(FullMessageId full_message_id, Status error) {
  fail_send_message(full_message_id, error.code() > 0 ? error.code() : 500,
                    error.message().str());  // TODO CHECK that status has always a code
}

void MessagesManager::fail_edit_message_media(FullMessageId full_message_id, Status &&error) {
  auto dialog_id = full_message_id.get_dialog_id();
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  MessageId message_id = full_message_id.get_message_id();
  CHECK(message_id.is_any_server());

  auto m = get_message(d, message_id);
  if (m == nullptr) {
    // message has already been deleted by the user or sent to inaccessible channel
    return;
  }
  CHECK(m->edited_content != nullptr);
  m->edit_promise.set_error(std::move(error));
  cancel_edit_message_media(dialog_id, m, "Failed to edit message. MUST BE IGNORED");
}

void MessagesManager::on_update_dialog_draft_message(DialogId dialog_id,
                                                     tl_object_ptr<telegram_api::DraftMessage> &&draft_message) {
  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive update chat draft in invalid " << dialog_id;
    return;
  }
  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Ignore update chat draft in unknown " << dialog_id;
    if (!have_input_peer(dialog_id, AccessRights::Read)) {
      LOG(ERROR) << "Have no read access to " << dialog_id << " to repair chat draft message";
    } else {
      send_get_dialog_query(dialog_id, Promise<Unit>());
    }
    return;
  }
  update_dialog_draft_message(d, get_draft_message(td_->contacts_manager_.get(), std::move(draft_message)), true, true);
}

bool MessagesManager::update_dialog_draft_message(Dialog *d, unique_ptr<DraftMessage> &&draft_message, bool from_update,
                                                  bool need_update_dialog_pos) {
  CHECK(d != nullptr);
  if (from_update && d->is_opened && d->draft_message != nullptr) {
    // send the update anyway, despite it shouldn't be applied client-side
    // return false;
  }
  if (draft_message == nullptr) {
    if (d->draft_message != nullptr) {
      d->draft_message = nullptr;
      if (need_update_dialog_pos) {
        update_dialog_pos(d, false, "update_dialog_draft_message", false);
      }
      send_update_chat_draft_message(d);
      return true;
    }
  } else {
    if (d->draft_message != nullptr && d->draft_message->reply_to_message_id == draft_message->reply_to_message_id &&
        d->draft_message->input_message_text == draft_message->input_message_text) {
      if (d->draft_message->date < draft_message->date) {
        if (need_update_dialog_pos) {
          update_dialog_pos(d, false, "update_dialog_draft_message 2");
        }
        d->draft_message->date = draft_message->date;
        return true;
      }
    } else {
      if (!from_update || d->draft_message == nullptr || d->draft_message->date <= draft_message->date) {
        d->draft_message = std::move(draft_message);
        if (need_update_dialog_pos) {
          update_dialog_pos(d, false, "update_dialog_draft_message 3", false);
        }
        send_update_chat_draft_message(d);
        return true;
      }
    }
  }
  return false;
}

void MessagesManager::on_update_dialog_is_pinned(FolderId folder_id, DialogId dialog_id, bool is_pinned) {
  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive pin of invalid " << dialog_id;
    return;
  }

  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    LOG(INFO) << "Can't apply updateDialogPinned with unknown " << dialog_id;
    on_update_pinned_dialogs(folder_id);
    return;
  }

  set_dialog_folder_id(d, folder_id);
  if (!is_pinned && d->pinned_order == DEFAULT_ORDER) {
    return;
  }
  set_dialog_is_pinned(d, is_pinned);
  update_dialog_pos(d, false, "on_update_dialog_is_pinned");
}

void MessagesManager::on_update_pinned_dialogs(FolderId folder_id) {
  // TODO logevent + delete_logevent_promise
  auto query_promise = PromiseCreator::lambda([actor_id = actor_id(this), folder_id](Unit /* ignore result */) {
    send_closure(actor_id, &MessagesManager::reload_pinned_dialogs, folder_id, Promise<Unit>());
  });

  // max ordinary pinned dialogs + max pinned secret chats + sponsored proxy
  size_t needed_dialogs = 2 * get_pinned_dialogs_limit(folder_id) + (folder_id == FolderId::main() ? 1 : 0);
  auto &list = get_dialog_list(folder_id);
  if (list.ordered_dialogs_.size() >= needed_dialogs) {
    query_promise.set_value(Unit());
  } else {
    load_dialog_list(folder_id, narrow_cast<int32>(needed_dialogs - list.ordered_dialogs_.size()), true,
                     std::move(query_promise));
  }
}

void MessagesManager::on_update_dialog_is_marked_as_unread(DialogId dialog_id, bool is_marked_as_unread) {
  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive marking as unread of invalid " << dialog_id;
    return;
  }

  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    // nothing to do
    return;
  }

  if (is_marked_as_unread == d->is_marked_as_unread) {
    return;
  }

  set_dialog_is_marked_as_unread(d, is_marked_as_unread);
}

void MessagesManager::set_dialog_is_marked_as_unread(Dialog *d, bool is_marked_as_unread) {
  if (td_->auth_manager_->is_bot()) {
    // just in case
    return;
  }

  CHECK(d != nullptr);
  CHECK(d->is_marked_as_unread != is_marked_as_unread);
  d->is_marked_as_unread = is_marked_as_unread;
  on_dialog_updated(d->dialog_id, "set_dialog_is_marked_as_unread");

  LOG(INFO) << "Set " << d->dialog_id << " is marked as unread to " << is_marked_as_unread;
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in set_dialog_is_marked_as_unread";
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatIsMarkedAsUnread>(d->dialog_id.get(), is_marked_as_unread));

  auto &list = get_dialog_list(d->folder_id);
  if (d->server_unread_count + d->local_unread_count == 0 && need_unread_counter(d->order) &&
      list.is_dialog_unread_count_inited_) {
    int32 delta = d->is_marked_as_unread ? 1 : -1;
    list.unread_dialog_total_count_ += delta;
    list.unread_dialog_marked_count_ += delta;
    if (is_dialog_muted(d)) {
      list.unread_dialog_muted_count_ += delta;
      list.unread_dialog_muted_marked_count_ += delta;
    }
    send_update_unread_chat_count(d->folder_id, d->dialog_id, true, "set_dialog_is_marked_as_unread");
  }
}

void MessagesManager::on_update_dialog_pinned_message_id(DialogId dialog_id, MessageId pinned_message_id) {
  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive pinned message in invalid " << dialog_id;
    return;
  }
  if (!pinned_message_id.is_valid() && pinned_message_id != MessageId()) {
    LOG(ERROR) << "Receive as pinned message " << pinned_message_id;
    return;
  }

  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    // nothing to do
    return;
  }

  if (d->pinned_message_id == pinned_message_id) {
    LOG(INFO) << "Pinned message in " << d->dialog_id << " is still " << pinned_message_id;
    if (!d->is_pinned_message_id_inited) {
      d->is_pinned_message_id_inited = true;
      on_dialog_updated(dialog_id, "on_update_dialog_pinned_message_id");
    }
    return;
  }

  set_dialog_pinned_message_id(d, pinned_message_id);
}

void MessagesManager::set_dialog_pinned_message_id(Dialog *d, MessageId pinned_message_id) {
  CHECK(d != nullptr);
  CHECK(d->pinned_message_id != pinned_message_id);
  d->pinned_message_id = pinned_message_id;
  d->is_pinned_message_id_inited = true;
  on_dialog_updated(d->dialog_id, "set_dialog_pinned_message_id");

  LOG(INFO) << "Set " << d->dialog_id << " pinned message to " << pinned_message_id;
  LOG_CHECK(d->is_update_new_chat_sent) << "Wrong " << d->dialog_id << " in set_dialog_pinned_message_id";
  send_closure(G()->td(), &Td::send_update,
               make_tl_object<td_api::updateChatPinnedMessage>(d->dialog_id.get(), pinned_message_id.get()));
}

void MessagesManager::repair_dialog_scheduled_messages(DialogId dialog_id) {
  if (td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    return;
  }

  // TODO create logevent
  get_dialog_scheduled_messages(dialog_id, PromiseCreator::lambda([actor_id = actor_id(this), dialog_id](Unit) {
                                  send_closure(G()->messages_manager(), &MessagesManager::get_dialog_scheduled_messages,
                                               dialog_id, Promise<Unit>());
                                }));
}

void MessagesManager::on_update_dialog_has_scheduled_server_messages(DialogId dialog_id,
                                                                     bool has_scheduled_server_messages) {
  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Receive has_scheduled_server_messages in invalid " << dialog_id;
    return;
  }
  if (td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    return;
  }

  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    // nothing to do
    return;
  }

  LOG(INFO) << "Receive has_scheduled_server_messages = " << has_scheduled_server_messages << " in " << dialog_id;
  if (d->has_scheduled_server_messages != has_scheduled_server_messages) {
    set_dialog_has_scheduled_server_messages(d, has_scheduled_server_messages);
  }
}

void MessagesManager::set_dialog_has_scheduled_server_messages(Dialog *d, bool has_scheduled_server_messages) {
  CHECK(d != nullptr);
  CHECK(d->has_scheduled_server_messages != has_scheduled_server_messages);
  d->has_scheduled_server_messages = has_scheduled_server_messages;
  repair_dialog_scheduled_messages(d->dialog_id);
  on_dialog_updated(d->dialog_id, "set_dialog_has_scheduled_server_messages");

  LOG(INFO) << "Set " << d->dialog_id << " has_scheduled_server_messages to " << has_scheduled_server_messages;

  send_update_chat_has_scheduled_messages(d);
}

void MessagesManager::set_dialog_has_scheduled_database_messages(DialogId dialog_id,
                                                                 bool has_scheduled_database_messages) {
  return set_dialog_has_scheduled_database_messages_impl(get_dialog(dialog_id), has_scheduled_database_messages);
}

void MessagesManager::set_dialog_has_scheduled_database_messages_impl(Dialog *d, bool has_scheduled_database_messages) {
  CHECK(d != nullptr);
  if (d->has_scheduled_database_messages == has_scheduled_database_messages) {
    return;
  }

  if (d->has_scheduled_database_messages && d->scheduled_messages != nullptr &&
      !d->scheduled_messages->message_id.is_yet_unsent()) {
    // to prevent race between add_message_to_database and check of has_scheduled_database_messages
    return;
  }

  CHECK(G()->parameters().use_message_db);

  d->has_scheduled_database_messages = has_scheduled_database_messages;
  on_dialog_updated(d->dialog_id, "set_dialog_has_scheduled_database_messages");
}

void MessagesManager::on_update_dialog_folder_id(DialogId dialog_id, FolderId folder_id) {
  auto d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    // nothing to do
    return;
  }

  set_dialog_folder_id(d, folder_id);
}

void MessagesManager::set_dialog_folder_id(Dialog *d, FolderId folder_id) {
  CHECK(d != nullptr);

  if (d->folder_id == folder_id) {
    if (!d->is_folder_id_inited) {
      LOG(INFO) << "Folder of " << d->dialog_id << " is still " << folder_id;
      d->is_folder_id_inited = true;
      on_dialog_updated(d->dialog_id, "set_dialog_folder_id");
    }
    return;
  }

  LOG(INFO) << "Change " << d->dialog_id << " folder from " << d->folder_id << " to " << folder_id;

  // first remove the dialog from the old chat list
  // this will send updateChatChatList if needed
  if (d->pinned_order != DEFAULT_ORDER) {
    set_dialog_is_pinned(d, false);
  }
  set_dialog_order(d, DEFAULT_ORDER, true, false, "set_dialog_folder_id old");

  // change the folder
  d->folder_id = folder_id;
  d->is_folder_id_inited = true;

  // update dialog position in the new folder
  // this will send updateChatChatList if needed
  update_dialog_pos(d, false, "set_dialog_folder_id new", true, false);

  on_dialog_updated(d->dialog_id, "set_dialog_folder_id");
}

void MessagesManager::on_create_new_dialog_success(int64 random_id, tl_object_ptr<telegram_api::Updates> &&updates,
                                                   DialogType expected_type, Promise<Unit> &&promise) {
  auto sent_messages = UpdatesManager::get_new_messages(updates.get());
  auto sent_messages_random_ids = UpdatesManager::get_sent_messages_random_ids(updates.get());
  if (sent_messages.size() != 1u || sent_messages_random_ids.size() != 1u) {
    LOG(ERROR) << "Receive wrong result for create group or channel chat " << oneline(to_string(updates));
    return on_create_new_dialog_fail(random_id, Status::Error(500, "Unsupported server response"), std::move(promise));
  }

  auto message = *sent_messages.begin();
  // int64 message_random_id = *sent_messages_random_ids.begin();
  // TODO check that message_random_id equals random_id after messages_createChat will be updated

  auto dialog_id = get_message_dialog_id(*message);
  if (dialog_id.get_type() != expected_type) {
    return on_create_new_dialog_fail(random_id, Status::Error(500, "Chat of wrong type has been created"),
                                     std::move(promise));
  }

  auto it = created_dialogs_.find(random_id);
  CHECK(it != created_dialogs_.end());
  CHECK(it->second == DialogId());

  it->second = dialog_id;

  const Dialog *d = get_dialog(dialog_id);
  if (d != nullptr && d->last_new_message_id.is_valid()) {
    // dialog have been already created and at least one non-temporary message was added,
    // i.e. we are not interested in the creation of dialog by searchMessages
    // then messages have already been added, so just set promise
    return promise.set_value(Unit());
  }

  if (pending_created_dialogs_.find(dialog_id) == pending_created_dialogs_.end()) {
    pending_created_dialogs_.emplace(dialog_id, std::move(promise));
  } else {
    LOG(ERROR) << dialog_id << " returned twice as result of chat creation";
    return on_create_new_dialog_fail(random_id, Status::Error(500, "Chat was created earlier"), std::move(promise));
  }

  td_->updates_manager_->on_get_updates(std::move(updates));
}

void MessagesManager::on_create_new_dialog_fail(int64 random_id, Status error, Promise<Unit> &&promise) {
  LOG(INFO) << "Clean up creation of group or channel chat";
  auto it = created_dialogs_.find(random_id);
  CHECK(it != created_dialogs_.end());
  CHECK(it->second == DialogId());
  created_dialogs_.erase(it);

  CHECK(error.is_error());
  promise.set_error(std::move(error));

  // repairing state by running get difference
  td_->updates_manager_->get_difference("on_create_new_dialog_fail");
}

void MessagesManager::on_dialog_bots_updated(DialogId dialog_id, vector<UserId> bot_user_ids) {
  if (td_->auth_manager_->is_bot()) {
    return;
  }

  auto d = get_dialog_force(dialog_id);
  if (d == nullptr || d->reply_markup_message_id == MessageId()) {
    return;
  }
  const Message *m = get_message_force(d, d->reply_markup_message_id, "on_dialog_bots_updated");
  if (m == nullptr || (m->sender_user_id.is_valid() && !td::contains(bot_user_ids, m->sender_user_id))) {
    LOG(INFO) << "Remove reply markup in " << dialog_id << ", because bot "
              << (m == nullptr ? UserId() : m->sender_user_id) << " isn't a member of the chat";
    set_dialog_reply_markup(d, MessageId());
  }
}

void MessagesManager::on_dialog_photo_updated(DialogId dialog_id) {
  auto d = get_dialog(dialog_id);  // called from update_user, must not create the dialog
  if (d != nullptr && d->is_update_new_chat_sent) {
    send_closure(G()->td(), &Td::send_update,
                 make_tl_object<td_api::updateChatPhoto>(
                     dialog_id.get(), get_chat_photo_object(td_->file_manager_.get(), get_dialog_photo(dialog_id))));
  }
}

void MessagesManager::on_dialog_title_updated(DialogId dialog_id) {
  auto d = get_dialog(dialog_id);  // called from update_user, must not create the dialog
  if (d != nullptr) {
    update_dialogs_hints(d);
    if (d->is_update_new_chat_sent) {
      send_closure(G()->td(), &Td::send_update,
                   make_tl_object<td_api::updateChatTitle>(dialog_id.get(), get_dialog_title(dialog_id)));
    }
  }
}

void MessagesManager::on_dialog_permissions_updated(DialogId dialog_id) {
  auto d = get_dialog(dialog_id);  // called from update_user, must not create the dialog
  if (d != nullptr && d->is_update_new_chat_sent) {
    send_closure(G()->td(), &Td::send_update,
                 td_api::make_object<td_api::updateChatPermissions>(
                     dialog_id.get(), get_dialog_permissions(dialog_id).get_chat_permissions_object()));
  }
}

void MessagesManager::on_dialog_user_is_contact_updated(DialogId dialog_id, bool is_contact) {
  CHECK(dialog_id.get_type() == DialogType::User);
  auto d = get_dialog(dialog_id);  // called from update_user, must not create the dialog
  if (d != nullptr && d->is_update_new_chat_sent) {
    if (d->know_action_bar) {
      if (is_contact) {
        if (d->can_block_user || d->can_add_contact) {
          d->can_block_user = false;
          d->can_add_contact = false;
          send_update_chat_action_bar(d);
        }
      } else {
        d->know_action_bar = false;
        if (have_input_peer(dialog_id, AccessRights::Read)) {
          repair_dialog_action_bar(dialog_id, "on_dialog_user_is_contact_updated");
        }
        // there is no need to change action bar
        on_dialog_updated(dialog_id, "on_dialog_user_is_contact_updated");
      }
    }
  }
}

void MessagesManager::on_dialog_user_is_blocked_updated(DialogId dialog_id, bool is_blocked) {
  CHECK(dialog_id.get_type() == DialogType::User);
  auto d = get_dialog(dialog_id);  // called from update_user_full, must not create the dialog
  if (d != nullptr && d->is_update_new_chat_sent) {
    if (d->know_action_bar) {
      if (is_blocked) {
        if (d->can_report_spam || d->can_share_phone_number || d->can_block_user || d->can_add_contact) {
          d->can_report_spam = false;
          d->can_share_phone_number = false;
          d->can_block_user = false;
          d->can_add_contact = false;
          send_update_chat_action_bar(d);
        }
      } else {
        d->know_action_bar = false;
        if (have_input_peer(dialog_id, AccessRights::Read)) {
          repair_dialog_action_bar(dialog_id, "on_dialog_user_is_blocked_updated");
        }
        // there is no need to change action bar
        on_dialog_updated(dialog_id, "on_dialog_user_is_blocked_updated");
      }
    }
  }
}

void MessagesManager::on_dialog_user_is_deleted_updated(DialogId dialog_id, bool is_deleted) {
  CHECK(dialog_id.get_type() == DialogType::User);
  auto d = get_dialog(dialog_id);  // called from update_user, must not create the dialog
  if (d != nullptr && d->is_update_new_chat_sent) {
    if (d->know_action_bar) {
      if (is_deleted) {
        if (d->can_share_phone_number || d->can_block_user || d->can_add_contact) {
          d->can_share_phone_number = false;
          d->can_block_user = false;
          d->can_add_contact = false;
          send_update_chat_action_bar(d);
        }
      } else {
        d->know_action_bar = false;
        if (have_input_peer(dialog_id, AccessRights::Read)) {
          repair_dialog_action_bar(dialog_id, "on_dialog_user_is_deleted_updated");
        }
        // there is no need to change action bar
        on_dialog_updated(dialog_id, "on_dialog_user_is_deleted_updated");
      }
    }
  }
}

DialogId MessagesManager::resolve_dialog_username(const string &username) const {
  auto cleaned_username = clean_username(username);
  auto it = resolved_usernames_.find(cleaned_username);
  if (it != resolved_usernames_.end()) {
    return it->second.dialog_id;
  }

  auto it2 = inaccessible_resolved_usernames_.find(cleaned_username);
  if (it2 != inaccessible_resolved_usernames_.end()) {
    return it2->second;
  }

  return DialogId();
}

DialogId MessagesManager::search_public_dialog(const string &username_to_search, bool force, Promise<Unit> &&promise) {
  string username = clean_username(username_to_search);
  if (username[0] == '@') {
    username = username.substr(1);
  }
  if (username.empty()) {
    promise.set_error(Status::Error(200, "Username is invalid"));
    return DialogId();
  }

  DialogId dialog_id;
  auto it = resolved_usernames_.find(username);
  if (it != resolved_usernames_.end()) {
    if (it->second.expires_at < Time::now()) {
      td_->create_handler<ResolveUsernameQuery>(Promise<>())->send(username);
    }
    dialog_id = it->second.dialog_id;
  } else {
    auto it2 = inaccessible_resolved_usernames_.find(username);
    if (it2 != inaccessible_resolved_usernames_.end()) {
      dialog_id = it2->second;
    }
  }

  if (dialog_id.is_valid()) {
    if (have_input_peer(dialog_id, AccessRights::Read)) {
      if (td_->auth_manager_->is_bot()) {
        force_create_dialog(dialog_id, "search public dialog", true);
      } else {
        const Dialog *d = get_dialog_force(dialog_id);
        if (!is_dialog_inited(d)) {
          send_get_dialog_query(dialog_id, std::move(promise));
          return DialogId();
        }
      }

      promise.set_value(Unit());
      return dialog_id;
    } else {
      // bot username maybe known despite there is no access_hash
      if (force || dialog_id.get_type() != DialogType::User) {
        force_create_dialog(dialog_id, "search public dialog", true);
        promise.set_value(Unit());
        return dialog_id;
      }
    }
  }

  td_->create_handler<ResolveUsernameQuery>(std::move(promise))->send(username);
  return DialogId();
}

void MessagesManager::send_get_dialog_notification_settings_query(DialogId dialog_id, Promise<Unit> &&promise) {
  if (td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    LOG(WARNING) << "Can't get notification settings for " << dialog_id;
    return promise.set_error(Status::Error(500, "Wrong getDialogNotificationSettings query"));
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    LOG(WARNING) << "Have no access to " << dialog_id << " to get notification settings";
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }

  LOG(INFO) << "Send GetDialogNotifySettingsQuery for " << dialog_id;
  auto &promises = get_dialog_notification_settings_queries_[dialog_id];
  promises.push_back(std::move(promise));
  if (promises.size() != 1) {
    // query has already been sent, just wait for the result
    return;
  }

  td_->create_handler<GetDialogNotifySettingsQuery>()->send(dialog_id);
}

void MessagesManager::send_get_scope_notification_settings_query(NotificationSettingsScope scope,
                                                                 Promise<Unit> &&promise) {
  if (td_->auth_manager_->is_bot()) {
    LOG(ERROR) << "Can't get notification settings for " << scope;
    return promise.set_error(Status::Error(500, "Wrong getScopeNotificationSettings query"));
  }

  td_->create_handler<GetScopeNotifySettingsQuery>(std::move(promise))->send(scope);
}

void MessagesManager::on_get_dialog_notification_settings_query_finished(DialogId dialog_id, Status &&status) {
  auto it = get_dialog_notification_settings_queries_.find(dialog_id);
  CHECK(it != get_dialog_notification_settings_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  get_dialog_notification_settings_queries_.erase(it);

  for (auto &promise : promises) {
    if (status.is_ok()) {
      promise.set_value(Unit());
    } else {
      promise.set_error(status.clone());
    }
  }
}

class MessagesManager::GetDialogFromServerLogEvent {
 public:
  DialogId dialog_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
  }
};

uint64 MessagesManager::save_get_dialog_from_server_logevent(DialogId dialog_id) {
  GetDialogFromServerLogEvent logevent{dialog_id};
  auto storer = LogEventStorerImpl<GetDialogFromServerLogEvent>(logevent);
  return binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::GetDialogFromServer, storer);
}

void MessagesManager::send_get_dialog_query(DialogId dialog_id, Promise<Unit> &&promise, uint64 logevent_id) {
  if (td_->auth_manager_->is_bot() || dialog_id.get_type() == DialogType::SecretChat) {
    if (logevent_id != 0) {
      binlog_erase(G()->td_db()->get_binlog(), logevent_id);
    }
    return promise.set_error(Status::Error(500, "Wrong getDialog query"));
  }
  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    if (logevent_id != 0) {
      binlog_erase(G()->td_db()->get_binlog(), logevent_id);
    }
    return promise.set_error(Status::Error(400, "Can't access the chat"));
  }

  auto &promises = get_dialog_queries_[dialog_id];
  promises.push_back(std::move(promise));
  if (promises.size() != 1) {
    if (logevent_id != 0) {
      LOG(INFO) << "Duplicate getDialog query for " << dialog_id;
      binlog_erase(G()->td_db()->get_binlog(), logevent_id);
    }
    // query has already been sent, just wait for the result
    return;
  }

  if (logevent_id == 0 && G()->parameters().use_message_db) {
    logevent_id = save_get_dialog_from_server_logevent(dialog_id);
  }
  if (logevent_id != 0) {
    auto result = get_dialog_query_logevent_id_.emplace(dialog_id, logevent_id);
    CHECK(result.second);
  }
  if (G()->close_flag()) {
    // request will be sent after restart
    return;
  }

  LOG(INFO) << "Send get " << dialog_id << " query";
  td_->create_handler<GetDialogQuery>()->send(dialog_id);
}

void MessagesManager::on_get_dialog_query_finished(DialogId dialog_id, Status &&status) {
  LOG(INFO) << "Finished getting " << dialog_id << " with result " << status;
  auto it = get_dialog_queries_.find(dialog_id);
  CHECK(it != get_dialog_queries_.end());
  CHECK(!it->second.empty());
  auto promises = std::move(it->second);
  get_dialog_queries_.erase(it);

  auto logevent_it = get_dialog_query_logevent_id_.find(dialog_id);
  if (logevent_it != get_dialog_query_logevent_id_.end()) {
    if (!G()->close_flag()) {
      binlog_erase(G()->td_db()->get_binlog(), logevent_it->second);
    }
    get_dialog_query_logevent_id_.erase(logevent_it);
  }

  for (auto &promise : promises) {
    if (status.is_ok()) {
      promise.set_value(Unit());
    } else {
      promise.set_error(status.clone());
    }
  }
}

bool MessagesManager::is_update_about_username_change_received(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->is_update_about_username_change_received(dialog_id.get_user_id());
    case DialogType::Chat:
      return true;
    case DialogType::Channel:
      return td_->contacts_manager_->get_channel_status(dialog_id.get_channel_id()).is_member();
    case DialogType::SecretChat:
      return true;
    case DialogType::None:
    default:
      UNREACHABLE();
      return false;
  }
}

void MessagesManager::on_dialog_username_updated(DialogId dialog_id, const string &old_username,
                                                 const string &new_username) {
  auto d = get_dialog(dialog_id);
  if (d != nullptr) {
    update_dialogs_hints(d);
  }
  if (!old_username.empty() && old_username != new_username) {
    resolved_usernames_.erase(clean_username(old_username));
    inaccessible_resolved_usernames_.erase(clean_username(old_username));
  }
  if (!new_username.empty()) {
    auto cache_time = is_update_about_username_change_received(dialog_id) ? USERNAME_CACHE_EXPIRE_TIME
                                                                          : USERNAME_CACHE_EXPIRE_TIME_SHORT;
    resolved_usernames_[clean_username(new_username)] = ResolvedUsername{dialog_id, Time::now() + cache_time};
  }
}

void MessagesManager::on_resolved_username(const string &username, DialogId dialog_id) {
  if (!dialog_id.is_valid()) {
    LOG(ERROR) << "Resolve username \"" << username << "\" to invalid " << dialog_id;
    return;
  }

  auto it = resolved_usernames_.find(clean_username(username));
  if (it != resolved_usernames_.end()) {
    LOG_IF(ERROR, it->second.dialog_id != dialog_id)
        << "Resolve username \"" << username << "\" to " << dialog_id << ", but have it in " << it->second.dialog_id;
    return;
  }

  inaccessible_resolved_usernames_[clean_username(username)] = dialog_id;
}

void MessagesManager::drop_username(const string &username) {
  inaccessible_resolved_usernames_.erase(clean_username(username));

  auto it = resolved_usernames_.find(clean_username(username));
  if (it == resolved_usernames_.end()) {
    return;
  }

  auto dialog_id = it->second.dialog_id;
  if (have_input_peer(dialog_id, AccessRights::Read)) {
    CHECK(dialog_id.get_type() != DialogType::SecretChat);
    send_get_dialog_query(dialog_id, Auto());
  }

  resolved_usernames_.erase(it);
}

const DialogPhoto *MessagesManager::get_dialog_photo(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->get_user_dialog_photo(dialog_id.get_user_id());
    case DialogType::Chat:
      return td_->contacts_manager_->get_chat_dialog_photo(dialog_id.get_chat_id());
    case DialogType::Channel:
      return td_->contacts_manager_->get_channel_dialog_photo(dialog_id.get_channel_id());
    case DialogType::SecretChat:
      return td_->contacts_manager_->get_secret_chat_dialog_photo(dialog_id.get_secret_chat_id());
    case DialogType::None:
    default:
      UNREACHABLE();
      return nullptr;
  }
}

string MessagesManager::get_dialog_title(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->get_user_title(dialog_id.get_user_id());
    case DialogType::Chat:
      return td_->contacts_manager_->get_chat_title(dialog_id.get_chat_id());
    case DialogType::Channel:
      return td_->contacts_manager_->get_channel_title(dialog_id.get_channel_id());
    case DialogType::SecretChat:
      return td_->contacts_manager_->get_secret_chat_title(dialog_id.get_secret_chat_id());
    case DialogType::None:
    default:
      UNREACHABLE();
      return string();
  }
}

string MessagesManager::get_dialog_username(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->get_user_username(dialog_id.get_user_id());
    case DialogType::Chat:
      return string();
    case DialogType::Channel:
      return td_->contacts_manager_->get_channel_username(dialog_id.get_channel_id());
    case DialogType::SecretChat:
      return td_->contacts_manager_->get_secret_chat_username(dialog_id.get_secret_chat_id());
    case DialogType::None:
    default:
      UNREACHABLE();
      return string();
  }
}

RestrictedRights MessagesManager::get_dialog_permissions(DialogId dialog_id) const {
  switch (dialog_id.get_type()) {
    case DialogType::User:
      return td_->contacts_manager_->get_user_default_permissions(dialog_id.get_user_id());
    case DialogType::Chat:
      return td_->contacts_manager_->get_chat_default_permissions(dialog_id.get_chat_id());
    case DialogType::Channel:
      return td_->contacts_manager_->get_channel_default_permissions(dialog_id.get_channel_id());
    case DialogType::SecretChat:
      return td_->contacts_manager_->get_secret_chat_default_permissions(dialog_id.get_secret_chat_id());
    case DialogType::None:
    default:
      UNREACHABLE();
      return RestrictedRights(false, false, false, false, false, false, false, false, false, false, false);
  }
}

void MessagesManager::send_dialog_action(DialogId dialog_id, const tl_object_ptr<td_api::ChatAction> &action,
                                         Promise<Unit> &&promise) {
  if (action == nullptr) {
    return promise.set_error(Status::Error(5, "Action must not be empty"));
  }

  if (!have_dialog_force(dialog_id)) {
    return promise.set_error(Status::Error(5, "Chat not found"));
  }

  auto can_send_status = can_send_message(dialog_id);
  if (can_send_status.is_error()) {
    if (td_->auth_manager_->is_bot()) {
      return promise.set_error(can_send_status.move_as_error());
    }
    return promise.set_value(Unit());
  }

  if (is_broadcast_channel(dialog_id)) {
    return promise.set_value(Unit());
  }

  auto dialog_type = dialog_id.get_type();
  if (dialog_type == DialogType::User || dialog_type == DialogType::SecretChat) {
    UserId user_id = dialog_type == DialogType::User
                         ? dialog_id.get_user_id()
                         : td_->contacts_manager_->get_secret_chat_user_id(dialog_id.get_secret_chat_id());
    if (!user_id.is_valid() || td_->contacts_manager_->is_user_bot(user_id) ||
        td_->contacts_manager_->is_user_deleted(user_id)) {
      return promise.set_value(Unit());
    }
    if (!td_->auth_manager_->is_bot() && !td_->contacts_manager_->is_user_status_exact(user_id)) {
      // return promise.set_value(Unit());
    }
  }

  if (dialog_type == DialogType::SecretChat) {
    tl_object_ptr<secret_api::SendMessageAction> send_action;
    switch (action->get_id()) {
      case td_api::chatActionCancel::ID:
        send_action = make_tl_object<secret_api::sendMessageCancelAction>();
        break;
      case td_api::chatActionTyping::ID:
        send_action = make_tl_object<secret_api::sendMessageTypingAction>();
        break;
      case td_api::chatActionRecordingVideo::ID:
        send_action = make_tl_object<secret_api::sendMessageRecordVideoAction>();
        break;
      case td_api::chatActionUploadingVideo::ID:
        send_action = make_tl_object<secret_api::sendMessageUploadVideoAction>();
        break;
      case td_api::chatActionRecordingVoiceNote::ID:
        send_action = make_tl_object<secret_api::sendMessageRecordAudioAction>();
        break;
      case td_api::chatActionUploadingVoiceNote::ID:
        send_action = make_tl_object<secret_api::sendMessageUploadAudioAction>();
        break;
      case td_api::chatActionUploadingPhoto::ID:
        send_action = make_tl_object<secret_api::sendMessageUploadPhotoAction>();
        break;
      case td_api::chatActionUploadingDocument::ID:
        send_action = make_tl_object<secret_api::sendMessageUploadDocumentAction>();
        break;
      case td_api::chatActionChoosingLocation::ID:
        send_action = make_tl_object<secret_api::sendMessageGeoLocationAction>();
        break;
      case td_api::chatActionChoosingContact::ID:
        send_action = make_tl_object<secret_api::sendMessageChooseContactAction>();
        break;
      case td_api::chatActionRecordingVideoNote::ID:
        send_action = make_tl_object<secret_api::sendMessageRecordRoundAction>();
        break;
      case td_api::chatActionUploadingVideoNote::ID:
        send_action = make_tl_object<secret_api::sendMessageUploadRoundAction>();
        break;
      case td_api::chatActionStartPlayingGame::ID:
        return promise.set_error(Status::Error(5, "Games are unsupported in secret chats"));
      default:
        UNREACHABLE();
    }
    send_closure(G()->secret_chats_manager(), &SecretChatsManager::send_message_action, dialog_id.get_secret_chat_id(),
                 std::move(send_action));
    promise.set_value(Unit());
    return;
  }

  tl_object_ptr<telegram_api::SendMessageAction> send_action;
  switch (action->get_id()) {
    case td_api::chatActionCancel::ID:
      send_action = make_tl_object<telegram_api::sendMessageCancelAction>();
      break;
    case td_api::chatActionTyping::ID:
      send_action = make_tl_object<telegram_api::sendMessageTypingAction>();
      break;
    case td_api::chatActionRecordingVideo::ID:
      send_action = make_tl_object<telegram_api::sendMessageRecordVideoAction>();
      break;
    case td_api::chatActionUploadingVideo::ID: {
      auto progress = static_cast<const td_api::chatActionUploadingVideo &>(*action).progress_;
      send_action = make_tl_object<telegram_api::sendMessageUploadVideoAction>(progress);
      break;
    }
    case td_api::chatActionRecordingVoiceNote::ID:
      send_action = make_tl_object<telegram_api::sendMessageRecordAudioAction>();
      break;
    case td_api::chatActionUploadingVoiceNote::ID: {
      auto progress = static_cast<const td_api::chatActionUploadingVoiceNote &>(*action).progress_;
      send_action = make_tl_object<telegram_api::sendMessageUploadAudioAction>(progress);
      break;
    }
    case td_api::chatActionUploadingPhoto::ID: {
      auto progress = static_cast<const td_api::chatActionUploadingPhoto &>(*action).progress_;
      send_action = make_tl_object<telegram_api::sendMessageUploadPhotoAction>(progress);
      break;
    }
    case td_api::chatActionUploadingDocument::ID: {
      auto progress = static_cast<const td_api::chatActionUploadingDocument &>(*action).progress_;
      send_action = make_tl_object<telegram_api::sendMessageUploadDocumentAction>(progress);
      break;
    }
    case td_api::chatActionChoosingLocation::ID:
      send_action = make_tl_object<telegram_api::sendMessageGeoLocationAction>();
      break;
    case td_api::chatActionChoosingContact::ID:
      send_action = make_tl_object<telegram_api::sendMessageChooseContactAction>();
      break;
    case td_api::chatActionStartPlayingGame::ID:
      send_action = make_tl_object<telegram_api::sendMessageGamePlayAction>();
      break;
    case td_api::chatActionRecordingVideoNote::ID:
      send_action = make_tl_object<telegram_api::sendMessageRecordRoundAction>();
      break;
    case td_api::chatActionUploadingVideoNote::ID: {
      auto progress = static_cast<const td_api::chatActionUploadingVideoNote &>(*action).progress_;
      send_action = make_tl_object<telegram_api::sendMessageUploadRoundAction>(progress);
      break;
    }
    default:
      UNREACHABLE();
  }

  auto &query_ref = set_typing_query_[dialog_id];
  if (!query_ref.empty() && !td_->auth_manager_->is_bot()) {
    LOG(INFO) << "Cancel previous set typing query";
    cancel_query(query_ref);
  }
  query_ref = td_->create_handler<SetTypingQuery>(std::move(promise))->send(dialog_id, std::move(send_action));
}

void MessagesManager::on_send_dialog_action_timeout(DialogId dialog_id) {
  LOG(INFO) << "Receive send_dialog_action timeout in " << dialog_id;
  Dialog *d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (can_send_message(dialog_id).is_error()) {
    return;
  }

  auto queue_id = get_sequence_dispatcher_id(dialog_id, MessageContentType::Photo);
  CHECK(queue_id & 1);

  auto queue_it = yet_unsent_media_queues_.find(queue_id);
  if (queue_it == yet_unsent_media_queues_.end()) {
    return;
  }

  pending_send_dialog_action_timeout_.add_timeout_in(dialog_id.get(), 4.0);

  CHECK(!queue_it->second.empty());
  MessageId message_id(queue_it->second.begin()->first);
  const Message *m = get_message(d, message_id);
  if (m == nullptr) {
    return;
  }
  if (m->forward_info != nullptr || m->had_forward_info || message_id.is_scheduled()) {
    return;
  }

  auto file_id = get_message_content_upload_file_id(m->content.get());
  if (!file_id.is_valid()) {
    LOG(ERROR) << "Have no file in "
               << to_string(get_message_content_object(m->content.get(), td_, m->date, m->is_content_secret));
    return;
  }
  auto file_view = td_->file_manager_->get_file_view(file_id);
  if (!file_view.is_uploading()) {
    return;
  }
  int64 total_size = file_view.expected_size();
  int64 uploaded_size = file_view.remote_size();
  int32 progress = 0;
  if (total_size > 0 && uploaded_size > 0) {
    if (uploaded_size > total_size) {
      uploaded_size = total_size;  // just in case
    }
    progress = static_cast<int32>(100 * uploaded_size / total_size);
  }

  td_api::object_ptr<td_api::ChatAction> action;
  switch (m->content->get_type()) {
    case MessageContentType::Animation:
    case MessageContentType::Audio:
    case MessageContentType::Document:
      action = td_api::make_object<td_api::chatActionUploadingDocument>(progress);
      break;
    case MessageContentType::Photo:
      action = td_api::make_object<td_api::chatActionUploadingPhoto>(progress);
      break;
    case MessageContentType::Video:
      action = td_api::make_object<td_api::chatActionUploadingVideo>(progress);
      break;
    case MessageContentType::VideoNote:
      action = td_api::make_object<td_api::chatActionUploadingVideoNote>(progress);
      break;
    case MessageContentType::VoiceNote:
      action = td_api::make_object<td_api::chatActionUploadingVoiceNote>(progress);
      break;
    default:
      return;
  }
  CHECK(action != nullptr);
  LOG(INFO) << "Send action in " << dialog_id << ": " << to_string(action);
  send_dialog_action(dialog_id, std::move(action), Auto());
}

void MessagesManager::on_active_dialog_action_timeout(DialogId dialog_id) {
  LOG(DEBUG) << "Receive active dialog action timeout in " << dialog_id;
  auto actions_it = active_dialog_actions_.find(dialog_id);
  if (actions_it == active_dialog_actions_.end()) {
    return;
  }
  CHECK(!actions_it->second.empty());

  auto now = Time::now();
  while (actions_it->second[0].start_time + DIALOG_ACTION_TIMEOUT < now + 0.1) {
    on_user_dialog_action(dialog_id, actions_it->second[0].user_id, nullptr, 0);

    actions_it = active_dialog_actions_.find(dialog_id);
    if (actions_it == active_dialog_actions_.end()) {
      return;
    }
    CHECK(!actions_it->second.empty());
  }

  LOG(DEBUG) << "Schedule next action timeout in " << dialog_id;
  active_dialog_action_timeout_.add_timeout_in(dialog_id.get(),
                                               actions_it->second[0].start_time + DIALOG_ACTION_TIMEOUT - now);
}

void MessagesManager::clear_active_dialog_actions(DialogId dialog_id) {
  LOG(DEBUG) << "Clear active dialog actions in " << dialog_id;
  auto actions_it = active_dialog_actions_.find(dialog_id);
  while (actions_it != active_dialog_actions_.end()) {
    CHECK(!actions_it->second.empty());
    on_user_dialog_action(dialog_id, actions_it->second[0].user_id, nullptr, 0);
    actions_it = active_dialog_actions_.find(dialog_id);
  }
}

tl_object_ptr<telegram_api::MessagesFilter> MessagesManager::get_input_messages_filter(SearchMessagesFilter filter) {
  switch (filter) {
    case SearchMessagesFilter::Empty:
      return make_tl_object<telegram_api::inputMessagesFilterEmpty>();
    case SearchMessagesFilter::Animation:
      return make_tl_object<telegram_api::inputMessagesFilterGif>();
    case SearchMessagesFilter::Audio:
      return make_tl_object<telegram_api::inputMessagesFilterMusic>();
    case SearchMessagesFilter::Document:
      return make_tl_object<telegram_api::inputMessagesFilterDocument>();
    case SearchMessagesFilter::Photo:
      return make_tl_object<telegram_api::inputMessagesFilterPhotos>();
    case SearchMessagesFilter::Video:
      return make_tl_object<telegram_api::inputMessagesFilterVideo>();
    case SearchMessagesFilter::VoiceNote:
      return make_tl_object<telegram_api::inputMessagesFilterVoice>();
    case SearchMessagesFilter::PhotoAndVideo:
      return make_tl_object<telegram_api::inputMessagesFilterPhotoVideo>();
    case SearchMessagesFilter::Url:
      return make_tl_object<telegram_api::inputMessagesFilterUrl>();
    case SearchMessagesFilter::ChatPhoto:
      return make_tl_object<telegram_api::inputMessagesFilterChatPhotos>();
    case SearchMessagesFilter::Call:
      return make_tl_object<telegram_api::inputMessagesFilterPhoneCalls>(0, false /*ignored*/);
    case SearchMessagesFilter::MissedCall:
      return make_tl_object<telegram_api::inputMessagesFilterPhoneCalls>(
          telegram_api::inputMessagesFilterPhoneCalls::MISSED_MASK, false /*ignored*/);
    case SearchMessagesFilter::VideoNote:
      return make_tl_object<telegram_api::inputMessagesFilterRoundVideo>();
    case SearchMessagesFilter::VoiceAndVideoNote:
      return make_tl_object<telegram_api::inputMessagesFilterRoundVoice>();
    case SearchMessagesFilter::Mention:
      return make_tl_object<telegram_api::inputMessagesFilterMyMentions>();
    default:
      UNREACHABLE();
      return nullptr;
  }
}

SearchMessagesFilter MessagesManager::get_search_messages_filter(
    const tl_object_ptr<td_api::SearchMessagesFilter> &filter) {
  if (filter == nullptr) {
    return SearchMessagesFilter::Empty;
  }
  switch (filter->get_id()) {
    case td_api::searchMessagesFilterEmpty::ID:
      return SearchMessagesFilter::Empty;
    case td_api::searchMessagesFilterAnimation::ID:
      return SearchMessagesFilter::Animation;
    case td_api::searchMessagesFilterAudio::ID:
      return SearchMessagesFilter::Audio;
    case td_api::searchMessagesFilterDocument::ID:
      return SearchMessagesFilter::Document;
    case td_api::searchMessagesFilterPhoto::ID:
      return SearchMessagesFilter::Photo;
    case td_api::searchMessagesFilterVideo::ID:
      return SearchMessagesFilter::Video;
    case td_api::searchMessagesFilterVoiceNote::ID:
      return SearchMessagesFilter::VoiceNote;
    case td_api::searchMessagesFilterPhotoAndVideo::ID:
      return SearchMessagesFilter::PhotoAndVideo;
    case td_api::searchMessagesFilterUrl::ID:
      return SearchMessagesFilter::Url;
    case td_api::searchMessagesFilterChatPhoto::ID:
      return SearchMessagesFilter::ChatPhoto;
    case td_api::searchMessagesFilterCall::ID:
      return SearchMessagesFilter::Call;
    case td_api::searchMessagesFilterMissedCall::ID:
      return SearchMessagesFilter::MissedCall;
    case td_api::searchMessagesFilterVideoNote::ID:
      return SearchMessagesFilter::VideoNote;
    case td_api::searchMessagesFilterVoiceAndVideoNote::ID:
      return SearchMessagesFilter::VoiceAndVideoNote;
    case td_api::searchMessagesFilterMention::ID:
      return SearchMessagesFilter::Mention;
    case td_api::searchMessagesFilterUnreadMention::ID:
      return SearchMessagesFilter::UnreadMention;
    default:
      UNREACHABLE();
      return SearchMessagesFilter::Empty;
  }
}

void MessagesManager::set_dialog_folder_id(DialogId dialog_id, FolderId folder_id, Promise<Unit> &&promise) {
  LOG(INFO) << "Receive setChatChatList request to change folder of " << dialog_id << " to " << folder_id;

  Dialog *d = get_dialog_force(dialog_id);
  if (d == nullptr) {
    return promise.set_error(Status::Error(3, "Chat not found"));
  }

  if (d->order == DEFAULT_ORDER) {
    return promise.set_error(Status::Error(400, "Chat is not in a chat list"));
  }

  if (d->folder_id == folder_id) {
    return promise.set_value(Unit());
  }

  if (!have_input_peer(dialog_id, AccessRights::Read)) {
    return promise.set_error(Status::Error(6, "Can't access the chat"));
  }

  set_dialog_folder_id(d, folder_id);

  if (dialog_id.get_type() != DialogType::SecretChat) {
    set_dialog_folder_id_on_server(dialog_id, false);
  }
  promise.set_value(Unit());
}

class MessagesManager::SetDialogFolderIdOnServerLogEvent {
 public:
  DialogId dialog_id_;
  FolderId folder_id_;

  template <class StorerT>
  void store(StorerT &storer) const {
    td::store(dialog_id_, storer);
    td::store(folder_id_, storer);
  }

  template <class ParserT>
  void parse(ParserT &parser) {
    td::parse(dialog_id_, parser);
    td::parse(folder_id_, parser);
  }
};

void MessagesManager::set_dialog_folder_id_on_server(DialogId dialog_id, bool from_binlog) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);

  if (!from_binlog && G()->parameters().use_message_db) {
    SetDialogFolderIdOnServerLogEvent logevent;
    logevent.dialog_id_ = dialog_id;
    logevent.folder_id_ = d->folder_id;
    auto storer = LogEventStorerImpl<SetDialogFolderIdOnServerLogEvent>(logevent);
    if (d->set_folder_id_logevent_id == 0) {
      d->set_folder_id_logevent_id =
          binlog_add(G()->td_db()->get_binlog(), LogEvent::HandlerType::SetDialogFolderIdOnServer, storer);
    } else {
      binlog_rewrite(G()->td_db()->get_binlog(), d->set_folder_id_logevent_id,
                     LogEvent::HandlerType::SetDialogFolderIdOnServer, storer);
    }
    d->set_folder_id_logevent_id_generation++;
  }

  Promise<> promise;
  if (d->set_folder_id_logevent_id != 0) {
    promise = PromiseCreator::lambda([actor_id = actor_id(this), dialog_id,
                                      generation = d->set_folder_id_logevent_id_generation](Result<Unit> result) {
      if (!G()->close_flag()) {
        send_closure(actor_id, &MessagesManager::on_updated_dialog_folder_id, dialog_id, generation);
      }
    });
  }

  // TODO do not send two queries simultaneously or use SequenceDispatcher
  td_->create_handler<EditPeerFoldersQuery>(std::move(promise))->send(dialog_id, d->folder_id);
}

void MessagesManager::on_updated_dialog_folder_id(DialogId dialog_id, uint64 generation) {
  auto d = get_dialog(dialog_id);
  CHECK(d != nullptr);
  LOG(INFO) << "Saved folder_id of " << dialog_id << " with logevent " << d->set_folder_id_logevent_id;
  if (d->set_folder_id_logevent_id_generation ==