/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kafka.log

import java.nio._
import java.util.concurrent.atomic._
import junit.framework.Assert._
import kafka.utils.TestUtils._
import kafka.message._
import org.junit.Test

class FileMessageSetTest extends BaseMessageSetTestCases {
  
  val messageSet = createMessageSet(messages)
  
  def createMessageSet(messages: Seq[Message]): FileMessageSet = {
    val set = new FileMessageSet(tempFile())
    set.append(new ByteBufferMessageSet(NoCompressionCodec, messages: _*))
    set.flush()
    set
  }

  /**
   * Test that the cached size variable matches the actual file size as we append messages
   */
  @Test
  def testFileSize() {
    assertEquals(messageSet.channel.size, messageSet.sizeInBytes)
    for(i <- 0 until 20) {
      messageSet.append(singleMessageSet("abcd".getBytes))
      assertEquals(messageSet.channel.size, messageSet.sizeInBytes)
    } 
  }
  
  /**
   * Test that adding invalid bytes to the end of the log doesn't break iteration
   */
  @Test
  def testIterationOverPartialAndTruncation() {
    testPartialWrite(0, messageSet)
    testPartialWrite(2, messageSet)
    testPartialWrite(4, messageSet)
    testPartialWrite(5, messageSet)
    testPartialWrite(6, messageSet)
  }
  
  def testPartialWrite(size: Int, messageSet: FileMessageSet) {
    val buffer = ByteBuffer.allocate(size)
    val originalPosition = messageSet.channel.position
    for(i <- 0 until size)
      buffer.put(0.asInstanceOf[Byte])
    buffer.rewind()
    messageSet.channel.write(buffer)
    // appending those bytes should not change the contents
    checkEquals(messages.iterator, messageSet.map(m => m.message).iterator)
  }
  
  /**
   * Iterating over the file does file reads but shouldn't change the position of the underlying FileChannel.
   */
  @Test
  def testIterationDoesntChangePosition() {
    val position = messageSet.channel.position
    checkEquals(messages.iterator, messageSet.map(m => m.message).iterator)
    assertEquals(position, messageSet.channel.position)
  }
  
  /**
   * Test a simple append and read.
   */
  @Test
  def testRead() {
    var read = messageSet.read(0, messageSet.sizeInBytes)
    checkEquals(messageSet.iterator, read.iterator)
    val items = read.iterator.toList
    val sec = items.tail.head
    read = messageSet.read(position = MessageSet.entrySize(sec.message), size = messageSet.sizeInBytes)
    assertEquals("Try a read starting from the second message", items.tail, read.toList)
    read = messageSet.read(MessageSet.entrySize(sec.message), MessageSet.entrySize(sec.message))
    assertEquals("Try a read of a single message starting from the second message", List(items.tail.head), read.toList)
  }
  
  /**
   * Test the MessageSet.searchFor API.
   */
  @Test
  def testSearch() {
    // append a new message with a high offset
    val lastMessage = new Message("test".getBytes)
    messageSet.append(new ByteBufferMessageSet(NoCompressionCodec, new AtomicLong(50), lastMessage))
    var position = 0
    assertEquals("Should be able to find the first message by its offset", 
                 OffsetPosition(0L, position), 
                 messageSet.searchFor(0, 0))
    position += MessageSet.entrySize(messageSet.head.message)
    assertEquals("Should be able to find second message when starting from 0", 
                 OffsetPosition(1L, position), 
                 messageSet.searchFor(1, 0))
    assertEquals("Should be able to find second message starting from its offset", 
                 OffsetPosition(1L, position), 
                 messageSet.searchFor(1, position))
    position += MessageSet.entrySize(messageSet.tail.head.message) + MessageSet.entrySize(messageSet.tail.tail.head.message)
    assertEquals("Should be able to find fourth message from a non-existant offset", 
                 OffsetPosition(50L, position), 
                 messageSet.searchFor(3, position))
    assertEquals("Should be able to find fourth message by correct offset", 
                 OffsetPosition(50L, position), 
                 messageSet.searchFor(50,  position))
  }
  
  /**
   * Test that the message set iterator obeys start and end slicing
   */
  @Test
  def testIteratorWithLimits() {
    val message = messageSet.toList(1)
    val start = messageSet.searchFor(1, 0).position
    val size = message.message.size
    val slice = messageSet.read(start, size)
    assertEquals(List(message), slice.toList)
  }
  
  /**
   * Test the truncateTo method lops off messages and appropriately updates the size
   */
  @Test
  def testTruncate() {
    val message = messageSet.toList(0)
    val end = messageSet.searchFor(1, 0).position
    messageSet.truncateTo(end)
    assertEquals(List(message), messageSet.toList)
    assertEquals(MessageSet.entrySize(message.message), messageSet.sizeInBytes)
  }
  
}
