/* Copyright 2015 The TensorFlow Authors. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
==============================================================================*/

// TODO(jeff,sanjay): Rename to tensorflow/public/c_api.h
#ifndef TENSORFLOW_PUBLIC_TENSOR_C_API_H_
#define TENSORFLOW_PUBLIC_TENSOR_C_API_H_

#include <stddef.h>
#include <stdint.h>

// --------------------------------------------------------------------------
// C API for TensorFlow.
//
// The API leans towards simplicity and uniformity instead of convenience
// since most usage will be by language specific wrappers.
//
// Conventions:
// * We use the prefix TF_ for everything in the API.
// * Objects are always passed around as pointers to opaque structs
//   and these structs are allocated/deallocated via the API.
// * TF_Status holds error information.  It is an object type
//   and therefore is passed around as a pointer to an opaque
//   struct as mentioned above.
// * Every call that has a TF_Status* argument clears it on success
//   and fills it with error info on failure.
//
// Questions left to address:
// * Might at some point need a way for callers to provide their own Env.
// * Maybe add TF_TensorShape that encapsulates dimension info.
//
// Design decisions made:
// * Backing store for tensor memory has an associated deallocation
//   function.  This deallocation function will point to client code
//   for tensors populated by the client.  So the client can do things
//   like shadowing a numpy array.
// * We do not provide TF_OK since it is not strictly necessary and we
//   are not optimizing for convenience.
// * We make assumption that one session has one graph.  This should be
//   fine since we have the ability to run sub-graphs.
// * We could allow NULL for some arguments (e.g., NULL options arg).
//   However since convenience is not a primary goal, we don't do this.
// * Devices are not in this API.  Instead, they are created/used internally
//   and the API just provides high level controls over the number of
//   devices of each type.

#ifdef __cplusplus
extern "C" {
#endif

// --------------------------------------------------------------------------
// TF_DataType holds the type for a scalar value.  E.g., one slot in a tensor.
// The enum values here are identical to corresponding values in types.proto.
typedef enum {
  TF_FLOAT = 1,
  TF_DOUBLE = 2,
  TF_INT32 = 3,  // Int32 tensors are always in 'host' memory.
  TF_UINT8 = 4,
  TF_INT16 = 5,
  TF_INT8 = 6,
  TF_STRING = 7,
  TF_COMPLEX64 = 8,  // Single-precision complex
  TF_COMPLEX = 8,    // Old identifier kept for API backwards compatibility
  TF_INT64 = 9,
  TF_BOOL = 10,
  TF_QINT8 = 11,     // Quantized int8
  TF_QUINT8 = 12,    // Quantized uint8
  TF_QINT32 = 13,    // Quantized int32
  TF_BFLOAT16 = 14,  // Float32 truncated to 16 bits.  Only for cast ops.
  TF_QINT16 = 15,    // Quantized int16
  TF_QUINT16 = 16,   // Quantized uint16
  TF_UINT16 = 17,
  TF_COMPLEX128 = 18,  // Double-precision complex
  TF_HALF = 19,
} TF_DataType;

// --------------------------------------------------------------------------
// TF_Code holds an error code.  The enum values here are identical to
// corresponding values in error_codes.proto.
typedef enum {
  TF_OK = 0,
  TF_CANCELLED = 1,
  TF_UNKNOWN = 2,
  TF_INVALID_ARGUMENT = 3,
  TF_DEADLINE_EXCEEDED = 4,
  TF_NOT_FOUND = 5,
  TF_ALREADY_EXISTS = 6,
  TF_PERMISSION_DENIED = 7,
  TF_UNAUTHENTICATED = 16,
  TF_RESOURCE_EXHAUSTED = 8,
  TF_FAILED_PRECONDITION = 9,
  TF_ABORTED = 10,
  TF_OUT_OF_RANGE = 11,
  TF_UNIMPLEMENTED = 12,
  TF_INTERNAL = 13,
  TF_UNAVAILABLE = 14,
  TF_DATA_LOSS = 15,
} TF_Code;

// --------------------------------------------------------------------------
// TF_Status holds error information.  It either has an OK code, or
// else an error code with an associated error message.
typedef struct TF_Status TF_Status;

// Return a new status object.
extern TF_Status* TF_NewStatus();

// Delete a previously created status object.
extern void TF_DeleteStatus(TF_Status*);

// Record <code, msg> in *s.  Any previous information is lost.
// A common use is to clear a status: TF_SetStatus(s, TF_OK, "");
extern void TF_SetStatus(TF_Status* s, TF_Code code, const char* msg);

// Return the code record in *s.
extern TF_Code TF_GetCode(const TF_Status* s);

// Return a pointer to the (null-terminated) error message in *s.  The
// return value points to memory that is only usable until the next
// mutation to *s.  Always returns an empty string if TF_GetCode(s) is
// TF_OK.
extern const char* TF_Message(const TF_Status* s);

// --------------------------------------------------------------------------
// TF_Buffer holds a pointer to a block of data and its associated length.
// Typically, the data consists of a serialized protocol buffer, but other data
// may also be held in a buffer.
//
// By default, TF_Buffer itself does not do any memory management of the
// pointed-to block.  If need be, users of this struct should specify how to
// deallocate the block by setting the `data_deallocator` function pointer.
typedef struct {
  const void* data;
  size_t length;
  void (*data_deallocator)(void* data, size_t length);
} TF_Buffer;

// Makes a copy of the input and sets an appropriate deallocator.  Useful for
// passing in read-only, input protobufs.
extern TF_Buffer* TF_NewBufferFromString(const void* proto, size_t proto_len);

// Useful for passing *out* a protobuf.
extern TF_Buffer* TF_NewBuffer();

extern void TF_DeleteBuffer(TF_Buffer*);

extern TF_Buffer TF_GetBuffer(TF_Buffer* buffer);

// --------------------------------------------------------------------------
// TF_Tensor holds a multi-dimensional array of elements of a single data type.
// For all types other than TF_STRING, the data buffer stores elements
// in row major order.  E.g. if data is treated as a vector of TF_DataType:
//
//   element 0:   index (0, ..., 0)
//   element 1:   index (0, ..., 1)
//   ...
//
// The format for TF_STRING tensors is:
//   start_offset: array[uint64]
//   data:         byte[...]
//
//   String length is encoded (varint?) starting at data[start_offset[i]]
//   String contents follow immediately after string length.

typedef struct TF_Tensor TF_Tensor;

// Return a new tensor that holds the bytes data[0,len-1].
//
// The data will be deallocated by a subsequent call to TF_DeleteTensor via:
//      (*deallocator)(data, len, deallocator_arg)
// Clients must provide a custom deallocator function so they can pass in
// memory managed by something like numpy.
extern TF_Tensor* TF_NewTensor(TF_DataType, const int64_t* dims, int num_dims,
                               void* data, size_t len,
                               void (*deallocator)(void* data, size_t len,
                                                   void* arg),
                               void* deallocator_arg);

// Destroy a tensor.
extern void TF_DeleteTensor(TF_Tensor*);

// Return the type of a tensor element.
extern TF_DataType TF_TensorType(const TF_Tensor*);

// Return the number of dimensions that the tensor has.
extern int TF_NumDims(const TF_Tensor*);

// Return the length of the tensor in the "dim_index" dimension.
// REQUIRES: 0 <= dim_index < TF_NumDims(tensor)
extern int64_t TF_Dim(const TF_Tensor* tensor, int dim_index);

// Return the size of the underlying data in bytes.
extern size_t TF_TensorByteSize(const TF_Tensor*);

// Return a pointer to the underlying data buffer.
extern void* TF_TensorData(const TF_Tensor*);

// --------------------------------------------------------------------------
// TF_SessionOptions holds options that can be passed during session creation.
typedef struct TF_SessionOptions TF_SessionOptions;

// Return a new options object.
extern TF_SessionOptions* TF_NewSessionOptions();

// Set the target in TF_SessionOptions.options.
// target can be empty, a single entry, or a comma separated list of entries.
// Each entry is in one of the following formats :
// "local"
// ip:port
// host:port
extern void TF_SetTarget(TF_SessionOptions* options, const char* target);

// Set the config in TF_SessionOptions.options.
// config should be a serialized tensorflow.ConfigProto proto.
// If config was not parsed successfully as a ConfigProto, record the
// error information in *status.
extern void TF_SetConfig(TF_SessionOptions* options, const void* proto,
                         size_t proto_len, TF_Status* status);

// Destroy an options object.
extern void TF_DeleteSessionOptions(TF_SessionOptions*);

// TODO(jeff,sanjay):
// - export functions to set Config fields

// --------------------------------------------------------------------------
// The new graph construction API, still under development.

// Represents a computation graph.  Graphs may be shared between sessions.
// Graphs are thread-safe when used as directed below.
typedef struct TF_Graph TF_Graph;

// Return a new graph object.
extern TF_Graph* TF_NewGraph();

// Destroy an options object.  Graph will be deleted once no more
// TFSessionWithGraph's are referencing it.
extern void TF_DeleteGraph(TF_Graph*);

// Node being built. The underlying graph must outlive this.
typedef struct TF_NodeDescription TF_NodeDescription;

// Node that has been added to the graph. Valid until the graph is
// deleted -- in particular adding a new node to the graph does not
// invalidate old TF_Node* pointers.
typedef struct TF_Node TF_Node;

// Represents a specific input or output of a node, e.g. to specify the
// specific output to pass as an input to an op.
typedef struct TF_Port {
  TF_Node* node;
  int index;  // Specifies the index of the input or output within node.
} TF_Port;

// Node will only be added to *graph when TF_FinishNode() is called
// (assuming TF_FinishNode() does not return an error).  *graph must
// not be deleted until after TF_FinishNode() is called.
extern TF_NodeDescription* TF_NewNode(TF_Graph* graph, const char* op_type,
                                      const char* node_name);

// Specify the device for `desc`.  Defaults to empty, meaning unconstrained.
extern void TF_SetDevice(TF_NodeDescription* desc, const char* device);

// The calls to TF_AddInput and TF_AddInputList must match (in number,
// order, and type) the op declaration.  For example, the "Concat" op
// has registration:
//   REGISTER_OP("Concat")
//       .Input("concat_dim: int32")
//       .Input("values: N * T")
//       .Output("output: T")
//       .Attr("N: int >= 2")
//       .Attr("T: type");
// that defines two inputs, "concat_dim" and "values" (in that order).
// You must use TF_AddInput() for the first input (since it takes a
// single tensor), and TF_AddInputList() for the second input (since
// it takes a list, even if you were to pass a list with a single
// tensor), as in:
//   TF_NodeDescription* desc = TF_NewNode(graph, "Concat", "c");
//   TF_Port concat_dim_input = {...};
//   TF_AddInput(desc, concat_dim_input);
//   TF_Port values_inputs[5] = {{...}, ..., {...}};
//   TF_AddInputList(desc, 5, values_inputs);

// For inputs that take a single tensor.
extern void TF_AddInput(TF_NodeDescription* desc, TF_Port input);

// For inputs that take a list of tensors.
// inputs must point to TF_Port[num_inputs].
extern void TF_AddInputList(TF_NodeDescription* desc, const TF_Port* inputs,
                            int num_inputs);

// Call once per control input to `desc`.
extern void TF_AddControlInput(TF_NodeDescription* desc, TF_Node* input);

// Call some TF_SetAttr*() function for every attr that is not
// inferred from an input and doesn't have a default value you wish to
// keep.

// `value` must point to a string of length `length` bytes.
extern void TF_SetAttrString(TF_NodeDescription* desc, const char* attr_name,
                             const void* value, int length);
// `values` and `lengths` both must have lengths `num_values`.
// `values[i]` must point to a string of length `lengths[i]` bytes.
extern void TF_SetAttrStringList(TF_NodeDescription* desc,
                                 const char* attr_name,
                                 const void* const* values, const int* lengths,
                                 int num_values);
extern void TF_SetAttrInt(TF_NodeDescription* desc, const char* attr_name,
                          int64_t value);
extern void TF_SetAttrIntList(TF_NodeDescription* desc, const char* attr_name,
                              const int64_t* values, int num_values);
extern void TF_SetAttrFloat(TF_NodeDescription* desc, const char* attr_name,
                            float value);
extern void TF_SetAttrFloatList(TF_NodeDescription* desc, const char* attr_name,
                                const float* values, int num_values);
extern void TF_SetAttrBool(TF_NodeDescription* desc, const char* attr_name,
                           unsigned char value);
extern void TF_SetAttrBoolList(TF_NodeDescription* desc, const char* attr_name,
                               const unsigned char* values, int num_values);
extern void TF_SetAttrType(TF_NodeDescription* desc, const char* attr_name,
                           TF_DataType value);
extern void TF_SetAttrTypeList(TF_NodeDescription* desc, const char* attr_name,
                               const TF_DataType* values, int num_values);

// Set `num_dims` to -1 to represent "unknown rank".  Otherwise,
// `dims` points to an array of length `num_dims`.  `dims[i]` must be
// >= -1, with -1 meaning "unknown dimension".
extern void TF_SetAttrShape(TF_NodeDescription* desc, const char* attr_name,
                            const int64_t* dims, int num_dims);
// `dims` and `num_dims` must point to arrays of length `num_shapes`.
// Set `num_dims[i]` to -1 to represent "unknown rank".  Otherwise,
// `dims[i]` points to an array of length `num_dims[i]`.  `dims[i][j]`
// must be >= -1, with -1 meaning "unknown dimension".
extern void TF_SetAttrShapeList(TF_NodeDescription* desc, const char* attr_name,
                                const int64_t* const* dims, const int* num_dims,
                                int num_shapes);
// `proto` must point to an array of `proto_len` bytes representing a
// binary-serialized TensorShapeProto.
extern void TF_SetAttrTensorShapeProto(TF_NodeDescription* desc,
                                       const char* attr_name, void* proto,
                                       int proto_len, TF_Status* status);
// `protos` and `proto_lens` must point to arrays of length `num_shapes`.
// `protos[i]` must point to an array of `proto_lens[i]` bytes
// representing a binary-serialized TensorShapeProto.
extern void TF_SetAttrTensorShapeProtoList(TF_NodeDescription* desc,
                                           const char* attr_name,
                                           const void* const* protos,
                                           const int* proto_lens,
                                           int num_shapes, TF_Status* status);

// This functions takes ownership of *value (the
// implementation will eventually call TF_DeleteTensor).
extern void TF_SetAttrTensor(TF_NodeDescription* desc, const char* attr_name,
                             TF_Tensor* value, TF_Status* status);
// This functions takes ownership of values[0]..values[num_values-1] (the
// implementation will eventually call TF_DeleteTensor on each).
extern void TF_SetAttrTensorList(TF_NodeDescription* desc,
                                 const char* attr_name,
                                 TF_Tensor* const* values, int num_values,
                                 TF_Status* status);

// `proto` should point to a sequence of bytes of length `proto_len`
// representing a binary serialization of an AttrValue protocol
// buffer.
extern void TF_SetAttrToAttrValueProto(TF_NodeDescription* desc,
                                       const char* attr_name, const void* proto,
                                       size_t proto_len, TF_Status* status);

// If this function succeeds:
//   * *status is set to an OK value,
//   * a TF_Node is added to the graph,
//   * a non-null value pointing to the added node is returned --
//     this value is valid until the underlying graph is deleted.
// Otherwise:
//   * *status is set to a non-OK value,
//   * the graph is not modified,
//   * a null value is returned.
// In either case, it deletes `desc`.
extern TF_Node* TF_FinishNode(TF_NodeDescription* desc, TF_Status* status);

// TF_Node functions.  Nodes are immutable once created, so these are all
// query functions.

extern const char* TF_NodeName(TF_Node* node);
extern const char* TF_NodeOpType(TF_Node* node);
extern const char* TF_NodeDevice(TF_Node* node);

extern int TF_NodeNumOutputs(TF_Node* node);
extern TF_DataType TF_NodeOutputType(TF_Port node_out);
extern int TF_NodeOutputListLength(TF_Node* node, const char* arg_name,
                                   TF_Status* status);

extern int TF_NodeNumInputs(TF_Node* node);
extern TF_DataType TF_NodeInputType(TF_Port node_in);
extern int TF_NodeInputListLength(TF_Node* node, const char* arg_name,
                                  TF_Status* status);

// In this code:
//   TF_Port producer = TF_NodeInput(consumer);
// There is an edge from producer.node's output (given by
// producer.index) to consumer.node's input (given by consumer.index).
extern TF_Port TF_NodeInput(TF_Port node_in);

// Get the number of current consumers of a node's output.  Note that
// this number can change when new nodes are added to the graph.
extern int TF_NodeOutputNumConsumers(TF_Port node_out);

// Get list of all current consumers of a node's output.  consumers
// must point to an array of length at least max_consumers (ideally
// set to TF_NodeOutputNumConsumer(node_out)).  Beware that a
// concurrent modification of the graph can increase the number of
// consumers of a node.  Returns the number of output consumers
// (should match TF_NodeOutputNumConsumers(node_out)).
extern int TF_NodeOutputConsumers(TF_Port node_out, TF_Port* consumers,
                                  int max_consumers);

// Get the number of control inputs to a node.
extern int TF_NodeNumControlInputs(TF_Node* node);

// Get list of all control inputs to a node.  control_inputs must
// point to an array of length max_control_inputs (ideally set to
// TF_NodeNumControlInputs(node)).  Returns the number of control
// inputs (should match TF_NodeNumControlInputs(node)).
extern int TF_NodeGetControlInputs(TF_Node* node, TF_Node** control_inputs,
                                   int max_control_inputs);

// Get the number of nodes that have *node as a control inputs.
// Note that this number can change when new nodes are added to the
// graph.
extern int TF_NodeNumControlOutputs(TF_Node* node);

// Get the list of nodes that have *node as a control input.
// control_outputs must point to an array of length at least
// max_control_outputs (ideally set to
// TF_NodeNumControlOutputs(node)). Beware that a concurrent
// modification of the graph can increase the number of control
// outputs.  Returns the number of control outputs (should match
// TF_NodeNumControlOutputs(node)).
extern int TF_NodeGetControlOutputs(TF_Node* node, TF_Node** control_outputs,
                                    int max_control_outputs);

// Sets `output_attr_value` to the binary-serialized AttrValue proto
// representation of the value of the `attr_name` attr of `node`.
extern void TF_NodeGetAttrValueProto(TF_Node* node, const char* attr_name,
                                     TF_Buffer* output_attr_value,
                                     TF_Status* status);

// Returns the node in the graph with `node_name`. Returns nullptr if
// no node found.
extern TF_Node* TF_GraphNodeByName(TF_Graph* graph, const char* node_name);

// Iterate through the nodes of a graph.  To use:
// size_t pos = 0;
// TF_Node* node;
// while ((node = TF_GraphNextNode(graph, &pos)) != nullptr) {
//   DoSomethingWithNode(node);
// }
extern TF_Node* TF_GraphNextNode(TF_Graph* graph, size_t* pos);

// Note: The following two functions may fail on very large protos in the
// future.

extern void TF_GraphToGraphDef(TF_Graph* graph, TF_Buffer* output_graph_def,
                               TF_Status* status);

extern void TF_NodeToNodeDef(TF_Node* node, TF_Buffer* output_node_def,
                             TF_Status* status);

// TODO(josh11b): Query attrs for a Node.

// TODO(cwhipkey): Query shape for node outputs.

// TODO(josh11b,mrry): Import GraphDef into TF_Graph.

// TODO(andydavis): Function to add gradients to a graph.

// TODO(josh11b): Register OpDef, available to all nodes added
// to this graph.

// The following two may both benefit from a subgraph-definition API
// that re-uses most of the graph-definition API.
// TODO(andydavis): Add functions to a graph.
// TODO(yuanbyu): Add while loop to graph.

// --------------------------------------------------------------------------
// The new session API that uses TF_Graph*.  The intent is this will
// replace the TF_ExtendGraph() API.

// TODO(josh11b): Rename this TF_Session once we delete the old API.
typedef struct TF_SessionWithGraph TF_SessionWithGraph;

// Return a new execution session with the associated graph, or NULL
// on error.  *graph must be a valid graph (not deleted or nullptr).
// This function will prevent the graph from being deleted until
// TF_DeleteSessionWithGraph() is called.  Does not take ownership of opts.
// TODO(josh11b): Rename this TF_NewSession() once we delete the old API.
extern TF_SessionWithGraph* TF_NewSessionWithGraph(
    TF_Graph* graph, const TF_SessionOptions* opts, TF_Status* status);

// Close a session. This contacts any other processes associated with this
// session, if applicable. This may not be called after
// TF_DeleteSessionWithGraph().
// TODO(josh11b): Rename this TF_CloseSession() once we delete the old API.
extern void TF_CloseSessionWithGraph(TF_SessionWithGraph*, TF_Status* status);

// Destroy a session object.  Even if error information is recorded in
// *status, this call discards all local resources associated with the
// session.  The session may not be used during or after this call
// (and the session drops its reference to the corresponding graph).
// TODO(josh11b): Rename this TF_DeleteSession() once we delete the old API.
extern void TF_DeleteSessionWithGraph(TF_SessionWithGraph*, TF_Status* status);

// See TF_Run() below.
extern void TF_SessionRun(TF_SessionWithGraph* session,
                          // RunOptions
                          const TF_Buffer* run_options,
                          // Input tensors
                          const TF_Port* inputs, TF_Tensor* const* input_values,
                          int ninputs,
                          // Output tensors
                          const TF_Port* outputs, TF_Tensor** output_values,
                          int noutputs,
                          // Target nodes
                          const TF_Node* const* target_nodes, int ntargets,
                          // RunMetadata
                          TF_Buffer* run_metadata,
                          // Output status
                          TF_Status*);

// See TF_PRunSetup() below.
extern void TF_SessionPRunSetup(TF_SessionWithGraph*,
                                // Input names
                                const TF_Port* inputs, int ninputs,
                                // Output names
                                const TF_Port* outputs, int noutputs,
                                // Target nodes
                                const TF_Node* const* target_nodes,
                                int ntargets,
                                // Output handle
                                const char** handle,
                                // Output status
                                TF_Status*);

// See TF_PRun() below.
extern void TF_SessionPRun(TF_SessionWithGraph*, const char* handle,
                           // Input tensors
                           const TF_Port* inputs,
                           TF_Tensor* const* input_values, int ninputs,
                           // Output tensors
                           const TF_Port* outputs, TF_Tensor** output_values,
                           int noutputs,
                           // Target nodes
                           const TF_Node* const* target_nodes, int ntargets,
                           // Output status
                           TF_Status*);

// --------------------------------------------------------------------------
// The deprecated session API.  Please switch to the above instead of
// TF_ExtendGraph().  TF_Session manages a single graph and execution.

typedef struct TF_Session TF_Session;

// Return a new execution session, or NULL on error.
extern TF_Session* TF_NewSession(const TF_SessionOptions*, TF_Status* status);

// Close a session.
extern void TF_CloseSession(TF_Session*, TF_Status* status);

// Destroy a session.  Even if error information is recorded in *status,
// this call discards all resources associated with the session.
extern void TF_DeleteSession(TF_Session*, TF_Status* status);

// Closes all existing sessions connected to the `target` specified in the
// `SessionOptions`, and frees shared resources in `containers` on `target'.
// If no containers are provided, all containers are cleared.
extern void TF_Reset(const TF_SessionOptions* opt, const char** containers,
                     int ncontainers, TF_Status* status);

// Treat the bytes proto[0,proto_len-1] as a serialized GraphDef and
// add the nodes in that GraphDef to the graph for the session.
extern void TF_ExtendGraph(TF_Session*, const void* proto, size_t proto_len,
                           TF_Status*);

// Run the graph associated with the session starting with the
// supplied inputs (inputs[0,ninputs-1]).  Regardless of success or
// failure, inputs[] become the property of the implementation (the
// implementation will eventually call TF_DeleteTensor on each input).
//
// Any NULL and non-NULL value combinations for (`run_options`,
// `run_metadata`) are valid.
//
//    - `run_options` may be NULL, in which case it will be ignored; or
//      non-NULL, in which case it must point to a `TF_Buffer` containing the
//      serialized representation of a `RunOptions` protocol buffer.
//    - `run_metadata` may be NULL, in which case it will be ignored; or
//      non-NULL, in which case it must point to an empty, freshly allocated
//      `TF_Buffer` that may be updated to contain the serialized representation
//      of a `RunMetadata` protocol buffer.
//
// The caller retains the ownership of `run_options` and/or `run_metadata` (when
// not NULL) and should manually call TF_DeleteBuffer on them.
//
// On success, the tensors corresponding to output_names[0,noutputs-1]
// are placed in outputs[], and these outputs[] become the property
// of the caller (the caller must eventually call TF_DeleteTensor on
// them).
//
// On failure, outputs[] contains NULLs.
extern void TF_Run(TF_Session*,
                   // RunOptions
                   const TF_Buffer* run_options,
                   // Input tensors
                   const char** input_names, TF_Tensor** inputs, int ninputs,
                   // Output tensors
                   const char** output_tensor_names, TF_Tensor** outputs,
                   int noutputs,
                   // Target nodes
                   const char** target_node_names, int ntargets,
                   // RunMetadata
                   TF_Buffer* run_metadata,
                   // Output status
                   TF_Status*);

// Set up the graph with the intended feeds and fetches for a sequence
// of partial run calls.
//
// On success, returns a handle that is used for subsequent PRun calls.
//
// On failure, out_status contains a tensorflow::Status with an error
// message.
// NOTE: This is EXPERIMENTAL and subject to change.
extern void TF_PRunSetup(TF_Session*,
                         // Input names
                         const char** input_names, int ninputs,
                         // Output names
                         const char** output_tensor_names, int noutputs,
                         // Target nodes
                         const char** target_node_names, int ntargets,
                         // Output handle
                         const char** handle,
                         // Output status
                         TF_Status*);

// Continue to run the graph with additional feeds and fetches. The
// execution state is uniquely identified by the handle.
// NOTE: This is EXPERIMENTAL and subject to change.
extern void TF_PRun(TF_Session*, const char* handle,
                    // Input tensors
                    const char** input_names, TF_Tensor** inputs, int ninputs,
                    // Output tensors
                    const char** output_tensor_names, TF_Tensor** outputs,
                    int noutputs,
                    // Target nodes
                    const char** target_node_names, int ntargets,
                    // Output status
                    TF_Status*);

// --------------------------------------------------------------------------
// Load plugins containing custom ops and kernels

// TF_Library holds information about dynamically loaded TensorFlow plugins.
typedef struct TF_Library TF_Library;

// Load the library specified by library_filename and register the ops and
// kernels present in that library.
//
// Pass "library_filename" to a platform-specific mechanism for dynamically
// loading a library. The rules for determining the exact location of the
// library are platform-specific and are not documented here.
// Expects the symbols "RegisterOps", "RegisterKernels", and "GetOpList", to be
// defined in the library.
//
// On success, place OK in status and return the newly created library handle.
// The caller owns the library handle.
//
// On failure, place an error status in status and return NULL.
extern TF_Library* TF_LoadLibrary(const char* library_filename,
                                  TF_Status* status);

// Get the OpList of OpDefs defined in the library pointed by lib_handle.
//
// Returns a TF_Buffer. The memory pointed to by the result is owned by
// lib_handle. The data in the buffer will be the serialized OpList proto for
// ops defined in the library.
extern TF_Buffer TF_GetOpList(TF_Library* lib_handle);

#ifdef __cplusplus
} /* end extern "C" */
#endif

#endif  // TENSORFLOW_PUBLIC_TENSOR_C_API_H_
