<?php
// $Id: simpletest.pages.inc,v 1.23 2010/01/15 04:16:57 dries Exp $

/**
 * @file
 * Page callbacks for simpletest module.
 */

/**
 * List tests arranged in groups that can be selected and run.
 */
function simpletest_test_form($form) {
  $form['tests'] = array(
    '#type' => 'fieldset',
    '#title' => t('Tests'),
    '#description' => t('Select the test(s) or test group(s) you would like to run, and click <em>Run tests</em>.'),
  );

  $form['tests']['table'] = array(
    '#theme' => 'simpletest_test_table',
  );

  // Generate the list of tests arranged by group.
  $groups = simpletest_test_get_all();
  foreach ($groups as $group => $tests) {
    $form['tests']['table'][$group] = array(
      '#collapsed' => TRUE,
    );

    foreach ($tests as $class => $info) {
      $form['tests']['table'][$group][$class] = array(
        '#type' => 'checkbox',
        '#title' => $info['name'],
        '#description' => $info['description'],
      );
    }
  }

  // Operation buttons.
  $form['tests']['op'] = array(
    '#type' => 'submit',
    '#value' => t('Run tests'),
  );
  $form['clean'] = array(
    '#type' => 'fieldset',
    '#collapsible' => FALSE,
    '#collapsed' => FALSE,
    '#title' => t('Clean test environment'),
    '#description' => t('Remove tables with the prefix "simpletest" and temporary directories that are left over from tests that crashed. This is intended for developers when creating tests.'),
  );
  $form['clean']['op'] = array(
    '#type' => 'submit',
    '#value' => t('Clean environment'),
    '#submit' => array('simpletest_clean_environment'),
  );

  return $form;
}

/**
 * Theme the test list generated by simpletest_test_form() into a table.
 *
 * @param $variables
 *   An associative array containing:
 *   - table: Form array that represent a table.
 *
 * @return
 *   HTML output.
 */
function theme_simpletest_test_table($variables) {
  $table = $variables['table'];

  drupal_add_css(drupal_get_path('module', 'simpletest') . '/simpletest.css');
  drupal_add_js(drupal_get_path('module', 'simpletest') . '/simpletest.js');

  // Create header for test selection table.
  $header = array(
    theme('table_select_header_cell'),
    array('data' => t('Test'), 'class' => array('simpletest_test')),
    array('data' => t('Description'), 'class' => array('simpletest_description')),
  );

  // Define the images used to expand/collapse the test groups.
  $js = array(
    'images' => array(
      theme('image', array('path' => 'misc/menu-collapsed.png', 'alt' => 'Expand', 'title' => 'Expand')),
      theme('image', array('path' => 'misc/menu-expanded.png', 'alt' => 'Collapsed', 'title' => 'Collapsed')),
    ),
  );

  // Cycle through each test group and create a row.
  $rows = array();
  foreach (element_children($table) as $key) {
    $element = &$table[$key];
    $row = array();

    // Make the class name safe for output on the page by replacing all
    // non-word/decimal characters with a dash (-).
    $test_class = strtolower(trim(preg_replace("/[^\w\d]/", "-", $key)));

    // Select the right "expand"/"collapse" image, depending on whether the
    // category is expanded (at least one test selected) or not.
    $collapsed = !empty($element['#collapsed']);
    $image_index = $collapsed ? 0 : 1;

    // Place-holder for checkboxes to select group of tests.
    $row[] = array('id' => $test_class, 'class' => array('simpletest-select-all'));

    // Expand/collapse image and group title.
    $row[] = array(
      'data' =>  '<div class="simpletest-image" id="simpletest-test-group-' . $test_class . '"></div>&nbsp;' .
                 '<label for="' . $test_class . '-select-all" class="simpletest-group-label">' . $key . '</label>',
      'style' => 'font-weight: bold;'
    );

    $row[] = '&nbsp;';

    $rows[] = array('data' => $row, 'class' => array('simpletest-group'));

    // Add individual tests to group.
    $current_js = array(
      'testClass' => $test_class . '-test',
      'testNames' => array(),
      'imageDirection' => $image_index,
      'clickActive' => FALSE,
    );

    // Sorting $element by children's #title attribute instead of by class name.
    uasort($element, '_simpletest_sort_by_title');

    // Cycle through each test within the current group.
    foreach (element_children($element) as $test_name) {
      $test = $element[$test_name];
      $row = array();

      $current_js['testNames'][] = $test['#id'];

      // Store test title and description so that checkbox won't render them.
      $title = $test['#title'];
      $description = $test['#description'];

      unset($test['#title']);
      unset($test['#description']);

      // Test name is used to determine what tests to run.
      $test['#name'] = $test_name;

      $row[] = drupal_render($test);
      $row[] = theme('indentation', array('size' => 1)) . '<label for="' . $test['#id'] . '">' . $title . '</label>';
      $row[] = '<div class="description">' . $description . '</div>';

      $rows[] = array('data' => $row, 'class' => array($test_class . '-test', ($collapsed ? 'js-hide' : '')));
    }
    $js['simpletest-test-group-' . $test_class] = $current_js;
    unset($table[$key]);
  }

  // Add js array of settings.
  drupal_add_js(array('simpleTest' => $js), 'setting');

  if (empty($rows)) {
    return '<strong>' . t('No tests to display.') . '</strong>';
  }
  else {
    return theme('table', array('header' => $header, 'rows' => $rows, 'attributes' => array('id' => 'simpletest-form-table')));
  }
}

/**
 * Sort element by title instead of by class name.
 */
function _simpletest_sort_by_title($a, $b) {
  // This is for parts of $element that are not an array.
  if (!isset($a['#title']) || !isset($b['#title'])) {
    return 1;
  }

  return strcasecmp($a['#title'], $b['#title']);
}

/**
 * Run selected tests.
 */
function simpletest_test_form_submit($form, &$form_state) {
  // Get list of tests.
  $tests_list = array();
  foreach ($form_state['values'] as $class_name => $value) {
    if (class_exists($class_name) && $value === 1) {
      $tests_list[] = $class_name;
    }
  }
  if (count($tests_list) > 0 ) {
    $test_id = simpletest_run_tests($tests_list, 'drupal');
    $form_state['redirect'] = 'admin/config/development/testing/results/' . $test_id;
  }
  else {
    drupal_set_message(t('No test(s) selected.'), 'error');
  }
}

/**
 * Test results form for $test_id.
 */
function simpletest_result_form($form, &$form_state, $test_id) {
  // Make sure there are test results to display and a re-run is not being performed.
  $results = array();
  if (is_numeric($test_id) && !$results = simpletest_result_get($test_id)) {
    drupal_set_message(t('No test results to display.'), 'error');
    drupal_goto('admin/config/development/testing');
    return $form;
  }

  // Load all classes and include CSS.
  drupal_add_css(drupal_get_path('module', 'simpletest') . '/simpletest.css');

  // Keep track of which test cases passed or failed.
  $filter = array(
    'pass' => array(),
    'fail' => array(),
  );

  // Summary result fieldset.
  $form['result'] = array(
    '#type' => 'fieldset',
    '#title' => t('Results'),
  );
  $form['result']['summary'] = $summary = array(
    '#theme' => 'simpletest_result_summary',
    '#pass' => 0,
    '#fail' => 0,
    '#exception' => 0,
    '#debug' => 0,
  );

  // Cycle through each test group.
  $header = array(t('Message'), t('Group'), t('Filename'), t('Line'), t('Function'), array('colspan' => 2, 'data' => t('Status')));
  $form['result']['results'] = array();
  foreach ($results as $group => $assertions) {
    // Create group fieldset with summary information.
    $info = call_user_func(array($group, 'getInfo'));
    $form['result']['results'][$group] = array(
      '#type' => 'fieldset',
      '#title' => $info['name'],
      '#description' => $info['description'],
      '#collapsible' => TRUE,
    );
    $form['result']['results'][$group]['summary'] = $summary;
    $group_summary = &$form['result']['results'][$group]['summary'];

    // Create table of assertions for the group.
    $rows = array();
    foreach ($assertions as $assertion) {
      $row = array();
      $row[] = $assertion->message;
      $row[] = $assertion->message_group;
      $row[] = basename($assertion->file);
      $row[] = $assertion->line;
      $row[] = $assertion->function;
      $row[] = simpletest_result_status_image($assertion->status);

      $class = 'simpletest-' . $assertion->status;
      if ($assertion->message_group == 'Debug') {
        $class = 'simpletest-debug';
      }
      $rows[] = array('data' => $row, 'class' => array($class));

      $group_summary['#' . $assertion->status]++;
      $form['result']['summary']['#' . $assertion->status]++;
    }
    $form['result']['results'][$group]['table'] = array(
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
    );

    // Set summary information.
    $group_summary['#ok'] = $group_summary['#fail'] + $group_summary['#exception'] == 0;
    $form['result']['results'][$group]['#collapsed'] = $group_summary['#ok'] && !$group_summary['#debug'];

    // Store test group (class) as for use in filter.
    $filter[$group_summary['#ok'] ? 'pass' : 'fail'][] = $group;
  }

  // Overal summary status.
  $form['result']['summary']['#ok'] = $form['result']['summary']['#fail'] + $form['result']['summary']['#exception'] == 0;

  // Actions.
  $form['#action'] = url('admin/config/development/testing/results/re-run');
  $form['action'] = array(
    '#type' => 'fieldset',
    '#title' => t('Actions'),
    '#attributes' => array('class' => array('container-inline')),
    '#weight' => -11,
  );

  $form['action']['filter'] = array(
    '#type' => 'select',
    '#title' => 'Filter',
    '#options' => array(
      'all' => t('All (@count)', array('@count' => count($filter['pass']) + count($filter['fail']))),
      'pass' => t('Pass (@count)', array('@count' => count($filter['pass']))),
      'fail' => t('Fail (@count)', array('@count' => count($filter['fail']))),
    ),
  );
  $form['action']['filter']['#default_value'] = ($filter['fail'] ? 'fail' : 'all');

  // Catagorized test classes for to be used with selected filter value.
  $form['action']['filter_pass'] = array(
    '#type' => 'hidden',
    '#default_value' => implode(',', $filter['pass']),
  );
  $form['action']['filter_fail'] = array(
    '#type' => 'hidden',
    '#default_value' => implode(',', $filter['fail']),
  );

  $form['action']['op'] = array(
    '#type' => 'submit',
    '#value' => t('Run tests'),
  );

  $form['action']['return'] = array(
    '#type' => 'link',
    '#title' => t('Return to list'),
    '#href' => 'admin/config/development/testing',
  );

  if (is_numeric($test_id)) {
    simpletest_clean_results_table($test_id);
  }

  return $form;
}

/**
 * Re-run the tests that match the filter.
 */
function simpletest_result_form_submit($form, &$form_state) {
  $pass = $form_state['values']['filter_pass'] ? explode(',', $form_state['values']['filter_pass']) : array();
  $fail = $form_state['values']['filter_fail'] ? explode(',', $form_state['values']['filter_fail']) : array();

  if ($form_state['values']['filter'] == 'all') {
    $classes = array_merge($pass, $fail);
  }
  else if ($form_state['values']['filter'] == 'pass') {
    $classes = $pass;
  }
  else {
    $classes = $fail;
  }

  if (!$classes) {
    $form_state['redirect'] = 'admin/config/development/testing';
    return;
  }

  $form_state_execute = array('values' => array());
  foreach ($classes as $class) {
    $form_state_execute['values'][$class] = 1;
  }

  simpletest_test_form_submit(array(), $form_state_execute);
}

/**
 * Add wrapper div with class based on summary status.
 *
 * @return HTML output.
 */
function theme_simpletest_result_summary($variables) {
  $form = $variables['form'];
  return '<div class="simpletest-' . ($form['#ok'] ? 'pass' : 'fail') . '">' . _simpletest_format_summary_line($form) . '</div>';
}

/**
 * Get test results for $test_id.
 *
 * @param $test_id The test_id to retrieve results of.
 * @return Array of results grouped by test_class.
 */
function simpletest_result_get($test_id) {
  $results = db_select('simpletest')
    ->fields('simpletest')
    ->condition('test_id', $test_id)
    ->orderBy('test_class')
    ->orderBy('message_id')
    ->execute();

  $test_results = array();
  foreach ($results as $result) {
    if (!isset($test_results[$result->test_class])) {
      $test_results[$result->test_class] = array();
    }
    $test_results[$result->test_class][] = $result;
  }
  return $test_results;
}

/**
 * Get the appropriate image for the status.
 *
 * @param $status Status string, either: pass, fail, exception.
 * @return HTML image or false.
 */
function simpletest_result_status_image($status) {
  // $map does not use drupal_static() as its value never changes.
  static $map;

  if (!isset($map)) {
    $map = array(
      'pass' => theme('image', array('path' => 'misc/watchdog-ok.png', 'alt' => t('Pass'))),
      'fail' => theme('image', array('path' => 'misc/watchdog-error.png', 'alt' => t('Fail'))),
      'exception' => theme('image', array('path' => 'misc/watchdog-warning.png', 'alt' => t('Exception'))),
      'debug' => theme('image', array('path' => 'misc/watchdog-warning.png', 'alt' => t('Debug'))),
    );
  }
  if (isset($map[$status])) {
    return $map[$status];
  }
  return FALSE;
}

/**
 * Provides settings form for SimpleTest variables.
 */
function simpletest_settings_form($form, &$form_state) {
  $form['general'] = array(
    '#type' => 'fieldset',
    '#title' => t('General'),
  );
  $form['general']['simpletest_clear_results'] = array(
    '#type' => 'checkbox',
    '#title' => t('Clear results after each complete test suite run'),
    '#description' => t('By default SimpleTest will clear the results after they have been viewed on the results page, but in some cases it may be useful to leave the results in the database. The results can then be viewed at <em>admin/config/development/testing/[test_id]</em>. The test ID can be found in the database, simpletest table, or kept track of when viewing the results the first time. Additionally, some modules may provide more analysis or features that require this setting to be disabled.'),
    '#default_value' => variable_get('simpletest_clear_results', TRUE),
  );
  $form['general']['simpletest_verbose'] = array(
    '#type' => 'checkbox',
    '#title' => t('Provide verbose information when running tests'),
    '#description' => t('The verbose data will be printed along with the standard assertions and is useful for debugging. The verbose data will be erased between each test suite run. The verbose data output is very detailed and should only be used when debugging.'),
    '#default_value' => variable_get('simpletest_verbose', FALSE),
  );

  $form['httpauth'] = array(
    '#type' => 'fieldset',
    '#title' => t('HTTP authentication'),
    '#description' => t('HTTP auth settings to be used by the SimpleTest browser during testing. Useful when the site requires basic HTTP authentication.'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );
  $form['httpauth']['simpletest_httpauth_method'] = array(
    '#type' => 'select',
    '#title' => t('Method'),
    '#options' => array(
      CURLAUTH_BASIC => t('Basic'),
      CURLAUTH_DIGEST => t('Digest'),
      CURLAUTH_GSSNEGOTIATE => t('GSS negotiate'),
      CURLAUTH_NTLM => t('NTLM'),
      CURLAUTH_ANY => t('Any'),
      CURLAUTH_ANYSAFE => t('Any safe'),
    ),
    '#default_value' => variable_get('simpletest_httpauth_method', CURLAUTH_BASIC),
  );
  $form['httpauth']['simpletest_httpauth_username'] = array(
    '#type' => 'textfield',
    '#title' => t('Username'),
    '#default_value' => variable_get('simpletest_httpauth_username', ''),
  );
  $form['httpauth']['simpletest_httpauth_password'] = array(
    '#type' => 'textfield',
    '#title' => t('Password'),
    '#default_value' => variable_get('simpletest_httpauth_password', ''),
  );

  return system_settings_form($form);
}
