/**

GDevelop - Pathfinding Behavior Extension
Copyright (c) 2010-2016 Florian Rival (Florian.Rival@gmail.com)
This project is released under the MIT License.
*/

#include "GDCore/Extensions/PlatformExtension.h"
#include "GDCore/Extensions/Metadata/MultipleInstructionMetadata.h"
#include "GDCore/Tools/Localization.h"
#include "GDCore/Project/BehaviorsSharedData.h"
#include "PathfindingBehavior.h"
#include "PathfindingObstacleBehavior.h"

void DeclarePathfindingBehaviorExtension(gd::PlatformExtension& extension) {
  extension
      .SetExtensionInformation(
          "PathfindingBehavior",
          _("Pathfinding behavior"),
          "Pathfinding allows to compute an efficient path for objects, "
          "avoiding obstacles on the way.",
          "Florian Rival",
          "Open source (MIT License)")
      .SetExtensionHelpPath("/behaviors/pathfinding");

  {
    gd::BehaviorMetadata& aut =
        extension.AddBehavior("PathfindingBehavior",
                              _("Pathfinding"),
                              "Pathfinding",
                              _("With this behavior, the object will move "
                                "while avoiding all objects that are "
                                "flagged as obstacles."),
                              "",
                              "CppPlatform/Extensions/AStaricon.png",
                              "PathfindingBehavior",
                              std::make_shared<PathfindingBehavior>(),
                              std::make_shared<gd::BehaviorsSharedData>());

#if defined(GD_IDE_ONLY)

    aut.SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("SetDestination",
                  _("Move to a position"),
                  _("Move the object to a position"),
                  _("Move _PARAM0_ to _PARAM3_;_PARAM4_"),
                  "Movement on the path",
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .AddCodeOnlyParameter("currentScene", "")

        .AddParameter("expression", _("Destination X position"))
        .AddParameter("expression", _("Destination Y position"))
        .SetFunctionName("MoveTo")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("PathFound",
                     _("Path found"),
                     _("Check if a path has been found."),
                     _("A path has been found for _PARAM0_"),
                     "Movement on the path",
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("PathFound")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("DestinationReached",
                     _("Destination reached"),
                     _("Check if the destination was reached."),
                     _("_PARAM0_ reached its destination"),
                     "Movement on the path",
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("DestinationReached")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("CellWidth",
                  _("Width of the cells"),
                  _("Change the width of the cells of the virtual grid."),
                  _("the width of the virtual cells"),
                  _("Virtual grid"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetCellWidth")
        .SetGetter("GetCellWidth")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("CellWidth",
                     _("Width of the virtual grid"),
                     _("Compare the width of the cells of the virtual grid."),
                     _("the width of the virtual cells"),
                     _("Virtual grid"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetCellWidth")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("CellHeight",
                  _("Height of the cells"),
                  _("Change the height of the cells of the virtual grid."),
                  _("the height of the virtual cells"),
                  _("Virtual grid"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetCellHeight")
        .SetGetter("GetCellHeight")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("CellHeight",
                     _("Height of the virtual grid"),
                     _("Compare the height of the cells of the virtual grid."),
                     _("the height of the virtual cells"),
                     _("Virtual grid"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetCellHeight")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("Acceleration",
                  _("Acceleration"),
                  _("Change the acceleration when moving the object"),
                  _("the acceleration on the path"),
                  _("Pathfinding configuration"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetAcceleration")
        .SetGetter("GetAcceleration")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("Acceleration",
                     _("Acceleration"),
                     _("Compare the acceleration when moving the object"),
                     _("the acceleration"),
                     _("Pathfinding configuration"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetAcceleration")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("MaxSpeed",
                  _("Maximum speed"),
                  _("Change the maximum speed when moving the object"),
                  _("the max. speed on the path"),
                  _("Pathfinding configuration"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetMaxSpeed")
        .SetGetter("GetMaxSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("MaxSpeed",
                     _("Maximum speed"),
                     _("Compare the maximum speed when moving the object"),
                     _("the max. speed"),
                     _("Pathfinding configuration"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetMaxSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("Speed",
                  _("Speed"),
                  _("Change the speed of the object on the path"),
                  _("the speed on the path"),
                  _("Movement on the path"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetSpeed")
        .SetGetter("GetSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("Speed",
                     _("Speed on its path"),
                     _("Compare the speed of the object on its path."),
                     _("the speed"),
                     _("Movement on the path"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddScopedCondition("MovementAngleIsAround",
                          _("Angle of movement on its path"),
                          _("Compare the angle of movement of an object on its path."),
                          _("Angle of movement of _PARAM0_ is _PARAM2_ (tolerance"
                              ": _PARAM3_ degrees)"),
                          _("Movement on the path"),
                         "CppPlatform/Extensions/AStaricon24.png",
                         "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .AddParameter("expression", _("Angle, in degrees"))
        .AddParameter("expression", _("Tolerance, in degrees"));

    aut.AddAction("AngularMaxSpeed",
                  _("Angular maximum speed"),
                  _("Change the maximum angular speed when moving the object"),
                  _("the max. angular speed on the path"),
                  _("Pathfinding configuration"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetAngularMaxSpeed")
        .SetGetter("GetAngularMaxSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition(
           "AngularMaxSpeed",
           _("Angular maximum speed"),
           _("Compare the maximum angular speed when moving the object"),
           _("the max. angular speed"),
           _("Pathfinding configuration"),
           "CppPlatform/Extensions/AStaricon24.png",
           "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetAngularMaxSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction(
           "AngleOffset",
           _("Rotation offset"),
           _("Change the rotation offset applied when moving the object"),
           _("the rotation offset on the path"),
           _("Pathfinding configuration"),
           "CppPlatform/Extensions/AStaricon24.png",
           "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetAngleOffset")
        .SetGetter("GetAngleOffset")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("AngleOffset",
                     _("Rotation offset"),
                     _("Compare the rotation offset when moving the object"),
                     _("the rotation offset"),
                     _("Pathfinding configuration"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetAngleOffset")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction(
           "ExtraBorder",
           _("Extra border"),
           _("Change the size of the extra border applied to the object when "
             "planning a path"),
           _("the size of the extra border on the path"),
           _("Pathfinding configuration"),
           "CppPlatform/Extensions/AStaricon24.png",
           "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetExtraBorder")
        .SetGetter("GetExtraBorder")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("ExtraBorder",
                     _("Extra border"),
                     _("Compare the size of the extra border applied to the "
                       "object when planning a path"),
                     _("the size of the extra border on the path"),
                     _("Pathfinding configuration"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetExtraBorder")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction(
           "AllowDiagonals",
           _("Diagonal movement"),
           _("Allow or restrict diagonal movement on the path"),
           _("Allow diagonal movement for _PARAM0_ on the path: _PARAM2_"),
           _("Pathfinding configuration"),
           "CppPlatform/Extensions/AStaricon24.png",
           "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .AddParameter("yesorno", _("Allow?"))
        .SetFunctionName("SetAllowDiagonals")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("DiagonalsAllowed",
                     _("Diagonal movement"),
                     _("Check if the object is allowed to move "
                       "diagonally on the path"),
                     _("Diagonal moves allowed for _PARAM0_"),
                     _("Pathfinding configuration"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("DiagonalsAllowed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddAction("RotateObject",
                  _("Rotate the object"),
                  _("Enable or disable rotation of the object on the path"),
                  _("Enable rotation of _PARAM0_ on the path: _PARAM2_"),
                  _("Pathfinding configuration"),
                  "CppPlatform/Extensions/AStaricon24.png",
                  "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .AddParameter("yesorno", _("Rotate object?"))
        .SetFunctionName("SetRotateObject")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddCondition("ObjectRotated",
                     _("Object rotated"),
                     _("Check if the object is rotated when traveling on "
                       "its path."),
                     _("_PARAM0_ is rotated when traveling on its path"),
                     _("Pathfinding configuration"),
                     "CppPlatform/Extensions/AStaricon24.png",
                     "CppPlatform/Extensions/AStaricon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("IsObjectRotated")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("GetNodeX",
                      _("Get a waypoint X position"),
                      _("Get next waypoint X position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .AddParameter("expression", _("Node index (start at 0!)"))
        .SetFunctionName("GetNodeX")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("GetNodeY",
                      _("Get a waypoint Y position"),
                      _("Get next waypoint Y position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .AddParameter("expression", _("Node index (start at 0!)"))
        .SetFunctionName("GetNodeY")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("NextNodeIndex",
                      _("Index of the next waypoint"),
                      _("Get the index of the next waypoint to reach"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetNextNodeIndex")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("NodeCount",
                      _("Waypoint count"),
                      _("Get the number of waypoints on the path"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetNodeCount")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("NextNodeX",
                      _("Get next waypoint X position"),
                      _("Get next waypoint X position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetNextNodeX")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("NextNodeY",
                      _("Get next waypoint Y position"),
                      _("Get next waypoint Y position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetNextNodeY")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("LastNodeX",
                      _("Last waypoint X position"),
                      _("Last waypoint X position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetLastNodeX")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("LastNodeY",
                      _("Last waypoint Y position"),
                      _("Last waypoint Y position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetLastNodeY")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("DestinationX",
                      _("Destination X position"),
                      _("Destination X position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetDestinationX")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("DestinationY",
                      _("Destination Y position"),
                      _("Destination Y position"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetDestinationY")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("Acceleration",
                      _("Acceleration"),
                      _("Acceleration of the object on the path"),
                      _("Pathfinding configuration"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetAcceleration")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("MaxSpeed",
                      _("Maximum speed"),
                      _("Maximum speed of the object on the path"),
                      _("Pathfinding configuration"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetMaxSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("Speed",
                      _("Speed"),
                      _("Speed of the object on the path"),
                      _("Movement on the path"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("AngularMaxSpeed",
                      _("Angular maximum speed"),
                      _("Angular maximum speed of the object on the path"),
                      _("Pathfinding configuration"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetAngularMaxSpeed")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("AngleOffset",
                      _("Rotation offset"),
                      _("Rotation offset applied the object on the path"),
                      _("Pathfinding configuration"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetAngleOffset")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("ExtraBorder",
                      _("Extra border size"),
                      _("Extra border applied the object on the path"),
                      _("Pathfinding configuration"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetExtraBorder")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("CellWidth",
                      _("Width of a cell"),
                      _("Width of the virtual grid"),
                      _("Virtual grid"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetCellWidth")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("CellHeight",
                      _("Height of a cell"),
                      _("Height of the virtual grid"),
                      _("Virtual grid"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .SetFunctionName("GetCellHeight")
        .SetIncludeFile("PathfindingBehavior/PathfindingRuntimeBehavior.h");

    aut.AddExpression("MovementAngle",
                    _("Angle of movement on its path"),
                    _("Angle of movement on its path"),
                    _("Movement on the path"),
                    "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior");

    aut.AddExpressionAndConditionAndAction("number",
                      "GridOffsetX",
                      _("Grid X offset"),
                      _("X offset of the virtual grid"),
                      _("X offset of the virtual grid"),
                      _("Virtual grid"),
                      "CppPlatform/Extensions/AStaricon24.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardParameters("number");

    aut.AddExpressionAndConditionAndAction("number",
                      "GridOffsetY",
                      _("Grid Y offset"),
                      _("Y offset of the virtual grid"),
                      _("Y offset of the virtual grid"),
                      _("Virtual grid"),
                      "CppPlatform/Extensions/AStaricon24.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingBehavior")
        .UseStandardParameters("number");

#endif
  }
  {
    gd::BehaviorMetadata& aut = extension.AddBehavior(
        "PathfindingObstacleBehavior",
        _("Obstacle for pathfinding"),
        "PathfindingObstacle",
        _("Flag the object as being an obstacle for pathfinding."),
        "",
        "CppPlatform/Extensions/pathfindingobstacleicon.png",
        "PathfindingObstacleBehavior",
        std::make_shared<PathfindingObstacleBehavior>(),
        std::make_shared<gd::BehaviorsSharedData>());

#if defined(GD_IDE_ONLY)
    aut.SetIncludeFile(
        "PathfindingBehavior/PathfindingObstacleRuntimeBehavior.h");

    aut.AddAction("Cost",
                  _("Cost"),
                  _("Change the cost of going through the object."),
                  _("the cost"),
                  _("Obstacles"),
                  "CppPlatform/Extensions/pathfindingobstacleicon24.png",
                  "CppPlatform/Extensions/pathfindingobstacleicon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingObstacleBehavior")
        .UseStandardOperatorParameters("number")
        .SetFunctionName("SetCost")
        .SetGetter("GetCost")
        .SetIncludeFile(
            "PathfindingBehavior/PathfindingObstacleRuntimeBehavior.h");

    aut.AddCondition("Cost",
                     _("Cost"),
                     _("Compare the cost of going through the object"),
                     _("the cost"),
                     _("Obstacles"),
                     "CppPlatform/Extensions/pathfindingobstacleicon24.png",
                     "CppPlatform/Extensions/pathfindingobstacleicon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingObstacleBehavior")
        .UseStandardRelationalOperatorParameters("number")
        .SetFunctionName("GetCost")
        .SetIncludeFile(
            "PathfindingBehavior/PathfindingObstacleRuntimeBehavior.h");

    aut.AddAction("SetImpassable",
                  _("Should object be impassable?"),
                  _("Decide if the object is an impassable obstacle"),
                  _("Set _PARAM0_ as an impassable obstacle: _PARAM2_"),
                  _("Obstacles"),
                  "CppPlatform/Extensions/pathfindingobstacleicon24.png",
                  "CppPlatform/Extensions/pathfindingobstacleicon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingObstacleBehavior")
        .AddParameter("yesorno", _("Impassable?"))
        .SetFunctionName("SetImpassable")
        .SetIncludeFile(
            "PathfindingBehavior/PathfindingObstacleRuntimeBehavior.h");

    aut.AddCondition("IsImpassable",
                     _("Is object impassable?"),
                     _("Check if the obstacle is impassable"),
                     _("_PARAM0_ is impassable"),
                     _("Obstacles"),
                     "CppPlatform/Extensions/pathfindingobstacleicon24.png",
                     "CppPlatform/Extensions/pathfindingobstacleicon16.png")

        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingObstacleBehavior")
        .SetFunctionName("IsImpassable")
        .SetIncludeFile(
            "PathfindingBehavior/PathfindingObstacleRuntimeBehavior.h");

    aut.AddExpression("Cost",
                      _("Cost"),
                      _("Obstacle cost"),
                      _("Obstacles"),
                      "CppPlatform/Extensions/AStaricon16.png")
        .AddParameter("object", _("Object"))
        .AddParameter("behavior", _("Behavior"), "PathfindingObstacleBehavior")
        .SetFunctionName("GetCost")
        .SetIncludeFile(
            "PathfindingBehavior/PathfindingObstacleRuntimeBehavior.h");

#endif
  }
}
