# revlog.py - storage back-end for mercurial
#
# This provides efficient delta storage with O(1) retrieve and append
# and O(changes) merge between branches
#
# Copyright 2005 Matt Mackall <mpm@selenic.com>
#
# This software may be used and distributed according to the terms
# of the GNU General Public License, incorporated herein by reference.

import zlib, struct, mdiff, sha, binascii, os, tempfile

def compress(text):
    return zlib.compress(text)

def decompress(bin):
    return zlib.decompress(bin)

def hash(text):
    return sha.sha(text).digest()

class revlog:
    indexformat = ">3l20s20s20s"
    nullid = hash("")

    def __init__(self, indexfile, datafile):
        self.indexfile = indexfile
        self.datafile = datafile
        self.index = []
        self.nodemap = { -1: self.nullid, self.nullid: -1 }
        # read the whole index for now, handle on-demand later
        try:
            n = 0
            i = self.open(self.indexfile).read()
            s = struct.calcsize(self.indexformat)
            for f in range(0, len(i), s):
                # offset, size, base, p1, p2, nodeid
                e = struct.unpack(self.indexformat, i[f:f + s])
                self.nodemap[e[5]] = n
                self.index.append(e)
                n += 1
        except IOError: pass

    def open(self, fn, mode="r"): return file(fn, mode)

    def hex(self, node): return binascii.hexlify(node)
    def bin(self, node): return binascii.unhexlify(node)
    
    def tip(self): return self.node(len(self.index) - 1)
    def count(self): return len(self.index)
    def node(self, rev): return rev < 0 and self.nullid or self.index[rev][5]
    def rev(self, node): return self.nodemap[node]
    def parents(self, node): return self.index[self.nodemap[node]][3:5]

    def start(self, rev): return self.index[rev][0]
    def length(self, rev): return self.index[rev][1]
    def end(self, rev): return self.start(rev) + self.length(rev)
    def base(self, rev): return self.index[rev][2]

    def revisions(self, list):
        # this can be optimized to do spans, etc
        # be stupid for now
        for r in list:
            yield self.revision(r)

    def revision(self, node):
        if node is self.nullid: return ""
        rev = self.rev(node)
        base = self.base(rev)
        start = self.start(base)
        end = self.end(rev)

        f = self.open(self.datafile)
        f.seek(start)
        data = f.read(end - start)

        last = self.length(base)
        text = decompress(data[:last])

        for r in range(base + 1, rev + 1):
            s = self.length(r)
            b = decompress(data[last:last + s])
            text = mdiff.patch(text, b)
            last = last + s

        (p1, p2) = self.parents(node)
        node = hash(p1 + p2 + text)
        if self.node(rev) != node:
            raise "Consistency check failed on %s:%d" % (self.datafile, rev)

        return text  

    def addrevision(self, text, p1=None, p2=None):
        if text is None: text = ""
        if p1 is None: p1 = self.tip()
        if p2 is None: p2 = self.nullid

        n = self.count()
        t = n - 1

        if n:
            start = self.start(self.base(t))
            end = self.end(t)
            prev = self.revision(self.tip())
            data = compress(mdiff.textdiff(prev, text))

        # full versions are inserted when the needed deltas
        # become comparable to the uncompressed text
        if not n or (end + len(data) - start) > len(text) * 2:
            data = compress(text)
            base = n
        else:
            base = self.base(t)

        offset = 0
        if t >= 0:
            offset = self.end(t)

        node = hash(p1 + p2 + text)
        e = (offset, len(data), base, p1, p2, node)
        
        self.index.append(e)
        entry = struct.pack(self.indexformat, *e)
        self.nodemap[node] = n
        self.open(self.indexfile, "a").write(entry)
        self.open(self.datafile, "a").write(data)
        return node

    def ancestor(self, a, b):
        def expand(e1, e2, a1, a2):
            ne = []
            for n in e1:
                (p1, p2) = self.parents(n)
                if p1 in a2: return p1
                if p2 in a2: return p2
                if 1 not in a1:
                    a1[p1] = 1
                    ne.append(p1)
                    if p2 >= 0 and p2 not in a1:
                        a1[p2] = 1
                        ne.append(p2)
            return expand(e2, ne, a2, a1)
        return expand([a], [b], {a:1}, {b:1})

    def mergedag(self, other, accumulate = None):
        """combine the nodes from other's DAG into ours"""
        old = self.tip()
        i = self.count()
        l = []

        # merge the other revision log into our DAG
        for r in range(other.count()):
            id = other.node(r)
            if id not in self.nodemap:
                (xn, yn) = other.parents(id)
                l.append((id, xn, yn))
                self.nodemap[id] = i
                i += 1

        # merge node date for new nodes
        r = other.revisions([e[0] for e in l])
        for e in l:
            t = r.next()
            if accumulate: accumulate(t)
            self.addrevision(t, e[1], e[2])

        # return the unmerged heads for later resolving
        return (old, self.tip())

    def resolvedag(self, old, new):
        """resolve unmerged heads in our DAG"""
        if old == new: return None
        a = self.ancestor(old, new)
        if old == a: return new
        return self.merge3(old, new, a)

    def merge3(self, my, other, base):
        """perform a 3-way merge and append the result"""
        def temp(prefix, node):
            (fd, name) = tempfile.mkstemp(prefix)
            f = os.fdopen(fd, "w")
            f.write(self.revision(node))
            f.close()
            return name

        a = temp("local", my)
        b = temp("remote", other)
        c = temp("parent", base)

        cmd = os.environ["HGMERGE"]
        r = os.system("%s %s %s %s" % (cmd, a, b, c))
        if r:
            raise "Merge failed, implement rollback!"

        t = open(a).read()
        return self.addrevision(t, my, other)

    def merge(self, other):
        """perform a merge and resolve resulting heads"""
        # this is where to handle rollback
        (o, n) = self.mergedag(other)
        return self.resolvedag(o, n)
