# hg.py - repository classes for mercurial
#
# Copyright 2005 Matt Mackall <mpm@selenic.com>
#
# This software may be used and distributed according to the terms
# of the GNU General Public License, incorporated herein by reference.

import sys, struct, sha, socket, os, time, base64, re, urllib2, byterange
from revlog import *

class filelog(revlog):
    def __init__(self, repo, path):
        self.repo = repo
        s = self.encodepath(path)
        revlog.__init__(self, os.path.join("index", s),
                        os.path.join("data/", s))

    def encodepath(self, path):
        s = sha.sha(path).digest()
        s = base64.encodestring(s)[:-3]
        s = re.sub("\+", "%", s)
        s = re.sub("/", "_", s)
        return s

    def open(self, file, mode = "r"):
        return self.repo.open(file, mode)
    def read(self, node):
        return self.revision(node)
    def add(self, text, p1=None, p2=None):
        return self.addrevision(text, p1, p2)

class manifest(revlog):
    def __init__(self, repo):
        self.repo = repo
        revlog.__init__(self, "00manifest.i", "00manifest.d")
    def open(self, file, mode="r"):
        return self.repo.open(file, mode)
    def read(self, node):
        text = self.revision(node)
        map = {}
        for l in mdiff.linesplit(text):
            map[l[41:-1]] = self.bin(l[0:40])
        return map
    def add(self, map, p1=None, p2=None):
        files = map.keys()
        files.sort()
        text = "".join(["%s %s\n" % (self.hex(map[f]), f)
                        for f in files])
        return self.addrevision(text, p1, p2)

class changelog(revlog):
    def __init__(self, repo):
        self.repo = repo
        revlog.__init__(self, "00changelog.i", "00changelog.d")
    def open(self, file, mode="r"):
        return self.repo.open(file, mode)
    def extract(self, text):
        l = mdiff.linesplit(text)
        manifest = self.bin(l[0][:-1])
        user = l[1][:-1]
        date = l[2][:-1]
        last = l.index("\n")
        files = [f[:-1] for f in l[3:last]]
        desc = "".join(l[last+1:])
        return (manifest, user, date, files, desc)
    def read(self, node):
        return self.extract(self.revision(node))
    def add(self, manifest, list, desc, p1=None, p2=None):
        try: user = os.environ["HGUSER"]
        except: user = os.environ["LOGNAME"] + '@' + socket.getfqdn()
        date = "%d %d" % (time.time(), time.timezone)
        list.sort()
        l = [self.hex(manifest), user, date] + list + ["", desc]
        text = "".join([e + "\n" for e in l])
        return self.addrevision(text, p1, p2)

    def merge3(self, my, other, base):
        pass

class repository:
    def __init__(self, ui, path=None, create=0):
        self.remote = 0
        if path and path[:7] == "http://":
            self.remote = 1
            self.path = path
        else:
            if not path:
                p = os.getcwd()
                while not os.path.isdir(os.path.join(p, ".hg")):
                    p = os.path.dirname(p)
                    if p == "/": raise "No repo found"
                path = p
            self.path = os.path.join(path, ".hg")

        self.root = path
        self.ui = ui

        if create:
            os.mkdir(self.path)  
            os.mkdir(os.path.join(self.path, "data"))
            os.mkdir(os.path.join(self.path, "index"))
            
        self.manifest = manifest(self)
        self.changelog = changelog(self)
        self.ignorelist = None

        if not self.remote:
            try:
                self.current = self.changelog.bin(self.open("current").read())
            except:
                self.current = None
      
    def open(self, path, mode="r"):
        f = self.join(path)
        if self.remote:
            return httprangereader(f)

        if mode != "r" and os.path.isfile(f):
            s = os.stat(f)
            if s.st_nlink > 1:
                file(f+".tmp", "w").write(file(f).read())
                os.rename(f+".tmp", f)
        
        return file(self.join(path), mode)

    def ignore(self, f):
        if self.ignorelist is None:
            self.ignorelist = []
            try:
                l = open(os.path.join(self.root, ".hgignore")).readlines()
                for pat in l:
                    self.ignorelist.append(re.compile(pat[:-1]))
            except IOError: pass
        for pat in self.ignorelist:
            if pat.search(f): return True
        return False

    def join(self, f):
        return os.path.join(self.path, f)

    def file(self, f):
        return filelog(self, f)

    def merge(self, other):
        changed = {}
        new = {}
        def accumulate(text):
            files = self.changelog.extract(text)[3]
            for f in files:
                changed[f] = 1

        # begin the import/merge of changesets
        self.ui.status("merging new changesets\n")
        (co, cn) = self.changelog.mergedag(other.changelog, accumulate)

        if co == cn: return

        # merge all files changed by the changesets,
        # keeping track of the new tips
        changed = changed.keys()
        changed.sort()
        for f in changed:
            sys.stdout.write(".")
            sys.stdout.flush()
            r = self.file(f)
            node = r.merge(other.file(f))
            if node:
                new[f] = node
        sys.stdout.write("\n")

        # begin the merge of the manifest
        self.ui.status("merging manifests\n")
        (mm, mo) = self.manifest.mergedag(other.manifest)
        ma = self.manifest.ancestor(mm, mo)

        # resolve the manifest to point to all the merged files
        self.ui.status("resolving manifests\n")
        mmap = self.manifest.read(mm) # mine
        omap = self.manifest.read(mo) # other
        amap = self.manifest.read(ma) # ancestor
        nmap = {}

        for f, mid in mmap.items():
            if f in omap:
                if mid != omap[f]: 
                    nmap[f] = new.get(f, mid) # use merged version
                else:
                    nmap[f] = new.get(f, mid) # they're the same
                del omap[f]
            elif f in amap:
                if mid != amap[f]: 
                    pass # we should prompt here
                else:
                    pass # other deleted it
            else:
                nmap[f] = new.get(f, mid) # we created it

        del mmap

        for f, oid in omap.items():
            if f in amap:
                if oid != amap[f]:
                    pass # this is the nasty case, we should prompt here too
                else:
                    pass # probably safe
            else:
                nmap[f] = new.get(f, oid) # remote created it

        del omap
        del amap

        node = self.manifest.add(nmap, mm, mo)

        # Now all files and manifests are merged, we add the changed files
        # and manifest id to the changelog
        self.ui.status("committing merge changeset\n")
        new = new.keys()
        new.sort()
        if co == cn: cn = -1

        edittext = "\n\n"+"".join(["HG: changed %s\n" % f for f in new])
        edittext = self.ui.edit(edittext)
        # we should give the user an opportunity to edit the changelog desc
        self.changelog.add(node, new, edittext, co, cn)

    def commit(self):
        try:
            remove = [ l[:-1] for l in self.open("to-remove") ]
        except IOError:
            remove = []

        update = self.diffdir(self.root)[0]

        # check in files
        new = {}
        for f in update:
            r = self.file(f)
            t = file(f).read()
            new[f] = r.add(t)

        # update manifest
        old = self.manifest.read(self.manifest.tip())
        old.update(new)
        for f in remove:
            del old[f]
        mnode = self.manifest.add(old)

        # add changeset
        new = new.keys()
        new.sort()

        edittext = "\n\n"+"".join(["HG: changed %s\n" % f for f in new])
        edittext = self.ui.edit(edittext)
        n = self.changelog.add(mnode, new, edittext)
        self.current = n
        self.open("current", "w").write(self.changelog.hex(n))

        # clear pending removals
        if remove: os.unlink(self.join("to-remove"))

        # update dircache
        l = self.manifest.read(mnode).keys()
        l.sort()
        stats = []
        for f in l:
            stats.append((os.stat(f), f))

        st = self.open("dircache", "w")
        for s, f in stats:
            e = (s.st_mode, s.st_size, s.st_mtime, len(f))
            e = struct.pack(">llll", *e)
            st.write(e + f)

    def checkdir(self, path):
        d = os.path.dirname(path)
        if not d: return
        if not os.path.isdir(d):
            self.checkdir(d)
            os.mkdir(d)

    def checkout(self, node):
        change = self.changelog.read(node)
        mmap = self.manifest.read(change[0])

        l = mmap.keys()
        l.sort()
        stats = []
        for f in l:
            r = self.file(f)
            t = r.revision(mmap[f])
            try:
                file(f, "w").write(t)
            except:
                self.checkdir(f)
                file(f, "w").write(t)
            stats.append((os.stat(f), f))

        st = self.open("dircache", "w")
        for s, f in stats:
            e = (s.st_mode, s.st_size, s.st_mtime, len(f))
            e = struct.pack(">llll", *e)
            st.write(e + f)

        self.current = node
        self.open("current", "w").write(self.changelog.hex(self.current))

    def diffdir(self, path):
        dc = {}
        try:
            st = self.open("dircache").read()
            pos = 0

            while pos < len(st):
                e = struct.unpack(">llll", st[pos:pos+16])
                l = e[3]
                pos += 16
                f = st[pos:pos + l]
                dc[f] = e[:3]
                pos += l
        except IOError: pass

        changed = []
        added = []

        mmap = {}
        if self.current:
            change = self.changelog.read(self.current)
            mmap = self.manifest.read(change[0])

        for dir, subdirs, files in os.walk(self.root):
            d = dir[len(self.root)+1:]
            if ".hg" in subdirs: subdirs.remove(".hg")
            
            for f in files:
                fn = os.path.join(d, f)
                try: s = os.stat(fn)
                except: continue
                if fn in dc:
                    c = dc[fn]
                    del dc[fn]
                    if c[1] != s.st_size:
                        changed.append(fn)
                    elif c[0] != s.st_mode or c[2] != s.st_mtime:
                        t1 = file(fn).read()
                        t2 = self.file(fn).revision(mmap[fn])
                        if t1 != t2:
                            changed.append(fn)
                else:
                    if self.ignore(fn): continue
                    added.append(fn)

        deleted = dc.keys()
        deleted.sort()

        return (changed, added, deleted)

    def add(self, list):
        st = self.open("dircache", "a")
        for f in list:
            s = os.stat(f)
            e = (s.st_mode, -1, s.st_mtime, len(f))
            e = struct.pack(">llll", *e)
            st.write(e + f)

    def remove(self, list):
        dl = self.open("to-remove", "a")
        for f in list:
            dl.write(f + "\n")

class ui:
    def __init__(self, verbose=False, debug=False):
        self.verbose = verbose
    def write(self, *args):
        for a in args:
            sys.stdout.write(str(a))
    def prompt(self, msg, pat):
        while 1:
            sys.stdout.write(msg)
            r = sys.stdin.readline()[:-1]
            if re.match(pat, r):
                return r
    def status(self, *msg):
        self.write(*msg)
    def warn(self, msg):
        self.write(*msg)
    def note(self, msg):
        if self.verbose: self.write(*msg)
    def debug(self, msg):
        if self.debug: self.write(*msg)
    def edit(self, text):
        (fd, name) = tempfile.mkstemp("hg")
        f = os.fdopen(fd, "w")
        f.write(text)
        f.close()

        print text, name

        editor = os.environ.get("EDITOR", "vi")
        r = os.system("%s %s" % (editor, name))
        if r:
            raise "Edit failed!"

        t = open(name).read()
        t = re.sub("(?m)^HG:.*\n", "", t)

        return t

    
class httprangereader:
    def __init__(self, url):
        self.url = url
        self.pos = 0
    def seek(self, pos):
        self.pos = pos
    def read(self, bytes=None):
        opener = urllib2.build_opener(byterange.HTTPRangeHandler())
        urllib2.install_opener(opener)
        req = urllib2.Request(self.url)
        end = ''
        if bytes: end = self.pos + bytes
        req.add_header('Range', 'bytes=%d-%s' % (self.pos, end))
        f = urllib2.urlopen(req)
        return f.read()
