import { MaybeTimestamp, asTimestamp } from './timestamp';

describe('util/timestamp', () => {
  describe('asTimestamp', () => {
    test.each`
      input                                        | expected
      ${new Date('2021-01-01T00:00:00.000Z')}      | ${'2021-01-01T00:00:00.000Z'}
      ${new Date('2021-01-01T00:00:00.000-03:00')} | ${'2021-01-01T03:00:00.000Z'}
      ${new Date('1999-01-01T00:00:00.000Z')}      | ${null}
      ${1609459200000}                             | ${'2021-01-01T00:00:00.000Z'}
      ${1609459200}                                | ${'2021-01-01T00:00:00.000Z'}
      ${-1}                                        | ${null}
      ${0}                                         | ${null}
      ${123}                                       | ${null}
      ${NaN}                                       | ${null}
      ${'2021-01-01T00:00:00.000Z'}                | ${'2021-01-01T00:00:00.000Z'}
      ${'2021-01-01'}                              | ${'2021-01-01T00:00:00.000Z'}
      ${'20210101000000'}                          | ${'2021-01-01T00:00:00.000Z'}
      ${'20211231235959'}                          | ${'2021-12-31T23:59:59.000Z'}
      ${'20210101000000+0000'}                     | ${'2021-01-01T00:00:00.000Z'}
      ${'20211231235959+0000'}                     | ${'2021-12-31T23:59:59.000Z'}
      ${'Jan 1, 2021'}                             | ${'2021-01-01T00:00:00.000Z'}
      ${'2021/01/01'}                              | ${'2021-01-01T00:00:00.000Z'}
      ${'2021-01-02T00:00:00+05:30'}               | ${'2021-01-01T18:30:00.000Z'}
      ${'2010-05-20T22:43:19-07:00'}               | ${'2010-05-21T05:43:19.000Z'}
      ${'2021-10-11 07:47:24 -0700'}               | ${'2021-10-11T14:47:24.000Z'}
      ${'Wed, 21 Oct 2015 07:28:00 GMT'}           | ${'2015-10-21T07:28:00.000Z'}
      ${null}                                      | ${null}
      ${undefined}                                 | ${null}
      ${{}}                                        | ${null}
      ${[]}                                        | ${null}
      ${'invalid date'}                            | ${null}
      ${'202x0101000000'}                          | ${null}
    `('$input -> $expected', ({ input, expected }) => {
      expect(asTimestamp(input)).toBe(expected);
      expect(MaybeTimestamp.parse(input)).toBe(expected);
    });
  });
});
