import { parseVersion } from './parser';
import { compare } from './version';

describe('modules/versioning/nuget/version', () => {
  describe('compare', () => {
    test.each`
      x                               | y                             | expected
      ${'17.4'}                       | ${'17.04'}                    | ${0}
      ${'1.4'}                        | ${'1.4.0'}                    | ${0}
      ${'1.0.110'}                    | ${'1.0.110.0'}                | ${0}
      ${'1.0.0'}                      | ${'1.0.0+c30d7625'}           | ${0}
      ${'1.022'}                      | ${'1.22.0.0'}                 | ${0}
      ${'23.2.3'}                     | ${'23.2.3.0'}                 | ${0}
      ${'1.3.42.10133'}               | ${'1.3.42.10133'}             | ${0}
      ${'1.0'}                        | ${'1.0.0.0'}                  | ${0}
      ${'1.23.01'}                    | ${'1.23.1'}                   | ${0}
      ${'1.45.6'}                     | ${'1.45.6.0'}                 | ${0}
      ${'1.45.6-Alpha'}               | ${'1.45.6-Alpha'}             | ${0}
      ${'1.6.2-BeTa'}                 | ${'1.6.02-beta'}              | ${0}
      ${'22.3.07     '}               | ${'22.3.07'}                  | ${0}
      ${'1.0'}                        | ${'1.0.0.0+beta'}             | ${0}
      ${'1.0.0.0+beta.2'}             | ${'1.0.0.0+beta.1'}           | ${0}
      ${'1.0.0'}                      | ${'1.0.0'}                    | ${0}
      ${'1.0.0-BETA'}                 | ${'1.0.0-beta'}               | ${0}
      ${'1.0.0-BETA+AA'}              | ${'1.0.0-beta+aa'}            | ${0}
      ${'1.0.0-BETA.X.y.5.77.0+AA'}   | ${'1.0.0-beta.x.y.5.77.0+aa'} | ${0}
      ${'1.0.0'}                      | ${'1.0.0+beta'}               | ${0}
      ${'1.0'}                        | ${'1.0.0.0'}                  | ${0}
      ${'1.0+test'}                   | ${'1.0.0.0'}                  | ${0}
      ${'1.0.0.1-1.2.A'}              | ${'1.0.0.1-1.2.a+A'}          | ${0}
      ${'1.0.01'}                     | ${'1.0.1.0'}                  | ${0}
      ${'0.0.0'}                      | ${'1.0.0'}                    | ${-1}
      ${'1.1.0'}                      | ${'1.0.0'}                    | ${1}
      ${'1.0.1'}                      | ${'1.0.0'}                    | ${1}
      ${'1.0.0-BETA'}                 | ${'1.0.0-beta2'}              | ${-1}
      ${'1.0.0+AA'}                   | ${'1.0.0-beta+aa'}            | ${1}
      ${'1.0.0-BETA+AA'}              | ${'1.0.0-beta'}               | ${0}
      ${'1.0.0-BETA.X.y.5.77.0+AA'}   | ${'1.0.0-beta.x.y.5.79.0+aa'} | ${-1}
      ${'1.2.3.4-RC+99'}              | ${'1.2.3.4-RC+99'}            | ${0}
      ${'1.2.3'}                      | ${'1.2.3'}                    | ${0}
      ${'1.2.3-Pre.2'}                | ${'1.2.3-Pre.2'}              | ${0}
      ${'1.2.3+99'}                   | ${'1.2.3+99'}                 | ${0}
      ${'1.2-Pre'}                    | ${'1.2.0-Pre'}                | ${0}
      ${'2.4.2'}                      | ${'2.4.1'}                    | ${1}
      ${'2.4-beta'}                   | ${'2.4-alpha'}                | ${1}
      ${'1.9'}                        | ${'2'}                        | ${-1}
      ${'1.9'}                        | ${'1.9.1'}                    | ${-1}
      ${'2.4.0'}                      | ${'2.4.0-beta'}               | ${1}
      ${'2.4.0-alpha'}                | ${'2.4.0'}                    | ${-1}
      ${'1.2.0-beta.333'}             | ${'1.2.0-beta.66'}            | ${1}
      ${'1.2.0-beta2'}                | ${'1.2.0-beta10'}             | ${1}
      ${'1.2.0.1'}                    | ${'1.2.0'}                    | ${1}
      ${'1.2.0.1'}                    | ${'1.2.0.1-beta'}             | ${1}
      ${'1.2.0.1-beta'}               | ${'1.2.0.1'}                  | ${-1}
      ${'1.2.0+1'}                    | ${'1.2.0'}                    | ${0}
      ${'1.2.0'}                      | ${'1.2.0+1'}                  | ${0}
      ${'1-a'}                        | ${'1-0'}                      | ${1}
      ${'1-a.b'}                      | ${'1-a'}                      | ${1}
      ${'1-a'}                        | ${'1-a.b'}                    | ${-1}
      ${'1.0.1'}                      | ${'1.0'}                      | ${1}
      ${'1.231'}                      | ${'1.23'}                     | ${1}
      ${'1.45.6'}                     | ${'1.4.5.6'}                  | ${1}
      ${'1.4.5.60'}                   | ${'1.4.5.6'}                  | ${1}
      ${'1.10'}                       | ${'1.01'}                     | ${1}
      ${'1.10-beta'}                  | ${'1.01-alpha'}               | ${1}
      ${'1.10.0-rc-2'}                | ${'1.01.0-RC-1'}              | ${1}
      ${'1.01'}                       | ${'1.01-RC-1'}                | ${1}
      ${'1.2-preview'}                | ${'1.01'}                     | ${1}
      ${'1.0.0'}                      | ${'0.0.0'}                    | ${1}
      ${'1.1.0'}                      | ${'1.0.0'}                    | ${1}
      ${'1.0.1'}                      | ${'1.0.0'}                    | ${1}
      ${'2.1.1'}                      | ${'1.999.9999'}               | ${1}
      ${'1.0.0-beta2'}                | ${'1.0.0-BETA'}               | ${1}
      ${'1.0.0+aa'}                   | ${'1.0.0-beta+AA'}            | ${1}
      ${'1.0.0-beta.1+AA'}            | ${'1.0.0-BETA'}               | ${1}
      ${'1.0.0-beta.x.y.5.79.0+aa'}   | ${'1.0.0-BETA.X.y.5.77.0+AA'} | ${1}
      ${'1.0.0-beta.x.y.5.790.0+abc'} | ${'1.0.0-BETA.X.y.5.79.0+AA'} | ${1}
    `('compare($x, $y) === $expected', ({ x, y, expected }) => {
      const a = parseVersion(x)!;
      const b = parseVersion(y)!;
      expect(a).toBeTruthy();
      expect(b).toBeTruthy();

      expect(compare(a, b)).toBe(expected);
    });
  });
});
