package rpm

import (
	"os"
	"sort"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"github.com/aquasecurity/trivy/pkg/fanal/types"
)

func TestParseRpmInfo(t *testing.T) {
	var tests = map[string]struct {
		path string
		pkgs types.Packages
	}{
		"Valid": {
			path: "./testdata/valid",
			// cp ./testdata/valid /path/to/testdir/Packages
			// rpm -qa --dbpath /path/to/testdir --queryformat "\{Name: \"%{NAME}\", Epoch: %{RPMTAG_EPOCHNUM}, Version: \"%{VERSION}\", Release: \"%{RELEASE}\", Arch: \"%{ARCH}\", Modularitylabel: \"%{RPMTAG_MODULARITYLABEL}\", Licenses: \[\]string\{\"%{LICENSE}\"\}, Maintainer: \"%{RPMTAG_VENDOR}\", Digest: \"md5:%{SIGMD5}\",\n" | sed "s/(none)//g" > 1.txt
			// rpm -qa --dbpath /path/to/testdir --queryformat "%{SOURCERPM}-%{RPMTAG_EPOCHNUM}\n" | awk -F"-" '{printf("SrcName: \""$0"\", SrcEpoch: "$NF", SrcVersion: \""$(NF-2)"\", SrcRelease: \""$(NF-1)"\"},\n")}' | sed -E 's/-[0-9.]+-.+.src.rpm-[0-9]//' | sed 's/.src.rpm//g' > 2.txt
			// paste -d " " 1.txt 2.txt
			pkgs: []types.Package{
				{Name: "centos-release", Epoch: 0, Version: "7", Release: "1.1503.el7.centos.2.8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:2eadb70af954964e77c9bc548a15d9f9", SrcName: "centos-release", SrcEpoch: 0, SrcVersion: "7", SrcRelease: "1.1503.el7.centos.2.8"},
				{Name: "filesystem", Epoch: 0, Version: "3.2", Release: "18.el7", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:8f701812eb72ce660122437a4ac730cc", SrcName: "filesystem", SrcEpoch: 0, SrcVersion: "3.2", SrcRelease: "18.el7"},
			},
		},
		"ValidBig": {
			path: "./testdata/valid_big",
			// cp ./testdata/valid_big /path/to/testdir/Packages
			// rpm -qa --dbpath /path/to/testdir --queryformat "\{Name: \"%{NAME}\", Epoch: %{RPMTAG_EPOCHNUM}, Version: \"%{VERSION}\", Release: \"%{RELEASE}\", Arch: \"%{ARCH}\", Modularitylabel: \"%{RPMTAG_MODULARITYLABEL}\", Licenses: \[\]string\{\"%{LICENSE}\"\}, Maintainer: \"%{RPMTAG_VENDOR}\", Digest: \"md5:%{SIGMD5}\",\n" | sed "s/(none)//g" > 1.txt
			// rpm -qa --dbpath /path/to/testdir --queryformat "%{SOURCERPM}-%{RPMTAG_EPOCHNUM}\n" | awk -F"-" '{printf("SrcName: \""$0"\", SrcEpoch: "$NF", SrcVersion: \""$(NF-2)"\", SrcRelease: \""$(NF-1)"\"},\n")}' | sed -E 's/-[0-9.]+-.+.src.rpm-[0-9]//' | sed 's/.src.rpm//g' > 2.txt
			// paste -d " " 1.txt 2.txt
			pkgs: []types.Package{
				{Name: "publicsuffix-list-dafsa", Epoch: 0, Version: "20180514", Release: "1.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:13fd895f6eb5b3c2ef15045f2e220777", SrcName: "publicsuffix-list", SrcEpoch: 0, SrcVersion: "20180514", SrcRelease: "1.fc28"},
				{Name: "libreport-filesystem", Epoch: 0, Version: "2.9.5", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:c595e91aebcebbc1c7465dfe13426f0c", SrcName: "libreport", SrcEpoch: 0, SrcVersion: "2.9.5", SrcRelease: "1.fc28"},
				{Name: "fedora-gpg-keys", Epoch: 0, Version: "28", Release: "5", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:4e4cae145b53af62ee15d4cc621b60e9", SrcName: "fedora-repos", SrcEpoch: 0, SrcVersion: "28", SrcRelease: "5"},
				{Name: "fedora-release", Epoch: 0, Version: "28", Release: "2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:3707b27e2ac3d5bcdefcddd9d92e5b53", SrcName: "fedora-release", SrcEpoch: 0, SrcVersion: "28", SrcRelease: "2"},
				{Name: "filesystem", Epoch: 0, Version: "3.8", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:cde59860e2772e9c64eb45b9f0d574d9", SrcName: "filesystem", SrcEpoch: 0, SrcVersion: "3.8", SrcRelease: "2.fc28"},
				{Name: "tzdata", Epoch: 0, Version: "2018e", Release: "1.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:bab8f9f16582030cbafc071512c6a6af", SrcName: "tzdata", SrcEpoch: 0, SrcVersion: "2018e", SrcRelease: "1.fc28"},
				{Name: "pcre2", Epoch: 0, Version: "10.31", Release: "10.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:b1f02e8b3f97058fbdba232a52e28213", SrcName: "pcre2", SrcEpoch: 0, SrcVersion: "10.31", SrcRelease: "10.fc28"},
				{Name: "glibc-minimal-langpack", Epoch: 0, Version: "2.27", Release: "32.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "Fedora Project", Digest: "md5:4e3741ca11df14c5bea06ad296510f59", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.27", SrcRelease: "32.fc28"},
				{Name: "glibc-common", Epoch: 0, Version: "2.27", Release: "32.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "Fedora Project", Digest: "md5:3b6df3648d3469d4132f78745df9fad3", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.27", SrcRelease: "32.fc28"},
				{Name: "bash", Epoch: 0, Version: "4.4.23", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:8849efa6e833effbe09d0fa08ed3c303", SrcName: "bash", SrcEpoch: 0, SrcVersion: "4.4.23", SrcRelease: "1.fc28"},
				{Name: "zlib", Epoch: 0, Version: "1.2.11", Release: "8.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"zlib and Boost"}, Maintainer: "Fedora Project", Digest: "md5:7a2dca79693b19ad8e42c7999cf61552", SrcName: "zlib", SrcEpoch: 0, SrcVersion: "1.2.11", SrcRelease: "8.fc28"},
				{Name: "bzip2-libs", Epoch: 0, Version: "1.0.6", Release: "26.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:1e021b45db1fbade03d37f43131d5624", SrcName: "bzip2", SrcEpoch: 0, SrcVersion: "1.0.6", SrcRelease: "26.fc28"},
				{Name: "libcap", Epoch: 0, Version: "2.25", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "Fedora Project", Digest: "md5:4518daad07126037cf776e303278799d", SrcName: "libcap", SrcEpoch: 0, SrcVersion: "2.25", SrcRelease: "9.fc28"},
				{Name: "libgpg-error", Epoch: 0, Version: "1.31", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:1dcdf8ec34984a2ecb78893eab1751da", SrcName: "libgpg-error", SrcEpoch: 0, SrcVersion: "1.31", SrcRelease: "1.fc28"},
				{Name: "libzstd", Epoch: 0, Version: "1.3.5", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2"}, Maintainer: "Fedora Project", Digest: "md5:2d6a9322d62140d6bf8d2eac4228ea4b", SrcName: "zstd", SrcEpoch: 0, SrcVersion: "1.3.5", SrcRelease: "1.fc28"},
				{Name: "expat", Epoch: 0, Version: "2.2.5", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:9fde9e0f69fd49801da2a6884c1a5f60", SrcName: "expat", SrcEpoch: 0, SrcVersion: "2.2.5", SrcRelease: "3.fc28"},
				{Name: "nss-util", Epoch: 0, Version: "3.38.0", Release: "1.0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:8afbc3c6148a05e1ccc5ea7073adc377", SrcName: "nss-util", SrcEpoch: 0, SrcVersion: "3.38.0", SrcRelease: "1.0.fc28"},
				{Name: "libcom_err", Epoch: 0, Version: "1.44.2", Release: "0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:adf1fd36edbe4f678a5bb6eb9829d1a3", SrcName: "e2fsprogs", SrcEpoch: 0, SrcVersion: "1.44.2", SrcRelease: "0.fc28"},
				{Name: "libffi", Epoch: 0, Version: "3.1", Release: "16.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:a360a17e7547b6661ab7bdb00fbef011", SrcName: "libffi", SrcEpoch: 0, SrcVersion: "3.1", SrcRelease: "16.fc28"},
				{Name: "libgcrypt", Epoch: 0, Version: "1.8.3", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:538afec59f776a8b7c6ecac251c7a592", SrcName: "libgcrypt", SrcEpoch: 0, SrcVersion: "1.8.3", SrcRelease: "1.fc28"},
				{Name: "libxml2", Epoch: 0, Version: "2.9.8", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:87aff1711a3a002493efb048bcd82433", SrcName: "libxml2", SrcEpoch: 0, SrcVersion: "2.9.8", SrcRelease: "4.fc28"},
				{Name: "libacl", Epoch: 0, Version: "2.2.53", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:948505b4f6b7873e070340bef6f37aac", SrcName: "acl", SrcEpoch: 0, SrcVersion: "2.2.53", SrcRelease: "1.fc28"},
				{Name: "sed", Epoch: 0, Version: "4.5", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:62f9dfcf1e032617111238569a4cb549", SrcName: "sed", SrcEpoch: 0, SrcVersion: "4.5", SrcRelease: "1.fc28"},
				{Name: "libmount", Epoch: 0, Version: "2.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:df960e8fb1a5eaebec58b0ec17e097a9", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "1.fc28"},
				{Name: "p11-kit", Epoch: 0, Version: "0.23.12", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:3fb0875e409f4530a132289ccf4bc5fb", SrcName: "p11-kit", SrcEpoch: 0, SrcVersion: "0.23.12", SrcRelease: "1.fc28"},
				{Name: "libidn2", Epoch: 0, Version: "2.0.5", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or LGPLv3+) and GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:46c6bd7103d10e4ba23458f61c56d9ba", SrcName: "libidn2", SrcEpoch: 0, SrcVersion: "2.0.5", SrcRelease: "1.fc28"},
				{Name: "libcap-ng", Epoch: 0, Version: "0.7.9", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:ec88100d3579fcdcfbf85095a85b6b9a", SrcName: "libcap-ng", SrcEpoch: 0, SrcVersion: "0.7.9", SrcRelease: "4.fc28"},
				{Name: "lz4-libs", Epoch: 0, Version: "1.8.1.2", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and BSD"}, Maintainer: "Fedora Project", Digest: "md5:4dccfb6692b54bd995455aba81c6301c", SrcName: "lz4", SrcEpoch: 0, SrcVersion: "1.8.1.2", SrcRelease: "4.fc28"},
				{Name: "libassuan", Epoch: 0, Version: "2.5.1", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:cb3ad31e3bdb7c4a6f6486d45ca0a6b1", SrcName: "libassuan", SrcEpoch: 0, SrcVersion: "2.5.1", SrcRelease: "3.fc28"},
				{Name: "keyutils-libs", Epoch: 0, Version: "1.5.10", Release: "6.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:8582115f8482e34565d78b8c054fdeaf", SrcName: "keyutils", SrcEpoch: 0, SrcVersion: "1.5.10", SrcRelease: "6.fc28"},
				{Name: "glib2", Epoch: 0, Version: "2.56.1", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:60a4da3abc6b4b9e85f517e435d8d6c9", SrcName: "glib2", SrcEpoch: 0, SrcVersion: "2.56.1", SrcRelease: "4.fc28"},
				{Name: "systemd-libs", Epoch: 0, Version: "238", Release: "9.git0e0aa59.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT"}, Maintainer: "Fedora Project", Digest: "md5:cedf8a4a17a04eb646eaab68995be995", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "238", SrcRelease: "9.git0e0aa59.fc28"},
				{Name: "dbus-libs", Epoch: 1, Version: "1.12.10", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:7cc0892e9871d2a97a688391c8178560", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.10", SrcRelease: "1.fc28"},
				{Name: "libtasn1", Epoch: 0, Version: "4.13", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:04014d6ea889d9f818b11950bc2c123e", SrcName: "libtasn1", SrcEpoch: 0, SrcVersion: "4.13", SrcRelease: "2.fc28"},
				{Name: "ca-certificates", Epoch: 0, Version: "2018.2.24", Release: "1.0.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:ea5c49c0c8a319c94f260ee07c23b1cf", SrcName: "ca-certificates", SrcEpoch: 0, SrcVersion: "2018.2.24", SrcRelease: "1.0.fc28"},
				{Name: "libarchive", Epoch: 0, Version: "3.3.1", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:ef366c4a5a532e07297569fe57520648", SrcName: "libarchive", SrcEpoch: 0, SrcVersion: "3.3.1", SrcRelease: "4.fc28"},
				{Name: "openssl", Epoch: 1, Version: "1.1.0h", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenSSL"}, Maintainer: "Fedora Project", Digest: "md5:26a9553c1706ede95925f32215ba9a60", SrcName: "openssl", SrcEpoch: 1, SrcVersion: "1.1.0h", SrcRelease: "3.fc28"},
				{Name: "libusbx", Epoch: 0, Version: "1.0.22", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:0a5650b09182d2b0134ee832acd5432e", SrcName: "libusbx", SrcEpoch: 0, SrcVersion: "1.0.22", SrcRelease: "1.fc28"},
				{Name: "libsemanage", Epoch: 0, Version: "2.8", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:5c29e3e9ed089e3fedf7298dabe503fd", SrcName: "libsemanage", SrcEpoch: 0, SrcVersion: "2.8", SrcRelease: "2.fc28"},
				{Name: "libutempter", Epoch: 0, Version: "1.1.6", Release: "14.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:ccfd737faad3d1be60f4c21839f5a9af", SrcName: "libutempter", SrcEpoch: 0, SrcVersion: "1.1.6", SrcRelease: "14.fc28"},
				{Name: "mpfr", Epoch: 0, Version: "3.1.6", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ and GPLv3+ and GFDL"}, Maintainer: "Fedora Project", Digest: "md5:a1339297aae3e31e8e1b1de00f7930", SrcName: "mpfr", SrcEpoch: 0, SrcVersion: "3.1.6", SrcRelease: "1.fc28"},
				{Name: "gnutls", Epoch: 0, Version: "3.6.3", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:6c813ffeecd51d2298766d86859395ca", SrcName: "gnutls", SrcEpoch: 0, SrcVersion: "3.6.3", SrcRelease: "4.fc28"},
				{Name: "gzip", Epoch: 0, Version: "1.9", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GFDL"}, Maintainer: "Fedora Project", Digest: "md5:efc5209278a36c1b62ee770a95e59929", SrcName: "gzip", SrcEpoch: 0, SrcVersion: "1.9", SrcRelease: "3.fc28"},
				{Name: "acl", Epoch: 0, Version: "2.2.53", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:9f72a1f855b12dc038e3f4921c50db92", SrcName: "acl", SrcEpoch: 0, SrcVersion: "2.2.53", SrcRelease: "1.fc28"},
				{Name: "nss-softokn-freebl", Epoch: 0, Version: "3.38.0", Release: "1.0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:e6eb1777436169f45799122ce5f9b427", SrcName: "nss-softokn", SrcEpoch: 0, SrcVersion: "3.38.0", SrcRelease: "1.0.fc28"},
				{Name: "nss", Epoch: 0, Version: "3.38.0", Release: "1.0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:bdd5b0f15a3ffce682979a9379d119d1", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.38.0", SrcRelease: "1.0.fc28"},
				{Name: "libmetalink", Epoch: 0, Version: "0.1.3", Release: "6.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:0b24a2205fff4c591e60c85dbd4dd67a", SrcName: "libmetalink", SrcEpoch: 0, SrcVersion: "0.1.3", SrcRelease: "6.fc28"},
				{Name: "libdb-utils", Epoch: 0, Version: "5.3.28", Release: "30.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2 and Sleepycat"}, Maintainer: "Fedora Project", Digest: "md5:8882fd2692a31eab11c2c1b35ff3717b", SrcName: "libdb", SrcEpoch: 0, SrcVersion: "5.3.28", SrcRelease: "30.fc28"},
				{Name: "file-libs", Epoch: 0, Version: "5.33", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:a946dfa89d79a10ff01b069bc1435dca", SrcName: "file", SrcEpoch: 0, SrcVersion: "5.33", SrcRelease: "7.fc28"},
				{Name: "libsss_idmap", Epoch: 0, Version: "1.16.3", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:ce3bea5356bcd30e4719571136134e71", SrcName: "sssd", SrcEpoch: 0, SrcVersion: "1.16.3", SrcRelease: "2.fc28"},
				{Name: "libsigsegv", Epoch: 0, Version: "2.11", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:b6b8636eb9487e009c872b86eb402657", SrcName: "libsigsegv", SrcEpoch: 0, SrcVersion: "2.11", SrcRelease: "5.fc28"},
				{Name: "krb5-libs", Epoch: 0, Version: "1.16.1", Release: "13.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:254a145d149c83917f97cb9545a68d9f", SrcName: "krb5", SrcEpoch: 0, SrcVersion: "1.16.1", SrcRelease: "13.fc28"},
				{Name: "libnsl2", Epoch: 0, Version: "1.2.0", Release: "2.20180605git4a062cf.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:971bc2ad15942d7f575c6ff7940d9c88", SrcName: "libnsl2", SrcEpoch: 0, SrcVersion: "1.2.0", SrcRelease: "2.20180605git4a062cf.fc28"},
				{Name: "python3-pip", Epoch: 0, Version: "9.0.3", Release: "2.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:bd24b8294b63337092d3b6f7bca57f1c", SrcName: "python-pip", SrcEpoch: 0, SrcVersion: "9.0.3", SrcRelease: "2.fc28"},
				{Name: "python3", Epoch: 0, Version: "3.6.6", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Python"}, Maintainer: "Fedora Project", Digest: "md5:a6407bed66cf643b299233c536265ffc", SrcName: "python3", SrcEpoch: 0, SrcVersion: "3.6.6", SrcRelease: "1.fc28"},
				{Name: "pam", Epoch: 0, Version: "1.3.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:e7ff450a56bf9534cbc9f2681964e49f", SrcName: "pam", SrcEpoch: 0, SrcVersion: "1.3.1", SrcRelease: "1.fc28"},
				{Name: "python3-gobject-base", Epoch: 0, Version: "3.28.3", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT"}, Maintainer: "Fedora Project", Digest: "md5:ddde5ae6a33bc0e7a3228e768e46c0ac", SrcName: "pygobject3", SrcEpoch: 0, SrcVersion: "3.28.3", SrcRelease: "1.fc28"},
				{Name: "python3-smartcols", Epoch: 0, Version: "0.3.0", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:06a8b79d535a16ca0dd000f1c56731e7", SrcName: "python-smartcols", SrcEpoch: 0, SrcVersion: "0.3.0", SrcRelease: "2.fc28"},
				{Name: "python3-iniparse", Epoch: 0, Version: "0.4", Release: "30.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT and Python"}, Maintainer: "Fedora Project", Digest: "md5:32cce6047f9ec5ae4e993193e3f6bee1", SrcName: "python-iniparse", SrcEpoch: 0, SrcVersion: "0.4", SrcRelease: "30.fc28"},
				{Name: "openldap", Epoch: 0, Version: "2.4.46", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenLDAP"}, Maintainer: "Fedora Project", Digest: "md5:c98aaadc1c628656c7f90a18759da517", SrcName: "openldap", SrcEpoch: 0, SrcVersion: "2.4.46", SrcRelease: "3.fc28"},
				{Name: "libseccomp", Epoch: 0, Version: "2.3.3", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2"}, Maintainer: "Fedora Project", Digest: "md5:26c6a917063d5654083974de6734112d", SrcName: "libseccomp", SrcEpoch: 0, SrcVersion: "2.3.3", SrcRelease: "2.fc28"},
				{Name: "npth", Epoch: 0, Version: "1.5", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:026afa87b50e9ec9812501188b6fd998", SrcName: "npth", SrcEpoch: 0, SrcVersion: "1.5", SrcRelease: "4.fc28"},
				{Name: "gpgme", Epoch: 0, Version: "1.10.0", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:5a044754d91ca7d0b4e4428b052de6", SrcName: "gpgme", SrcEpoch: 0, SrcVersion: "1.10.0", SrcRelease: "4.fc28"},
				{Name: "json-c", Epoch: 0, Version: "0.13.1", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:2d0c0ea344c78a2faeab869c9474f9", SrcName: "json-c", SrcEpoch: 0, SrcVersion: "0.13.1", SrcRelease: "2.fc28"},
				{Name: "libyaml", Epoch: 0, Version: "0.1.7", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:4e0f94eba782aa090fe6487142772aba", SrcName: "libyaml", SrcEpoch: 0, SrcVersion: "0.1.7", SrcRelease: "5.fc28"},
				{Name: "libpkgconf", Epoch: 0, Version: "1.4.2", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "Fedora Project", Digest: "md5:5721f7af98fca6a80db3d51fe0ad7312", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.fc28"},
				{Name: "pkgconf-pkg-config", Epoch: 0, Version: "1.4.2", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "Fedora Project", Digest: "md5:f297678d95f451aa7132d15aff3246a7", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.fc28"},
				{Name: "iptables-libs", Epoch: 0, Version: "1.6.2", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and Artistic Licence 2.0 and ISC"}, Maintainer: "Fedora Project", Digest: "md5:436c3b926769e66ee6c471295ec03069", SrcName: "iptables", SrcEpoch: 0, SrcVersion: "1.6.2", SrcRelease: "3.fc28"},
				{Name: "device-mapper-libs", Epoch: 0, Version: "1.02.146", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2"}, Maintainer: "Fedora Project", Digest: "md5:04d7f026ae3222f91454908a33a9b030", SrcName: "lvm2", SrcEpoch: 0, SrcVersion: "2.02.177", SrcRelease: "5.fc28"},
				{Name: "systemd-pam", Epoch: 0, Version: "238", Release: "9.git0e0aa59.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT and GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:eb61bc3dd551e9096261751d61f59599", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "238", SrcRelease: "9.git0e0aa59.fc28"},
				{Name: "systemd", Epoch: 0, Version: "238", Release: "9.git0e0aa59.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT and GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:5cf717691cbe7965cb9b66007d8cb6af", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "238", SrcRelease: "9.git0e0aa59.fc28"},
				{Name: "elfutils-default-yama-scope", Epoch: 0, Version: "0.173", Release: "1.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:df83018bdb1dbe9ff48dcecb8a0f6696", SrcName: "elfutils", SrcEpoch: 0, SrcVersion: "0.173", SrcRelease: "1.fc28"},
				{Name: "libcurl", Epoch: 0, Version: "7.59.0", Release: "6.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:bc6fada59774858c98511c28a1aad381", SrcName: "curl", SrcEpoch: 0, SrcVersion: "7.59.0", SrcRelease: "6.fc28"},
				{Name: "python3-librepo", Epoch: 0, Version: "1.8.1", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:e67d38dde04912162900ae0cd30ba871", SrcName: "librepo", SrcEpoch: 0, SrcVersion: "1.8.1", SrcRelease: "7.fc28"},
				{Name: "rpm-plugin-selinux", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:cc86e6bf2d8bf448550c8b65278006d9", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "rpm", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:a4038012a8f482c1cbcb27af42fbca06", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "libdnf", Epoch: 0, Version: "0.11.1", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:07f592ead4364677e6d1a0d7a3c55b82", SrcName: "libdnf", SrcEpoch: 0, SrcVersion: "0.11.1", SrcRelease: "3.fc28"},
				{Name: "rpm-build-libs", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+ with exceptions"}, Maintainer: "Fedora Project", Digest: "md5:e0a88c9a5e2d7e63af38eb5376aeeff8", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "python3-rpm", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:147e27b17f631923e5518c340a425909", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "dnf", Epoch: 0, Version: "2.7.5", Release: "12.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "Fedora Project", Digest: "md5:bd88715ed29242ad4306a98df0ad531a", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "2.7.5", SrcRelease: "12.fc28"},
				{Name: "deltarpm", Epoch: 0, Version: "3.6", Release: "25.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:e7e9ed52b29cb90390af12f7854171af", SrcName: "deltarpm", SrcEpoch: 0, SrcVersion: "3.6", SrcRelease: "25.fc28"},
				{Name: "sssd-client", Epoch: 0, Version: "1.16.3", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:65132a690812808d55d37f060d4e46c0", SrcName: "sssd", SrcEpoch: 0, SrcVersion: "1.16.3", SrcRelease: "2.fc28"},
				{Name: "cracklib-dicts", Epoch: 0, Version: "2.9.6", Release: "13.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:05f6ece3e3c711fc56b860f4489c829d", SrcName: "cracklib", SrcEpoch: 0, SrcVersion: "2.9.6", SrcRelease: "13.fc28"},
				{Name: "tar", Epoch: 2, Version: "1.30", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:ea2a310b216d89abc0cab39cc2d1ac77", SrcName: "tar", SrcEpoch: 2, SrcVersion: "1.30", SrcRelease: "3.fc28"},
				{Name: "diffutils", Epoch: 0, Version: "3.6", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:ed08723b1610b914fd87ec36271fffc5", SrcName: "diffutils", SrcEpoch: 0, SrcVersion: "3.6", SrcRelease: "4.fc28"},
				{Name: "langpacks-en", Epoch: 0, Version: "1.0", Release: "12.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:926a126cefbdaad1bf20a46984fd5891", SrcName: "langpacks", SrcEpoch: 0, SrcVersion: "1.0", SrcRelease: "12.fc28"},
				{Name: "gpg-pubkey", Epoch: 0, Version: "9db62fb1", Release: "59920156", Arch: "None", Modularitylabel: "", Licenses: []string{"pubkey"}, Maintainer: "", Digest: "", SrcName: "", SrcEpoch: 0, SrcVersion: "", SrcRelease: ""},
				{Name: "libgcc", Epoch: 0, Version: "8.1.1", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD"}, Maintainer: "Fedora Project", Digest: "md5:84352a20bcd304bd78f817bfbe7c13c1", SrcName: "gcc", SrcEpoch: 0, SrcVersion: "8.1.1", SrcRelease: "5.fc28"},
				{Name: "pkgconf-m4", Epoch: 0, Version: "1.4.2", Release: "1.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ with exceptions"}, Maintainer: "Fedora Project", Digest: "md5:c271acc9d36e787c084c50308c3fc06c", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.fc28"},
				{Name: "dnf-conf", Epoch: 0, Version: "2.7.5", Release: "12.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "Fedora Project", Digest: "md5:50220d930013ef33381d349cd964bcd4", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "2.7.5", SrcRelease: "12.fc28"},
				{Name: "fedora-repos", Epoch: 0, Version: "28", Release: "5", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:c3581989c0096ce0099764d4aa60ef8c", SrcName: "fedora-repos", SrcEpoch: 0, SrcVersion: "28", SrcRelease: "5"},
				{Name: "setup", Epoch: 0, Version: "2.11.4", Release: "1.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:51d1c9b034bde4b27cb105cc09efad78", SrcName: "setup", SrcEpoch: 0, SrcVersion: "2.11.4", SrcRelease: "1.fc28"},
				{Name: "basesystem", Epoch: 0, Version: "11", Release: "5.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:9a4682815de24f51b21232b086db3535", SrcName: "basesystem", SrcEpoch: 0, SrcVersion: "11", SrcRelease: "5.fc28"},
				{Name: "ncurses-base", Epoch: 0, Version: "6.1", Release: "5.20180224.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:7f2aaf7720a07969215fe1a23a090179", SrcName: "ncurses", SrcEpoch: 0, SrcVersion: "6.1", SrcRelease: "5.20180224.fc28"},
				{Name: "libselinux", Epoch: 0, Version: "2.8", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:923f92e5c4bba85713a5a3dfd06656e1", SrcName: "libselinux", SrcEpoch: 0, SrcVersion: "2.8", SrcRelease: "1.fc28"},
				{Name: "ncurses-libs", Epoch: 0, Version: "6.1", Release: "5.20180224.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:e0e0590d378c6a92dc54bbf0e6e22aeb", SrcName: "ncurses", SrcEpoch: 0, SrcVersion: "6.1", SrcRelease: "5.20180224.fc28"},
				{Name: "glibc", Epoch: 0, Version: "2.27", Release: "32.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "Fedora Project", Digest: "md5:826d5c9c240f4f4caa40e2856bdd2bf6", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.27", SrcRelease: "32.fc28"},
				{Name: "libsepol", Epoch: 0, Version: "2.8", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:396caa3e34a4895ab9a399f3d6a6bf4c", SrcName: "libsepol", SrcEpoch: 0, SrcVersion: "2.8", SrcRelease: "1.fc28"},
				{Name: "xz-libs", Epoch: 0, Version: "5.2.4", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:f02af3d27b5312ef030e10ee3f1dc762", SrcName: "xz", SrcEpoch: 0, SrcVersion: "5.2.4", SrcRelease: "2.fc28"},
				{Name: "info", Epoch: 0, Version: "6.5", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:6146023e357c6201b0e1765b9dd988a2", SrcName: "texinfo", SrcEpoch: 0, SrcVersion: "6.5", SrcRelease: "4.fc28"},
				{Name: "libdb", Epoch: 0, Version: "5.3.28", Release: "30.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2 and Sleepycat"}, Maintainer: "Fedora Project", Digest: "md5:8c1ce08d80b2ae50f8d9929a56d3f2f1", SrcName: "libdb", SrcEpoch: 0, SrcVersion: "5.3.28", SrcRelease: "30.fc28"},
				{Name: "elfutils-libelf", Epoch: 0, Version: "0.173", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:d34e7ba3ed7f6d6eded91362840326a6", SrcName: "elfutils", SrcEpoch: 0, SrcVersion: "0.173", SrcRelease: "1.fc28"},
				{Name: "popt", Epoch: 0, Version: "1.16", Release: "14.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:5a7b62c8ec9969e76d0bb03fa2de3c8c", SrcName: "popt", SrcEpoch: 0, SrcVersion: "1.16", SrcRelease: "14.fc28"},
				{Name: "nspr", Epoch: 0, Version: "4.19.0", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:531a0b52f4de1039af8c86bb65c34391", SrcName: "nspr", SrcEpoch: 0, SrcVersion: "4.19.0", SrcRelease: "1.fc28"},
				{Name: "libxcrypt", Epoch: 0, Version: "4.1.2", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and BSD and Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:fdc13ebcc1832ddd86d6935e8cb55c2d", SrcName: "libxcrypt", SrcEpoch: 0, SrcVersion: "4.1.2", SrcRelease: "1.fc28"},
				{Name: "lua-libs", Epoch: 0, Version: "5.3.4", Release: "10.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:f0da7c13dd005e9e6082c2064b0a36cd", SrcName: "lua", SrcEpoch: 0, SrcVersion: "5.3.4", SrcRelease: "10.fc28"},
				{Name: "libuuid", Epoch: 0, Version: "2.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:a76da5a5aa56cf31d66581ff43a2629a", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "1.fc28"},
				{Name: "readline", Epoch: 0, Version: "7.0", Release: "11.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:07a436e6d82014e8bdb5a14376957e75", SrcName: "readline", SrcEpoch: 0, SrcVersion: "7.0", SrcRelease: "11.fc28"},
				{Name: "libattr", Epoch: 0, Version: "2.4.48", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:d6f4bf85cf8a1d33a4fe6fbb5ed5db13", SrcName: "attr", SrcEpoch: 0, SrcVersion: "2.4.48", SrcRelease: "3.fc28"},
				{Name: "coreutils-single", Epoch: 0, Version: "8.29", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:9f532cc8c9a619d35d34bb503c4a15b2", SrcName: "coreutils", SrcEpoch: 0, SrcVersion: "8.29", SrcRelease: "7.fc28"},
				{Name: "libblkid", Epoch: 0, Version: "2.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:dce635c80efb14ac8541acb93d2b3b49", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "1.fc28"},
				{Name: "gmp", Epoch: 1, Version: "6.1.2", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ or GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:602014b65fab8b889447bd3bf2791af0", SrcName: "gmp", SrcEpoch: 1, SrcVersion: "6.1.2", SrcRelease: "7.fc28"},
				{Name: "libunistring", Epoch: 0, Version: "0.9.10", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLV2+ or LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:e1b8cbbd9032775feddc924209176b6d", SrcName: "libunistring", SrcEpoch: 0, SrcVersion: "0.9.10", SrcRelease: "1.fc28"},
				{Name: "sqlite-libs", Epoch: 0, Version: "3.22.0", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:c8bcf9c9600dba85b9b6e21615b9ef9f", SrcName: "sqlite", SrcEpoch: 0, SrcVersion: "3.22.0", SrcRelease: "4.fc28"},
				{Name: "audit-libs", Epoch: 0, Version: "2.8.4", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:91627d355f76a4e490a6a327480bff60", SrcName: "audit", SrcEpoch: 0, SrcVersion: "2.8.4", SrcRelease: "2.fc28"},
				{Name: "chkconfig", Epoch: 0, Version: "1.10", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "Fedora Project", Digest: "md5:f19b3a5e681b99fe7a274bffe69dea84", SrcName: "chkconfig", SrcEpoch: 0, SrcVersion: "1.10", SrcRelease: "4.fc28"},
				{Name: "libsmartcols", Epoch: 0, Version: "2.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:079826f5b1d991b34f3673580a940635", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "1.fc28"},
				{Name: "pcre", Epoch: 0, Version: "8.42", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:35a6932b995f80a724b4b24baa6fe8e7", SrcName: "pcre", SrcEpoch: 0, SrcVersion: "8.42", SrcRelease: "3.fc28"},
				{Name: "grep", Epoch: 0, Version: "3.1", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:69fffa003ecf55cbae2d265a1db73b9d", SrcName: "grep", SrcEpoch: 0, SrcVersion: "3.1", SrcRelease: "5.fc28"},
				{Name: "crypto-policies", Epoch: 0, Version: "20180425", Release: "5.git6ad4018.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:7b5456e60b3904649ef88ed6d6eb0847", SrcName: "crypto-policies", SrcEpoch: 0, SrcVersion: "20180425", SrcRelease: "5.git6ad4018.fc28"},
				{Name: "gdbm-libs", Epoch: 1, Version: "1.14.1", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:8d02dd64e50a570e0435e690974ed87c", SrcName: "gdbm", SrcEpoch: 1, SrcVersion: "1.14.1", SrcRelease: "4.fc28"},
				{Name: "p11-kit-trust", Epoch: 0, Version: "0.23.12", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:1551c99a0806b5f020087330fe5e195c", SrcName: "p11-kit", SrcEpoch: 0, SrcVersion: "0.23.12", SrcRelease: "1.fc28"},
				{Name: "openssl-libs", Epoch: 1, Version: "1.1.0h", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenSSL"}, Maintainer: "Fedora Project", Digest: "md5:8c1c855c5ce8faf1c068fc2e04456dc2", SrcName: "openssl", SrcEpoch: 1, SrcVersion: "1.1.0h", SrcRelease: "3.fc28"},
				{Name: "ima-evm-utils", Epoch: 0, Version: "1.1", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "Fedora Project", Digest: "md5:f0f17fdf3829aa5a371d58b113251124", SrcName: "ima-evm-utils", SrcEpoch: 0, SrcVersion: "1.1", SrcRelease: "2.fc28"},
				{Name: "gdbm", Epoch: 1, Version: "1.14.1", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:5c8164ecf3c03d355dc5547e947ad887", SrcName: "gdbm", SrcEpoch: 1, SrcVersion: "1.14.1", SrcRelease: "4.fc28"},
				{Name: "gobject-introspection", Epoch: 0, Version: "1.56.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+, LGPLv2+, MIT"}, Maintainer: "Fedora Project", Digest: "md5:33f5f6a58146988df94221944d02daa7", SrcName: "gobject-introspection", SrcEpoch: 0, SrcVersion: "1.56.1", SrcRelease: "1.fc28"},
				{Name: "shadow-utils", Epoch: 2, Version: "4.6", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:da5727d6020eca44eecf0dc2ada776b6", SrcName: "shadow-utils", SrcEpoch: 2, SrcVersion: "4.6", SrcRelease: "1.fc28"},
				{Name: "libpsl", Epoch: 0, Version: "0.20.2", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:9f6915752f645003359b143c09d109de", SrcName: "libpsl", SrcEpoch: 0, SrcVersion: "0.20.2", SrcRelease: "2.fc28"},
				{Name: "nettle", Epoch: 0, Version: "3.4", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ or GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:de21f7ba2b85723f9e410dcfc3503c2d", SrcName: "nettle", SrcEpoch: 0, SrcVersion: "3.4", SrcRelease: "2.fc28"},
				{Name: "libfdisk", Epoch: 0, Version: "2.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:f262682df2eae5a78157f3ba4fe2486b", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "1.fc28"},
				{Name: "cracklib", Epoch: 0, Version: "2.9.6", Release: "13.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:6ef0f1b104fd8644707036f80c3929c2", SrcName: "cracklib", SrcEpoch: 0, SrcVersion: "2.9.6", SrcRelease: "13.fc28"},
				{Name: "libcomps", Epoch: 0, Version: "0.1.8", Release: "11.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:9cbbf1a738742b0c4f7f831ab6a97f80", SrcName: "libcomps", SrcEpoch: 0, SrcVersion: "0.1.8", SrcRelease: "11.fc28"},
				{Name: "nss-softokn", Epoch: 0, Version: "3.38.0", Release: "1.0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:5fe60496f97ce6327d779173f943bfb0", SrcName: "nss-softokn", SrcEpoch: 0, SrcVersion: "3.38.0", SrcRelease: "1.0.fc28"},
				{Name: "nss-sysinit", Epoch: 0, Version: "3.38.0", Release: "1.0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:56d7739e0735f26c771d9a31ee0b6402", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.38.0", SrcRelease: "1.0.fc28"},
				{Name: "libksba", Epoch: 0, Version: "1.3.5", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(LGPLv3+ or GPLv2+) and GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:64f47804b68844f3eaadb64625ecccfc", SrcName: "libksba", SrcEpoch: 0, SrcVersion: "1.3.5", SrcRelease: "7.fc28"},
				{Name: "kmod-libs", Epoch: 0, Version: "25", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:fa3e05ce583ca814c8523493164d2dd7", SrcName: "kmod", SrcEpoch: 0, SrcVersion: "25", SrcRelease: "2.fc28"},
				{Name: "libsss_nss_idmap", Epoch: 0, Version: "1.16.3", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:4f0a858d8d28c2192ba4ef1f71643edd", SrcName: "sssd", SrcEpoch: 0, SrcVersion: "1.16.3", SrcRelease: "2.fc28"},
				{Name: "libverto", Epoch: 0, Version: "0.3.0", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:36eca4dbcb08f145085a1f277ad2f2d3", SrcName: "libverto", SrcEpoch: 0, SrcVersion: "0.3.0", SrcRelease: "5.fc28"},
				{Name: "gawk", Epoch: 0, Version: "4.2.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv2+ and LGPLv2+ and BSD"}, Maintainer: "Fedora Project", Digest: "md5:c76ff7273f0569d10fe934f134f9f380", SrcName: "gawk", SrcEpoch: 0, SrcVersion: "4.2.1", SrcRelease: "1.fc28"},
				{Name: "libtirpc", Epoch: 0, Version: "1.0.3", Release: "3.rc2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"SISSL and BSD"}, Maintainer: "Fedora Project", Digest: "md5:3622fd09756d62e568cd8ef6f758da5b", SrcName: "libtirpc", SrcEpoch: 0, SrcVersion: "1.0.3", SrcRelease: "3.rc2.fc28"},
				{Name: "python3-libs", Epoch: 0, Version: "3.6.6", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Python"}, Maintainer: "Fedora Project", Digest: "md5:2df2a10f40a769626c393ee8036890b6", SrcName: "python3", SrcEpoch: 0, SrcVersion: "3.6.6", SrcRelease: "1.fc28"},
				{Name: "python3-setuptools", Epoch: 0, Version: "39.2.0", Release: "6.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:de97dffcc340039f09d416095226d667", SrcName: "python-setuptools", SrcEpoch: 0, SrcVersion: "39.2.0", SrcRelease: "6.fc28"},
				{Name: "libpwquality", Epoch: 0, Version: "1.4.0", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD or GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:07c137a1fde1d839050fab8d9571a339", SrcName: "libpwquality", SrcEpoch: 0, SrcVersion: "1.4.0", SrcRelease: "7.fc28"},
				{Name: "util-linux", Epoch: 0, Version: "2.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:6df6e3111307017cdc0d3be66a9ad3d2", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "1.fc28"},
				{Name: "python3-libcomps", Epoch: 0, Version: "0.1.8", Release: "11.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:015886962d2f403e28c54116787de72a", SrcName: "libcomps", SrcEpoch: 0, SrcVersion: "0.1.8", SrcRelease: "11.fc28"},
				{Name: "python3-six", Epoch: 0, Version: "1.11.0", Release: "3.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:b3e94962b65a8d8073b964d5be7975bb", SrcName: "python-six", SrcEpoch: 0, SrcVersion: "1.11.0", SrcRelease: "3.fc28"},
				{Name: "cyrus-sasl-lib", Epoch: 0, Version: "2.1.27", Release: "0.2rc7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD with advertising"}, Maintainer: "Fedora Project", Digest: "md5:b19a2d49c32921b418b3d33384daafb8", SrcName: "cyrus-sasl", SrcEpoch: 0, SrcVersion: "2.1.27", SrcRelease: "0.2rc7.fc28"},
				{Name: "libssh", Epoch: 0, Version: "0.8.2", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:fccdd0ffe60f86721e8bd4e8eac22562", SrcName: "libssh", SrcEpoch: 0, SrcVersion: "0.8.2", SrcRelease: "1.fc28"},
				{Name: "qrencode-libs", Epoch: 0, Version: "3.4.4", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:d3ff72e3c3ef756417edb9eae5418635", SrcName: "qrencode", SrcEpoch: 0, SrcVersion: "3.4.4", SrcRelease: "5.fc28"},
				{Name: "gnupg2", Epoch: 0, Version: "2.2.8", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:54cd97bcdc9bdda5bb70302f0a15c426", SrcName: "gnupg2", SrcEpoch: 0, SrcVersion: "2.2.8", SrcRelease: "1.fc28"},
				{Name: "python3-gpg", Epoch: 0, Version: "1.10.0", Release: "4.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:a94041676e57089ef07d088cd75a0fc9", SrcName: "gpgme", SrcEpoch: 0, SrcVersion: "1.10.0", SrcRelease: "4.fc28"},
				{Name: "libargon2", Epoch: 0, Version: "20161029", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain or ASL 2.0"}, Maintainer: "Fedora Project", Digest: "md5:8af9f36b09b9685e6ba5f6241c01739e", SrcName: "argon2", SrcEpoch: 0, SrcVersion: "20161029", SrcRelease: "5.fc28"},
				{Name: "libmodulemd", Epoch: 0, Version: "1.6.2", Release: "2.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:835bc0b681c5104b26cfad5fc22a20b5", SrcName: "libmodulemd", SrcEpoch: 0, SrcVersion: "1.6.2", SrcRelease: "2.fc28"},
				{Name: "pkgconf", Epoch: 0, Version: "1.4.2", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "Fedora Project", Digest: "md5:b47f825995f3846e0fb24dc9d477489f", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.fc28"},
				{Name: "libpcap", Epoch: 14, Version: "1.9.0", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD with advertising"}, Maintainer: "Fedora Project", Digest: "md5:8a640dd6fcceecbfbe3ca5ae964943be", SrcName: "libpcap", SrcEpoch: 14, SrcVersion: "1.9.0", SrcRelease: "1.fc28"},
				{Name: "device-mapper", Epoch: 0, Version: "1.02.146", Release: "5.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "Fedora Project", Digest: "md5:cd04b9d971aab244a99600118b021f5e", SrcName: "lvm2", SrcEpoch: 0, SrcVersion: "2.02.177", SrcRelease: "5.fc28"},
				{Name: "cryptsetup-libs", Epoch: 0, Version: "2.0.4", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:df3fcc57a6a4b9776be8feeb41e2426a", SrcName: "cryptsetup", SrcEpoch: 0, SrcVersion: "2.0.4", SrcRelease: "1.fc28"},
				{Name: "elfutils-libs", Epoch: 0, Version: "0.173", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "Fedora Project", Digest: "md5:bc22d891929666e57b0572afec13fd1b", SrcName: "elfutils", SrcEpoch: 0, SrcVersion: "0.173", SrcRelease: "1.fc28"},
				{Name: "dbus", Epoch: 1, Version: "1.12.10", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:ee3f72fbc8a89d136d100db721431454", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.10", SrcRelease: "1.fc28"},
				{Name: "libnghttp2", Epoch: 0, Version: "1.32.1", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:aa8a98c6d5a648db8e4f4c60a687cf10", SrcName: "nghttp2", SrcEpoch: 0, SrcVersion: "1.32.1", SrcRelease: "1.fc28"},
				{Name: "librepo", Epoch: 0, Version: "1.8.1", Release: "7.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:ca7b6f87d7b63decc8986ca383b3d4a0", SrcName: "librepo", SrcEpoch: 0, SrcVersion: "1.8.1", SrcRelease: "7.fc28"},
				{Name: "curl", Epoch: 0, Version: "7.59.0", Release: "6.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "Fedora Project", Digest: "md5:508c4c2ad02cffe7d550f15d60a2e131", SrcName: "curl", SrcEpoch: 0, SrcVersion: "7.59.0", SrcRelease: "6.fc28"},
				{Name: "rpm-libs", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+ with exceptions"}, Maintainer: "Fedora Project", Digest: "md5:522ef24ea7270005a459b2836051f4b3", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "libsolv", Epoch: 0, Version: "0.6.35", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "Fedora Project", Digest: "md5:7de1966ef13546320718670d8174f693", SrcName: "libsolv", SrcEpoch: 0, SrcVersion: "0.6.35", SrcRelease: "1.fc28"},
				{Name: "python3-hawkey", Epoch: 0, Version: "0.11.1", Release: "3.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:8699eceb6522941c1218010bc2862058", SrcName: "libdnf", SrcEpoch: 0, SrcVersion: "0.11.1", SrcRelease: "3.fc28"},
				{Name: "rpm-sign-libs", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+ with exceptions"}, Maintainer: "Fedora Project", Digest: "md5:4c0ed3291ed6592709845d6a9c119a2f", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "python3-dnf", Epoch: 0, Version: "2.7.5", Release: "12.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "Fedora Project", Digest: "md5:55528c254ac96f57623bbaa3ae9eb41c", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "2.7.5", SrcRelease: "12.fc28"},
				{Name: "dnf-yum", Epoch: 0, Version: "2.7.5", Release: "12.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "Fedora Project", Digest: "md5:5f1ecbeb04b81e6d3433232a3d04f17e", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "2.7.5", SrcRelease: "12.fc28"},
				{Name: "rpm-plugin-systemd-inhibit", Epoch: 0, Version: "4.14.1", Release: "9.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "Fedora Project", Digest: "md5:ff3e305325cf0052a13b2865cc28d1b7", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.1", SrcRelease: "9.fc28"},
				{Name: "nss-tools", Epoch: 0, Version: "3.38.0", Release: "1.0.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "Fedora Project", Digest: "md5:118c6a98328213346b8a4cc4dba26884", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.38.0", SrcRelease: "1.0.fc28"},
				{Name: "openssl-pkcs11", Epoch: 0, Version: "0.4.8", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and BSD"}, Maintainer: "Fedora Project", Digest: "md5:2f91bf3dff53be5197f5ef12daac76a6", SrcName: "openssl-pkcs11", SrcEpoch: 0, SrcVersion: "0.4.8", SrcRelease: "1.fc28"},
				{Name: "vim-minimal", Epoch: 2, Version: "8.1.328", Release: "1.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Vim and MIT"}, Maintainer: "Fedora Project", Digest: "md5:5ad20c562d35c2b3f15fa65dd6e31b7d", SrcName: "vim", SrcEpoch: 2, SrcVersion: "8.1.328", SrcRelease: "1.fc28"},
				{Name: "glibc-langpack-en", Epoch: 0, Version: "2.27", Release: "32.fc28", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "Fedora Project", Digest: "md5:3f4fbbebb7e3205a955e6aecec486f83", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.27", SrcRelease: "32.fc28"},
				{Name: "rootfiles", Epoch: 0, Version: "8.1", Release: "22.fc28", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "Fedora Project", Digest: "md5:2f9c0518edf47a80c269222c354f2a70", SrcName: "rootfiles", SrcEpoch: 0, SrcVersion: "8.1", SrcRelease: "22.fc28"},
			},
		},
		"ValidWithModularitylabel": {
			path: "./testdata/valid_with_modularitylabel",
			// cp ./testdata/valid_with_modularitylabel /path/to/testdir/Packages
			// rpm -qa --dbpath /path/to/testdir --queryformat "\{Name: \"%{NAME}\", Epoch: %{RPMTAG_EPOCHNUM}, Version: \"%{VERSION}\", Release: \"%{RELEASE}\", Arch: \"%{ARCH}\", Modularitylabel: \"%{RPMTAG_MODULARITYLABEL}\", Licenses: \[\]string\{\"%{LICENSE}\"\}, Maintainer: \"%{RPMTAG_VENDOR}\", Digest: \"md5:%{SIGMD5}\",\n" | sed "s/(none)//g" > 1.txt
			// rpm -qa --dbpath /path/to/testdir --queryformat "%{SOURCERPM}-%{RPMTAG_EPOCHNUM}\n" | awk -F"-" '{printf("SrcName: \""$0"\", SrcEpoch: "$NF", SrcVersion: \""$(NF-2)"\", SrcRelease: \""$(NF-1)"\"},\n")}' | sed -E 's/-[0-9.]+-.+.src.rpm-[0-9]//' | sed 's/.src.rpm//g' > 2.txt
			// paste -d " " 1.txt 2.txt
			pkgs: []types.Package{
				{Name: "perl-podlators", Epoch: 0, Version: "4.11", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and FSFAP"}, Maintainer: "CentOS", Digest: "md5:bd2b28862f36b3cec0e1d5fb74d5edb7", SrcName: "perl-podlators", SrcEpoch: 0, SrcVersion: "4.11", SrcRelease: "1.el8"},
				{Name: "python3-setuptools-wheel", Epoch: 0, Version: "39.2.0", Release: "5.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:86b613f3e1fbf33755d1b34d3e1cb098", SrcName: "python-setuptools", SrcEpoch: 0, SrcVersion: "39.2.0", SrcRelease: "5.el8"},
				{Name: "perl-Pod-Perldoc", Epoch: 0, Version: "3.28", Release: "396.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:fc8b1db50f574413212744f6f888c3d2", SrcName: "perl-Pod-Perldoc", SrcEpoch: 0, SrcVersion: "3.28", SrcRelease: "396.el8"},
				{Name: "perl-IO-Socket-SSL", Epoch: 0, Version: "2.066", Release: "4.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:896f66d11d6b66dd0fe2f849b41febed", SrcName: "perl-IO-Socket-SSL", SrcEpoch: 0, SrcVersion: "2.066", SrcRelease: "4.el8"},
				{Name: "perl-URI", Epoch: 0, Version: "1.73", Release: "3.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:b66d85e3853e61853c8334dd507a77a0", SrcName: "perl-URI", SrcEpoch: 0, SrcVersion: "1.73", SrcRelease: "3.el8"},
				{Name: "filesystem", Epoch: 0, Version: "3.8", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:b3b5004f3478d8b6692e392260597d96", SrcName: "filesystem", SrcEpoch: 0, SrcVersion: "3.8", SrcRelease: "2.el8"},
				{Name: "emacs-filesystem", Epoch: 1, Version: "26.1", Release: "5.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv3+ and CC0-1.0"}, Maintainer: "CentOS", Digest: "md5:94218736a9d0203615311c0eb679b00a", SrcName: "emacs", SrcEpoch: 1, SrcVersion: "26.1", SrcRelease: "5.el8"},
				{Name: "git", Epoch: 0, Version: "2.18.4", Release: "2.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:c8a5db38b49ac1f39df666f5fbf3c6e8", SrcName: "git", SrcEpoch: 0, SrcVersion: "2.18.4", SrcRelease: "2.el8_2"},
				{Name: "pcre2", Epoch: 0, Version: "10.32", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:2e9a6d375bb106ef8dbb9f37fa960b17", SrcName: "pcre2", SrcEpoch: 0, SrcVersion: "10.32", SrcRelease: "1.el8"},
				{Name: "vim-common", Epoch: 2, Version: "8.0.1763", Release: "13.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Vim and MIT"}, Maintainer: "CentOS", Digest: "md5:ff0b106bea9523285c1c4ef88558adc4", SrcName: "vim", SrcEpoch: 2, SrcVersion: "8.0.1763", SrcRelease: "13.el8"},
				{Name: "ncurses-libs", Epoch: 0, Version: "6.1", Release: "7.20180224.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:e22fb3c519b0d6c0b1544ade616647a1", SrcName: "ncurses", SrcEpoch: 0, SrcVersion: "6.1", SrcRelease: "7.20180224.el8"},
				{Name: "vim-enhanced", Epoch: 2, Version: "8.0.1763", Release: "13.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Vim and MIT"}, Maintainer: "CentOS", Digest: "md5:62728e1b7906f37e24e446db101cbebc", SrcName: "vim", SrcEpoch: 2, SrcVersion: "8.0.1763", SrcRelease: "13.el8"},
				{Name: "glibc-common", Epoch: 0, Version: "2.28", Release: "101.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:59fd06b7b2ff75b002fbdc1de4e84b8f", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.28", SrcRelease: "101.el8"},
				{Name: "openssl-devel", Epoch: 1, Version: "1.1.1c", Release: "15.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenSSL"}, Maintainer: "CentOS", Digest: "md5:dc02d8f2b675a1a062e37364789e980c", SrcName: "openssl", SrcEpoch: 1, SrcVersion: "1.1.1c", SrcRelease: "15.el8"},
				{Name: "bash", Epoch: 0, Version: "4.4.19", Release: "10.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:8b1a1d644da5e718f7a1cd76769e3f83", SrcName: "bash", SrcEpoch: 0, SrcVersion: "4.4.19", SrcRelease: "10.el8"},
				{Name: "popt-devel", Epoch: 0, Version: "1.16", Release: "14.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:510a39ef7fcc2ddcc1b9a8bdef0cce12", SrcName: "popt", SrcEpoch: 0, SrcVersion: "1.16", SrcRelease: "14.el8"},
				{Name: "libarchive-devel", Epoch: 0, Version: "3.3.2", Release: "8.el8_1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:55aebe7f19c587ed2d5eac64c255b9f5", SrcName: "libarchive", SrcEpoch: 0, SrcVersion: "3.3.2", SrcRelease: "8.el8_1"},
				{Name: "bzip2-libs", Epoch: 0, Version: "1.0.6", Release: "26.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:dd9abf1cc65154142a808c839f4dda55", SrcName: "bzip2", SrcEpoch: 0, SrcVersion: "1.0.6", SrcRelease: "26.el8"},
				{Name: "xz-lzma-compat", Epoch: 0, Version: "5.2.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:b361e08322d3397cff6b98f18612a75e", SrcName: "xz", SrcEpoch: 0, SrcVersion: "5.2.4", SrcRelease: "3.el8"},
				{Name: "libgpg-error", Epoch: 0, Version: "1.31", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:e84b78135beff56ee4835fed39d7401c", SrcName: "libgpg-error", SrcEpoch: 0, SrcVersion: "1.31", SrcRelease: "1.el8"},
				{Name: "libdb-devel", Epoch: 0, Version: "5.3.28", Release: "37.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2 and Sleepycat"}, Maintainer: "CentOS", Digest: "md5:14528e1de2b3ea9df2c3bee88bea7673", SrcName: "libdb", SrcEpoch: 0, SrcVersion: "5.3.28", SrcRelease: "37.el8"},
				{Name: "elfutils-libelf", Epoch: 0, Version: "0.178", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "CentOS", Digest: "md5:cd8b135e1df7ffbfb140b41ab26f87aa", SrcName: "elfutils", SrcEpoch: 0, SrcVersion: "0.178", SrcRelease: "7.el8"},
				{Name: "libgomp", Epoch: 0, Version: "8.3.1", Release: "5.el8.0.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:2356db2ff13d8d90ede9c428e79327b1", SrcName: "gcc", SrcEpoch: 0, SrcVersion: "8.3.1", SrcRelease: "5.el8.0.2"},
				{Name: "libxcrypt", Epoch: 0, Version: "4.1.1", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and BSD and Public Domain"}, Maintainer: "CentOS", Digest: "md5:5d0ae947dd72ad1f5b27c8ab209e9b12", SrcName: "libxcrypt", SrcEpoch: 0, SrcVersion: "4.1.1", SrcRelease: "4.el8"},
				{Name: "gettext-libs", Epoch: 0, Version: "0.19.8.1", Release: "17.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and GPLv3+"}, Maintainer: "CentOS", Digest: "md5:8967e0fd86f922ac3fe079bb64ec85c4", SrcName: "gettext", SrcEpoch: 0, SrcVersion: "0.19.8.1", SrcRelease: "17.el8"},
				{Name: "sqlite-libs", Epoch: 0, Version: "3.26.0", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:123153073f9818beb07847d632f6bbb7", SrcName: "sqlite", SrcEpoch: 0, SrcVersion: "3.26.0", SrcRelease: "6.el8"},
				{Name: "cpp", Epoch: 0, Version: "8.3.1", Release: "5.el8.0.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:0578fcab8e65239fc0b5000723873379", SrcName: "gcc", SrcEpoch: 0, SrcVersion: "8.3.1", SrcRelease: "5.el8.0.2"},
				{Name: "libstdc++", Epoch: 0, Version: "8.3.1", Release: "5.el8.0.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:cb453d84e5e5758ed5e5cb27247ad32b", SrcName: "gcc", SrcEpoch: 0, SrcVersion: "8.3.1", SrcRelease: "5.el8.0.2"},
				{Name: "m4", Epoch: 0, Version: "1.4.18", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:65b81ee9fe7f8936bafa29901124e3a4", SrcName: "m4", SrcEpoch: 0, SrcVersion: "1.4.18", SrcRelease: "7.el8"},
				{Name: "popt", Epoch: 0, Version: "1.16", Release: "14.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:91f1fc1e5257f088cf83fcde67e1597f", SrcName: "popt", SrcEpoch: 0, SrcVersion: "1.16", SrcRelease: "14.el8"},
				{Name: "libgpg-error-devel", Epoch: 0, Version: "1.31", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:139964288d8e4c634dd609f40ef40fe3", SrcName: "libgpg-error", SrcEpoch: 0, SrcVersion: "1.31", SrcRelease: "1.el8"},
				{Name: "readline", Epoch: 0, Version: "7.0", Release: "10.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:53395adc92c214ce3ce3f63c19180d19", SrcName: "readline", SrcEpoch: 0, SrcVersion: "7.0", SrcRelease: "10.el8"},
				{Name: "glibc-headers", Epoch: 0, Version: "2.28", Release: "101.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:14929ccf48957fe03d69f587efe9d942", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.28", SrcRelease: "101.el8"},
				{Name: "json-c", Epoch: 0, Version: "0.13.1", Release: "0.2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:eb194624bd2d5fd5966ea9a6927525c7", SrcName: "json-c", SrcEpoch: 0, SrcVersion: "0.13.1", SrcRelease: "0.2.el8"},
				{Name: "glibc-devel", Epoch: 0, Version: "2.28", Release: "101.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:e522ba17ee3f0d9e73ba581ec0dbf5d9", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.28", SrcRelease: "101.el8"},
				{Name: "libacl", Epoch: 0, Version: "2.2.53", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:cd43f03193723ee093e7705f1c5c6f39", SrcName: "acl", SrcEpoch: 0, SrcVersion: "2.2.53", SrcRelease: "1.el8"},
				{Name: "perl-Thread-Queue", Epoch: 0, Version: "3.13", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:6f6fe2a2414cdc6fe4ef25c47b079c62", SrcName: "perl-Thread-Queue", SrcEpoch: 0, SrcVersion: "3.13", SrcRelease: "1.el8"},
				{Name: "sed", Epoch: 0, Version: "4.5", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:3ddec62cedf49c241d6ab77613daaf0a", SrcName: "sed", SrcEpoch: 0, SrcVersion: "4.5", SrcRelease: "1.el8"},
				{Name: "isl", Epoch: 0, Version: "0.16.1", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:7e07c5719518014f7be119290bd3c713", SrcName: "isl", SrcEpoch: 0, SrcVersion: "0.16.1", SrcRelease: "6.el8"},
				{Name: "libmount", Epoch: 0, Version: "2.32.1", Release: "22.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:bc5feadfa4bfce68811894bb73f493c4", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "22.el8"},
				{Name: "libtool", Epoch: 0, Version: "2.4.6", Release: "25.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+ and GFDL"}, Maintainer: "CentOS", Digest: "md5:46ac6d2c69ea159d33657a2cd9525be4", SrcName: "libtool", SrcEpoch: 0, SrcVersion: "2.4.6", SrcRelease: "25.el8"},
				{Name: "audit-libs", Epoch: 0, Version: "3.0", Release: "0.17.20191104git1c2f876.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:85d66fe7059cab257dff0b1a99e59479", SrcName: "audit", SrcEpoch: 0, SrcVersion: "3.0", SrcRelease: "0.17.20191104git1c2f876.el8"},
				{Name: "libgcrypt-devel", Epoch: 0, Version: "1.8.3", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:660426abcb73c094d1d5a97224b435d2", SrcName: "libgcrypt", SrcEpoch: 0, SrcVersion: "1.8.3", SrcRelease: "4.el8"},
				{Name: "libsmartcols", Epoch: 0, Version: "2.32.1", Release: "22.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:6533915dd27fba66917d6cdca81a67fc", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "22.el8"},
				{Name: "nodejs-full-i18n", Epoch: 1, Version: "10.21.0", Release: "3.module_el8.2.0+391+8da3adc6", Arch: "x86_64", Modularitylabel: "nodejs:10:8020020200707141642:6a468ee4", Licenses: []string{"MIT and ASL 2.0 and ISC and BSD"}, Maintainer: "CentOS", Digest: "md5:06c123561d40c914ddb262744ad30403", SrcName: "nodejs", SrcEpoch: 1, SrcVersion: "10.21.0", SrcRelease: "3.module_el8.2.0+391+8da3adc6"},
				{Name: "lua-libs", Epoch: 0, Version: "5.3.4", Release: "11.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:3ec414ac7e8a8f90968c86da820336cb", SrcName: "lua", SrcEpoch: 0, SrcVersion: "5.3.4", SrcRelease: "11.el8"},
				{Name: "nodejs", Epoch: 1, Version: "10.21.0", Release: "3.module_el8.2.0+391+8da3adc6", Arch: "x86_64", Modularitylabel: "nodejs:10:8020020200707141642:6a468ee4", Licenses: []string{"MIT and ASL 2.0 and ISC and BSD"}, Maintainer: "CentOS", Digest: "md5:bac7919c2369f944f9da510bbd01370b", SrcName: "nodejs", SrcEpoch: 1, SrcVersion: "10.21.0", SrcRelease: "3.module_el8.2.0+391+8da3adc6"},
				{Name: "p11-kit", Epoch: 0, Version: "0.23.14", Release: "5.el8_0", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:77b1d075374eae3c0fa1a2ffa004120a", SrcName: "p11-kit", SrcEpoch: 0, SrcVersion: "0.23.14", SrcRelease: "5.el8_0"},
				{Name: "libbabeltrace", Epoch: 0, Version: "1.5.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT and GPLv2"}, Maintainer: "CentOS", Digest: "md5:bd7c93a25505f05172f8039d5ba03d4a", SrcName: "babeltrace", SrcEpoch: 0, SrcVersion: "1.5.4", SrcRelease: "3.el8"},
				{Name: "gzip", Epoch: 0, Version: "1.9", Release: "9.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GFDL"}, Maintainer: "CentOS", Digest: "md5:d540a593d8d1a3e01a685dbe12294a37", SrcName: "gzip", SrcEpoch: 0, SrcVersion: "1.9", SrcRelease: "9.el8"},
				{Name: "libatomic_ops", Epoch: 0, Version: "7.6.2", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and MIT"}, Maintainer: "CentOS", Digest: "md5:6f1ef12f2494056f5cbafb60b23d483c", SrcName: "libatomic_ops", SrcEpoch: 0, SrcVersion: "7.6.2", SrcRelease: "3.el8"},
				{Name: "libunistring", Epoch: 0, Version: "0.9.9", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "CentOS", Digest: "md5:7769c82a0a6aa692f4778f71f6fdec0c", SrcName: "libunistring", SrcEpoch: 0, SrcVersion: "0.9.9", SrcRelease: "3.el8"},
				{Name: "guile", Epoch: 5, Version: "2.0.14", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+"}, Maintainer: "CentOS", Digest: "md5:575e446ce886def641e2b1c46db9480d", SrcName: "guile", SrcEpoch: 5, SrcVersion: "2.0.14", SrcRelease: "7.el8"},
				{Name: "libassuan", Epoch: 0, Version: "2.5.1", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and GPLv3+"}, Maintainer: "CentOS", Digest: "md5:44d9976e6073475aebe11bef3ff8dc33", SrcName: "libassuan", SrcEpoch: 0, SrcVersion: "2.5.1", SrcRelease: "3.el8"},
				{Name: "gdb", Epoch: 0, Version: "8.2", Release: "12.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and LGPLv3+ and BSD and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:0ca27d09cfd7d1091691bcdc0bafe3c9", SrcName: "gdb", SrcEpoch: 0, SrcVersion: "8.2", SrcRelease: "12.el8"},
				{Name: "gdbm-libs", Epoch: 1, Version: "1.18", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:8efb9ab97fa0223d11d0857febae479d", SrcName: "gdbm", SrcEpoch: 1, SrcVersion: "1.18", SrcRelease: "1.el8"},
				{Name: "platform-python-setuptools", Epoch: 0, Version: "39.2.0", Release: "6.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:474f0b4e883902feae0a35bf08bb1b3e", SrcName: "python-setuptools", SrcEpoch: 0, SrcVersion: "39.2.0", SrcRelease: "6.el8"},
				{Name: "libtasn1", Epoch: 0, Version: "4.13", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:6fb9739221265fd08745407c915f792d", SrcName: "libtasn1", SrcEpoch: 0, SrcVersion: "4.13", SrcRelease: "3.el8"},
				{Name: "python3-setuptools", Epoch: 0, Version: "39.2.0", Release: "6.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:5c536abe6bb6f1df908aa0507a61bceb", SrcName: "python-setuptools", SrcEpoch: 0, SrcVersion: "39.2.0", SrcRelease: "6.el8"},
				{Name: "lzo", Epoch: 0, Version: "2.08", Release: "14.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:2e8c7912284d0c40f98a93649366097c", SrcName: "lzo", SrcEpoch: 0, SrcVersion: "2.08", SrcRelease: "14.el8"},
				{Name: "python3-pip", Epoch: 0, Version: "9.0.3", Release: "18.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)"}, Maintainer: "CentOS", Digest: "md5:5c07b4e25764b5f90d020849f23e1015", SrcName: "python-pip", SrcEpoch: 0, SrcVersion: "9.0.3", SrcRelease: "18.el8"},
				{Name: "grep", Epoch: 0, Version: "3.1", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:94ed80f88be0c0b77082a51035c5e53f", SrcName: "grep", SrcEpoch: 0, SrcVersion: "3.1", SrcRelease: "6.el8"},
				{Name: "python2-pip-wheel", Epoch: 0, Version: "9.0.3", Release: "18.module_el8.3.0+478+7570e00c", Arch: "noarch", Modularitylabel: "python27:2.7:8030020200831201838:851f4228", Licenses: []string{"MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)"}, Maintainer: "CentOS", Digest: "md5:cf364e483d3b1f2e47c035520d322c04", SrcName: "python2-pip", SrcEpoch: 0, SrcVersion: "9.0.3", SrcRelease: "18.module_el8.3.0+478+7570e00c"},
				{Name: "dbus-libs", Epoch: 1, Version: "1.12.8", Release: "10.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:e895b9b8d703c114cb15c7564df6de96", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.8", SrcRelease: "10.el8_2"},
				{Name: "python2-pip", Epoch: 0, Version: "9.0.3", Release: "18.module_el8.3.0+478+7570e00c", Arch: "noarch", Modularitylabel: "python27:2.7:8030020200831201838:851f4228", Licenses: []string{"MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)"}, Maintainer: "CentOS", Digest: "md5:a70ae1bfca6b1617bfc1cc87a05a8bd6", SrcName: "python2-pip", SrcEpoch: 0, SrcVersion: "9.0.3", SrcRelease: "18.module_el8.3.0+478+7570e00c"},
				{Name: "dhcp-libs", Epoch: 12, Version: "4.3.6", Release: "40.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "CentOS", Digest: "md5:1f247bfb2d46e59b1fe552896cfcf0d9", SrcName: "dhcp", SrcEpoch: 12, SrcVersion: "4.3.6", SrcRelease: "40.el8"},
				{Name: "python2", Epoch: 0, Version: "2.7.17", Release: "2.module_el8.3.0+478+7570e00c", Arch: "x86_64", Modularitylabel: "python27:2.7:8030020200831201838:851f4228", Licenses: []string{"Python"}, Maintainer: "CentOS", Digest: "md5:111e33c5066b23212de6c1299ddf0f85", SrcName: "python2", SrcEpoch: 0, SrcVersion: "2.7.17", SrcRelease: "2.module_el8.3.0+478+7570e00c"},
				{Name: "procps-ng", Epoch: 0, Version: "3.3.15", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ and GPLv2 and GPLv2+ and GPLv3+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:030ea9850814d2c60c5af0ddb2f69c72", SrcName: "procps-ng", SrcEpoch: 0, SrcVersion: "3.3.15", SrcRelease: "1.el8"},
				{Name: "python2-rpmUtils", Epoch: 0, Version: "0.1", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "", Digest: "md5:61a8edb4eaae41ce7e3be3b82c19a091", SrcName: "python-rpmUtils", SrcEpoch: 0, SrcVersion: "0.1", SrcRelease: "1.el8"},
				{Name: "xz", Epoch: 0, Version: "5.2.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and Public Domain"}, Maintainer: "CentOS", Digest: "md5:2acee73f3a4b9738e0133043726a13f7", SrcName: "xz", SrcEpoch: 0, SrcVersion: "5.2.4", SrcRelease: "3.el8"},
				{Name: "rpm", Epoch: 0, Version: "4.14.3", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:d2caa895a1f0373abe18afe94b371c29", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.3", SrcRelease: "4.el8"},
				{Name: "gdbm", Epoch: 1, Version: "1.18", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:69ce5734e1fd54cdd298b4667d76616f", SrcName: "gdbm", SrcEpoch: 1, SrcVersion: "1.18", SrcRelease: "1.el8"},
				{Name: "python3-rpm", Epoch: 0, Version: "4.14.3", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:0eb23c1126d20050844447c82dbbddfa", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.3", SrcRelease: "4.el8"},
				{Name: "shadow-utils", Epoch: 2, Version: "4.6", Release: "8.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:211de7a014813ec64ce2bb8800b336b6", SrcName: "shadow-utils", SrcEpoch: 2, SrcVersion: "4.6", SrcRelease: "8.el8"},
				{Name: "libfdisk", Epoch: 0, Version: "2.32.1", Release: "22.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:886e9927f7ebfcc4c907a870f40d9c09", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "22.el8"},
				{Name: "mpfr", Epoch: 0, Version: "3.1.6", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ and GPLv3+ and GFDL"}, Maintainer: "CentOS", Digest: "md5:7504c4f1b7aab1cf22f4896a6c35e91f", SrcName: "mpfr", SrcEpoch: 0, SrcVersion: "3.1.6", SrcRelease: "1.el8"},
				{Name: "snappy", Epoch: 0, Version: "1.1.7", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:459dcaf6d9799509b291148959e3a2ac", SrcName: "snappy", SrcEpoch: 0, SrcVersion: "1.1.7", SrcRelease: "5.el8"},
				{Name: "libmetalink", Epoch: 0, Version: "0.1.3", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:a97ac81965ec113b881058f001c770f6", SrcName: "libmetalink", SrcEpoch: 0, SrcVersion: "0.1.3", SrcRelease: "7.el8"},
				{Name: "libksba", Epoch: 0, Version: "1.3.5", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(LGPLv3+ or GPLv2+) and GPLv3+"}, Maintainer: "CentOS", Digest: "md5:74d2b871d0f179895daee53422a52b94", SrcName: "libksba", SrcEpoch: 0, SrcVersion: "1.3.5", SrcRelease: "7.el8"},
				{Name: "ethtool", Epoch: 2, Version: "5.0", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:ec93a9c27813a31ebce90966632b8edd", SrcName: "ethtool", SrcEpoch: 2, SrcVersion: "5.0", SrcRelease: "2.el8"},
				{Name: "libmnl", Epoch: 0, Version: "1.0.4", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:003fa2a9079721368b658c63d8832e20", SrcName: "libmnl", SrcEpoch: 0, SrcVersion: "1.0.4", SrcRelease: "6.el8"},
				{Name: "libpcap", Epoch: 14, Version: "1.9.0", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD with advertising"}, Maintainer: "CentOS", Digest: "md5:4d2a5ad234af8d3ebe34fc032aa8dfdd", SrcName: "libpcap", SrcEpoch: 14, SrcVersion: "1.9.0", SrcRelease: "3.el8"},
				{Name: "libseccomp", Epoch: 0, Version: "2.4.1", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2"}, Maintainer: "CentOS", Digest: "md5:6237a23621d5be842f2db331714e7215", SrcName: "libseccomp", SrcEpoch: 0, SrcVersion: "2.4.1", SrcRelease: "1.el8"},
				{Name: "gawk", Epoch: 0, Version: "4.2.1", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv2+ and LGPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:c47002ae2e1dddb7747b3a6e813e821d", SrcName: "gawk", SrcEpoch: 0, SrcVersion: "4.2.1", SrcRelease: "1.el8"},
				{Name: "libnsl2", Epoch: 0, Version: "1.2.0", Release: "2.20180605git4a062cf.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:7be9bc3ac6e7e4d56de085c410a8e20d", SrcName: "libnsl2", SrcEpoch: 0, SrcVersion: "1.2.0", SrcRelease: "2.20180605git4a062cf.el8"},
				{Name: "krb5-libs", Epoch: 0, Version: "1.17", Release: "18.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:800e946bba1c5c102ccdb4159c046ce6", SrcName: "krb5", SrcEpoch: 0, SrcVersion: "1.17", SrcRelease: "18.el8"},
				{Name: "crypto-policies", Epoch: 0, Version: "20191128", Release: "2.git23e1bf1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:a6cdcd759a11870401ea0021c2a1496c", SrcName: "crypto-policies", SrcEpoch: 0, SrcVersion: "20191128", SrcRelease: "2.git23e1bf1.el8"},
				{Name: "platform-python", Epoch: 0, Version: "3.6.8", Release: "23.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Python"}, Maintainer: "CentOS", Digest: "md5:5674cc4a141996200466adabc7e51f5c", SrcName: "python3", SrcEpoch: 0, SrcVersion: "3.6.8", SrcRelease: "23.el8"},
				{Name: "libdb", Epoch: 0, Version: "5.3.28", Release: "37.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2 and Sleepycat"}, Maintainer: "CentOS", Digest: "md5:293f618e23dd98b899b17e526e052883", SrcName: "libdb", SrcEpoch: 0, SrcVersion: "5.3.28", SrcRelease: "37.el8"},
				{Name: "pam", Epoch: 0, Version: "1.3.1", Release: "8.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:f83fb370c1196ddc5e979ddf009b40b2", SrcName: "pam", SrcEpoch: 0, SrcVersion: "1.3.1", SrcRelease: "8.el8"},
				{Name: "gnutls", Epoch: 0, Version: "3.6.8", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:9d8812427df36d252b285cdc3ea1bfc9", SrcName: "gnutls", SrcEpoch: 0, SrcVersion: "3.6.8", SrcRelease: "11.el8_2"},
				{Name: "kmod-libs", Epoch: 0, Version: "25", Release: "16.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:09bde70809bfa4bf789c0ad78f31dc15", SrcName: "kmod", SrcEpoch: 0, SrcVersion: "25", SrcRelease: "16.el8"},
				{Name: "ima-evm-utils", Epoch: 0, Version: "1.1", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:c5dccadfe8fd2c247d6785651fb218c5", SrcName: "ima-evm-utils", SrcEpoch: 0, SrcVersion: "1.1", SrcRelease: "5.el8"},
				{Name: "libcurl-minimal", Epoch: 0, Version: "7.61.1", Release: "12.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:97c7f1247354d8a7561b5a26dd184ebb", SrcName: "curl", SrcEpoch: 0, SrcVersion: "7.61.1", SrcRelease: "12.el8"},
				{Name: "cyrus-sasl-lib", Epoch: 0, Version: "2.1.27", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD with advertising"}, Maintainer: "CentOS", Digest: "md5:7a3b1a78bd4838e0f6324f5bba9c148e", SrcName: "cyrus-sasl", SrcEpoch: 0, SrcVersion: "2.1.27", SrcRelease: "1.el8"},
				{Name: "libdb-utils", Epoch: 0, Version: "5.3.28", Release: "37.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and LGPLv2 and Sleepycat"}, Maintainer: "CentOS", Digest: "md5:e4324aa7441337796a429e9f92455ede", SrcName: "libdb", SrcEpoch: 0, SrcVersion: "5.3.28", SrcRelease: "37.el8"},
				{Name: "libsolv", Epoch: 0, Version: "0.7.7", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:b742b6c750a836cf3ac712046b1479fa", SrcName: "libsolv", SrcEpoch: 0, SrcVersion: "0.7.7", SrcRelease: "1.el8"},
				{Name: "libmodulemd1", Epoch: 0, Version: "1.8.16", Release: "0.2.8.2.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:6b71512e8c92424e2bbf51ac393fb29c", SrcName: "libmodulemd", SrcEpoch: 0, SrcVersion: "2.8.2", SrcRelease: "1.el8"},
				{Name: "gnupg2", Epoch: 0, Version: "2.2.9", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:0b4519bf8b18306203885e2108c81aca", SrcName: "gnupg2", SrcEpoch: 0, SrcVersion: "2.2.9", SrcRelease: "1.el8"},
				{Name: "python3-libdnf", Epoch: 0, Version: "0.39.1", Release: "6.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:38a109680c9ff4ce9d1c8f3597c346c4", SrcName: "libdnf", SrcEpoch: 0, SrcVersion: "0.39.1", SrcRelease: "6.el8_2"},
				{Name: "python3-gpg", Epoch: 0, Version: "1.10.0", Release: "6.el8.0.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:f1bb67f41c054b7a241452a9db8c72f4", SrcName: "gpgme", SrcEpoch: 0, SrcVersion: "1.10.0", SrcRelease: "6.el8.0.1"},
				{Name: "dnf-data", Epoch: 0, Version: "4.2.17", Release: "7.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "CentOS", Digest: "md5:9e2017577948cc0fb98c7907611ee902", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "4.2.17", SrcRelease: "7.el8_2"},
				{Name: "dbus-common", Epoch: 1, Version: "1.12.8", Release: "10.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:77d77f4770ed1b5442e40d6d8fd10d58", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.8", SrcRelease: "10.el8_2"},
				{Name: "device-mapper", Epoch: 8, Version: "1.02.169", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:4e2e93bf6d5fc21511b94184b03be6b9", SrcName: "lvm2", SrcEpoch: 8, SrcVersion: "2.03.08", SrcRelease: "3.el8"},
				{Name: "cryptsetup-libs", Epoch: 0, Version: "2.2.2", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:97b1e392c1b2c834b1233239b95bd0cc", SrcName: "cryptsetup", SrcEpoch: 0, SrcVersion: "2.2.2", SrcRelease: "1.el8"},
				{Name: "elfutils-libs", Epoch: 0, Version: "0.178", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "CentOS", Digest: "md5:11d18e7fceb1aacb16e96dbebdf78557", SrcName: "elfutils", SrcEpoch: 0, SrcVersion: "0.178", SrcRelease: "7.el8"},
				{Name: "systemd", Epoch: 0, Version: "239", Release: "31.el8_2.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:e8944e79d0bd5f79c2746813db1244ee", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "239", SrcRelease: "31.el8_2.2"},
				{Name: "iputils", Epoch: 0, Version: "20180629", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:ca9a89c22af93423852ecbd08d3abf59", SrcName: "iputils", SrcEpoch: 0, SrcVersion: "20180629", SrcRelease: "2.el8"},
				{Name: "libkcapi", Epoch: 0, Version: "1.1.1", Release: "16_1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD or GPLv2"}, Maintainer: "CentOS", Digest: "md5:d765e8153b4adaa1b9d759650ef97cce", SrcName: "libkcapi", SrcEpoch: 0, SrcVersion: "1.1.1", SrcRelease: "16_1.el8"},
				{Name: "systemd-udev", Epoch: 0, Version: "239", Release: "31.el8_2.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:36012002d579569d43dfa958dbbaf95c", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "239", SrcRelease: "31.el8_2.2"},
				{Name: "dracut-network", Epoch: 0, Version: "049", Release: "70.git20200228.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:e91224a402dbebc12de2db7fb4747e4a", SrcName: "dracut", SrcEpoch: 0, SrcVersion: "049", SrcRelease: "70.git20200228.el8"},
				{Name: "python3-dnf", Epoch: 0, Version: "4.2.17", Release: "7.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "CentOS", Digest: "md5:32a841a42030f8a4768761feb148c477", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "4.2.17", SrcRelease: "7.el8_2"},
				{Name: "yum", Epoch: 0, Version: "4.2.17", Release: "7.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "CentOS", Digest: "md5:315b9809e784fb291a584f24244e641e", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "4.2.17", SrcRelease: "7.el8_2"},
				{Name: "binutils", Epoch: 0, Version: "2.30", Release: "73.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:f6fcd230323217f591c9f96172dae25f", SrcName: "binutils", SrcEpoch: 0, SrcVersion: "2.30", SrcRelease: "73.el8"},
				{Name: "vim-minimal", Epoch: 2, Version: "8.0.1763", Release: "13.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Vim and MIT"}, Maintainer: "CentOS", Digest: "md5:ed73caeb8c3746e48ced83354900908b", SrcName: "vim", SrcEpoch: 2, SrcVersion: "8.0.1763", SrcRelease: "13.el8"},
				{Name: "less", Epoch: 0, Version: "530", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ or BSD"}, Maintainer: "CentOS", Digest: "md5:c95ece51b133ab6ce794a40e777072d2", SrcName: "less", SrcEpoch: 0, SrcVersion: "530", SrcRelease: "1.el8"},
				{Name: "rootfiles", Epoch: 0, Version: "8.1", Release: "22.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:8396ccef88edaffce4a8d5eb1ac4aa66", SrcName: "rootfiles", SrcEpoch: 0, SrcVersion: "8.1", SrcRelease: "22.el8"},
				{Name: "centos-gpg-keys", Epoch: 0, Version: "8.2", Release: "2.2004.0.2.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:48b8047e1811de4b7702c3954af7f50c", SrcName: "centos-release", SrcEpoch: 0, SrcVersion: "8.2", SrcRelease: "2.2004.0.2.el8"},
				{Name: "centos-repos", Epoch: 0, Version: "8.2", Release: "2.2004.0.2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:36b1ffd95fc9f6aa9b970635dcc1b2cb", SrcName: "centos-release", SrcEpoch: 0, SrcVersion: "8.2", SrcRelease: "2.2004.0.2.el8"},
				{Name: "tzdata", Epoch: 0, Version: "2020d", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:6ea2473648de3aaee55267c11000854d", SrcName: "tzdata", SrcEpoch: 0, SrcVersion: "2020d", SrcRelease: "1.el8"},
				{Name: "ca-certificates", Epoch: 0, Version: "2020.2.41", Release: "80.0.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:3e438f05cd93729e37e44d76a8059e57", SrcName: "ca-certificates", SrcEpoch: 0, SrcVersion: "2020.2.41", SrcRelease: "80.0.el8_2"},
				{Name: "perl-Exporter", Epoch: 0, Version: "5.72", Release: "396.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:4f56f0f2098313624fdcd567a5c644fc", SrcName: "perl-Exporter", SrcEpoch: 0, SrcVersion: "5.72", SrcRelease: "396.el8"},
				{Name: "perl-Carp", Epoch: 0, Version: "1.42", Release: "396.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:4b23c1071bd0bd3f46aa4bc06d3cc6bb", SrcName: "perl-Carp", SrcEpoch: 0, SrcVersion: "1.42", SrcRelease: "396.el8"},
				{Name: "perl-parent", Epoch: 1, Version: "0.237", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:3a1e8a3596d9a3ca9021a40c141adca3", SrcName: "perl-parent", SrcEpoch: 1, SrcVersion: "0.237", SrcRelease: "1.el8"},
				{Name: "nss-util", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:f4a99f6bc7e92ebfb9ae2470de3f6de4", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "nss-softokn", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:0245449c679c87662bab54d4f57b8956", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "nss-sysinit", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:094e2ee6a5c5ce63f17098268e8dc066", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "nss-softokn-freebl-devel", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:0d4300a7577be8b229ddb403bdbf79c9", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "perl-macros", Epoch: 4, Version: "5.26.3", Release: "416.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:ae436db14693a9b49ecd5d02ef877a13", SrcName: "perl", SrcEpoch: 4, SrcVersion: "5.26.3", SrcRelease: "416.el8"},
				{Name: "perl-Socket", Epoch: 4, Version: "2.027", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:6805c843a6f77fb26c9213320e0c54d1", SrcName: "perl-Socket", SrcEpoch: 4, SrcVersion: "2.027", SrcRelease: "3.el8"},
				{Name: "perl-Unicode-Normalize", Epoch: 0, Version: "1.25", Release: "396.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:8461c73c94cc877ae2103add615572f8", SrcName: "perl-Unicode-Normalize", SrcEpoch: 0, SrcVersion: "1.25", SrcRelease: "396.el8"},
				{Name: "perl-IO", Epoch: 0, Version: "1.38", Release: "416.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:24ea3921fbc00498cc592aa5fa5513c4", SrcName: "perl", SrcEpoch: 0, SrcVersion: "5.26.3", SrcRelease: "416.el8"},
				{Name: "perl-constant", Epoch: 0, Version: "1.33", Release: "396.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:dbb827cf7e21d66176d41c7280d1da8b", SrcName: "perl-constant", SrcEpoch: 0, SrcVersion: "1.33", SrcRelease: "396.el8"},
				{Name: "perl-threads-shared", Epoch: 0, Version: "1.58", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:f7ca1f9a2bb956e0fcb5c0921352b23c", SrcName: "perl-threads-shared", SrcEpoch: 0, SrcVersion: "1.58", SrcRelease: "2.el8"},
				{Name: "perl-MIME-Base64", Epoch: 0, Version: "3.15", Release: "396.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and MIT"}, Maintainer: "CentOS", Digest: "md5:8ed7ed3ebaf11f7a51f28e8602e60020", SrcName: "perl-MIME-Base64", SrcEpoch: 0, SrcVersion: "3.15", SrcRelease: "396.el8"},
				{Name: "perl-Time-Local", Epoch: 1, Version: "1.280", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:d9ff68ed7dce4a5bb9a13c85270f9908", SrcName: "perl-Time-Local", SrcEpoch: 1, SrcVersion: "1.280", SrcRelease: "1.el8"},
				{Name: "perl-Digest", Epoch: 0, Version: "1.17", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:3288d677385ef645566a458dae131af2", SrcName: "perl-Digest", SrcEpoch: 0, SrcVersion: "1.17", SrcRelease: "395.el8"},
				{Name: "perl-Net-SSLeay", Epoch: 0, Version: "1.88", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Artistic 2.0"}, Maintainer: "CentOS", Digest: "md5:d060dd8a7177e91cee9f963348ab8305", SrcName: "perl-Net-SSLeay", SrcEpoch: 0, SrcVersion: "1.88", SrcRelease: "1.el8"},
				{Name: "perl-TermReadKey", Epoch: 0, Version: "2.37", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(Copyright only) and (Artistic or GPL+)"}, Maintainer: "CentOS", Digest: "md5:1ca4492dd5719a92713930073a8fc89c", SrcName: "perl-TermReadKey", SrcEpoch: 0, SrcVersion: "2.37", SrcRelease: "7.el8"},
				{Name: "perl-Pod-Escapes", Epoch: 1, Version: "1.07", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:06bdeb217473ea5c468916d046240269", SrcName: "perl-Pod-Escapes", SrcEpoch: 1, SrcVersion: "1.07", SrcRelease: "395.el8"},
				{Name: "perl-Mozilla-CA", Epoch: 0, Version: "20160104", Release: "7.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:e517a072a8b0ef27b42a5472efb1bd17", SrcName: "perl-Mozilla-CA", SrcEpoch: 0, SrcVersion: "20160104", SrcRelease: "7.el8"},
				{Name: "fipscheck", Epoch: 0, Version: "1.5.0", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:33be21c19bde710c0b22e48c6ab19c91", SrcName: "fipscheck", SrcEpoch: 0, SrcVersion: "1.5.0", SrcRelease: "4.el8"},
				{Name: "which", Epoch: 0, Version: "2.21", Release: "12.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3"}, Maintainer: "CentOS", Digest: "md5:183da0ec6dab25a25210bb1be5733f46", SrcName: "which", SrcEpoch: 0, SrcVersion: "2.21", SrcRelease: "12.el8"},
				{Name: "libpsl", Epoch: 0, Version: "0.20.2", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:bdc70122875df1e028ba0654b6566384", SrcName: "libpsl", SrcEpoch: 0, SrcVersion: "0.20.2", SrcRelease: "5.el8"},
				{Name: "pcre2-utf32", Epoch: 0, Version: "10.32", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:793c13660ceec0a0f375a98c077b1144", SrcName: "pcre2", SrcEpoch: 0, SrcVersion: "10.32", SrcRelease: "1.el8"},
				{Name: "openssl", Epoch: 1, Version: "1.1.1c", Release: "15.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenSSL"}, Maintainer: "CentOS", Digest: "md5:a2d45c750a52617caa18d0f23c80534e", SrcName: "openssl", SrcEpoch: 1, SrcVersion: "1.1.1c", SrcRelease: "15.el8"},
				{Name: "perl-Term-Cap", Epoch: 0, Version: "1.17", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:a36d892cdd3e4c40b70ebf3ebd307ece", SrcName: "perl-Term-Cap", SrcEpoch: 0, SrcVersion: "1.17", SrcRelease: "395.el8"},
				{Name: "libpkgconf", Epoch: 0, Version: "1.4.2", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "CentOS", Digest: "md5:c1d476e8247ae535374ebffccfb2ca5c", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.el8"},
				{Name: "pkgconf-pkg-config", Epoch: 0, Version: "1.4.2", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "CentOS", Digest: "md5:6a5f9694de9b98ca71ed47b605ba977a", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.el8"},
				{Name: "nss-util-devel", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:7e8988fa06b53b21885b838eee5005ae", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "libcom_err-devel", Epoch: 0, Version: "1.45.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:64fde397ca1b863536fc9b1de81cdbfe", SrcName: "e2fsprogs", SrcEpoch: 0, SrcVersion: "1.45.4", SrcRelease: "3.el8"},
				{Name: "libverto-devel", Epoch: 0, Version: "0.3.0", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:532a02d519c64b120a5574d579b4f476", SrcName: "libverto", SrcEpoch: 0, SrcVersion: "0.3.0", SrcRelease: "5.el8"},
				{Name: "libselinux-devel", Epoch: 0, Version: "2.9", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:c8de4e1f3790622c27baf4a2807e4d2e", SrcName: "libselinux", SrcEpoch: 0, SrcVersion: "2.9", SrcRelease: "3.el8"},
				{Name: "libkadm5", Epoch: 0, Version: "1.17", Release: "18.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:5b42f084acf13e7c87d03a2327b62dc3", SrcName: "krb5", SrcEpoch: 0, SrcVersion: "1.17", SrcRelease: "18.el8"},
				{Name: "openssh-clients", Epoch: 0, Version: "8.0p1", Release: "4.el8_1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:78ac3735eebc747b931cc7712cc0e46f", SrcName: "openssh", SrcEpoch: 0, SrcVersion: "8.0p1", SrcRelease: "4.el8_1"},
				{Name: "git-core-doc", Epoch: 0, Version: "2.18.4", Release: "2.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:b5e58ac9279ac3502dc6f53feecb64c1", SrcName: "git", SrcEpoch: 0, SrcVersion: "2.18.4", SrcRelease: "2.el8_2"},
				{Name: "krb5-devel", Epoch: 0, Version: "1.17", Release: "18.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:d7ed22f52db48539363e1388925f0246", SrcName: "krb5", SrcEpoch: 0, SrcVersion: "1.17", SrcRelease: "18.el8"},
				{Name: "perl-Encode", Epoch: 4, Version: "2.97", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and Artistic 2.0 and UCD"}, Maintainer: "CentOS", Digest: "md5:9c13163ba1dba6de88e23119fbbb07c7", SrcName: "perl-Encode", SrcEpoch: 4, SrcVersion: "2.97", SrcRelease: "3.el8"},
				{Name: "perl-Getopt-Long", Epoch: 1, Version: "2.50", Release: "4.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:8402f4c723f38564bab926f618d4a638", SrcName: "perl-Getopt-Long", SrcEpoch: 1, SrcVersion: "2.50", SrcRelease: "4.el8"},
				{Name: "libgcc", Epoch: 0, Version: "8.3.1", Release: "5.el8.0.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:6d2466b49e2e159a9ffbc965d87539b5", SrcName: "gcc", SrcEpoch: 0, SrcVersion: "8.3.1", SrcRelease: "5.el8.0.2"},
				{Name: "perl-Pod-Usage", Epoch: 4, Version: "1.69", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:7ca3785d080ee4a20cb030cff8d14650", SrcName: "perl-Pod-Usage", SrcEpoch: 4, SrcVersion: "1.69", SrcRelease: "395.el8"},
				{Name: "python3-pip-wheel", Epoch: 0, Version: "9.0.3", Release: "16.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)"}, Maintainer: "CentOS", Digest: "md5:c7af612313ed1ff4b0831c8e2a811ef4", SrcName: "python-pip", SrcEpoch: 0, SrcVersion: "9.0.3", SrcRelease: "16.el8"},
				{Name: "perl-HTTP-Tiny", Epoch: 0, Version: "0.074", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:450794533f6287021ea470ccdb71a39a", SrcName: "perl-HTTP-Tiny", SrcEpoch: 0, SrcVersion: "0.074", SrcRelease: "1.el8"},
				{Name: "perl-libnet", Epoch: 0, Version: "3.11", Release: "3.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:84cf9f5f0d65cd76fdc6916aa342c1f3", SrcName: "perl-libnet", SrcEpoch: 0, SrcVersion: "3.11", SrcRelease: "3.el8"},
				{Name: "setup", Epoch: 0, Version: "2.12.2", Release: "5.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:d55f10b24cc01ff684bb2380bcc453b8", SrcName: "setup", SrcEpoch: 0, SrcVersion: "2.12.2", SrcRelease: "5.el8"},
				{Name: "file", Epoch: 0, Version: "5.33", Release: "13.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:814793b843457f406144e67030207c01", SrcName: "file", SrcEpoch: 0, SrcVersion: "5.33", SrcRelease: "13.el8"},
				{Name: "basesystem", Epoch: 0, Version: "11", Release: "5.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:f988bd6b76df626c40d6e051ee7629c9", SrcName: "basesystem", SrcEpoch: 0, SrcVersion: "11", SrcRelease: "5.el8"},
				{Name: "perl-Git", Epoch: 0, Version: "2.18.4", Release: "2.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:1186b063cb093bd7e40f35d6d6d2edd5", SrcName: "git", SrcEpoch: 0, SrcVersion: "2.18.4", SrcRelease: "2.el8_2"},
				{Name: "ncurses-base", Epoch: 0, Version: "6.1", Release: "7.20180224.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:b159cfc17482713cecfbfd8e031a91f2", SrcName: "ncurses", SrcEpoch: 0, SrcVersion: "6.1", SrcRelease: "7.20180224.el8"},
				{Name: "vim-filesystem", Epoch: 2, Version: "8.0.1763", Release: "13.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"Vim and MIT"}, Maintainer: "CentOS", Digest: "md5:747cf7810d58760c910bf51f0fae0722", SrcName: "vim", SrcEpoch: 2, SrcVersion: "8.0.1763", SrcRelease: "13.el8"},
				{Name: "libselinux", Epoch: 0, Version: "2.9", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:6ab87f69ce519580828da70906fc05a2", SrcName: "libselinux", SrcEpoch: 0, SrcVersion: "2.9", SrcRelease: "3.el8"},
				{Name: "gpm-libs", Epoch: 0, Version: "1.20.7", Release: "15.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and GPLv2+ with exceptions and GPLv3+ and Verbatim and Copyright only"}, Maintainer: "CentOS", Digest: "md5:897dcd70f96bda1474ee53eb454a307f", SrcName: "gpm", SrcEpoch: 0, SrcVersion: "1.20.7", SrcRelease: "15.el8"},
				{Name: "glibc-minimal-langpack", Epoch: 0, Version: "2.28", Release: "101.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:83a4dc4baf1266a1818a32d3115c3697", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.28", SrcRelease: "101.el8"},
				{Name: "file-devel", Epoch: 0, Version: "5.33", Release: "13.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:3bdc4b880ab807baac63dd9e32a18c17", SrcName: "file", SrcEpoch: 0, SrcVersion: "5.33", SrcRelease: "13.el8"},
				{Name: "glibc", Epoch: 0, Version: "2.28", Release: "101.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and LGPLv2+ with exceptions and GPLv2+ and GPLv2+ with exceptions and BSD and Inner-Net and ISC and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:5cb33fd16d51470238a7a4ec125016ba", SrcName: "glibc", SrcEpoch: 0, SrcVersion: "2.28", SrcRelease: "101.el8"},
				{Name: "nss-devel", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:b7a2c2d0f38735307380c0e6ad6ad433", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "libsepol", Epoch: 0, Version: "2.9", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:38605757e0d48009beca6e82c6bdc8d5", SrcName: "libsepol", SrcEpoch: 0, SrcVersion: "2.9", SrcRelease: "1.el8"},
				{Name: "xz-devel", Epoch: 0, Version: "5.2.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:03d0dba57b4185d8b1ee4fdecc7bdafc", SrcName: "xz", SrcEpoch: 0, SrcVersion: "5.2.4", SrcRelease: "3.el8"},
				{Name: "xz-libs", Epoch: 0, Version: "5.2.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Public Domain"}, Maintainer: "CentOS", Digest: "md5:96bf8a25552450c4992aeb824bb55c87", SrcName: "xz", SrcEpoch: 0, SrcVersion: "5.2.4", SrcRelease: "3.el8"},
				{Name: "wget", Epoch: 0, Version: "1.19.5", Release: "8.el8_1.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:dd754ca4ef35c74593a6d3c4785affa9", SrcName: "wget", SrcEpoch: 0, SrcVersion: "1.19.5", SrcRelease: "8.el8_1.1"},
				{Name: "libcap", Epoch: 0, Version: "2.26", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:5fe96e3190b4c9ae5bfcdb5268e7a69f", SrcName: "libcap", SrcEpoch: 0, SrcVersion: "2.26", SrcRelease: "3.el8"},
				{Name: "strace", Epoch: 0, Version: "4.24", Release: "9.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPL-2.1+ and GPL-2.0+"}, Maintainer: "CentOS", Digest: "md5:0d086902619eb98efc1c415c910d2064", SrcName: "strace", SrcEpoch: 0, SrcVersion: "4.24", SrcRelease: "9.el8"},
				{Name: "info", Epoch: 0, Version: "6.5", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:e368cb98b8ae037ff471725c2cea363e", SrcName: "texinfo", SrcEpoch: 0, SrcVersion: "6.5", SrcRelease: "6.el8"},
				{Name: "gdb-gdbserver", Epoch: 0, Version: "8.2", Release: "11.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and LGPLv3+ and BSD and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:f1e0d599b08052a503433bb9e4d19490", SrcName: "gdb", SrcEpoch: 0, SrcVersion: "8.2", SrcRelease: "11.el8"},
				{Name: "libcom_err", Epoch: 0, Version: "1.45.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:ec30a7983356b3de9e83f1dda2eecd8b", SrcName: "e2fsprogs", SrcEpoch: 0, SrcVersion: "1.45.4", SrcRelease: "3.el8"},
				{Name: "libcroco", Epoch: 0, Version: "0.6.12", Release: "4.el8_2.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2"}, Maintainer: "CentOS", Digest: "md5:15aa0ca9c1889bdda4537f342f8df60d", SrcName: "libcroco", SrcEpoch: 0, SrcVersion: "0.6.12", SrcRelease: "4.el8_2.1"},
				{Name: "libxml2", Epoch: 0, Version: "2.9.7", Release: "7.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:18a89ea76ea4278d068b24672a91f048", SrcName: "libxml2", SrcEpoch: 0, SrcVersion: "2.9.7", SrcRelease: "7.el8"},
				{Name: "libmpc", Epoch: 0, Version: "1.0.2", Release: "9.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ and GFDL"}, Maintainer: "CentOS", Digest: "md5:2c68aca463647ae457efae8232d80a86", SrcName: "libmpc", SrcEpoch: 0, SrcVersion: "1.0.2", SrcRelease: "9.el8"},
				{Name: "expat", Epoch: 0, Version: "2.2.5", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:0599f3c75b448b9c88e478b764361704", SrcName: "expat", SrcEpoch: 0, SrcVersion: "2.2.5", SrcRelease: "3.el8"},
				{Name: "gettext", Epoch: 0, Version: "0.19.8.1", Release: "17.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:b0c5f450b122286eb0d173e9c1aaa23d", SrcName: "gettext", SrcEpoch: 0, SrcVersion: "0.19.8.1", SrcRelease: "17.el8"},
				{Name: "libuuid", Epoch: 0, Version: "2.32.1", Release: "22.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:490942418c894fbc00e1bed23dd61096", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "22.el8"},
				{Name: "autoconf", Epoch: 0, Version: "2.69", Release: "27.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GFDL"}, Maintainer: "CentOS", Digest: "md5:9c5483a0a68d7fd73aec444ac2a9d631", SrcName: "autoconf", SrcEpoch: 0, SrcVersion: "2.69", SrcRelease: "27.el8"},
				{Name: "chkconfig", Epoch: 0, Version: "1.11", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:33ae796d1b96f631732e6553d5e67e26", SrcName: "chkconfig", SrcEpoch: 0, SrcVersion: "1.11", SrcRelease: "1.el8"},
				{Name: "kernel-headers", Epoch: 0, Version: "4.18.0", Release: "193.28.1.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and Redistributable, no modification permitted"}, Maintainer: "CentOS", Digest: "md5:cb027fddb5bfd3e90f75340b4b86a00b", SrcName: "kernel", SrcEpoch: 0, SrcVersion: "4.18.0", SrcRelease: "193.28.1.el8_2"},
				{Name: "gmp", Epoch: 1, Version: "6.1.2", Release: "10.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ or GPLv2+"}, Maintainer: "CentOS", Digest: "md5:d97c6201b7617cb78474f88a10f7da34", SrcName: "gmp", SrcEpoch: 1, SrcVersion: "6.1.2", SrcRelease: "10.el8"},
				{Name: "libxcrypt-devel", Epoch: 0, Version: "4.1.1", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and BSD and Public Domain"}, Maintainer: "CentOS", Digest: "md5:aec174890471f284aa6e03020ff76443", SrcName: "libxcrypt", SrcEpoch: 0, SrcVersion: "4.1.1", SrcRelease: "4.el8"},
				{Name: "libattr", Epoch: 0, Version: "2.4.48", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:bdc22fe0cdcb03a5014e6275c6539003", SrcName: "attr", SrcEpoch: 0, SrcVersion: "2.4.48", SrcRelease: "3.el8"},
				{Name: "gettext-common-devel", Epoch: 0, Version: "0.19.8.1", Release: "17.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:0383d8a86aba24f71729c918d949d967", SrcName: "gettext", SrcEpoch: 0, SrcVersion: "0.19.8.1", SrcRelease: "17.el8"},
				{Name: "coreutils-single", Epoch: 0, Version: "8.30", Release: "7.el8_2.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:68abfec2549d36f258bd2c2e513fbddc", SrcName: "coreutils", SrcEpoch: 0, SrcVersion: "8.30", SrcRelease: "7.el8_2.1"},
				{Name: "automake", Epoch: 0, Version: "1.16.1", Release: "6.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GFDL and Public Domain and MIT"}, Maintainer: "CentOS", Digest: "md5:82fb5fbeda4c984163fa56c0d87f88d2", SrcName: "automake", SrcEpoch: 0, SrcVersion: "1.16.1", SrcRelease: "6.el8"},
				{Name: "libblkid", Epoch: 0, Version: "2.32.1", Release: "22.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:081c1805273069f7933c6a2bc62f2a5b", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "22.el8"},
				{Name: "gcc", Epoch: 0, Version: "8.3.1", Release: "5.el8.0.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ with exceptions and LGPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:3380ab4e7a454e71dc39e99c060dfcda", SrcName: "gcc", SrcEpoch: 0, SrcVersion: "8.3.1", SrcRelease: "5.el8.0.2"},
				{Name: "libcap-ng", Epoch: 0, Version: "0.7.9", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:dbb245f0f1383ee7e20511a832a9caeb", SrcName: "libcap-ng", SrcEpoch: 0, SrcVersion: "0.7.9", SrcRelease: "5.el8"},
				{Name: "gettext-devel", Epoch: 0, Version: "0.19.8.1", Release: "17.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and GPLv3+"}, Maintainer: "CentOS", Digest: "md5:bdc16337577ae7ba9540b00bfd49fe75", SrcName: "gettext", SrcEpoch: 0, SrcVersion: "0.19.8.1", SrcRelease: "17.el8"},
				{Name: "libffi", Epoch: 0, Version: "3.1", Release: "21.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:75f0d6ae08ec69f1d9c600ce05bfad37", SrcName: "libffi", SrcEpoch: 0, SrcVersion: "3.1", SrcRelease: "21.el8"},
				{Name: "make", Epoch: 1, Version: "4.2.1", Release: "10.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:5e35f9e2dafcf4717463a5cccd6363eb", SrcName: "make", SrcEpoch: 1, SrcVersion: "4.2.1", SrcRelease: "10.el8"},
				{Name: "libzstd", Epoch: 0, Version: "1.4.2", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD and GPLv2"}, Maintainer: "CentOS", Digest: "md5:6cc827a8449f9baed1dbad4f5992d2a1", SrcName: "zstd", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "2.el8"},
				{Name: "npm", Epoch: 1, Version: "6.14.4", Release: "1.10.21.0.3.module_el8.2.0+391+8da3adc6", Arch: "x86_64", Modularitylabel: "nodejs:10:8020020200707141642:6a468ee4", Licenses: []string{"MIT and ASL 2.0 and ISC and BSD"}, Maintainer: "CentOS", Digest: "md5:c36d61a68ea091683d7a2c88e19239c1", SrcName: "nodejs", SrcEpoch: 1, SrcVersion: "10.21.0", SrcRelease: "3.module_el8.2.0+391+8da3adc6"},
				{Name: "lz4-libs", Epoch: 0, Version: "1.8.1.2", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and BSD"}, Maintainer: "CentOS", Digest: "md5:a5e16dd8650587ebf1e85350bb7fb063", SrcName: "lz4", SrcEpoch: 0, SrcVersion: "1.8.1.2", SrcRelease: "4.el8"},
				{Name: "libtool-ltdl", Epoch: 0, Version: "2.4.6", Release: "25.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:e5553e445ddba10c79d2f430439f24cb", SrcName: "libtool", SrcEpoch: 0, SrcVersion: "2.4.6", SrcRelease: "25.el8"},
				{Name: "libgcrypt", Epoch: 0, Version: "1.8.3", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:453cb9a4dcfba2847247a6004adb8a40", SrcName: "libgcrypt", SrcEpoch: 0, SrcVersion: "1.8.3", SrcRelease: "4.el8"},
				{Name: "libipt", Epoch: 0, Version: "1.6.1", Release: "8.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:68edc160e21686087d8486cdf060073e", SrcName: "libipt", SrcEpoch: 0, SrcVersion: "1.6.1", SrcRelease: "8.el8"},
				{Name: "cracklib", Epoch: 0, Version: "2.9.6", Release: "15.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:3ced50b8ce5c91849d67c8e6f75f5cd1", SrcName: "cracklib", SrcEpoch: 0, SrcVersion: "2.9.6", SrcRelease: "15.el8"},
				{Name: "gc", Epoch: 0, Version: "7.6.4", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:d16a971513963ab4666f0aef05b733b6", SrcName: "gc", SrcEpoch: 0, SrcVersion: "7.6.4", SrcRelease: "3.el8"},
				{Name: "libidn2", Epoch: 0, Version: "2.2.0", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or LGPLv3+) and GPLv3+"}, Maintainer: "CentOS", Digest: "md5:206b757710d3924958eeb8bd423243bd", SrcName: "libidn2", SrcEpoch: 0, SrcVersion: "2.2.0", SrcRelease: "1.el8"},
				{Name: "gdb-headless", Epoch: 0, Version: "8.2", Release: "12.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GPLv3+ with exceptions and GPLv2+ and GPLv2+ with exceptions and GPL+ and LGPLv2+ and LGPLv3+ and BSD and Public Domain and GFDL"}, Maintainer: "CentOS", Digest: "md5:35ade298b366dbb4e1a451d391957065", SrcName: "gdb", SrcEpoch: 0, SrcVersion: "8.2", SrcRelease: "12.el8"},
				{Name: "file-libs", Epoch: 0, Version: "5.33", Release: "13.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:99f7be5b7e16f6c8881a5f0c7201d727", SrcName: "file", SrcEpoch: 0, SrcVersion: "5.33", SrcRelease: "13.el8"},
				{Name: "epel-release", Epoch: 0, Version: "8", Release: "8.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:ffafcace455ddb4eff21913fabc48caa", SrcName: "epel-release", SrcEpoch: 0, SrcVersion: "8", SrcRelease: "8.el8"},
				{Name: "keyutils-libs", Epoch: 0, Version: "1.5.10", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:abf58d0542bb7f9113858fce1e2aa9b4", SrcName: "keyutils", SrcEpoch: 0, SrcVersion: "1.5.10", SrcRelease: "6.el8"},
				{Name: "platform-python-pip", Epoch: 0, Version: "9.0.3", Release: "18.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MIT and Python and ASL 2.0 and BSD and ISC and LGPLv2 and MPLv2.0 and (ASL 2.0 or BSD)"}, Maintainer: "CentOS", Digest: "md5:a1a1e7b786b556224de6cb0559470b94", SrcName: "python-pip", SrcEpoch: 0, SrcVersion: "9.0.3", SrcRelease: "18.el8"},
				{Name: "p11-kit-trust", Epoch: 0, Version: "0.23.14", Release: "5.el8_0", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:801f7a4fef4a8c568c336fdfb3031b21", SrcName: "p11-kit", SrcEpoch: 0, SrcVersion: "0.23.14", SrcRelease: "5.el8_0"},
				{Name: "python36", Epoch: 0, Version: "3.6.8", Release: "2.module_el8.3.0+562+e162826a", Arch: "x86_64", Modularitylabel: "python36:3.6:8030020201104034153:24f1489c", Licenses: []string{"Python"}, Maintainer: "CentOS", Digest: "md5:37556e1c897b3116f014573121cb26f4", SrcName: "python36", SrcEpoch: 0, SrcVersion: "3.6.8", SrcRelease: "2.module_el8.3.0+562+e162826a"},
				{Name: "pcre", Epoch: 0, Version: "8.42", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:9c05cc476028e5b01edd29104ec2e9b3", SrcName: "pcre", SrcEpoch: 0, SrcVersion: "8.42", SrcRelease: "4.el8"},
				{Name: "python2-setuptools-wheel", Epoch: 0, Version: "39.0.1", Release: "12.module_el8.3.0+478+7570e00c", Arch: "noarch", Modularitylabel: "python27:2.7:8030020200831201838:851f4228", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:920a10248c3d9f8b8d9a6e456259a58c", SrcName: "python2-setuptools", SrcEpoch: 0, SrcVersion: "39.0.1", SrcRelease: "12.module_el8.3.0+478+7570e00c"},
				{Name: "systemd-libs", Epoch: 0, Version: "239", Release: "31.el8_2.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT"}, Maintainer: "CentOS", Digest: "md5:a316b3f18d3fd2ae1a08e2a877f41ae7", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "239", SrcRelease: "31.el8_2.2"},
				{Name: "python2-libs", Epoch: 0, Version: "2.7.17", Release: "2.module_el8.3.0+478+7570e00c", Arch: "x86_64", Modularitylabel: "python27:2.7:8030020200831201838:851f4228", Licenses: []string{"Python"}, Maintainer: "CentOS", Digest: "md5:5530a6dceb2fef1bd62c82e9933dff3c", SrcName: "python2", SrcEpoch: 0, SrcVersion: "2.7.17", SrcRelease: "2.module_el8.3.0+478+7570e00c"},
				{Name: "dbus-tools", Epoch: 1, Version: "1.12.8", Release: "10.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:7d9ee2812adf87829a32fb0b026fe137", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.8", SrcRelease: "10.el8_2"},
				{Name: "python2-setuptools", Epoch: 0, Version: "39.0.1", Release: "12.module_el8.3.0+478+7570e00c", Arch: "noarch", Modularitylabel: "python27:2.7:8030020200831201838:851f4228", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:2bde3ee3de071c004b14a3faac2ab8f0", SrcName: "python2-setuptools", SrcEpoch: 0, SrcVersion: "39.0.1", SrcRelease: "12.module_el8.3.0+478+7570e00c"},
				{Name: "libusbx", Epoch: 0, Version: "1.0.22", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:8ccdca006e6ce8c7c62ac19db25e7734", SrcName: "libusbx", SrcEpoch: 0, SrcVersion: "1.0.22", SrcRelease: "1.el8"},
				{Name: "gpg-pubkey", Epoch: 0, Version: "ce977fe0", Release: "5db1f171", Arch: "None", Modularitylabel: "", Licenses: []string{"pubkey"}, Maintainer: "", Digest: "", SrcName: "", SrcEpoch: 0, SrcVersion: "", SrcRelease: ""},
				{Name: "rpm-libs", Epoch: 0, Version: "4.14.3", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+ with exceptions"}, Maintainer: "CentOS", Digest: "md5:35f8555f8534e5acf6ca14e950fe0635", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.3", SrcRelease: "4.el8"},
				{Name: "squashfs-tools", Epoch: 0, Version: "4.3", Release: "19.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:23db010940aed94245b743afe8aabfec", SrcName: "squashfs-tools", SrcEpoch: 0, SrcVersion: "4.3", SrcRelease: "19.el8"},
				{Name: "rpm-build-libs", Epoch: 0, Version: "4.14.3", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+ with exceptions"}, Maintainer: "CentOS", Digest: "md5:66f97999bf52dc143aeaedf4750bc001", SrcName: "rpm", SrcEpoch: 0, SrcVersion: "4.14.3", SrcRelease: "4.el8"},
				{Name: "libsemanage", Epoch: 0, Version: "2.9", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:dabbf5bee55cc3853ae33c8e739c59b1", SrcName: "libsemanage", SrcEpoch: 0, SrcVersion: "2.9", SrcRelease: "2.el8"},
				{Name: "libutempter", Epoch: 0, Version: "1.1.6", Release: "14.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:d8773aceac7955c99c3d89a95568c180", SrcName: "libutempter", SrcEpoch: 0, SrcVersion: "1.1.6", SrcRelease: "14.el8"},
				{Name: "acl", Epoch: 0, Version: "2.2.53", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:483792b8b5f9eb8be7dc4407733118d0", SrcName: "acl", SrcEpoch: 0, SrcVersion: "2.2.53", SrcRelease: "1.el8"},
				{Name: "nettle", Epoch: 0, Version: "3.4.1", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv3+ or GPLv2+"}, Maintainer: "CentOS", Digest: "md5:6bf682f2d691cb408aa61a2601d0f922", SrcName: "nettle", SrcEpoch: 0, SrcVersion: "3.4.1", SrcRelease: "1.el8"},
				{Name: "libcomps", Epoch: 0, Version: "0.1.11", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:61f1c2ca663b8226890e72a85e94f058", SrcName: "libcomps", SrcEpoch: 0, SrcVersion: "0.1.11", SrcRelease: "4.el8"},
				{Name: "findutils", Epoch: 1, Version: "4.6.0", Release: "20.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:5ca7479cd7d1cdbe30746ce1c53d6bc1", SrcName: "findutils", SrcEpoch: 1, SrcVersion: "4.6.0", SrcRelease: "20.el8"},
				{Name: "cpio", Epoch: 0, Version: "2.12", Release: "8.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:7ddd4e1548f05e3884f62166277480e7", SrcName: "cpio", SrcEpoch: 0, SrcVersion: "2.12", SrcRelease: "8.el8"},
				{Name: "ipcalc", Epoch: 0, Version: "0.2.4", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:4564057e3ae90a2296967da77f71cb87", SrcName: "ipcalc", SrcEpoch: 0, SrcVersion: "0.2.4", SrcRelease: "4.el8"},
				{Name: "libnghttp2", Epoch: 0, Version: "1.33.0", Release: "3.el8_2.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:54279556ffeb8d7a527c7d6d157965e3", SrcName: "nghttp2", SrcEpoch: 0, SrcVersion: "1.33.0", SrcRelease: "3.el8_2.1"},
				{Name: "iptables-libs", Epoch: 0, Version: "1.8.4", Release: "10.el8_2.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and Artistic 2.0 and ISC"}, Maintainer: "CentOS", Digest: "md5:5e4704a82f9b3370b941fddcb34c8031", SrcName: "iptables", SrcEpoch: 0, SrcVersion: "1.8.4", SrcRelease: "10.el8_2.1"},
				{Name: "libsigsegv", Epoch: 0, Version: "2.11", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:5625ecad632168eecb8960821b0abeca", SrcName: "libsigsegv", SrcEpoch: 0, SrcVersion: "2.11", SrcRelease: "5.el8"},
				{Name: "libverto", Epoch: 0, Version: "0.3.0", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:4b04a34ead82e762637fae7d79adf926", SrcName: "libverto", SrcEpoch: 0, SrcVersion: "0.3.0", SrcRelease: "5.el8"},
				{Name: "libtirpc", Epoch: 0, Version: "1.1.4", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"SISSL and BSD"}, Maintainer: "CentOS", Digest: "md5:8b4eec2ef36446061ee886a625584830", SrcName: "libtirpc", SrcEpoch: 0, SrcVersion: "1.1.4", SrcRelease: "4.el8"},
				{Name: "openssl-libs", Epoch: 1, Version: "1.1.1c", Release: "15.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenSSL"}, Maintainer: "CentOS", Digest: "md5:035906e9d19890fe1f95b1e8d4b1a890", SrcName: "openssl", SrcEpoch: 1, SrcVersion: "1.1.1c", SrcRelease: "15.el8"},
				{Name: "python3-libs", Epoch: 0, Version: "3.6.8", Release: "23.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"Python"}, Maintainer: "CentOS", Digest: "md5:24366dc108d8d78edaf5a0297620741c", SrcName: "python3", SrcEpoch: 0, SrcVersion: "3.6.8", SrcRelease: "23.el8"},
				{Name: "libpwquality", Epoch: 0, Version: "1.4.0", Release: "9.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD or GPLv2+"}, Maintainer: "CentOS", Digest: "md5:5b6c127112b13b51d39696fe3eb6ae50", SrcName: "libpwquality", SrcEpoch: 0, SrcVersion: "1.4.0", SrcRelease: "9.el8"},
				{Name: "util-linux", Epoch: 0, Version: "2.32.1", Release: "22.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2 and GPLv2+ and LGPLv2+ and BSD with advertising and Public Domain"}, Maintainer: "CentOS", Digest: "md5:f6b9e0111f2f92d2a3cfe3091237a049", SrcName: "util-linux", SrcEpoch: 0, SrcVersion: "2.32.1", SrcRelease: "22.el8"},
				{Name: "glib2", Epoch: 0, Version: "2.56.4", Release: "8.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:5cd98b131a43e30a121edb9f26697a57", SrcName: "glib2", SrcEpoch: 0, SrcVersion: "2.56.4", SrcRelease: "8.el8"},
				{Name: "iproute", Epoch: 0, Version: "5.3.0", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and Public Domain"}, Maintainer: "CentOS", Digest: "md5:d8fba74d3a00fae03acc06cf82eac90b", SrcName: "iproute", SrcEpoch: 0, SrcVersion: "5.3.0", SrcRelease: "1.el8"},
				{Name: "kmod", Epoch: 0, Version: "25", Release: "16.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:57442c44df12acfc5a6164cbd3780511", SrcName: "kmod", SrcEpoch: 0, SrcVersion: "25", SrcRelease: "16.el8"},
				{Name: "curl", Epoch: 0, Version: "7.61.1", Release: "12.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:c305531060535f28771ea538447cc0a9", SrcName: "curl", SrcEpoch: 0, SrcVersion: "7.61.1", SrcRelease: "12.el8"},
				{Name: "openldap", Epoch: 0, Version: "2.4.46", Release: "11.el8_1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"OpenLDAP"}, Maintainer: "CentOS", Digest: "md5:435b641c2259efbca591c3692fb3ee4f", SrcName: "openldap", SrcEpoch: 0, SrcVersion: "2.4.46", SrcRelease: "11.el8_1"},
				{Name: "python3-libcomps", Epoch: 0, Version: "0.1.11", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:5b6f724e320aac06d2dbb21171455287", SrcName: "libcomps", SrcEpoch: 0, SrcVersion: "0.1.11", SrcRelease: "4.el8"},
				{Name: "libarchive", Epoch: 0, Version: "3.3.2", Release: "8.el8_1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:3e813ad4d33701372cb8dcbf7e13bab6", SrcName: "libarchive", SrcEpoch: 0, SrcVersion: "3.3.2", SrcRelease: "8.el8_1"},
				{Name: "libyaml", Epoch: 0, Version: "0.1.7", Release: "5.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:7c8bef326f7edfab7928fbaaeee8c652", SrcName: "libyaml", SrcEpoch: 0, SrcVersion: "0.1.7", SrcRelease: "5.el8"},
				{Name: "npth", Epoch: 0, Version: "1.5", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:b944c570e2db54dd8f8d3ab4d7b7990d", SrcName: "npth", SrcEpoch: 0, SrcVersion: "1.5", SrcRelease: "4.el8"},
				{Name: "gpgme", Epoch: 0, Version: "1.10.0", Release: "6.el8.0.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:23c61df97de39d050a78bfe752b271fe", SrcName: "gpgme", SrcEpoch: 0, SrcVersion: "1.10.0", SrcRelease: "6.el8.0.1"},
				{Name: "libdnf", Epoch: 0, Version: "0.39.1", Release: "6.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:fffff8e4851ab715207b238f149bbbdd", SrcName: "libdnf", SrcEpoch: 0, SrcVersion: "0.39.1", SrcRelease: "6.el8_2"},
				{Name: "python3-hawkey", Epoch: 0, Version: "0.39.1", Release: "6.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:b91b61327f7e96d7cfc60c43131e344c", SrcName: "libdnf", SrcEpoch: 0, SrcVersion: "0.39.1", SrcRelease: "6.el8_2"},
				{Name: "libreport-filesystem", Epoch: 0, Version: "2.9.5", Release: "10.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:688a38734bfb36dcd3aa27147e733f48", SrcName: "libreport", SrcEpoch: 0, SrcVersion: "2.9.5", SrcRelease: "10.el8"},
				{Name: "dhcp-common", Epoch: 12, Version: "4.3.6", Release: "40.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "CentOS", Digest: "md5:cff0499703ca84229ef6cdb351b72248", SrcName: "dhcp", SrcEpoch: 12, SrcVersion: "4.3.6", SrcRelease: "40.el8"},
				{Name: "dbus-daemon", Epoch: 1, Version: "1.12.8", Release: "10.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:5650c215b979d90eda12fbc0f3029c02", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.8", SrcRelease: "10.el8_2"},
				{Name: "device-mapper-libs", Epoch: 8, Version: "1.02.169", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2"}, Maintainer: "CentOS", Digest: "md5:5efcaa95e0028e060cbb5e8654429d5c", SrcName: "lvm2", SrcEpoch: 8, SrcVersion: "2.03.08", SrcRelease: "3.el8"},
				{Name: "elfutils-default-yama-scope", Epoch: 0, Version: "0.178", Release: "7.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ or LGPLv3+"}, Maintainer: "CentOS", Digest: "md5:d610d0c6277635bcb67b234bc252e3ab", SrcName: "elfutils", SrcEpoch: 0, SrcVersion: "0.178", SrcRelease: "7.el8"},
				{Name: "systemd-pam", Epoch: 0, Version: "239", Release: "31.el8_2.2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+ and MIT and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:c7bcefdda4c92a3ddce3361e73a6fb04", SrcName: "systemd", SrcEpoch: 0, SrcVersion: "239", SrcRelease: "31.el8_2.2"},
				{Name: "dbus", Epoch: 1, Version: "1.12.8", Release: "10.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPLv2+ or AFL) and GPLv2+"}, Maintainer: "CentOS", Digest: "md5:0527c3681f77bd4775f23298020c568c", SrcName: "dbus", SrcEpoch: 1, SrcVersion: "1.12.8", SrcRelease: "10.el8_2"},
				{Name: "dhcp-client", Epoch: 12, Version: "4.3.6", Release: "40.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "CentOS", Digest: "md5:188f208523e4eff06746a39355aae3f9", SrcName: "dhcp", SrcEpoch: 12, SrcVersion: "4.3.6", SrcRelease: "40.el8"},
				{Name: "libkcapi-hmaccalc", Epoch: 0, Version: "1.1.1", Release: "16_1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD or GPLv2"}, Maintainer: "CentOS", Digest: "md5:386b48136f3955caddf97c0526e0fee0", SrcName: "libkcapi", SrcEpoch: 0, SrcVersion: "1.1.1", SrcRelease: "16_1.el8"},
				{Name: "dracut", Epoch: 0, Version: "049", Release: "70.git20200228.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:6ebe6c201c226b6df68e38a6d9e2c0c3", SrcName: "dracut", SrcEpoch: 0, SrcVersion: "049", SrcRelease: "70.git20200228.el8"},
				{Name: "dracut-squash", Epoch: 0, Version: "049", Release: "70.git20200228.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:477bd22fb0a32db1be3435b173f6e8a7", SrcName: "dracut", SrcEpoch: 0, SrcVersion: "049", SrcRelease: "70.git20200228.el8"},
				{Name: "dnf", Epoch: 0, Version: "4.2.17", Release: "7.el8_2", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ and GPLv2 and GPL"}, Maintainer: "CentOS", Digest: "md5:d11048989f73ec620c1e1d1df594e893", SrcName: "dnf", SrcEpoch: 0, SrcVersion: "4.2.17", SrcRelease: "7.el8_2"},
				{Name: "kexec-tools", Epoch: 0, Version: "2.0.20", Release: "14.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:a826184bec8587e019019077336578fa", SrcName: "kexec-tools", SrcEpoch: 0, SrcVersion: "2.0.20", SrcRelease: "14.el8"},
				{Name: "tar", Epoch: 2, Version: "1.30", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+"}, Maintainer: "CentOS", Digest: "md5:9cf6795c42907b004d73bbb404a037fc", SrcName: "tar", SrcEpoch: 2, SrcVersion: "1.30", SrcRelease: "4.el8"},
				{Name: "hostname", Epoch: 0, Version: "3.20", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:6f9f94b9611168a6baa489eeae0d1f05", SrcName: "hostname", SrcEpoch: 0, SrcVersion: "3.20", SrcRelease: "6.el8"},
				{Name: "langpacks-en", Epoch: 0, Version: "1.0", Release: "12.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+"}, Maintainer: "CentOS", Digest: "md5:170167632b0fe83fe2b3f2b1296109e2", SrcName: "langpacks", SrcEpoch: 0, SrcVersion: "1.0", SrcRelease: "12.el8"},
				{Name: "gpg-pubkey", Epoch: 0, Version: "8483c65d", Release: "5ccc5b19", Arch: "None", Modularitylabel: "", Licenses: []string{"pubkey"}, Maintainer: "", Digest: "", SrcName: "", SrcEpoch: 0, SrcVersion: "", SrcRelease: ""},
				{Name: "centos-release", Epoch: 0, Version: "8.2", Release: "2.2004.0.2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:f028acfc4904a465f8e47a49e8a8fc05", SrcName: "centos-release", SrcEpoch: 0, SrcVersion: "8.2", SrcRelease: "2.2004.0.2.el8"},
				{Name: "zlib", Epoch: 0, Version: "1.2.11", Release: "16.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"zlib and Boost"}, Maintainer: "CentOS", Digest: "md5:43b76aa847c359e8e634127cc2e8ed28", SrcName: "zlib", SrcEpoch: 0, SrcVersion: "1.2.11", SrcRelease: "16.el8_2"},
				{Name: "librepo", Epoch: 0, Version: "1.11.0", Release: "3.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:e57245767bd4ce9a832b412d56aff46e", SrcName: "librepo", SrcEpoch: 0, SrcVersion: "1.11.0", SrcRelease: "3.el8_2"},
				{Name: "bind-export-libs", Epoch: 32, Version: "9.11.13", Release: "6.el8_2.1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:bb03154a73645929b9521e1c6cea762b", SrcName: "bind", SrcEpoch: 32, SrcVersion: "9.11.13", SrcRelease: "6.el8_2.1"},
				{Name: "perl-libs", Epoch: 4, Version: "5.26.3", Release: "416.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and HSRL and MIT and UCD"}, Maintainer: "CentOS", Digest: "md5:6a77515e3417539336ed3f49908f83c7", SrcName: "perl", SrcEpoch: 4, SrcVersion: "5.26.3", SrcRelease: "416.el8"},
				{Name: "perl-Scalar-List-Utils", Epoch: 3, Version: "1.49", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:ae9fd554ccfa042dc074779abf5f2e58", SrcName: "perl-Scalar-List-Utils", SrcEpoch: 3, SrcVersion: "1.49", SrcRelease: "2.el8"},
				{Name: "nspr", Epoch: 0, Version: "4.25.0", Release: "2.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:1a03fabf3824c5fa596f3160a89d3aa1", SrcName: "nspr", SrcEpoch: 0, SrcVersion: "4.25.0", SrcRelease: "2.el8_2"},
				{Name: "nss-softokn-freebl", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:2dc45b206ef714ffe6914e0af559ddf8", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "nss", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:e5a5911dab0c66f9fd756f7845278a8d", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "perl-Text-ParseWords", Epoch: 0, Version: "3.30", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:bf3b84c83bffa76bd3af79141fe72394", SrcName: "perl-Text-ParseWords", SrcEpoch: 0, SrcVersion: "3.30", SrcRelease: "395.el8"},
				{Name: "perl-Term-ANSIColor", Epoch: 0, Version: "4.06", Release: "396.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:c0fc2f73ed2179d13f01ef64e42008ad", SrcName: "perl-Term-ANSIColor", SrcEpoch: 0, SrcVersion: "4.06", SrcRelease: "396.el8"},
				{Name: "perl-Errno", Epoch: 0, Version: "1.28", Release: "416.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:f0355f3f5fcd20f240b9a59e0f0dff88", SrcName: "perl", SrcEpoch: 0, SrcVersion: "5.26.3", SrcRelease: "416.el8"},
				{Name: "perl-Text-Tabs+Wrap", Epoch: 0, Version: "2013.0523", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"TTWL"}, Maintainer: "CentOS", Digest: "md5:68fe36e969406f0e90c6ee13b59b03e3", SrcName: "perl-Text-Tabs+Wrap", SrcEpoch: 0, SrcVersion: "2013.0523", SrcRelease: "395.el8"},
				{Name: "perl-File-Path", Epoch: 0, Version: "2.15", Release: "2.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:26125ecafbd51e07539b96c2699308bd", SrcName: "perl-File-Path", SrcEpoch: 0, SrcVersion: "2.15", SrcRelease: "2.el8"},
				{Name: "perl-PathTools", Epoch: 0, Version: "3.74", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and BSD"}, Maintainer: "CentOS", Digest: "md5:7cbc1c39bf3c5c7425bf73e5235d00d3", SrcName: "perl-PathTools", SrcEpoch: 0, SrcVersion: "3.74", SrcRelease: "1.el8"},
				{Name: "perl-threads", Epoch: 1, Version: "2.21", Release: "2.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:1a94ed19723e65a29945e59d8dfb5cf4", SrcName: "perl-threads", SrcEpoch: 1, SrcVersion: "2.21", SrcRelease: "2.el8"},
				{Name: "perl-interpreter", Epoch: 4, Version: "5.26.3", Release: "416.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and (GPLv2+ or Artistic) and BSD and Public Domain and UCD"}, Maintainer: "CentOS", Digest: "md5:e6124e5f67e224ff12cd136480af150a", SrcName: "perl", SrcEpoch: 4, SrcVersion: "5.26.3", SrcRelease: "416.el8"},
				{Name: "perl-IO-Socket-IP", Epoch: 0, Version: "0.39", Release: "5.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:9db2eaec9a186e8a7b92990683a12044", SrcName: "perl-IO-Socket-IP", SrcEpoch: 0, SrcVersion: "0.39", SrcRelease: "5.el8"},
				{Name: "perl-File-Temp", Epoch: 0, Version: "0.230.600", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:4a996a6bcd7b0cda3b3d8dc5b4e712fe", SrcName: "perl-File-Temp", SrcEpoch: 0, SrcVersion: "0.230.600", SrcRelease: "1.el8"},
				{Name: "perl-Digest-MD5", Epoch: 0, Version: "2.55", Release: "396.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and BSD"}, Maintainer: "CentOS", Digest: "md5:baf4430def945ca345c13bd147f60e7c", SrcName: "perl-Digest-MD5", SrcEpoch: 0, SrcVersion: "2.55", SrcRelease: "396.el8"},
				{Name: "perl-Error", Epoch: 1, Version: "0.17025", Release: "2.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"(GPL+ or Artistic) and MIT"}, Maintainer: "CentOS", Digest: "md5:235ea1ae7ab96868efe5090a4eb7bf1b", SrcName: "perl-Error", SrcEpoch: 1, SrcVersion: "0.17025", SrcRelease: "2.el8"},
				{Name: "perl-Data-Dumper", Epoch: 0, Version: "2.167", Release: "399.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:e7ff09d4e428471fd85587fbe9260fca", SrcName: "perl-Data-Dumper", SrcEpoch: 0, SrcVersion: "2.167", SrcRelease: "399.el8"},
				{Name: "perl-Storable", Epoch: 1, Version: "3.11", Release: "3.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:7575af2e91c5f435c050e1b3f453c9f3", SrcName: "perl-Storable", SrcEpoch: 1, SrcVersion: "3.11", SrcRelease: "3.el8"},
				{Name: "fipscheck-lib", Epoch: 0, Version: "1.5.0", Release: "4.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:9c891d4da9388c333268d7f2e3a9bcb1", SrcName: "fipscheck", SrcEpoch: 0, SrcVersion: "1.5.0", SrcRelease: "4.el8"},
				{Name: "openssh", Epoch: 0, Version: "8.0p1", Release: "4.el8_1", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:cbc935bae411018f5812f57975b259f3", SrcName: "openssh", SrcEpoch: 0, SrcVersion: "8.0p1", SrcRelease: "4.el8_1"},
				{Name: "publicsuffix-list-dafsa", Epoch: 0, Version: "20180723", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:8ce7a82bce196a11a42960b007ef0867", SrcName: "publicsuffix-list", SrcEpoch: 0, SrcVersion: "20180723", SrcRelease: "1.el8"},
				{Name: "pkgconf-m4", Epoch: 0, Version: "1.4.2", Release: "1.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPLv2+ with exceptions"}, Maintainer: "CentOS", Digest: "md5:398894375a82b4e1d5f7f0b38d3bec99", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.el8"},
				{Name: "pcre2-utf16", Epoch: 0, Version: "10.32", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:019c499b3107a465297726933eb4c78f", SrcName: "pcre2", SrcEpoch: 0, SrcVersion: "10.32", SrcRelease: "1.el8"},
				{Name: "ncurses", Epoch: 0, Version: "6.1", Release: "7.20180224.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MIT"}, Maintainer: "CentOS", Digest: "md5:93bfe403590240ca37823287b0cb4669", SrcName: "ncurses", SrcEpoch: 0, SrcVersion: "6.1", SrcRelease: "7.20180224.el8"},
				{Name: "libsecret", Epoch: 0, Version: "0.18.6", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:debd45dfee226d419b9cb0f66242aa5b", SrcName: "libsecret", SrcEpoch: 0, SrcVersion: "0.18.6", SrcRelease: "1.el8"},
				{Name: "pkgconf", Epoch: 0, Version: "1.4.2", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"ISC"}, Maintainer: "CentOS", Digest: "md5:c5c251d6f960076fcbae5a6e61fe23d1", SrcName: "pkgconf", SrcEpoch: 0, SrcVersion: "1.4.2", SrcRelease: "1.el8"},
				{Name: "nspr-devel", Epoch: 0, Version: "4.25.0", Release: "2.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:abe6ed3ed16d2bd85698372a81a6f782", SrcName: "nspr", SrcEpoch: 0, SrcVersion: "4.25.0", SrcRelease: "2.el8_2"},
				{Name: "nss-softokn-devel", Epoch: 0, Version: "3.53.1", Release: "11.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"MPLv2.0"}, Maintainer: "CentOS", Digest: "md5:4967426f9704426480fdb4df4e0991f0", SrcName: "nss", SrcEpoch: 0, SrcVersion: "3.53.1", SrcRelease: "11.el8_2"},
				{Name: "libsepol-devel", Epoch: 0, Version: "2.9", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:74728353391a1f9393e0bb7ec7f8e1c8", SrcName: "libsepol", SrcEpoch: 0, SrcVersion: "2.9", SrcRelease: "1.el8"},
				{Name: "pcre2-devel", Epoch: 0, Version: "10.32", Release: "1.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:8550def6c3764e92e2ff20d036fa30e7", SrcName: "pcre2", SrcEpoch: 0, SrcVersion: "10.32", SrcRelease: "1.el8"},
				{Name: "zlib-devel", Epoch: 0, Version: "1.2.11", Release: "16.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"zlib and Boost"}, Maintainer: "CentOS", Digest: "md5:a27299e14cacfca9442fdff1604139f3", SrcName: "zlib", SrcEpoch: 0, SrcVersion: "1.2.11", SrcRelease: "16.el8_2"},
				{Name: "libedit", Epoch: 0, Version: "3.1", Release: "23.20170329cvs.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"BSD"}, Maintainer: "CentOS", Digest: "md5:bcd179f588923a78be4b41d5ec7a7a39", SrcName: "libedit", SrcEpoch: 0, SrcVersion: "3.1", SrcRelease: "23.20170329cvs.el8"},
				{Name: "git-core", Epoch: 0, Version: "2.18.4", Release: "2.el8_2", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2"}, Maintainer: "CentOS", Digest: "md5:ba87010c46d7b15221de570dca65076d", SrcName: "git", SrcEpoch: 0, SrcVersion: "2.18.4", SrcRelease: "2.el8_2"},
				{Name: "keyutils-libs-devel", Epoch: 0, Version: "1.5.10", Release: "6.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv2+ and LGPLv2+"}, Maintainer: "CentOS", Digest: "md5:c2a3fa1cb645573a2b44c54118c43762", SrcName: "keyutils", SrcEpoch: 0, SrcVersion: "1.5.10", SrcRelease: "6.el8"},
				{Name: "groff-base", Epoch: 0, Version: "1.22.3", Release: "18.el8", Arch: "x86_64", Modularitylabel: "", Licenses: []string{"GPLv3+ and GFDL and BSD and MIT"}, Maintainer: "CentOS", Digest: "md5:6ed091baa7cbbcf9205fce2f24fa48cb", SrcName: "groff", SrcEpoch: 0, SrcVersion: "1.22.3", SrcRelease: "18.el8"},
				{Name: "perl-Pod-Simple", Epoch: 1, Version: "3.35", Release: "395.el8", Arch: "noarch", Modularitylabel: "", Licenses: []string{"GPL+ or Artistic"}, Maintainer: "CentOS", Digest: "md5:4634886be1066e76224760899bd706d5", SrcName: "perl-Pod-Simple", SrcEpoch: 1, SrcVersion: "3.35", SrcRelease: "395.el8"},
			},
		},
	}
	a := rpmPkgAnalyzer{}
	for testname, tc := range tests {
		t.Run(testname, func(t *testing.T) {
			f, err := os.Open(tc.path)
			require.NoError(t, err)
			defer f.Close()

			got, _, err := a.parsePkgInfo(f)
			require.NoError(t, err)

			sort.Sort(tc.pkgs)
			sort.Sort(got)

			for i := range got {
				got[i].ID = ""
				got[i].DependsOn = nil // TODO: add tests
			}

			assert.Equal(t, tc.pkgs, got)
		})
	}
}

func Test_splitFileName(t *testing.T) {
	tests := []struct {
		name     string
		filename string
		wantName string
		wantVer  string
		wantRel  string
		wantErr  bool
	}{
		{
			name:     "valid name",
			filename: "glibc-2.17-307.el7.1.src.rpm",
			wantName: "glibc",
			wantVer:  "2.17",
			wantRel:  "307.el7.1",
			wantErr:  false,
		},
		{
			name:     "invalid name",
			filename: "elasticsearch-5.6.16-1-src.rpm",
			wantName: "",
			wantVer:  "",
			wantRel:  "",
			wantErr:  true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			gotName, gotVer, gotRel, err := splitFileName(tt.filename)
			if tt.wantErr {
				assert.Error(t, err)
			} else {
				assert.NoError(t, err)
			}
			assert.Equal(t, tt.wantName, gotName)
			assert.Equal(t, tt.wantVer, gotVer)
			assert.Equal(t, tt.wantRel, gotRel)
		})
	}
}
