#!/usr/bin/env bash
# Upload Uppy releases to Edgly.net CDN. Copyright (c) 2018, Transloadit Ltd.
#
# This file:
#
#  - Assumes EDGLY_KEY and EDGLY_SECRET are available (e.g. set via Travis secrets)
#  - Tries to load env.sh instead, if not
#  - Checks if a tag is being built (on Travis - otherwise opts to continue execution regardless)
#  - Installs AWS CLI if needed
#  - Assumed a fully built uppy is in root dir (unless a specific tag was specified, then it's fetched from npm)
#  - Runs npm pack, and stores files to e.g. https://transloadit.edgly.net/releases/uppy/v0.30.3/uppy.css
#  - Uses local package by default, if [version] argument was specified, takes package from npm
#
# Run as:
#
#  ./upload-to-cdn.sh [version]
#
# To upload all versions in one go (DANGER):
#
#  git tag |awk -Fv '{print "./bin/upload-to-cdn.sh "$2}' |bash
#
# Authors:
#
#  - Kevin van Zonneveld <kevin@transloadit.com>
set -o pipefail
set -o errexit
set -o nounset
# set -o xtrace

# Set magic variables for current FILE & DIR
__dir="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
__file="${__dir}/$(basename "${BASH_SOURCE[0]}")"
__base="$(basename ${__file} .sh)"
__root="$(cd "$(dirname "${__dir}")" && pwd)"

function fatal () {
  echo "❌ ${*}";
  exit 1
}

pushd "${__root}" > /dev/null 2>&1
  if [ -n "${TRAVIS:-}" ]; then
    if [ "${TRAVIS_PULL_REQUEST:-}" != "false" ]; then
      echo "On Travis (TRAVIS is '${TRAVIS}'), I'm not pushing releases to the CDN for pull requests (TRAVIS_PULL_REQUEST is '${TRAVIS_PULL_REQUEST}')"
      exit 0
    fi
    if [[ ! "$TRAVIS_COMMIT_MESSAGE" =~ ^Release* ]]; then
      echo "On Travis (TRAVIS is '${TRAVIS}'), I'm not pushing releases to the CDN unless commit message starts with 'Release' (TRAVIS_COMMIT_MESSAGE is '${TRAVIS_COMMIT_MESSAGE}')"
      exit 0
    fi
  fi


  if [ -z "${EDGLY_KEY:-}" ] && [ -f ./env.sh ]; then
    source ./env.sh
  fi
  [ -z "${EDGLY_KEY:-}" ] && fatal "Unable to find or source EDGLY_KEY env var"

  type aws || pip install --user awscli

  remoteVersion="${1:-}"
  version="${remoteVersion}"
  if [ -z "${remoteVersion}" ]; then
    localVersion=$(node -pe "require('./packages/uppy/package.json').version")
    echo "${localVersion}"
    version="${localVersion}"
  fi

  majorVersion=$(echo "${version}" |awk -F. '{print $1}')

  echo -n "--> Check if not overwriting an existing tag ... "
  env \
    AWS_ACCESS_KEY_ID="${EDGLY_KEY}" \
    AWS_SECRET_ACCESS_KEY="${EDGLY_SECRET}" \
  aws s3 ls \
    --region="us-east-1" \
  "s3://crates.edgly.net/756b8efaed084669b02cb99d4540d81f/default/releases/uppy/v${version}/uppy.min.css" > /dev/null 2>&1 && fatal "Tag ${version} already exists"
  echo "✅"


  echo "--> Obtain relevant npm files for robodog ${version} ... "
  pushd packages/@uppy/robodog
    if [ -z "${remoteVersion}" ]; then
      npm pack || fatal "Unable to fetch "
    else
      npm pack "@uppy/robodog@${remoteVersion}"
    fi
  popd > /dev/null 2>&1
  echo "✅"
  rm -rf /tmp/robodog-to-edgly
  mkdir -p /tmp/robodog-to-edgly
  cp -af "packages/@uppy/robodog/uppy-robodog-${version}.tgz" /tmp/robodog-to-edgly/
  tar zxvf "packages/@uppy/robodog/uppy-robodog-${version}.tgz" -C /tmp/robodog-to-edgly/

  echo "--> Upload robodog to edgly.net CDN"
  pushd /tmp/robodog-to-edgly/package/dist
    # --delete \
    env \
      AWS_ACCESS_KEY_ID="${EDGLY_KEY}" \
      AWS_SECRET_ACCESS_KEY="${EDGLY_SECRET}" \
    aws s3 sync \
      --region="us-east-1" \
      --exclude 'node_modules/*' \
    ./ "s3://crates.edgly.net/756b8efaed084669b02cb99d4540d81f/default/releases/uppy/v${version}"
    echo "Saved https://transloadit.edgly.net/releases/uppy/v${version}/"
  popd > /dev/null 2>&1
  rm -rf /tmp/robodog-to-edgly


  echo "--> Obtain relevant npm files for uppy ${version} ... "
  pushd packages/uppy
    if [ -z "${remoteVersion}" ]; then
      npm pack || fatal "Unable to fetch "
    else
      npm pack "uppy@${remoteVersion}"
    fi
  popd > /dev/null 2>&1
  echo "✅"
  rm -rf /tmp/uppy-to-edgly
  mkdir -p /tmp/uppy-to-edgly
  cp -af "packages/uppy/uppy-${version}.tgz" /tmp/uppy-to-edgly/
  tar zxvf "packages/uppy/uppy-${version}.tgz" -C /tmp/uppy-to-edgly/

  echo "--> Upload uppy to edgly.net CDN"
  pushd /tmp/uppy-to-edgly/package/dist
    # --delete \
    env \
      AWS_ACCESS_KEY_ID="${EDGLY_KEY}" \
      AWS_SECRET_ACCESS_KEY="${EDGLY_SECRET}" \
    aws s3 sync \
      --region="us-east-1" \
      --exclude 'node_modules/*' \
    ./ "s3://crates.edgly.net/756b8efaed084669b02cb99d4540d81f/default/releases/uppy/v${version}"
    echo "Saved https://transloadit.edgly.net/releases/uppy/v${version}/"
  popd > /dev/null 2>&1
  rm -rf /tmp/uppy-to-edgly


  echo "${version}" | env \
    AWS_ACCESS_KEY_ID="${EDGLY_KEY}" \
    AWS_SECRET_ACCESS_KEY="${EDGLY_SECRET}" \
  aws s3 cp \
    --region="us-east-1" \
    --content-type="text/plain" \
  - "s3://crates.edgly.net/756b8efaed084669b02cb99d4540d81f/default/releases/uppy/latest.txt"
  echo "Saved https://transloadit.edgly.net/releases/uppy/latest.txt"
  echo "${version}" | env \
    AWS_ACCESS_KEY_ID="${EDGLY_KEY}" \
    AWS_SECRET_ACCESS_KEY="${EDGLY_SECRET}" \
  aws s3 cp \
    --region="us-east-1" \
    --content-type="text/plain" \
  - "s3://crates.edgly.net/756b8efaed084669b02cb99d4540d81f/default/releases/uppy/v${majorVersion}-latest.txt"
  echo "Saved https://transloadit.edgly.net/releases/uppy/v${majorVersion}-latest.txt"
popd > /dev/null 2>&1
